/* Copyright (c) 1988,1989 by Sozobon, Limited.  Author: Johann Ruegg
 *           (c) 1990 - 2009 by H. Robbers.   ANSI upgrade.
 *
 * This file is part of AHCC.
 *
 * AHCC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * AHCC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AHCC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *	e2.c
 *
 *	Expression tree routines.
 *
 *	Constant folding, typing of nodes, simple transformations.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "param.h"

#include "body.h"
#include "decl.h"
#include "d2.h"
#include "expr.h"
#include "e2.h"
#include "gen.h"
#include "gsub.h"
#include "md.h"
#include "out.h"

#define debug_x G.yflags['x'-'a']
#define debug_s G.yflags['s'-'a']
#define debugQ  G.xflags['q'-'a']
#define debugC (G.xflags['c'-'a'])		/* casting */
#define debugM  G.xflags['m'-'a']

#define ICONRIGHT 1			/* icons to the left give better code (moveq #n,Dn) */

static void impl_deref(NP np, TP tp)
{
	NP cp = copyone(np);
	np->token = DEREF;
	np->tt = E_UNARY;
	clr_flgs(np);
	cp->left = np->left;

/* 11'09 HR: forgot about the right part */

/*	messagen(cp, "implicit deref %s %s",
		np->left ? "L" : "", np->right ? "R" : "");
*/
	cp->right = np->right;
	np->right = nil;

	np->left = cp;
	np->type = tp;
	to_nct(np);
	name_to_str(np, "impl_deref");
}

/* new function name seen in expr */
static TP new_func(NP op)
{
	TP np;
	D_(_x, "new_func");
	/* we know left, right and type are nil */
	np = e_copyone_t(op);
	np->type = basic_type(K_PROC);
	to_nct(np);
	np->sc = K_EXTERN;
	globl_sym(np);
	np->nflags.f.nheap = op->nflags.f.nheap;		/* symtab must own the name */
	op->nflags.f.nheap = NCOPY;
	D_1(_x, "after new_func", np);
	return np;
}

static void see_func(NP np)	/* restricted version of see_id() */
{
	TP tp = all_syms((TP)np, 0);
	D_2(_x, "after all_syms see_func", np, tp);
	if (tp eq nil)
	{
#if FOR_A
		errorn(np, "undefined function");
#else
		warnn(np, "no prototype for");
#endif
		tp = new_func(np);
	}
	if (tp->sc eq K_REG)
		np->rno = tp->rno;
	/* we need these for the implicit deref */
	np->sc = tp->sc;
	np->val.i = tp->offset;
	np->lbl = tp->lbl;
	np->cflgs.f.undef   = tp->cflgs.f.undef;
	np->cflgs.f.see_reg = tp->cflgs.f.see_reg;
	np->cflgs.f.cdec    = tp->cflgs.f.cdec;
	clr_flgs(np);
	np->type = tp->type;
	to_nct(np);
}

/* (type yielding func) (args) */
static void function_type(NP np, short context)
{
	NP lp; TP typ;

	D_(_x, "function_type");

	lp = np->left;
#if FOR_A
	if (lp->token ne STMT)
#endif
		if (lp->token eq ID)
			see_func(lp);		/* may be new ID; */
		else
			form_types(lp, context, 0);

	typ = lp->type;

	if (    typ->token ne K_PROC
		and typ->token ne L_PROC)		/* still not a func ?? */
	{

/*   ANSI:	!!I hate this!!
	imlicit dereferencing does not belong to C,
	its not done anywhere else in C,
	AND its done only 1 deep, so it is not even complete.

	Try (with Pure C too):
	 'void (**toff)();
	  void p(void) { toff();}'			you could do 2 impicit derefs,

	no no, you must do '(*toff)();'		This is sooo stupid.
*/
		TP ttp = typ;
		typ = nil;
		if (ttp->token eq REFTO)
		{
			ttp = ttp->type;
			if (   ttp->token eq K_PROC
				or ttp->token eq L_PROC)
			{
				impl_deref(lp, ttp);
				typ = ttp->type;	/* return type */
			}
		}
	othw
		typ = typ->type;			/* The return type */
	}

	if (typ)
	{
		if (typ->token eq REFTO)
			np->rno = AREG;
#if FLOAT
		elif (is_hw_fp(typ->ty))		/* was np ????? */
			np->rno = FREG;
#endif
	othw
		error("call non-function");
		typ = basic_type(G.for_A ? T_VOID : T_INT);
	}

	np->type = typ;
	to_nct(np);

	D_1(_x, "after function_type", np);
}

static void toomany(NP np)
{
	errorn(np, "too many parameters");
}

static void toofew(NP np)
{
	errorn(np, "too few parameters");
}

#if PROMOTE
/* 3'11 HR promote by casts (simplify onearg()) */
static void promote(NP arg)
{
	NP  np = arg->left;
	TP  tp = np->type,
	   itp = default_type(-1, 0);
#if FLOAT
	TP ftp = basic_type(T_REAL);
#endif

	switch (tp->ty)
	{
		case ET_U:
		case ET_S:			/* promote int to default int size */
			if (tp->size < itp->size)
			{
				Cast(np, itp, ARGCNV, "Promote int");
				arg->type = np->type;
				to_nct(arg);
			}
		break;
#if FLOAT
		case ET_R:			/* promote real to default real size */
			if (tp->size < ftp->size)
			{
				Cast(np, ftp, ARGCNV, "Promote float");
				arg->type = np->type;
				to_nct(arg);
			}
		break;
#endif
	}
}

static void promote_args(NP np)
{
	while (np)
	{
		promote (np);
		np = np->right;
	}
}
#endif

static void match_args(NP np)	/* and insert argcasts  */
{
	TP tl, tls; NP args;
	short i, j;
	bool loc = np->left->type->token eq L_PROC;

	D_(_x, "match_args");

	tl = np->left->type;

	if ( tl->list eq nil and np->right eq nil ) return;		/* 11'09 HR check before ANSI args */

	if (!tl->tflgs.f.ans_args)
	{
		warnn(np->left, "no args matching for");
#if PROMOTE
		promote_args(np->right);
#endif
		return;
	}

	tl = tl->list;					/* list is ID or REFTO */
	args = np->right;				/* right is arg expr */

	if ( tl eq nil )
	{
		toomany(np);
		return;
	}

	/* tl now !nil */

	if ( args eq nil )
	{
		if ( yields_ty(tl, T_VARGL) eq nil )
			toofew(np);
		return;
	}

	/* both tl and args now !nil */

	tls = tl;
	j = 0;
	while(tls)
	{
		if (yields_ty(tls, T_VARGL)) break;
		j++;
		tls = tls->next;
	}
	/* no of args in typelist excl vargl */

	i = 1;								/* already excluded 0 args */
	if ( args->token eq ARG )
		i = args->val.i;					/* no of args in CALL */

	D_D(_x, send_msg("i(args): %d\tj(list): %d\n", i, j);)

	if (i<j)
	{
		toofew(np);
		return;
	}

	if ( i>j and !tls )					/* T_VARGL leaves tls!nil */
	{
		toomany(np);
		return;
	}

	while (args)
	{
		args->rno = -1;
		if (arg_check(args, tl, loc) eq 0)		/* false: found  T_VARGL */
		{
			args->eflgs.f.varg = 1;
#if PROMOTE
			promote_args(args);
#endif
			return;
		}
		args->type = args->left->type;			/* already to_nct */
		tl = tl->next;
		args = args->right;
	}
}

/* np points to ID or DEREF or SELECT node
	type is a COPY
	type token is ROW  */

static void see_array(NP np)
{
	NP tp;

#if BIP_ASM
	if (G.for_S)
		return;
#endif
	D_1(_x, "see_array", np);
	tp = copyone(np);
	tp->left = np->left;
	np->size = tp->type->size;		/* keep total size */
	tp->type = tp->type->type;		/*	hier verdwijnt Arr of, maar staat nog in np
									    dus houdt zijn NCT ?*/
	np->left = tp;
	np->token = TAKEAD;
	np->tt = E_UNARY;
	name_to_str(np, "array_to_ptr");
	array_to_ptr((TP)np);			/* Arr of --> Arref to */
#if FOR_A
	np->type->name = np->left->name;	/* new 'for' syntax: must have name of terminal array */
	np->type->nflags.f.nheap = NCOPY;
#endif
	D_1(_x, "after see_array", tp);
}

/* (struct|union).ID */
static void member_type(NP xp)
{
	NP rp, lp; TP sup, rv;

	D_1(_x, "member_type", xp);

	rp = xp->right;
	lp = xp->left;
	sup = lp->type;

/* already checked that rp->token is ID */

	if ( !is_aggreg(sup) )
	{
		errorn(lp, "select non-%s/%s", graphic[K_STRUCT], graphic[K_UNION]);
		rv = default_type(-1, 0);
	othw
		rv = tlook(sup->list, (TP)rp);
		if (rv eq nil)
		{
			errorn(rp, "not member");
			rv = default_type(-1, 0);
		othw
			D_1(_x, "member found", rv);
			xp->val.i = rv->offset;
			if (rv->fld.width)
				xp->fld = rv->fld;
/*			{
				xp->fldw = rv->fldw;
				xp->fldo = rv->fldo;
			}
*/			xp->name = rv->name;		/* remember selection's name */
			xp->nflags.f.nheap = NCOPY;
			KEEP_QUAL(xp, lp);
			rv = rv->type;
		}
	}

	xp->type = rv;
	to_nct(xp);

	/* change to UNARY op */
	xp->tt = E_UNARY;
	freenode(rp);
	xp->right = nil;

	/* change ROW OF to REF TO */
	if (rv->token eq ROW)
		see_array(xp);
	D_1(_x, "after member_type", xp);
}

/* used for CASE value, row index decl, enum value,
   size of bitfields & #if truth value	*/
global long confold_value(NP np, short context)
{
	long l;

	D_(_x, "confold_value");

	if (np)
	{
		form_types(np, context, 0);	/* confold now in form_types */

		if (np->token eq ICON)
		{
			l = np->val.i;
			freenode(np);
			return l;
		}
	}

	D_1(_x, "confold_value no ICON", np);
	error("needs constant expression");
	return 0;
}

#if BIP_ASM

/* similar to get_arglist, but calls questx() */

/* used in assembler: The association of ( in eg d(a0) is different from a call */
global
NP asm_expr(void)
{
	NP np, ind, e1;

	D_(_x, "asm_expr");

	np = questx();		/* xpr without commas */

	if (np)
	{
		if ( (np->category & ASM) eq 0)
		{
			if (np->token eq DEREF)		/* 05'11 HR: unary '*' --> Instruction Counter */
				np->token = INSTR;
			elif (cur->token eq PAREN)		/* followed by (...) */
			{
				ind = npcur(); advnode();
				e1 = get_expr();
				eat(NERAP);
				if (e1)
				{
					ind->token = REGINDISP;
					ind->tt = E_SPEC;
					ind->left = np;
					ind->right = e1;
					np = ind;
				othw
					error("empty ()");
					freenode(np);
					freenode(ind);
					return nil;
				}
			othw
			/* if questx() yields a COMMA expr, the expression must
			   have been held between parentheses;
			   questx() --> primary() :: '(' --> get_expr()
			*/
				if (np->token eq COMMA)		/* (... , ...) */
				{
					np->token = REGINDX;
					np->tt = E_SPEC;
				}
			}
		}
	}

	form_types(np, FORSIDE, 0);	/* includes confold */

	return np;
}
#endif

static void newicon(NP np, long x)
{
	D_(_x, "newicon");

	np->token = ICON;
	np->val.i = x;
	new_name(np, "%ld", x);
	np->tt = E_LEAF;
	if (np->left)
	{
		freenode(np->left);
		np->left = nil;
	}
	if (np->right)
	{
		freenode(np->right);
		np->right = nil;
	}
}

#if FLOAT
global
void newfcon(NP np, double x, short r_or_f)
{
	D_(_x, "newfcon");

	if (r_or_f eq RCON)
	{
		new_rnode(np, x);
	othw
		np->token = FCON;
		np->val.f = x;
	}
	new_name(np, "%g", x);
	np->tt = E_LEAF;
	if (np->left)
	{
		freenode(np->left);
		np->left = nil;
	}
	if (np->right)
	{
		freenode(np->right);
		np->right = nil;
	}
}
#endif

static void insptrto(NP np)
{
	D_(_x, "insptrto");
	{
		NP op = copyone(np);

		np->left = op;
		np->token = TAKEAD;
		np->tt = E_UNARY;
		name_to_str(np, "&Proc");
	}

/* position in flow of see_id changed
   must now complete types */
	{
		TP op = basic_type(REFTO);		/* construct TAKEAD's type */
		name_to_str(op, "ProcRef");
		op->type = np->type;		/* func */
		to_nct(op);
		np->type = op;				/* ref to func */
		not_nct(np);
	}
}

#if BIP_ASM
static
TP asm_symbol(NP np)
{
	TP tp;

	tp = allocTn(1);
	tp->type = asm_type();
	to_nct(tp);
	tp->name = np->name;
	tp->nflags.f.nheap = np->nflags.f.nheap;
	tp->nflags.f.dot   = np->nflags.f.dot;
	np->nflags.f.nheap = NCOPY;				/* symtab owns name */
	tp->sc   = K_AHEAD;
#if DOTLABELS
	tp->area_info.id = (tp->nflags.f.dot and G.scope) ? area_id : 0;	/* 2'11 HR */
#else
	tp->area_info.id = 0;
#endif

	tp->area_info.class = in_class;

	np->area_info = tp->area_info;
	np->sc = tp->sc;				/* 11'10 HR */

#if DOTLABELS
	if (tp->nflags.f.dot)
		np->lbl = dot_sym(tp);
	else
#endif
	if (G.scope)
		loc_sym(tp);
	else
		globl_sym(tp);

	return tp;
}
#endif

static
TP undef_symbol(NP np)
{
	TP tp;

	errorn(np, "undefined");
	tp = allocTn(1);
	tp->cflgs.f.undef = 1;			/* to avoid a lot of identical err msgs */
	np->cflgs.f.undef = 1;
	tp->type   = default_type(-1, 0);
	to_nct(tp);
	tp->offset = 0;
	tp->name   = np->name;
	tp->nflags.f.nheap = np->nflags.f.nheap;
	np->nflags.f.nheap = NCOPY;				/* symtab owns name */
	if (G.scope)
		loc_sym(tp);
	else
		globl_sym(tp);

	return tp;
}

static void see_id(NP np, short context, short usage)
{
	TP tp;
#if BIP_ASM
	void asm_offs(NP, TP);
#endif

	D_(_x, "see_id");

	if (usage ne FORLVAL)
		usage = FORSEE;

#if BIP_ASM && DOTLABELS
	if (G.dot_seen)
	{
		G.dot_seen = false;
		np->nflags.f.dot = 1;
	}

	if (np->nflags.f.dot)
		tp = tlook(G.dot_labels, (TP)np);
	else
#endif
		tp = all_syms((TP)np, usage);

	if (tp eq nil)
	{
		if (context eq FORTRUTH)		/*for #if expressions */
		{
			np->token = ICON;
			np->val.i = 0;				/* constant representing falsehood */
			np->type = default_type(-1, 0);
			to_nct(np);
			return;
		}
#if BIP_ASM
		/* if asm, make symbol table entry for advance reference (K_AHEAD) */
		if (G.for_S)
		{
			tp = asm_symbol(np);
			asm_offs(np, tp);		/* convert offset ID and K_AUTO to ICON */
		}
		else
#endif
			tp = undef_symbol(np);
	othw
#if USAGE
		if (context eq FORLVAL)			/* 12'08 HR usage tracking */
			tp->tflgs.f.inid = 1;
#endif
		if (tp->token eq ICON)
			np->token = ICON;			/* t.off is same location as e.val.i */
		elif (tp->sc eq K_REG)
			np->rno = tp->rno;

		np->sc    = tp->sc;
		np->val.i = tp->offset;
		np->area_info = tp->area_info;
		np->lbl   = tp->lbl;
		take_flgs(np, tp);
#if BIP_ASM
		if (G.for_S)
			asm_offs(np, tp);		/* convert offset ID and K_AUTO to ICON */
#endif
	}

	np->type = tp->type;
	to_nct(np);
	tp = np->type;

	/* special conversions */
	if (   tp->token eq K_PROC
		or tp->token eq L_PROC)		/* local procedure */
	{
		insptrto(np);
	}

#if FOR_A
	if (G.for_A)
		if (tp->tflgs.f.derefto)
			impl_deref(np, tp->type);
#endif
}

static TP xcon_ty(NP lp)
{
	if (lp->token eq ICON)	return basic_type(icon_ty(lp));
#if FLOAT
	if (lp->token eq RCON)	return basic_type(T_REAL);
	if (lp->token eq FCON)	return basic_type(T_FLOAT);
#endif
#if LONGLONG
	if (lp->token eq LCON)	return basic_type(T_LLONG);
#endif
	return nil;
}

/* UNARY */

static void ucanon(NP np)	/* somewhat more straightforward */
{
	D_1(_x, "ucanon", np);

	if (np->token eq K_SIZEOF)
	{
/*  I needed the true pointer size(4) in the Arr_of type node.
	so I put the total size if a array in the TAKEAD expression node */
		newicon(np, (np->left->type->tflgs.f.saw_array)
					? np->left->      size
					: np->left->type->size
				);
		np->type = basic_type(SIZE_T);
		to_nct(np);
		return;
	}

	if (np->token eq K_OFFSET)
	{
		if (np->left->token eq SELECT)
		{
			newicon(np, np->left->val.i);
			np->type = basic_type(SIZE_T);
			to_nct(np);
		}
		else
			error("must select a member");

		return;
	}

	if (np->token eq TOFFSET)
	{
		TP mp;
		mp = tlook(np->type->list, np->left);
		if (mp)
		{
			newicon(np, mp->offset);
			np->type = basic_type(SIZE_T);
			to_nct(np);
		}
		else
			errorn(np->type, "'%s' not member of ", np->left->name);
		return;
	}

#if FLOAT
	if (np->left->token eq FCON)
	{
		if (np->token eq NEGATE)
		{
			np->type = np->left->type;
			to_nct(np);
			newfcon(np, -np->left->val.f, FCON);
		}
		elif (np->token eq UPLUS)
		{
			np->type = np->left->type;
			to_nct(np);
			newfcon(np, np->left->val.f, FCON);
		}
	}
	elif (np->left->token eq RCON)
	{
		if (np->token eq NEGATE)
		{
			np->type = np->left->type;
			to_nct(np);
			newfcon(np, -getrcon(np->left), RCON);	/* dbl s/w indirect voor RLNODE  */
		}
		elif (np->token eq UPLUS)
		{
			np->type = np->left->type;
			to_nct(np);
			newfcon(np, getrcon(np->left), RCON);
		}
	}
#endif

/* Here LCON folding */
	elif (np->left->token eq ICON)
	{
/* D_1(_x, "ucanon left icon", np); */
		switch (np->token)
		{
		case UPLUS:
			np->type = np->left->type;
			to_nct(np);
			newicon(np, np->left->val.i);
		break;
		case NEGATE:
			np->type = np->left->type;
			to_nct(np);
			newicon(np, -np->left->val.i);
		break;
		case BINNOT:
			np->type = np->left->type;
			to_nct(np);
			newicon(np, ~np->left->val.i);
		break;
		case NOT:
			np->type = np->left->type;
			to_nct(np);
#if BIP_ASM
			if (G.for_S)
				newicon(np, ~np->left->val.i);
			else
#endif
				newicon(np, !np->left->val.i);
		break;
		}
	}
}

/* switch pseudo-commutative op */
static void swt_op(NP np)
{
	short newtok;

	D_(_x, "swt_op");

	switch (np->token)
	{
	case LTEQ:		newtok = GREATER;	break;
	case GTEQ:		newtok = LESS;		break;
	case LESS:		newtok = GTEQ;		break;
	case GREATER:	newtok = LTEQ;		break;
	default:
		return;
	}
	np->token = newtok;
	name_to_str(np, graphic[newtok]);
}

bool stronger(TP xp, TP  yp);		/* ex MD.C */

/* BINARY 2 ICON's */
static void b2i(NP np)
{
	long l, r, x;
	D_(_x, "b2i");

	r = np->right->val.i;
	l = np->left->val.i;

	switch (np->token)
	{
	case PLUS:
			x = l + r;	break;
	case MINUS:
			x = l - r;	break;
	case TIMES:
			x = l * r;	break;
	case DIV:
			x = l / r;	break;
	case MOD:
			x = l % r;	break;
	case GREATER:
			x = l > r;	break;
	case LESS:
			x = l < r;	break;
	case LTEQ:
			x = l <= r;	break;		/* 03'09 oooops ( was >= ) */
	case GTEQ:
			x = l >= r;	break;		/* 03'09 oooops ( was <= ) */
	case EQUALS:
			x = l eq r;	break;
	case NOTEQ:
			x = l ne r;	break;
	case BINAND:
			x = l & r;	break;
	case BINOR:
			x = l | r;	break;
	case BINEOR:
			x = l ^ r;	break;
	case LSHIFT:
			x = l << r;	break;
	case RSHIFT:
			x = l >> r;	break;
	default:
		return;
	}

	np->eflgs.i = np->left->eflgs.i > np->right->eflgs.i
					? np->left->eflgs.i
					: np->right->eflgs.i;
	if (!(np->eflgs.f.see_u or np->eflgs.f.see_l))
		np->val.i = x,
		np->type = basic_type(icon_ty(np));
	else
		np->type = stronger(np->left->type, np->right->type)
				? np->left->type
				: np->right->type;
	to_nct(np);

	newicon(np, x);
}


#if FLOAT
/* BINARY 2 RCON's */
static void b2f(NP np, double l, double r)
{
	double x;
	short ix, isint;

	D_(_x, "b2f");

	isint = 0;

	switch (np->token)
	{
	case PLUS:
			x = l + r;	break;
	case MINUS:
			x = l - r;	break;
	case TIMES:
			x = l * r;	break;
	case DIV:
			x = l / r;	break;
	case GREATER:
			ix = l > r;  isint++;	break;
	case LESS:
			ix = l < r;  isint++;	break;
	case LTEQ:
			ix = l <= r; isint++;	break;	/* 03'09 oooops ( was >= ) */
	case GTEQ:
			ix = l >= r; isint++;	break;	/* 03'09 oooops ( was <= ) */
	case EQUALS:
			ix = l eq r; isint++;	break;
	case NOTEQ:
			ix = l ne r; isint++;	break;
	default:
		return;
	}

	to_nct(np);

	if (isint)
	{
		newicon(np, ix);
		np->type = basic_type(T_BOOL);	/* ty ET_B */
	othw
		newfcon(np, x, RCON);	/* intermediate arithm allways double */
		np->type = basic_type(T_REAL);
	}
}
#endif

/* 10'10 HR: Now only called for X_THEN */
static void xcanon(NP np)
{
	long l = np->left->val.i;
	NP onp, tp = np;

	D_(_x, "xcanon");

	tp = tp->right;				/* X_ELSE node */

	if (l)
	{							/* take true side */
		onp = tp->left;
		freenode(tp->right);
	othw						/* take false side */
		onp = tp->right;
		freenode(tp->left);
	}

	freenode(np->left);
	freeunit(np->right);
	copyinto(np, onp);
	onp->nflags.f.nheap = NCOPY;
	freeunit(onp);
	D_1(_x, "after xcanon", np);
}

static void collaps(NP np, long truth)
{
	np->token = ICON;
	np->val.i = truth;
	np->tt = E_LEAF;
	freenode(np->left);
	np->left = nil;
	freenode(np->right);
	np->right = nil;
}

static void pull_right(NP np)
{
	NP onp = np->right;
	freenode(np->left);			/* icon */
	copyinto(np, onp);
	onp->nflags.f.nheap = NCOPY;
	freeunit(onp);
}

static void pull_left(NP np)
{
	NP onp = np->left;
	freenode(np->right);		/* icon */
	copyinto(np, onp);
	onp->nflags.f.nheap = NCOPY;
	freeunit(onp);
}

/* 10'10 HR: AND, OR */
static void ycanon(NP np)
{
	short lt = np->left ->token eq ICON,
	      rt = np->right->token eq ICON,
	      tok = np->token;
	long l = np->left ->val.i,
	     r = np->right->val.i;

	if (lt and rt)
		if (tok eq AND)
			collaps(np, l and r);
		else
			collaps(np, l or  r);
	elif (lt)
		if (tok eq AND and l eq 0)	/* false for sure */
			collaps(np, 0);
		elif (tok eq OR and l ne 0)	/* true for sure  */
			collaps(np, 1);
		else
			pull_right(np);			/* dont bother about icon */
	elif (rt)
		if (tok eq AND and r eq 0)	/* false for sure */
			collaps(np, 0);
		elif (tok eq OR and r ne 0)	/* true for sure  */
			collaps(np, 1);
		else
			pull_left(np);			/* dont bother about icon */
}

/* confold integrated in form_types:
	its easier when you have the types already determined. */
static void bcanon(NP np)
{
	short ltok, rtok;
	NP tp;

	D_1(_x, "[1]bcanon", np);

	ltok = np->left->token;
	rtok = np->right->token;

#if ICONRIGHT
												/* previous */
	if (!is_icon(ltok) and is_icon(rtok) and np->token eq MINUS)
	{
		np->token = PLUS;					/* X - con --> X + -con (This sure helps :-) */
		np->right->val.i = -np->right->val.i;
	}

	if (is_con(ltok) and !is_con(rtok))		/* left is ?CON, right is not */
	{
		if (is_C_A_NA(np->token))
		{						/* reverse sides  - put CON on right */
			tp = np->left;
			np->left = np->right;
			np->right = tp;
			if (is_C_NA(np->token))
				swt_op(np);
			D_1(_x, "[2a]bcanon", np);
		}
		return;
	}
#else
	/* opposite of previous: put CON on left !!!!! (more asm optimizations possible (moveq #,Dn)) */
	/* MINUS --> PLUS For better performance (more often) of b_assoc (now left_assoc & right_assoc) */

	if (is_icon(ltok) and !is_icon(rtok) and np->token eq MINUS)
	{
		np->token = PLUS;					/* con - X --> -con + X (This sure helps :-) */
		np->left->val.i = -np->left->val.i;
	}

	if (is_con(rtok) and !is_con(ltok))
	{
								/* right is ?CON, left is not */
		if (is_C_A_NA(np->token))
		{						/* reverse sides  - put CON on left */
			tp = np->right;
			np->right = np->left;
			np->left = tp;
			if (is_C_NA(np->token))
				swt_op(np);
		}
		D_1(_x, "[2b]bcanon", np);
		return;
	}
#endif

	if (is_icon(ltok) and is_icon(rtok))		/* both int */
	{
#if BIP_ASM
		if (np->left->area_info.id ne np->right->area_info.id)
		{
			send_msg("%d %d\n", np->left->area_info.id, np->right->area_info.id);
			error("undefined address arithmatic between different areas");
			return;
		}
#endif
		b2i(np);
		D_1(_x, "[3]bcanon", np);
		return;
	}

#if FLOAT
	if (is_con(rtok) and is_con(ltok))	/* mixed int, short or long real */
	{
		double l, r;

		if (ltok eq RCON)
			l = getrcon(np->left);		/* intermediate real arithmatic allways extended precision */
		elif (ltok eq FCON)
			l = np->left->val.f;
		else
			l = np->left->val.i;

		if (rtok eq RCON)
			r = getrcon(np->right);		/* idem */
		elif (rtok eq FCON)
			r = np->right->val.f;
		else
			r = np->right->val.i;

		b2f(np, l, r);
		D_1(_x, "[4]bcanon", np);
	}
#endif
}

/* canon for assoc. & comm. op */
/* Johann: this code will almost never be executed, but it was fun. */
/* OK, but it wasnt perfect ;-)
		See changes in bcanon, especially reversion of MINUS --> PLUS for constants
		Now it is done quite a lot more often.

		I made a left and right version of b_assoc to make the process
		independant of the position of the constants. Whether put left or right.
 */

static bool right_assoc(NP np)
{
	NP lp, rp;
	short tok;

	D_(_x, "right_assoc");

	lp = np->left;
	if (lp->token ne np->token)
		return false;
									/* left is same op as np */
	rp = np->right;
	tok = lp->right->token;
	if (!is_con(tok))
		return false;
	/* left.right is ?CON */
	tok = rp->token;
	if (is_con(tok))
	{
					/* have 2 CONS l.r and r -- put together on r */
		NP	ep;
		ep = lp->left;
		np->left = ep;
		np->right = lp;
		lp->left = rp;
					/* can now fold 2 CONS */
		/*	warn("right_assoc together right %ld %ld\n", lp->left->type->size, lp->right->type->size); */
		bcanon(lp);
		/*	console("confolded to %ld, %lx\n", lp->val.i, lp->type); */
	othw
					/* have 1 CON at l.r -- move to top right */
		NP	kp;
		kp = lp->right;
		lp->right = rp;
		np->right = kp;
		/*	warn("right_assoc top right\n"); */
		castdefault(lp->left, lp->right, lp->type);
	}

/*	D_1(_x, "na right_assoc", np);
*/
	return true;
}

static bool left_assoc(NP np)
{
	NP rp, lp;
	short tok;

	D_(_x, "left_assoc");

	rp = np->right;
	if (rp->token ne np->token)
		return false;
									/* right is same op as np */
	lp = np->left;
	tok = rp->left->token;
	if (!is_con(tok))
		return false;
	/* right.left is ?CON */
	tok = lp->token;
	if (is_con(tok))
	{
					/* have 2 CONS r.l and r -- put together on l */
		NP	ep;
		ep = rp->right;
		np->right = ep;
		np->left = rp;
		rp->right = lp;
					/* can now fold 2 CONS */
		bcanon(rp);
	othw
					/* have 1 CON at r.l -- move to top left */
		NP	kp;
		kp = rp->left;
		rp->left = lp;
		np->left = kp;
		castdefault(rp->left, rp->right, rp->type);
	}

/*	D_1(_x, "na left_assoc", np);
*/
	return true;
}

static bool is_useless(NP np, long val)
{
	NP rp = np->right, lp = np->left;
	if (is_icon(rp->token) and rp->val.i eq val)
	{
		freenode(rp);
		copyinto(np, lp);
		lp->nflags.f.nheap = NCOPY;
		freeunit(lp);
		return TRUE;
	}
	return FALSE;
}

static short save_scon(NP np)		/* for the sake of size moved to here from gen.c */
{
	NP tp;

	D_(_x, "save_Scon");

	tp = copyone(np);
	tp->nflags.f.nheap = np->nflags.f.nheap;		/* reverses default copyone action. */
	np->nflags.f.nheap = NCOPY;
	tp->lbl = new_lbl();
	tp->nflags.f.n_brkpr = 1;
#if NODESTATS
		G.ncnt[tp->nt]--;
		G.ncnt[GENODE]++;
#endif
	tp->nt = GENODE;
	addcode(tp, "L1:\nXZ");
	put_fifo(&G.strsave, &G.strlast, tp);
	set_class(np, STR_class);
	np->area_info.disp = std_areas->ar[np->area_info.class].size;
	std_areas->ar[np->area_info.class].size += tp->size + 1;

	return tp->lbl;
}

/* lcanon integrated in leaf_types, confold integrated in form_types */
static void leaf_types(NP np, short context, short usage)
{
	D_D(_x, send_msg("leaf_types:%s,%s\t", np->name, prcntxt(context));)

	switch (np->token)
	{
	case ID:
		see_id(np, context, usage);
		if (np->type->token eq ROW)
			if (context ne FORLAINIT)		/* ANSI: init local arrays */
				see_array(np);				/* change to &ID; insert UNARY & node */
		return;
	case SCON:
		if (!G.for_S)
			np->lbl = save_scon(np);
		np->type = basic_type(SCON);
	esac
	case MEMBER:
		np->type = default_type(-1, 0);
		np->token = ID;
	esac
	case TSIZEOF:
		newicon(np, np->type->size),
		np->type = basic_type(SIZE_T);
	esac
	case K_TRUE:
		newicon(np, 1);
		np->type = basic_type(T_BOOL);
	esac
	case K_FALSE:
		newicon(np, 0);
		np->type = basic_type(T_BOOL);
	esac
	case T_VOID:
		newicon(np, 0);
		np->type = basic_type(T_VOID);
	esac
	case K_NIL:							/* (void *)0L */
		newicon(np, 0);
		np->type = basic_type(T_ULONG);
	esac
	case K_NOP:					/* 03'09 */
	case K_REST:
	case K_SAVE:
		np->type = basic_type(T_VOID);
	esac
#if FOR_A
	case STMT:
		return;		/* type already established */
#endif
	default:								/* must be ?CON */
#if BIP_ASM
		if (np->category & ASM)
			np->type = basic_type(T_SHORT);
		else
#endif
		if ((np->type = xcon_ty(np)) eq nil)
		{
#if C_DEBUG
			CE_NX(np, "Weird leaf %s", ptok(np->token));
#else
			CE_NX(np, "Weird leaf: %d", np->token);
#endif
			np->type = default_type(-1, 0);
		}
	}
	to_nct(np);
}

static
void unary_types(NP np)
{
	NP lp; TP tp;

	D_D(_x, send_msg("unary_types:%s\t", np->name);)

	ucanon(np);			/* confold integrated in form_types */
						/* its easier when the types are already determined */

	if (np->tt ne E_UNARY)		/* probably done something */
		return;

	if (external_unary_op(np))		/* s/w dbl: now we know the type for */
		return;						/*	external defined unary operators */

	lp = np->left;
	tp = lp->type;				/* default */
#if FLOAT
	if (np->cflgs.f.rlop)		/* fpu monops */
	{
		mustty(lp, R_ARITH);
		tp = basic_type(T_REAL);
	}
	else
#endif
	switch (np->token)
	{
		case TOFFSET:
		break;
	case POSTINCR:
		mustlval(lp);
		mustty(lp, R_SCALAR);
		if (postincdec(np, tp, ASSIGN PLUS, INCR, MINUS))	/* X++ becomes (++X - 1) */
			return;
		break;
	case POSTDECR:
		mustlval(lp);
		mustty(lp, R_SCALAR);
		if (postincdec(np, tp, ASSIGN MINUS, DECR, PLUS))
			return;
		break;
	case INCR:
		mustlval(lp);
		mustty(lp, R_SCALAR);
		if (incdec(np, tp, ASSIGN PLUS))			/* ++X becomes X+=1 */
			return;
		break;
	case DECR:
		mustlval(lp);
		mustty(lp, R_SCALAR);
		if (incdec(np, tp, ASSIGN MINUS))
			return;
		break;
	case DEREF:
		if (mustty(lp, R_POINTER))
		{
			tp = default_type(-1, 0);			/* error */
			break;
		}

		KEEP_QUAL(np, tp);
		tp = tp->type;

/*		if (tp->token eq K_PROC)		/* HR spurious &fu */
			message("spurious (*func)");
*/
		np->type = tp;
		to_nct(np);
		/* ROW OF to REF TO */
		if (tp->token eq ROW)
			see_array(np);
		return;
	case TAKEAD:
		if (lp->token eq TAKEAD and lp->type->tflgs.f.saw_array)
		{
			/* change back to array_of */
			TP ltp = lp->type;
			NP keep = lp->left;
			np->left = keep;
			keep->type = ltp;
			not_nct(keep);
			ltp->token = ROW;
			ltp->size = lp->size;
			ltp->ty = ET_A;
			name_to_str(ltp, "array_of");
			ltp->tflgs.f.saw_array = 0;
			freeunit(lp);
			lp = keep;
		}
		mustlval(lp);
		if (lp->cflgs.f.see_reg)		/* actually specified 'register' */
			errorn(lp, "%s reg_var", graphic[ADDRESS]);
		tp = basic_type(REFTO);
		tp->type = lp->type;
		to_nct(tp);
		KEEP_QUAL(tp, lp);
		np->type = tp;
		return;
#if BIP_ASM
	case INSTR:				/* 05'11 HR: '*' operand */
		if (lp->token ne ICON)
		{
			errorn(lp, "+/- integer constant required");
			np->val.i = 0;
		}
		else
			np->val.i = lp->val.i;
		np->tt = E_LEAF;
		np->left = nil;
		freenode(lp);
		np->type = basic_type(T_LONG);
		to_nct(np);
		return;
#endif
	case NEGATE:
#if BIP_ASM
		if (G.for_S and lp->token eq REGINDIRECT)
		{
			*np = *lp;
			freeunit(lp);
			np->token = MINUSREGIND;
			return;
		}
#endif
		mustty(lp, R_ARITH);
#if COLDFIRE
		if (G.Coldfire)
		{
			cast_up(lp, basic_type(np->ty eq ET_S ? T_LONG : T_ULONG), "Cast neg");
			tp = lp->type;
		}
		else
#endif
		if (is_CC(lp))			/* 06'11 HR */
			cast_up(lp, basic_type(np->ty eq ET_S ? T_INT : T_UINT), "cast_cc unary minus");
		break;
	case UPLUS:
		mustty(lp, R_ARITH);
		if (is_CC(lp))			/* 06'11 HR */
			cast_up(lp, basic_type(np->ty eq ET_S ? T_INT : T_UINT), "cast_cc unary plus");
		break;
	case IMPCNV:
	case EXPCNV:
	case ARGCNV:
		if (np->type->token ne T_VOID)
		{
			mustty(lp, R_CC|R_SCALAR);
			mustty(np, R_CC|R_SCALAR);
		}

#if LONGLONG
		if (is_con(lp->token) or lp->token eq LCON)
#else
		if (is_con(lp->token))
#endif
		/* This happens when a cast is generated prior to call do_expr() */
		{
			short sv = is_nct(np);
							/* copy con --> np, tp is new type, remove old con */
			tp = np->type;	/* ...CNV's type = new type */
			*np = *lp;
			freeunit(lp);
			cast_con(np, tp);
			np->type = tp;
			if (sv)
				to_nct(np);
			else
				not_nct(np);
			return;
		}
		if (!G.nmerrors and G.casttab)
			external_cast(np);
		return;				/* type already specified */
	case NOT:				/* always yields a true boolean */
		mustty(lp, R_CC|R_SCALAR);
		if (!is_CC(lp))
			cmp_zero(lp);
		np->type = CC_type(np->left, nil);
		not_nct(np);
		return;
	case BINNOT:
		mustty(lp, R_INTEGRAL);
#if COLDFIRE
		if (G.Coldfire)
			tp = basic_type(T_ULONG);
#endif
		break;
	case BSWP:			/* 03'09 byte swap */
		mustty(lp, R_INTEGRAL);
		tp = basic_type(widen(lp->type->token));
		cast_up(lp, tp, "Cast bswap");
		break;
	case HALT:			/* 03'09 stop */
		mustty(lp, R_INTEGRAL);
		if (lp->token ne ICON)
			error("stop needs integer constant");
		tp = basic_type(T_VOID);
		break;
	case SETSR:			/* 03'09 move to   SR */
		tp = basic_type(T_VOID);
		/* fall thru */
	case GETSR:			/* 03.09 move from SR */
		mustty(lp, R_INTEGRAL);
		break;
	case GETSETSR:		/* 03'09 move from & move to SR */
		mustty(lp, R_INTEGRAL);
		tp = basic_type(T_SHORT);
		break;
#if BIP_ASM
	case REGINDIRECT:		/* 11'09 HR forgot these */
	case REGINDPLUS:
	break;
#endif
	default:
		CE_X("unknown unary operator '%s'", np->name);
		tp = default_type(-1, 0);
	}

	np->type = tp;
	to_nct(np);
}

static
bool must_soft(NP np, TP tp, short op)
{
#if FLOAT
	if (tp->ty ne ET_R)
#endif
	{
		if (    tp->size eq SIZE_L
#if FLOAT || COLDFIRE
		    and !(G.i2_68020 or G.Coldfire)
#endif
		   )
		{
			np->type = tp;
			to_nct(np);

			/* extracodes: now we know left & right types for external defined binary operators */

			if (op eq (np->token&TOKMASK))	/* if not transformed */
			{
				bool bex = external_binary_op(np);
				if (!bex)
					error("long '%s' and no function defined for it in 'ahcc_rt.h'", graphic[op]);
			}
			return true;
		}
	}

	return false;
}

#if COLDFIRE
global
void cold_con(NP np, TP tp)
{
	if (G.Coldfire and tp->size eq SIZE_L)
	{
		NP lp = np->left, rp = np->right;
		if (is_icon(lp->token) and lp->token eq np->token)	/* both icons (confold) */
			return;
		if (is_icon(lp->token))
			lp->token = COLDCON;		/* to be saved in l_eval */
		if (is_con(rp->token))
			rp->token = COLDCON;
	}
}
#endif

static
bool can_shift(NP rp, TP tp, TP ltp)
{
	if (    is_icon(rp->token)
#if FLOAT
		and tp->ty ne ET_R
#endif
#if COLDFIRE
		and !(G.Coldfire and ltp->size < DOT_L)		/* 09'10 HR: must check left size */
#endif
		and is2pow(&rp->val.i)
	   )
		return true;

	return false;
}

static
void can_moveq(NP np)
{
	NP rp = np->right,
	   lp = np->left;

	if (rp->token eq ICON)
		/* simple and small optimization (use moveq when possible) */
		if (    is_C_A(np->token)
		    and is_signed_byte(rp->val.i)	/* moveq is sign extending!! */
		   )
		{
			message("%ld --> right size %ld", rp->val.i, np->type->size);
			np->left = rp;
			np->right = lp;
		}
}

global void binary_types(NP np, short context, short usage)
{
	TP tp, ltp, rtp;
	NP lp, rp;
	short op;
	bool assignop, pow2;

	D_D(_x, send_msg("binary_types:%s,%s\t", np->name, prcntxt(context));)

	op = np->token&TOKMASK;
	assignop = isassign(np->token);

	if (op ne X_ELSE)		/* confold integrated */
	{
		if (assignop)
			mustlval(np->left);
/* 10'10 HR xcanon only correct for X_THEN
			symmetric function ycanon for 2 icon's
			other combinations were already handled correctly.
*/
		if (op eq X_THEN and np->left->token eq ICON)
			xcanon(np);
		elif (    (op eq AND or op eq OR)
			  and (   np->left ->token eq ICON
		           or np->right->token eq ICON
		          )
		     )
		{
			ycanon(np);
			return;
		}
		else
		{
			bcanon(np);
			if (is_C_A(np->token))
			#if ICONRIGHT
				if (!right_assoc(np))	/* 2 versions of old b_assoc (for right and left operator chains) */
					left_assoc(np);
			#else
				if (!left_assoc(np))
					right_assoc(np);
			#endif
		}

		if (np->tt ne E_BIN)	/* must have done something */
			return;
	}

	lp = np->left;
	rp = np->right;
	ltp = lp->type;
	rtp = rp->type;

	if (lp->cflgs.f.undef)		/* reducing err msgs */
		lp->type = rtp;
	if (rp->cflgs.f.undef)
		rp->type = ltp;

	tp = default_type(-1, 0);

	if (assignop)
		const_check(lp, nil, context);

	switch (op)
	{
	case TIMES:
#if BIP_ASM
		if (G.for_S and (lp->category & ASMREG) ne 0)
		{
			np->token = ASMSCALE;
			tp = lp->type;
			break;
		}

		/* else fall thru */
#endif
	case DIV:
		if (   mustty(lp, R_ARITH)
		    or mustty(rp, R_ARITH) ) break;
		tp = normalty(np, 0);
		if (is_useless(np, 1))
			return;				/* removes ICON so no bcanon needed */

/* If target is ST, conversion to shift must be done before soft long mul/div */
#if TWOPOW
		pow2 = can_shift(rp, tp, ltp);				/* 09'10 HR must check ltp for Coldfire !!! */
		if (pow2)
		{								/* is2pow doesnt change off if false */
			np->token &= (ASSIGN 0);
			np->token |= op eq TIMES ? LSHIFT : RSHIFT;

			if (!assignop)
				castdefault(lp, rp, tp);
			else
			{
				castasmagic(lp, rp);
				tp = hardasmagic(np, ltp, 0);
			}
		}
		else
#endif
		{
			tp = basic_type(widen(tp->token));		/* widen alleen voor mul,div,mod */
			if (!assignop)
				castdefault(lp, rp, tp);
			if (assignop)
			{
				tp = ltp;
				castasop(lp, rp);
				tp = hardas(np, tp);
			}

			if (must_soft(np, tp, op))
				return;

#if COLDFIRE
			cold_con(np, tp);
#endif

		}

		break;
	case MOD:
		if (   mustty(lp, R_ARITH)
		    or mustty(rp, R_ARITH) ) break;
		tp = normalty(np, 0);
		if (is_useless(np, 1))
			return;				/* removes ICON so no bcanon needed */

		tp = basic_type(widen(tp->token));		/* widen alleen voor mul,div,mod */
		if (!assignop)
			castdefault(lp, rp, tp);
		if (assignop)
		{
			tp = ltp;
			castasop(lp, rp);
			tp = hardas(np, tp);
		}

		if (must_soft(np, tp, op))
			return;

#if COLDFIRE
		cold_con(np, tp);
#endif
		break;
	case AND:
	case OR:			/* recursive call of  binary_types :: no more fall thru */
#if FOR_A
		if (    G.for_A
		    and !is_CC(lp) and ltp->token eq T_BOOL
		    and !is_CC(rp) and rtp->token eq T_BOOL)
		{
			np->token = np->token eq AND ? BINAND : BINOR;
			/* both boolean variables: fall through to BIN op's !!This is a concession.
			   In fact you must abolish && and || from the language,
			   and always use & and | for both boolean and binary operation */
			binary_types(np, FORSIDE, usage);	/* process the changed and/or */
			return;
		}
#endif
		mustty(lp, R_CC|R_SCALAR);
		mustty(rp, R_CC|R_SCALAR);
		if (!is_CC(lp))			/* only if types are different */
			cmp_zero(lp);
		if (!is_CC(rp))			/*   "     */
			cmp_zero(rp);
		np->type = CC_type(np->left, np->right);		/* makes special type ET_?C */
		not_nct(np);
		external_binary_op(np);
		return;
	case BINAND:
	case BINOR:
	case BINEOR:
		if (   mustty(lp, R_BIN)		/* true boolean: R_BIN concession to C */
		    or mustty(rp, R_BIN) ) break;
		if (op ne BINAND and is_useless(np, 0))
			return;
		tp = bitsty(np, G.Coldfire);

		if (!assignop)
			castdefault(lp, rp, tp);
		else
		{
			castasmagic(lp, rp);
			tp = hardasmagic(np, ltp, G.Coldfire);
		}
		break;
	case MINUS:
#if BIP_ASM					/* difference between ID's in same area */
		if (	G.for_S
			and lp->token eq ID and rp->token eq ID
		    /* and lp->area_info.id eq rp->area_info.id */
		   )
		{
/*			message("ID-ID: l id%d + %ld, r id%d + %ld :: %ld",
				lp->area_info.id,
				lp->area_info.disp,
				rp->area_info.id,
				rp->area_info.disp,
				lp->area_info.disp - rp->area_info.disp        );
*/
			if (!(lp->sc eq K_AHEAD or rp->sc eq K_AHEAD) )
			{
				if (lp->area_info.id ne rp->area_info.id)
					error("labels must be in same area");
				else
				{
					np->token = ICON;
					np->val.i = lp->area_info.disp - rp->area_info.disp;
					np->tt = E_LEAF;
					freenode(lp);
					freenode(rp);
					np->left = nil;
					np->right = nil;
				}
			}
			tp = basic_type(T_LONG);
			break;
		}
		/* else fall thru */
#endif
	case INDEX:
	case PLUS:
		if (   mustty(lp, R_SCALAR)
			or mustty(rp, R_SCALAR) ) break;
		if (is_useless(np, 0))
			return;
		if (ltp->token eq REFTO or rtp->token eq REFTO)
		{
			if (op eq PLUS)
				np->token = (np->token & ~TOKMASK) | INDEX;	/* ptr + i */

			tp = scalety(np);	/* may change lp &| rp (former addty() */
			lp = np->left;
			rp = np->right;
		othw
			tp = normalty(np, G.Coldfire);
		}

		if (assignop)
		{
			castasmagic(lp, rp);
			tp = hardasmagic(np, ltp, G.Coldfire);
		}
		elif (!(ltp->token eq REFTO or rtp->token eq REFTO))
			castdefault(lp, rp, tp);
	/*	else
		    casts already done in scalety */

		break;
	case LSHIFT:		/* shift count(right) is used module 64 so can be any integral type */
	case RSHIFT:
		if (   mustty(lp, R_INTEGRAL)
			or mustty(rp, R_INTEGRAL) ) break;
		if (is_useless(np, 0))
			return;
#if 0
		if (lp->token eq ICON and ltp->size eq DOT_B)	/* 1 << n */
			ltp = basic_type(widen(ltp->token));		/* most compilers do this */

/* ANSI: the type of the shiftcount does not influence the type of the result.
   A problem with shift on 68000 is that
    lsl #n,Dy		shift count is 3 bits	treated as unsigned (with a special treatment for 0)
	lsl Dx,Dy		shift count is 6 bits 	Dx modulo 64 	The manual is not clear about a sign
		So the best for the right part is do nothing, because even casting from byte to int doesnt
		do any good;
	In that case 'a<<-3' or 'a<<325' must be considered unpredictable
*/
#endif

/* cast. (for practical reasons (and compatability with Pure C.) ) */
		tp = shiftty(np, G.Coldfire);

		if (!assignop)					/* 03'02 HR */
			castdefault(lp, rp, tp);
		else							/* 03'02 HR */
		{
			castasmagic(lp, rp);
			tp = hardasmagic(np, ltp, G.Coldfire);
		}
		break;
	case LESS:
	case GREATER:
	case LTEQ:
	case GTEQ:
		if (   mustty(lp, R_SCALAR)
			or mustty(rp, R_SCALAR) ) break;
		chkcmp(np);
		if (is_zero(rp))
			ins_zero(rp);
		else
			cast_compare(lp, rp);

		np->type = CC_type(np->left, np->right);
		not_nct(np);
		external_binary_op(np);
		return;
	case EQUALS:
	case NOTEQ:
		if (   mustty(lp, R_CC|R_SCALAR)
			or mustty(rp, R_CC|R_SCALAR) ) break;
		chkcmp(np);
		if (is_zero(rp))
			ins_zero(rp);
		else
			cast_compare(lp, rp);
		np->type = CC_type(np->left, nil);
		not_nct(np);
		external_binary_op(np);
		return;
	case X_THEN:
		mustty(lp, AC_BOOL);
		tp = rp->type;
		if (!is_CC(lp))
			cmp_zero(lp);
		break;
	case X_ELSE:
		if (   mustty(lp, R_CC|R_ASSN)
			or mustty(rp, R_CC|R_ASSN) ) break;
		tp = colonty(np);
		if (!is_aggreg(tp))
			castdefault(lp, rp, tp);
		break;
	case ASSIGNMT:
		if (   mustlval(lp)
			or mustty  (lp, R_ASSN) ) break;		/* use R_STRUCT for arrays */
		asn_check (ltp, rp, context);
		const_check(lp, rp, context);
		tp = ltp;
		castasn(lp, rp);
		fieldas(np, lp);
		break;
	case PULL:		/* pull argregs from the stack */
	case ARG:		/* does arrive here only if G.h_cdecl_calling (else CE ) */
	case ARGP:
	case COMMA:
		tp = rtp;
		break;
	default:
		CE_X("unknown binary operator '%s'", np->name);
	}

	np->type = tp;
	to_nct(np);

#if 0
	#if BIP_ASM
		if (!G.for_S)
	#endif
			if (!isassign(np->token) and np->token ne TIMES)
				can_moveq(np);
#endif

	external_binary_op(np);
}

global void call_types(NP np, short context)	/* args in list */
{
	NP rp = np->right;

	D_(_x, "call_types");

	if (rp)				 /* voor args */
	{
		while (rp)
		{
			form_types(rp->left, context, 0);
			rp->type = rp->left->type;
			to_nct(rp);
			rp = rp->right;
		}
		np->tt = E_BIN;
	}
	else
		np->tt = E_UNARY;

	D_1(_x, "after call_types", np);
}

static void asmsize_type(NP np)		/* only called for operands */
{
	TP tp;
	NP lp = np->left, rp = np->right;

D_(_x, "asmsize_type");

	tp = lp->type;

	if (rp->token eq ID)
	{
		short c = tolower(*rp->name);

		if (    *(rp->name + 1) eq 0
			and (c eq 'w' or c eq 'l')
		   )
		{
			*np = *lp;
			freenode(rp);
			freenode(lp);
			np->type = tp;
			to_nct(np);
			if (c eq 'w')
				np->eflgs.f.asm_w = 1;
			elif (c eq 'l')
				np->eflgs.f.asm_l = 1;
D_1(_x, "after asmsize_type", np);
			return;
		}
	}

	error("asm size syntax error");
	np->type = tp;
	to_nct(np);
}

global void form_types(NP np, short context, short usage)
{
	if (np eq nil) return;

D_1(_x, "form_types", np);

	switch (np->tt)
	{
	case E_SPEC:
		D_D(_x, send_msg("spec_types:%s,%s\t", np->name, prcntxt(context));)

		switch (np->token)		/* special cases */
		{
#if BIP_ASM
		case ASM_SELECT:
			form_types(np->left, context, usage);
			member_type(np);
			return;
#endif
		case SELECT:
			form_types(np->left, context, usage);
#if BIP_ASM
			if (G.for_S)				/* id.w or id.l */
				asmsize_type(np);
			else
#endif
				member_type(np);
			return;
		case CALL:
			call_types(np, context);
			function_type(np, context);
			match_args(np);			/* also do argassigns if proc is local */
			arg_regs(np);
			return;
		}
/*	case E_SPEC fall through (be sure they are in fact binary) */
	case E_BIN:
/*		messagen(np->left, "asmt usage %s", prcntxt(context));
		if (np->token eq ASSIGNMT)
			form_types(np->left, context, FORLVAL);		/* 12'08 HR usage tracking */
		else
*/			form_types(np->left,  context, usage);
		form_types(np->right, context, usage);
		binary_types(np,      context, usage);
		break;

	case E_UNARY:
/*		messagen(np->left, "unary usage %s", prcntxt(context));
		if (usage eq FORLVAL and context eq FORLINIT)			/* 12'08 HR usage tracking */
			form_types(np->left,  context, 0);
		else
*/			form_types(np->left,  context, usage);
		unary_types(np);
		break;

	case E_LEAF:
		leaf_types(np, context, usage);
		break;
	}
}

static void gen_expr(NP np, short context)
{
	untype(np);
	branch_tree(np);
	genx(np, context);
}

/* general expression */
global void do_expr(NP np, short context)
{
	D_D(_x, (send_msg("do_expr: %s\t", prcntxt(context)), printnode(np));)

	G.prtab->tmps = 0;
/*	'and !G.nmerrors' if want only one error per expression	*/
	if (np and !G.nmerrors)
	{
		form_types(np, context, 0);
#if 0 /* COLDFIRE 03'11 HR: not a good idee. */
		if (    G.Coldfire
		    and context eq SWITCH
		    and np->size < SIZE_L
		   )
		{
		   cast_up(np, basic_type(T_USHORT), "castu switch");
		   cast_up(np, basic_type(T_ULONG ), "castl switch");
		}
#endif
		if (!G.nmerrors)
			gen_expr(np, context);
	}
}

/* expression returning struct or union */
static void str_expr(NP np, TP typ)
{
	D_D(_x, (send_msg("str_expr:\t"), printnode(np));)

	G.prtab->tmps = 0;
	if (np and !G.nmerrors)
	{
		form_types(np, RETSTRU, 0);
		if (!G.nmerrors and similar_type(0,1,np->type, typ, 0) eq 0)
			error("bad %s return", graphic[np->type->token]);
		if (!G.nmerrors)
			gen_expr(np, RETSTRU);
	}
}

static short ret_context(TP typ)
{
#if FLOAT
	if (   (    !G.use_FPU
	        and typ->token eq T_REAL
	       )
	    or typ->ty eq ET_A
	   )
		return RETSTRU;
	else
#endif
	if (    typ->token eq REFTO
/*		and typ->cflgs.f.cdec eq 0		/* 09'10 HR */
*/	   )
		return INA0;
	else
		return IND0;		/* implies INF0 */
}

global void ret_expr(NP np, TP typ)
{
	if (is_aggreg(typ))
	{
		D_(_x, "ret_stru");
		str_expr(np, typ);
	othw
		Cast(np, typ, IMPCNV, "Ret_cast");
		do_expr(np, ret_context(typ));
	}
}

global bool coercable(TP prt, TP tp)
{
	if (    (prt->ty eq ET_S or prt->ty eq ET_U)
		and ( tp->ty eq ET_S or  tp->ty eq ET_U)
		)
		return true;	/* both integral types */
	if (prt->ty eq ET_P and tp->ty eq ET_P)
		if (   (prt->type and prt->type->token eq T_VOID)
			or ( tp->type and  tp->type->token eq T_VOID)
		   )
			return true;
	if (similar_type(0,0,prt, tp, 0))
		return true;
	return false;
}

#if FOR_A
#include "a_expr.h"
#endif