/* Copyright (c) 1988,1989 by Sozobon, Limited.  Author: Johann Ruegg
 *           (c) 1990 - 2009 by H. Robbers.   ANSI upgrade.
 *
 * This file is part of AHCC.
 *
 * AHCC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * AHCC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AHCC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *	nodes.c
 *
 *	Node allocation, deallocation, searching
 *	and other node handling
 */

#define PROCESS_H
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>

#include "lmem.h"
#include "param.h"
#include "shell/xref.h"

#define debugL (G.xflags['l'-'a'])
#define debugU (G.xflags['u'-'a'])

#define CHKFREE 1		/* set independent from C_DEBUG: check 'freeing free/bas node' */
#define CHKNODE 0		/* set independent from C_DEBUG: check node type */

global short nodesmade = 0, nodesavail = 0;

#define zeroize(xp) *((long*)xp)++ = 0

#include "zernodes.h"

NODESPACE initspace[XXr] =
{
	{sizeof(XNODE), 100, true, 3, 2, zerXnode, "Preprocessor"},
	{sizeof(TNODE), 200, true, 3, 2, zerTnode, "Typelist    "},
	{sizeof( NODE),  75, true, 3, 2, zerEnode, "Expression  "},
	{sizeof(VNODE),  25, true, 3, 2, zerVnode, "Various     "},
	{sizeof(INODE), 100, true, 3, 2, zerInode, "Instruction "},
	{sizeof(BNODE),  25, true, 3, 2, zerBnode, "Block       "}
};

NODESPACE xspace[XXr];

#define BLKMAX (32768-blockprefix-unitprefix)

#if NODESTATS
void nodecnts(void)
{
	console("\nxnodes copied: %ld\n\n", G.xcopied);
	console("usage:\n");
	console("STNODES: %ld\n", G.ncnt[STNODE]);
	console("DFNODES: %ld\n", G.ncnt[DFNODE]);
	console("TLNODES: %ld\n", G.ncnt[TLNODE]);
	console("EXNODES: %ld\n", G.ncnt[EXNODE]);
	console("GENODES: %ld\n", G.ncnt[GENODE]);
	console("BLNODES: %ld\n", G.ncnt[BLNODE]);
	console("FLNODES: %ld\n", G.ncnt[FLNODE]);
	console("CSNODES: %ld\n", G.ncnt[CSNODE]);
	console("TPNODES: %ld\n", G.ncnt[TPNODE]);
	console("PRNODES: %ld\n", G.ncnt[PRNODE]);
	console("RLNODES: %ld\n", G.ncnt[RLNODE]);
	console("LLNODES: %ld\n", G.ncnt[LLNODE]);
	console("LBNODES: %ld\n", G.ncnt[LBNODE]);
	console("IFNODES: %ld\n", G.ncnt[IFNODE]);
	console("INNODES: %ld\n", G.ncnt[INNODE]);
	console("BKNODES: %ld\n", G.ncnt[BKNODE]);

	console("        -----\n");
	console("NODES  : %ld\n\n",
	         G.ncnt[0]
	        +G.ncnt[STNODE]
	        +G.ncnt[EXNODE]
			+G.ncnt[GENODE]+G.ncnt[LBNODE]
			+G.ncnt[BLNODE]+G.ncnt[FLNODE]
			+G.ncnt[CSNODE]+G.ncnt[TPNODE]
			+G.ncnt[PRNODE]+G.ncnt[RLNODE]+G.ncnt[LLNODE]
			+G.ncnt[IFNODE]+G.ncnt[INNODE]+G.ncnt[BKNODE]);
	console("match   ops: %ld\n", G.operators);
	console("id's       : %ld\n", G.dictionary_names);
	console("heap names : %ld\n", G.heap_names);
	console("symbols    : %ld\n", G.symbols);
	console("defines    : %ld\n", G.defs);
	console("strcode    : %ld\n", G.strcode);
	console("npcode     : %ld\n", G.npcode);
}
#endif

static
bool bas_or_free(void *vp)
{
	NP np = vp;
	return np->nflags.f.bas ne 0 or np->nflags.f.free ne 0;
}

global void name_to_str(void *vp, string s)
{
	NP np = vp;

	free_name(np);
	np->name = s;
}

global void free_name(void *vp)			/* already made secure */
{
	NP np = vp;
	if (np->nflags.f.nheap eq NHEAP)
	{
		CC_xfree(np->name, 240);
		np->name = "__freed__";
		np->nflags.f.nheap = NCOPY;
	}
}

void (*BOM)(void) = nil;

static
void *noderow(NODESPACE *sp, void *np, short i)
{
	size_t s = sp->nodesize;
	char *xp = np;
	do
	{
		xp += s;					/* first node is used for space chain */
		((XP)xp)->nflags.f.free = 1;
		(char *)(((XP)xp)->next) = xp+s;
	} while (--i);
	((XP)xp)->next = nil;
	return (char *)np+s;
}

VpI		waitexit;

static
void new_space(NODESPACE *sp, short ty)
{
	XP np;
	long size = sp->start;

#if C_DEBUG
	np = NS_xcalloc(1, size * sp->nodesize, AH_TEST_SPACE + ty, CC_ranout);
#else
	np = NS_xmalloc(   size * sp->nodesize, AH_TEST_SPACE + ty, CC_ranout);
#endif
	if (np)
	{
		np->val.i = size;
	
#if NODESTATS
		if (G.al_list_stats)
			console("new %s node space(%ld): %ldK\n", sp->name, sp->nodesize, (size*sp->nodesize+512)/1024);
#endif
		sp->start = sp->exponential ? (size*sp->tel)/sp->noem : size;
		if (sp->start*sp->nodesize >= BLKMAX)
			sp->start = BLKMAX/sp->nodesize;
#if NODESTATS
		sp->made += size-1;
		sp->avail += size-1;
#endif
		np->next  = sp->space;		/* chain allocated blocks for later free() */
		sp->space = np;
		sp->list  = noderow(sp, np, size-1);
	}
}

#if C_DEBUG
bool check_node(void *vp, short ty)
{
	NP np = vp;
	if (np->nflags.f.bas)
	{ warn("CW: free%cunit() on resident node %s%lx", ty, pntype(np->nt), np);	/* extra check */
		return true; }
	if (np->nflags.f.free)
	{ error("CE: free%cunit() on free node %s%lx", ty, pntype(np->nt), np);
		return true; }
	return false;		/* no error */
}

bool check_free(void *vp, short ty,
		short t1, short t2, short t3, short t4, short t5, short t6, short t7, short t8)
{
	NP np = vp;
	if (np->nflags.f.bas)
	{
		warnn(np, "CW: free%cnode on resident node %s%lx", ty, pntype(np->nt), np);	/* extra check */
		return true;
	}
	if (np->nflags.f.free)
	{
#if BIP_CC
		warn("CW: freeing free %cnode, %lx %s file %s line %ld", ty,
			np, ptok(np->token), xref_file_by_number(auto_dependencies, np->fileno), np->lineno);
#else
		warn("CW: freeing free %cnode, %lx %s fileno %d line %ld", ty,
			np, ptok(np->token), np->fileno, np->lineno);
#endif
		printnode(np);
		if (    np->nt ne t1 and np->nt ne t2 and np->nt ne t3 and np->nt ne t4
		    and np->nt ne t5 and np->nt ne t6 and np->nt ne t7 and np->nt ne t8)
			warn("CE: no %cnode: node type: '%s'", ty, pntype(np->nt));
		return true;
	}
	if (    np->nt ne t1 and np->nt ne t2 and np->nt ne t3 and np->nt ne t4
	    and np->nt ne t5 and np->nt ne t6 and np->nt ne t7 and np->nt ne t8)
	{
#if BIP_CC
		warn("CW: free%cnode on no %cnode [%s]%d %s, file %s line %ld", ty, ty,
			pntype(np->nt), np->nt, ptok(np->token), xref_file_by_number(auto_dependencies, np->fileno), np->lineno);
#else
		warn("CW: free%cnode on no %cnode [%s]%d %s, fileno %d line %ld", ty, ty,
			pntype(np->nt), np->nt, ptok(np->token), np->fileno, np->lineno);
#endif
		if (np->token eq ID)	send_msg("\t\t%s\n", np->name);
		return true;
	}

	return false;
}
#else
bool check_node(void *vp, short ty)
{
	if (bas_or_free(vp))
		return true;
	return false;			/* no error */
}

bool check_free(void *vp, short ty,
		short t1, short t2, short t3, short t4, short t5, short t6, short t7, short t8)
{
	NP np = vp;
	if (bas_or_free(np)) return true;
	if (    np->nt ne t1 and np->nt ne t2 and np->nt ne t3 and np->nt ne t4
	    and np->nt ne t5 and np->nt ne t6 and np->nt ne t7 and np->nt ne t8)
		return true;
	return false;
}
#endif

global XP allocXn(bool clr)
{
	NODESPACE *sp = &xspace[XNr];
	XP np;

	if (!sp->list)
		new_space(sp, XNr);
	np = sp->list;			/* The allocation proper */
	sp->list = np->next;
	if (clr)
		(*sp->zer)(np);
	np->nt = STNODE;

#if NODESTATS
	sp->avail--;
	G.ncnt[STNODE]++;
#endif

	line_number(np);

	D_D(L, send_msg("+%s%lx\t", pntype(np->nt), np);)
	return np;
}

global void freeXunit(XP np)
{
	NODESPACE *sp = &xspace[XNr];
#if CHKNODE
	if (check_node(np, 'X'))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->nflags.f.free = 1;
	np->nt = 0;
	np->next = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
	D_D(L, send_msg("%s%lx-\t", pntype(np->nt), np);)
}

global void freeXn(XP np)
{
	while (np)
	{
		XP nxt;
#if CHKFREE
		if (check_free(np, 'X', STNODE, DFNODE, DFNODE, DFNODE, DFNODE, DFNODE, DFNODE, DFNODE))
			return;
#endif
		nxt = np->next;
		if (np->right)
			freeXn(np->right);
		if (np->tseq and !np->nflags.f.n_ct)
			freeXn(np->tseq);
#if FLOAT
		if (np->token eq RCON)
			freeVunit(np->val.dbl);
#endif
#if LONGLONG
		if (np->token eq LCON)
			freeVunit(np->val.dbl);
#endif
		freeXunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

global XP copyXone(XP np)
{
	XP nx = allocXn(0);		/* no clear */
	*nx = *np;
	nx->nflags.f.nheap = NCOPY;
	nx->right = nil; 	/* break right links */
#if FLOAT
	if (np->token eq RCON)
		nx->val.dbl = copyVone(np->val.dbl);
#endif
#if LONGLONG
	if (np->token eq LCON)
		nx->val.dbl = copyVone(np->val.dbl);
#endif
	line_number(nx);

	return nx;
}


global TP allocTn(bool clr)
{
	NODESPACE *sp = &xspace[TLr];
	TP np;

	if (!sp->list)
		new_space(sp, TLr);
	np = sp->list;			/* The allocation proper */
	sp->list = np->next;
	if (clr)
		(*sp->zer)(np);
	np->nt = TLNODE;
	np->rno = -1;

#if NODESTATS
	sp->avail--;
	G.ncnt[TLNODE]++;
#endif
	line_number(np);
	return np;
}

global void freeTunit(TP np)
{
	NODESPACE *sp = &xspace[TLr];
#if CHKNODE
	if (check_node(np, 'T'))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->nflags.f.free = 1;
	np->next = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
}

global void freeTn(TP np)
{
	while (np)
	{
		TP nxt;
#if CHKFREE
		if (check_free(np, 'T', TLNODE, TLNODE, TLNODE, TLNODE, TLNODE, TLNODE, TLNODE, TLNODE))
			return;
#endif
		nxt = np->next;
		if (np->list)
			freeTn(np->list);
		if (np->type and !is_nct(np))
			freeTn(np->type);
		freeTunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

global BP allocBn(void)
{
	NODESPACE *sp = &xspace[BKr];
	BP np;

	if (!sp->list)
		new_space(sp, BKr);
	np = sp->list;			/* The allocation proper */
	sp->list = np->next;
	(*sp->zer)(np);
	np->nt = BKNODE;

#if NODESTATS
	sp->avail--;
	G.ncnt[BKNODE]++;
#endif
	return np;
}

global void freeBunit(BP np)
{
	NODESPACE *sp = &xspace[BKr];
#if CHKNODE
	if (check_node(np, 'B'))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->nflags.f.free = 1;
	np->next = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
}

global void freeBn(BP np)
{
	while (np)
	{
		BP nxt;
#if CHKFREE
		if (check_free(np, 'B', BLNODE, BLNODE, BLNODE, BLNODE, BLNODE, BLNODE, BLNODE, BLNODE))
			return;
#endif
		nxt = np->next;
		if (np->chain)
			freeBn(np->chain);
		if (np->symbol and !is_nct(np))
			freeBn(np->symbol);
		freeBunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

global IP allocIn(void)
{
	NODESPACE *sp = &xspace[INr];
	IP np;

	if (!sp->list)
		new_space(sp, INr);	/* N.B !! INr only used in console msg */
	np = sp->list;			/* The allocation proper */
	sp->list = np->next;
	(*sp->zer)(np);
	np->nt = INNODE;		/* 03'11 HR: back in again; needed for debugging */

#if NODESTATS
	sp->avail--;
	G.ncnt[INNODE]++;
#endif
	return np;
}

global void freeIunit(IP np)
{
	NODESPACE *sp = &xspace[INr];
#if CHKNODE
	if (check_node(np, 'I'))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->arg = nil;
	np->nflags.f.free = 1;
	np->next = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
}

global void freeIn(IP np)
{
	while (np)
	{
		IP nxt;
#if CHKFREE
		if (check_free(np, 'I', INNODE, INNODE, INNODE, INNODE, INNODE, INNODE, INNODE, INNODE))
			return;
#endif
		nxt = np->next;
#if OPTBUG
		if (np->bugstr)
			CC_xfree(np->bugstr, 241);
#endif
		freeIunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

#if LOST_NODES
static
void lost_xnodes(void)
{
	if debugU
	{
		long i;
		NODESPACE *sp = &xspace[XNr];
		XP xp = sp->space;
		while(xp)
		{
			XP nx = xp->next;
			long s = xp->val.i;
			for (i = 1; i<s; i++)
				if (xp[i].nflags.f.free eq 0)
				{
					XP np = &xp[i];
					np->right = 0;
					if (np->token eq REFTO and np->next)
						np->next->next = 0;
					else
						np->next = 0;
					np->tseq = 0;
					console("**** lost Xnode %s ****\n", ptok(np->token));
	/*				printnode(np);		*/
	/*				pm_print(&xp[i]);	*/
				}
			xp = nx;
		}
	}
}

static
void lost_vnodes(void)
{
	if debugU
	{
		long i;
		NODESPACE *sp = &xspace[VNr];
		VP xp = sp->space;
		while(xp)
		{
			VP nx = xp->next;
			long s = xp->vval;
			for (i = 1; i<s; i++)
				if (xp[i].nflags.f.free eq 0)
				{
					VP np = &xp[i];
					np->inner = 0;
					if (np->token eq REFTO and np->next)
						np->next->next = 0;
					else
						np->next = 0;
					np->codep = 0;
					console("**** lost Vnode %s ****\n", ptok(np->token));
	/*				printnode(np);		*/
	/*				pm_print(&xp[i]);	*/
				}
			xp = nx;
		}
	}
}

static
void lost_tnodes(void)
{
	if debugU
	{
		long i;
		NODESPACE *sp = &xspace[TLr];
		TP xp = sp->space;
		while(xp)
		{
			TP nx = xp->next;
			long s = xp->offset;
			for (i = 1; i<s; i++)
				if (xp[i].nflags.f.free eq 0)
				{
					TP np = &xp[i];
					np->list = 0;
					if (np->token eq REFTO and np->next)
						np->next->next = 0;
					else
						np->next = 0;
					np->type = 0;
					console("**** lost Tnode %s ****\n", ptok(np->token));
	/*				printnode(np);		*/
	/*				pm_print(&xp[i]);	*/
				}
			xp = nx;
		}
	}
}

static
void lost_nodes(void)
{
	if debugU
	{
		long i;
		NODESPACE *sp = &xspace[EXr];
		NP xp = sp->space;
		while(xp)
		{
			NP nx = xp->left;
			long s = xp->val.i;
			for (i = 1; i<s; i++)
				if (xp[i].nflags.f.free eq 0)
				{
					NP np = &xp[i];
					np->right = 0;
					if (np->token eq REFTO and np->left)
						np->left->left = 0;
					else
						np->left = 0;
					np->type = 0;
					console("**** lost G/E node %s ****\n", ptok(np->token));
	/*				printnode(np);		*/
	/*				pm_print(&xp[i]);	*/
				}
			xp = nx;
		}
	}
}
#endif

global void freenodespace(short ty)
{
	NODESPACE *sp = &xspace[ty];

#if NODESTATS
	if (G.al_list_stats)
		console("%s node space (%ld) = %ldK\n", sp->name,
				sp->nodesize, (((long)sp->avail*sp->nodesize)+1023)/1024);
	if (sp->made ne sp->avail)
	{
		short n = sp->made - sp->avail;
		if (G.al_list_stats)
			console("made : %d, available: %d\n", sp->made, sp->avail);
		console("lost %d %s node%s(%ld)!!!\n", n, sp->name, pluralis(abs(n)), sp->nodesize);
	#if LOST_NODES
		if debugU
			if (n > 0)
			{
				if   (ty eq XNr) lost_xnodes();
				elif (ty eq TLr) lost_tnodes();
				elif (ty eq VNr) lost_vnodes();
				elif (ty eq EXr) lost_nodes();
			}
	#endif
	}
#endif

	while(sp->space)
	{
		XP xp = sp->space;
		xp = xp->next;
		NS_xfree(sp->space, 242);
		sp->space = xp;
	}
	sp->list = nil;
}

global NP allocnode(short nt)
{
	NODESPACE *sp = &xspace[EXr];
	NP np;

	if (!sp->list)
		new_space(sp, EXr);

	np = sp->list;			/* The allocation proper */
	sp->list = np->left;
	if (nt >= 0)			/* -1 if contents are copied anyway */
	{
		(*sp->zer)(np);
		np->nt = nt;
		np->rno = -1;
	}

#if NODESTATS
	sp->avail--;
	if (nt < 0)
		nt = 0;
	G.ncnt[nt]++;
#endif
	line_number(np);

	D_D(L, send_msg("+%s%lx\t", pntype(nt), np);)
	return np;
}

global
NP gx_node(void)
{
	NP gp = allocnode(GENODE);
	return gp;
}

global void freeunit(NP np)
{
	NODESPACE *sp = &xspace[EXr];
#if CHKNODE
	if (check_node(np, ' '))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->nflags.f.free = 1;
	np->left = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
	D_D(L, send_msg("%s%lx-\t", pntype(np->nt), np);)
}

global void new_gp(NP ex, short tok)
{
	gpbase = gp = gx_node();
	gp->token = tok;
	gp->right = ex;
}

global void expr_gp(NP ex, short tok)
{
	gp->left = gx_node();
	gp = gp->left;
	gp->token = tok;
	gp->right = ex;
}

global void next_gp(NP np)
{
	gp->left = np ? np : gx_node();
	gp = gp->left;
}

global void save_name(void *vp, long l, string s)
{
	NP np = vp;

	np->nflags.f.nheap = NCOPY;

	if (np->token eq ID)
		np->name = to_dictionary(s);
	else
		np->name = to_name_heap(l, s);
}

global short new_name(void *vp, string text, ... )
{
	NP np = vp;
	char tus[256]; short l;
	va_list argpoint;

	free_name(np);

	va_start(argpoint, text);
	l = vsprintf(tus, text, argpoint);
	va_end(argpoint);

	save_name(np, l, tus);
	return l;
}

global void freenode(NP np)
{
	while (np)
	{
		NP nxt;
#if CHKFREE
		if (check_free(np, ' ', EXNODE, GENODE, GENODE, GENODE, GENODE, GENODE, GENODE, GENODE))
			return;
#endif
		nxt = np->left;
		if (np->right)
			freenode(np->right);
		if (np->type and !is_nct(np))
			if (np->nt eq GENODE or np->nt eq TPNODE)
				freeVn((VP)np->type);
			else
				freeTn(np->type);
		if (np->nt eq GENODE)
			if (np->betw)
				freeVn(np->betw);
#if FLOAT
		if (np->token eq RCON)
			freeVn(np->val.dbl);
#endif
#if LONGLONG
		if (np->token eq LCON)
			freeVn(np->val.dbl);
#endif
		freeunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

global VP allocVn(short nt)
{
	NODESPACE *sp = &xspace[VNr];
	VP np;

	if (!sp->list)
		new_space(sp, VNr);

	np = sp->list;			/* The allocation proper */
	sp->list = np->next;
	if (nt >= 0)			/* -1 if contents are copied anyway */
	{
		(*sp->zer)(np);
		np->nt = nt;
	}

#if NODESTATS
	sp->avail--;
	if (nt < 0)
		nt = 0;
	G.ncnt[nt]++;
#endif
	line_number(np);

	D_D(L, send_msg("+%s%lx\t", pntype(nt), np);)
	return np;
}

global void freeVunit(VP np)
{
	NODESPACE *sp = &xspace[VNr];
#if CHKNODE
	if (check_node(np, 'V'))
		return;
#elif CHKFREE
	if (bas_or_free(np))
		return;
#endif
	free_name(np);
	np->nflags.f.free = 1;
	np->next = sp->list;
	sp->list = np;
#if NODESTATS
	sp->avail++;
#endif
	D_D(L, send_msg("%s%lx-\t", pntype(np->nt), np);)
}

global void freeVn(VP np)
{
	while (np)
	{
		VP nxt;
#if CHKFREE
		if (check_free(np, 'V', FLNODE, CSNODE, TPNODE, PRNODE, RLNODE, LLNODE, IFNODE, LBNODE))
			return;
#endif
		nxt = np->next;
		if (np->inner)
			freeVn(np->inner);
		if (np->codep and !is_nct(np))
			if (np->nt eq TPNODE)
				freeVn(np->codep);
			else
				freeTn((TP)np->codep);
		if (np->nt eq FLNODE)
			freeVn(np->F.out);
		freeVunit(np);
		np = nxt;		/* every subnode recursive except left */
	}
}

global NP childname(NP np)
{
	if (np->nt eq GENODE)
		while (np->eflgs.f.lname or np->eflgs.f.rname)
		{
			NP cp = (np->eflgs.f.lname) ? np->left : np->right;
			if (cp eq nil)
				break;
			np = cp;
		}
	return np;
}

global void send_name(void *vp)
{										/* NB: name may contain '%' */
	NP np = vp;
	if (np)
		if (np->name)
#if 1
			console("%s", np->name);
#else
			console("name(%lx,%lx)", np, np->name);
#endif
		else
			console("~~");
}

global void put_fifo(NP *first, NP *last, NP np)
{
	if (*last)
		(*last)->left = np;
	else
		*first = np;
	*last = np;
}

global void put_lifo(TP *head, TP np)
{
	if (*head)
		np->next = *head;
	*head = np;
}

global void putn_lifo(NP *head, NP np)
{
	if (*head)
		np->left = *head;
	*head = np;
}

global void putv_lifo(VP *head, VP np)
{
	if (*head)
		np->next = *head;
	*head = np;
}

global void * init_symtab(short key)
{
	void * al = CC_xcalloc(1, sizeof(void *) * numhash, key, CC_ranout);
	if (al eq nil)
		error("Ran out of memory");
	return al;
}

global void init_hlist(TP list[])
{
	short i;

	for (i = 0; i < numhash; i++)
		list[i] = nil;
}

global void init_xlist(XP list[])
{
	short i;

	for (i = 0; i < numhash; i++)
		list[i] = nil;
}

global void put_hlist(TP *list, TP np)	/* put after *list */
{
	if (*list)
		np->next = *list;
	*list = np;
}

global void put_xlist(XP *list, XP np)	/* put after *list */
{
	if (*list)
		np->next = *list;
	*list = np;
#if NODESTATS
	G.defs++;
#endif
}

global void * tlook(void *vt, void *nv)
{
	AP pt = vt, np = nv;
	while (pt)
	{
		if (pt->name eq np->name)
			return pt;
		pt = pt->left;
	}
	return nil;
}

global NP copynode(NP op)
{
	NP np;

	if (op eq nil) return nil;

	if (op->nflags.f.bas)
		return op;

	np = allocnode(-1);

	D_D(L, send_msg(":=%lx\t", op);)

	*np = *op;

	line_number(np);

#if NODESTATS
	G.ncnt[0]--;
	G.ncnt[np->nt]++;
#endif
	if (np->right)
		np->right = copynode(np->right);
	if (np->left)
		np->left = copynode(np->left);
#if FLOAT
	if (np->token eq RCON)
		np->val.dbl = copyVone(np->val.dbl);
#endif
#if LONGLONG
	if (np->token eq LCON)
		np->val.dbl = copyVone(np->val.dbl);
#endif
	if (np->type)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
	return np;
}

global TP e_copyone_t(NP op)
{
	TP np = allocTn(0);
	*np = *(TP)op;
	line_number(np);
	np->nt = TLNODE;
	if (np->type)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
	return np;
}

global NP t_copyone_e(TP op)
{
	NP np = allocnode(EXNODE);
	*(TP)np = *op;
	line_number(np);
	np->nt = EXNODE;
	np->right = nil;
	np->left = nil;
	if (np->type)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
	return np;
}

global NP t_copy_e(TP op)		/* complete copy */
{
	NP np;

	if (op eq nil) return nil;

	D_B(if (op->nt eq GENODE) {errorn(op, "CE: t_copy_e GENODE");return (NP)op;})
	np = allocnode(EXNODE);

	D_D(L, send_msg(":=%lx\t", op);)

	*(TP)np = *op;		/* enodes are larger */
	np->nt = EXNODE;
	line_number(np);
	if (op->list)
		np->right = t_copy_e(op->list);
	if (op->next)
		np->left = t_copy_e(op->next);
	if (np->type)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
	return np;
}

global NP t_to_e(TP tp)
{
	NP np = allocnode(-1);
	*(TP)np = *tp;
	np->nt = EXNODE;
	line_number(np);
	#if NODESTATS
		G.ncnt[0]--;
		G.ncnt[np->nt]++;
	#endif
	freeTunit(tp);
	return np;
}

global TP copyTnode(TP op)
{
	TP np;

	if (op eq nil) return nil;

	if (op->nflags.f.bas)
		return op;

	np = allocTn(0);

	D_D(L, send_msg(":=%lx\t", op);)

	*np = *op;
	line_number(np);

	if (np->list)
		np->list = copyTnode(np->list);
	if (np->next)
		np->next = copyTnode(np->next);
	if (np->type)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
	return np;
}

global void copyinto(NP ip, NP np)	/* left & right of ip must have been catered for */
{
	D_B(if (np->nt eq GENODE) errorn(np, "CE: copyinto GENODE");)
	if (ip->type and !is_nct(ip))
		freeTn(ip->type);
	free_name(ip);
#if NODESTATS
	G.ncnt[ip->nt]--;
	G.ncnt[np->nt]++;
#endif
	*ip = *np;
	line_number(ip);
}

static
void generic_copyone(void *ng)
{
	AP np = ng;

	line_number(np);
#if NODESTATS
	G.ncnt[0]--;
	G.ncnt[np->nt]++;
#endif
	np->right = nil;
	np->left = nil;
	if (np->nt ne GENODE and np->var.info)
		to_nct(np);
	np->nflags.f.nheap = NCOPY;
}

global NP copyone(NP op)
{
	NP np;

	if (op eq nil) return nil;
	np = allocnode(-1);
	*np = *op;

	D_D(L, send_msg(":=%lx\t", op);)

	generic_copyone(np);

#if FLOAT
	if (np->token eq RCON)
		np->val.dbl = nil;
#endif
#if LONGLONG
	if (np->token eq LCON)
		np->val.dbl = nil;
#endif

	return np;
}

global VP copyVone(VP op)
{
	VP np;

	if (op eq nil) return nil;
	np = allocVn(-1);
	*np = *op;

	D_D(L, send_msg(":=%lx\t", op);)

	generic_copyone(np);

	return np;
}

global TP copyTone(TP op)
{
	TP np;

	if (op eq nil) return nil;
	np = allocTn(0);
	*np = *op;

	D_D(L, send_msg(":=%lx\t", op);)

	generic_copyone(np);

	if (np->nflags.f.bas)
		np->nflags.f.bas = 0;
	return np;
}

global NP for_old_copyone(NP np)
{
	NP newp;

	if (np eq nil) return nil;
	newp = allocnode(-1);
	*newp = *np;

	D_D(L, send_msg("=:%lx\t", np);)

#if NODESTATS
	G.ncnt[0]--;
	G.ncnt[newp->nt]++;
#endif
	np->right = nil;
	np->left = nil;
#if FLOAT
	if (np->token eq RCON)
		np->val.dbl = nil;
#endif
#if LONGLONG
	if (np->token eq LCON)
		np->val.dbl = nil;
#endif
	line_number(newp);
	np->nflags.f.nheap = NCOPY;
	return newp;
}

global NP nthnode(NP np, short n)			/* not used yet */
{
	while (n--)
		if (np eq nil)
			return nil;
		else
			np = np->left;
	return np;
}

global NP rthnode(NP np, short n)			/* only used in pre.c */
{
	while (n--)
		if (np eq nil)
			return nil;
		else
			np = np->right;
	return np;
}

/*  hashing method from very old gnu ld, but still good */

typedef struct hashdic
{
	struct hashdic *link;
	char name[0];
} DIC;

void ** dictionary;
static MEMBASE dictionary_mem;
global short numhash = NHASH;

short hash(string key)
{
	string s = key;
	short k = 0;

	while (*s)
#if 1
		k = (((k << 1) + (k >> 14)) ^ (*s++)) & 0x3fff;
#else
		k += *s++;
#endif
	return k % numhash;
}

/* Get the dictionary entry for the identifier named key.
   Create one if there is none.  */

char * to_dictionary(char *key)	/* for ID's only */
{
	short hashval;
	DIC *bp;

	/* Determine the proper bucket.  */

	hashval = hash(key);

	/* Search the bucket.  */

	bp = dictionary[hashval];
	while (bp)
	{
		if (strcmp (key, bp->name) eq 0)
			return bp->name;
		bp = bp->link;
	}

	/* Nothing was found; create a new dictionary entry.  */

	bp = CC_qalloc(&dictionary_mem, sizeof(DIC) + strlen(key) + 1, CC_ranout, AH_CC_DIC);
	if (bp)
	{
		strcpy (bp->name, key);

		/* Add the entry to the bucket.  */

		bp->link = dictionary[hashval];
		dictionary[hashval] = bp;
#if NODESTATS
		G.dictionary_names++;
#endif
		return bp->name;
	}
	else
		return "~~";
}

VpV free_dictionary
{
#if NODESTATS
	console("unique id's: %ld\n", G.dictionary_names);
	console("heap's     : %ld\n", G.heap_names);
#endif
	free_membase(&dictionary_mem);
	CC_xfree(dictionary, 243);
}

static
char * to_name_heap(short l, char *n)	/* may contain \0 */
{
	char *s, *to;

	if (l eq 1)
	{
		/* very common (about 30 %) */
		if (*n eq ' ')
			return " ";
		if (*n eq '0')
			return "0";
		if (*n eq '1')
			return "1";
	}

	to = CC_qalloc(&dictionary_mem, l + 1, CC_ranout, AH_CC_NAMES);
	if (to)
	{
		s = to;
		while(l--) *s++ = *n++;
		*s = 0;
#if NODESTATS
		G.heap_names++;
#endif
		return to;
	}
	else
		return "~~";
}

VpV init_dictionary
{
#if NODESTATS
	G.heap_names = 0;
	G.dictionary_names = 0;
#endif
	dictionary = CC_xcalloc(1, sizeof(void *) * numhash, AH_INIT_NAMES, CC_ranout);
	init_membase(&dictionary_mem, 8192, 0, "dictionary_memory", nil);
}

#if FLOAT
global void new_rnode(NP np, double x)
{
	VP dp = np->val.dbl;
	if (np->token ne RCON)
		dp = allocVn(RLNODE);
	np->token = RCON;
	dp->rval = x;
	np->val.dbl = dp;
}

global double getrcon(NP np)
{
	VP dp = np->val.dbl;
	if (np->token eq RCON)
		if (dp)
			return dp->rval;
		else
			CE_("RCON no val.dbl");
	else
		CE_("no rnode");
	return 0;
}
#endif

#if LONGLONG
global void new_lnode(NP np, __ll x)
{
	VP dp = np->val.dbl;
	if (np->token ne LCON)
		dp = allocVn(LLNODE);
	np->token = LCON;
	dp->llval = x;
	np->val.dbl = dp;
}

global __ll getlcon(NP np, short which)
{
	__ll ll_0 = {0,0};
	VP dp = np->val.dbl;
	if (np->token eq LCON)
		if (dp)
			return dp->llval;
		else
			CE_("LCON no val.ll");
	else
	{
		CE_("no lnode");
		send_msg(" -------- [%d]\n", which);
	}
	return ll_0;
}
#endif
