/* Copyright (c) 2004 - 2008 by H. Robbers Amsterdam.
 *
 * This file is part of AHCC.
 *
 * AHCC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * AHCC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AHCC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * xref.c			the projects cross reference
 *                  database.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <tos.h>
#include "common/mallocs.h"
#include "common/aaaa_lib.h"
#include "common/hierarch.h"

#include "aa_intro.h"
#include "xref.h"
#include "common/config.h"
#include "shlcfg.h"
#include "common/ipff.h"
#include "ahcc/lmem.h"

MEMBASE xref_mem;

DEP *auto_dependencies = nil, *prj_dependencies = nil;

XREF_BASE identifiers = { nil, nil, nil },
          filenames   = { nil, nil, nil };		/* 04'09 */

bool dep_changed = false;

void	send_msg	(char *text, ...);

void ftab(FILE *fp, short lvl)
{
	if (lvl > 0)
		while (lvl--)
			fprintf(fp, "\t");
}

void xref_list_files(FILE *fp, DEP *root, short lvl)
{
	DEP *dp = root;
	if (fp eq stdout)
	{
		while (dp)
		{
			send_msg("%d>[%d]%s\n", lvl, dp->data->file_number, dp->data->name);
			xref_list_files(fp, dp->depend, lvl + 1);
			dp = dp->next;
		}
	othw
		if (root)
		{
			ftab(fp, lvl - 1);
			fprintf(fp, "{\n");
		}
		while (dp)
		{
			ftab(fp, lvl);
			fprintf(fp, "[%d]%s\n", dp->data->file_number, dp->data->name);
			xref_list_files(fp, dp->depend, lvl + 1);
	
			dp = dp->next;
		}
		if (root)
		{
			ftab(fp, lvl - 1);
			fprintf(fp, "}\n");
		}
	}
}

void xref_free_files(DEP *root)
{
	DEP *dp = root;
	
	while (dp)
	{
		DEP *nx = dp->next;
		xref_free_files(dp->depend);
		free(dp);
		dp = nx;
	}
}

void xref_list_ides(FILE *fp, XREF *x)
{
	while (x)
	{
		if break_in
			break;

		if (fp eq stdout)
			send_msg("%5d [%d/%ld]\t%s\n", x->n, x->file_number, x->line_number, x->name);
		else
		{
			fprintf(fp, "\t[%d/%ld", x->file_number, x->line_number);
			if (x->scope)
				fprintf(fp, ",%d", x->scope);
			fprintf(fp,"]%s\n", x->name);
		}
		x = x->next;
	}
}

void xref_list_ordered(FILE *fp, XREF *x, char arrow, short lvl)
{
	if (x)
	{
		if (fp eq stdout)
		{
			if break_in
				return;

			xref_list_ordered(fp, x->less,  '<', lvl + 1);
#if 0
			send_msg("%d>%c[%d/%ld]\t%s\t%d\n", lvl, arrow, x->file_number, x->line_number, x->name, x->n);
#else
			send_msg("%5d [%d/%ld]\t%s\t\n", x->n, x->file_number, x->line_number, x->name);
#endif
			xref_list_ordered(fp, x->great, '>', lvl + 1);
		othw
			xref_list_ordered(fp, x->less,  0, lvl + 1);
#if 0
			ftab(fp, lvl - 1);
#endif
			fprintf(fp, "\t[%d/%ld", x->file_number, x->line_number);
			if (x->scope)
				fprintf(fp, ",%d", x->scope);
			fprintf(fp,"]%s\n", x->name);
			xref_list_ordered(fp, x->great, 0, lvl + 1);
		}
	}
}

DEP * xref_find_file(DEP *root, char *name, bool deep)
{	
	DEP *dp = root;

	if (root)
	{
		while (dp)
		{
			if (strcmp(name, dp->data->name) eq 0)
				return dp;
			dp = dp->next;
		}
		if (deep)
			return xref_find_file(root->depend, name, deep);
	}
	return dp;
}

#if 0
void dep_change(bool b, short which)
{
	if (b ne dep_changed)
		send_msg("[%d] dep_changed --> %d\n", which, b);
	dep_changed = b;
}
#else
#define dep_change(b, w) dep_changed = b
#endif

global
DEP * xref_new_file(DEP **in, char *name, short count)
{
	DEP *last = *in,
	    *dp = xmalloc(sizeof(*dp), AH_PRJ);

	if (!dp)
		send_msg("ran out of memory for project database\n");
	else
	{
		dp->data = xref_new_ide(10, &filenames, name, 0, 0, 0);
		if (dp->data)
		{
			if (last)
			{
				while(last->next)
					last = last->next;
				last->next = dp;
			}
			else
				*in = dp;
	
			dp->next = nil;
			dp->depend = nil;				/* later */
	
			if (count < 0)
				dp->data->file_number = ++filecount;
			else
			{
				dp->data->file_number = count;
				if (count > filecount)
					filecount = count;
			}
		}
		dep_change(true, 1);
	}

	return dp;
}

char *xref_file_by_number(DEP *root, short number)
{
	DEP *dp = root;

	while (dp)
	{
		if (dp->data->file_number eq number)
			return dp->data->name;
		dp = dp->next;
	}
	
	return "~~~";
}

void xref_init(void)
{
	init_membase(&xref_mem, 8192, 0, "xref memory base", nil);
	auto_dependencies = nil;
	prj_dependencies  = nil;
	identifiers.cur   = nil;
	identifiers.first = nil;
	identifiers.last  = nil;
	filenames.cur     = nil;
	filenames.first   = nil;
	filenames.last    = nil;
	dep_change(false, 2);
	filecount = 0;
}

void xref_free_dep(void)
{
	XA_free_all(nil, AH_XREF, -1);
	xref_free_files(auto_dependencies);
	xref_free_files(prj_dependencies);
	auto_dependencies = nil;
	prj_dependencies = nil;
	free_membase(&xref_mem);
	identifiers.cur   = nil;
	identifiers.first = nil;
	identifiers.last  = nil;
	filenames.cur     = nil;
	filenames.first   = nil;
	filenames.last    = nil;
	filecount = high_prj;		/* reset highest file number */
	dep_change(false, 3);
}

extern char depfn[], mkfn[];

void xref_write_dep(void)
{
	FILE *fp;
	char buf[8200];
#if GEMSHELL
	mbumble;
#endif
	fp = fopen(depfn, "w");
	setvbuf(fp, buf, _IOFBF, 8192);			/* set full buffereing (Doesnt flush at newline!!!) */
	if (fp eq nil)
		send_msg("Cant create %s\n", depfn);
	else
	{
		extern char *ahcc_version;
		fprintf(fp, "ahcc_dep AHCC %s project database\n", ahcc_version);
		fprintf(fp, "files %d\n", filecount);
		fprintf(fp, "dependencies\n");
		xref_list_files(fp, auto_dependencies, 1);
		if (identifiers.first)
		{
			fprintf(fp, "\nidentifiers\n{\n");
			xref_list_ides(fp, identifiers.first);
			fprintf(fp,"}\n");
		}
		fclose(fp);
	}

	if (SHL_cfg.v)
		send_msg("saved dependencies %s\n", depfn);
#if GEMSHELL
	mpijl;
#endif
}

static
void match_prj(char *fn, short fileno)
{
	DEP *x = xref_find_file(prj_dependencies, fn, 0);
	if (x)
	{
		if (x->data->file_number ne fileno)
			x->data->file_number = fileno;
	}
}

static
void match_deps(DEP *deps)
{
	if (deps)
	{
		DEP *jp = prj_dependencies;
		
		while (jp)
		{
			char *fn = jp->data->name;
			short fileno = jp->data->file_number;

			DEP *x = xref_find_file(deps, fn, 0);

			if (x eq nil)		/* new file in project */
			{
				char *f;
				f = xref_file_by_number(deps, fileno);		/* is the number in use? */
				if (*f ne '~')
					jp->data->file_number = ++filecount;
			}

			jp = jp->next;
		}
	}
}

DEP * xref_read_list(short lvl)
{
	DEP *deps = nil;

	do
	{
		if (sk() eq '[')
		{
			DEP *this;
			short fileno; char fn[256];

			skc();
			fileno = idec();
			if (fileno > filecount)			/* 04'09 */
				filecount = fileno;
			if (sk() eq ']') skc();
			fstr(fn);
			this = xref_new_file(&deps, fn, fileno);
			if (lvl eq 0)
				match_prj(fn, fileno);		/* transfer autodep filename to prj */
			if (sk() eq '{')
			{
				skc();
				if (this)
					this->depend = xref_read_list(lvl + 1);
				if (sk() eq '}') skc();
			}
		}
		else
			break;
	}
	od

	return deps;
}	

static short cmp;
static char *r;
static XREF *last;

short bfind_ide(XREF *x)
{
	if (x)
	{
		last = x;
		cmp = strcmp(r, x->name);
		if (cmp < 0)
			return bfind_ide(x->less);
		if (cmp > 0)
			return bfind_ide(x->great);
	}
	return cmp;
}

XREF *xref_find_ide(XREF_BASE *ides, char *n)
{
	cmp = 1;
	last = nil;
	r = n;
	cmp = bfind_ide(ides->cur);
	return cmp eq 0 ? last : nil;
}

static
void ins_ide(XREF_BASE *ides, XREF *x)
{
	if (ides->cur)
	{
		if (cmp < 0)
			last->less = x;
		else
			last->great = x;
	}
	else
		ides->cur = x;

	if (ides->first eq nil)
	{
		ides->first = ides->last = x;
		x->n = 1;
	othw
		x->n = ides->last->n + 1;
		ides->last->next = x;
		ides->last = x;
	}
	
	x->next = nil;
}

XREF * xref_new_ide(short which, XREF_BASE *ides, char *ide, short file, long line, short scope)
{
	XREF *x = xref_find_ide(ides, ide);
	if (x)
	{
		if (scope > x->scope)			/* keep highest scope */
		{
			x->scope = scope;
			x->line_number = line;
			x->file_number = file;
			dep_change(true, 4);
		}
		elif (    scope eq x->scope
		      and scope ne -1
		      and(x->line_number ne line or x->file_number ne file))
		{
			x->line_number = line;
			x->file_number = file;
			dep_change(true, 5);
		}
	/*	else
			unchanged */
	othw
		short l = strlen(ide);
		x = qalloc(&xref_mem, sizeof(*x) + l + 1, "", "identifiers");
		if (x)
		{
			x->line_number = line;
			x->file_number = file;
			x->scope = scope;
			strcpy(x->name, ide);
			x->less = nil;
			x->great = nil;
			ins_ide(ides, x);
			dep_change(true, 6);
		}
	}

	return x;
}

void xref_read_ides(XREF_BASE *ides)
{
	ides->cur = nil;

	if (sk() eq '{')
	{
		skc();

		do
		{
			if (sk() eq '[')
			{
				char rstr[IPFF_L + 2];
				short fileno = 0, ty = 0;
				long lineno = 1;

				skc();
				fileno = idec();
				if (sk() eq '/')
				{
					skc();
					lineno = dec();
				}
				if (sk() eq ',')
				{
					skc();
					ty = idec();
				}
				if (sk() eq ']') skc();
				str(rstr);
				if (*rstr)
					xref_new_ide(1, ides, rstr, fileno, lineno, ty);
			}
			else
				break;
		}
		od

		if (sk() eq '}') skc();
	}
}

DEP * xref_read_dep(void)
{
	short fl;
	long l;
	char *s;
	DEP *deps = nil;

#if GEMSHELL
	mbumble;
#endif

	s = Fload(depfn, &fl, &l, "while loading deps", AH_XREF);
	if (s)
	{
		char rstr[IPFF_L + 2];

		send_msg("Loading %s\n", depfn);

		ipff_in(s);
		sk();
		str(rstr);
		if (strcmp(rstr,"ahcc_dep") eq 0)
		{
			sknl();
			str(rstr);
			if (strcmp(rstr, "files") eq 0)
			{
				sk();
				filecount = idec();
				sk();
				str(rstr);

				if (strcmp(rstr, "dependencies") eq 0)
				{
					if (SHL_cfg.v)
						send_msg("%s present\n", rstr);
					if (sk() eq '{')
					{
						skc();
						deps = xref_read_list(0);
						if (sk() eq '}') skc();
					}
				}
	
				sk();
				str(rstr);
				if (strcmp(rstr, "identifiers") eq 0)
				{
					if (SHL_cfg.v)
						send_msg("%s present\n", rstr);
					xref_read_ides(&identifiers);
				}
	
				ffree(s, 100);
				dep_change(false, 7);
				match_deps(deps);		/* renumber new project files not in auto_dependencies */

	#if GEMSHELL
				mpijl;
	#endif
				return deps;
			}
		}

		ffree(s, 101);
		send_msg("Format error in %s\n", depfn);
	}
	else
		send_msg("No dependencies for %s\n", mkfn);

	dep_change(false, 8);
	
#if GEMSHELL
	mpijl;
#endif
	return nil;	
}
