/* Copyright (c) 2004 - 2009 by H. Robbers.
 *
 * This file is part of AHCC.
 *
 * AHCC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * AHCC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AHCC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * assembler: parse routines for as_parse
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "param.h"

#include "decl.h"
#include "d2.h"
#include "expr.h"
#include "e2.h"
#include "inst.h"
#include "opt.h"
#include "out.h"
#include "reg.h"
#include "po.h"
#include "as.h"
#include "as_plib.h"

#define debug_a G.yflags['a'-'a']
#define debugA G.xflags['a'-'a']

static
char *ordinal[] =
{
	"size ",
	"first ",
	"second ",
	"third ",
	"fourth ",
	"fifth ",
	"sixth ",
	"seventh ",
	"eighth ",
	"",				/* unspecified */
	0
};

global
short opnd_err(short n, char *msg, short which)
{
	if (n < 0 or n > 9)
		n = 9;			/* we really need enumerating array initializations */
#if C_DEBUG
	error("[%d]invalid %soperand: %s", which, ordinal[n], msg);
#else
	error("invalid %soperand: %s", ordinal[n], msg);
#endif
	return 1;
}

static char * modes[] =
{
	"all",
	"control alterable",
	"control addressable",
	"memory alterable",
	"memory addressable",
	"data alterable",
	"data addressable",
	"alterable",
	"control"
};

global
void ea_error(short n, short type)
{
	error("only '%s' modes allowed for %s operand", modes[type&0xff], ordinal[n]);
}

char *amodes[] =
{
	"NONE",
	"REG",
	"REGI",
	"REGID",
	"REGIDX",
	"REGIDXX",
	"PCD",
	"PCDX",
	"PCDXX",
	"IMM",
	"ABS",
	"RLST",
	"*12*",
	"*13*",
	"*14*",
	"*15*"
};

char *aflags[] =
{
	"XLONG",
	"SYMB",
	"ABSW",
	"BDISP",
	"ODISP",
	"MIND",
	"POSTI/INC",
	"PRE_I/DEC"
};

string pr_amode(ushort m)
{
	static char s[128];
	short i = 0;

	strcpy(s, amodes[m&15]);
	m >>= 8;
	while (m)
	{
		if (m&1)
		{
			strcat(s,"|");
			strcat(s, aflags[i]);
		}
		i++;
		m >>= 1;
	}

	return s;
}

void pr_opnd(short n, OPND *a, char *text)
{
	if (a)
	{
		send_msg("%s%s%s, ", ordinal[n], text, pr_amode(a->amode));
		send_msg("disp %ld, areg %d, ireg %d, ", a->disp, a->areg, a->ireg);
		send_msg("[%d]%s+%ld\n", a->aname, a->astr ? a->astr : "~~", a->namedisp);
		if (a->outd)
			if (a->outd->amode ne NONE)
			{
				send_msg("\t");
				pr_opnd(n, a->outd, "OUTD ");
			}
	}
	else
		send_msg("~~ operand\n");
}

AREA_INFO zero_info ={0,0,0};

global
void asm_offs(NP np, TP tp)
{
	if (np->sc eq K_AUTO)
	{
		np->token = tp->token  = ICON;
		np->val.i = tp->offset;
		tp->type  = basic_type(T_LONG);
		np->area_info = zero_info;
	}
	elif (np->area_info.class eq OFFS_class)
	{
		np->token = tp->token  = ICON;
		np->val.i = tp->offset = np->area_info.disp;
		tp->type  = basic_type(T_LONG);
		np->area_info = zero_info;
	}
}

global
OPND *new_arg(void)
{
	OPND *new = CC_qalloc(&opndmem, sizeof(*new), CC_ranout, AH_CC_INST_OP);
	if (new)
		pzero(new);
	return new;
}

global
bool check_ea(OPND *a, short type)
{
	short reg = a->areg, mo = a->amode;

	if (type & AM_DN)
		if (mo eq REG and ISD(reg))
			return true;
	if (type & AM_INC)
		if (mo eq (REGI|INC))
			return true;
	if (type & AM_DEC)
		if (mo eq (REGI|DEC))
			return true;

	mo = MM(mo);
	type &= 0xff;

	switch (type)
	{
		case AM_CTRL:
			return    check_ea(a, AM_C_ADD)
			       and mo ne IMM;
		case AM_C_ALT:
			return (    check_ea(a, AM_ALTER)
			        and check_ea(a, AM_C_ADD) );
		case AM_C_ADD:
			if (mo eq REG)
				return false;
			if (mo eq REGI and (a->amode & (INC|DEC)) ne 0)
				return false;
			return true;
		case AM_M_ALT:
			return (   check_ea(a, AM_C_ALT)
			        or (    mo eq REGI
			            and (a->amode & (INC|DEC)) ne 0) );
		case AM_M_ADD:
			return mo ne REG;
		case AM_D_ALT:
			return (   check_ea(a, AM_M_ALT)
			        or (mo eq REG and ISD(reg)) );
		case AM_D_ADD:
			if (mo eq REG and ISA(reg))
				return false;
			return true;
		case AM_ALTER:
			if (mo eq IMM or mo eq PCD or mo eq PCDX or mo eq PCDXX)
				return false;
			return true;
	}

	return true;
}

global
short isize(string allowed, short def)		/* 12'10 HR: allow for .label */
{
	short sz = -1;
	G.dot_seen = false;
	if (cur->token eq SELECT)
	{
		fadvnode();
		if (cur->token ne ID)
			opnd_err(0, "no size identifier", 1001);
		else
		{
			if   (allowed[0] ne ' ' and stricmp(cur->name, "b") eq 0)
				sz = DOT_B;
			elif (allowed[1] ne ' ' and stricmp(cur->name, "w") eq 0)
				sz = DOT_W;
			elif (allowed[2] ne ' ' and stricmp(cur->name, "l") eq 0)
				sz = DOT_L;
			elif (allowed[3] ne ' ' and stricmp(cur->name, "s") eq 0)
				sz = DOT_B;			/* short branche operand */
			elif (allowed[4] ne ' ' and stricmp(cur->name, "x") eq 0)
				sz = DOT_X;
			elif (allowed[5] ne ' ' and stricmp(cur->name, "d") eq 0)
				sz = DOT_D;
			elif (allowed[6] ne ' ' and stricmp(cur->name, "p") eq 0)
				sz = DOT_P;
			elif (allowed[7] ne ' ' and stricmp(cur->name, "s") eq 0)
				sz = DOT_S+1;		/* single precision floating point */

			if (sz ne -1)
				fadvnode();
			else
				G.dot_seen = true;
		}
	}
	else
		sz = def;

	return sz;
}

#if 0
global
short bra_size(short def)		/* allow for .label */
{
	short sz = -1;
	G.dot_seen = false;
	if (cur->token eq SELECT)
	{
		fadvnode();
		if (cur->token ne ID)
			opnd_err(0, "no size identifier", 1001);
		else
		{
			if   (stricmp(cur->name, "b") eq 0)
				sz = DOT_B;
			elif (stricmp(cur->name, "w") eq 0)
				sz = DOT_W;
			elif (stricmp(cur->name, "l") eq 0)
				sz = DOT_L;
			elif (stricmp(cur->name, "s") eq 0)
				sz = DOT_B;			/* short branche operand */

			if (sz ne -1)
				fadvnode();
			else
				G.dot_seen = true;
		}
	}
	else
		sz = def;

	return sz;
}
#endif

global
bool check_imm(long l, short sz, short n)
{
	long check = l;

	switch(sz)
	{
		case DOT_B:
			check &= 0xffffff00;
			if (check ne 0 and check ne 0xffffff00)
				return opnd_err(n, "out of range", 1003), false;
		break;
		case DOT_W:
			check &= 0xffff0000;
			if (check ne 0 and check ne 0xffff0000)
				return opnd_err(n, "out of range", 1004), false;
		break;
	}

	return true;
}

global
void chk_glob(OPND *a, void * e1)
{
	TP np = e1;
	TP rv = tlook(symtab[hash(np->name)],np);

	if (rv)
		if (rv->sc eq K_GLOBAL)
			a->gl = true;
}

static
bool asm_select(OPND *a, NP e1, short am)
{
	if (e1->token eq ASM_SELECT)		/* strunion @ member */
	{
		if (e1->left->sc eq K_TYPE)		/* declarer @ member (offsetof) */
		{
			a->amode = am;
			a->disp  = e1->val.i;
		othw							/* variable @ member */
			a->astr  = e1->left->name;
			a->tlab  = e1->left->lbl;
			a->areg  = e1->area_info.id;
			a->amode = am | SYMB;
			a->disp  = e1->val.i;
		}

		return true;
	}

	return false;
}

static
void name_size(OPND *a, NP np)
{
	if (np->eflgs.f.asm_w)
		a->amode |= ABSW;
}

static
short name_xpr(OPND *a, NP e1, short am)  /* id, n or id+n */
{
	am |= a->amode & 0xff00;		/* keep flags */

	if (asm_select(a, e1, am))
		return 0;

	if (    e1->token eq PLUS
	    and e1->left ->token eq ID
	    and e1->right->token eq ICON
	   )
	{
		a->astr = e1->left->name;
		a->tlab = e1->left->lbl;
		a->areg = e1->left->area_info.id;
		a->amode = am | SYMB;
		chk_glob(a, e1->left);
		a->disp = e1->right->val.i;
		name_size(a, e1->left);
	}
	elif (e1->token eq ID)
	{
		a->astr = e1->name;
		a->tlab = e1->lbl;
		a->areg = e1->area_info.id;
		a->amode = am | SYMB;
		chk_glob(a, e1);
		name_size(a, e1);
	}
	elif (e1->token eq ICON)
	{
		a->disp = e1->val.i;
		a->amode = am;
		name_size(a, e1);
	}
	else
		return 7;

/*	if (tysz eq DOT_W)
		a->amode |= ABSW;
*/	return 0;
}

global
bool immediate(OPND *a, bool cons, short n)
{
	NP e1; short err = 0;

	if (cur->token eq PREP)
		fadvnode();
	e1 = asm_expr();

	if (e1)
	{
		short err = name_xpr(a, e1, IMM);
		if (err)
			opnd_err(n, "not id, n or id+n", 1005);
		if (cons and MAB(a->amode) ne IMM)
			err = opnd_err(n, "needs constant immediate", 1006);
		freenode(e1);
	}
	else
		err = opnd_err(n, "missing", 1007);

	return err eq 0;
}

global
short p_reg(NP np)
{
	short tok = np->token;
	if (np->category & ASMREG)
		if (tok > K_A7)
			return D0 + (tok - K_D0);
		else
			return A0 + (tok - K_A0);
	return -1;
}

#if FLOAT
global
short p_freg(NP np)
{
	short tok = np->token;
	if (np->category & ASM)
		if (tok >= K_F0 and tok <= K_F7)
			return F0 + (tok - K_F0);
	return -1;
}
#endif

global
bool p_isareg(void * vp)
{
	XP np = vp;
	if (np->category & ASMREG)
		if (np->token <= K_A7)
			return true;
	return false;
}

global
bool p_isdreg(void * vp)
{
	XP np = vp;
	if (np->category & ASMREG)
		if (np->token >= K_D0)
			return true;
	return false;
}

#if FLOAT
global
bool p_isfreg(void * vp)
{
	XP np = vp;
	if (np->category & ASM)
		if (np->token >= K_F0 and np->token <= K_F7)
			return true;
	return false;
}
#endif

static
short base_reg(OPND *a, NP np)
{
	short m;

	if (np->token eq K_PC)
		m = PCD;
	else
	{
		if (p_isareg(np))
		{
			a->areg = p_reg(np);
			m = a->disp eq 0 ? REGI : REGID;
		}
		elif (p_isdreg(np))
		{
			if ((G.CPU & _H) eq 0)
				return 12;
			a->areg = -1;
			a->ireg = p_reg(np);
			m = REGIDXX;
		}
		else
			return 13;
	}
	a->amode |= m;
	return 0;
}

static
void mem_ind(OPND *a, NP np, short which, NP *bd, NP *An, NP *Xn)
{
	if (np->category & ASMREG)
	{
		*An = np;			/* ([reg]...) */
		if (p_isdreg(np))
			a->amode |= PREI;
	othw
		if (np->token eq COMMA)		/* ([x,y...]...) */
		{
			if (np->left->token eq COMMA)		/* ([bd,An,Xn]...) */
			{
				*bd = np->left->left;
				*An = np->left->right;
				*Xn = np->right;
				a->amode |= PREI;
			}
			elif (np->left->category & ASMREG)
			{
				*An = np->left;		/* ([An,Xn]...) */
				*Xn = np->right;
				a->amode |= PREI;
			othw
				*bd = np->left;		/* ([bd,An]...) */
				*An = np->right;
				if (p_isdreg(np->right))
					a->amode |= PREI;
			}
		}
		else
			*bd = np;				/* ([bd]...) */
	}
}

static
NP opt_scale(OPND *a, NP xp, short n)
{
	if (xp->tt eq E_BINARY and xp->token eq ASMSCALE)
	{
		if ((G.CPU & _H) eq 0)		/* cpu id must be high */
			opnd_err(n, "scale factor not allowed", 1008);
		else
		{
			short scale = xp->right->val.i;
			xp = xp->left;
			if (scale ne 2 and scale ne 4 and scale ne 8)
				opnd_err(n, "scale factor must be 2, 4, or 8", 1009);
			else
				a->scale = scale eq 2 ? 1 : scale eq 4 ? 2 : 3;
		}
	}

	return xp;
}

short make_opnd(short n, OPND *a, NP bd, NP An, NP Xn, short m1, short m2)
{
	short err = 0, mode = m2;

	D_1(_a, "base disp", bd);

	if (bd)
	{
		err = name_xpr(a, bd, m2);
		if (err)
			opnd_err(n, "not id, n or id+n", 1010);
		a->amode |= BDISP;
	}

	D_1(_a, "base reg", An);

	if (An)
	{
		if (An->token eq K_PC)
			mode = m1;
		elif (p_isareg(An))
			a->areg = p_reg(An);
		else
		{
			if (Xn)
				err = opnd_err(n, "too many registers", 1011); 			/* syntax error */
			Xn = An;
			An = nil;
		}
	}

	D_1(_a, "index reg", Xn);

	if (Xn)
	{
		Xn = opt_scale(a, Xn, n);

		if (p_isareg(Xn) or p_isdreg(Xn))
		{
			a->ireg = p_reg(Xn);
			if (Xn->eflgs.f.asm_l)			/* 04'09 use ASM_W|ASM_L */
				a->amode |= XLONG;
			elif (   !(Xn->eflgs.f.asm_l or Xn->eflgs.f.asm_w)
			      and G.aw_Xnl
			     )
				a->amode |= XLONG;
		}
		else
			err = opnd_err(n, "invalid index register", 1012);
	}

	a->amode &= 0xff00;			/* keep flags */
	a->amode |= mode;

	return err;
}

/* the number of good asm extended operand expression trees is finite
   and actually not extremely large.

   Some trees might just have the same effect

   bd = base displacement
   An = base register
   Xn = index register
   od = outer displacement

   Note that there is a difference between An and Xn such that
   An can only be a address reg and Xn can also be a data reg

   Xn and od also involve a choice between post & pre memory indexing

([bd,An,Xn],od)		/* 1111 */
([bd,An],Xn,od)

([bd,An,Xn])		/* 1110 */
([bd,An],Xn)

([bd,An],od)		/* 1101 */

([bd,An])			/* 1100 */

([bd,Xn],od)		/* 1011 */
([bd],Xn,od)

([bd,Xn],od)		/* 1010 */
([bd],Xn,od)

([bd],od)			/* 1001 */

([bd])				/* 1000 */

([An,Xn],od)		/* 0111 */
([An],Xn,od)

([An,Xn])			/* 0110 */
([An],Xn)

([An],od)			/* 0101 */

([An])				/* 0100 */

([Xn],od)			/* 0011 */

([Xn])				/* 0010 */
(Xn)

(od)				/* 0001 */

()					/* 0000 */

(bd,An,Xn)			/*  111 */
(bd,An)				/*  110 */
(bd,Xn)				/*  101 */
(bd)				/*  100 */
(An,Xn)				/*  011 */
(An)				/*  010 */
(Xn)				/*  001 */

()					/*  000 */

The other (normal) addressing modes are quite uncomplicated
*/

char *cpu_n_avail = "addressing mode not available on current cpu type";

global
OPND * p_ea(OPND *a, short type, short n)
{
	short reg, err = 0;
	NP e1, lp, rp,
	   bd = nil,
	   An = nil,
	   Xn = nil,
	   oud = nil;

	a->amode = NONE;

	if (cur->token eq PREP)
		return immediate(a, IMMXPR, n), a;

	e1 = asm_expr();

	D_1(_a, "operand tree:", e1);

	if (e1)
	{
		switch (e1->tt)
		{
			case E_LEAF:
				if (e1->token eq INSTR)		/* 05'11 HR: '*' operand */
				{
					a->amode = INSTD;
					a->disp = e1->val.i;
				othw					/* simple: register or name or number */
					reg = p_reg(e1);
					if (reg >= 0)
					{
						a->amode = REG;
						a->areg = reg;
					}
					else
						err = name_xpr(a, e1, ABS);
				}
			break;
			case E_UNARY:
				if (!asm_select(a, e1, ABS))
				{
					lp = e1->left, rp = e1->right;
					if (e1->token eq REGINDIM)		/* memory indirect ([...]) */
					{
						if ((G.CPU & _H) eq 0) { err = opnd_err(n, cpu_n_avail, 1013); break; }
						a->areg = -1, a->ireg = -1;

						a->amode |= MIND;
						mem_ind(a, lp, 1, &bd, &An, &Xn);		/* dispatch [] elements */
						err = make_opnd(n, a, bd, An, Xn, PCDXX, REGIDXX);
					}
					elif (lp->tt eq E_LEAF)			/* simple indirection */
					{
						if (e1->token eq REGINDIRECT)		/* ...REGIND... always have valid reg */
						{
							if (lp->token eq K_PC)
								a->amode = PCD;
							elif (p_isareg(lp))
							{
								a->amode = REGI;
								a->areg = p_reg(lp);
							othw
								if ((G.CPU & _H) eq 0) { err = opnd_err(n, cpu_n_avail, 1014); break; }

								a->amode = REGIDXX;
								a->areg = -1;
								a->ireg = p_reg(lp);
							}
						}
						elif (e1->token eq REGINDPLUS)
						{
							a->amode = REGI|INC;
							a->areg = p_reg(lp);
						}
						elif (e1->token eq MINUSREGIND)
						{
							a->amode = REGI|DEC;
							a->areg = p_reg(lp);
						}
					}
				}
			break;
			case E_BINARY:					/* simple asm expression: number, name or name+number */
				err = name_xpr(a, e1, ABS);
				if (err)
					opnd_err(n, "not id, n or id+n", 1015);
			break;
			case E_SPEC:					/* complex: extended addressing modes */
				lp = e1->left, rp = e1->right;
				if (e1->token eq REGINDISP)
				{
					err = name_xpr(a, lp, 0);
					if (err) { opnd_err(n, "not id, id+n or n", 1016); break; }

					if (rp->tt eq E_LEAF)			/* base register only */
					{
						if (!check_imm(lp->val.i, DOT_W, n))
							err = 3;
						if (err eq 0)
						{
							err = base_reg(a, rp);
							if (err) { opnd_err(n, "invalid base register", 1017); break; }
						}
					}
					elif (rp->token eq COMMA)		/* both base & index register */
					{
						if (!check_imm(lp->val.i, DOT_B, n))
							err = 3;
						if (err eq 0)
						{
							An = rp->left;
							Xn = rp->right;
							err = make_opnd(n, a, nil, An, Xn, PCDX, REGIDX);
						}
					}
				}
				elif (e1->token eq REGINDX)			/* 2 or 3 elements separated by comma's */
				{									/* REGINDX is last comma */
					if ((G.CPU & _H) eq 0) { err = opnd_err(n, cpu_n_avail, 1018); break; }

					a->areg = -1, a->ireg = -1;

					if (lp->token eq REGINDIM)		/* ([...] */
					{
						a->amode |= MIND;
						if (rp->token ne COMMA)		/* ([...], od) */
						{
							mem_ind(a, lp->left, 2, &bd, &An, &Xn);  /* dispatch [] elements */
							if (Xn eq 0 and (rp->category & ASMREG) ne 0)	/* ([bd,An],Xn) */
							{
								a->amode |= POSTI;
								Xn = rp;
							}
							else
								oud = rp;
						}
					}
					elif (lp->token eq COMMA)		/* (x,,) */
					{
						NP cp = lp->left;

						if (cp->token eq REGINDIM)	/* ([..],,) */
						{
							a->amode |= MIND;
							mem_ind(a, cp->left, 3, &bd, &An, &Xn);

							if (Xn)
								err = opnd_err(n, "too many registers", 1019);

							a->amode |= POSTI;
							Xn = lp->right;
							oud = e1->right;
						othw
							Xn = rp;
							An = lp->right;
							bd = cp;
						}
					othw							/* (,) */
						Xn = nil;
						An = rp;
						bd = lp;
					}

					err = make_opnd(n, a, bd, An, Xn, PCDXX, REGIDXX);

					if (oud)
					{
						err = name_xpr(a->outd, oud, ABS);
						if (err)
							opnd_err(n, "invalid outer displacement", 1020);
						else
							a->amode |= ODISP;
					}
				}
			break;
		}

		freenode(e1);
	}

	if ((uchar)a->amode eq NONE)
		opnd_err(n, "syntax", 9999);
	elif (err eq 0)
	{
		D_D(A,pr_opnd(n, a, "OPERAND:\n");)
		if (check_ea(a, type))
			return a;
		else
			ea_error(n, type);
	}

	return nil;
}
