/*  Copyright (c) 1993 - 2008 by Henk Robbers Amsterdam.
 *
 * This file is part of CALC.
 *
 * CALC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CALC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CALC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *		SHPARSE.C
 */

#include <string.h>
#include <ctype.h>
#include <math.h>

#include "common/aaaa_lib.h"
#include "parse.h"
#include "sheet.h"

static SH_SHEET cell;
static bool isformula;

int colspace,rowspace;

int denotation_space(int big, int root)
/* Returns the width in spaces of number denotation
	( root: number of digits in system )
	( big: biggest number that must fit in the space) */
{
	return big ? log(big+1)/log(root)+1 : 1;
}

int abs(int a) { return a < 0 ? -a : a; }
int alert_msg(char *t, ... );

IDES * find_ide(char *s)
{
	IDES *id = parsepar.ides;
	char *ide = is_label(s, nil);

	while (id)
	{
		if (strcmp(ide, id->name) eq 0)
		{
/*			sprintf(ide, "[1][ got: %s | %d,%d ][ Ok ]", id->name, id->col, id->row);
			form_alert(1, ide);
*/
			return id;
		}
		id = id->next;
	}

	return nil;
}

/* double shrange(T o, T l, T r) */
PARSE_RANGE shrange
{
	int i, j, lr, rr, lc, rc, step;
	CELL c;

	l.v = 0.0;

	step = abs(o.step);
	if (step eq 0)
		step = 1;

	lr = rr = l.row;
	if (r.row < l.row)
		lr = r.row;
	else
		rr = r.row;

	lc = rc = l.col;
	if (r.col < l.col)
		lc = r.col;
	else
		rc = r.col;

	for (i = lc; i <= rc; i += step)
		for (j = lr; j <= rr; j += step)
			if (    i >= 0 and i < MAXCOLS
			    and j >= 0 and j < MAXROWS
			   )
				if ( (c = pget_c(cell, i, j)) ne 0L)
					if (c->attr eq FORM or c->attr eq VAL)
						l.v += c->val;

	return l.v;
}

bool cellname(char **input, int *col, int *row)
/* Returns true if the string is the start of a formula, false otherwise.
   Also returns the column and row of the formula. */
{
	int len, c = 0, r = 0;
	char *start, *s=*input, numstring[10];

	start = s;

	if (*s eq '[')		/* relative name */
	{
		s++;
		s += get_int(s, &c);
		if (*s eq ',')
		{
			s++;
			s += get_int(s, &r);
		}

		if (*s eq ']')
		{
			s++;
			c += sh_col;
			r += sh_row;
			if (    c >= 0
			    and c <  MAXCOLS
			    and r >= 0
			    and r <  MAXROWS)
			{
				*col = c;
				*row = r;
				*input = s;
				return true;
			}
		}
	othw
		loop(len,colspace)
			if (*s < 'a' or *s > 'z')
				break;
			else
				s++;

		if(len ne 0)
		{
			strncpy(numstring, start, len);
			numstring[len] = '\0';
			c = crbv(numstring,'a');
			if (c < MAXCOLS)
			{
				start = s;

				loop(len,rowspace)
					if(!isdigit(*s))
						break;
					else
						s++;

				if(len ne 0)
				{
					strncpy(numstring, start, len);
					numstring[len] = '\0';
					r = cdbv(numstring)-1;
					if(r < MAXROWS and r >= 0)
					{
						*col = c;
						*row = r;
						*input=s;
						return true;
					}
				}
			}
		}
	}
	*input=s;
	return false;
}

PARSE_NEXT shnext
{
	bool iscell = false;
	char *save;
	C=BADTOK;

	while (*STR eq ' ') STR++;

	if (*STR eq 0)
	{
		C = eoln;
		return C;
	}

	save = STR;
	iscell = cellname(&STR, &C.col, &C.row);

	if (iscell)
	{
		CELL c = pget_c(cell, C.col, C.row);
		C.t = NUM;
		C.p = PRIMARY;
		C.nam = "N";
		isformula = true;
		if (c)
			C.v = c->val;
		else
			C.v  = 0.0;
	othw
		STR = parselook(save);

		if (C.t eq IDE)
		{
			IDES *id = find_ide(C.nam);
			if (id)
			{
				C.t = NUM;
				C.p = PRIMARY;
				C.col = id->col;
				C.row = id->row;
				C.v = 0;
				isformula=true;
			}
			else
				C = bad_tok("undefined ide");
		}
		else
		if (C.t eq LHELM)
		{
			CELL c;
			double c2 = 343.0*343,
				   pi = M_PI,
				   O = 0.0, V = 0.0, f = 0.0;
			c = pget_c(cell, sh_col, sh_row - 1);
			if (c)
				O = c->val;
			c = pget_c(cell, sh_col, sh_row - 2);
			if (c)
				V = c->val;
			c = pget_c(cell, sh_col, sh_row - 3);
			if (c)
				f = c->val;
			C.v =((c2*O)/(4*pi*pi*f*f*V))-.5*sqrt(pi*O);
			C.t = NUM;
			isformula = true;
		}
		else
		if (C.t eq FHELM)
		{
			CELL c;
			double c2 = 343.0*343,
				   pi = M_PI,
				   O = 0.0,V=0.0,l=0.0;
			c=pget_c(cell,sh_col,sh_row-1);
			if (c)
				l=c->val;
			c=pget_c(cell,sh_col,sh_row-2);
			if (c)
				O=c->val;
			c=pget_c(cell,sh_col,sh_row-3);
			if (c)
				V=c->val;
			C.v=sqrt((c2*O)/(4*pi*pi*V*((sqrt(pi*O)/2)+l)));
			C.t=NUM;
			isformula=true;
		}
	}
	return C;
}

CELL get_c(int col, int row)
{
	return pget_c(cell, col, row);
}

/*			char ftxt[128];
			sprintf(ftxt,"[1][ token %d | %s ][ Ok]", n.t, p->text);
			form_alert(1,ftxt);
*/

/* double shderef(T r, int *attr) */
PARSE_DEREF shderef
{
	CELL p = pget_c(cell, r.col, r.row);
	*attr = 0;
	if (p)
	{
		T n;
		char *save = STR;
		char *t = p->text;
		while(*t >= 'a' and *t <= 'z') t++;		/* simple lower case 'name=' */
		if (*t eq '-' and *(t+1) eq '>')
		{
			t+=2;
			STR = t;
			n = parse();
			if (C.p ne EOLN)
				n = bad_tok("<-text");
			if (is_bad(n))
				*attr = BAD;
			STR = save;
			return n.v;
		}
		elif (*t eq ':' and *(t+1) eq ':')
			return p->val;
	}
	return r.v;
}

double shparse(SH_SHEET c, IDES *ides, char *s, uchar *attr)
{
	T n;
	isformula = false;

	cell=c;

	parsedef(shnext, ides, s, &parsetab, parse_binop, parse_unop, shrange, shderef);

	n = parse();

	if (C.p ne EOLN)		/* other things after what is actually */
		n=bad_tok("txt");	/* itself a valid expression, */
							/* must be treated as text */
	if (n.t eq NUM and isformula)
		*attr=FORM;
	else
		if (n.t eq NUM)
			*attr=VAL;
		else
			*attr=TXT,n.v=0.0;

	return n.v;
}
