/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stdarg.h>

/* #define NDEBUG */
#include <assert.h>

#include "config.h"

#include "a8cas.h"
#include "a8cas_file.h"
#include "format_cas.h"
#include "format_fsk.h"
#include "format_hex.h"
#include "format_sndfile.h"
#include "format_raw.h"
#include "io.h"

enum { DEFAULT_AUDIO_SAMPLERATE = 44100 };
enum { DEFAULT_AUDIO_BITS= 8 };

/* Number of file formats that will be tried when a file is opened with
   A8CAS_FORMAT_ANY. It's 1 less than size of the open_functions array,
   because the RAW format should be used only when the user explicitely
   requests it. */
enum { NUM_TRY_FORMATS = 4 };
#define MAX_FORMAT_NUM A8CAS_FORMAT_RAW

int seek_units[] = {
	FORMAT_CAS_SEEK_UNIT,
	FORMAT_HEX_SEEK_UNIT,
	FORMAT_FSK_SEEK_UNIT,
	FORMAT_SNDFILE_SEEK_UNIT,
	FORMAT_RAW_SEEK_UNIT
};

static int specific_open(unsigned int format, A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	switch(format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_open(file, path, info);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_open(file, path, info);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_open(file, path, info);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_open(file, path, info);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_open(file, path, info);
	}
}

static int specific_close(unsigned int format, A8CAS_FILE *file)
{
	switch(format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_close(file);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_close(file);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_close(file);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_close(file);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_close(file);
	}
}

static int try_open_file(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	if (file->format != A8CAS_FORMAT_ANY) {
		/* Format given by user, check if it matches */
		if (specific_open(file->format, file, path, info) == 0)
			return 0;
		else {
			A8CAS_errnum errnum = file->errnum; /* specific_close resets errnum */
			specific_close(file->format, file);
			file->errnum = errnum;
			file->specific_data = NULL;
			if (errnum == A8CAS_ERR_BADFILE)
				file->errnum = A8CAS_ERR_NOMATCH;
			return 1;
		}
	} else { /* Format not given by user */
		unsigned int i;
		if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
			/* For writing format must ge given explicitly. */
			file->errnum = A8CAS_ERR_INVALID;
			return 1;
		}
		/* Format not given, try opening sequentially.
		   Note: we do NOT recognise the raw format automatically. */
		for (i = 1; i < A8CAS_FORMAT_RAW; i ++) {
			file->errnum = A8CAS_ERR_NONE;
			if (specific_open(i, file, path, info) == 0) {
				file->format = i;
				return 0;
			}
			else {
				A8CAS_errnum errnum = file->errnum;
				specific_close(i, file);
				file->errnum = errnum;
				file->specific_data = NULL;
				if (errnum != A8CAS_ERR_BADFILE)
					return 1;
			}
		}
		assert(file->errnum == A8CAS_ERR_BADFILE);
		return 1;
	}
}

int A8CAS_read(A8CAS_FILE *file, A8CAS_signal *sig)
{
#if 1
	return (*file->read_func)(file, sig);
#else
	/* Debug version */
	int retval = (*file->read_func)(file, sig);
	A8CAS_log(file, "A8CAS_read: %d / %d/%f\n", retval, sig->signal, (double)sig->length / sig->rate);
	return retval;
#endif
}

int A8CAS_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	return (*file->read_with_audio_func)(file, sig, num_samples);
}

int A8CAS_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_read_audio(file, samples, num_samples);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_read_audio(file, samples, num_samples);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_read_audio(file, samples, num_samples);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_read_audio(file, samples, num_samples);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_read_audio(file, samples, num_samples);
	}
}

int A8CAS_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	/* Signals of length 0 cause problems in CAS_ENCODE. So they are
	   ignored a bit earlier. */
	if (sig->length == 0)
		return 0;
	return (*file->write_func)(file, sig);
}

int A8CAS_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	/* Signals of length 0 cause problems in CAS_ENCODE. So they are
	   ignored a bit earlier. */
	if (sig->length == 0)
		return 0;
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_write_audio(file, sig, samples);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_write_audio(file, sig, samples);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_write_audio(file, sig, samples);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_write_audio(file, sig, samples);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_write_audio(file, sig, samples);
	}
}

unsigned int A8CAS_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_read_bytes(file, bytes, size, baudrate);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_read_bytes(file, bytes, size, baudrate);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_read_bytes(file, bytes, size, baudrate);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_read_bytes(file, bytes, size, baudrate);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_read_bytes(file, bytes, size, baudrate);
	}
}

unsigned int A8CAS_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_read_signals(file, sigs, size);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_read_signals(file, sigs, size);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_read_signals(file, sigs, size);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_read_signals(file, sigs, size);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_read_signals(file, sigs, size);
	}
}

int A8CAS_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	return (*file->write_bytes_func)(file, bytes, size, baudrate);
}

int A8CAS_flush(A8CAS_FILE *file)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_flush(file);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_flush(file);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_flush(file);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_flush(file);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_flush(file);
	}
}

A8CAS_FILE* A8CAS_open(char const *path, A8CAS_info *info)
{
	A8CAS_FILE* file;

	/* Check the mode provided by the user */
	if (info->mode < A8CAS_READ || info->mode > A8CAS_WRRD) {
		info->errnum = A8CAS_ERR_INVALID;
		return NULL;
	}

	/* Check the format provided by the user */
	if (info->format > MAX_FORMAT_NUM) {
		info->errnum = A8CAS_ERR_INVALID;
		return NULL;
	}

	/* Allocate the memory for the file pointer */
	if ((file = malloc((size_t)sizeof(*file))) == NULL) {
		info->errnum = A8CAS_ERR_NOMEM;
		return NULL;
	}

	info->errnum = A8CAS_ERR_NONE;
	file->specific_data = NULL;
	file->mode = info->mode;
	file->format = info->format;
	file->description = NULL;
	file->errnum = info->errnum;

	file->log_stream = NULL;

	/* Try "opening" the file: determine its format, allocate memory, load initial data, etc. */
	if (try_open_file(file, path, info) != 0)
	{
		info->errnum = file->errnum;
		A8CAS_close(file);
		return NULL;
	}

	/* Copying additional open information (description, sample rate, etc.) */
	info->mode = file->mode;
	info->format = file->format;
	if (info->mode == A8CAS_READ || info->mode == A8CAS_RDWR) /* Copy read description only when reading */
		info->description = file->description;
	info->errnum = file->errnum;

	file->audio.samplerate = DEFAULT_AUDIO_SAMPLERATE;
	file->audio.bits = DEFAULT_AUDIO_BITS;
	
	{
		/* Set the default crosstalk volume. */
		float vol = 0.05f;
		A8CAS_set_param(file, A8CAS_PARAM_CROSSTALK_VOLUME, &vol);
	}

	return file;
}

int A8CAS_close(A8CAS_FILE *file)
{
	int errnum = 0;

	/* file format = A8CAS_FORMAT_ANY only if an error happened during file open,
	   before allocating the file_specific structure. */
	if (file->format != A8CAS_FORMAT_ANY && specific_close(file->format, file) != 0) {
		errnum = file->errnum;
	}
	/* Free the description read from file. */
	free(file->description);
	free(file);
	return errnum;
}

enum A8CAS_errnum A8CAS_error(A8CAS_FILE *file)
{
	return file->errnum;
}

/* ---- Block-navigation functions ---- */

unsigned long A8CAS_tell(A8CAS_FILE *file)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_tell(file);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_tell(file);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_tell(file);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_tell(file);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_tell(file);
	}
}

unsigned long A8CAS_size(A8CAS_FILE *file)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_size(file);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_size(file);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_size(file);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_size(file);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_size(file);
	}
}

int A8CAS_seek(A8CAS_FILE *file, unsigned long position)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_seek(file, position);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_seek(file, position);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_seek(file, position);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_seek(file, position);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_seek(file, position);
	}
}

int A8CAS_seek_unit(A8CAS_FILE *file)
{
	return seek_units[file->format - 1];
}

/* ---- Audio playback functions ---- */
void A8CAS_set_audio(A8CAS_FILE *file, unsigned int samplerate, unsigned int bits)
{
	file->audio.samplerate = samplerate;
	file->audio.bits = bits;
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		FORMAT_CAS_set_audio(file);
		return;
	case A8CAS_FORMAT_HEX:
		FORMAT_HEX_set_audio(file);
		return;
	case A8CAS_FORMAT_FSK:
		FORMAT_FSK_set_audio(file);
		return;
	case A8CAS_FORMAT_SNDFILE:
		FORMAT_SNDFILE_set_audio(file);
		return;
	default: /* case A8CAS_FORMAT_RAW */
		FORMAT_RAW_set_audio(file);
		return;
	}
}

/* ---- Functions for adjusting encoding/decoding options and parameters ---- */
int A8CAS_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_set_param(file, type, value);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_set_param(file, type, value);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_set_param(file, type, value);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_set_param(file, type, value);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_set_param(file, type, value);
	}
}

int A8CAS_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	switch(file->format) {
	case A8CAS_FORMAT_CAS:
		return FORMAT_CAS_get_param(file, type, value);
	case A8CAS_FORMAT_HEX:
		return FORMAT_HEX_get_param(file, type, value);
	case A8CAS_FORMAT_FSK:
		return FORMAT_FSK_get_param(file, type, value);
	case A8CAS_FORMAT_SNDFILE:
		return FORMAT_SNDFILE_get_param(file, type, value);
	default: /* case A8CAS_FORMAT_RAW */
		return FORMAT_RAW_get_param(file, type, value);
	}
}

/* ---- Functions for turbo systems ---- */
int A8CAS_switch_PWM_decoding(A8CAS_FILE *file, int on)
{
	return A8CAS_set_param(file, A8CAS_PARAM_PWM_DECODE, &on);
}

int A8CAS_set_PWM_tolerance(A8CAS_FILE *file, unsigned short int tolerance)
{
	return A8CAS_set_param(file, A8CAS_PARAM_PWM_TOLERANCE, &tolerance);
}

int A8CAS_PWM_tolerance(A8CAS_FILE *file, unsigned short int *tolerance)
{
	return A8CAS_get_param(file, A8CAS_PARAM_PWM_TOLERANCE, &tolerance);
}

/* ---- Functions for debugging ---- */
void A8CAS_set_log_stream(A8CAS_FILE *file, FILE *stream)
{
	file->log_stream = stream;
}

int A8CAS_log(A8CAS_FILE *file, const char *fmt, ...)
{
	va_list fmt_args;
	int retval = 0;

	if (file->log_stream != NULL) {
		va_start(fmt_args, fmt);
		retval = vfprintf(file->log_stream, fmt, fmt_args);
		va_end(fmt_args);
	}

	return retval;
}

int const A8CAS_version = (A8CAS_VERSION_MAJOR << 16) | A8CAS_VERSION_MINOR;
