/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <string.h>
#include <limits.h>

#include "format_fsk.h"
#include "a8cas_file.h"
#include "a8cas.h"
#include "commons.h"
#include "format_stdio_based.h"
#include "io.h"
#include "read_audio.h"
#include "dyn_array.h"

/*#define NDEBUG*/

/* Minimum length of a MARK signal to be considered an IRG, in seconds */
#define MIN_IRG_LENGTH_S 0.05

#define READ_BUF_SIZE 256

/*#define WRITE_NUM_SAMPLES*/

typedef struct file_specific {
	FILE *file;
	char read_buffer[READ_BUF_SIZE];
	A8CAS_signal prev_sig;
#ifdef WRITE_NUM_SAMPLES
	unsigned int num_samples;
#endif
	/* Array to store block offsets - including offset after the last block */
	long *block_offsets;
	DYN_ARRAY_t block_array;
	long file_offset;
	unsigned int current_block;
	struct {
		double length;
		long offset;
	} prev_mark;
	

	READ_AUDIO_t read_audio;
} file_specific;

/* ------------------------------------------------------------------------ *
 * Block offset control                                                     *
 * ------------------------------------------------------------------------ */

#if 0
static void report_blocks(A8CAS_FILE *file, char const *desc)
{
	unsigned int end = A8CAS_size(file) + 1;
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int i;

	A8CAS_log(file, "%s: current block = %lu\n", desc, A8CAS_tell(file));
	for (i = 0; i < end; i ++) {
		A8CAS_log(file, "Block %u: offset=%lu\n", i, (long unsigned)data->block_offsets[i]);
	}
}
#endif

static void update_current_block_after_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int next_block = data->current_block + 1;

	if (next_block >= data->block_array.fill) {
		return;
	}
	if (data->file_offset >= data->block_offsets[next_block])
		data->current_block = next_block;
}

static void truncate_block_offsets(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	assert(data->block_offsets[data->current_block] <= data->file_offset);

	if (data->block_array.fill == 1) /* Only start block */
		return;

	/* If we are at the last block (that is, file's end) we just decrease
	   the block number - because when writing, we don't want to
	   constantly point to the file's end. */
	if (data->current_block == data->block_array.fill - 1) {
		data->current_block --;
		return;
	}
	/* We are not at the last block. */
	/* If we are at the beginning of the block, then we just truncate the
	   block array. However if we are in the middle of a block, then we
	   assume that the next block offset (maybe the last one, pointing to
	   the file's end) exists, and update its value to the current file
	   offset, thus "moving" the file's end. Then we also truncate. */
	if (data->block_offsets[data->current_block] == data->file_offset) {
		data->block_array.fill = data->current_block + 1;
		/* We decrease the block number - because when writing, we don't want
		   to constantly point to the file's end. */
		if (data->current_block > 0)
			data->current_block --;
	} else {
		/* Not at the block's beginning */
		data->block_offsets[data->current_block + 1] = data->file_offset;
		data->block_array.fill = data->current_block + 2;
	}
}

/* Update (possibly add a new one) block offsets array during a write
   operation. */
static int update_block_offsets_after_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->block_array.fill == 1) {
		if (data->block_offsets[0] != data->file_offset) {
			if (DYN_ARRAY_add(&data->block_array, &data->file_offset) != 0) {
				file->errnum = A8CAS_ERR_NOMEM;
				return 1;
			}
		}
	}
	if (sig->signal == 1) {
		data->prev_mark.length += (double)sig->length / sig->rate;
	} else {
		if (data->prev_mark.length >= MIN_IRG_LENGTH_S &&
		    (data->block_array.fill > 2 || data->prev_mark.offset != data->block_offsets[0])) {
			data->block_offsets[++data->current_block] = data->prev_mark.offset;
			if (DYN_ARRAY_add(&data->block_array, &data->file_offset) != 0) {
				file->errnum = A8CAS_ERR_NOMEM;
				return 1;
			}
		} else
			data->block_offsets[data->current_block + 1] = data->file_offset;
		data->prev_mark.length = 0.0;
		data->prev_mark.offset = data->file_offset;
	}

	return 0;
}

static int flush_data(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 0;
	if (data->prev_sig.length > 0) {
		int written = fprintf(data->file, "%d %u/%u\n", data->prev_sig.signal, data->prev_sig.length, data->prev_sig.rate);
		if (written < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}
		data->file_offset += written;
		if (update_block_offsets_after_write(file, &data->prev_sig) != 0)
			return 1;
		data->prev_sig.length = 0;
	}
	return 0;
}

int FORMAT_FSK_close(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int retval = 0;

	if (data != NULL) {
		if (flush_data(file) != 0)
			retval = 1;
#ifdef WRITE_NUM_SAMPLES
		fprintf(data->file, "%u samples\n", data->num_samples);
#endif
		if (data->block_offsets != NULL)
			free(data->block_offsets);
		if (data->file != NULL && fclose(data->file) != 0) {
			file->errnum = A8CAS_ERR_FCLOSE;
			retval = 1;
		}
		free(data);
	}

	return retval;
}

int FORMAT_FSK_flush(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;
	if (fflush(data->file) != 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Read functions                                                           *
 * ------------------------------------------------------------------------ */

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig);

static int specific_read(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;
	size_t length;
	int signal;

	if (A8CAS_fatal_error(file))
		return 1;
	if (fgets(data->read_buffer, READ_BUF_SIZE, data->file) == NULL) {
		file->errnum = feof(data->file)? A8CAS_ERR_EOF: A8CAS_ERR_FREAD;
		return 1;
	}
	
	length = strlen(data->read_buffer);
	if (data->read_buffer[length - 1] != '\n') {
		file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_LIMIT;
		return 1;
	}

	data->file_offset += length;
	update_current_block_after_read(file);

	if (sscanf(data->read_buffer, "%d %u/%u", &signal, &sig->length, &sig->rate) != 3) {
		file->errnum = A8CAS_ERR_BADFILE;
		return 1;
	}
	if ((signal != 0 && signal != 1) || sig->rate == 0) {
		file->errnum = A8CAS_ERR_BADFILE;
		return 1;
	}
	sig->signal = (char)signal;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int FORMAT_FSK_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

unsigned int FORMAT_FSK_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

static int specific_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Audio playback functions                                                 *
 * ------------------------------------------------------------------------ */

void FORMAT_FSK_set_audio(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_reconfigure(&data->read_audio, file->audio.samplerate, file->audio.bits);
}

static int specific_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	return READ_AUDIO_read_with_audio(&data->read_audio, sig, num_samples);
}

int FORMAT_FSK_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_read_audio(&data->read_audio, samples, num_samples);
	return num_samples;
}

/* ------------------------------------------------------------------------ *
 * Read after write functions                                               *
 * ------------------------------------------------------------------------ */
 
static void reset_before_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read;
	file->read_with_audio_func = &specific_read_with_audio;
	file->write_func = &specific_write_after_read;
	file->errnum = A8CAS_ERR_NONE;
	READ_AUDIO_reset(&data->read_audio);
}

static int common_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;

	/* After writing, the file's offset points at the file's end,
	   but the current block is not the last one. Move the current block
	   pointer to the end. */
	data->current_block = data->block_array.fill - 1;

	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}
	reset_before_read(file);

	return 0;
}

static int specific_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read(file, sig);
}

static int specific_read_with_audio_after_write(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read_with_audio(file, sig, num_samples);
}

/* ------------------------------------------------------------------------ *
 * Write functions                                                          *
 * ------------------------------------------------------------------------ */

static int specific_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	/* Write output only if new signal is different than the previous one. */
	if (sig->signal == data->prev_sig.signal && sig->rate == data->prev_sig.rate) {
		data->prev_sig.length += sig->length;
	} else {
		if (flush_data(file) != 0)
			return 1;
		data->prev_sig = *sig;
	}

#ifdef WRITE_NUM_SAMPLES
	data->num_samples += sig->length;
#endif
	return 0;
}

int FORMAT_FSK_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((*file->write_func)(file, sig) != 0)
		return 1;
	FSK_MOD_generate(&data->read_audio.mod.fsk, samples, sig->length, sig->signal);
	return 0;
}


static void reset_before_write(A8CAS_FILE *file)
{
	file->read_func = &specific_read_after_write;
	file->read_with_audio_func = &specific_read_with_audio_after_write;
	file->write_func = &specific_write;
	file->errnum = A8CAS_ERR_NONE;
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (IO_ftruncate(data->file, data->file_offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	truncate_block_offsets(file);

	reset_before_write(file);

	return specific_write(file, sig);
}

/* ------------------------------------------------------------------------ *
 * Block & seeking functions                                                *
 * ------------------------------------------------------------------------ */

unsigned long FORMAT_FSK_tell(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->current_block;
}

unsigned long FORMAT_FSK_size(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->block_array.fill - 1;
}

int FORMAT_FSK_seek(A8CAS_FILE *file, unsigned long position)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int block_num = (unsigned int) position;

	if (flush_data(file) != 0)
		return 1;
	if (block_num >= data->block_array.fill)
		block_num = data->block_array.fill - 1;
	if (fseek(data->file, data->block_offsets[block_num], SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	clearerr(data->file);

	data->current_block = block_num;
	data->file_offset = data->block_offsets[block_num];

	data->prev_mark.offset = data->file_offset;
	data->prev_mark.length = 0.0;

	reset_before_read(file);
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Functions for adjusting encoding/decoding options and parameters         *
 * ------------------------------------------------------------------------ */
int FORMAT_FSK_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	file_specific *data = (file_specific*) file->specific_data;
	switch (type) {
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		if (READ_AUDIO_set_crosstalk_volume(&data->read_audio, *((float *)value)))
			return 1;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

int FORMAT_FSK_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		*((float *) value) = data->read_audio.crosstalk_volume;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Initialisation functions                                                 *
 * ------------------------------------------------------------------------ */

static int read_description(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	DYN_ARRAY_t array;

	if (DYN_ARRAY_init(&array, 256, sizeof(char), (void **)(&file->description)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* Read description. */
	for (;;) {
		int byte = fgetc(data->file);
		if (byte == EOF) {
			file->errnum = (ferror(data->file) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE);
			return 1;
		}
		{
			char char_byte = (char) byte;
			if (char_byte == '\n')
				break;
			if (DYN_ARRAY_add(&array, &char_byte) != 0) {
				file->errnum = A8CAS_ERR_NOMEM;
				return 1;
			}
		}
	}
	data->file_offset += array.fill + 1; /* +1 because \n was read from file */

	{
		char const nul = '\0';
		if (DYN_ARRAY_add(&array, &nul) != 0 /* End the string with Nul */
		    || DYN_ARRAY_strip(&array) != 0) {
			file->errnum = A8CAS_ERR_NOMEM;
			return 1;
		}
	}
	return 0;
}

static int initialise_block_offsets_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (DYN_ARRAY_init(&data->block_array, 256, sizeof(long), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	if (DYN_ARRAY_add(&data->block_array, &data->file_offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	data->current_block = 0;
	return 0;
}

static int initialise_block_offsets_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	double length = 0.0;
	long offset = data->file_offset;
	/* The first block is an exception - it always points to the first
	   signal in the file, and that signal does not need to be IRG (like
	   in all other blocks save for the last one). This const is used
	   to distinguish the exceptional situation. */
	long const start_offset = offset;
	int prev_signal = 0;
	A8CAS_signal sig;

	/* Assume that at the beginning of this function the file's position
	   indicator lies after the end of the description - that is, just at
	   the beginning of the first block. */
	if (DYN_ARRAY_init(&data->block_array, 256, sizeof(long), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	if (DYN_ARRAY_add(&data->block_array, &start_offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	for (;;) {
		if (specific_read(file, &sig) != 0) {
			if (file->errnum == A8CAS_ERR_EOF) {
				/* If IRG found at end, save offset of it. */
				if (!(prev_signal == 1 && length >= MIN_IRG_LENGTH_S)) {
					/* Otherwise, save offset to end-of-file instead. */
					offset = data->file_offset;
				}
				if (offset != start_offset
				    && DYN_ARRAY_add(&data->block_array, &offset) != 0) {
					file->errnum = A8CAS_ERR_NOMEM;
					return 1;
				}
				/* End-of-file reached. */
				break;
			}
			else /* Read error */
				return 1;
		}
		if (sig.signal == 0) {
			if (prev_signal == 1 && length >= MIN_IRG_LENGTH_S) {
				if (offset != start_offset
				    && DYN_ARRAY_add(&data->block_array, &offset) != 0) {
					file->errnum = A8CAS_ERR_NOMEM;
					return 1;
				}
			}
			length = 0.0;
			offset = data->file_offset;
		} else
			length += (double) sig.length / sig.rate;
		prev_signal = sig.signal;
	}
	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* Jump back to block 0. */
	return FORMAT_FSK_seek(file, 0);
}

int FORMAT_FSK_open(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	struct file_specific *data;

	if ((data = file->specific_data = malloc(sizeof (*data))) == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* NULLize allocatable pointers. */
	data->file = NULL;
	data->block_offsets = NULL;

	file->write_bytes_func = &specific_write_bytes;

	/* Initialize this values so closing the file won't "flush" anything to
	   the disk at this moment. */
	data->prev_sig.length = 0;
	data->prev_sig.rate = 0;

	READ_AUDIO_init(&data->read_audio, file);

	/* Open the file descriptor */
	if ((data->file = FORMAT_STDIO_BASED_open(path, file->mode)) == NULL) {
		file->errnum = A8CAS_ERR_FOPEN;
		return 1;
	}

	if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
		char const *description = (info->description == NULL? "": info->description);
		if ((data->file_offset = fprintf(data->file, "A8CAS-FSK\n%s\n", description)) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}

		if (initialise_block_offsets_write(file) != 0)
			return 1;

		data->prev_mark.offset = data->file_offset;
		data->prev_mark.length = 0.0;

		reset_before_write(file);
	} else { /* file->mode == A8CAS_READ || file->mode == A8CAS_RDWR */
		if (fgets(data->read_buffer, 11, data->file) == NULL) {
			file->errnum = ferror(data->file)? A8CAS_ERR_FREAD: A8CAS_ERR_BADFILE;
			return 1;
		}

		if (strcmp(data->read_buffer, "A8CAS-FSK\n") != 0) {
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
		data->file_offset = 10;
		if (read_description(file) != 0)
			return 1;

		if (initialise_block_offsets_read(file) != 0)
			return 1;
		info->samplerate = 0;

		reset_before_read(file);
	}

#ifdef WRITE_NUM_SAMPLES
	data->num_samples = 0;
#endif
	return 0;
}
