/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>
#include <ctype.h>
#include <inttypes.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>

#include "format_hex.h"

#include "a8cas_file.h"
#include "a8cas.h"
#include "cas_decode.h"
#include "cas_encode.h"
#include "cas_header.h"
#include "dyn_array.h"
#include "format_stdio_based.h"
#include "io.h"
#include "read_audio.h"

#define READ_BUF_SIZE 16
#define FSCANF_BUF_SIZE 21

typedef struct block_offset_t {
	long offset;
	uint16_t baudrate;

	uint16_t pwm_rate;
	int pwm_lsb_first;
	int pwm_rising_edge_first;
} block_offset_t;

typedef struct file_specific {
	FILE *file;
	char read_buffer[READ_BUF_SIZE];

	block_offset_t *block_offsets;
	/* Number of records + 1 (we also hold the offset of the last block's
	 * end): */
	DYN_ARRAY_t block_array;
	unsigned int current_block;
	long current_block_offset;

	int search_for_newline;

	CAS_DECODE_t cas_decode;
	CAS_ENCODE_t cas_encode;
	READ_AUDIO_t read_audio;
} file_specific;

/* ------------------------------------------------------------------------ *
 * Block offset control                                                     *
 * ------------------------------------------------------------------------ */

static void truncate_block_offsets(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	/* We are at the beginning of the block, so we just truncate the
	   block array. */
	data->block_array.fill = data->current_block + 1;
}

static int add_offset_after_write_block(A8CAS_FILE *file, uint16_t baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	block_offset_t offset;

	offset.baudrate = baudrate;
	if ((offset.offset = ftell(data->file)) == -1) {
		file->errnum = A8CAS_ERR_FTELL;
		return 1;
	}
	
	if (DYN_ARRAY_add(&data->block_array, &offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	data->current_block ++;

	return 0;
}

static int flush_data(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 0;
	return CAS_ENCODE_flush(&data->cas_encode);
}

int FORMAT_HEX_close(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int retval = 0;

	if (data != NULL) {
		if (flush_data(file) != 0)
			retval = 1;
		CAS_ENCODE_free(&data->cas_encode);
		if (data->block_offsets != NULL)
			free(data->block_offsets);
		if (data->file != NULL && fclose(data->file) != 0) {
			file->errnum = A8CAS_ERR_FCLOSE;
			retval = 1;
		}
		free(data);
	}

	return retval;
}

int FORMAT_HEX_flush(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;
	if (fflush(data->file) != 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig);
static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate);

static int fscanf_to_whitespace(FILE *stream, const char *fmt, ...)
{
	va_list fmt_args;
	int retval = 0;
	char buf[FSCANF_BUF_SIZE];
	int nonspace = 0;
	unsigned int i;
	for (i = 0; i < FSCANF_BUF_SIZE; i ++) {
		int c = getc(stream);
		if (c == EOF)
			return EOF;
		if (c == '\n') {
			if (ungetc(c, stream) == EOF)
				return EOF;
			break;
		}
		if (c == '\0' || c == ' ') {
			if (nonspace) {
				if (ungetc(c, stream) == EOF)
					return EOF;
				break;
			}
		} else
			nonspace = 1;
		buf[i] = (char) c;
	}
	buf[i] = '\0';

/*printf("buf: %s\n", buf);*/
	va_start(fmt_args, fmt);
	retval = vsscanf(buf, fmt, fmt_args);
	va_end(fmt_args);

	return retval;
	
}

/* ------------------------------------------------------------------------ *
 * CAS_DECODE callback functions                                            *
 * ------------------------------------------------------------------------ */

/* Returns 0 on success, else returns 1 and sets file->errnum */
static int read_cas_header(A8CAS_FILE *file, CAS_HEADER_t *header)
{
	size_t size;
	unsigned int aux;
	file_specific *data = (file_specific*) file->specific_data;

	/* Search for a newline character. */
	if (data->search_for_newline) {
		for (;;) {
			int byte = fgetc(data->file);
			if (byte == EOF) {
				file->errnum = (ferror(data->file) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE);
				return 1;
			}
			if ((char)byte == '\n')
				break;
		}
	} else
		data->search_for_newline = 1;
	if ((data->current_block_offset = ftell(data->file)) == -1) {
		file->errnum = A8CAS_ERR_FTELL;
		return 1;
	}

	size = fread(&header->type, sizeof(uint32_t), 1, data->file);
	if (size == 0) {
		file->errnum = (feof(data->file) ? A8CAS_ERR_EOF: A8CAS_ERR_FREAD);
		/* Since reading can be called several times after end of file,
		   we must assure that the next call won't search for
		   a \n character. */
		data->search_for_newline = 0;
		return 1;
	}
	/* Scan for AUX value */
	switch (header->type) {
	case BLOCK_TYPE_FUJI:
		header->aux = 0;
		break;
	case BLOCK_TYPE_PWMD:
		{
			int result = fscanf_to_whitespace(data->file, "%u", &aux); /* width of impulse "0" */
			if (result != 1) {
				file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
				return 1;
			}
			if (aux > 0xff) {
				file->errnum = A8CAS_ERR_LIMIT;
				return 1;
			}
			header->aux = (uint16_t)aux;
			result = fscanf_to_whitespace(data->file, "%u", &aux); /* width of impulse "1" */
			if (result != 1) {
				file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
				return 1;
			}
			if (aux > 0xff) {
				file->errnum = A8CAS_ERR_LIMIT;
				return 1;
			}
			header->aux |= (uint16_t)aux << 8;
		}
		data->read_audio.mod_type = MOD_TYPE_PWM;
		break;
	case BLOCK_TYPE_PWMS:
		{
			char buf[FSCANF_BUF_SIZE];
			int i;
			aux=0;
			for (i = 0; i < 2; i ++) {
				/* both params can be in any order */
				int result = fscanf_to_whitespace(data->file, "%s", &buf);
				if (result != 1) {
					file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
					return 1;
				}
				aux |= (strcmp(buf,"msb_first") == 0 ? 4 : 
					strcmp(buf,"lsb_first") == 0 ? 0 :
					strcmp(buf,"rising_edge_first") == 0 ? 2 :
					strcmp(buf,"falling_edge_first") == 0 ? 1 : 256);
				if (aux >= 256) {
					/* String not recognised. */
					file->errnum = A8CAS_ERR_BADFILE;
					return 1;
				}
			}
			if ((aux & 3) == 0 || (aux & 3) == 3) {
				/* one and only one edge should be first */
				file->errnum = A8CAS_ERR_BADFILE;
				return 1;
			}
			header->aux = aux;
			break;
		}
	default:
		{
			int result = fscanf_to_whitespace(data->file, "%u", &aux);
			if (result != 1) {
				file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
				return 1;
			}
			if (aux > 0xffff) {
				file->errnum = A8CAS_ERR_LIMIT;
				return 1;
			}
			header->aux = (uint16_t)aux;
			switch (header->type) {
			case BLOCK_TYPE_PWMC:
			case BLOCK_TYPE_PWML:
				data->read_audio.mod_type = MOD_TYPE_PWM;
				break;
			case BLOCK_TYPE_DATA:
			case BLOCK_TYPE_FSK:
				data->read_audio.mod_type = MOD_TYPE_FSK;
			}
		}
	}
	return 0;
}

static int read_data_uint8_hex(A8CAS_FILE *file, uint8_t *byte)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int read_byte;
	int result = fscanf_to_whitespace(data->file, "%x", &read_byte);

	if (result != 1) {
		if (result == EOF && ferror(data->file)) {
			file->errnum = A8CAS_ERR_FREAD;
			return 1;
		}
		/* Else, it's a block's end. */
		data->current_block ++;
		return 2;
	}
	if (read_byte > 0xff) {
		file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}
	*byte = (uint8_t)read_byte;
	return 0;
}

static int read_data_uint16_dec(A8CAS_FILE *file, uint16_t *length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int read_length;
	int result = fscanf_to_whitespace(data->file, "%u", &read_length);

	if (result != 1) {
		if (result == EOF && ferror(data->file)) {
			file->errnum = A8CAS_ERR_FREAD;
			return 1;
		}
		/* Else, it's a block's end. */
		data->current_block ++; /* !!!!!!!!!!!!!! */
		return 2;
	}
	if (read_length > 0xffff) {
		file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}
	*length = (uint16_t)read_length;
	return 0;
}

static int read_pwms_block(A8CAS_FILE *file, uint16_t *rate)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int read_rate;
	int result = fscanf_to_whitespace(data->file, "%u", &read_rate);

	if (result != 1) {
		file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
		return 1;
	}
	if (read_rate > 0xffff) {
		file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}
	*rate = (uint16_t)read_rate;
	return 0;
}

static int read_pwmc_signal(A8CAS_FILE *file, uint8_t *length, uint16_t *count)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int read_val;
	int result = fscanf_to_whitespace(data->file, "%u", &read_val);

	if (result != 1) {
		if (result == EOF && ferror(data->file)) {
			file->errnum = A8CAS_ERR_FREAD;
			return 1;
		}
		/* Else, it's a block's end. */
		data->current_block ++;
		return 2;
	}
	if (read_val > 0xff) {
		file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}
	*length = (uint8_t)read_val;

	result = fscanf_to_whitespace(data->file, "%u", &read_val);

	if (result != 1) {
		file->errnum = (result == EOF && ferror(data->file)) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE;
		return 1;
	}
	if (read_val > 0xffff) {
		file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}
	*count = (uint16_t)read_val;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Read functions                                                           *
 * ------------------------------------------------------------------------ */

static int specific_read(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read(&data->cas_decode, sig);
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int FORMAT_HEX_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_bytes(&data->cas_decode, bytes, size, baudrate);
}

unsigned int FORMAT_HEX_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_signals(&data->cas_decode, sigs, size);
}

static int specific_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write_bytes(&data->cas_encode, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Audio playback functions                                                 *
 * ------------------------------------------------------------------------ */

void FORMAT_HEX_set_audio(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_reconfigure(&data->read_audio, file->audio.samplerate, file->audio.bits);
}


static int specific_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	return READ_AUDIO_read_with_audio(&data->read_audio, sig, num_samples);
}

int FORMAT_HEX_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_read_audio(&data->read_audio, samples, num_samples);
	return num_samples;
}

/* ------------------------------------------------------------------------ *
 * Read after write functions                                               *
 * ------------------------------------------------------------------------ */
 
static void reset_before_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read;
	file->read_with_audio_func = &specific_read_with_audio;
	file->write_func = &specific_write_after_read;
	file->write_bytes_func = &specific_write_bytes_after_read;
	file->errnum = A8CAS_ERR_NONE;
	CAS_DECODE_reset(&data->cas_decode,
			 data->block_offsets[data->current_block].baudrate,
			 data->block_offsets[data->current_block].pwm_rate,
			 data->block_offsets[data->current_block].pwm_lsb_first,
			 data->block_offsets[data->current_block].pwm_rising_edge_first);
	READ_AUDIO_reset(&data->read_audio);
}

static int common_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;

	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}
	reset_before_read(file);

	return 0;
}

static int specific_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read(file, sig);
}

static int specific_read_with_audio_after_write(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read_with_audio(file, sig, num_samples);
}

/* ------------------------------------------------------------------------ *
 * CAS_ENCODE callback functions                                            *
 * ------------------------------------------------------------------------ */

static int write_baud_block(A8CAS_FILE *file, uint16_t baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;
	if (fprintf(data->file, "baud %05"PRIu16"\n", baudrate) < 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

static int write_data_block_bytes(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int i;

	if (fprintf(data->file, "data %05"PRIu16, irg_ms) < 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	for(i = 0; i < length; i ++) {
		if (fprintf(data->file, " %02"PRIx8, buffer[i]) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}
	}

	return 0;
}

static int write_standard_block(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int checksum = 0;
	unsigned int i;

	if (write_data_block_bytes(file, irg_ms, buffer, length) != 0)
		return 1;

	for (i = 0; i < length - 1; i ++) {
		if ((checksum += buffer[i]) > 0xff) {
			checksum++;
			checksum &= 0xff;
		}
	}

	if (fprintf(data->file, " ; standard record; length=%"PRIu16", checksum=%02x ", length, checksum) < 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}

	if (checksum == buffer[length - 1]) {
		if (fprintf(data->file, "OK\n") < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}
	} else {
		if (fprintf(data->file, "BAD\n") < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}
	}
	return 0;
}

static int write_turbo2600_block(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int i;

	/* 12 if there's a header, 0 if there's not a header */
	int vectors_start = 12;

	/* The record must be long enough to fit the Turbo2600 header. */
	while (length >= vectors_start + 10) {
		unsigned int checksum = 0;
		uint16_t start = buffer[vectors_start] + (buffer[vectors_start + 1] << 8);
		uint16_t end = buffer[vectors_start + 2] + (buffer[vectors_start + 3] << 8);
		uint16_t sub_length;
		if (end < start) /* Malformed Turbo2600 block */
			break;

		sub_length = end - start + vectors_start + 10;
		if (sub_length > length) /* Malformed Turbo2600 block */
			break;

		if (write_data_block_bytes(file, irg_ms, buffer, sub_length) != 0)
			return 1;
		irg_ms = 0;

		/* Compute header checksum. */
		for (i = 0; i < vectors_start + 8; i ++) {
			if ((checksum += buffer[i]) > 0xff) {
				checksum++;
				checksum &= 0xff;
			}
		}

		if (fprintf(data->file, " ; turbo2600 record; length=%"PRIu16", header checksum=%02x ", sub_length, checksum) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}

		if (checksum == buffer[vectors_start + 8]) {
			if (fprintf(data->file, "OK") < 0) {
				file->errnum = A8CAS_ERR_FWRITE;
				return 1;
			}
		} else {
			if (fprintf(data->file, "BAD") < 0) {
				file->errnum = A8CAS_ERR_FWRITE;
				return 1;
			}
		}

		/* Compute data checksum. */
		checksum = 0;
		for (i = 9 + vectors_start; i < sub_length - 1; i ++) {
			if ((checksum += buffer[i]) > 0xff) {
				checksum++;
				checksum &= 0xff;
			}
		}

		if (fprintf(data->file, "; data checksum=%02x ", checksum) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}

		if (checksum == buffer[sub_length - 1]) {
			if (fprintf(data->file, "OK\n") < 0) {
				file->errnum = A8CAS_ERR_FWRITE;
				return 1;
			}
		} else {
			if (fprintf(data->file, "BAD\n") < 0) {
				file->errnum = A8CAS_ERR_FWRITE;
				return 1;
			}
		}

		if (buffer[vectors_start + 6] & 0x01)
			vectors_start = 12;
		else
			vectors_start = 0;
		buffer += sub_length;
		length -= sub_length;
	}

	if (length > 0) /* Malformed Turbo2600 block */
		return write_standard_block(file, irg_ms, buffer, length);
	return 0;
}

static int write_data_block(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	if (length >= 26) {
		static uint8_t const t2600_headers[2][12] = {
			/* Turbo 2600 block of variable length */
			{ 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfc, 0xfc, 0xf8, 0xf0, 0xc7 },
			/* Turbo 2600 block of fixed length */
			{ 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfc, 0xfc, 0xf8, 0xf0, 0xc7 }
		};
		if (memcmp(buffer, t2600_headers[0], 12) == 0 || memcmp(buffer, t2600_headers[1], 12) == 0)
			return write_turbo2600_block(file, irg_ms, buffer, length);
	}
	return write_standard_block(file, irg_ms, buffer, length);
}

static int write_fsk_block(A8CAS_FILE *file, uint16_t irg_ms, uint16_t *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned long duration = 0;
	unsigned int i;

	if (fprintf(data->file, "fsk  %05"PRIu16, irg_ms) < 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	for(i = 0; i < length; i ++) {
		if (fprintf(data->file, " %"PRIu8, buffer[i]) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}
		duration += (unsigned long)buffer[i];
	}

	if (fprintf(data->file, " ; length=%"PRIu16"; duration=%lu ms\n", length, duration/10) < 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}

	return 0;
}

/* ------------------------------------------------------------------------ *
 * Write functions                                                          *
 * ------------------------------------------------------------------------ */

static int specific_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write(&data->cas_encode, sig);
}

int FORMAT_HEX_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((*file->write_func)(file, sig) != 0)
		return 1;
	FSK_MOD_generate(&data->read_audio.mod.fsk, samples, sig->length, sig->signal);
	return 0;
}

static void reset_before_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read_after_write;
	file->read_with_audio_func = &specific_read_with_audio_after_write;
	file->write_func = &specific_write;
	file->write_bytes_func = &specific_write_bytes;
	file->errnum = A8CAS_ERR_NONE;
	CAS_ENCODE_reset(&data->cas_encode, data->block_offsets[data->current_block].baudrate);
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (FORMAT_HEX_seek(file, data->current_block) != 0)
		return 1;
	if (IO_ftruncate(data->file, data->block_offsets[data->current_block].offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	truncate_block_offsets(file);

	reset_before_write(file);

	return specific_write(file, sig);
}

static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (FORMAT_HEX_seek(file, data->current_block) != 0)
		return 1;
	if (IO_ftruncate(data->file, data->block_offsets[data->current_block].offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	truncate_block_offsets(file);

	reset_before_write(file);

	return specific_write_bytes(file, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Block & seeking functions                                                *
 * ------------------------------------------------------------------------ */

unsigned long FORMAT_HEX_tell(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->current_block;
}

unsigned long FORMAT_HEX_size(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	return (unsigned long)data->block_array.fill - 1;
}

int FORMAT_HEX_seek(A8CAS_FILE *file, unsigned long position)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int block_num = (unsigned int) position;

	if (flush_data(file) != 0)
		return 1;
	if (block_num >= data->block_array.fill)
		block_num = data->block_array.fill - 1;
	if (fseek(data->file, data->block_offsets[block_num].offset, SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	clearerr(data->file);

	data->current_block = block_num;
	data->current_block_offset = data->block_offsets[block_num].offset;
	data->search_for_newline = 0;

	reset_before_read(file);
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Functions for adjusting encoding/decoding options and parameters         *
 * ------------------------------------------------------------------------ */
int FORMAT_HEX_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		if (CAS_ENCODE_set_block_header_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		if (CAS_ENCODE_set_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		if (CAS_ENCODE_set_stop_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		if (CAS_ENCODE_set_bit_middle(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		if (CAS_ENCODE_set_block_header_length(&data->cas_encode, *((unsigned int *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		if (CAS_ENCODE_set_baudrate_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		if (READ_AUDIO_set_crosstalk_volume(&data->read_audio, *((float *)value)))
			return 1;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

int FORMAT_HEX_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		 *((double *) value) = data->cas_encode.block_header_deviation;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.bit_deviation;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.stop_bit_deviation;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		 *((double *) value) = data->cas_encode.bit_middle;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		 *((unsigned int *) value) = data->cas_encode.block_header_length;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		 *((double *) value) = data->cas_encode.baudrate_deviation;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		*((float *) value) = data->read_audio.crosstalk_volume;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Initialisation functions                                                 *
 * ------------------------------------------------------------------------ */

static int initialise_block_offsets_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	block_offset_t offset;

	offset.offset = data->current_block_offset;
	offset.baudrate = 0;
	offset.pwm_lsb_first = 0;
	offset.pwm_rising_edge_first = 0;

	
	if (DYN_ARRAY_init(&data->block_array, 256, sizeof(block_offset_t), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	if (DYN_ARRAY_add(&data->block_array, &offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	data->current_block = 0;
	return 0;
}

static int read_description(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	DYN_ARRAY_t array;

	/*if (file->description != NULL) ?*/
	free(file->description);

	if (DYN_ARRAY_init(&array, 256, sizeof(char), (void **)(&file->description)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* Read description. */
	for (;;) {
		int byte;
		if ((byte = fgetc(data->file)) == EOF) {
			file->errnum = (ferror(data->file) ? A8CAS_ERR_FREAD : A8CAS_ERR_BADFILE);
			return 1;
		}
		{
			char char_byte = (char) byte;
			if (char_byte == '\n')
				break;
			if (array.fill == 0 && char_byte == ' ') /* skip first space */
				continue;
			if (DYN_ARRAY_add(&array, &char_byte) != 0) {
				file->errnum = A8CAS_ERR_NOMEM;
				return 1;
			}
		}
	}
	data->search_for_newline = 0;

	{
		char const nul = '\0';
		if (DYN_ARRAY_add(&array, &nul) != 0 /* End the string with Nul */
		    || DYN_ARRAY_strip(&array) != 0) {
			file->errnum = A8CAS_ERR_NOMEM;
			return 1;
		}
	}
	return 0;
}

static int initialise_block_offsets_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	block_offset_t cur_block;

	int after_ignored_block = 0;

	cur_block.baudrate = 0;
	cur_block.pwm_lsb_first = 0;
	cur_block.pwm_rising_edge_first = 0;

	if (DYN_ARRAY_init(&data->block_array, 10, sizeof(block_offset_t), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	for (;;) {
		/* Read next block. */
		CAS_HEADER_t header;
		if (read_cas_header(file, &header) != 0) {
			if (file->errnum == A8CAS_ERR_EOF) {
				/* If no block left, add the offset of the
				   place after the last block. */
				cur_block.offset = data->current_block_offset;
				if (!after_ignored_block &&
				    DYN_ARRAY_add(&data->block_array, &cur_block) != 0) {
					file->errnum = A8CAS_ERR_NOMEM;
					return 1;
				}
				break;
			}
			else /* Read error */
				return 1;
		}
		/* Add new offset to the table. */
		cur_block.offset = data->current_block_offset;
		if (!after_ignored_block &&
		    DYN_ARRAY_add(&data->block_array, &cur_block) != 0) {
			file->errnum = A8CAS_ERR_NOMEM;
			return 1;
		}
		switch (header.type) {
		case BLOCK_TYPE_BAUD:
			/* If it was a BAUD block, get the baudrate and ignore the
			   block in the offsets' table. */
			cur_block.baudrate = header.aux;
			after_ignored_block = 1;
			break;
		case BLOCK_TYPE_PWMS:
			{
				uint16_t read_rate;
				if (read_pwms_block(file, &read_rate) != 0)
					return 1;
				cur_block.pwm_rate = read_rate;
				cur_block.pwm_lsb_first = ( header.aux & 0x4 ) == 0;
				cur_block.pwm_rising_edge_first = ( header.aux & 0x3 ) >> 1;
				after_ignored_block = 1;
			break;
			}
		case BLOCK_TYPE_DATA:
		case BLOCK_TYPE_FSK:
		case BLOCK_TYPE_PWMD:
		case BLOCK_TYPE_PWMC:
		case BLOCK_TYPE_PWML:
			after_ignored_block = 0;
			break;
		case BLOCK_TYPE_FUJI:
			after_ignored_block = 1;
			break;
		default:
			/* don't do bad things */
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
	}
	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* Jump back to block 0. */
	return FORMAT_HEX_seek(file, 0);
}

int FORMAT_HEX_open(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	file_specific *data;

	if ((data = file->specific_data = malloc(sizeof (*data))) == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* NULLize allocatable pointers. */
	data->file = NULL;
	data->block_offsets = NULL;
	CAS_ENCODE_init(&data->cas_encode);

	READ_AUDIO_init(&data->read_audio, file);

	data->cas_decode.file = file;
	data->cas_decode.read_header_func = &read_cas_header;
	data->cas_decode.read_fsk_signal_func = &read_data_uint16_dec;
	data->cas_decode.read_data_byte_func = &read_data_uint8_hex;
	data->cas_decode.read_pwms_block_func = &read_pwms_block;
	data->cas_decode.read_pwmd_byte_func = &read_data_uint8_hex;
	data->cas_decode.read_pwmc_signal_func = &read_pwmc_signal;
	data->cas_decode.read_pwml_signal_func = &read_data_uint16_dec;
	data->cas_decode.read_description_func = &read_description;

	data->cas_encode.file = file;
	data->cas_encode.write_baud_block_func = &write_baud_block;
	data->cas_encode.write_data_block_func = &write_data_block;
	data->cas_encode.write_fsk_block_func = &write_fsk_block;
	data->cas_encode.add_block_offset_func = &add_offset_after_write_block;

	data->read_audio.mod_type = MOD_TYPE_FSK;

	/* Open the file descriptor */
	if ((data->file = FORMAT_STDIO_BASED_open(path, file->mode)) == NULL) {
		file->errnum = A8CAS_ERR_FOPEN;
		return 1;
	}

	/* Initialise to avoid errors when flush_data() is called. */
	if (CAS_ENCODE_alloc(&data->cas_encode) != 0)
		return 1;
	CAS_ENCODE_reset(&data->cas_encode, 0);

	if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
		char const *description = (info->description == NULL? "": info->description);
		if ((data->current_block_offset = fprintf(data->file, "A8CAS-HEX\nFUJI %s\n", description)) < 0) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}

		if (initialise_block_offsets_write(file) != 0)
			return 1;
		reset_before_write(file);
	} else { /* file->mode == A8CAS_READ || file->mode == A8CAS_RDWR */
		CAS_HEADER_t header;
		file->description = NULL;

		if (fgets(data->read_buffer, 11, data->file) == NULL) {
			file->errnum = ferror(data->file)? A8CAS_ERR_FREAD: A8CAS_ERR_BADFILE;
			return 1;
		}

		if (strcmp(data->read_buffer, "A8CAS-HEX\n") != 0) {
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}

		data->search_for_newline=0; /* fgets got newline already */ 

		if (read_cas_header(file, &header) != 0 || header.type != BLOCK_TYPE_FUJI) {
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
		if (read_description(file) != 0)
			return 1;

		if (initialise_block_offsets_read(file) != 0)
			return 1;

		info->samplerate = 0;
		reset_before_read(file);
	}

	return 0;
}
