/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <string.h>

#include "fsk_demod.h"
#include "downsample.h"

enum { MIN_THRESHOLD = 3 };
float const FSK_DEMOD_DEFAULT_SILENCE_THRESHOLD = 0.005f;

/*#define SND_DEBUG*/
#ifdef SND_DEBUG
#include <sndfile.h>
static SNDFILE *snd1;
static short int buf1[1024];
static unsigned int pos = 0;

#endif

void FSK_DEMOD_init(FSK_DEMOD_t *demod, unsigned int samplerate)
{
	DOWNSAMPLE_init(&demod->downsample, samplerate, FSK_DEMOD_downsample_rate);

	demod->prev_signal = 1;
	demod->denoise_threshold = 0;
	demod->min_silence_length = 6;
	demod->silence_left = 0;

	demod->index = 0;
	memset(demod->history, 0, sizeof (demod->history));
	memset(demod->history_sum8, 0, sizeof (demod->history_sum8));
	memset(demod->history_sum6, 0, sizeof (demod->history_sum6));
#ifdef SND_DEBUG
	{
		SF_INFO info;
		info.format = SF_FORMAT_WAV | SF_FORMAT_PCM_16;
		info.samplerate = samplerate;
		info.channels = 1;
		snd1 = sf_open("snd1.wav", SFM_WRITE, &info);
	}
#endif
}

static int advance_downsampled(FSK_DEMOD_t *demod, short int sample)
{
	int const index = demod->index;

	float * const hist = &demod->history[index];
	float * const hist8 = &demod->history_sum8[index];
	float * const hist6 = &demod->history_sum6[index];

	demod->index = (index - 1) & 31;

	/* A circular buffer of size 32 implemented by duplicating the 32-size
	   buffer two times in a 64-size array. */
	hist[0] = hist[32] = ((float)sample + 0.5)/ 32767.5;
	hist8[0] = hist8[32] = hist[0] + hist[8] + hist[16];
	hist6[0] = hist6[32] = hist[0] + hist[6] + hist[12] + hist[18];

	{
	/* This is an optimised verion of:
	   float const sin0 = hist8[0]*sin(0)  + hist8[1]*sin(pi/4)   + hist8[2]*sin(pi/2)   + hist8[3]*sin(3*pi/4) +
	               hist8[4]*sin(pi) + hist8[5]*sin(5*pi/4) + hist8[6]*sin(3*pi/2) + hist8[7]*sin(7*pi/4);
	   float const cos0 = hist8[0]*cos(0)  + hist8[1]*cos(pi/4)   + hist8[2]*cos(pi/2)   + hist8[3]*cos(3*pi/4) +
	               hist8[4]*cos(pi) + hist8[5]*cos(5*pi/4) + hist8[6]*cos(3*pi/2) + hist8[7]*cos(7*pi/4);
	 */
	static float const sin1_4pi = 0.707107f; /* sin(pi/4) = sin(3*pi/4) = -sin(5*pi/4) = -sin(7*pi/4) */
	float const hist8_1m5 = hist8[1] - hist8[5];
	float const hist8_3m7 = hist8[3] - hist8[7];
	float const sin0 = (hist8_1m5 + hist8_3m7) * sin1_4pi + hist8[2] - hist8[6];
	float const cos0 = (hist8_1m5 - hist8_3m7) * sin1_4pi + hist8[0] - hist8[4];

	/* And this is an optimised version of:
	   float const sin1 = hist6[0]*sin(0)  + hist6[1]*sin(pi/3)   + hist6[2]*sin(2*pi/3) +
	               hist6[3]*sin(pi) + hist6[4]*sin(4*pi/3) + hist6[5]*sin(5*pi/3);
	   cloat const cos1 = hist6[0]*cos(0)  + hist6[1]*cos(pi/3)   + hist6[2]*cos(2*pi/3) +
	               hist6[3]*cos(pi) + hist6[4]*cos(4*pi/3) + hist6[5]*cos(5*pi/3);
	 */
	static float const cos1_3pi = 0.5f; /* cos(pi/3)=-cos(2*pi/3)=-cos(4*pi/3)=cos(5*pi/3) */
	static float const sin1_3pi = 0.86602540378443864676372317075294f; /* sin(pi/3)=sin(2*pi/3)=-sin(4*pi/3)=-sin(5*pi/3) */
	float const hist6_1m4 = hist6[1] - hist6[4];
	float const hist6_2m5 = hist6[2] - hist6[5];
	float const sin1 = (hist6_1m4 + hist6_2m5) * sin1_3pi;
	float const cos1 = (hist6_1m4 - hist6_2m5) * cos1_3pi + hist6[0] - hist6[3];

	/* Each entry in history_sum8 is a sum of 3 samples. Each entry in history_sum6 is a sum of 4 samples.
	   Therefore we need to re-scale the resulting ZERO value by (3/4)^2 = 0.5625 to match the "weight"
	   of the ONE value. */
	float one = (sin1 * sin1 + cos1 * cos1);
	float zero = (sin0 * sin0 + cos0 * cos0) * 0.5625;

	int signal = (one > zero);

/* Initial denoising - do #if 0 to turn it off. */
#if 1
	if (signal != demod->prev_signal) {
		if (demod->denoise_threshold < MIN_THRESHOLD) {
			demod->denoise_threshold ++;
			signal = demod->prev_signal;
		} else {
			demod->denoise_threshold = 0;
			demod->prev_signal = signal;
		}
	} else {
		if (demod->denoise_threshold > 0) {
			demod->denoise_threshold --;
		}
	}
#else
	demod->prev_signal = signal;
#endif


	return signal;
	}
}

/* Detect silence longer than 6 samples (32000Hz). */
static int signal_silence(FSK_DEMOD_t *demod)
{
	unsigned int i = demod->index + 1;
	float dif = demod->history[i] - demod->history[i + 1];
	float th = demod->silence_threshold;
	if (demod->silence_left == 0) {
		/* Silence not yed kicked in. */
		if (dif > th || dif < -th) {
			demod->min_silence_length = 6;
			return 0;
		} else if (--demod->min_silence_length == 0) {
			demod->silence_left = 32;
			return 1;
		}
		return 0;
	} else {
		/* Silence already kicked in and active. */
		if (dif > th || dif < -th) {
			demod->min_silence_length = 6;
			return (--demod->silence_left != 0);
		} else if (demod->min_silence_length > 0)
			demod->min_silence_length --;
		demod->silence_left = 32;
		return 1;
	}
}

int FSK_DEMOD_advance(FSK_DEMOD_t *demod, short int sample)
{
	int sig;
	if (DOWNSAMPLE_next(&demod->downsample, &sample) != 0)
		sig = advance_downsampled(demod, sample);
	else
		sig = demod->prev_signal;

	if (signal_silence(demod))
		sig = 1;

#ifdef SND_DEBUG
	{
		buf1[pos ++] = sig * 32000 - 16000;
		if (pos == 1024) {
			sf_write_short(snd1, buf1, 1024);
			pos = 0;
		}
	}
#endif
	return sig;
}
