/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#ifndef FSK_DEMOD_H
#define FSK_DEMOD_H

#include "downsample.h"

/* For saving on Atari tapes, POKEY's sound generator is used. POKEY's base
   frequency is set to 1789790/28 Hz (AUDCTL register set to 0). (The value
   given is for NTSC - however the difference between NTSC and PAL frequencies
   is negligible). The frequencies saved on tape are created by POKEY with
   AUDF register set to 5 for MARK tone and 7 for SPACE tone. The formula to
   compute frequencies is BASE_FREQ/(2*(AUDF+1)).

   The outcome is, that if we sampled an audio input with frequency of
   1789790/28/2 Hz, then lengths of both full MARK and SPACE periods would be
   an integer - exactly 6 and 8 samples, respectively.

   Having a sample rate that is an integreal multipler of a frequency's period
   is mandatory to construct a matched filter, so we downsample the audio
   to a 1789790/28/2 Hz for that very reason. */
enum { FSK_DEMOD_downsample_rate = 1789790/28/2 };
extern float const FSK_DEMOD_DEFAULT_SILENCE_THRESHOLD; /* 0.005 */

typedef struct FSK_DEMOD_t {
	int index;
	float history[64];
	float history_sum8[64];
	float history_sum6[64];

	int prev_signal;
	unsigned int denoise_threshold;

	/* If 6 consecutive samples differ by less than this value, the signal is considered silence,
	   and MARK is returned. Then there have to be 32 "non-silent" samples to return to normal operation. */
	float silence_threshold;
	unsigned int min_silence_length; /* Number of silent samples (initially 6) that must pass for "silence mode" to kick in. */
	unsigned int silence_left; /* Number of samples that must still pass before "silence mode" is deactivated. */

	DOWNSAMPLE_t downsample;
} FSK_DEMOD_t;

void FSK_DEMOD_init(FSK_DEMOD_t *demod, unsigned int samplerate);

int FSK_DEMOD_advance(FSK_DEMOD_t *demod, short int sample);
#endif /* FSK_DEMOD_H */