/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <math.h>
#include <limits.h>

#include "pwm_mod.h"

static void generate_8bit(PWM_MOD_t *mod, void *buffer, unsigned int num_samples, int signal)
{
	float volume = mod->volume;
	unsigned int i, j;
	unsigned char *buf = (unsigned char *)buffer;

	for (i = 0, j = mod->write_channel; i < num_samples; i ++, j += mod->num_channels) {
		buf[i] = (unsigned char)((signal ? 1.0f : 0.0f) * volume);
	}
}

static void generate_16bit(PWM_MOD_t *mod, void *buffer, unsigned int num_samples, int signal)
{
	static float const add = (float) (SHRT_MAX + SHRT_MIN) / 2.0f;
	float volume = mod->volume;
	unsigned int i, j;
	short *buf = (short *)buffer;

	for (i = 0, j = mod->write_channel; i < num_samples; i ++, j += mod->num_channels) {
		buf[j] = (short)((signal ? 1.0f : -1.0f) * volume + add);
	}
}

void PWM_MOD_init(PWM_MOD_t *mod, unsigned int samplerate, unsigned int bits, unsigned int num_channels, unsigned int write_channel)
{
	if (bits == 16)
		mod->gen_func = &generate_16bit;
	else /* bits == 8 */
		mod->gen_func = &generate_8bit;
	mod->num_channels = num_channels;
	mod->write_channel = write_channel;
	PWM_MOD_set_volume(mod, 1.0f);
}

void PWM_MOD_set_volume(PWM_MOD_t *mod, float volume)
{
	/* Scale volume to the chosen bitrate. */
	if (mod->gen_func == &generate_16bit)
		mod->volume = volume * (float) (SHRT_MAX - SHRT_MIN) / 2.0f;
	else
		mod->volume = volume * (float) UCHAR_MAX / 2.0f;
}
 
void PWM_MOD_generate(PWM_MOD_t *mod, void *buffer, unsigned int num_samples, int signal)
{
	(*mod->gen_func)(mod, buffer, num_samples, signal);
}
