/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include "read_audio.h"
#include "a8cas_file.h"
#include "a8cas.h"
#include "fsk_mod.h"
#include "pwm_mod.h"
#include "commons.h"

void READ_AUDIO_init(READ_AUDIO_t *read_audio, A8CAS_FILE *file)
{
	read_audio->file = file;
	read_audio->samplerate = 0;
	read_audio->pending_samples = 0.0;
	read_audio->pending_signal = 1;
	read_audio->mod_type = MOD_TYPE_FSK;
}

void READ_AUDIO_reconfigure(READ_AUDIO_t *read_audio, unsigned int samplerate, unsigned int bits)
{
	read_audio->samplerate = samplerate;
	read_audio->pending_samples = 0.0;
	FSK_MOD_init(&read_audio->mod.fsk, samplerate, bits, 1, 0, A8CAS_MARK_FREQ, A8CAS_SPACE_FREQ);
	PWM_MOD_init(&read_audio->mod.pwm, samplerate, bits, 1, 0);
	READ_AUDIO_set_crosstalk_volume(read_audio, read_audio->crosstalk_volume);
}

void READ_AUDIO_reset(READ_AUDIO_t *read_audio)
{
/*	read_audio->pending_samples = 0.0;*/
}

int READ_AUDIO_read_with_audio(READ_AUDIO_t *read_audio, A8CAS_signal *sig, unsigned int *num_samples)
{
	double num_samples_with_frac;
	if ((*read_audio->file->read_func)(read_audio->file, sig) != 0) /* Read error */
		return 1;
	read_audio->pending_signal = sig->signal;
	num_samples_with_frac = (double)sig->length * read_audio->samplerate / sig->rate + read_audio->pending_samples;
	*num_samples = (unsigned int) num_samples_with_frac;
	read_audio->pending_samples = num_samples_with_frac - *num_samples;
	return 0;
}

void READ_AUDIO_read_audio(READ_AUDIO_t *read_audio, void *samples, unsigned int num_samples)
{
	if (read_audio->mod_type == MOD_TYPE_FSK)
		FSK_MOD_generate(&read_audio->mod.fsk, samples, num_samples, read_audio->pending_signal);
	else
		PWM_MOD_generate(&read_audio->mod.pwm, samples, num_samples, read_audio->pending_signal);
}

int READ_AUDIO_set_crosstalk_volume(READ_AUDIO_t *read_audio, float value)
{
	FSK_MOD_set_volume(&read_audio->mod.fsk, value);
	PWM_MOD_set_volume(&read_audio->mod.pwm, value);
	read_audio->crosstalk_volume = value;
	return 0;
}
