/*
 * img_tape_a8cas.c - support for tape files using liba8cas
 *
 * Copyright (C) 2011 Tomasz Krasuski
 * Copyright (C) 2011 Atari800 development team (see DOC/CREDITS)
 *
 * This file is part of the Atari800 emulator project which emulates
 * the Atari 400, 800, 800XL, 130XE, and 5200 8-bit computers.
 *
 * Atari800 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Atari800 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Atari800; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <a8cas.h>

#include "atari.h"
#include "config.h"
#include "img_tape.h"
#include "memory.h"
#include "util.h"

/* Index of each entry in this array correspond to a value in IMG_TAPE_format_t. */
static A8CAS_format format_conv[] = {
	A8CAS_FORMAT_CAS,
	A8CAS_FORMAT_RAW,
	A8CAS_FORMAT_SNDFILE
};

struct IMG_TAPE_t {
	A8CAS_FILE *file;
	unsigned int samplerate;
#ifdef SYNCHRONIZED_SOUND
	unsigned int audio_samplerate;
#endif
};

int IMG_TAPE_FileSupported(char const *filename, UBYTE const start_bytes[4])
{
	A8CAS_info cas_info;
	A8CAS_FILE *file;

	cas_info.format = A8CAS_FORMAT_ANY;
	cas_info.mode = A8CAS_RDWR;
	if ((file = A8CAS_open(filename, &cas_info)) == NULL)
		return FALSE;
	A8CAS_close(file);
	return TRUE;
}

IMG_TAPE_t *IMG_TAPE_Open(char const *filename, int *writable, int *is_raw, char const **description)
{
	A8CAS_info cas_info;
	IMG_TAPE_t *img;
	A8CAS_FILE *file;

	cas_info.format = A8CAS_FORMAT_ANY;
	cas_info.mode = A8CAS_RDWR;
	if ((file = A8CAS_open(filename, &cas_info)) == NULL) {
		/* Try opening read-only. */
		cas_info.mode = A8CAS_READ;
		if ((file = A8CAS_open(filename, &cas_info)) == NULL) {
			/* Try opening as a raw file. */
			cas_info.mode = A8CAS_RDWR;
			cas_info.format = A8CAS_FORMAT_RAW;
			if ((file = A8CAS_open(filename, &cas_info)) == NULL) {
				/* Last attempt, try opening as a raw read-only file. */
				cas_info.mode = A8CAS_READ;
				if ((file = A8CAS_open(filename, &cas_info)) == NULL)
					return NULL;
			}
		}
	}
	img = Util_malloc(sizeof(IMG_TAPE_t));
	img->file = file;
	img->samplerate = cas_info.samplerate;

	*writable = cas_info.mode == A8CAS_RDWR;
	*is_raw = cas_info.format == A8CAS_FORMAT_RAW;
	*description = cas_info.description;

	return img;
}

unsigned int IMG_TAPE_GetSamplerate(IMG_TAPE_t const *file)
{
	return file->samplerate;
}

void IMG_TAPE_Close(IMG_TAPE_t *file)
{
	A8CAS_close(file->file);
	free(file);
}

IMG_TAPE_t *IMG_TAPE_Create(char const *filename, IMG_TAPE_format_t format, char const *description)
{
	A8CAS_info cas_info;
	IMG_TAPE_t *img;
	A8CAS_FILE *file;

	cas_info.format = format_conv[format];
	cas_info.samplerate = 44100;
	cas_info.description = description;
	cas_info.mode = A8CAS_WRRD;
	if ((file = A8CAS_open(filename, &cas_info)) == NULL || cas_info.mode != A8CAS_WRRD)
		return NULL;

	img = Util_malloc(sizeof(IMG_TAPE_t));
	img->file = file;
	img->samplerate = cas_info.samplerate;

	return img;
}

int IMG_TAPE_Read(IMG_TAPE_t *file, unsigned int *length, int *signal)
{
	A8CAS_signal sig;

	if (A8CAS_read(file->file, &sig) != 0)
		return FALSE;

	*length = (Atari800_tv_mode == Atari800_TV_NTSC ? Atari800_FPS_NTSC : Atari800_FPS_PAL) * 114 * Atari800_tv_mode * sig.length / sig.rate;
	*signal = sig.signal;

	return TRUE;
}

int IMG_TAPE_Write(IMG_TAPE_t *file, unsigned int length, int signal)
{
	A8CAS_signal sig;
	sig.signal = signal;
	sig.length = length;
	sig.rate = (Atari800_tv_mode == Atari800_TV_NTSC ? Atari800_FPS_NTSC : Atari800_FPS_PAL) * 114 * Atari800_tv_mode;

	return A8CAS_write(file->file, &sig) == 0;
}

int IMG_TAPE_Flush(IMG_TAPE_t *file)
{
	return A8CAS_flush(file->file) == 0;
}

void IMG_TAPE_SetPWMDecode(IMG_TAPE_t *file, int value)
{
	A8CAS_set_param(file->file, A8CAS_PARAM_PWM_DECODE, &value);
}

void IMG_TAPE_SetPWMTolerance(IMG_TAPE_t *file, unsigned short int value)
{
	A8CAS_set_param(file->file, A8CAS_PARAM_PWM_TOLERANCE, &value);
}

#ifdef SYNCHRONIZED_SOUND
void IMG_TAPE_SetCrosstalkVolume(IMG_TAPE_t *file, float value)
{
	A8CAS_set_param(file->file, A8CAS_PARAM_CROSSTALK_VOLUME, &value);
}
#endif

int IMG_TAPE_PositionInBlocks(IMG_TAPE_t *file)
{
	return A8CAS_seek_unit(file->file) == A8CAS_SEEK_UNIT_BLOCKS;
}

unsigned int IMG_TAPE_GetPosition(IMG_TAPE_t *file)
{
	return (unsigned int)A8CAS_tell(file->file);
}

unsigned int IMG_TAPE_GetSize(IMG_TAPE_t *file)
{
	return (unsigned int)A8CAS_size(file->file);
}

void IMG_TAPE_Seek(IMG_TAPE_t *file, unsigned int position)
{
	A8CAS_seek(file->file, (unsigned long)position);
}

#ifdef SYNCHRONIZED_SOUND
void IMG_TAPE_SetAudio(IMG_TAPE_t *file, unsigned int samplerate, unsigned int bits)
{
	A8CAS_set_audio(file->file, samplerate, bits);
	file->audio_samplerate = samplerate;
}

int IMG_TAPE_WriteAudio(IMG_TAPE_t *file, unsigned int num_samples, int signal, UBYTE *audio_buffer)
{
	A8CAS_signal sig;
	sig.signal = signal;
	sig.length = num_samples;
	sig.rate = file->audio_samplerate;

	return A8CAS_write_audio(file->file, &sig, audio_buffer) == 0;
}

int IMG_TAPE_ReadWithAudio(IMG_TAPE_t *file, unsigned int *length, int *signal, unsigned int *num_samples)
{
	A8CAS_signal sig;

	if (A8CAS_read_with_audio(file->file, &sig, num_samples) != 0)
		return FALSE;

	*length = (Atari800_tv_mode == Atari800_TV_NTSC ? Atari800_FPS_NTSC : Atari800_FPS_PAL) * 114 * Atari800_tv_mode * sig.length / sig.rate;
	*signal = sig.signal;

	return TRUE;
}

int IMG_TAPE_ReadAudio(IMG_TAPE_t *file, UBYTE *audio_buffer, unsigned int num_samples)
{
	return A8CAS_read_audio(file->file, audio_buffer, num_samples);
}
#endif

int IMG_TAPE_SkipToData(IMG_TAPE_t *file, int ms)
{
	A8CAS_signal sig;
	double delay_s = (double) ms / 1000.0; /* delay in seconds, in floating-point */

	while (delay_s > 0.0) {
		/* Skip signals. */
		if (A8CAS_read(file->file, &sig) != 0)
			/* Ignore the eventual error, assume it is the end of file */
			return FALSE;
		delay_s -= (double)sig.length / sig.rate;
/*		Log_print("Skipped; delay: %f", delay_s);*/
	}

	/* Skip any remaining non-byte signals */
	{
		A8CAS_signal buffer[256];
		while (A8CAS_read_signals(file->file, buffer, 256) != 0);
	}
	return A8CAS_error(file->file) == A8CAS_ERR_NONE; /* 0 when error/EOF, 1 otherwise */
}

int IMG_TAPE_ReadToMemory(IMG_TAPE_t *file, UWORD dest_addr, int length)
{
	UBYTE buffer[256]; /* Atari records aren't typically longer than 132 bytes anyway. */
	unsigned int checksum = 0;
	length ++; /* Increase length so that checksum will be also read. */

	for (;;) {
		int i;
		unsigned int baudrate; /* Ignored. */
		unsigned int fill = A8CAS_read_bytes(file->file, buffer, (length > 256 ? 256: length), &baudrate);

		if (fill == 0) /* Record ended too early, or read error. */
			break;
/*		for (i = 0; i < fill; i ++)
			fprintf(stderr, " %02x", buffer[i]);*/
		length -= fill;
		if (length == 0)
			/* End of a record - so don't treat the last byte (checksum) as data.
			   After this, the read checksum is found at buffer[fill]. */
			fill --;
		MEMORY_CopyToMem(buffer, dest_addr, fill); /* Checkum is not copied. */
		/* Update checksum. */
		for (i = 0; i < fill; i ++)
			checksum += buffer[i];
		if (length == 0) { /* Wanted number of bytes has been read */
			while (checksum > 0xff)
				checksum = (checksum & 0xff) + (checksum >> 8);
/*			fprintf(stderr, " fill: %u, dest_addr: %u, check: %02x, good? %i\n", fill, dest_addr, checksum, checksum == buffer[fill]);*/
			return checksum == buffer[fill];
		}
		dest_addr += fill;
	}

	/* Deal with an error. */
	return (A8CAS_error(file->file) == A8CAS_ERR_NONE) - 1; /* -1 when error/EOF, 0 otherwise */
}

int IMG_TAPE_WriteFromMemory(IMG_TAPE_t *file, UWORD src_addr, int length, int gap)
{
	/* Write leader. */
	if (gap > 0) {
		A8CAS_signal sig;
		sig.signal = 1;
		sig.length = gap;
		sig.rate = 1000;
		if (A8CAS_write(file->file, &sig) != 0)
			return FALSE;
	}

	/* Write block bytes. */
	{
		UBYTE buffer[256];
		unsigned int checksum = 0;

		for (;;) {
			int i;
			unsigned int to_write = (length > 255 ? 255 : length);
		
			length -= to_write;
			MEMORY_CopyFromMem(src_addr, buffer, to_write);
			/* Update checksum. */
			for (i = 0; i < to_write; i ++)
				checksum += buffer[i];
			if (length == 0) {
				/* End of a record - also write checksum. */
				while (checksum > 0xff)
					checksum = (checksum & 0xff) + (checksum >> 8);
				buffer[to_write ++] = checksum;
				return !A8CAS_write_bytes(file->file, buffer, to_write, 600);
			}
			else if (A8CAS_write_bytes(file->file, buffer, to_write, 600) != 0)
				return FALSE;
			src_addr += to_write;
		}
	}
}
