/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>

#include "cas_decode.h"

#include "a8cas_file.h"
#include "a8cas.h"
#include "cas_header.h"

enum { DEFAULT_BAUDRATE = 600,
       DEFAULT_PWM_RATE = 44100 };

static int read_recognise_block(CAS_DECODE_t *decode, A8CAS_signal *sig);
static int read_data_block_next_byte(CAS_DECODE_t *decode, A8CAS_signal *sig);

static int read_data_block_next_bit(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	char current;

	sig->length = 0;

	sig->signal = (decode->current_10bit & decode->bit_mask) != 0;
	while (decode->bit_mask != 0x400
	       && (current = (decode->current_10bit & decode->bit_mask) != 0) == sig->signal) {
		sig->length ++;
		(decode->bit_mask) <<= 1;
	};
	sig->rate = decode->baudrate;
	if (decode->bit_mask == 0x400)
		/* End of byte */
		decode->read_func = &read_data_block_next_byte;
	return 0;
}

/* Invoked before processing the next byte from the buffer, this function
 * splits the byte on 10 bits and fills the current_10bit value. Then the
 * bit processing starts.
 * Returns 0 on success, otherwise 1 and sets file->errnum.
 */
static int read_data_block_next_byte(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	uint8_t byte;
	int retval = (*decode->read_data_byte_func)(decode->file, &byte);

	if (retval == 2) {
		/* End of DATA block */
		decode->read_func = &read_recognise_block;
		return read_recognise_block(decode, sig);
	}
	else if (retval != 0)
		return 1;
	decode->current_10bit = ((unsigned int)byte << 1) | 0x200;
	decode->bit_mask = 1;
	decode->read_func = &read_data_block_next_bit;
	return read_data_block_next_bit(decode, sig);
}

static int read_pwmd_block_next_byte(CAS_DECODE_t *decode, A8CAS_signal *sig);

static int read_pwmd_block_next_bit(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	int length;

	sig->rate = decode->pwm_params.rate;

	if (decode->pwm_params.rising_edge_first)
		sig->signal = (decode->current_10bit & 0x100) != 0;
	else
		sig->signal = (decode->current_10bit & 0x100) == 0;


	length = ((decode->current_10bit & decode->bit_mask) == 0) ? decode->pwm_params.length_0 / 2 
								   : decode->pwm_params.length_1 / 2;
	if ((decode->current_10bit & 0x100) == 0) {
		/* if length is odd, add 1 sample to second half */
		if (length % 2 != 0)
			length ++;

		if (decode->pwm_params.lsb_first) {
			/* read from least to most */
			(decode->bit_mask) <<= 1;
			if (decode->bit_mask == 0x100)
				/* End of byte */
				decode->read_func = &read_pwmd_block_next_byte;
		}
		else {
			/* read from most to least */
			(decode->bit_mask) >>= 1;
			if (decode->bit_mask == 0)
				/* End of byte */
				decode->read_func = &read_pwmd_block_next_byte;
		}
	}

	sig->length = length;
	decode->current_10bit ^= 0x100; /* will get next bit every 2nd call */

	return 0;
}

static int read_pwmd_block_next_byte(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	uint8_t byte;
	int retval = (*decode->read_pwmd_byte_func)(decode->file, &byte);

	if (retval == 2) {
		/* End of PWMD block */
		decode->read_func = &read_recognise_block;
		return read_recognise_block(decode, sig);
	}
	else if (retval != 0)
		return 1;

	/* will use 9th bit to distinguish first/second half of impulse */
	decode->current_10bit = ((unsigned int)byte) | 0x100;
	if (decode->pwm_params.lsb_first)
		decode->bit_mask = 0x1; /* will shift mask left */
	else
		decode->bit_mask = 0x80; /* will shift mask right */

	decode->read_func = &read_pwmd_block_next_bit;
	return read_pwmd_block_next_bit(decode, sig);
}

static int read_fsk_block_next_signal(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	uint16_t length;
	int retval;

	sig->signal = (char)decode->bit_mask;
	decode->bit_mask = ! decode->bit_mask;
	retval = (*decode->read_fsk_signal_func)(decode->file, &length);
	if (retval == 2) {
		/* End of FSK block */
		decode->read_func = &read_recognise_block;
		return read_recognise_block(decode, sig);
	}
	else if (retval != 0)
		return 1;
	sig->length = (unsigned int) length;
	sig->rate = 10000;
	return 0;
}

static int read_pwmc_block_next_group(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	uint8_t length;
	uint16_t count;
	int retval;

	retval = (*decode->read_pwmc_signal_func)(decode->file, &length, &count);
	if (retval == 0) {
		decode->pwm_params.length_0 = (uint16_t) length;
		decode->pwm_params.counter = count;
	}
	else return retval;
	return 0;
}

static int read_pwmc_block_next_signal(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	int retval;

	if (decode->pwm_params.counter == 0) {
		retval = read_pwmc_block_next_group(decode, sig);
		if (retval == 2) {
			/* End of PWMC block */
			decode->read_func = &read_recognise_block;
			return read_recognise_block(decode, sig);
		}
		else if (retval != 0)
			return 1;
	}
	if (decode->pwm_params.rising_edge_first)
		sig->signal = decode->bit_mask;
	else
		sig->signal = ! decode->bit_mask;
	sig->length = decode->pwm_params.length_0/2;
	sig->rate = decode->pwm_params.rate;
	decode->bit_mask = ! decode->bit_mask;
	if (decode->bit_mask) {
		/* decrease counter every 2 signal states (one impulse)*/
		decode->pwm_params.counter --; 
		/* if length is odd, add 1 sample to second half */
		if (decode->pwm_params.length_0 % 2 !=0)
			sig->length++;
	}
	return 0;
}

static int read_pwml_block_next_signal(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	uint16_t length;
	int retval;

	retval = (*decode->read_pwml_signal_func)(decode->file, &length);
	if (retval == 2) {
		/* End of PWMC block */
		decode->read_func = &read_recognise_block;
		return read_recognise_block(decode, sig);
	}
	else if (retval != 0)
			return 1;
	if (decode->pwm_params.rising_edge_first)
		sig->signal = decode->bit_mask;
	else
		sig->signal = ! decode->bit_mask;
	sig->length = length;
	sig->rate = decode->pwm_params.rate;
	decode->bit_mask = ! decode->bit_mask;
	return 0;
}

/* Invoked just after reading a new CAS block, this function sets FSK to
 * the signal of the new block's IRG gap.
 */
static void read_irg(CAS_DECODE_t *decode, A8CAS_signal *sig, CAS_HEADER_t *header)
{
	switch (header->type) {
	case BLOCK_TYPE_DATA:
	case BLOCK_TYPE_FSK:
		sig->signal = 1;
		break;
	case BLOCK_TYPE_PWMC:
	case BLOCK_TYPE_PWML:
		/* this is just to make things pretty, nobody will notice ;) */
		sig->signal = ! decode->pwm_params.rising_edge_first;
	}
	sig->rate = 1000;
	sig->length = header->aux;
}

/* The "entry" point for all CAS read functions. Its purpose is to read the
   next block's header, recognise its type and redirect the cas_decode->read_func
   pointer so that next calls to read functions will perform block-specific
   actions.
   Returns 0 on success; otherwise 1 and sets file->errnum. */
static int read_recognise_block(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	for (;;) {
		CAS_HEADER_t header;
		if ((*decode->read_header_func)(decode->file, &header) != 0)
			return 1;
		/* Remember not to use the header.length field - it's not initialised. */
		switch (header.type) {
		case BLOCK_TYPE_FUJI:
			if ((*decode->read_description_func)(decode->file) != 0)
				return 1;
			break;
		case BLOCK_TYPE_BAUD:
			if (header.aux == 0) {
				/* Baudrate of 0 is not allowed (divide by 0!). */
				decode->file->errnum = A8CAS_ERR_BADFILE;
				return 1;
			}
			decode->baudrate = header.aux;
			break;
		case BLOCK_TYPE_DATA:
			decode->read_func = &read_data_block_next_byte;
			read_irg(decode, sig, &header);
			return 0;
		case BLOCK_TYPE_FSK:
			decode->read_func = &read_fsk_block_next_signal;
			decode->bit_mask = 0;
			read_irg(decode, sig, &header);
			return 0;
		case BLOCK_TYPE_PWMS:
			/* set "samplerate" of PWM signal etc. */
			{
				uint16_t read_rate;
				if ((*decode->read_pwms_block_func)(decode->file, &read_rate) != 0)
					return 1;
				decode->pwm_params.rate = read_rate;
				decode->pwm_params.lsb_first = ( header.aux & 0x4 ) == 0;
				decode->pwm_params.rising_edge_first = ( header.aux & 0x2 ) >> 1;
			}
			break;
		case BLOCK_TYPE_PWMD:
			decode->read_func = &read_pwmd_block_next_byte;
			decode->pwm_params.length_1 = header.aux >> 8;
			decode->pwm_params.length_0 = header.aux & 0xff;
			return read_pwmd_block_next_byte(decode,sig);
		case BLOCK_TYPE_PWMC:
			decode->read_func = &read_pwmc_block_next_signal;
			decode->bit_mask = 1;
			decode->pwm_params.counter = 0;
			read_irg(decode, sig, &header);
			return 0;
		case BLOCK_TYPE_PWML:
			decode->read_func = &read_pwml_block_next_signal;
			decode->bit_mask = decode->pwm_params.rising_edge_first;
			read_irg(decode, sig, &header);
			return 0;
		default:
			decode->file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
	}
	return 0;
}

void CAS_DECODE_reset(CAS_DECODE_t *decode, uint16_t baudrate,
		      uint16_t pwm_rate, int pwm_lsb_first, int pwm_rising_edge_first)
{
	decode->read_func = &read_recognise_block;
	decode->baudrate = (baudrate == 0 ? DEFAULT_BAUDRATE : baudrate);
	decode->pwm_params.rate = (pwm_rate == 0 ? DEFAULT_PWM_RATE : pwm_rate);
	decode->pwm_params.lsb_first = pwm_lsb_first;
	decode->pwm_params.rising_edge_first = pwm_rising_edge_first;
	decode->last_irg_signal.length = 0;
}

int CAS_DECODE_read(CAS_DECODE_t *decode, A8CAS_signal *sig)
{
	return (*decode->read_func)(decode, sig);
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int CAS_DECODE_read_bytes(CAS_DECODE_t *decode, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	unsigned int written = 0;

	for (;;) {

		if (written >= size) {
			/* Result buffer full. */
			decode->file->errnum = A8CAS_ERR_NONE;
			return written;
		}

		if (decode->last_irg_signal.length != 0) {
			decode->file->errnum = A8CAS_ERR_NONE;
			return written;
		}
		if (decode->read_func == &read_recognise_block) {
			if (read_recognise_block(decode, &decode->last_irg_signal) != 0) {
				if (decode->file->errnum == A8CAS_ERR_EOF)
					return written;
				else
					return 0;
			}
			continue;
		}
		if (decode->read_func != &read_data_block_next_byte) {
			/* Reading a signal; no full byte to read. */
			decode->file->errnum = A8CAS_ERR_NONE;
			return written;
		}

		*baudrate = (unsigned int) decode->baudrate;
		for ( ; written < size; written ++) {
			int retval = (*decode->read_data_byte_func)(decode->file, &bytes[written]);
			if (retval == 2) {
				/* End of data block */
				decode->read_func = &read_recognise_block;
				break;
			}
			else if (retval != 0)
				/* Some error. */
				return 0;
		}
	}
}

unsigned int CAS_DECODE_read_signals(CAS_DECODE_t *decode, A8CAS_signal *sigs, unsigned int size)
{
	unsigned int written = 0;

	for (;;) {
		if (written >= size) {
			decode->file->errnum = A8CAS_ERR_NONE;
			return written;
		}

		if (decode->last_irg_signal.length != 0) {
			sigs[written ++] = decode->last_irg_signal;
			decode->last_irg_signal.length = 0;
			continue;
		}
		if (decode->read_func == &read_recognise_block) {
			if (read_recognise_block(decode, &decode->last_irg_signal) != 0) {
				if (decode->file->errnum == A8CAS_ERR_EOF)
					return written;
				else
					return 0;
			}
			continue;
		}

		if (decode->read_func == &read_data_block_next_byte) {
			/* There's a byte to read. */
			decode->file->errnum = A8CAS_ERR_NONE;
			return written;
		}
		
		{
			if ((*decode->read_func)(decode, &sigs[written++]) != 0);
				return 0;
		}
	}
}
