/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "cas_encode.h"

#include "a8cas_file.h"
#include "a8cas.h"
#include "commons.h"
/*#include "denoise.h"*/

#define DEBUG_LOG 0

/* Minimum length of a MARK signal to be considered an IRG, in seconds */
#define MIN_IRG_LENGTH_S 0.05

/* Max length of a bit in a DATA block, in seconds. */
#define MAX_BIT_LENGTH (1.0/200.0)
#define MAX_NOISE_LENGTH 0.01

/* ======== Functions for manipulating the circular signals buffer ======== */

static void add_signal(CAS_ENCODE_t *encode, double length)
{
	assert((encode->bit_signals.r_pos + encode->bit_signals.fill) % encode->bit_signals.buf_size == encode->bit_signals.w_pos);

	encode->bit_signals.buf[encode->bit_signals.w_pos ++] = length;
	encode->bit_signals.sum += length;
	encode->bit_signals.fill ++;
	if (encode->bit_signals.w_pos >= encode->bit_signals.buf_size)
		encode->bit_signals.w_pos = 0;
	assert(encode->bit_signals.fill <= encode->bit_signals.buf_size);
	assert((encode->bit_signals.r_pos + encode->bit_signals.fill) % encode->bit_signals.buf_size == encode->bit_signals.w_pos);
}

static void remove_signals(CAS_ENCODE_t *encode, unsigned int num)
{
	assert((encode->bit_signals.r_pos + encode->bit_signals.fill) % encode->bit_signals.buf_size == encode->bit_signals.w_pos);
	assert(encode->bit_signals.fill >= num);

	if ((encode->bit_signals.fill -= num) == 0) {
		encode->bit_signals.sum = 0.0;
		encode->bit_signals.r_pos = encode->bit_signals.w_pos;
	} else {
		while (num > 0) {
			encode->bit_signals.sum -= encode->bit_signals.buf[encode->bit_signals.r_pos];
			if (++encode->bit_signals.r_pos >= encode->bit_signals.buf_size)
				encode->bit_signals.r_pos = 0;
			num --;
		}
	}

	assert((encode->bit_signals.r_pos + encode->bit_signals.fill) % encode->bit_signals.buf_size == encode->bit_signals.w_pos);
}

#define remove_signal(encode) remove_signals(encode, 1)

static char current_signal(CAS_ENCODE_t const *encode)
{
	assert(encode->bit_signals.fill > 0);
	return (encode->bit_signals.r_pos & 1) ^ 1;
}

static double current_signal_length(CAS_ENCODE_t const *encode)
{
	assert(encode->bit_signals.fill > 0);
	return encode->bit_signals.buf[encode->bit_signals.r_pos];
}

static double signal_length_at(CAS_ENCODE_t const *encode, unsigned int idx)
{
	assert(idx < encode->bit_signals.buf_size);
	assert(encode->bit_signals.fill > idx);
	return encode->bit_signals.buf[(encode->bit_signals.r_pos + idx) % encode->bit_signals.buf_size];
}

#if DEBUG_LOG
static void debug_write_signals(CAS_ENCODE_t *encode)
{
	unsigned int i;

	A8CAS_log(encode->file, "Offset=%u, Number of buffered signals=%u: ", encode->bit_signals.r_pos, encode->bit_signals.fill);
	for (i = 0; i < encode->bit_signals.fill; i ++)
		A8CAS_log(encode->file, " %f/%u", signal_length_at(encode, i), i & 1 );
	A8CAS_log(encode->file, "\n");
}
#endif

/* ======== Write block functions ======== */

static int write_baud_block(CAS_ENCODE_t *encode)
{
	double baudrate = (double) encode->baudrate;
	double deviation = 1.0 + encode->baudrate_deviation;

	if (encode->new_baudrate > baudrate * deviation ||
	    baudrate > encode->new_baudrate * deviation) {
		uint16_t new_baud_int;
		if (encode->new_baudrate >= 65536.0) {
			A8CAS_log(encode->file, "Too high baudrate=%f - probably noise", encode->new_baudrate);
			encode->new_baudrate = 65535.0;
		}
		/* Round to int and check if there's still a baudrate change. */
		new_baud_int = (uint16_t)nearbyint(encode->new_baudrate);
		if (new_baud_int != encode->baudrate) {
			encode->baudrate = new_baud_int;
			if ((*encode->write_baud_block_func)(encode->file, encode->baudrate) != 0)
				return 1;
		}
	}
	return 0;
}

static int write_data_block(CAS_ENCODE_t *encode)
{
	if (write_baud_block(encode) != 0)
		return 1;
	if ((*encode->write_data_block_func)(encode->file, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.data, encode->block_buffer_fill) != 0)
		return 1;
	return (*encode->add_block_offset_func)(encode->file, encode->baudrate);
}

static int write_fsk_block(CAS_ENCODE_t *encode, uint16_t prev_irg_ms, uint16_t *buffer, unsigned int fill)
{
	if (prev_irg_ms == 0 && fill == 0)
		/* The chunk would be empty (sometimes happens on flush when
		   prev_irg rounds down to 0) - don't write. */
		return 0;
	if ((*encode->write_fsk_block_func)(encode->file, prev_irg_ms, buffer, fill) != 0)
		return 1;
	return (*encode->add_block_offset_func)(encode->file, encode->baudrate);
}

/* ======== */

/* After reading a byte, adjust the read baudrate a little in the middle of
   a DATA block, to prevent eventual framing errors in very long blocks. */
static void adjust_bit_length(CAS_ENCODE_t *encode, double byte_length)
{
	double new_bit_length = byte_length / 10.0;
	double max_bit_length = encode->bit_length * 1.1;
	double min_bit_length = encode->bit_length * 0.9;
	
	if (new_bit_length < max_bit_length && new_bit_length > min_bit_length) {
		/* New bit length does not vary too much, so adjust the baudrate. */
/*		double old_bit_length = encode->bit_length;*/
		/* The computation in its original form: */
		encode->bit_length *= (new_bit_length / encode->bit_length - 1.0) / 10.0 + 1.0;
/*		encode->bit_length = new_bit_length / 10.0 + 0.9 * encode->bit_length;*/
/*		A8CAS_log(encode->file, "bitrate adjusted: %f -> %f (%f) at byte %u\n", old_bit_length, encode->bit_length, new_bit_length, encode->block_buffer_fill);*/
	}
}

static int recognise_block_init(CAS_ENCODE_t *encode);
static int write_irg(CAS_ENCODE_t *encode);
static int recognise_fsk_signals(CAS_ENCODE_t *encode);

static int recognise_noise_after_data_block(CAS_ENCODE_t *encode)
{
	do {
		double length = current_signal_length(encode);
		if (length > MAX_NOISE_LENGTH) {
			encode->write_func = &write_irg;
			return write_irg(encode);
		}
		encode->prev_irg_ms += length;
		remove_signal(encode);
	} while (encode->bit_signals.fill != 0);
	return 0;
}

static void merge_signals(CAS_ENCODE_recognise_byte_t *signals, unsigned int pos, unsigned int *fill)
{
	unsigned int i, j;
	assert(pos >= 1);
	assert(pos < *fill - 1);
	
	signals[pos-1].length += signals[pos].length + signals[pos+1].length;
	for (i = pos, j = pos + 2; j < *fill; i ++, j ++) {
		signals[i] = signals[j];
	}
	*fill -= 2;
}

static int recognise_byte(CAS_ENCODE_t *encode, double byte_length)
{
	CAS_ENCODE_recognise_byte_t *signals = encode->recognise_byte_buf;
	double min_length = encode->bit_length * 0.1; /* Anything shorted will be ignored */

	unsigned int read_offset = 0;
	unsigned int fill;
	double sum = 0.0;

	unsigned int i;

	unsigned int pos;
	double position;
	uint16_t byte = 0;
	unsigned int mask = 0x01;
	unsigned int num_of_end_mark_bits = 1;

	double min_dist_from_bit_change_pos = encode->bit_length * (0.5 - encode->bit_deviation);

	{
		double real_sum = 0.0;
/*		A8CAS_log(encode->file, "sum: %f, byte_length: %f\n", encode->bit_signals.sum, byte_length);*/
		while (real_sum < byte_length) {
/*			A8CAS_log(encode->file, "d1 sum: %f\n", sum);*/
			double length = signal_length_at(encode, read_offset);
			real_sum += length;
/*			A8CAS_log(encode->file, "d2\n");*/
			if (read_offset & 0x01)
				length -= encode->bit_1_0_diff;
			else
				length += encode->bit_1_0_diff;
			if (length < 0.0)
				length = 0.0;
			signals[read_offset].length = length;
			signals[read_offset].start = sum;
			sum += length;
			read_offset ++;
		}
	}

	adjust_bit_length(encode, sum);

	if (read_offset & 1) { /* Odd number of signals read, therefore the last is "0" */
		A8CAS_log(encode->file, "Framing error at byte %u\n", encode->block_buffer_fill);
		return 0;
	}

	if (sum > encode->bit_length * (10.0 + encode->stop_bit_deviation) &&
	    sum < encode->bit_length * 15.0) { /* Too long stop bit */
		A8CAS_log(encode->file, "Too long stop bit at byte %u\n", encode->block_buffer_fill);
		return 0;
	}

	fill = read_offset;

	/* Merge too short signals - treat them as minor "interference". */
	for (i = 2; i < fill - 1; ) {
		if (signals[i].length < min_length) {
			/* Merge this signal with its neighbour(s). */
			if (signals[i+1].length >= min_length || i+1 == fill-1) {
				merge_signals(signals, i, &fill);
				continue;
			} else {
				double delete_i_pos = (signals[i+1].start+signals[i+1].length) / encode->bit_length;
				double delete_i1_pos = signals[i].start / encode->bit_length;
				/* Compute distances from the ideal bit border positions. */
				delete_i_pos = fabs(delete_i_pos - round(delete_i_pos));
				delete_i1_pos = fabs(delete_i1_pos - round(delete_i1_pos));
				assert(delete_i_pos <= 0.5);
				assert(delete_i1_pos <= 0.5);
				if (delete_i_pos < delete_i1_pos)
					merge_signals(signals, i, &fill);
				else
					merge_signals(signals, i+1, &fill);
			}
		} else
			i ++;
	}

	/* Decode a byte. */
	pos = 0;
	i = 0;
	position = encode->bit_length * (encode->bit_middle + pos);
	for (;;) {
		if (!(signals[i].start < position && signals[i].start + signals[i].length > position)) {
			/* Found a signal between recognised bits. */
			A8CAS_log(encode->file, "Too short signal at byte %u, signal %u\n", encode->block_buffer_fill, i);
			return 0;
		}
		if (position - signals[i].start < min_dist_from_bit_change_pos ||
		    signals[i].start + signals[i].length - position < min_dist_from_bit_change_pos) {
			A8CAS_log(encode->file, "Ambiguous bit at byte %u, signal %u\n", encode->block_buffer_fill, i);
			return 0;
		}
		
		if (i & 1) {
			byte |= mask;
			num_of_end_mark_bits ++;
		} else
			num_of_end_mark_bits = 1;
		mask <<= 1;
		pos ++;
		if (pos >= 9)
			break;
		position = encode->bit_length * (encode->bit_middle + pos);
		if (signals[i].start + signals[i].length < position)
			i ++;
	}

	byte >>= 1;
/*	{
		unsigned int i;
		A8CAS_log(encode->file, "byte: %x!! pos: %u, size: %u, bit_length: %f diff: %f\n", (unsigned int)byte, encode->block_buffer_fill, encode->bit_signals.fill, encode->bit_length, encode->bit_1_0_diff);
		for (i = 0; i < encode->bit_signals.fill; i ++)
			A8CAS_log(encode->file, "%f ", signal_length_at(encode, i));
		A8CAS_log(encode->file, "\n");
	}*/
	encode->block_buffer.data[encode->block_buffer_fill ++] = byte;

	assert(read_offset <= encode->bit_signals.fill);
	assert(read_offset >= 1);

	remove_signals(encode, read_offset - 1);

	assert(current_signal(encode) == 1);

	encode->bit_signals.buf[encode->bit_signals.r_pos] -= encode->bit_length * num_of_end_mark_bits;
	encode->bit_signals.sum -= encode->bit_length * num_of_end_mark_bits;
	return 1;
}

static int recognise_data_bytes(CAS_ENCODE_t *encode)
{
	for (;;) {
		double byte_length = encode->bit_length * (9.0 + encode->bit_middle);

		if (encode->bit_signals.sum < byte_length) {
			/* Not enough length of signals to decode a single byte. */
			if (encode->bit_signals.fill >= 20) {
				/* ... But there's no free space in the buffer. */
				/* Assume it's noise. */
				if (encode->block_buffer_fill > 0) {
					if (write_data_block(encode) != 0)
						return 1;
					encode->block_buffer_fill = 0;
					encode->prev_irg_ms = 0.0;
				}
				encode->write_func = &recognise_noise_after_data_block;
				return recognise_noise_after_data_block(encode);
			}
			return 0;
		}

		if (encode->block_buffer_fill == 65535) {
			/* DATA block size too big. */
			encode->file->errnum = A8CAS_ERR_LIMIT;
			return 1;
		}

		/* Adjust the byte's length. */
		{
			double sum = 0.0;
			unsigned int offset = 0;

			while (sum < byte_length) {
				double length = signal_length_at(encode, offset);
				sum += length;
				offset ++;
			}
			adjust_bit_length(encode, sum);
		}

		if (!recognise_byte(encode, byte_length)) {
			if (encode->block_buffer_fill > 0) {
				if (write_data_block(encode) != 0)
					return 1;
				encode->block_buffer_fill = 0;
				encode->prev_irg_ms = 0.0;
				return recognise_block_init(encode);
			} else {
				encode->write_func = &recognise_fsk_signals;
				return recognise_fsk_signals(encode);
			}
		}

		if (current_signal_length(encode) >= encode->bit_length * encode->bit_deviation) {
			/* Write DATA block. */
			A8CAS_log(encode->file, "End of data block at byte %u, found IRG=%f\n", encode->block_buffer_fill, current_signal_length(encode));
			if (encode->block_buffer_fill > 0 &&
			    write_data_block(encode) != 0)
					return 1;
			encode->block_buffer_fill = 0;
			encode->prev_irg_ms = 0.0;
			encode->write_func = &write_irg;
			return write_irg(encode);
		} else {
			remove_signal(encode);
		}
	}
}

static int recognise_fsk_signals(CAS_ENCODE_t *encode)
{
	do {
		if (current_signal_length(encode) >= MIN_IRG_LENGTH_S && current_signal(encode)) {
			/* Found an IRG. */
			if (write_fsk_block(encode, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.fsk, encode->block_buffer_fill) != 0)
				return 1;
			encode->block_buffer_fill = 0;
			encode->prev_irg_ms = 0.0;
			encode->write_func = &write_irg;
			return write_irg(encode);
		}

		if (encode->block_buffer_fill == CAS_ENCODE_FSK_BUF_SIZE) {
			/* Artificially split the FSK block. */
			if (write_fsk_block(encode, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.fsk, encode->block_buffer_fill) != 0)
				return 1;
			/* Assume that CAS_ENCODE_FSK_BUF_SIZE is an odd number;
			therefore the last value in the FSK block corresponds to
			a SPACE signal. */
			encode->block_buffer_fill = 0;
			assert(current_signal(encode) == (char) 1);
			encode->prev_irg_ms = 0.0;
			encode->write_func = &write_irg;
			return write_irg(encode);
		}

		/* Convert secs to 1/10's of ms. */
		{
			unsigned int length = (unsigned int)nearbyint(current_signal_length(encode) * 10000.0);
			while (length > 65535) {
				/* Artificially split the FSK block. */
				assert(current_signal(encode) == 0);
				encode->block_buffer.fsk[encode->block_buffer_fill ++] = 65535;
				if (write_fsk_block(encode, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.fsk, encode->block_buffer_fill) != 0)
					return 1;
				encode->block_buffer_fill = 0;
				encode->prev_irg_ms = 0.0;
				length -= 65535;
			}
			encode->block_buffer.fsk[encode->block_buffer_fill ++] = (uint16_t)length;
			remove_signal(encode);
		}
	} while (encode->bit_signals.fill != 0);
	return 0;
}

/* Recognises standard blocks starting with 0x55 55. */
static void adjust_bit_1_0_diff(CAS_ENCODE_t *encode)
{
	unsigned int i;
	if (encode->bit_signals.fill >= 20) {
		double min_length = encode->bit_length * (1.0 - encode->block_header_deviation);
		double max_length = encode->bit_length * (1.0 + encode->block_header_deviation);

		double length0 = 0.0;
		double length1 = 0.0;

		for (i = 0; i < 20; i ++) {
			double length = signal_length_at(encode, i);
			if (length > max_length || length < min_length) {
				return;
			}
			if (i & 1)
				length1 += length;
			else
				length0 += length;
		}

		encode->bit_1_0_diff = (length1 - length0) / 20.0;
		/* Divide by 20 to get half the difference between MARK and SPACE bit lengths. */
		A8CAS_log(encode->file, "Adjusted bit_1_0_diff = %f\n", encode->bit_1_0_diff);
	}
}

/* Returns 1 on success, 0 on failure. */
static int is_byte_length_good(CAS_ENCODE_t *encode, double *byte_length, unsigned int signals_to_check, double *deviation_sum)
{
	double bit_length = *byte_length / 10.0;
	unsigned int i;
	double next_full_byte = *byte_length;
	double all_sum = 0.0;
	double old_sum = 0.0;
	unsigned int bytes_num = 0;

	#define stop_bit_tolerance (encode->block_header_deviation)

	*deviation_sum = 0.0;

	/* Check for framing errors. */
	for (i = 0; i < signals_to_check; i += 2) {
		if (signal_length_at(encode, i + 1) >= MIN_IRG_LENGTH_S) {
			/* Do not inlude the ending IRG signal in the measurement. */
			if (bytes_num == 0)
				return 0;
			break;
		}
		all_sum += signal_length_at(encode, i) + signal_length_at(encode, i + 1);
		if (all_sum < next_full_byte - stop_bit_tolerance * bit_length)
			/* Not a framing bit yet */
			continue;
		if (all_sum > next_full_byte + stop_bit_tolerance * bit_length) {
			/* Framing error */
#if DEBUG_LOG
			A8CAS_log(encode->file, "framing error at %u(%f) - sum=%f, NFB=%f\n", i + 1, signal_length_at(encode, i + 1), all_sum, next_full_byte);
#endif
			return 0;
		}
		next_full_byte = all_sum + *byte_length;
		old_sum = all_sum;
		bytes_num ++;
	}
	/* No framing errors found. */

	*byte_length = old_sum / bytes_num;
	/* *BYTE_LENGTH has been adjusted. Now check whether bit lengths are
	   around 1/10 of the byte length. */
	bit_length = *byte_length / 10.0;
#if DEBUG_LOG
	A8CAS_log(encode->file, "Adjusted byte length=%f, bit length=%f, i=%u\n", *byte_length, bit_length, i);
#endif
	for (i = 0; i < signals_to_check; i ++) {
		double length = signal_length_at(encode, i);
		double num_bits;
		double scaled_length = length / bit_length;
		double deviation;
		/* If we've encountered an IRG, then assume that the byte is OK. */
		if (length >= MIN_IRG_LENGTH_S && (i & 1)) {
			return 1;
		}
		num_bits = round(scaled_length);
		if (num_bits == 0.0)
			num_bits = 1.0;
#if DEBUG_LOG
		A8CAS_log(encode->file, "sigtocheck: %i=%f, num_bits=%f, scaled_length=%f\n", i, length, num_bits, scaled_length);
#endif
		deviation = fabs(scaled_length - num_bits);
		if (deviation > encode->block_header_deviation) {
			/* Too much deviation in bit length. */
#if DEBUG_LOG
			A8CAS_log(encode->file, "deviation at %u(%f) - scaled length=%f, max deviation=%f\n", i, length, scaled_length, encode->block_header_deviation);
#endif
			return 0;
		}
		*deviation_sum += deviation;
	}
	return 1;
}

/* Recognises standard and non-standard blocks (not starting with 0x55 55).
   Tries to determine the bit length and checks for framing errors. */
static int recognise_data_block(CAS_ENCODE_t *encode)
{
	unsigned int i;
	double all_sum = 0.0;
	double result_byte_length = 0.0;
	unsigned int signals_to_check = encode->bit_signals.fill & ~(unsigned int)0x1; /* Even value */

	/* Safe assumption - each signal can add at most 1.0 to the deviation. */
	double min_deviation = 2.0 * signals_to_check;

#if DEBUG_LOG
	debug_write_signals(encode);
#endif
	A8CAS_log(encode->file, "Attempting to recognise data block...");
	/* Assume that there are at least 2 signals in SIGNAL_LENGTHS. */
	for (i = 0; i < signals_to_check; i += 2) {
		double byte_length;
		double deviation;
		all_sum += signal_length_at(encode, i) + signal_length_at(encode, i + 1);
		byte_length = all_sum;

#if DEBUG_LOG
		A8CAS_log(encode->file, "proposed byte length: %f, bit length: %f\n", byte_length, byte_length / 10.0);
#endif
		if (is_byte_length_good(encode, &byte_length, signals_to_check, &deviation) &&
		    deviation <= min_deviation) {
			result_byte_length = byte_length;
			/* Multiply by 1.0001 to account fp-math inexactness (when comparing zeros). */
			min_deviation = deviation + 0.0001;
		}
		
	}
	if (result_byte_length == 0.0) {
		A8CAS_log(encode->file, " failed\n");
		return 0;
	}

	encode->bit_length= result_byte_length / 10.0;
	encode->new_baudrate = 1.0 / encode->bit_length;
	A8CAS_log(encode->file, " succeeded: bit_length=%f\n", encode->bit_length);
	adjust_bit_1_0_diff(encode);
	return 1;
}

static int recognise_block_now(CAS_ENCODE_t *encode)
{
	/* Avoid recognising isolated "bumps" as bytes. If number of stored
	   signals is < 4, then it's an isolated bump. */
	if (encode->bit_signals.fill >= 4 &&
	    recognise_data_block(encode))
		encode->write_func = &recognise_data_bytes;
	else
		/* It's not a DATA block. */
		encode->write_func = &recognise_fsk_signals;
	return (*encode->write_func)(encode);
}

static int wait_for_enough_signals(CAS_ENCODE_t *encode)
{
	while (encode->bit_signals.chk_pos < encode->bit_signals.fill) {
		if ((encode->bit_signals.chk_pos & 1) &&
		    signal_length_at(encode, encode->bit_signals.chk_pos) > MIN_IRG_LENGTH_S)
			return recognise_block_now(encode);
		encode->bit_signals.chk_pos ++;
	}

	if (encode->bit_signals.fill >= encode->block_header_length) {
		if (recognise_data_block(encode))
			encode->write_func = &recognise_data_bytes;
		else
			encode->write_func = &recognise_fsk_signals;
		return (*encode->write_func)(encode);
	}
	return 0;
}

static int recognise_block_init(CAS_ENCODE_t *encode)
{
	encode->bit_signals.chk_pos = 0;
	encode->write_func = &wait_for_enough_signals;
	return wait_for_enough_signals(encode);
}

static int write_irg(CAS_ENCODE_t *encode)
{
	if (current_signal(encode)) {
		encode->prev_irg_ms += current_signal_length(encode) * 1000.0;
		remove_signal(encode);
		while (encode->prev_irg_ms >= 65536.0) {
			if (write_fsk_block(encode, (uint16_t)65535, NULL, 0) != 0)
				return 1;
			encode->prev_irg_ms -= 65535.0;
		}
		return 0;
	}

	A8CAS_log(encode->file, "new block...\n");
	return recognise_block_init(encode);
}

static int process_signals(CAS_ENCODE_t *encode)
{
	double length = encode->sig.length;
	assert(((encode->bit_signals.w_pos & 0x1) ^ 0x1) == encode->sig.signal);

	add_signal(encode, length);
	return (*encode->write_func)(encode);
}

static int write_after_write_bytes(CAS_ENCODE_t *encode);

static int flush_block_buffer(CAS_ENCODE_t *encode)
{
	if (encode->block_buffer_fill > 0) {
		if (encode->write_func == &recognise_data_bytes || encode->write_func == &write_after_write_bytes) {
/*			A8CAS_log(encode->file, "out: bit signals sum: %f, bit length: %f, byte length: %f\n", encode->bit_signals.sum, encode->bit_length, encode->bit_length * (9.0 + BIT_ENCODE_SHIFT));
			{
				unsigned int i;
				A8CAS_log(encode->file, "pos: %u, size: %u, bit_length: %f\n", encode->block_buffer_fill, encode->bit_signals.fill, encode->bit_length);
				for (i = 0; i < encode->bit_signals.fill; i ++)
					A8CAS_log(encode->file, "%f ", signal_length_at(encode, i));
				A8CAS_log(encode->file, "\n");
			}*/
			if (write_data_block(encode) != 0)
				return 1;
		} else {
			assert(encode->write_func == &recognise_fsk_signals);
			if (write_fsk_block(encode, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.fsk, encode->block_buffer_fill) != 0)
				return 1;
		}
		encode->block_buffer_fill = 0;
		encode->prev_irg_ms = 0.0;
		encode->write_func = &write_irg;
	}
	return 0;
}

int CAS_ENCODE_flush(CAS_ENCODE_t *encode)
{
	if (encode->sig.length > 0.0) {
/*		A8CAS_log(encode->file, "write sig: %i, len: %f\n", encode->sig.signal, encode->sig.length);*/
		if (process_signals(encode) != 0)
			return 1;
		encode->sig.length = 0.0;
	}

	if (flush_block_buffer(encode) != 0)
		return 1;

	while (encode->bit_signals.fill > 0) {
		encode->bit_signals.chk_pos = 0;
		encode->write_func = &recognise_block_now;
		if (recognise_block_now(encode) != 0)
			return 1;
		if (flush_block_buffer(encode) != 0)
			return 1;
	}

	encode->bit_signals.r_pos = encode->bit_signals.w_pos = 0;

	if (encode->prev_irg_ms * 1000.0 >= MIN_IRG_LENGTH_S) {
		if (write_fsk_block(encode, (uint16_t)nearbyint(encode->prev_irg_ms), encode->block_buffer.fsk, 0) != 0)
			return 1;
	}
	encode->prev_irg_ms = 0.0;
	return 0;
}

void CAS_ENCODE_init(CAS_ENCODE_t *encode)
{
	encode->bit_deviation = 0.25;
	encode->stop_bit_deviation = 10000.0;
	encode->bit_middle = 0.5;
	encode->block_header_deviation = 0.5;
	encode->baudrate_deviation = 0.05;
	encode->bit_signals.buf = NULL;
	encode->recognise_byte_buf = NULL;
	encode->bit_signals.buf = NULL;
}

int CAS_ENCODE_alloc(CAS_ENCODE_t *encode)
{
	return CAS_ENCODE_set_block_header_length(encode, 20);
}

void CAS_ENCODE_reset(CAS_ENCODE_t *encode, uint16_t baudrate)
{
	encode->sig.signal = 1;
	encode->sig.length = 0.0;
	encode->baudrate = baudrate;
	encode->prev_irg_ms = 0.0;
	encode->bit_signals.fill = 0;
	encode->bit_signals.r_pos = encode->bit_signals.w_pos = 0;
	encode->bit_signals.sum = 0.0;
	encode->bit_length = 0.0;
	encode->bit_1_0_diff = 0.0;
	encode->block_buffer_fill = 0;
	encode->write_func = &write_irg;
}

void CAS_ENCODE_free(CAS_ENCODE_t *encode)
{
	if (encode->bit_signals.buf != NULL) {
		free(encode->bit_signals.buf);
		encode->bit_signals.buf = NULL;
	}
}

int CAS_ENCODE_write(CAS_ENCODE_t *encode, A8CAS_signal const *sig)
{
	double length = (double) sig->length / sig->rate;

	if (encode->sig.signal == sig->signal) {
		encode->sig.length += length;
	} else {
/*		A8CAS_log(encode->file, "write sig: %i, len: %f\n", encode->sig.signal, encode->sig.length);*/
		if (encode->sig.length == 0.0) {
			/* First signal ever (or first signal after rewind). */
			assert(sig->signal == 0);
			assert(encode->bit_signals.fill == 0);
			encode->bit_signals.w_pos = encode->bit_signals.r_pos = 1;
		} else {
			if (process_signals(encode) != 0)
				return 1;
		}
		encode->sig.signal = sig->signal;
		encode->sig.length = length;
	}
	return 0;
}

/* This function is set as write_func only after performing CAS_ENCODE_write_bytes. */
static int write_after_write_bytes(CAS_ENCODE_t *encode)
{
	/* After a write_bytes operation, the first "normal" write operation
	   should flush the data buffer. */
	if (flush_block_buffer(encode) != 0)
		return 1;
	encode->write_func = &write_irg;
	return write_irg(encode);
}

int CAS_ENCODE_write_bytes(CAS_ENCODE_t *encode, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	
	if (encode->sig.length > 0.0) {
		if (process_signals(encode) != 0)
			return 1;
		encode->sig.length = 0.0;
	}
		encode->sig.signal = 1;

	if (!(encode->write_func == &write_irg || encode->write_func == &write_after_write_bytes ||
	      (encode->write_func == &recognise_data_bytes && encode->bit_signals.fill == 0))) {
#if DEBUG_LOG
		A8CAS_log(encode->file, "flush (irg %i) (rdata %i) (rfsk %i) (rbnow %i) (wait %i) (noise %i)\n", encode->write_func == &write_irg, encode->write_func == &recognise_data_bytes, encode->write_func == &recognise_fsk_signals,
		          encode->write_func == &recognise_block_now, encode->write_func == &wait_for_enough_signals, encode->write_func == &recognise_noise_after_data_block);
#endif
		if (CAS_ENCODE_flush(encode) != 0)
			return 1;
	}

	encode->bit_signals.w_pos = encode->bit_signals.r_pos = 0;
	assert(encode->write_func == &write_irg || encode->write_func == &write_after_write_bytes ||
	      (encode->write_func == &recognise_data_bytes && encode->bit_signals.fill == 0));

	if (encode->block_buffer_fill + size > 65535) {
		/* DATA block size too big. */
		encode->file->errnum = A8CAS_ERR_LIMIT;
		return 1;
	}

	/* Set the new baudrate */
	encode->bit_length = 1.0 / baudrate;
	encode->new_baudrate = baudrate;
	
	memcpy(encode->block_buffer.data + encode->block_buffer_fill, bytes, size);
	encode->block_buffer_fill += size;

	encode->write_func = &write_after_write_bytes;
	return 0;
}

/* Functions for setting parameters. */
int CAS_ENCODE_set_bit_deviation(CAS_ENCODE_t *encode, double value)
{
	if (value < 0.0 || value > 0.5) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	encode->bit_deviation = value;
	return 0;
}

int CAS_ENCODE_set_stop_bit_deviation(CAS_ENCODE_t *encode, double value)
{
	if (value < 0.0) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	encode->stop_bit_deviation = value;
	return 0;
}

int CAS_ENCODE_set_bit_middle(CAS_ENCODE_t *encode, double value)
{
	if (value < 0.0 || value > 1.0) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	encode->bit_middle = value;
	return 0;
}

int CAS_ENCODE_set_block_header_length(CAS_ENCODE_t *encode, unsigned int value)
{
	if (value < 10 || (value & 1)) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	if (encode->bit_signals.buf != NULL &&
	    encode->bit_signals.fill != 0) {
		encode->file->errnum = A8CAS_ERR_NOMATCH;
		return 1;
	}
	free(encode->bit_signals.buf);
	free(encode->recognise_byte_buf);
	encode->bit_signals.w_pos = encode->bit_signals.r_pos = 0;

	/* Function recognise_data_bytes() requires the bit_signals buffer size to be at least 20. */
	encode->bit_signals.buf_size = value < 20 ? 20 : value;
	if ((encode->bit_signals.buf = malloc(sizeof(double) * encode->bit_signals.buf_size)) == NULL ||
	    (encode->recognise_byte_buf = malloc(sizeof(CAS_ENCODE_recognise_byte_t) * encode->bit_signals.buf_size)) == NULL) {
		encode->file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}
	encode->block_header_length = value;
	return 0;
}

int CAS_ENCODE_set_block_header_deviation(CAS_ENCODE_t *encode, double value)
{
	if (value < 0.0) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	encode->block_header_deviation = value;
	return 0;
}

int CAS_ENCODE_set_baudrate_deviation(CAS_ENCODE_t *encode, unsigned int value)
{
	if (value < 0.0) {
		encode->file->errnum = A8CAS_ERR_INVALID;
		return 1;
	}
	encode->baudrate_deviation = value;
	return 0;
}
