/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#ifndef CAS_ENCODE_H
#define CAS_ENCODE_H

#include <inttypes.h>

#include "a8cas.h"

enum { CAS_ENCODE_DATA_BUF_SIZE = 65535 };

/* CAS_ENCODE_FSK_BUF_SIZE must be an odd number. */
enum { CAS_ENCODE_FSK_BUF_SIZE = 65535 / 2 };

/* CAS_DECODE_SIGNAL_BUF_SIZE must be an even number.
   Type of signal is determined by index % 2. */
enum { CAS_ENCODE_SIGNAL_BUF_SIZE = 50 };

typedef struct CAS_ENCODE_recognise_byte_t {
	double start;
	double length;
} CAS_ENCODE_recognise_byte_t;

typedef struct CAS_ENCODE_t CAS_ENCODE_t;

struct CAS_ENCODE_t {
	/* The values below should be set by user */
	A8CAS_FILE *file;
	int (*write_baud_block_func)(A8CAS_FILE*, uint16_t);
	int (*write_data_block_func)(A8CAS_FILE*, uint16_t, uint8_t const *, uint16_t);
	/* The function might modify the buffer given as the 3rd parameter. */
	int (*write_fsk_block_func)(A8CAS_FILE*, uint16_t, uint16_t *, uint16_t);
	int (*add_block_offset_func)(A8CAS_FILE*, uint16_t);

	/* Parameters adjustable by user */
	double block_header_deviation; /* Default: 0.25 */
	double bit_deviation; /* Default: 0.25 */
	double stop_bit_deviation; /* Default: higher than IRG length, so no stop bit deviation will be measured. */
	double bit_middle; /* Default: 0.5 */
	unsigned int block_header_length; /* Must be even. Default: 20. */
	double baudrate_deviation; /* Min. 0. Default: 0.05 */ 

	/* Internal variables */
	struct {
		char signal;
		double length;
	} sig;

	uint16_t baudrate;

	double prev_irg_ms;
	struct {
		unsigned int fill;
		unsigned int r_pos;
		unsigned int w_pos;
		unsigned int chk_pos;
		double sum;
		double *buf;
		unsigned int buf_size; /* Not lower than 20. Must be even. */
	} bit_signals;

	double bit_length;
	double new_baudrate;
	double bit_1_0_diff;

	CAS_ENCODE_recognise_byte_t *recognise_byte_buf;

	union {
		uint8_t data[CAS_ENCODE_DATA_BUF_SIZE];
		uint16_t fsk[CAS_ENCODE_FSK_BUF_SIZE];
	} block_buffer;
	uint16_t block_buffer_fill;

	int (*write_func)(CAS_ENCODE_t*);
};

/* Sets default values for encode adjustments */
void CAS_ENCODE_init(CAS_ENCODE_t *encode);

int CAS_ENCODE_alloc(CAS_ENCODE_t *encode);

void CAS_ENCODE_free(CAS_ENCODE_t *encode);

void CAS_ENCODE_reset(CAS_ENCODE_t *encode, uint16_t baudrate);

int CAS_ENCODE_write(CAS_ENCODE_t *encode, A8CAS_signal const *sig);

int CAS_ENCODE_write_bytes(CAS_ENCODE_t *encode, unsigned char const *bytes, unsigned int size, unsigned int baudrate);

int CAS_ENCODE_flush(CAS_ENCODE_t *encode);

/* Functions for setting parameters. They check if a value is correct; if not, they return 1. */
int CAS_ENCODE_set_block_header_deviation(CAS_ENCODE_t *encode, double value);
int CAS_ENCODE_set_bit_deviation(CAS_ENCODE_t *encode, double value);
int CAS_ENCODE_set_stop_bit_deviation(CAS_ENCODE_t *encode, double value);
int CAS_ENCODE_set_bit_middle(CAS_ENCODE_t *encode, double value);
int CAS_ENCODE_set_block_header_length(CAS_ENCODE_t *encode, unsigned int value);
int CAS_ENCODE_set_baudrate_deviation(CAS_ENCODE_t *encode, unsigned int value);

#endif /* CAS_ENCODE_H */
