/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "format_cas.h"

#include "a8cas_file.h"
#include "a8cas.h"
#include "cas_decode.h"
#include "cas_encode.h"
#include "cas_header.h"
#include "commons.h"
#include "dyn_array.h"
#include "endianness.h"
#include "format_stdio_based.h"
#include "io.h"
#include "read_audio.h"

/*#define REPORT_BLOCKS*/

enum { FSK_BUF_SIZE = 64 };
enum { DATA_BUF_SIZE = FSK_BUF_SIZE * 2 };
enum { PWMC_BUF_SIZE = DATA_BUF_SIZE / 3 * 3}; /* rounded to multiples of 3 */

typedef struct block_offset_t {
	long offset;
	uint16_t baudrate;

	uint16_t pwm_rate;
	int pwm_lsb_first;
	int pwm_rising_edge_first;
} block_offset_t;

typedef struct file_specific {
	FILE *file;
	union {
		uint16_t fsk[FSK_BUF_SIZE];
		uint8_t data[DATA_BUF_SIZE];
		uint8_t pwmc[PWMC_BUF_SIZE];
	} buffer;
	unsigned int buffer_pos;
	block_offset_t *block_offsets;
	/* Number of records + 1 (we also hold the offset of the last block's
	 * end): */
	DYN_ARRAY_t block_array;
	unsigned int current_block;

	uint16_t read_length; /* Number of bytes/signals left in currently-read block */
	CAS_DECODE_t cas_decode;
	CAS_ENCODE_t cas_encode;
	READ_AUDIO_t read_audio;
} file_specific;

/* ------------------------------------------------------------------------ *
 * Block offset control                                                     *
 * ------------------------------------------------------------------------ */

#ifdef REPORT_BLOCKS
static void report_blocks(A8CAS_FILE *file, char const *desc)
{
	unsigned int end = A8CAS_size(file) + 1;
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int i;

	A8CAS_log(file, "%s: current block = %lu\n", desc, A8CAS_tell(file));
	for (i = 0; i < end; i ++) {
		A8CAS_log(file, "Block %u: offset=%lu, baudrate=%u\n", i, (long unsigned)data->block_offsets[i].offset, (unsigned int)data->block_offsets[i].baudrate);
	}
}
#endif

static void truncate_block_offsets(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	/* We are at the beginning of the block, so we just truncate the
	   block array. */
	data->block_array.fill = data->current_block + 1;
}

static int add_offset_after_write_block(A8CAS_FILE *file, uint16_t baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	block_offset_t offset;

	offset.baudrate = baudrate;
	if ((offset.offset = ftell(data->file)) == -1) {
		file->errnum = A8CAS_ERR_FTELL;
		return 1;
	}
	
	if (DYN_ARRAY_add(&data->block_array, &offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	data->current_block ++;
#ifdef REPORT_BLOCKS
	report_blocks(file, "add_offset");
#endif

	return 0;
}

static int flush_data(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 0;
	return CAS_ENCODE_flush(&data->cas_encode);
}


int FORMAT_CAS_close(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int retval = 0;

	if (data != NULL) {
		if (flush_data(file) != 0)
			retval = 1;
		CAS_ENCODE_free(&data->cas_encode);
		if (data->block_offsets != NULL)
			free(data->block_offsets);
		if (data->file != NULL && fclose(data->file) != 0) {
			file->errnum = A8CAS_ERR_FCLOSE;
			retval = 1;
		}
		free(data);
	}

	return retval;
}

int FORMAT_CAS_flush(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;
	if (fflush(data->file) != 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig);
static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate);

/* ------------------------------------------------------------------------ *
 * CAS_DECODE callback functions                                            *
 * ------------------------------------------------------------------------ */

/* Returns 0 on success, else returns 1 and sets file->errnum */
static int read_cas_header(A8CAS_FILE *file, CAS_HEADER_t *header)
{
	uint16_t aux_buffer[2];
	size_t size;
	file_specific *data = (file_specific*) file->specific_data;

	size = fread(&header->type, sizeof(uint32_t), 1, data->file);
	if (size == 0) {
		file->errnum = feof(data->file)? A8CAS_ERR_EOF: A8CAS_ERR_FREAD;
		return 1;
	}

	size = fread(aux_buffer, sizeof(uint16_t), 2, data->file);
	if (size != 2) {
		file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_FREAD;
		return 1;
	}
	header->length = LE_SHORT(aux_buffer[0]); /* Needed by initialise_block_offsets_read. */
	header->aux = LE_SHORT(aux_buffer[1]);
	switch (header->type) {
		case BLOCK_TYPE_FSK:
			data->read_length = LE_SHORT(aux_buffer[0]) / 2;
			data->read_audio.mod_type = MOD_TYPE_FSK;
			break;
		case BLOCK_TYPE_PWMS:
			data->read_length = LE_SHORT(aux_buffer[0]) / 2;
			break;
		case BLOCK_TYPE_PWMC:
		case BLOCK_TYPE_PWMD:
			data->read_length = LE_SHORT(aux_buffer[0]);
			data->read_audio.mod_type = MOD_TYPE_PWM;
			break;
		case BLOCK_TYPE_PWML:
			data->read_length = LE_SHORT(aux_buffer[0]) / 2;
			data->read_audio.mod_type = MOD_TYPE_PWM;
			break;
		case BLOCK_TYPE_DATA:
			data->read_length = LE_SHORT(aux_buffer[0]);
			data->read_audio.mod_type = MOD_TYPE_FSK;
			break;
		default:
			data->read_length = LE_SHORT(aux_buffer[0]);
	}
	/* Indicate that the next read shoud fill the buffer. */
	data->buffer_pos = DATA_BUF_SIZE;
	return 0;
}

static int read_data_uint8(A8CAS_FILE *file, uint8_t *byte)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->read_length == 0) {
		data->current_block ++;
		return 2;
	}

	if (data->buffer_pos >= DATA_BUF_SIZE) { /* Need to fill the buffer */
		/* Make the amount to read fit the buffer */
		unsigned int read_length = (data->read_length > DATA_BUF_SIZE ? DATA_BUF_SIZE : data->read_length);

		if (fread(data->buffer.data, sizeof(uint8_t), read_length, data->file) != read_length) {
			file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_FREAD;
			return 1;
		}

		data->buffer_pos = 0;
	}
	*byte = data->buffer.data[data->buffer_pos ++];
	data->read_length --;
	return 0;
}

static int read_data_uint16(A8CAS_FILE *file, uint16_t *length)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->read_length == 0) {
		data->current_block ++;
		return 2;
	}

	if (data->buffer_pos >= FSK_BUF_SIZE) { /* Need to fill the buffer */
		/* Make the amount to read fit the buffer */
		unsigned int read_length = (data->read_length > FSK_BUF_SIZE ? FSK_BUF_SIZE : data->read_length);

		if (fread(data->buffer.fsk, sizeof(uint16_t), read_length, data->file) != read_length) {
			file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_FREAD;
			return 1;
		}

		data->buffer_pos = 0;
	}
	*length = LE_SHORT(data->buffer.fsk[data->buffer_pos ++]);
	data->read_length --;
	return 0;
}

static int read_pwms_block(A8CAS_FILE *file, uint16_t *rate)
{
	uint16_t buf;
	file_specific *data = (file_specific*) file->specific_data;

	if (fread(&buf, sizeof(uint16_t), 1, data->file) == 0) {
		file->errnum = feof(data->file)? A8CAS_ERR_EOF: A8CAS_ERR_FREAD;
		return 1;
	}
	*rate = LE_SHORT(buf);
	data->read_length --;
	return 0;
}

static int read_pwmc_signal(A8CAS_FILE *file, uint8_t *length, uint16_t *count)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->read_length == 0) {
		data->current_block ++;
		return 2;
	}

	assert(sizeof(uint16_t) == 2 * sizeof(uint8_t));

	if (data->buffer_pos >= PWMC_BUF_SIZE) { /* Need to fill the buffer */
		/* Make the amount to read fit the buffer */
		unsigned int read_length = (data->read_length > PWMC_BUF_SIZE ? PWMC_BUF_SIZE : data->read_length);

		if (fread(data->buffer.pwmc, sizeof(uint8_t), read_length, data->file) != read_length) {
			file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_FREAD;
			return 1;
		}

		data->buffer_pos = 0;
	}
	*length = data->buffer.pwmc[data->buffer_pos ++];
	*count = (uint16_t)data->buffer.pwmc[data->buffer_pos ++];
	*count |= ((uint16_t)data->buffer.pwmc[data->buffer_pos ++]) << 8;
	data->read_length -= 3;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Read functions                                                           *
 * ------------------------------------------------------------------------ */

static int specific_read(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read(&data->cas_decode, sig);
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int FORMAT_CAS_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_bytes(&data->cas_decode, bytes, size, baudrate);
}

unsigned int FORMAT_CAS_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_signals(&data->cas_decode, sigs, size);
}

static int specific_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write_bytes(&data->cas_encode, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Audio playback functions                                                 *
 * ------------------------------------------------------------------------ */

void FORMAT_CAS_set_audio(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_reconfigure(&data->read_audio, file->audio.samplerate, file->audio.bits);
}

static int specific_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	return READ_AUDIO_read_with_audio(&data->read_audio, sig, num_samples);
}

int FORMAT_CAS_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_read_audio(&data->read_audio, samples, num_samples);
	return num_samples;
}

/* ------------------------------------------------------------------------ *
 * Read after write functions                                               *
 * ------------------------------------------------------------------------ */
 
static void reset_before_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read;
	file->read_with_audio_func = &specific_read_with_audio;
	file->write_func = &specific_write_after_read;
	file->write_bytes_func = &specific_write_bytes_after_read;
	file->errnum = A8CAS_ERR_NONE;
	CAS_DECODE_reset(&data->cas_decode,
			 data->block_offsets[data->current_block].baudrate,
			 data->block_offsets[data->current_block].pwm_rate,
			 data->block_offsets[data->current_block].pwm_lsb_first,
			 data->block_offsets[data->current_block].pwm_rising_edge_first);
	READ_AUDIO_reset(&data->read_audio);
}

static int common_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;

	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}
	reset_before_read(file);

	return 0;
}

static int specific_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read(file, sig);
}

static int specific_read_with_audio_after_write(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read_with_audio(file, sig, num_samples);
}

/* ------------------------------------------------------------------------ *
 * CAS_ENCODE callback functions                                            *
 * ------------------------------------------------------------------------ */

static int write_cas_header(A8CAS_FILE *file, CAS_HEADER_t *header)
{
	file_specific *data = (file_specific*) file->specific_data;
	uint16_t aux_buffer[2];
	size_t size;

	size = fwrite(&header->type, sizeof(uint32_t), 1, data->file);
	if (size == 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}

	aux_buffer[0] = LE_SHORT(header->length);
	aux_buffer[1] = LE_SHORT(header->aux);
	size = fwrite(aux_buffer, sizeof(uint16_t), 2, data->file);
	if (size != 2) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}

	return 0;
}

static int write_baud_block(A8CAS_FILE *file, uint16_t baudrate)
{
	CAS_HEADER_t header;
	header.type = BLOCK_TYPE_BAUD;
	header.length = 0;
	header.aux = baudrate;
	return write_cas_header(file, &header);
}

static int write_data_block(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	CAS_HEADER_t header;
	header.type = BLOCK_TYPE_DATA;
	header.length = length;
	header.aux = irg_ms;
	if (write_cas_header(file, &header) != 0)
		return 1;

	if (fwrite(buffer, sizeof(uint8_t), length, data->file) != length) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

static int write_fsk_block(A8CAS_FILE *file, uint16_t irg_ms, uint16_t *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	CAS_HEADER_t header;
	int i;
	header.type = BLOCK_TYPE_FSK;
	header.length = length * 2;
	header.aux = irg_ms;
	if (write_cas_header(file, &header) != 0)
		return 1;

	/* Convert the buffer to proper endianness. */
	for (i = 0; i < length; i ++)
		buffer[i] = LE_SHORT(buffer[i]);
		
	if (fwrite(buffer, sizeof(uint16_t), length, data->file) != length) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Write functions                                                          *
 * ------------------------------------------------------------------------ */

static int specific_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write(&data->cas_encode, sig);
}

int FORMAT_CAS_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((*file->write_func)(file, sig) != 0)
		return 1;
	FSK_MOD_generate(&data->read_audio.mod.fsk, samples, sig->length, sig->signal);
	return 0;
}

static void reset_before_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read_after_write;
	file->read_with_audio_func = &specific_read_with_audio_after_write;
	file->write_func = &specific_write;
	file->write_bytes_func = &specific_write_bytes;
	file->errnum = A8CAS_ERR_NONE;
	CAS_ENCODE_reset(&data->cas_encode, data->block_offsets[data->current_block].baudrate);
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (FORMAT_CAS_seek(file, data->current_block) != 0)
		return 1;
	if (IO_ftruncate(data->file, data->block_offsets[data->current_block].offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	truncate_block_offsets(file);

	reset_before_write(file);

	return specific_write(file, sig);
}

static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (FORMAT_CAS_seek(file, data->current_block) != 0)
		return 1;
	if (IO_ftruncate(data->file, data->block_offsets[data->current_block].offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	truncate_block_offsets(file);

	reset_before_write(file);

	return specific_write_bytes(file, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Block & seeking functions                                                *
 * ------------------------------------------------------------------------ */

unsigned long FORMAT_CAS_tell(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->current_block;
}

unsigned long FORMAT_CAS_size(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	return (unsigned long)data->block_array.fill - 1;
}

int FORMAT_CAS_seek(A8CAS_FILE *file, unsigned long position)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int block_num = (unsigned int) position;

	if (flush_data(file) != 0)
		return 1;
	if (block_num >= data->block_array.fill)
		block_num = data->block_array.fill - 1;
	if (fseek(data->file, data->block_offsets[block_num].offset, SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	clearerr(data->file);

	data->current_block = block_num;
	reset_before_read(file);
#ifdef REPORT_BLOCKS
	report_blocks(file, "go_to");
#endif
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Functions for adjusting encoding/decoding options and parameters         *
 * ------------------------------------------------------------------------ */
int FORMAT_CAS_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		if (CAS_ENCODE_set_block_header_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		if (CAS_ENCODE_set_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		if (CAS_ENCODE_set_stop_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		if (CAS_ENCODE_set_bit_middle(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		if (CAS_ENCODE_set_block_header_length(&data->cas_encode, *((unsigned int *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		if (CAS_ENCODE_set_baudrate_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		if (READ_AUDIO_set_crosstalk_volume(&data->read_audio, *((float *)value)))
			return 1;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

int FORMAT_CAS_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		 *((double *) value) = data->cas_encode.block_header_deviation;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.bit_deviation;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.stop_bit_deviation;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		 *((double *) value) = data->cas_encode.bit_middle;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		 *((unsigned int *) value) = data->cas_encode.block_header_length;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		 *((double *) value) = data->cas_encode.baudrate_deviation;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		*((float *) value) = data->read_audio.crosstalk_volume;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Initialisation functions                                                 *
 * ------------------------------------------------------------------------ */


static int initialise_block_offsets_write(A8CAS_FILE *file, uint16_t block1_length)
{
	file_specific *data = (file_specific*) file->specific_data;
	block_offset_t offset;

	offset.offset = block1_length + 8;
	offset.baudrate = 0;
	offset.pwm_rate = 0;
	offset.pwm_lsb_first = 0;
	offset.pwm_rising_edge_first = 0;
	
	if (DYN_ARRAY_init(&data->block_array, 256, sizeof(block_offset_t), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	if (DYN_ARRAY_add(&data->block_array, &offset) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	data->current_block = 0;
	return 0;
}

static int read_description(A8CAS_FILE *file)
{
	uint16_t length;
	file_specific *data = (file_specific*) file->specific_data;
	length = data->read_length;

	file->description = realloc(file->description,length + 1);

	if (file->description == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	if (fread(file->description, sizeof(char), length, data->file) != length) {
		file->errnum = feof(data->file)? A8CAS_ERR_BADFILE: A8CAS_ERR_FREAD;
		return 1;
	}

	file->description[length] = '\0';
	return 0;
}

/* Creates a "directory" of block offsets in the CAS file, for future use when
   rewinding the tape. Fills the struct file's "block_offsets" field.
   Returns 0 on success, otherwise 1 and sets file->errnum. */
static int initialise_block_offsets_read(A8CAS_FILE *file, uint16_t block1_length)
{
	file_specific *data = (file_specific*) file->specific_data;

	block_offset_t cur_block;
	int after_ignored_block = 0;

	/* Assume that at the beginning of this function the file's position
	   indicator lies after the end of the FUJI block, and that the FUJI
	   block's length is stored in block1_length. */
	cur_block.offset = block1_length + 8;
	cur_block.baudrate = 0;
	cur_block.pwm_rate = 0;
	cur_block.pwm_lsb_first = 0;
	cur_block.pwm_rising_edge_first = 0;


	if (DYN_ARRAY_init(&data->block_array, 10, sizeof(block_offset_t), (void **)(&data->block_offsets)) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	for (;;) {
		/* Read next block. */
		CAS_HEADER_t header;
		if (read_cas_header(file, &header) != 0) {
			if (file->errnum == A8CAS_ERR_EOF) {
				/* If no block left, add the offset of the
				   place after the last block. */
				if (!after_ignored_block &&
				    DYN_ARRAY_add(&data->block_array, &cur_block) != 0) {
					file->errnum = A8CAS_ERR_NOMEM;
					return 1;
				}
				break;
			}
			else /* Read error */
				return 1;
		}
		if (!after_ignored_block &&
		    DYN_ARRAY_add(&data->block_array, &cur_block) != 0) {
			file->errnum = A8CAS_ERR_NOMEM;
			return 1;
		}
		switch (header.type) {
		/* If it's a BAUD block, get the baudrate but ignore the
		   block in the offsets' table. */
		case BLOCK_TYPE_BAUD:
			cur_block.baudrate = header.aux;
			after_ignored_block = 1;
			break;
		case BLOCK_TYPE_DATA:
		case BLOCK_TYPE_FSK:
		case BLOCK_TYPE_PWMD:
		case BLOCK_TYPE_PWMC:
		case BLOCK_TYPE_PWML:
			/* Add new offset to the table. */
			after_ignored_block = 0;
			break;
		case BLOCK_TYPE_PWMS:
			{
				uint16_t read_rate;
				if (read_pwms_block(file,&read_rate) != 0) {
					return 1;
				}
				cur_block.pwm_rate = read_rate;
				cur_block.pwm_lsb_first = ( header.aux & 0x4 ) == 0;
				cur_block.pwm_rising_edge_first = ( header.aux & 0x3 ) >> 1;
				after_ignored_block = 1;
				/* data bytes were taken - the rest of block is shorter */
				header.length -= sizeof(uint16_t);
				/* but adjust ofsset */
				cur_block.offset += sizeof(uint16_t);
				break;
			}
		case BLOCK_TYPE_FUJI:
			after_ignored_block = 1;
			break;
		default:
			/* don't do bad things */
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
		/* Skip remaining data in the block. */
		if (fseek(data->file, (long) header.length, SEEK_CUR) != 0) {
			file->errnum = A8CAS_ERR_FSEEK;
			return 1;
		}
		cur_block.offset += header.length + 8;
	}

	if (DYN_ARRAY_strip(&data->block_array) != 0) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* Jump back to block 0. */
	return FORMAT_CAS_seek(file, 0);
}

int FORMAT_CAS_open(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	file_specific *data;

	if ((data = file->specific_data = malloc(sizeof (*data))) == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* NULLize allocatable pointers. */
	data->file = NULL;
	data->block_offsets = NULL;
	CAS_ENCODE_init(&data->cas_encode);

	READ_AUDIO_init(&data->read_audio, file);

	data->cas_decode.file = file;
	data->cas_decode.read_header_func = &read_cas_header;
	data->cas_decode.read_fsk_signal_func = &read_data_uint16;
	data->cas_decode.read_data_byte_func = &read_data_uint8;
	data->cas_decode.read_pwms_block_func = &read_pwms_block;
	data->cas_decode.read_pwmd_byte_func = &read_data_uint8;
	data->cas_decode.read_pwmc_signal_func = &read_pwmc_signal;
	data->cas_decode.read_pwml_signal_func = &read_data_uint16;
	data->cas_decode.read_description_func = &read_description;

	data->cas_encode.file = file;
	data->cas_encode.write_baud_block_func = &write_baud_block;
	data->cas_encode.write_data_block_func = &write_data_block;
	data->cas_encode.write_fsk_block_func = &write_fsk_block;
	data->cas_encode.add_block_offset_func = &add_offset_after_write_block;

	/*data->read_audio.mod_type = MOD_TYPE_FSK;*/

	/* Open the file descriptor */
	if ((data->file = FORMAT_STDIO_BASED_open(path, file->mode)) == NULL) {
		file->errnum = A8CAS_ERR_FOPEN;
		return 1;
	}

	/* Initialise to avoid errors when flush_data() is called. */
	if (CAS_ENCODE_alloc(&data->cas_encode) != 0)
		return 1;
	CAS_ENCODE_reset(&data->cas_encode, 0);

	if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
		char const *description = (info->description == NULL? "": info->description);
		CAS_HEADER_t header;
		header.type = BLOCK_TYPE_FUJI;
		header.length = strlen(description);
		header.aux = 0;
		if (write_cas_header(file, &header) != 0)
			return 1;
		if (fwrite(description, sizeof(char), header.length, data->file) != header.length) {
			file->errnum = A8CAS_ERR_FWRITE;
			return 1;
		}

		if (initialise_block_offsets_write(file, header.length) != 0)
			return 1;
		reset_before_write(file);
	} else { /* file->mode == A8CAS_READ || file->mode == A8CAS_RDWR */
		CAS_HEADER_t header;
		file->description = NULL;

		if (read_cas_header(file, &header) != 0 || header.type != BLOCK_TYPE_FUJI) {
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
		if (read_description(file) !=0 )
		    return 1;
		if (initialise_block_offsets_read(file, header.length) != 0)
			return 1;
		info->samplerate = 0;
		reset_before_read(file);
	}

#ifdef REPORT_BLOCKS
	report_blocks(file, "initialise");
#endif
	return 0;
}
