/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "format_raw.h"

#include "a8cas_file.h"
#include "a8cas.h"
#include "cas_decode.h"
#include "cas_encode.h"
#include "cas_header.h"
#include "format_stdio_based.h"
#include "io.h"
#include "read_audio.h"

/*#define REPORT_BLOCKS*/

enum { BLOCK_SIZE = 132 };

typedef struct file_specific {
	FILE *file;
	uint8_t buffer[BLOCK_SIZE];
	unsigned int buffer_pos;
	long offset;
	long length;
	int reading_all0;

	CAS_DECODE_t cas_decode;
	CAS_ENCODE_t cas_encode;
	READ_AUDIO_t read_audio;
} file_specific;

/* ------------------------------------------------------------------------ *
 * Block offset control                                                     *
 * ------------------------------------------------------------------------ */
static int add_offset_after_write_block(A8CAS_FILE *file, uint16_t baudrate)
{
	return 0;
}

static int flush_data(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 0;
	return CAS_ENCODE_flush(&data->cas_encode);
}

int FORMAT_RAW_close(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int retval = 0;

	if (data != NULL) {
		if (flush_data(file) != 0)
			retval = 1;
		CAS_ENCODE_free(&data->cas_encode);
		if (data->file != NULL && fclose(data->file) != 0) {
			file->errnum = A8CAS_ERR_FCLOSE;
			retval = 1;
		}
		free(data);
	}

	return retval;
}

int FORMAT_RAW_flush(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;
	if (fflush(data->file) != 0) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}
	return 0;
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig);
static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate);

/* ------------------------------------------------------------------------ *
 * CAS_DECODE callback functions                                            *
 * ------------------------------------------------------------------------ */

/* Returns 0 on success, else returns 1 and sets file->errnum */
static int read_cas_header(A8CAS_FILE *file, CAS_HEADER_t *header)
{
	file_specific *data = (file_specific*) file->specific_data;


	if (data->offset >= data->length) {
		if (data->reading_all0) {
			/* EOF block */
			data->buffer[2] = 0xfe;
			memset(data->buffer + 3, 0, 128);
		} else {
			file->errnum = A8CAS_ERR_EOF;
			return 1;
		}
	} else {
		size_t size = fread(data->buffer + 3, sizeof(uint8_t), 128, data->file);
		if (size < 128) {
			if (!feof(data->file)) {
				file->errnum = A8CAS_ERR_FREAD;
				return 1;
			}
			data->buffer[2] = 0xfa;
			memset(data->buffer + 3 + size, 0, 127 - size);
			data->buffer[130] = (uint8_t)size;
		}
		else
			data->buffer[2] = 0xfc;
	}

	/* Compute checksum */
	{
		int i;
		unsigned int checksum = 0;
		for (i = 0; i < 131; i ++)
			checksum += data->buffer[i];
		while (checksum > 0xff)
			checksum = (checksum & 0xff) + (checksum >> 8);
		data->buffer[131] = checksum;
	}

	header->type = BLOCK_TYPE_DATA;
	header->aux = (data->offset == 0 ? 20000 : 250);

	data->buffer_pos = 0;
	return 0;
}

static void next_block(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->offset >= data->length) {
		if (data->reading_all0)
			data->reading_all0 = 0;
	} else {
		data->offset += 128;
		if (data->offset >= data->length) {
			data->reading_all0 = 1;
			data->offset = data->length;
		}
	}
}

static int read_data_byte(A8CAS_FILE *file, uint8_t *byte)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->buffer_pos >= BLOCK_SIZE) {
		next_block(file);
		return 2;
	}

	*byte = data->buffer[data->buffer_pos ++];
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Read functions                                                           *
 * ------------------------------------------------------------------------ */

static int specific_read(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read(&data->cas_decode, sig);
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int FORMAT_RAW_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_bytes(&data->cas_decode, bytes, size, baudrate);
}

unsigned int FORMAT_RAW_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_DECODE_read_signals(&data->cas_decode, sigs, size);
}

static int specific_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write_bytes(&data->cas_encode, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Audio playback functions                                                 *
 * ------------------------------------------------------------------------ */

void FORMAT_RAW_set_audio(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_reconfigure(&data->read_audio, file->audio.samplerate, file->audio.bits);
}

static int specific_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	return READ_AUDIO_read_with_audio(&data->read_audio, sig, num_samples);
}

int FORMAT_RAW_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	READ_AUDIO_read_audio(&data->read_audio, samples, num_samples);
	return num_samples;
}

/* ------------------------------------------------------------------------ *
 * Read after write functions                                               *
 * ------------------------------------------------------------------------ */
 
static void reset_before_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read;
	file->read_with_audio_func = &specific_read_with_audio;
	file->write_func = &specific_write_after_read;
	file->write_bytes_func = &specific_write_bytes_after_read;
	file->errnum = A8CAS_ERR_NONE;
	CAS_DECODE_reset(&data->cas_decode, 600, 44100, 0, 0);
	READ_AUDIO_reset(&data->read_audio);
}

static int common_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (flush_data(file) != 0)
		return 1;

	reset_before_read(file);
	data->reading_all0 = 0;

	return 0;
}

static int specific_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read(file, sig);
}

static int specific_read_with_audio_after_write(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	if (common_read_after_write(file, sig) != 0)
		return 1;
	return specific_read_with_audio(file, sig, num_samples);
}

/* ------------------------------------------------------------------------ *
 * CAS_ENCODE callback functions                                            *
 * ------------------------------------------------------------------------ */

static int write_baud_block(A8CAS_FILE *file, uint16_t baudrate)
{
	return 0;
}

static int write_data_block(A8CAS_FILE *file, uint16_t irg_ms, uint8_t const *buffer, uint16_t length)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int write_length;

	if (length < 4) { /* Not a standard SIO record */
		file->errnum = A8CAS_ERR_BADFILE;
		return 1;
	}
	if (buffer[0] == 0x55 && buffer[1] == 0x55) {
		/* Only standard SIO blocks are supported by raw format. */
		switch (buffer[2]) {
		case 0xfc: /* Full 128-byte record */
			write_length = length - 4;
			break;
		case 0xfa: /* Partially-filled record */
			write_length = buffer[length - 2];
			if (write_length >= length - 4) {
				file->errnum = A8CAS_ERR_BADFILE;
				return 1;
			}
			break;
		case 0xfe: /* EOF block */
			return 0;
		default: /* Not a standard SIO block */
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}
	}
	else {
		file->errnum = A8CAS_ERR_BADFILE;
		return 1;
	}

	if (fwrite(buffer + 3, sizeof(uint8_t), write_length, data->file) != write_length) {
		file->errnum = A8CAS_ERR_FWRITE;
		return 1;
	}

	data->offset += write_length;
	data->length = data->offset;
	return 0;
}

static int write_fsk_block(A8CAS_FILE *file, uint16_t irg_ms, uint16_t *buffer, uint16_t length)
{
	/* FSK blocks are not supported. */
	file->errnum = A8CAS_ERR_BADFILE;
	return 1;
}

/* ------------------------------------------------------------------------ *
 * Write functions                                                          *
 * ------------------------------------------------------------------------ */

static int specific_write(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (A8CAS_fatal_error(file))
		return 1;
	return CAS_ENCODE_write(&data->cas_encode, sig);
}

int FORMAT_RAW_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((*file->write_func)(file, sig) != 0)
		return 1;
	FSK_MOD_generate(&data->read_audio.mod.fsk, samples, sig->length, sig->signal);
	return 0;
}

static void reset_before_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	file->read_func = &specific_read_after_write;
	file->read_with_audio_func = &specific_read_with_audio_after_write;
	file->write_func = &specific_write;
	file->write_bytes_func = &specific_write_bytes;
	file->errnum = A8CAS_ERR_NONE;
	CAS_ENCODE_reset(&data->cas_encode, 600);
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (fseek(data->file, data->offset, SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	if (IO_ftruncate(data->file, data->offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	data->length = data->offset;

	reset_before_write(file);

	return specific_write(file, sig);
}

static int specific_write_bytes_after_read(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (fseek(data->file, data->offset, SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	if (IO_ftruncate(data->file, data->offset) != 0) {
		file->errnum = A8CAS_ERR_FTRUNCATE;
		return 1;
	}
	data->length = data->offset;

	reset_before_write(file);

	return specific_write_bytes(file, bytes, size, baudrate);
}

/* ------------------------------------------------------------------------ *
 * Block & seeking functions                                                *
 * ------------------------------------------------------------------------ */

unsigned long FORMAT_RAW_tell(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned long pos = (unsigned long)(data->offset >> 7);

	if (data->offset >= data->length) {
		if (data->reading_all0)
			pos ++;
		else
			pos += 2;
	}
	return pos;
}

unsigned long FORMAT_RAW_size(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	/* Remembering about the last "all 0's" record.
	   0->1
	   1->2
	   ...
	   128->2
	   129->3
	   ...
	   256->3
	   257->4
	   ... */
	return (unsigned long)((data->length - 1) / 128 + 2);
}

int FORMAT_RAW_seek(A8CAS_FILE *file, unsigned long position)
{
	file_specific *data = (file_specific*) file->specific_data;
	long offset = position * 128;

	if (flush_data(file) != 0)
		return 1;

	if (offset >= data->length) {
		data->reading_all0 = (offset - data->length < 128);
		data->offset = data->length;
	}
	else {
		data->reading_all0 = 0;
		data->offset = offset;
	}

	if (fseek(data->file, data->offset, SEEK_SET) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}
	clearerr(data->file);

	reset_before_read(file);
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Functions for adjusting encoding/decoding options and parameters         *
 * ------------------------------------------------------------------------ */
int FORMAT_RAW_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		if (CAS_ENCODE_set_block_header_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		if (CAS_ENCODE_set_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		if (CAS_ENCODE_set_stop_bit_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		if (CAS_ENCODE_set_bit_middle(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		if (CAS_ENCODE_set_block_header_length(&data->cas_encode, *((unsigned int *)value)))
			return 1;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		if (CAS_ENCODE_set_baudrate_deviation(&data->cas_encode, *((double *)value)))
			return 1;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		if (READ_AUDIO_set_crosstalk_volume(&data->read_audio, *((float *)value)))
			return 1;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

int FORMAT_RAW_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_BLOCK_HEADER_DEVIATION:
	case A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION:
		 *((double *) value) = data->cas_encode.block_header_deviation;
		break;
	case A8CAS_PARAM_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.bit_deviation;
		break;
	case A8CAS_PARAM_STOP_BIT_DEVIATION:
		 *((double *) value) = data->cas_encode.stop_bit_deviation;
		break;
	case A8CAS_PARAM_BIT_TIMESHIFT:
		 *((double *) value) = data->cas_encode.bit_middle;
		break;
	case A8CAS_PARAM_BLOCK_HEADER_LENGTH:
		 *((unsigned int *) value) = data->cas_encode.block_header_length;
		break;
	case A8CAS_PARAM_BAUDRATE_DEVIATION:
		 *((double *) value) = data->cas_encode.baudrate_deviation;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		*((float *) value) = data->read_audio.crosstalk_volume;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Initialisation functions                                                 *
 * ------------------------------------------------------------------------ */

/* Measures the file's length.
   Returns 0 on success, otherwise 1 and sets file->errnum. */
static int initialise_length(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (fseek(data->file, 0l, SEEK_END) != 0) {
		file->errnum = A8CAS_ERR_FSEEK;
		return 1;
	}

	if ((data->length = ftell(data->file)) == -1) {
		file->errnum = A8CAS_ERR_FTELL;
		return 1;
	}

	rewind(data->file);
	return 0;
}

int FORMAT_RAW_open(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	file_specific *data;

	if ((data = file->specific_data = malloc(sizeof (*data))) == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}

	/* NULLize allocatable pointers. */
	data->file = NULL;
	CAS_ENCODE_init(&data->cas_encode);

	data->offset = 0l;
	data->reading_all0 = 0;

	READ_AUDIO_init(&data->read_audio, file);

	data->cas_decode.file = file;
	data->cas_decode.read_header_func = &read_cas_header;
	data->cas_decode.read_fsk_signal_func = NULL;
	data->cas_decode.read_data_byte_func = &read_data_byte;
	data->cas_decode.read_pwms_block_func = NULL;
	data->cas_decode.read_pwmd_byte_func = NULL;
	data->cas_decode.read_pwmc_signal_func = NULL;
	data->cas_decode.read_pwml_signal_func = NULL;
	data->cas_decode.read_description_func = NULL;

	data->cas_encode.file = file;
	data->cas_encode.write_baud_block_func = &write_baud_block;
	data->cas_encode.write_data_block_func = &write_data_block;
	data->cas_encode.write_fsk_block_func = &write_fsk_block;
	data->cas_encode.add_block_offset_func = &add_offset_after_write_block;

	/* Open the file descriptor */
	if ((data->file = FORMAT_STDIO_BASED_open(path, file->mode)) == NULL) {
		file->errnum = A8CAS_ERR_FOPEN;
		return 1;
	}

	/* Initialise to avoid errors when flush_data() is called. */
	if (CAS_ENCODE_alloc(&data->cas_encode) != 0)
		return 1;
	CAS_ENCODE_reset(&data->cas_encode, 600);

	if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
		data->length = 0;
		reset_before_write(file);
	} else { /* file->mode == A8CAS_READ || file->mode == A8CAS_RDWR */
		if (initialise_length(file) != 0)
			return 1;
		info->samplerate = 0;
		reset_before_read(file);
	}

	data->buffer[0] = data->buffer[1] = 0x55; /* The header won't change at any time. */

	return 0;
}
