/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sndfile.h>

#include <assert.h>

#include "format_sndfile.h"
#include "a8cas_file.h"
#include "a8cas.h"
#include "commons.h"
#include "fsk_demod.h"
#include "fsk_mod.h"
#include "pwm_demod.h"
#include "pwm_mod.h"
#include "resample.h"

#define DEFAULT_BUF_SIZE 256
#define AUDIO_BUF_SIZE 1024

typedef struct file_specific file_specific;

struct file_specific {
	SNDFILE *file;
	SF_INFO snd_info;

/* Fields used while writing */
	struct {
		FSK_MOD_t fsk;
		PWM_MOD_t pwm;
	} mod;
	float leftover_sample_length;
	short buffer[DEFAULT_BUF_SIZE];
	unsigned int buffer_write_fill;

/* Fields used while reading */
	/* specific_read */
	char next_signal;
	/* read_fill_buffer */
	unsigned int buffer_read_pos;
	unsigned int buffer_read_fill;
	sf_count_t frames_fitting;
	unsigned int signal_channel_num;
	unsigned int audio_channel_num;
	struct {
		FSK_DEMOD_t fsk;
		PWM_DEMOD_t pwm;
	} demod;
	int (*advance_demod_func)(file_specific*, short int);

/* Fields used while block counting and rewinding */
	/* Number of samples already read / number of the next sample
	 * (starting at 0) */
	sf_count_t file_offset;

/* Fields used while reading with audio */
	RESAMPLE_t resample;
	struct {
		short int buffer[AUDIO_BUF_SIZE];
		unsigned int pos;
	} audio;
	unsigned short int crosstalk_volume;

/* Fields used while writing with audio */
	struct {
		FSK_MOD_t fsk;
		PWM_MOD_t pwm;
	} audio_mod;

};

/* ------------------------------------------------------------------------ *
 * Block offset control                                                     *
 * ------------------------------------------------------------------------ */

static int flush_data(file_specific *data)
{
	if (data->buffer_write_fill != 0) {
		if (sf_write_short(data->file, data->buffer, (sf_count_t)data->buffer_write_fill) != (sf_count_t)data->buffer_write_fill)
			return A8CAS_ERR_SNDFILE;
		data->buffer_write_fill = 0;
	}
	return A8CAS_ERR_NONE;
}

int FORMAT_SNDFILE_close(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	int retval = 0;

	if (data != NULL) {
		if ((file->errnum = flush_data(data)) != A8CAS_ERR_NONE)
			retval = 1;
		if (data->file != NULL && sf_close(data->file) != 0) {
			file->errnum = A8CAS_ERR_SNDFILE;
			retval = 1;
		}
		free(data);
	}

	return retval;
}

int FORMAT_SNDFILE_flush(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((file->errnum = flush_data(data)) != A8CAS_ERR_NONE)
		return 1;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Read functions                                                           *
 * ------------------------------------------------------------------------ */
static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig);

static int read_fill_buffer(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((data->buffer_read_fill = sf_read_short(data->file, data->buffer, data->frames_fitting)) == 0) {
		file->errnum = (sf_error(data->file) == SF_ERR_NO_ERROR)? A8CAS_ERR_EOF: A8CAS_ERR_SNDFILE;
		return 1;
	}

	data->buffer_read_pos = 0;
	return 0;
}

static int next_sample(A8CAS_FILE *file, short int *signal_sample, short int *audio_sample)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (data->buffer_read_pos >= data->buffer_read_fill) {
		if (read_fill_buffer(file) != 0)
			return 1;
	}
	*signal_sample = data->buffer[data->buffer_read_pos + data->signal_channel_num];
	if (data->snd_info.channels == 1) /* Mono sound */
		/* Adjust the audio track according to crosstalk volume. */
		*audio_sample = data->buffer[data->buffer_read_pos + data->audio_channel_num] * data->crosstalk_volume / USHRT_MAX;
	else
		/* Add some amount of data track to the audio track due to crosstalk. */
		*audio_sample = data->buffer[data->buffer_read_pos + data->audio_channel_num] + *signal_sample * data->crosstalk_volume / USHRT_MAX;

	data->buffer_read_pos += data->snd_info.channels;
	data->file_offset ++;
	return 0;
}

static void fill_audio_buffer(A8CAS_FILE *file, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	while (!data->resample.need && *num_samples < AUDIO_BUF_SIZE) {
		short int audio_sample;
		if (RESAMPLE_next(&data->resample, &audio_sample)) {
			if (file->audio.bits == 16)
				((short int *)data->audio.buffer)[(*num_samples) ++] = audio_sample;
			else
				((unsigned char *)data->audio.buffer)[(*num_samples) ++] = (unsigned char)((audio_sample >> 8) + 0x80);
		}
	}
}

static int specific_read(A8CAS_FILE *file, A8CAS_signal *sig) {
	file_specific *data = (file_specific*) file->specific_data;

	sig->length = 0;
	sig->rate = (unsigned int) data->snd_info.samplerate;
	sig->signal = data->next_signal;

	for (;;) {
		short int sample;
		short int audio_sample;

		if (next_sample(file, &sample, &audio_sample) != 0) {
			if (file->errnum == A8CAS_ERR_EOF && sig->length != 0) {
				return 0;
			}
			return 1;
		}
		sig->length ++;

		if ((*data->advance_demod_func)(data, sample) != data->next_signal) {
			/* Signal has changed -> exit the loop */
			data->next_signal = !data->next_signal;
			return 0;
		}
	}
}

/* ------------------------------------------------------------------------ *
 * Fuunctions to read full bytes/signals                                    *
 * ------------------------------------------------------------------------ */

unsigned int FORMAT_SNDFILE_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

unsigned int FORMAT_SNDFILE_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

static int specific_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate)
{
	file->errnum = A8CAS_ERR_UNSUPPORTED;
	return 0;
}

/* ------------------------------------------------------------------------ *
 * Audio playback functions                                                 *
 * ------------------------------------------------------------------------ */

void FORMAT_SNDFILE_set_audio(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	FSK_MOD_init(&data->audio_mod.fsk, file->audio.samplerate, file->audio.bits, 1, 0, A8CAS_MARK_FREQ, A8CAS_SPACE_FREQ);
	RESAMPLE_init(&data->resample, data->snd_info.samplerate, file->audio.samplerate);
}

static int specific_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	sig->length = 0;
	sig->rate = (unsigned int) data->snd_info.samplerate;
	sig->signal = data->next_signal;

	*num_samples = 0;

	data->audio.pos = 0;

	for (;;) {
		short int sample;
		short int audio_sample;

		fill_audio_buffer(file, num_samples);
		if (*num_samples >= AUDIO_BUF_SIZE)
			return 0;
		if (next_sample(file, &sample, &audio_sample) != 0) {
			if (file->errnum == A8CAS_ERR_EOF && sig->length != 0) {
				return 0;
			}
			return 1;
		}
		sig->length ++;

		if (RESAMPLE_next(&data->resample, &audio_sample)) {
			if (file->audio.bits == 16)
				((short int *)data->audio.buffer)[(*num_samples) ++] = audio_sample;
			else
				((unsigned char *)data->audio.buffer)[(*num_samples) ++] = (unsigned char)((audio_sample >> 8) + 0x80);
		}

		fill_audio_buffer(file, num_samples);
		if ((*data->advance_demod_func)(data, sample) != data->next_signal) {
			/* Signal has changed -> exit the loop */
			data->next_signal = !data->next_signal;
			return 0;
		}
		if (*num_samples >= AUDIO_BUF_SIZE)
			return 0;
	}
}

int FORMAT_SNDFILE_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (num_samples > AUDIO_BUF_SIZE - data->audio.pos)
		num_samples = AUDIO_BUF_SIZE - data->audio.pos;
	if (file->audio.bits == 16)
		memcpy(samples, ((short int *)data->audio.buffer) + data->audio.pos, num_samples << 1);
	else
		memcpy(samples, ((unsigned char *)data->audio.buffer) + data->audio.pos, num_samples);
	data->audio.pos += num_samples;
	return num_samples;
}

/* ------------------------------------------------------------------------ *
 * Read after write functions                                               *
 * ------------------------------------------------------------------------ */

static void reset_before_read(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	data->buffer_read_pos = 0;
	data->buffer_read_fill = 0;
	data->buffer_write_fill = 0;
	data->next_signal = 1;

	file->read_func = &specific_read;
	file->read_with_audio_func = &specific_read_with_audio;
	file->write_func = &specific_write_after_read;
	file->errnum = A8CAS_ERR_NONE;
}

static int common_read_after_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((file->errnum = flush_data(data)) != A8CAS_ERR_NONE)
		return 1;

	/* libsndfile keeps track of read and write locations separately. Have to
	   seek the file's read position to that of its write position. */
	if (sf_seek(data->file, 0, SEEK_CUR) == -1) {
		file->errnum = A8CAS_ERR_SNDFILE;
		return 1;
	}

	reset_before_read(file);
	return 0;
}

static int specific_read_after_write(A8CAS_FILE *file, A8CAS_signal *sig)
{
	if (common_read_after_write(file) != 0)
		return 1;
	return specific_read(file, sig);
}

static int specific_read_with_audio_after_write(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples)
{
	if (common_read_after_write(file) != 0)
		return 1;
	return specific_read_with_audio(file, sig, num_samples);
}

/* ------------------------------------------------------------------------ *
 * Write functions                                                          *
 * ------------------------------------------------------------------------ */

static int specific_write_fsk_signal(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;
	unsigned int sample_time;
	unsigned int samples_to_write;

	double length = data->leftover_sample_length + (float)sig->length * data->snd_info.samplerate / sig->rate;
	sample_time = samples_to_write = (unsigned int) length;
	
	data->leftover_sample_length = length - sample_time;
/*printf("length: %u\n", sample_time);*/
	for (;;)
	{
		unsigned int free_space = DEFAULT_BUF_SIZE - data->buffer_write_fill;
		if (samples_to_write >= free_space) { /* wrap */
			FSK_MOD_generate(&data->mod.fsk, data->buffer + data->buffer_write_fill,
			                 free_space, sig->signal);
			data->buffer_write_fill = DEFAULT_BUF_SIZE;
			if ((file->errnum = flush_data(data)) != A8CAS_ERR_NONE)
				return 1;
			data->buffer_write_fill = 0;
			samples_to_write -= free_space;
		} else {
			FSK_MOD_generate(&data->mod.fsk, data->buffer + data->buffer_write_fill,
			                 samples_to_write, sig->signal);
			data->buffer_write_fill += samples_to_write;
			data->file_offset += (sf_count_t)sample_time;
			data->snd_info.frames = data->file_offset;
			return 0;
		}
	}
}

int FORMAT_SNDFILE_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples)
{
	file_specific *data = (file_specific*) file->specific_data;

	if ((*file->write_func)(file, sig) != 0)
		return 1;
	FSK_MOD_generate(&data->audio_mod.fsk, samples, sig->length, sig->signal);
	return 0;
}


static void reset_before_write(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;

	memset(data->buffer, (short)0, DEFAULT_BUF_SIZE);
	data->buffer_write_fill = 0;
	data->leftover_sample_length = 0.f;

	file->write_func = &specific_write_fsk_signal;
	file->read_func = &specific_read_after_write;
	file->read_with_audio_func = &specific_read_with_audio_after_write;
	file->errnum = A8CAS_ERR_NONE;
}

static int specific_write_after_read(A8CAS_FILE *file, A8CAS_signal const *sig)
{
	file_specific *data = (file_specific*) file->specific_data;

	if (sf_command(data->file, SFC_FILE_TRUNCATE, &data->file_offset, sizeof(sf_count_t)) != 0) {
		file->errnum = A8CAS_ERR_SNDFILE;
		return 1;
	}
	data->snd_info.frames = data->file_offset;

	reset_before_write(file);
	return specific_write_fsk_signal(file, sig);
}

/* ------------------------------------------------------------------------ *
 * Block & seeking functions                                                *
 * ------------------------------------------------------------------------ */

unsigned long FORMAT_SNDFILE_tell(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->file_offset;
}

unsigned long FORMAT_SNDFILE_size(A8CAS_FILE *file)
{
	file_specific *data = (file_specific*) file->specific_data;
	return (unsigned long)data->snd_info.frames;
}

int FORMAT_SNDFILE_seek(A8CAS_FILE *file, unsigned long position)
{
	file_specific *data = (file_specific*) file->specific_data;
	sf_count_t pos = (sf_count_t) position;

	if ((file->errnum = flush_data(data)) != A8CAS_ERR_NONE)
		return 1;
	if (pos >= data->snd_info.frames)
		pos = data->snd_info.frames;
	if (sf_seek(data->file, pos, SEEK_SET) == -1) {
		file->errnum = A8CAS_ERR_SNDFILE;
		return 1;
	}

	data->file_offset = pos;

	reset_before_read(file);

	return 0;
}

/* ------------------------------------------------------------------------ *
 * Functions for turbo systems                                              *
 * ------------------------------------------------------------------------ */
static int advance_fsk_demod(file_specific *data, short int sample)
{
	return FSK_DEMOD_advance(&data->demod.fsk, sample);
}

static int advance_pwm_demod(file_specific *data, short int sample)
{
	return PWM_DEMOD_advance(&data->demod.pwm, sample);
}

/* ------------------------------------------------------------------------ *
 * Functions for adjusting encoding/decoding options and parameters         *
 * ------------------------------------------------------------------------ */
int FORMAT_SNDFILE_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_PWM_DECODE:
		if (*((int *) value)) {
			PWM_DEMOD_init(&data->demod.pwm);
			data->advance_demod_func = &advance_pwm_demod;
		} else {
			FSK_DEMOD_init(&data->demod.fsk, data->snd_info.samplerate);
			data->advance_demod_func = &advance_fsk_demod;
		}
		break;
	case A8CAS_PARAM_PWM_TOLERANCE:
		data->demod.pwm.tolerance = *((unsigned short int *) value);
		break;
	case A8CAS_PARAM_SILENCE_LEVEL:
		data->demod.fsk.silence_threshold = *((float *)value);
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		FSK_MOD_set_volume(&data->audio_mod.fsk, *((float *)value));
		data->crosstalk_volume = *((float *)value) * USHRT_MAX;
		break;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}

int FORMAT_SNDFILE_get_param(A8CAS_FILE *file, A8CAS_param type, void *value)
{
	file_specific *data = (file_specific*) file->specific_data;

	switch (type) {
	case A8CAS_PARAM_PWM_DECODE:
		*((int *) value) = (data->advance_demod_func == &advance_pwm_demod);
		break;
	case A8CAS_PARAM_PWM_TOLERANCE:
		*((unsigned short int *) value) = data->demod.pwm.tolerance;
		break;
	case A8CAS_PARAM_SILENCE_LEVEL:
		*((float *) value) = data->demod.fsk.silence_threshold;
		break;
	case A8CAS_PARAM_CROSSTALK_VOLUME:
		*((float *) value) = (float)data->crosstalk_volume / USHRT_MAX;
	default:
		file->errnum = A8CAS_ERR_UNSUPPORTED;
		return 1;
	}
	return 0;
}


/* ------------------------------------------------------------------------ *
 * Initialisation functions                                                 *
 * ------------------------------------------------------------------------ */

int FORMAT_SNDFILE_open(A8CAS_FILE *file, char const *path, A8CAS_info *info)
{
	file_specific *data;
	int sf_mode;

	if ((data = file->specific_data = malloc(sizeof (*data))) == NULL) {
		file->errnum = A8CAS_ERR_NOMEM;
		return 1;
	}
	/* NULLize allocatable pointers. */
	data->file = NULL;

	data->buffer_write_fill = 0;

	file->write_bytes_func = &specific_write_bytes;

	data->file_offset = 0;

	switch (file->mode) {
	case A8CAS_WRITE:
		sf_mode = SFM_WRITE;
		break;
	case A8CAS_READ:
		sf_mode = SFM_READ;
		break;
	default: /* A8CAS_RDWR, A8CAS_WRRD */
		sf_mode = SFM_RDWR;
	}

	if (file->mode == A8CAS_WRITE || file->mode == A8CAS_WRRD) {
		data->snd_info.samplerate = info->samplerate;
		data->snd_info.channels = 1;
		data->snd_info.format = SF_FORMAT_WAV | SF_FORMAT_PCM_16;

		if ((data->file = sf_open(path, sf_mode, &data->snd_info)) == NULL) {
			file->errnum = A8CAS_ERR_SNDFILE;
			return 1;
		}

		data->snd_info.frames = 0;

		reset_before_write(file);
	} else { /* file->mode == A8CAS_READ || file->mode == A8CAS_RDWR */
		data->snd_info.format = 0;
		if ((data->file = sf_open(path, sf_mode, &data->snd_info)) == NULL) {
			file->errnum = A8CAS_ERR_BADFILE;
			return 1;
		}

		/* If samplerate is too low, it won't be possible to properly
		   decode FSK signals. */
		if (data->snd_info.samplerate < FSK_DEMOD_downsample_rate) {
			file->errnum = A8CAS_ERR_SAMPLERATE;
			return 1;
		}
		/* If number of channels is greater than the buffer's size,
		 * I won't be able to read a sound frame properly. */
		if (data->snd_info.channels > DEFAULT_BUF_SIZE) {
			file->errnum = A8CAS_ERR_LIMIT;
			return 1;
		}

		data->next_signal = 1;

		info->samplerate = data->snd_info.samplerate;

		reset_before_read(file);
	}

	data->frames_fitting = (sf_count_t) (DEFAULT_BUF_SIZE / data->snd_info.channels);
	data->signal_channel_num = (data->snd_info.channels >= 2 ? 1 : 0);
	data->audio_channel_num = 0;

	{
		int value = 0;
		FORMAT_SNDFILE_set_param(file, A8CAS_PARAM_PWM_DECODE, &value);
	}
	data->demod.pwm.tolerance = PWM_DEMOD_DEFAULT_TOLERANCE;
	data->demod.fsk.silence_threshold = FSK_DEMOD_DEFAULT_SILENCE_THRESHOLD;

	data->crosstalk_volume = USHRT_MAX / 20;
	FSK_MOD_init(&data->mod.fsk, data->snd_info.samplerate, 16,
	             data->snd_info.channels, data->signal_channel_num,
	             A8CAS_MARK_FREQ, A8CAS_SPACE_FREQ);

	return 0;
}
