/*
 * Copyright (c) 2010-2013 Tomasz Krasuski (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
/*#define OUTPUT_AUDIO*/
/*#define DEBUG_BLOCKS*/

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <string.h>

#include <a8cas.h>

#ifdef OUTPUT_AUDIO
#include <sndfile.h>
#define AUDIO_BUFFER_SIZE 4096
#endif /* OUTPUT_AUDIO */

#include "config.h"

enum {
	PARAM_HEADER_LENGTH = 0,
	PARAM_HEADER_DEVIATION,
	PARAM_BIT_DEVIATION,
	PARAM_STOP_BIT_DEVIATION,
	PARAM_BIT_TIMESHIFT,
	PARAM_BAUDRATE_DEVIATION,
	PARAM_SILENCE_LEVEL,
	PARAM_SIZE /* Number of A8CAS parameters supported by a8cas-convert */
};

static void print_help(char const *name)
{
	/* The help string has to be split into segments of length supported
	   by the ISO C standard (509 bytes). */
	fprintf(stderr, "Usage: %s <IN_file> <OUT_file>\n"
	                "  -s, --samplerate=integer             Output sample rate (default: 44100)\n"
	                "  -f, --format=c|f|h|s|r               Output format (default: \"c\")\n"
	                "  -d, --desc=string                    CAS description\n"
	                "  -r, --raw                            Input file is in raw format\n"
	                "  Adjustments for recognising blocks of bytes:\n"
	                "      --header-length=integer          Number of signals needed to recognise block\n", name);
	fprintf(stderr, "      --header-deviation=float         Max. deviation between bit lengths to recognise block\n"
	                "      --bit-deviation=float            Max. deviation from ideal bit lengths in records' bytes\n"
	                "      --stop-bit-deviation=float       Max. deviation from ideal stop bit lengths in records' bytes\n"
	                "      --bit-timeshift=float            Distance of the shift when decoding bits (0.5 is the middle)\n");
	fprintf(stderr, "      --baudrate-deviation=float       Max. ratio between blocks' baudrates to avoid a baud block\n"
	                "      --silence-level=float            Max. amplitude (0..1) under which the signal will be treated as silence\n"
	                "  -v, --version                        Show version number and exit\n"
	                "  -?, --help                           Show this help message and exit\n");
}

static void print_version(void)
{
	printf("A8CAS-Convert v" VERSION "\n");
}

static char const *open_error_string(A8CAS_errnum errnum)
{
	switch (errnum) {
	case A8CAS_ERR_NOMEM:
		return "No memory";
	case A8CAS_ERR_BADFILE:
		return "File format not recognised";
	case A8CAS_ERR_SAMPLERATE:
		return "Too low sample rate in file";
	default:
		return NULL;
	}
}

static char const *set_param_error_string(A8CAS_errnum errnum)
{
	switch (errnum) {
	case A8CAS_ERR_UNSUPPORTED:
		return "File format does not support this parameter";
	case A8CAS_ERR_INVALID:
		return "Value not valid";
	case A8CAS_ERR_NOMATCH:
		return "Parameter cannot be changed at this moment";
	case A8CAS_ERR_NOMEM:
		return "No memory";
	default:
		return NULL;
	}
}

static char const *read_error_string(A8CAS_errnum errnum)
{
	switch (errnum) {
	case A8CAS_ERR_NOMEM:
		return "No memory";
	case A8CAS_ERR_BADFILE:
		return "Broken file structure";
	default:
		return NULL;
	}
}

static char const *write_error_string(A8CAS_errnum errnum)
{
	switch (errnum) {
	case A8CAS_ERR_NOMEM:
		return "No memory";
	case A8CAS_ERR_BADFILE:
		return "File format does not support the data being written";
	default:
		return NULL;
	}
}

int main(int argc, char * const argv[])
{
	A8CAS_FILE *in_file;
	A8CAS_FILE *out_file;
	int error;
	char const *in_name = NULL, *out_name = NULL;
/*	static int mark_freq = A8CAS_MARK_FREQ;
	static int space_freq = A8CAS_SPACE_FREQ;*/
	A8CAS_signal sig;
	A8CAS_info in_info;
	A8CAS_info out_info;

	int header_length;
	double header_deviation;
	double bit_deviation;
	double stop_bit_deviation;
	double bit_timeshift;
	double baudrate_deviation;
	float silence_level;

	struct {
		char const * const name;
		A8CAS_param const a8cas_id;
		int set;
		A8CAS_FILE ** const file;
		void * const value;
	} params[PARAM_SIZE] = {
		{ "header-length", A8CAS_PARAM_BLOCK_HEADER_LENGTH, 0, &out_file, &header_length },
		{ "header-deviation", A8CAS_PARAM_BLOCK_HEADER_DEVIATION, 0, &out_file, &header_deviation },
		{ "bit-deviation", A8CAS_PARAM_BIT_DEVIATION, 0, &out_file, &bit_deviation },
		{ "stop-bit-deviation", A8CAS_PARAM_STOP_BIT_DEVIATION, 0, &out_file, &stop_bit_deviation },
		{ "bit-timeshift", A8CAS_PARAM_BIT_TIMESHIFT, 0, &out_file, &bit_timeshift },
		{ "baudrate-deviation", A8CAS_PARAM_BAUDRATE_DEVIATION, 0, &out_file, &baudrate_deviation },
		{ "silence-level", A8CAS_PARAM_SILENCE_LEVEL, 0, &in_file, &silence_level },
	};

#ifdef OUTPUT_AUDIO
	SNDFILE *output_audio;
	SF_INFO snd_info;
	short audio_buffer[AUDIO_BUFFER_SIZE];
#endif /* OUTPUT_AUDIO */
#ifdef DEBUG_BLOCKS
	unsigned int in_block;
	unsigned int out_block;
#endif /* DEBUG_BLOCKS */

	static struct option const long_options[] = {
		{"samplerate", required_argument, NULL, 's'},
		{"format", required_argument, NULL, 'f'},
		{"desc", required_argument, NULL, 'd'},
		{"raw", no_argument, NULL, 'r'},
		{"header-length", required_argument, NULL, 256 + PARAM_HEADER_LENGTH},
		{"header-deviation", required_argument, NULL, 256 + PARAM_HEADER_DEVIATION},
		{"bit-deviation", required_argument, NULL, 256 + PARAM_BIT_DEVIATION},
		{"stop-bit-deviation", required_argument, NULL, 256 + PARAM_STOP_BIT_DEVIATION},
		{"bit-timeshift", required_argument, NULL, 256 + PARAM_BIT_TIMESHIFT},
		{"baudrate-deviation", required_argument, NULL, 256 + PARAM_BAUDRATE_DEVIATION},
		{"silence-level", required_argument, NULL, 256 + PARAM_SILENCE_LEVEL},
		{"version", no_argument, NULL, 'v'},
		{"help", no_argument, NULL, '?'},
		{0, 0, 0, 0}
	};
	int opt;

	in_info.format = A8CAS_FORMAT_ANY;
	in_info.mode = A8CAS_READ;

	out_info.samplerate = 44100;
	out_info.format = A8CAS_FORMAT_CAS;
	out_info.mode = A8CAS_WRITE;
	out_info.description = NULL;

	while ((opt = getopt_long(argc, argv, "s:f:d:rv?", long_options, NULL)) != -1) {
		switch (opt) {
		case 's':
			out_info.samplerate = atoi(optarg);
			break;
		case 'f':
			/* Check the format option */
			error = 0;
			if (strlen(optarg) != 1)
				error = 1;
			switch (optarg[0]) {
			case 'c':
				out_info.format = A8CAS_FORMAT_CAS;
				break;
			case 'f':
				out_info.format = A8CAS_FORMAT_FSK;
				break;
			case 'h':
				out_info.format = A8CAS_FORMAT_HEX;
				break;
			case 's':
				out_info.format = A8CAS_FORMAT_SNDFILE;
				break;
			case 'r':
				out_info.format = A8CAS_FORMAT_RAW;
				break;
			default:
				error = 1;
			}
			if (error != 0) {
				fprintf(stderr, "-f: bad format given\n");
				exit(EXIT_FAILURE);
			}
			break;
		case 'd':
			out_info.description = optarg;
			break;
		case 'r':
			in_info.format = A8CAS_FORMAT_RAW;
			break;
		case 256 + PARAM_HEADER_LENGTH:
			header_length = atoi(optarg);
			params[PARAM_HEADER_LENGTH].set = 1;
			break;
		case 256 + PARAM_HEADER_DEVIATION:
			header_deviation = strtod(optarg, NULL);
			params[PARAM_HEADER_DEVIATION].set = 1;
			break;
		case 256 + PARAM_BIT_DEVIATION:
			bit_deviation = strtod(optarg, NULL);
			params[PARAM_BIT_DEVIATION].set = 1;
			break;
		case 256 + PARAM_STOP_BIT_DEVIATION:
			stop_bit_deviation = strtod(optarg, NULL);
			params[PARAM_STOP_BIT_DEVIATION].set = 1;
			break;
		case 256 + PARAM_BIT_TIMESHIFT:
			bit_timeshift = strtod(optarg, NULL);
			params[PARAM_BIT_TIMESHIFT].set = 1;
			break;
		case 256 + PARAM_BAUDRATE_DEVIATION:
			baudrate_deviation = strtod(optarg, NULL);
			params[PARAM_BAUDRATE_DEVIATION].set = 1;
			break;
		case 256 + PARAM_SILENCE_LEVEL:
			silence_level = strtof(optarg, NULL);
			params[PARAM_SILENCE_LEVEL].set = 1;
			break;
		case 'v':
			print_version();
			exit(EXIT_SUCCESS);
		case '?':
			print_help(argv[0]);
			exit(EXIT_FAILURE);
		}
	}

	if (optind >= argc) {
		fprintf(stderr, "Missing input filename\n");
		exit(EXIT_FAILURE);
	}
	in_name = argv[optind];
	if (optind + 1 >= argc) {
		fprintf(stderr, "Missing output filename\n");
		exit(EXIT_FAILURE);
	}
	out_name = argv[optind + 1];

	if ((in_file = A8CAS_open(in_name, &in_info)) == NULL) {
		char const * const err_string = open_error_string(in_info.errnum);
		if (err_string == NULL)
			fprintf(stderr, "Can't open %s (code %d)\n", in_name, in_info.errnum);
		else
			fprintf(stderr, "Can't open %s (%s)\n", in_name, err_string);
		exit(EXIT_FAILURE);
	}
	A8CAS_set_log_stream(in_file, stderr);

	if (out_info.description == NULL)
		out_info.description = in_info.description;

	if ((out_file = A8CAS_open(out_name, &out_info)) == NULL) {
		char const * const err_string = open_error_string(out_info.errnum);
		if (err_string == NULL)
			fprintf(stderr, "Can't open %s (code %d)\n", out_name, out_info.errnum);
		else
			fprintf(stderr, "Can't open %s (%s)\n", out_name, err_string);
		A8CAS_close(in_file);
		exit(EXIT_FAILURE);
	}
	A8CAS_set_log_stream(out_file, stderr);

	/* Set parameters. */
	{
		int i;
		for (i = 0; i < PARAM_SIZE; ++i) {
			if (params[i].set) {
				if (A8CAS_set_param(*params[i].file, params[i].a8cas_id, params[i].value)) {
					error = A8CAS_error(*params[i].file);
					{
						char const * const err_string = set_param_error_string(error);
						if (err_string == NULL)
							fprintf(stderr, "Can't set parameter %s (code %d)\n", params[i].name, error);
						else
							fprintf(stderr, "Can't set parameter %s (%s)\n", params[i].name, err_string);
					}
					A8CAS_close(out_file);
					A8CAS_close(in_file);
					exit(EXIT_FAILURE);
				}
			}
		}
	}
	
#ifdef OUTPUT_AUDIO
	snd_info.samplerate = sample_rate;
	snd_info.channels = 1;
	snd_info.format = SF_FORMAT_WAV | SF_FORMAT_PCM_16;
	if ((output_audio = sf_open("output_audio.wav", SFM_WRITE, &snd_info)) == NULL) {
		fprintf(stderr, "Can't open output_audio.wav\n");
		A8CAS_close(out_file);
		A8CAS_close(in_file);
		exit(EXIT_FAILURE);
	}
	A8CAS_set_audio(in_file, sample_rate, 16);
#endif /* OUTPUT_AUDIO */

#ifdef DEBUG_BLOCKS
	printf("In blocks: %u, out_blocks: %u\n", A8CAS_num_blocks(in_file), A8CAS_num_blocks(out_file));
	in_block = A8CAS_current_block(in_file);
	out_block = A8CAS_current_block(out_file);
	printf("In block: %u, out block: %u\n", in_block, out_block);
#endif /* DEBUG_BLOCKS */

	error = 0;
	for (;;) {
#ifdef OUTPUT_AUDIO
		if (A8CAS_read_audio(in_file, &sig, audio_buffer, AUDIO_BUFFER_SIZE) != 0) {
#else
		if (A8CAS_read(in_file, &sig) != 0) {
#endif /* OUTPUT_AUDIO */
			if ((error = A8CAS_error(in_file)) != A8CAS_ERR_EOF) {
				char const * const err_string = read_error_string(error);
				if (err_string == NULL)
					fprintf(stderr, "Can't read from %s (code %d)\n", in_name, error);
				else
					fprintf(stderr, "Can't read from %s (%s)\n", in_name, err_string);
				A8CAS_close(in_file);
				A8CAS_close(out_file);
#ifdef OUTPUT_AUDIO
				sf_close(output_audio);
#endif /* OUTPUT_AUDIO */
				exit(EXIT_FAILURE);
			} else
				break;
		}
#ifdef DEBUG_BLOCKS
		{
			unsigned int new_block = A8CAS_current_block(in_file);
			if (new_block != in_block) {
				in_block = new_block;
				printf("In block: %u\n", in_block);
			}
		}
#endif /* DEBUG_BLOCKS */

/*		printf("signal: %i %u/%u\n", sig.signal, sig.length, sig.rate);*/
		if (A8CAS_write(out_file, &sig) != 0) {
			error = A8CAS_error(out_file);
			{
				char const * const err_string = write_error_string(error);
				if (err_string == NULL)
					fprintf(stderr, "Can't write to %s (code %d)\n", out_name, error);
				else
					fprintf(stderr, "Can't write to %s (%s)\n", out_name, err_string);
			}
			A8CAS_close(in_file);
			A8CAS_close(out_file);
#ifdef OUTPUT_AUDIO
			sf_close(output_audio);
#endif /* OUTPUT_AUDIO */
			exit(EXIT_FAILURE);
		}
#ifdef DEBUG_BLOCKS
		{
			unsigned int new_block = A8CAS_current_block(out_file);
			if (new_block != out_block) {
				out_block = new_block;
				printf("Out block: %u\n", out_block);
			}
		}
#endif /* DEBUG_BLOCKS */

#ifdef OUTPUT_AUDIO
		if (sf_write_short(output_audio, audio_buffer, sig.length) != sig.length) {
			fprintf(stderr, "Can't write to output_audio.wav\n");
			A8CAS_close(in_file);
			A8CAS_close(out_file);
			sf_close(output_audio);
			exit(EXIT_FAILURE);
		}
#endif /* OUTPUT_AUDIO */
	}

#ifdef OUTPUT_AUDIO
	if (sf_close(output_audio) != 0) {
		fprintf(stderr, "Can't close output_audio.wav\n");
		A8CAS_close(out_file);
		A8CAS_close(in_file);
		exit(EXIT_FAILURE);
	}
#endif /* OUTPUT_AUDIO */

	if ((error = A8CAS_close(out_file)) != 0) {
		fprintf(stderr, "Can't close %s (code %d)\n", out_name, error);
		A8CAS_close(in_file);
		exit(EXIT_FAILURE);
	}
	if ((error = A8CAS_close(in_file)) != 0) {
		fprintf(stderr, "Can't close %s (code %d)\n", in_name, error);
		exit(EXIT_FAILURE);
	}

	exit(EXIT_SUCCESS);
}
