/*
 * Copyright (c) 2010-2013 Tomasz Krasuski (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>

#include <a8cas.h>

#include "config.h"

int main(int argc, char const *argv[])
{
	A8CAS_FILE *file;
	A8CAS_info info;
	int error;

	if (argc < 2) {
		fprintf(stderr, "A8CAS-Info v" VERSION
		        "\nUsage: %s <cassette_file>\n", argv[0]);
		exit(EXIT_FAILURE);
	}

	info.format = A8CAS_FORMAT_ANY;
	info.mode = A8CAS_READ;
	
	if ((file = A8CAS_open(argv[1], &info)) == NULL) {
		fprintf(stderr, "Can't open %s (code %d)\n", argv[1], info.errnum);
		exit(EXIT_FAILURE);
	}

	printf("Filename: %s\n", argv[1]);

	{
		char *format_string;
		switch (info.format) {
		case A8CAS_FORMAT_CAS:
			format_string = "CAS";
			break;
		case A8CAS_FORMAT_HEX:
			format_string = "HEX";
			break;
		case A8CAS_FORMAT_FSK:
			format_string = "FSK";
			break;
		case A8CAS_FORMAT_SNDFILE:
			format_string = "SNDFILE";
			break;
		default:
			format_string = "not recognised!";
		}
		printf("Format: %s\n", format_string);
	}

	if (info.description == NULL)
		printf("Description not applicable\n");
	else
		printf("Description: %s\n", info.description);

	{
		char *unit_string;
		if (A8CAS_seek_unit(file) == A8CAS_SEEK_UNIT_BLOCKS)
			unit_string = "blocks";
		else
			unit_string = "samples";
		printf("Size: %lu %s\n", A8CAS_size(file), unit_string);
	}

	if (info.format == A8CAS_FORMAT_SNDFILE)
		printf("Sample rate: %u Hz\n", info.samplerate);
	else
		printf("Sample rate not applicable\n");
	
	if ((error = A8CAS_close(file)) != A8CAS_ERR_NONE) {
		fprintf(stderr, "Can't close %s (code %d)\n", argv[1], error);
		exit(EXIT_FAILURE);
	}

	exit(EXIT_SUCCESS);
}
