/*
 * cassette.c - cassette emulation
 *
 * Copyright (C) 2001 Piotr Fusik
 * Copyright (C) 2001-2011 Atari800 development team (see DOC/CREDITS)
 *
 * This file is part of the Atari800 emulator project which emulates
 * the Atari 400, 800, 800XL, 130XE, and 5200 8-bit computers.
 *
 * Atari800 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Atari800 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Atari800; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include "antic.h"
#include "atari.h"
#include "cpu.h"
#include "cassette.h"
#include "cfg.h"
#include "esc.h"
#include "img_tape.h"
#include "log.h"
#include "util.h"
#include "pokey.h"

#if HAVE_LIBA8CAS
#include "pia.h"
#ifdef SYNCHRONIZED_SOUND
#include "syncsound.h"
/* Uncomment to write debug informations on audio buffer under/overflow. */
/*#define DEBUG_AUDIO*/
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

static IMG_TAPE_t *cassette_file = NULL;

/* Time till the end of the current tape event (byte or gap), in CPU ticks. */
static double event_time_left = 0.0;

char CASSETTE_filename[FILENAME_MAX];
CASSETTE_status_t CASSETTE_status = CASSETTE_STATUS_NONE;
int CASSETTE_write_protect = FALSE;
int CASSETTE_record = FALSE;
static int cassette_writable = FALSE;
static int cassette_readable = FALSE;

char CASSETTE_description[CASSETTE_DESCRIPTION_MAX];

/* For operations with SIO-patch */
static int cassette_gapdelay = 0;	/* in ms, includes leader and all gaps */

static int cassette_motor = 0;

int CASSETTE_hold_start_on_reboot = 0;
int CASSETTE_hold_start = 0;
int CASSETTE_press_space = 0;
/* Indicates whether the tape has ended. During saving the value is always 0;
   during loading it is equal to (CASSETTE_GetPosition() >= CASSETTE_GetSize()). */
static int eof_of_tape = 0;

static unsigned int prev_cpu_clock;

#if HAVE_LIBA8CAS
int CASSETTE_position_in_blocks;

static int current_signal;
static int cassette_samplerate;

enum CASSETTE_TURBO_TYPE_t CASSETTE_turbo_type = CASSETTE_TURBO_NONE;
static char const * const turbo_type_cfg_strings[CASSETTE_TURBO_TYPE_SIZE] = {
	"NONE",
	"AST-XC12",
	"BLIZZARD",
	"HARD",
	"MANUAL",
	"MANUAL-REV"
};

enum TURBO_CONTROL_t {
	TURBO_CONTROL_NONE,
	TURBO_CONTROL_MANUAL,
	TURBO_CONTROL_DATA_OUT,
	TURBO_CONTROL_COMMAND,
	TURBO_CONTROL_COMMAND_MOTOR
};
static enum TURBO_CONTROL_t turbo_control;
static int turbo_descend_is_1;
int CASSETTE_turbo_state = 0;
static int file_turbo_state = 0;
int CASSETTE_turbo_tolerance = CASSETTE_DEFAULT_TURBO_TOLERANCE;

#ifdef SYNCHRONIZED_SOUND
int CASSETTE_play_audio = 1;
unsigned int CASSETTE_audio_volume = CASSETTE_DEFAULT_AUDIO_VOLUME;
unsigned int CASSETTE_crosstalk_volume = CASSETTE_DEFAULT_CROSSTALK_VOLUME;

static struct {
	unsigned int samplerate;
	unsigned int n_pokeys;
	unsigned int bit16;
	unsigned int buf_size;
	int write_pos;
	double sample_pos;
	unsigned int samples_left_in_signal;
	UBYTE *buffer; /* When BUFFER is NULL, it indicates that audio was not initialised. */
} audio = { 44100, 2, 16, 0, 0, 0.0, 0, NULL };
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

static void CassetteWrite(int num_ticks);
static void CassetteRead(int num_ticks);
static void (*cassette_write_func)(int) = &CassetteWrite;
static void (*cassette_read_func)(int) = &CassetteRead;

/* Call this function after each change of
   cassette_motor, CASSETTE_status or eof_of_tape. */
static void UpdateFlags(void)
{
	cassette_readable = cassette_motor &&
	                    (CASSETTE_status == CASSETTE_STATUS_READ_WRITE ||
	                     CASSETTE_status == CASSETTE_STATUS_READ_ONLY) &&
	                     !eof_of_tape;
	cassette_writable = cassette_motor &&
	                    CASSETTE_status == CASSETTE_STATUS_READ_WRITE &&
	                    !CASSETTE_write_protect;
}

#if HAVE_LIBA8CAS
static void UpdateTurboState(void)
{
	if (CASSETTE_turbo_state != file_turbo_state &&
	    cassette_file != NULL &&
	    !CASSETTE_record) {
		IMG_TAPE_SetPWMDecode(cassette_file, CASSETTE_turbo_state);
		file_turbo_state = CASSETTE_turbo_state;
	}
}

static void ResetTurboState(void)
{
	switch (turbo_control) {
	case TURBO_CONTROL_DATA_OUT:
		CASSETTE_turbo_state = !POKEY_serial_data_output;
		break;
	case TURBO_CONTROL_COMMAND:
		CASSETTE_turbo_state = !(PIA_PBCTL & 0x08);
		break;
	case TURBO_CONTROL_COMMAND_MOTOR:
		CASSETTE_turbo_state = !(PIA_PBCTL & 0x08) && cassette_motor;
		break;
	default:
		break;
	}
	UpdateTurboState();
}

static void UpdateTurboTolerance(void)
{
	unsigned short int value = CASSETTE_turbo_tolerance * USHRT_MAX / 10 / CASSETTE_MAX_TURBO_TOLERANCE;
	IMG_TAPE_SetPWMTolerance(cassette_file, value);
}

#ifdef SYNCHRONIZED_SOUND
static void UpdateCrosstalkVolume(void)
{
	float value = (float)CASSETTE_crosstalk_volume / CASSETTE_MAX_CROSSTALK_VOLUME;
	IMG_TAPE_SetCrosstalkVolume(cassette_file, value);
}
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

int CASSETTE_ReadConfig(char *string, char *ptr)
{
	if (strcmp(string, "CASSETTE_FILENAME") == 0)
		Util_strlcpy(CASSETTE_filename, ptr, sizeof(CASSETTE_filename));
	else if (strcmp(string, "CASSETTE_LOADED") == 0) {
		int value = Util_sscanbool(ptr);
		if (value == -1)
			return FALSE;
		CASSETTE_status = (value ? CASSETTE_STATUS_READ_WRITE : CASSETTE_STATUS_NONE);
	}
	else if (strcmp(string, "CASSETTE_WRITE_PROTECT") == 0) {
		int value = Util_sscanbool(ptr);
		if (value == -1)
			return FALSE;
		CASSETTE_write_protect = value;
	}
#if HAVE_LIBA8CAS
	else if (strcmp(string, "CASSETTE_TURBO_TYPE") == 0) {
		int i = CFG_MatchTextParameter(ptr, turbo_type_cfg_strings, CASSETTE_TURBO_TYPE_SIZE);
		if (i < 0)
			return FALSE;
		CASSETTE_SetTurboType(i);
	}
#ifdef SYNCHRONIZED_SOUND
	else if (strcmp(string, "CASSETTE_PLAY_AUDIO") == 0)
		return (CASSETTE_play_audio = Util_sscanbool(ptr)) != -1;
	else if (strcmp(string, "CASSETTE_AUDIO_VOLUME") == 0)
		return (CASSETTE_audio_volume = Util_sscandec(ptr)) != -1;
	else if (strcmp(string, "CASSETTE_CROSSTALK_VOLUME") == 0) {
		int value = Util_sscandec(ptr);
		if (value == -1)
			return FALSE;
		CASSETTE_SetCrosstalkVolume(value);
	}
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */
	else return FALSE;
	return TRUE;
}
void CASSETTE_WriteConfig(FILE *fp)
{
	fprintf(fp, "CASSETTE_FILENAME=%s\n", CASSETTE_filename);
	fprintf(fp, "CASSETTE_LOADED=%d\n", CASSETTE_status != CASSETTE_STATUS_NONE);
	fprintf(fp, "CASSETTE_WRITE_PROTECT=%d\n", CASSETTE_write_protect);
#if HAVE_LIBA8CAS
	fprintf(fp, "CASSETTE_TURBO_TYPE=%s\n", turbo_type_cfg_strings[CASSETTE_turbo_type]);
#ifdef SYNCHRONIZED_SOUND
	fprintf(fp, "CASSETTE_PLAY_AUDIO=%d\n", CASSETTE_play_audio);
	fprintf(fp, "CASSETTE_AUDIO_VOLUME=%u\n", CASSETTE_audio_volume);
	fprintf(fp, "CASSETTE_CROSSTALK_VOLUME=%u\n", CASSETTE_crosstalk_volume);
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */
}

int CASSETTE_Initialise(int *argc, char *argv[])
{
	int i;
	int j;
	int protect = FALSE; /* Is write-protect requested in command line? */

	prev_cpu_clock = ANTIC_CPU_CLOCK;

	for (i = j = 1; i < *argc; i++) {
		int i_a = (i + 1 < *argc);		/* is argument available? */
		int a_m = FALSE;			/* error, argument missing! */
#if HAVE_LIBA8CAS
		int a_i = FALSE;			/* error, argument invalid! */
#endif /* HAVE_LIBA8CAS */


		if (strcmp(argv[i], "-tape") == 0) {
			if (i_a) {
				Util_strlcpy(CASSETTE_filename, argv[++i], sizeof(CASSETTE_filename));
				CASSETTE_status = CASSETTE_STATUS_READ_WRITE;
				/* Reset any write-protection read from config file. */
				CASSETTE_write_protect = FALSE;
			}
			else a_m = TRUE;
		}
		else if (strcmp(argv[i], "-boottape") == 0) {
			if (i_a) {
				Util_strlcpy(CASSETTE_filename, argv[++i], sizeof(CASSETTE_filename));
				CASSETTE_status = CASSETTE_STATUS_READ_WRITE;
				/* Reset any write-protection read from config file. */
				CASSETTE_write_protect = FALSE;
				CASSETTE_hold_start = 1;
			}
			else a_m = TRUE;
		}
		else if (strcmp(argv[i], "-tape-readonly") == 0)
			protect = TRUE;
#if HAVE_LIBA8CAS
		else if (strcmp(argv[i], "-tape-turbo") == 0) {
			if (i_a) {
				int idx = CFG_MatchTextParameter(argv[++i], turbo_type_cfg_strings, CASSETTE_TURBO_TYPE_SIZE);
				if (idx < 0)
					a_i = TRUE;
				else
					CASSETTE_SetTurboType(idx);
			}
			else a_m = TRUE;
		}
#endif /* HAVE_LIBA8CAS */
		else {
			if (strcmp(argv[i], "-help") == 0) {
				Log_print("\t-tape <file>       Insert cassette image");
				Log_print("\t-boottape <file>   Insert cassette image and boot it");
				Log_print("\t-tape-readonly     Mark the attached cassette image as read-only");
#if HAVE_LIBA8CAS
				Log_print("\t-tape-turbo none|ast-xc12|blizzard|hard|manual|manual-rev");
				Log_print("\t                   Choose tape recorder's turbo modification");
#endif /* HAVE_LIBA8CAS */
			}
			argv[j++] = argv[i];
		}

		if (a_m) {
			Log_print("Missing argument for '%s'", argv[i]);
			return FALSE;
		}
#if HAVE_LIBA8CAS
		else if (a_i) {
			Log_print("Invalid argument for '%s'", argv[--i]);
			return FALSE;
		}
#endif /* HAVE_LIBA8CAS */
	}

	*argc = j;

	/* If CASSETTE_status was set in this function or in CASSETTE_ReadConfig(),
	   then tape is to be mounted. */
	if (CASSETTE_status != CASSETTE_STATUS_NONE && CASSETTE_filename[0] != '\0') {
		/* Tape is mounted unprotected by default - overrun it if needed. */
		protect = protect || CASSETTE_write_protect;
		if (!CASSETTE_Insert(CASSETTE_filename)) {
			CASSETTE_status = CASSETTE_STATUS_NONE;
			Log_print("Cannot open cassette image %s", CASSETTE_filename);
		}
		else if (protect)
			CASSETTE_ToggleWriteProtect();
	}

	return TRUE;
}

void CASSETTE_Exit(void)
{
	CASSETTE_Remove();
#if HAVE_LIBA8CAS && defined(SYNCHRONIZED_SOUND)
	if (audio.buffer != NULL) {
		free(audio.buffer);
	}
#endif
}

int CASSETTE_Insert(const char *filename)
{
	int writable;
	int is_raw;
	char const *description;

	IMG_TAPE_t *file = IMG_TAPE_Open(filename, &writable, &is_raw, &description);
	if (file == NULL)
		return FALSE;

	CASSETTE_Remove();
	cassette_file = file;
	/* Guard against providing CASSETTE_filename as parameter. */
	if (CASSETTE_filename != filename)
		strcpy(CASSETTE_filename, filename);
	eof_of_tape = 0;

	CASSETTE_status = (writable ? CASSETTE_STATUS_READ_WRITE : CASSETTE_STATUS_READ_ONLY);
	event_time_left = 0.0;
	if (description != NULL)
		Util_strlcpy(CASSETTE_description, description, sizeof(CASSETTE_description));

	CASSETTE_write_protect = FALSE;
	CASSETTE_record = FALSE;
	UpdateFlags();

	cassette_gapdelay = 0;

#if HAVE_LIBA8CAS
	cassette_samplerate = IMG_TAPE_GetSamplerate(cassette_file);
	CASSETTE_position_in_blocks = IMG_TAPE_PositionInBlocks(cassette_file);
	file_turbo_state = 0;
	ResetTurboState();
	UpdateTurboTolerance();
#ifdef SYNCHRONIZED_SOUND
	audio.sample_pos = 0.0;
	audio.samples_left_in_signal = 0;
	IMG_TAPE_SetAudio(cassette_file, audio.samplerate, (audio.bit16 ? 16 : 8));
	UpdateCrosstalkVolume();
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

	if (is_raw)
		return 2;
	else
		return TRUE;
}

void CASSETTE_Remove(void)
{
	if (cassette_file != NULL) {
		IMG_TAPE_Close(cassette_file);
		cassette_file = NULL;
	}
	CASSETTE_status = CASSETTE_STATUS_NONE;
	CASSETTE_description[0] = '\0';
	UpdateFlags();
}

static int CreateBlankImage(IMG_TAPE_format_t format, char const *filename, char const *description)
{
	IMG_TAPE_t *file = IMG_TAPE_Create(filename, format, description);
	if (file == NULL)
		return FALSE;

	CASSETTE_Remove(); /* Unmount any previous tape image */
	cassette_file = file;
	Util_strlcpy(CASSETTE_filename, filename, sizeof(CASSETTE_filename));
	if (description != NULL)
		Util_strlcpy(CASSETTE_description, description, sizeof(CASSETTE_description));
	CASSETTE_status = CASSETTE_STATUS_READ_WRITE;
	event_time_left = 0.0;
	cassette_gapdelay = 0;
	eof_of_tape = 0;
	CASSETTE_record = TRUE;
	CASSETTE_write_protect = FALSE;
	UpdateFlags();

#if HAVE_LIBA8CAS
	cassette_samplerate = IMG_TAPE_GetSamplerate(cassette_file);
	CASSETTE_position_in_blocks = IMG_TAPE_PositionInBlocks(cassette_file);
	file_turbo_state = 0;
	ResetTurboState();
	UpdateTurboTolerance();
#ifdef SYNCHRONIZED_SOUND
	audio.sample_pos = 0.0;
	audio.samples_left_in_signal = 0;
	IMG_TAPE_SetAudio(cassette_file, audio.samplerate, (audio.bit16 ? 16 : 8));
	UpdateCrosstalkVolume();
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

	return TRUE;
}

int CASSETTE_CreateCAS(const char *filename, char const *description)
{
	return CreateBlankImage(IMG_TAPE_FORMAT_CAS, filename, description);
}

#if HAVE_LIBA8CAS
int CASSETTE_CreateWAV(char const *filename)
{
	return CreateBlankImage(IMG_TAPE_FORMAT_WAV, filename, NULL);
}

int CASSETTE_CreateRaw(char const *filename)
{
	return CreateBlankImage(IMG_TAPE_FORMAT_RAW, filename, NULL);
}
#endif /* HAVE_LIBA8CAS */

unsigned int CASSETTE_GetPosition(void)
{
	if (cassette_file == NULL)
		return 0;
#if HAVE_LIBA8CAS
	if (CASSETTE_position_in_blocks)
#endif /* HAVE_LIBA8CAS */
		return IMG_TAPE_GetPosition(cassette_file) + 1;
#if HAVE_LIBA8CAS
	else
		return IMG_TAPE_GetPosition(cassette_file) / cassette_samplerate;
#endif /* HAVE_LIBA8CAS */
}

unsigned int CASSETTE_GetSize(void)
{
	if (cassette_file == NULL)
		return 0;
#if HAVE_LIBA8CAS
	if (CASSETTE_position_in_blocks)
#endif /* HAVE_LIBA8CAS */
		return IMG_TAPE_GetSize(cassette_file);
#if HAVE_LIBA8CAS
	else
		return IMG_TAPE_GetSize(cassette_file) / cassette_samplerate;
#endif /* HAVE_LIBA8CAS */
}

void CASSETTE_Seek(unsigned int position)
{
	if (cassette_file != NULL) {
#if HAVE_LIBA8CAS
		if (CASSETTE_position_in_blocks) {
#endif /* HAVE_LIBA8CAS */
			if (position > 0)
				position --;
			IMG_TAPE_Seek(cassette_file, position);
#if HAVE_LIBA8CAS
		} else
			IMG_TAPE_Seek(cassette_file, position * cassette_samplerate);
#endif /* HAVE_LIBA8CAS */
		event_time_left = 0.0;
		eof_of_tape = 0;
		CASSETTE_record = FALSE;
#if HAVE_LIBA8CAS && defined(SYNCHRONIZED_SOUND)
		audio.samples_left_in_signal = 0;
#endif /* HAVE_LIBA8CAS && defined(SYNCHORNIZED_SOUND) */
		UpdateFlags();
	}
}

int CASSETTE_IOLineStatus(void)
{
	/* if motor off and EOF return always 1 (equivalent the mark tone) */
	if (!cassette_readable || CASSETTE_record) {
		return 1;
	}
	if (!ESC_enable_sio_patch)
		CASSETTE_AddScanLine();
#if HAVE_LIBA8CAS
	if (CASSETTE_turbo_state && turbo_descend_is_1)
		return !(int)current_signal;
	else
		return (int)current_signal;
#else /* HAVE_LIBA8CAS */
	return IMG_TAPE_SerinStatus(cassette_file, (int)event_time_left);
#endif /* HAVE_LIBA8CAS */
}

#if !HAVE_LIBA8CAS
void CASSETTE_PutByte(int byte)
{
	if (!ESC_enable_sio_patch && cassette_writable && CASSETTE_record)
		IMG_TAPE_WriteByte(cassette_file, byte, POKEY_AUDF[POKEY_CHAN3] + POKEY_AUDF[POKEY_CHAN4]*0x100);
}
#endif /* HAVE_LIBA8CAS */

/* set motor status
 1 - on, 0 - off
 remark: the real evaluation is done in AddScanLine*/
void CASSETTE_TapeMotor(int onoff)
{
	if (cassette_motor != onoff) {
		if (CASSETTE_record && cassette_writable)
			/* Recording disabled, flush the tape */
			IMG_TAPE_Flush(cassette_file);
		cassette_motor = onoff;
		UpdateFlags();
#if HAVE_LIBA8CAS
		ResetTurboState();
#endif /* HAVE_LIBA8CAS */
	}
}

int CASSETTE_ToggleWriteProtect(void)
{
	if (CASSETTE_status != CASSETTE_STATUS_READ_WRITE)
		return FALSE;
	CASSETTE_write_protect = !CASSETTE_write_protect;
	UpdateFlags();
	return TRUE;
}

int CASSETTE_ToggleRecord(void)
{
	if (CASSETTE_status == CASSETTE_STATUS_NONE)
		return FALSE;
	CASSETTE_record = !CASSETTE_record;
	if (CASSETTE_record)
		eof_of_tape = FALSE;
	else if (cassette_writable)
		/* Recording disabled, flush the tape */
		IMG_TAPE_Flush(cassette_file);
	event_time_left = 0.0;
	UpdateFlags();
#if HAVE_LIBA8CAS
	UpdateTurboState();
#endif /* HAVE_LIBA8CAS */
	/* Return FALSE to indicate that recording will not work. */
	return !CASSETTE_record || (CASSETTE_status == CASSETTE_STATUS_READ_WRITE && !CASSETTE_write_protect);
}

#if HAVE_LIBA8CAS
void CASSETTE_ToggleTurboState(void)
{
	CASSETTE_turbo_state = !CASSETTE_turbo_state;
	UpdateTurboState();
}

void CASSETTE_SetTurboType(enum CASSETTE_TURBO_TYPE_t type)
{
	CASSETTE_turbo_type = type;

	switch (type) {
	case CASSETTE_TURBO_AST_XC12:
		turbo_control = TURBO_CONTROL_COMMAND;
		turbo_descend_is_1 = 0;
		break;
	case CASSETTE_TURBO_BLIZZARD:
		turbo_control = TURBO_CONTROL_DATA_OUT;
		turbo_descend_is_1 = 0;
		break;
	case CASSETTE_TURBO_HARD:
		/* PWM circuit is enabled by asserting COMMAND and turning on the motor. */
		turbo_control = TURBO_CONTROL_COMMAND_MOTOR;
		turbo_descend_is_1 = 0;
		break;
	case CASSETTE_TURBO_MANUAL:
		turbo_control = TURBO_CONTROL_MANUAL;
		turbo_descend_is_1 = 0;
		break;
	case CASSETTE_TURBO_MANUAL_REV:
		turbo_control = TURBO_CONTROL_MANUAL;
		turbo_descend_is_1 = 1;
		break;
	default:
		turbo_control = TURBO_CONTROL_NONE;
	}
	CASSETTE_turbo_state = 0;
	ResetTurboState();
}

void CASSETTE_SetTurboTolerance(int value)
{
	CASSETTE_turbo_tolerance = value;
	if (cassette_file != NULL)
		UpdateTurboTolerance();
}

#ifdef SYNCHRONIZED_SOUND
void CASSETTE_SetCrosstalkVolume(unsigned int value)
{
	CASSETTE_crosstalk_volume = value;
	if (cassette_file != NULL)
		UpdateCrosstalkVolume();
}


/* 16 bit mixing */
static void mix(SWORD *dst, SWORD *src, int sndn, int volume)
{
	int s1;
	SWORD s2;
	int val;

	while (sndn--) {
		s1 = (int) *src;
		s1 = s1*volume/CASSETTE_DEFAULT_AUDIO_VOLUME;
		s2 = *dst;
		src++;
		val = s1 + s2;
		if (val > 32767) val = 32767;
		if (val < -32768) val = -32768;
		*dst++ = val;
		if (audio.n_pokeys == 2) {
			s2 = *dst;
			val = s1 + s2;
			if (val > 32767) val = 32767;
			if (val < -32768) val = -32768;
			*dst++ = val;
		}
	}
}

/* 8 bit mixing */
static void mix8(UBYTE *dst, UBYTE *src, int sndn, int volume)
{
	SWORD s1, s2;
	SWORD val;

	while (sndn--) {
		s1 = (SWORD)(*src) - 0x80;
		s1 = s1*volume/CASSETTE_DEFAULT_AUDIO_VOLUME;
		s2 = (SWORD)(*dst) - 0x80;
		src++;
		val = s1 + s2;
		if (val > 127) val = 127;
		if (val < -128) val = -128;
		*dst++ = (UBYTE)(val + 0x80);
		if (audio.n_pokeys == 2) {
			s2 = (SWORD)(*dst) - 0x80;
			val = s1 + s2;
			if (val > 127) val = 127;
			if (val < -128) val = -128;
			*dst++ = (UBYTE)(val + 0x80);
		}
	}
}

static void WriteSilence(int num_samples)
{
	if (audio.bit16)
		memset(audio.buffer + (audio.write_pos << 1), 0, num_samples << 1);
	else
		memset(audio.buffer + audio.write_pos, 127, num_samples);
	audio.write_pos += num_samples;
}

static void CassetteWriteAudio(int num_ticks)
{
	int num_samples = 0;

	event_time_left -= num_ticks;
	if (event_time_left < 0.0) {
		num_samples = ceil((-event_time_left) / SYNCSOUND_ticks_per_sample);
		event_time_left += num_samples * SYNCSOUND_ticks_per_sample;
	}
	if ((audio.bit16 ? (audio.write_pos + num_samples) << 1 : audio.write_pos + num_samples) > audio.buf_size) {
/*		Log_print("Cassette audio buffer overload: %f", event_time_left);*/
		audio.write_pos = 0;
	}

/*	Log_print("caswrite: num_samples=%i", num_samples);*/
	if (cassette_writable && num_samples > 0) {
/*	Log_print("addscanline: numsamples=%i, signal=%u, tick=%i", num_samples, POKEY_serial_data_output, ANTIC_CPU_CLOCK);*/
		if (!IMG_TAPE_WriteAudio(cassette_file, num_samples, POKEY_serial_data_output, audio.buffer + (audio.bit16 ? audio.write_pos << 1 : audio.write_pos))) {
			WriteSilence(num_samples);
			return;
		}
		audio.write_pos += num_samples;
	} else {
		WriteSilence(num_samples);
	}
}

static void CassetteReadAudio(int num_ticks)
{
	int num_samples = 0;

	if (cassette_readable) {
		event_time_left -= num_ticks;
		while (event_time_left < 0.0) {
			if (audio.samples_left_in_signal > 0) {
				int free_space = (int)((audio.bit16 ? audio.buf_size >> 1 : audio.buf_size) - audio.write_pos);
				unsigned int samples_to_read;
				if (free_space == 0) {
					audio.write_pos = 0;
#ifdef DEBUG_AUDIO					
					Log_print("Audio buffer overflow by %u samples", audio.samples_left_in_signal);
#endif /* DEBUG_AUDIO */
					continue;
				}
				samples_to_read = (free_space < audio.samples_left_in_signal ? free_space : audio.samples_left_in_signal);
				IMG_TAPE_ReadAudio(cassette_file, audio.buffer + (audio.bit16 ? audio.write_pos << 1 : audio.write_pos), samples_to_read);
				audio.samples_left_in_signal -= samples_to_read;
				audio.sample_pos += samples_to_read * SYNCSOUND_ticks_per_sample;
				audio.write_pos += samples_to_read;
				continue;
			}

			{
				unsigned int length;
				if (!IMG_TAPE_ReadWithAudio(cassette_file, &length, &current_signal, &audio.samples_left_in_signal)) {
					/* Ignore the eventual error, assume it is the end of file */
					eof_of_tape = 1;
					UpdateFlags();
					break;
				}
				event_time_left += length;
			}
		}
	}

	audio.sample_pos -= num_ticks;
	if (audio.sample_pos < 0.0) {
		num_samples = ceil((-audio.sample_pos) / SYNCSOUND_ticks_per_sample);
		audio.sample_pos += num_samples * SYNCSOUND_ticks_per_sample;
	}

	{
		int free_space = (int)((audio.bit16 ? audio.buf_size >> 1 : audio.buf_size) - audio.write_pos);
		if (free_space < num_samples) {
#ifdef DEBUG_AUDIO					
			Log_print("Audio buffer overflow by %u samples", num_samples);
#endif /* DEBUG_AUDIO */
			audio.write_pos = 0;
		}
	}

	if (cassette_readable) {
		int free_space = (int)((audio.bit16 ? audio.buf_size >> 1 : audio.buf_size) - audio.write_pos);
		unsigned int samples_to_read = (free_space > audio.samples_left_in_signal ? audio.samples_left_in_signal : free_space);
#ifdef DEBUG_AUDIO					
		if (samples_to_read < num_samples)
			Log_print("Not enough samples to read: %i", num_samples - samples_to_read);
#endif /* DEBUG_AUDIO */
		IMG_TAPE_ReadAudio(cassette_file, audio.buffer + (audio.bit16 ? audio.write_pos << 1 : audio.write_pos), samples_to_read);
		audio.write_pos += samples_to_read;
		audio.samples_left_in_signal -= samples_to_read;
		num_samples -= samples_to_read;
		if (num_samples < 0)
			audio.sample_pos -= num_samples * SYNCSOUND_ticks_per_sample;
	}

	if (num_samples > 0)
		WriteSilence(num_samples);
}

void CASSETTE_InitAudio(unsigned int samplerate, unsigned int n_pokeys, int b16)
{
	audio.samplerate = samplerate;
	audio.n_pokeys = n_pokeys;
	audio.bit16 = b16;
	if (audio.buffer != NULL) {
		free(audio.buffer);
	}
	/* AUDIO.BUF_SIZE has to be a few (5 is OK) bytes larger than needed,
	   because number of CPU cycles per frame is not constant - it varies
	   a bit because CPU emulation stops only after performing a full 6502
	   instruction (which can last 2 to 6 cycles). */
	audio.buf_size = ((unsigned int)ceil(SYNCSOUND_samples_per_frame) + 5) * (b16 ? 2 : 1);
	audio.buffer = Util_malloc(audio.buf_size);
	
	audio.write_pos = 0;
	audio.sample_pos = 0.0;
	if (cassette_file != NULL)
		IMG_TAPE_SetAudio(cassette_file, samplerate, (b16 ? 16 : 8));
	cassette_write_func = &CassetteWriteAudio;
	cassette_read_func = &CassetteReadAudio;
}

void CASSETTE_UpdateSound(void *sndbuffer, int sndn)
{
	unsigned int numsamples = audio.write_pos;
	if (ESC_enable_sio_patch) {
		/* When SIO patch is active, CASSETTE_AddScanLine is not
		   called. Therefore we must update PREV_CPU_CLOCK here, so
		   that CASSETTE_AddScanLine won't fail horribly when the SIO
		   patch is turned off. */
		prev_cpu_clock = ANTIC_CPU_CLOCK;
		return;
	}
	if (!CASSETTE_play_audio || CASSETTE_audio_volume == 0)
		return;
	if (audio.n_pokeys == 2)
		sndn >>= 1;

	if (numsamples >= sndn) {
/*		Log_print("Update reserve: %i, sndn=%i, %i", numsamples - sndn, sndn, audio.buf_size);*/
		numsamples = sndn;
	}
/*	else {
		Log_print("Update underflow: %i", sndn - numsamples);
	}*/
	if (audio.bit16)
		mix((SWORD *)sndbuffer, (SWORD *)audio.buffer, numsamples, CASSETTE_audio_volume);
	else
		mix8((UBYTE *)sndbuffer, (UBYTE *)audio.buffer, numsamples, CASSETTE_audio_volume);
	if (numsamples >= sndn) {
		/* Audio buffer's size (numsamples) is computed in CASSETTE_InitAudio()
		   according to SYNCSOUND_samples_per_frame, so it is at most a few bytes
		   larger that sndn. Therefore the copying below would copy only a few
		   bytes. It's easier and simpler to do that instead of maintaining a
		   circular buffer. */
		if (audio.bit16)
			memmove(audio.buffer, audio.buffer + (sndn << 1), (audio.write_pos - sndn) << 1);
		else
			memmove(audio.buffer, audio.buffer + sndn, audio.write_pos - sndn);
		audio.write_pos -= sndn;
	} else {
		audio.write_pos = 0;
	}
}
#endif /* SYNCHRONIZED_SOUND */
#endif /* HAVE_LIBA8CAS */

static void CassetteWrite(int num_ticks)
{
/*	Log_print("caswrite: num_samples=%i", num_samples);*/
	if (cassette_writable) {
/*	Log_print("addscanline: numsamples=%i, signal=%u, tick=%i", num_samples, POKEY_serial_data_output, ANTIC_CPU_CLOCK);*/
#if HAVE_LIBA8CAS
		IMG_TAPE_Write(cassette_file, num_ticks, POKEY_serial_data_output);
#else
		IMG_TAPE_WriteAdvance(cassette_file, num_ticks);
#endif /* HAVE_LIBA8CAS */
	}
}

static void CassetteRead(int num_ticks)
{
	if (cassette_readable) {
		event_time_left -= num_ticks;
		while (event_time_left < 0.0) {
			unsigned int length;
#if HAVE_LIBA8CAS
			if (!IMG_TAPE_Read(cassette_file, &length, &current_signal)) {
#else
			if (!IMG_TAPE_Read(cassette_file, &length)) {
#endif /* HAVE_LIBA8CAS */
				/* Ignore the eventual error, assume it is the end of file */
				eof_of_tape = 1;
				UpdateFlags();
				return;
			}
			event_time_left += length;
/*			Log_print("event_time_left: %f", event_time_left);*/
		}
	}
}

void CASSETTE_AddScanLine(void)
{
	int num_ticks = (int)(ANTIC_CPU_CLOCK - prev_cpu_clock);
	prev_cpu_clock = ANTIC_CPU_CLOCK;
#if HAVE_LIBA8CAS
	ResetTurboState();
#endif /* HAVE_LIBA8CAS */

	if (CASSETTE_record)
		(*cassette_write_func)(num_ticks);
	else
		(*cassette_read_func)(num_ticks);
}

/* --- Functions for loading/saving with SIO patch --- */

int CASSETTE_AddGap(int gaptime)
{
	cassette_gapdelay += gaptime;
	if (cassette_gapdelay < 0)
		cassette_gapdelay = 0;
	return cassette_gapdelay;
}

/* Indicates that a loading leader is expected by the OS */
void CASSETTE_LeaderLoad(void)
{
	if (CASSETTE_record)
		CASSETTE_ToggleRecord();
	CASSETTE_TapeMotor(TRUE);
	cassette_gapdelay = 9600;
	/* registers for SETVBV: third system timer, ~0.1 sec */
	CPU_regA = 3;
	CPU_regX = 0;
	CPU_regY = 5;
}

/* indicates that a save leader is written by the OS */
void CASSETTE_LeaderSave(void)
{
	if (!CASSETTE_record)
		CASSETTE_ToggleRecord();
	CASSETTE_TapeMotor(TRUE);
	cassette_gapdelay = 19200;
	/* registers for SETVBV: third system timer, ~0.1 sec */
	CPU_regA = 3;
	CPU_regX = 0;
	CPU_regY = 5;
}

int CASSETTE_ReadToMemory(UWORD dest_addr, int length)
{
/*	Log_print("ReadWithPatch! length: %u, dest_addr: %u, cassette_gapdelay: %i, event_time_left: %f", length, dest_addr, cassette_gapdelay, event_time_left);*/
	CASSETTE_TapeMotor(1);
	if (!cassette_readable)
		return FALSE;

	/* Convert event_time_left to ms ( event_time_left * 1000 / 1789790 ) and subtract. */
	cassette_gapdelay -= event_time_left / 1789; /* better accuracy not needed */
	if (!IMG_TAPE_SkipToData(cassette_file, cassette_gapdelay)) {
		/* Ignore the eventual error, assume it is the end of file */
		cassette_gapdelay = 0;
		eof_of_tape = 1;
		UpdateFlags();
		return FALSE;
	}
	cassette_gapdelay = 0;

	/* Load bytes */
	switch (IMG_TAPE_ReadToMemory(cassette_file, dest_addr, length)) {
	case TRUE:
		return TRUE;
	case -1: /* Read error/EOF */
		eof_of_tape = 1;
		UpdateFlags();
		/* FALLTHROUGH */
	default: /* case FALSE */
		return FALSE;
	}
}

int CASSETTE_WriteFromMemory(UWORD src_addr, int length)
{
	int result;
/*	Log_print("WriteWithPatch! length: %u, src_addr: %u, cassette_gapdelay: %i, event_time_left: %f", length, src_addr, cassette_gapdelay, event_time_left);*/
	CASSETTE_TapeMotor(1);
	if (!cassette_writable)
		return 0;

	result = IMG_TAPE_WriteFromMemory(cassette_file, src_addr, length, cassette_gapdelay);
	cassette_gapdelay = 0;
	return result;
}


/*
vim:ts=4:sw=4:
*/
