#ifndef CASSETTE_H_
#define CASSETTE_H_

#include <stdio.h>		/* for FILE and FILENAME_MAX */

#include "config.h"
#include "atari.h"		/* for UBYTE */

#define CASSETTE_DESCRIPTION_MAX 256

extern char CASSETTE_filename[FILENAME_MAX];
extern char CASSETTE_description[CASSETTE_DESCRIPTION_MAX];
typedef enum {
	CASSETTE_STATUS_NONE,
	CASSETTE_STATUS_READ_ONLY,
	CASSETTE_STATUS_READ_WRITE
} CASSETTE_status_t;
extern CASSETTE_status_t CASSETTE_status;

/* Used in Atari800_Initialise during emulator initialisation */
int CASSETTE_Initialise(int *argc, char *argv[]);
void CASSETTE_Exit(void);
/* Config file read/write */
int CASSETTE_ReadConfig(char *string, char *ptr);
void CASSETTE_WriteConfig(FILE *fp);

/* Attaches a tape image. Also resets CASSETTE_write_protect to FALSE.
   Returns FALSE on failure, TRUE if inserting tape succeeded, and 2
   if the file format is not recognised, so the tape is inserted as a raw
   binary file. */
int CASSETTE_Insert(const char *filename);
void CASSETTE_Remove(void);
/* Creates a new file in CAS format. DESCRIPTION can be NULL.
   Returns TRUE on success, FALSE otherwise. */
int CASSETTE_CreateCAS(char const *filename, char const *description);
#if HAVE_LIBA8CAS
int CASSETTE_CreateWAV(char const *filename);
int CASSETTE_CreateRaw(char const *filename);
#endif

extern int CASSETTE_hold_start;
extern int CASSETTE_hold_start_on_reboot; /* preserve hold_start after reboot */
extern int CASSETTE_press_space;

/* Is cassette file write-protected? Don't change directly, use CASSETTE_ToggleWriteProtect(). */
extern int CASSETTE_write_protect;
/* Switches RO/RW. Fails with FALSE if the tape cannot be switched to RW. */
int CASSETTE_ToggleWriteProtect(void);

/* Is cassette record button pressed? Don't change directly, use CASSETTE_ToggleRecord(). */
extern int CASSETTE_record;
/* If tape is mounted, switches recording on/off (otherwise return FALSE).
   Recording operations would fail if the tape is read-only. In such
   situation, when switching recording on the function returns FALSE. */
int CASSETTE_ToggleRecord(void);

void CASSETTE_Seek(unsigned int position);
/* Returns status of the DATA IN line. */
int CASSETTE_IOLineStatus(void);
#if !HAVE_LIBA8CAS
/* Put a byte into the cas file.
   The block is being written at first putbyte of the subsequent block */
void CASSETTE_PutByte(int byte);
#endif
/* Set motor status: 1 - on, 0 - off */
void CASSETTE_TapeMotor(int onoff);
/* Advance the tape by a scanline. */
void CASSETTE_AddScanLine(void);
#if !HAVE_LIBA8CAS
/* Reset cassette serial transmission; call when resseting POKEY by SKCTL. */
void CASSETTE_ResetPOKEY(void);
#endif

/* Return size in blocks/seconds of the currently-mounted tape file. */
unsigned int CASSETTE_GetSize(void);
/* Return current position (block number counted from 1/second) of the mounted tape. */
unsigned int CASSETTE_GetPosition(void);
#if HAVE_LIBA8CAS
/* If false, then current file's position is given in seconds. Value is
   undefined when a tape is not inserted. */
extern int CASSETTE_position_in_blocks;

#ifdef SYNCHRONIZED_SOUND
/* Audio output functions */
extern int CASSETTE_play_audio;
extern unsigned int CASSETTE_audio_volume;
extern unsigned int CASSETTE_crosstalk_volume;
enum {
	CASSETTE_DEFAULT_AUDIO_VOLUME = 50,
	CASSETTE_MAX_AUDIO_VOLUME = 99,
	CASSETTE_DEFAULT_CROSSTALK_VOLUME = 5,
	CASSETTE_MAX_CROSSTALK_VOLUME = 99
};
/* VALUE must be between 0 (silence) and 99 (100% crosstalk). */
void CASSETTE_SetCrosstalkVolume(unsigned int value);

void CASSETTE_InitAudio(unsigned int samplerate, unsigned int n_pokeys, int b16);
void CASSETTE_UpdateSound(void *sndbuffer, int sndn);
#endif /* SYNCHRONIZED_SOUND */

/* Support for turbo systems */
enum CASSETTE_TURBO_TYPE_t {
	CASSETTE_TURBO_NONE = 0,
	CASSETTE_TURBO_AST_XC12,
	CASSETTE_TURBO_BLIZZARD,
	CASSETTE_TURBO_HARD,
	CASSETTE_TURBO_MANUAL,
	CASSETTE_TURBO_MANUAL_REV,
	/* Number of values in enumerator */
	CASSETTE_TURBO_TYPE_SIZE
};
extern enum CASSETTE_TURBO_TYPE_t CASSETTE_turbo_type;
void CASSETTE_SetTurboType(enum CASSETTE_TURBO_TYPE_t type);

extern int CASSETTE_turbo_state;
void CASSETTE_ToggleTurboState(void);
void CASSETTE_SetTurboTolerance(int value);
extern int CASSETTE_turbo_tolerance;
enum {
	CASSETTE_DEFAULT_TURBO_TOLERANCE = 3,
	CASSETTE_MAX_TURBO_TOLERANCE = 99
};

#endif /* HAVE_LIBA8CAS */

/* --- Functions used by patched SIO --- */
/* -- SIO_Handler() -- */
int CASSETTE_AddGap(int gaptime);
/* Reads a record from tape and copies its contents (max. LENGTH bytes,
   excluding the trailing checksum) to memory starting at address DEST_ADDR.
   Returns FALSE if number of bytes in record doesn't equal LENGTH, or
   checksum is incorrect, or there was a read error/end of file; otherwise
   returns TRUE. */
int CASSETTE_ReadToMemory(UWORD dest_addr, int length);
/* Reads LENGTH bytes from memory starting at SRC_ADDR and writes them as
   a record (with added checksum) to tape. Returns FALSE if there was a write
   error, TRUE otherwise. */
int CASSETTE_WriteFromMemory(UWORD src_addr, int length);
/* -- Other -- */
void CASSETTE_LeaderLoad(void);
void CASSETTE_LeaderSave(void);

#endif /* CASSETTE_H_ */
