#include "DirEntry.h"
#include "stringops.h"
#include "petscii.h"
#include "files.h"

#define CBMDOS_FILETYPE		0x0F
#define CBMDOS_PROTECTED	0x40
#define CBMDOS_CLOSED		0x80

#define ATRDOS_OPENED		0x01
#define ATRDOS_DOS2			0x02
#define ATRDOS_LOCKED		0x20
#define ATRDOS_INUSE		0x40
#define ATRDOS_DELETED		0x80

DirEntryHeader *DirEntryHeader::DeepCopy(void)
{
	DirEntryHeader *ne;
	int i;

	ne = new DirEntryHeader;
	ne->Entry = 0;
	if (Entry) {
		ne->Entry = new unsigned char[256];
		for (i=0;i<256;i++) ne->Entry[i] = Entry[i];
	}
	return (ne);
}

void DirEntryHeader::SetEntry(DirEntryHeader *e)
{
	int i;

	for (i=2;i<256;i++) Entry[i] = e->Entry[i];
}

void DirEntryHeader::Exit(void)
{
	if (Entry) delete[] Entry;
}

SP<StringStream> DirEntryHeader::GetHeaderASCII(void)
{
	SP<ByteBuffer> buf;

	if (!Entry) return 0;

	buf = new ByteBuffer(&Entry[0x90], 16);
	return (petscii2filename(buf));
}

void DirEntryHeader::SetHeaderASCII(SP<StringStream> filename)
{
	SP<ByteBuffer> pet;
	int i;

	if (filename != 0) {
		pet = filename2petscii(filename);
		for (i=0;i<16;i++) Entry[i+0x90] = pet->GetByte(i);
	}
}

SP<StringStream> DirEntryHeader::GetIdASCII(void)
{
	SP<StringStream> sid;
	SP<ByteBuffer> id;

	if (!Entry) return 0;

	id = new ByteBuffer(&Entry[0xA2], 6);
	id->SetByte(0, 5);
	sid = petscii2filename(id);
	sid->GetString()[5] = 0;

	return (sid);
}

void DirEntryHeader::SetIdASCII(SP<StringStream> filename)
{
	SP<ByteBuffer> pet;
	unsigned char c;
	int i;

	if (filename != 0) {
		pet = filename2petscii(filename);
		for (i=0;i<5;i++) {
			c = pet->GetByte(i);
			if (c == 0xA0) c = 0x20;
			Entry[i+0xA2] = c;
		}
		if (Entry[0xA4] == 0x20) Entry[0xA4] = 0xA0;
	}
}

void DirEntryHeader::SetProtection(bool protection)
{
	if (protection) Entry[2] = 0x42;
	else Entry[2] = 0x41;
}

bool DirEntryHeader::GetProtection(void)
{
	return (Entry[2] != 0x41);
}

unsigned short DirEntryHeader::GetBlockCount(void)
{
	unsigned short blocks;
	int i;

	blocks = 0;

	for (i=0x04;i<0x90;i+=4) blocks += Entry[i];
	blocks -= Entry[0x48];

	return (blocks);
}

void DirEntryHeader::SetBlockCount(unsigned short blocks)
{
	int i;

	if (blocks != GetBlockCount()) {
		for (i=0x04;i<0x48;i+=4) Entry[i] = 0;
		for (i=0x4c;i<0x90;i+=4) Entry[i] = 0;

		for (i=0x04;i<0x90;i+=4) {
			if (i != 0x48) {
				if (blocks > 255) Entry[i] = 255;
				else Entry[i] = (unsigned char)blocks;
				blocks -= Entry[i];
			}
		}
	}
}

/**********************************************************************************/

DirEntryD64 *DirEntryD64::DeepCopy(void)
{
	DirEntryD64 *ne;
	int i;

	ne = new DirEntryD64;
	ne->Entry = 0;
	if (Entry) {
		ne->Entry = new unsigned char[32];
		for (i=0;i<32;i++) ne->Entry[i] = Entry[i];
	}
	return (ne);
}

void DirEntryD64::SetEntry(DirEntryD64 *e)
{
	int i;

	for (i=2;i<32;i++) Entry[i] = e->Entry[i];
}

void DirEntryD64::Exit(void)
{
	if (Entry) delete[] Entry;
}

SP<StringStream> DirEntryD64::GetFileNameASCII(void)
{
	SP<ByteBuffer> buf;

	if (!Entry) return 0;

	buf = new ByteBuffer(&Entry[5], 16);
	return (petscii2filename(buf));
}

void DirEntryD64::SetFileNameASCII(SP<StringStream> filename)
{
	SP<ByteBuffer> pet;
	int i;

	if (filename != 0) {
		pet = filename2petscii(filename);
		for (i=0;i<16;i++) Entry[i+5] = pet->GetByte(i);
	}
}

void DirEntryD64::SetFileType(unsigned char filetype)
{
	Entry[2] = (Entry[2] & (~CBMDOS_FILETYPE)) | (filetype & CBMDOS_FILETYPE);
}

unsigned char DirEntryD64::GetFileType(void)
{
	return (Entry[2] & CBMDOS_FILETYPE);
}

void DirEntryD64::SetProtection(bool protection)
{
	if (protection) Entry[2] |= CBMDOS_PROTECTED;
	else Entry[2] &= ~CBMDOS_PROTECTED;
}

bool DirEntryD64::GetProtection(void)
{
	return ((Entry[2] & CBMDOS_PROTECTED) != 0);
}

void DirEntryD64::SetClosed(bool closed)
{
	if (closed) Entry[2] |= CBMDOS_CLOSED;
	else Entry[2] &= ~CBMDOS_CLOSED;
}

bool DirEntryD64::GetClosed(void)
{
	return ((Entry[2] & CBMDOS_CLOSED) != 0);
}

unsigned short DirEntryD64::GetBlockCount(void)
{
	unsigned short blocks;

	blocks  = Entry[0x1F];
	blocks <<= 8;
	blocks |= Entry[0x1E];

	return (blocks);
}

void DirEntryD64::SetBlockCount(unsigned short blocks)
{
	Entry[0x1E] = (unsigned char)(blocks & 0x00FF);
	blocks >>= 8;
	Entry[0x1F] = (unsigned char)(blocks);
}

/**********************************************************************************/

DirEntryATR *DirEntryATR::DeepCopy(void)
{
	DirEntryATR *ne;
	int i;

	ne = new DirEntryATR;
	ne->Entry = 0;
	if (Entry) {
		ne->Entry = new unsigned char[16];
		for (i=0;i<16;i++) ne->Entry[i] = Entry[i];
	}
	return (ne);
}

void DirEntryATR::SetEntry(DirEntryATR *e)
{
	int i;

	for (i=0;i<16;i++) Entry[i] = e->Entry[i];
}

void DirEntryATR::Exit(void)
{
	if (Entry) delete[] Entry;
}

SP<StringStream> DirEntryATR::GetFileNameASCII(void)
{
	unsigned char temp[20];
	int i;
	SP<StringStream> str;

	for (i=0;i<16;i++) temp[i] = Entry[i+5];
	temp[16] = 0;

	i = 8;
	while (i > 0) {
		if (temp[i-1] != 0x20) break;
		i--;
	}
	temp[i] = '.';
	temp[i+1] = Entry[13];
	temp[i+2] = Entry[14];
	temp[i+3] = Entry[15];
	temp[i+4] = 0;

	str = new StringStream(petscii2ascii(temp, stringlen((char *)temp)));

	return (str);
}

void DirEntryATR::SetFileNameASCII(SP<StringStream> filename)
{
	SP<ByteBuffer> pet;
	SP<StringStream> str;
	int i;
	unsigned char c;

	if (filename != 0) {
		for (i=0;i<11;i++) Entry[i+5] = 0x20;
		str = getnameonly(filename);
		pet = filename2petscii(str);
		for (i=0;i<8;i++) {
			c = pet->GetByte(i);
			if ((!c) || (c == 0xA0)) break;
			Entry[i+5] = c;
		}
		str = getextension(filename);
		if (str == 0) return;
		if (str->GetSize() <= 1) return;
		pet = filename2petscii(str);
		for (i=0;i<3;i++) {
			c = pet->GetByte(i+1);
			if ((!c) || (c == 0xA0)) break;
			Entry[i+5+8] = c;
		}
	}
}

bool DirEntryATR::IsNeverUsed(void)
{
	return (Entry[0] == 0);
}

void DirEntryATR::SetLocked(bool protection)
{
	if (protection) Entry[0] |= ATRDOS_LOCKED;
	else Entry[0] &= ~ATRDOS_LOCKED;
}

bool DirEntryATR::GetLocked(void)
{
	return ((Entry[0] & ATRDOS_LOCKED) != 0);
}

void DirEntryATR::SetOpened(bool closed)
{
	if (closed) Entry[0] |= ATRDOS_OPENED;
	else Entry[0] &= ~ATRDOS_OPENED;
}

bool DirEntryATR::GetOpened(void)
{
	return ((Entry[0] & ATRDOS_OPENED) != 0);
}

void DirEntryATR::SetDeleted(bool deleted)
{
	if (deleted) Entry[0] |= ATRDOS_DELETED;
	else Entry[0] &= ~ATRDOS_DELETED;
}

bool DirEntryATR::GetDeleted(void)
{
	return ((Entry[0] & ATRDOS_DELETED) != 0);
}

void DirEntryATR::SetInUse(bool inuse)
{
	if (inuse) Entry[0] |= ATRDOS_INUSE;
	else Entry[0] &= ~ATRDOS_INUSE;
}

bool DirEntryATR::GetInUse(void)
{
	return ((Entry[0] & ATRDOS_INUSE) != 0);
}

void DirEntryATR::SetDOS2(bool dos2)
{
	if (dos2) Entry[0] |= ATRDOS_DOS2;
	else Entry[0] &= ~ATRDOS_DOS2;
}

bool DirEntryATR::GetDOS2(void)
{
	return ((Entry[0] & ATRDOS_DOS2) != 0);
}

unsigned short DirEntryATR::GetBlockCount(void)
{
	unsigned short blocks;

	blocks  = Entry[2];
	blocks <<= 8;
	blocks |= Entry[1];

	return (blocks);
}

void DirEntryATR::SetBlockCount(unsigned short blocks)
{
	Entry[1] = (unsigned char)(blocks & 0x00FF);
	blocks >>= 8;
	Entry[2] = (unsigned char)(blocks);
}
