#include "FileZipCode.h"
#include "StringStream.h"
#include "stringops.h"
#include "files.h"
#include "debug.h"
#include <stdio.h>

extern unsigned char GetMaxSectors40(int cyl);
extern int GetTrackOffset40(int cyl);

static unsigned char starttrack[6] = { 1, 9, 17, 26, 36, 41 };

static bool ParseZipCodeData(unsigned char *data, int size, short mintrack, short maxtrack, unsigned char *buffer)
{
	short track, sector, ms, i, k, m;
	int offs;
	unsigned char crmode, zsect, fill, packbyte, packsize, packlen;
	unsigned char *block;
	unsigned char *limit;

	limit = &data[size];
	for (track=mintrack;track<= maxtrack;track++)
	{
		ms = GetMaxSectors40(track);
		for (sector=0;sector<ms;sector++)
		{
			if (data >= limit) return false;
			crmode = *data++;
			if ((crmode & 0x3F) != track) return false;

			offs = GetTrackOffset40(track);
			if (offs < 0) return false;

			if (data >= limit) return false;
			zsect = *data++;
			if (zsect >= ms) return false;

			block = &buffer[(offs+zsect) << 8];
			switch (crmode >> 6)
			{
			case 0:
				if (data >= limit) return false;
				for (i=0;i<256;i++) {
					if (data >= limit) return false;
					block[i] = *data++;
				}
				break;
			case 1:
				if (data >= limit) return false;
				fill = *data++;
				for (i=0;i<256;i++) block[i] = fill;
				break;
			case 2:
				if (data >= limit) return false;
				packsize = *data++;
				if (data >= limit) return false;
				packbyte = *data++;
				if ((data+packsize) >= limit) return false;
				i = 0;
				k = 0;
				while (i<packsize) {
					fill = data[i];
					i++;
					if (fill != packbyte) {
						block[k] = fill;
						k++;
						if (k > 256) return false;
					} else {
						packlen = data[i];
						i++;
						fill = data[i];
						i++;
						for (m=0;m<packlen;m++) {
							block[k] = fill;
							k++;
							if (k > 256) return false;
						}
					}					
				}
				data += packsize;
				if (k != 256) return false;
				break;
			case 3:
				return false;
			}
		}
	}

	return true;
}

static int ZipCodeFilenameOffset(const char *filename)
{
	int foffs, fsize;
	const char *fname;

	if (filename == 0) return (-1);

	foffs = FindFilenameOffset(filename);
	fsize = stringlen(filename)-foffs;
	fname = &filename[foffs];

	// filename has to start with "1!" but not "1!!" (different ZipCode format)
	if (fsize >= 3) {
		if ((fname[0] == '1') && (fname[1] == '!') && (fname[2] != '!')) return (foffs);
	}

	// filename can end with "1.z64"
	fsize -= 5;
	if (fsize >= 0) {
		fname += fsize;
		if ((stringcmp(fname, "1.z64")) || (stringcmp(fname, "1.Z64"))) return (fsize+foffs);
	}

	return (-1);
}

extern bool TestZipCodeFileName(const char *filename)
{
	return (ZipCodeFilenameOffset(filename) >= 0);
}

int ParseZipCode(const char *filename, unsigned char *buffer)
{
	char path[FILENAME_MAX];
	char *fname;
	int i, k, fsize, foffs;
	bool ok;

	FILE *f;
	unsigned char *fdata;
	unsigned char *data;
	int size;

	if ((filename == 0) || (buffer == 0)) return (0);

	fsize = stringlen(filename);
	for (i=0;i<=fsize;i++) path[i] = filename[i];

	foffs = ZipCodeFilenameOffset(filename);
	if (foffs < 0) return (0);
	fname = &path[foffs];

	fdata = new unsigned char[65536];
	for (i=1;i<6;i++) {
		for (k=0;k<65536;k++) fdata[k] = 0;

		ok = false;
		size = 0;
		*fname = (char)(i + '0');
		f = fopen(path, "rb");
		if (f) {
			size = fread((void *)fdata, 1, 65536, f);
			fclose(f);
		}

		if ((size < 256) || (size > 64000)) {
			ok |= (i > 4);
			break;
		}

		if (i == 1) {
			data = &fdata[4];
			if ((fdata[0] != 0xFE) || (fdata[1] != 0x03)) break;
		} else {
			data = &fdata[2];
			if ((fdata[0] != 0x00) || (fdata[1] != 0x04)) break;
		}

#ifdef _DEBUG
		sDPrintF("ZipCode file: %s\n", path);
#endif

		ok = ParseZipCodeData(data, size, starttrack[i-1], starttrack[i]-1, buffer);
		if (!ok) break;
	}
	delete[] fdata;

	return (ok);
}

class StringStream *StripFilename(const char *filename)
{
	char path[FILENAME_MAX];
	int i, fsize, foffs;
	char c;

	if (ZipCodeFilenameOffset(filename) < 0) return (0);

	fsize = stringlen(filename);
	for (i=0;i<=fsize;i++) path[i] = filename[i];

	foffs = FindFilenameOffset(filename);

	i = fsize-1;
	while (i >= foffs) {
		if (path[i] == '.') {
			path[i] = 0;
			break;
		}
		i--;
	}

	if ((path[foffs] == '1') && (path[foffs+1] == '!')) {
		i = foffs;
		do {
			c = path[i+2];
			path[i] = c;
			i++;
		} while (c != 0);
	}

	return (new StringStream(path));
}
