#include "GlobalConfig.h"
#include "stringops.h"
#include "files.h"
#include "FileImage.h"
#include "d64.h"
#include "atr.h"
#include "ArgStream.h"
#include "StringStream.h"

#include <stdio.h>
#include "debug.h"

#define ARC64CMD_NULL		0
#define ARC64CMD_OPEN		1
#define ARC64CMD_STORE		2
#define ARC64CMD_SETIP		3
#define ARC64CMD_SETPORT	4
#define ARC64CMD_SETTIMEOUT	5
#define ARC64CMD_SETRETRIES	6
#define ARC64CMD_ADD		7
#define ARC64CMD_NEWD64		8
#define ARC64CMD_NEWATR		9
#define ARC64CMD_SETX64		21
#define ARC64CMD_SETX128	22
#define ARC64CMD_SETXVIC	23
#define ARC64CMD_SETXPLUS4	24
#define ARC64CMD_SETYAPE	25
#define ARC64CMD_SETA800W	26
#define ARC64CMD_SETD64		27
#define ARC64CMD_SETATR		28
#define ARC64CMD_SETUSET18		29
#define ARC64CMD_SETDIRT18		30
#define ARC64CMD_SETINTERLEAVE	31
#define ARC64CMD_SETDIRINTERLV	32
#define ARC64CMD_SETSTRATEGY	33

GlobalConfig *Config;

ArgToken CmdTokens[]=
{
	{ "-o",			ARC64CMD_OPEN },
	{ "-s",			ARC64CMD_STORE },
	{ "-n",			ARC64CMD_SETIP },
	{ "-p",			ARC64CMD_SETPORT },
	{ "-a",			ARC64CMD_ADD },
	{ "-d64",		ARC64CMD_NEWD64 },
	{ "-atr",		ARC64CMD_NEWATR },
	{ "-uset18",	ARC64CMD_SETUSET18 },
	{ "-dirt18",	ARC64CMD_SETDIRT18 },
	{ "-interleave",	ARC64CMD_SETINTERLEAVE },
	{ "-dirinterleave",	ARC64CMD_SETDIRINTERLV },
	{ "-strategy",		ARC64CMD_SETSTRATEGY },
	{ 0, ARC64CMD_NULL }
};


ArgToken SettingTokens[]=
{
	{ "x64path",	ARC64CMD_SETX64 },
	{ "x128path",	ARC64CMD_SETX128 },
	{ "xvicpath",	ARC64CMD_SETXVIC },
	{ "xplus4path",	ARC64CMD_SETXPLUS4 },
	{ "yapepath",	ARC64CMD_SETYAPE },
	{ "a800wpath",	ARC64CMD_SETA800W },
	{ "d64",		ARC64CMD_SETD64 },
	{ "atr",		ARC64CMD_SETATR },
	{ "ip",			ARC64CMD_SETIP },
	{ "port",		ARC64CMD_SETPORT },
	{ "timeout",	ARC64CMD_SETTIMEOUT },
	{ "retries",	ARC64CMD_SETRETRIES },
	{ 0, ARC64CMD_NULL }
};

/******************************************************************************/

GlobalConfig::GlobalConfig(SP<class StringStream> filepath)
{
	Path       = filepath;

	X64Path    = new StringStream("");
	X128Path   = new StringStream("");
	XVICPath   = new StringStream("");
	XPLUS4Path = new StringStream("");
	YAPEPath   = new StringStream("");
	A800WPath  = new StringStream("");

	DefaultImage = CONFIG_DEFAULTIMAGE_D64;

	IP         = new StringStream("192.168.0.64");
	Port       = 6462;
	UseTrack18 = 0;
	DirTrack18Only = 1;
	Interleave = 10;
	DirInterleave = 3;
	Strategy   = D64STRATEGY_CLOSEST;

	Changed    = 0;
}

GlobalConfig::~GlobalConfig(void)
{
	Path       = 0;
	IP         = 0;
	X64Path    = 0;
	X128Path   = 0;
	XVICPath   = 0;
	XPLUS4Path = 0;
	YAPEPath   = 0;
	A800WPath  = 0;
}

SP<class StringStream> GlobalConfig::GetIP(void)
{
	return (IP);
}

int GlobalConfig::GetNetPort(void)
{
	return (Port);
}

SP<StringStream> GlobalConfig::GetX64Path(void)
{
	return (X64Path);
}

SP<StringStream> GlobalConfig::GetX128Path(void)
{
	return (X128Path);
}

SP<StringStream> GlobalConfig::GetXVICPath(void)
{
	return (XVICPath);
}

SP<StringStream> GlobalConfig::GetXPLUS4Path(void)
{
	return (XPLUS4Path);
}

SP<StringStream> GlobalConfig::GetYAPEPath(void)
{
	return (YAPEPath);
}

SP<StringStream> GlobalConfig::GetA800WPath(void)
{
	return (A800WPath);
}

static int IsIP(const char *str)
{
	int i, k, cnt, x;
	char c;

	i = 0;
	for (k=0;k<4;k++) {
		if (k > 0) {
			if (str[i] != '.') return (0);
			i++;
		}

		x = 0;
		cnt = 0;
		while(str[i]) {
			c = str[i];
			if ((c < '0') || (c > '9')) break;
			x = ((x * 10) + (c - '0'));
			cnt++;
			i++;
		}
		if ((cnt == 0) || (cnt > 3)) return (0);
		if (x >= 256) return (0);
	}

	return (1);
}

void GlobalConfig::SetIP(SP<StringStream> str)
{
	if (str == 0) return;

	if (!IsIP(str->GetString())) return;

	if (!IP->IsEqual(str)) {
		IP = str;
		Changed = 1;
	}
}

void GlobalConfig::SetNetPort(int p)
{
	if (Port != p) {
		Port = p;
		Changed = 1;
	}
}

void GlobalConfig::SetDefaultImage(int d)
{
	if (DefaultImage != d) {
		DefaultImage = d;
		Changed = 1;
	}
}

void GlobalConfig::SetX64Path(SP<StringStream> str)
{
	if (!X64Path->IsEqual(str)) {
		X64Path = str;
		Changed = 1;
	}
}

void GlobalConfig::SetX128Path(SP<StringStream> str)
{
	if (!X128Path->IsEqual(str)) {
		X128Path = str;
		Changed = 1;
	}
}

void GlobalConfig::SetXVICPath(SP<StringStream> str)
{
	if (!XVICPath->IsEqual(str)) {
		XVICPath = str;
		Changed = 1;
	}
}

void GlobalConfig::SetXPLUS4Path(SP<StringStream> str)
{
	if (!XPLUS4Path->IsEqual(str)) {
		XPLUS4Path = str;
		Changed = 1;
	}
}

void GlobalConfig::SetYAPEPath(SP<StringStream> str)
{
	if (!YAPEPath->IsEqual(str)) {
		YAPEPath = str;
		Changed = 1;
	}
}

void GlobalConfig::SetA800WPath(SP<StringStream> str)
{
	if (!A800WPath->IsEqual(str)) {
		A800WPath = str;
		Changed = 1;
	}
}

/*********************************************************************************************/

#include "FileImage.h"
#include "FileContainer.h"

class FileImage *ConvertToImage(const char *path, class FileImage *img)
{
	class FileContainer *fc;
	FILE *f;
	fpos_t s;
	unsigned char *data;
	int size;
	int type, ok;
	SP<StringStream> spath;

	if (path == 0) return (0);

	spath = new StringStream(path);

	type = getfiletype(path);
	ok = 0;

	data = 0;
	size = 0;

	f = fopen(path, "rb");
	if (f) {
		fseek(f, 0, SEEK_END);
		fgetpos(f, &s);
		fseek(f, 0, SEEK_SET);
		size = (int)s;
		if (size <= 2*1024*1024) {
			data = new unsigned char[size];
			if (size > 0) {
				size = fread((void *)data, 1, size, f);
			}
		}
		fclose(f);
	}

	if (data != 0) {
		img = CreateImage(spath, type, data, size);

		if (img == 0) {
			fc = ParseFile(spath, type, data, size);
			if (fc != 0) {
				img = new FileD64(IMAGETYPE_D64_170K);
				ok = fc->AddTo(img);
				delete fc;
			}
		}

		delete[] data;
	}

	return (img);
}

static class FileImage *LoadImageFromAnyFile(class FileImage *img, const char *path)
{
	int type;

	if (path == 0) return (img);

	type = getfiletype(path);
	switch (type)
	{
	case FILETYPE_D64:
	case FILETYPE_ZIP:
	case FILETYPE_ATR:
	case FILETYPE_XFD:
	case FILETYPE_DCM:
		img = CreateImage(path);
		break;

	case FILETYPE_COM:
	case FILETYPE_EXE:
		if (img == 0) img = new FileATR(IMAGETYPE_ATR_130K);
		img = ConvertToImage(path, img);
		break;

	default:
		if (img == 0) img = new FileD64(IMAGETYPE_D64_170K);
		img = ConvertToImage(path, img);
		break;
	}

	return (img);
}

static SP<FileImage> ParseString(ArgToken *tokens, GlobalConfig *config, SP<FileImage> fd64, char *cmdstring)
{
	ArgStreamReader *cmdline;
	SP<StringStream> argword;
	int cmd, argcount, token, type;

	cmd = ARC64CMD_NULL;
	argcount = 0;

	cmdline = new ArgStreamReader(cmdstring);
	while (!cmdline->AtEnd()) {
		argword = new StringStream(cmdline->GetWord());
		token = cmdline->GetToken(tokens, argword->GetString());
		if ((argcount == 0) && (token == ARC64CMD_NULL)) {
			fd64 = LoadImageFromAnyFile(fd64.GetNativePointer(), argword->GetString());
		}

#ifdef _DEBUG
		sDPrintF("%s %i\n", argword->GetString(), token);
#endif

		switch (token)
		{
		case (ARC64CMD_NEWD64):
			fd64 = new FileD64(IMAGETYPE_D64_170K);
			break;
		case (ARC64CMD_NEWATR):
			fd64 = new FileATR(IMAGETYPE_ATR_130K);
			break;
		case (ARC64CMD_OPEN):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				fd64 = CreateImage(argword->GetString());
			}
			break;
		case (ARC64CMD_STORE):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				if (fd64 != 0) {
					fd64->SetPath(argword);
					fd64->Save(fd64->GetPath()->GetString());
				}
			}
			break;
		case (ARC64CMD_ADD):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				if (fd64 != 0) {
//					fd64->Add(argword);
				}
			}
			break;

		case (ARC64CMD_SETD64):
			config->SetDefaultImage(CONFIG_DEFAULTIMAGE_D64);
			break;
		case (ARC64CMD_SETATR):
			config->SetDefaultImage(CONFIG_DEFAULTIMAGE_ATR);
			break;
		case (ARC64CMD_SETUSET18):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->UseTrack18 = (argword->ParseInteger() != 0);
			}
			break;
		case (ARC64CMD_SETDIRT18):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->DirTrack18Only = (argword->ParseInteger() != 0);
			}
			break;
		case (ARC64CMD_SETINTERLEAVE):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->Interleave = argword->ParseInteger();
			}
			break;
		case (ARC64CMD_SETDIRINTERLV):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->DirInterleave = argword->ParseInteger();
			}
			break;
		case (ARC64CMD_SETSTRATEGY):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->Strategy = argword->ParseInteger();
			}
			break;

		case (ARC64CMD_SETIP):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetIP(argword);
			}
			break;
		case (ARC64CMD_SETPORT):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetNetPort(argword->ParseInteger());
			}
			break;
		case (ARC64CMD_SETX64):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetX64Path(argword);
			}
			break;
		case (ARC64CMD_SETX128):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetX128Path(argword);
			}
			break;
		case (ARC64CMD_SETXVIC):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetXVICPath(argword);
			}
			break;
		case (ARC64CMD_SETXPLUS4):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetXPLUS4Path(argword);
			}
			break;
		case (ARC64CMD_SETYAPE):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetYAPEPath(argword);
			}
			break;
		case (ARC64CMD_SETA800W):
			argword = new StringStream(cmdline->GetWord());
			if (argword != 0) {
				config->SetA800WPath(argword);
			}
			break;
		}
		argcount++;
	}
	delete cmdline;

	return (fd64);
}

SP<FileImage> GlobalConfig::ParseCmds(SP<FileImage> fd64, char *cmdline)
{
	return ParseString(CmdTokens, this, fd64, cmdline);
}

SP<FileImage> GlobalConfig::Load(void)
{
	FILE *f;
	char *buffer;
	long size;
	SP<FileImage> fd64;

	fd64 = 0;
	f = fopen(Path->GetString(), "rb");
	if (f) {
		fseek(f, 0, SEEK_END);
		size = ftell(f);
		fseek(f, 0, SEEK_SET);

		buffer = new char[size+1];
		fread((void *)buffer, 1, size, f);
		fclose(f);
		buffer[size] = 0;

		fd64 = ParseString(SettingTokens, this, fd64, buffer);
		delete[] buffer;
	}

	if (fd64 == 0) {
		if (DefaultImage != CONFIG_DEFAULTIMAGE_ATR) {
			fd64 = new FileD64(IMAGETYPE_D64_170K);
		} else {
			fd64 = new FileATR(IMAGETYPE_ATR_130K);
		}
	}

	Changed = 0;

	return (fd64);
}

static char *FindTokenString(ArgToken *tlist, int token)
{
	int i;

	i = 0;
	while (tlist[i].Token != ARC64CMD_NULL) {
		if (tlist[i].Token == token) return (tlist[i].String);
		i++;
	}

	return (0);
}

static void SaveSetting(FILE *f, int token, char *str, int quotes)
{
	char *tkn;
	int len;

	tkn = FindTokenString(SettingTokens, token);
	if (tkn != 0) {
		fwrite((void *)tkn, 1, stringlen(tkn), f);
		if (str != 0) {
			fwrite((void *)" ", 1, 1, f);
			if (quotes) fwrite((void *)"\"", 1, 1, f);
			len = stringlen(str);
			if (len > 0) fwrite((void *)str, 1, len, f);
			if (quotes) fwrite((void *)"\"", 1, 1, f);
		}
		fwrite((void *)"\n", 1, 1, f);
	}
}

void GlobalConfig::Save(void)
{
	FILE *f;
	SP<StringStream> str;

	if (!Changed) return;
	if (Path == 0) return;

	f = fopen(Path->GetString(), "wb");
	if (f) {
		SaveSetting(f, ARC64CMD_SETX64, X64Path->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETX128, X128Path->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETXVIC, XVICPath->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETXPLUS4, XPLUS4Path->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETYAPE, YAPEPath->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETA800W, A800WPath->GetString(), 1);
		SaveSetting(f, ARC64CMD_SETIP, IP->GetString(), 1);
		str = new StringStream(Port);
		SaveSetting(f, ARC64CMD_SETPORT, str->GetString(), 0);
		if (DefaultImage == CONFIG_DEFAULTIMAGE_ATR) SaveSetting(f, ARC64CMD_SETATR, 0, 0);
		if (DefaultImage == CONFIG_DEFAULTIMAGE_D64) SaveSetting(f, ARC64CMD_SETD64, 0, 0);
		fclose(f);

		Changed = 0;
	}
}
