#include "StringStream.h"
#include "ByteBuffer.h"
#include "debug.h"

#ifdef _DEBUG
static int strcounter = 0;
#endif

static int stringlength(const char *string)
{
	int i;

	i = 0;
	if (string) {
		while (string[i]) i++;
	}

	return (i);
}

static void stringcopy(char *dst, const char *src)
{
	int i;

	i = 0;

	if (src && dst) {
		while (src[i]) {
			dst[i] = src[i];
			i++;
		}
		dst[i] = 0;
	}
}

static char *stringclone(const char *string)
{
	char *clone;

	if (!string) return 0;

	clone = new char[stringlength(string)+1];
	stringcopy(clone, string);

	return clone;
}

void StringStream::Construct(const char *str)
{
	if (str == 0) {
		String = 0;
		Size = 0;
	} else {
		String = stringclone(str);
		Size = stringlength(String);
	}

#ifdef _DEBUG
	strcounter++;
	if (String != 0) sDPrintF("StringStream %i \"%s\"\n", strcounter, String);
	else sDPrintF("StringStream %i (NULL)\n", strcounter);
#endif
}

StringStream::~StringStream(void)
{
#ifdef _DEBUG
	strcounter--;
	if (String != 0) sDPrintF("StringStream %i \"%s\"\n", strcounter, String);
	else sDPrintF("StringStream %i (NULL)\n", strcounter);
#endif

	if (String != 0) delete[] String;
}

StringStream::StringStream(void)
{
	Construct(0);
}

StringStream::StringStream(const char *string)
{
	Construct(string);
}

StringStream::StringStream(SP<StringStream> str)
{
	if (str == 0) Construct(0);
	else Construct(str->GetString());
}

StringStream::StringStream(SP<ByteBuffer> bb)
{
	char *tmp;
	int size;

	size = bb->GetSize();
	tmp = new char[size+1];

	bb->Put((unsigned char *)tmp);
	tmp[size] = 0;

	Construct(tmp);

	delete[] tmp;
}

StringStream::StringStream(int value)
{
	int i, k;
	char tmp1[16];
	char tmp2[16];

	if (value == 0)
	{
		tmp2[0] = '0';
		tmp2[1] = 0;

	} else {

		i = 0;
		k = 0;

		if (value < 0) {
			value = -value;
			tmp2[0] = '-';
			k = 1;
		}

		while (value > 0) {
			tmp1[i] = value % 10;
			value /= 10;
			i++;
		}

		while (i > 0) {
			i--;
			tmp2[k] = tmp1[i] + '0';
			k++;
		}
		tmp2[k] = 0;
	}

	Construct(tmp2);
}

char *StringStream::GetString(void)
{
	return (String);
}

int StringStream::GetSize(void)
{
	return (Size);
}

char StringStream::GetChar(int index)
{
	if ((index < 0) || (index >= Size) || (String == 0)) return 0;

	return (String[index]);
}

SP<StringStream> StringStream::Append(const char *str2)
{
	int i, size2;
	char *nstr;
	SP<StringStream> ns;

	size2 = stringlength(str2);
	if ((str2 == 0) || (size2 <= 0)) return (new StringStream(String));

	if ((Size <= 0) || (String == 0)) {
		return (new StringStream(str2));
	} else {
		nstr = new char[Size+size2+1];
		for (i=0;i<Size ;i++) nstr[i] = String[i];
		for (i=0;i<size2;i++) nstr[i+Size] = str2[i];
		nstr[Size+size2] = 0;
		ns = new StringStream(nstr);
		delete[] nstr;
		return (ns);
	}
}

SP<StringStream> StringStream::Append(SP<StringStream> string)
{
	if (string == 0) return (0);
	return Append(string->GetString());
}

int StringStream::ParseInteger(void)
{
	int i, value;
	char c;

	i = 0;
	value = 0;

	while (i < Size) {
		c = String[i];
		if ((c < '0') || (c > '9')) break;

		value *= 10;
		value += (c - '0');

		i++;
	}

	return (value);
}

SP<StringStream> StringStream::Cut(int index)
{
	if ((String == 0) || (index > Size)) return (new StringStream());

	return (new StringStream(&String[index]));
}

SP<StringStream> StringStream::GetExtension(void)
{
	int i;
	char c;

	if ((String != 0) && (Size > 0)) {

		for (i=(Size-1);i>=0;i--) {
			c = String[i];
			if ((c == ':') || (c == '/') || (c == '\\')) break;
			if (c == '.') return (Cut(i+1));
		}
	}

	return (new StringStream());
}

static int TestEqual(const char *str1, const char *str2)
{
	int i;

	if (str1 == str2) return (1);
	if ((str1 == 0) || (str2 == 0)) return (0);

	i = 0;
	while (1) {
		if (str1[i] != str2[i]) return (0);
		if (str1[i] == 0) return (1);
		i++;
	}
}

int StringStream::IsEqual(SP<StringStream> str)
{
	return (TestEqual(GetString(), str->GetString()));
}
