#include "atr.h"
#include "stringops.h"
#include "files.h"
#include "cbmlist.h"
#include "petscii.h"
#include "c64file.h"
#include "files.h"
#include <stdio.h>
#include <stdlib.h>
#include "debug.h"
#include "DirEntry.h"
#include "StringStream.h"

/*

Different sector offsets: DOS starts counting with sector 1, drive with sector 0.
The code refers to DOS sectors.

DOS format:

Sector 360 (0xB390): BAM (VTOC) (starting at offset 0xB39A), bit 7...0 = sectors 0...7 etc
Sector 361-368 (0xB410): directory

*/

FileATR::FileATR()
{
	SectorsPerTrack = 0;
	SectorSize = 0;
	map = 0;

	DebugPrint("FileATR(void);");
}

FileATR::FileATR(int type)
{
	Init(type);

	DebugPrint("FileATR(int type);");
}

FileATR::~FileATR()
{
	DebugPrint("~FileATR(void);");

	if (map != 0) delete[] map;
}

bool FileATR::HasHeader(void)
{
	return false;
}

int FileATR::GetImageSystem(void)
{
	return (IMAGESYSTEM_ATR);
}

void FileATR::InitMap()
{
	if (map) {
		delete[] map;
		map = 0;
	}

	if (TotalSectors > 0) {
		map = new unsigned char[TotalSectors];
		ClearMap();
	}
}

void FileATR::ClearMap()
{
	int i;

	if (map) {
		for (i=0;i<TotalSectors;i++) map[i] = 0;
	}
}

bool FileATR::AlreadyVisited(unsigned short sector)
{
	if (!IsValidBlock(sector)) return false;
	return (map[sector] != 0);
}

void FileATR::SetVisitedFlag(unsigned short sector)
{
	if (IsValidBlock(sector)) {
		map[sector] = 1;
	}
}


void FileATR::Init(int type)
{
	int i;
	unsigned char *block;
	unsigned short useable;

	Type = type;
	Flags = IMAGEFLAG_NEW;
	Tracks = 40;
	SectorsPerTrack = 18;
	SectorSize = 128;
	useable = 707;
	map = 0;

	switch (type)
	{
	default:
		Image = 0;
		Size = 0;
		return;

	case (IMAGETYPE_ATR_130K):
		SectorsPerTrack = 26;
		useable = 1010;
		break;

	case (IMAGETYPE_ATR_90K):
		break;

	case (IMAGETYPE_ATR_180K):
		SectorSize = 256;
		break;
	}

	TotalSectors = Tracks*SectorsPerTrack;
	Size = TotalSectors*SectorSize;
	if (SectorSize == 256) Size -= (3*128);		// first 3 sectors always have a size of 128 bytes
	Image = new unsigned char[Size];

	for (i=0;i<Size;i++) Image[i] = 0;
	block = GetBlock(360);
	block[0] = 2;
	block[1] = useable & 0x00FF;
	block[2] = useable >> 8;

	InitMap();
	Validate();
}

void FileATR::Init(SP<StringStream> path, const unsigned char *data, int size)
{
	int i;

	SetPath(path);
	Flags &= ~IMAGEFLAG_NEW;

	if (size > Size) size = Size;
	for (i=0;i<size;i++) Image[i] = data[i];
	for (i=size;i<Size;i++) Image[i] = 0;

	InitMap();
//	TestDirectory();
}

class FileImage *FileATR::Clone(void)
{
	FileATR *img;

	img = new FileATR(Type);
	img->Init(Path, Image, Size);

	return (img);
}

static int GetWord(const unsigned char *data)
{
	return (data[0] | (data[1] << 8));
}

FileATR *CreateATR(SP<StringStream> path, int type, const unsigned char *data, int size)
{
	FileATR *img;
	int bcnt, bsize, isize, itype;

	img = 0;
	itype = IMAGETYPE_NULL;

	if (type == FILETYPE_ATR) {

		// ATR header = 16 bytes
		if (size < 16) return (0);
		size -= 16;

		// ATR ID
		if (GetWord(&data[0]) != 0x0296) return (0);

		bcnt = GetWord(&data[2]);
		bsize = GetWord(&data[4]);

		// 90k
		isize = 40*18*128;
		if (size == isize) {
			if ((bcnt == (isize>>4)) && (bsize == 128)) itype = IMAGETYPE_ATR_90K;
		}

		// 130k
		isize = 40*26*128;
		if (size == isize) {
			if ((bcnt == (isize>>4)) && (bsize == 128)) itype = IMAGETYPE_ATR_130K;
		}

		// 180k
		isize = ((40*18*256)-(3*128));
		if (size == isize) {
			if ((bcnt == (isize>>4)) && (bsize == 256)) itype = IMAGETYPE_ATR_180K;
		}

		data += 16;
	}

	if (type == FILETYPE_XFD) {
		// 90k
		if (size == (40*18*128)) itype = IMAGETYPE_ATR_90K;

		// 130k
		if (size == (40*26*128)) itype = IMAGETYPE_ATR_130K;

		// 180k
		if (size == ((40*18*256)-(3*128))) itype = IMAGETYPE_ATR_180K;
	}

	if (itype != IMAGETYPE_NULL) {
		img = new FileATR(itype);
		img->Init(path, data, size);
	}

	return (img);
}

/***************************************************************************************************/

/* helper: translate sector number to BAM byte mask */

unsigned char GetMaskBAM(short sector)
{
	if (sector <= 0) return (0);
	return (1 << (((sector) & 7) ^ 7));
}

/* erase BAM and free all blocks */

void FileATR::ClearBAM()
{
	unsigned char *bam;
	short i;

	bam = GetBlock(360);

	switch (GetType())
	{
	default:
		break;
	case (IMAGETYPE_ATR_90K):
		bam[0x0A] = 0x7F;
		for (i=1;i<90;i++) bam[i+10] = 0xFF;
		break;
	case (IMAGETYPE_ATR_130K):
		bam[0x0A] = 0x7F;
		for (i=1;i<90;i++) bam[i+10] = 0xFF;
		bam = GetBlock(1024);
		for (i=0;i<0x7A;i++) bam[i] = 0xFF;
		bam[0x7A] = 0x2F;	// 303 extra blocks on ED
		bam[0x7B] = 0x01;
		bam[0x54] = 0x7F;
		break;
	}
}

unsigned char FileATR::ReadBAMByte(short sector)
{
	unsigned char bambyte;

	bambyte = 0;
	switch (GetType())
	{
	default:
		break;
	case (IMAGETYPE_ATR_90K):
		bambyte = GetBlock(360)[(sector >> 3) + 10];
		break;
	case (IMAGETYPE_ATR_130K):
		if (sector < 720) bambyte = GetBlock(360)[(sector >> 3) + 10];
		else
			bambyte = GetBlock(1024)[(sector >> 3) - 6];
		break;
	}
	return (bambyte);
}

void FileATR::WriteBAMByte(short sector, unsigned char bambyte)
{
	switch (GetType())
	{
	default:
		break;
	case (IMAGETYPE_ATR_90K):
		GetBlock(360)[(sector >> 3) + 10] = bambyte;
		break;
	case (IMAGETYPE_ATR_130K):
		if (sector < 720) {
			GetBlock(360)[(sector >> 3) + 10] = bambyte;
			if (sector >= 48) GetBlock(1024)[(sector >> 3) - 6] = bambyte;
		} else {
			GetBlock(1024)[(sector >> 3) - 6] = bambyte;
		}
		break;
	}
}

/* test if a block is allocated. TRUE if block is free */

bool FileATR::IsFreeBlock(short sector)
{
	if (!IsValidBlock(sector)) return false;

	return ((ReadBAMByte(sector) & GetMaskBAM(sector)) != 0);
}

/* allocate a block */

void FileATR::AllocBlock(short sector)
{
	if (!IsValidBlock(sector)) return;

	WriteBAMByte(sector, (ReadBAMByte(sector) & (~GetMaskBAM(sector))));
}

/* free a block */

void FileATR::FreeBlock(short sector)
{
	if (!IsValidBlock(sector)) return;

	WriteBAMByte(sector, (ReadBAMByte(sector) | GetMaskBAM(sector)));
}

/* validate a file */

void FileATR::ValidateLinks(short sector, short count)
{
	if (!IsValidBlock(sector)) return;
	if (!IsFreeBlock(sector)) return;
	AllocBlock(sector);
	ValidateLinks(GetBlockLink(sector), count+1);
}

void FileATR::ValidateBlockCount(void)
{
	unsigned short free;
	short i;
	unsigned char *block;

	free = 0;
	switch (GetType())
	{
	default:
		break;
	case (IMAGETYPE_ATR_90K):
		for (i=0;i<720;i++) {
			if (IsFreeBlock(i)) free++;
		}
		block = GetBlock(360);
		block[3] = (unsigned char)(free & 0x00FF);
		block[4] = (unsigned char)(free >> 8);
		break;
	case (IMAGETYPE_ATR_130K):
		for (i=0;i<720;i++) {
			if (IsFreeBlock(i)) free++;
		}
		block = GetBlock(360);
		block[3] = (unsigned char)(free & 0x00FF);
		block[4] = (unsigned char)(free >> 8);

		free = 0;
		for (i=720;i<1024;i++) {
			if (IsFreeBlock(i)) free++;
		}
		block = GetBlock(1024);
		block[0x7A] = (unsigned char)(free & 0x00FF);
		block[0x7B] = (unsigned char)(free >> 8);
		break;
	}
}

/* validate all files and directory */

void FileATR::Validate()
{
	unsigned short sector, i;
	unsigned char *direntry;

	ClearBAM();
	for (i=0;i<4;i++) AllocBlock(i);
	for (i=360;i<(360+9);i++) AllocBlock(i);

	for (i=1;i<=64;i++) {
		direntry = GetDirEntry(i);
		if (!direntry) break;
		sector  = direntry[4];
		sector <<= 8;
		sector |= direntry[3];
		ValidateLinks(sector, 0);
	}
	ValidateBlockCount();
}

/***************************************************************************************************/

/* Returns a pointer to a free dir entry or NULL */

unsigned char *FileATR::GetEmptyDirEntry()
{
	unsigned long i;
	unsigned char *block;

	block = GetBlock(361);

	i = 0;
	while (i<64)
	{
		if ((*block) & 0x80) return block;
		if ((*block) == 0x00) return block;
		i++;
		block += 16;
	}
	return 0;	/* end of dir reached */
}

/* TRUE if block exists or FALSE for invalid track/sector */

bool FileATR::IsValidBlock(short sector)
{
	if ((sector <= 0) || (sector >= TotalSectors)) return false;
	return true;
}

/* returns pointer to block in D64 or NULL for illegal track/sector */

unsigned char *FileATR::GetBlock(short sector)
{
	if (!IsValidBlock(sector)) return 0;
	sector--;
	if (SectorSize != 256) return &Image[sector*SectorSize];
	if (sector < 3) return &Image[sector<<7];
	else return &Image[(sector<<8)-384];
}

short FileATR::GetBlockLink(short sector)
{
	unsigned char *block;
	short offset;

	block = GetBlock(sector);
	if (!block) return (0);
	offset = block[SectorSize-3] & 0x03;
	offset <<= 8;
	offset |= (short)block[SectorSize-2];
	return (offset);
}

static char tempbuffer[_MAX_PATH+1024];

/***************************************************************************************************/

void FileATR::TestDirectory()
{
	int i, o;
	unsigned char *block;
	bool ok;

	Flags &= ~(IMAGEFLAG_CORRUPT_DIRLINK | IMAGEFLAG_LOOPED_DIR | IMAGEFLAG_INVALID_BAM);

	ClearMap();

	block = Image;
	Image = new unsigned char[Size];
	for (i=0;i<Size;i++) {
		Image[i] = block[i];
	}

	Validate();

	o = 359 * SectorSize;
	ok = true;
	for (i=(o+0x01);i<(o+0x05);i++) {
		if (block[i] != Image[i]) ok = false;
	}
	for (i=(o+0x0A);i<(o+0x64);i++) {
		if (block[i] != Image[i]) ok = false;
	}
	if (GetType() == IMAGETYPE_ATR_130K) {
		o = 1023 * SectorSize;
		for (i=(o+0x00);i<(o+0x7C);i++) {
			if (block[i] != Image[i]) ok = false;
		}
	}
	delete[] Image;
	Image = block;

	if (!ok) Flags |= IMAGEFLAG_INVALID_BAM;
}

CbmList *FileATR::GetDirectory()
{
	unsigned long i, k, m, bfree;
	CbmList *nl;
	unsigned char *block;
	unsigned char *cblock;
	unsigned char line[32];
	DirEntryATR *direntry;

	ClearMap();

	block = GetBlock(361);
	if (!block) return 0;
	nl = new CbmList;
	nl->Init(CHARSET_ATARI);

	// generate fake header
	for (k=0;k<32;k++) line[k] = 0x20;
	k = 5+printstring((char *)&line[5], "IMAGE");
	line[k] = ' ';
	for (k=0;k<5;k++) line[k+5] |= 0x80;
	atascii2screen(line, 32);
	nl->Add(line);

	direntry = new DirEntryATR;
	i = 0;
	m = 0;
	while (m < 64) {
		if (!GetDirEntry(m+1)) break;
		direntry->Entry = GetDirEntry(m+1);

		for (k=0;k<32;k++) line[k] = 0x20;
		if (direntry->GetLocked()) line[0] = '*';
		if (!direntry->GetInUse()) {
			line[1] = '<';
			line[13] = '>';
		}
		for (k=0;k<11;k++) line[k+2] = block[i+5+k];
		bfree = block[i+2];
		bfree <<= 8;
		bfree |= block[i+1];
		if (bfree > 999) bfree = 999;
		for (k=0;k<3;k++) {
			line[16-k] = (unsigned char)((bfree % 10) + '0');
			bfree /= 10;
		}
		atascii2screen(line, 32);
		nl->Add(line);
		i += 16;
		m ++;
		if ((m & 7) == 0) {
			i = 0;
			block = GetBlock((short)(361+(m >> 3)));
		}
	}
	delete[] direntry;

	block = GetBlock(360);
	cblock = GetBlock(1024);

	bfree = block[4];
	bfree <<= 8;
	bfree |= block[3];
	if (SectorsPerTrack == 26)
	{
		bfree += cblock[0x7A];
		bfree += ((unsigned long)(cblock[0x7B])) << 8;
	}

	for (k=0;k<32;k++) line[k] = 0x20;
	k = 4+printstring((char *)&line[4], "FREE SECTORS");
	line[k] = ' ';
	if (bfree > 999)
	{
		bfree = 999;
		line[3] = '+';
	}
	for (k=0;k<3;k++)
	{
		line[2-k] = (unsigned char)((bfree % 10) + '0');
		bfree /= 10;
	}
	atascii2screen(line, 32);
	nl->Add(line);

	return nl;
}

/***************************************************************************************************/

long FileATR::GetFileLength(unsigned short sector)
{
	unsigned char blockused[1040];
	unsigned char *block;
	int length, i, offset;

	for (i=0;i<1040;i++) blockused[i] = 0;

	length = 0;
	do
	{
		if (!IsValidBlock(sector)) return length;
		if (blockused[sector]) return length;
		blockused[sector] = 1;

		block = GetBlock(sector);
		if (!block) return length;
		offset = GetBlockLink(sector);
		length += block[SectorSize-1];
		if (offset == 0) return length;
		sector = offset;
	} while (true);
}

unsigned char *FileATR::GetFileBuffer(unsigned short sector)
{
	unsigned char blockused[1040];
	unsigned char *block;
	unsigned char *file;
	int length, i, offset;

	length = GetFileLength(sector);
	if (length <= 0) return 0;
	file = new unsigned char[length];

	for (i=0;i<1040;i++) blockused[i] = 0;

	length = 0;
	do
	{
		if (!IsValidBlock(sector)) return file;
		if (blockused[sector]) return file;
		blockused[sector] = 1;

		block = GetBlock(sector);
		if (!block) return file;
		for (i=0;i<block[SectorSize-1];i++) file[length+i] = block[i];
		offset = GetBlockLink(sector);
		length += block[SectorSize-1];
		if (offset == 0) return file;
		sector = offset;
	} while (true);
}

C64File *FileATR::GetFile(unsigned long index)
{
	C64File *nf;
	unsigned long i, idx, size;
	unsigned short sector;
	unsigned char *block;
	unsigned char *data;
	unsigned char name[16];

	block = GetBlock(361);
	i = 0;
	idx = 0;
	while (true)
	{
		if (i >= 64) return 0;
		if (*block == 0) return 0;
		if (!(*block & 0x80))
		{
			if (idx == index-1) break;
			idx++;
		}
		block += 16;
		i++;
	}
	sector = block[4];
	sector <<= 8;
	sector |= block[3];

	size = GetFileLength(sector);
	data = GetFileBuffer(sector);
	nf = 0;

	if (data)
	{
		for (i=0;i<11;i++) name[i] = block[i+5];
		for (i=11;i<16;i++) name[i] = 0;

		nf = new C64File(new ByteBuffer(name, 16), data, size, block[0], FILEFORMAT_ATARI);
	}

	return nf;
}

void FileATR::CreateATRHeader(unsigned char *header)
{
	int i;

	i = TotalSectors << 3;
	header[0] = 0x96;
	header[1] = 0x02;
	header[2] = (unsigned char)(i & 0x00FF);
	header[3] = (unsigned char)(i >> 8);
	header[4] = (unsigned char)(SectorSize & 0x00FF);
	header[5] = (unsigned char)(SectorSize >> 8);
	for (i=6;i<16;i++) header[i] = 0;
}

SP<StringStream> FileATR::SaveFile(SP<StringStream> temppath, unsigned long index)
{
	C64File *cf;
	FILE *f;
	SP<StringStream> fp;
	SP<StringStream> nm;

	if (!Image) return (new StringStream((char *)0));

	fp = 0;
	if (index == 0) {
		nm = GetPath();
		if (nm == 0) nm = new StringStream("new.atr");
		fp = getfilepath(temppath, nm->GetString());
		remove(fp->GetString());

		Save(fp->GetString());
	} else {
		cf = GetFile(index);
		if (cf) {
			fp = cf->Save(temppath);
			delete cf;
		}
	}

	return fp;
}

/***************************************************************************************************/

#include "debug.h"

short FileATR::AllocNextBlock(void)
{
	short sector;

	for (sector=1;sector<1024;sector++)
	{
		if (IsFreeBlock(sector)) {
			AllocBlock(sector);
			return (sector);
		}
	}

	return (0);
}

short FileATR::WriteData(unsigned char *data, unsigned long size, unsigned short filenum,
						 unsigned char *direntry)
{
	short first, link, bsize, sz, fsize, i;
	unsigned char *block;

	bsize = SectorSize-3;
	fsize = 0;

	link = AllocNextBlock();
	first = link;
	do {
		if (!link) return 0;
		block = GetBlock(link);

		// update direntry block counter
		fsize++;
		direntry[1] = (unsigned char)(fsize & 0x00FF);
		direntry[2] = (unsigned char)(fsize >> 8);

		// put data into sector
		if (size < bsize) sz = size;
		else sz = bsize;
		for (i=0;i<sz;i++) block[i] = data[i];
		for (i=sz;i<bsize;i++) block[i] = data[i];
		data += sz;
		size -= sz;

		link = 0;
		if (size > 0) link = AllocNextBlock();
		block[SectorSize-3] = (unsigned char)(((link>>8) & 0x03) | (filenum << 2));
		block[SectorSize-2] = (unsigned char)(link & 0xFF);
		block[SectorSize-1] = (unsigned char)(sz & 0xFF); // | shortsector;

	} while (size > 0);

	return (first);
}


bool FileATR::WriteFile(SP<ByteBuffer> name, unsigned char type, unsigned char *data, unsigned long size)
{
	unsigned char *direntry;
	unsigned char *block;
	unsigned short link;
	unsigned char *stuff;
	unsigned char c;
	short i, ppos, nsize, filenum;

	if (!data) return false;

	block = GetBlock(361);
	direntry = GetEmptyDirEntry();
	filenum = (short)((direntry-block) >> 4);

	// initialize dir entry
	direntry[0] = type & 0xFE;	// close file (will be "opened" on disk full)
	direntry[1] = 0;
	direntry[2] = 0;
	direntry[3] = 0;
	direntry[4] = 0;
	for (i=0;i<11;i++) direntry[i+5] = 0x20;

	stuff = name->GetBuffer();
	nsize = name->GetSize();

	ppos = nsize-1;
	while (ppos >= 0) {
		if (stuff[ppos] == '.') break;
		ppos--;
	}
	if (ppos < 0) ppos = nsize;

	for (i=0;(i<8)&&(i<ppos);i++) {
		c = stuff[i];
		if (c == 0xA0) break;
		direntry[i+5] = c;
	}

	ppos++;
	stuff += ppos;
	nsize -= ppos;
	for (i=0;(i<3)&&(i<nsize);i++) {
		c = stuff[i];
		if (c == 0xA0) break;
		direntry[i+5+8] = c;
	}

	link = WriteData(data, size, filenum, direntry);
	ValidateBlockCount();
	if (!link) return false;
	direntry[3] = (unsigned char)(link & 0x00FF);
	direntry[4] = (unsigned char)(link >> 8);

	return true;
}

bool FileATR::WriteFile(C64File *file)
{
	if (!file) return true;
	return WriteFile(file->GetName(), 0x42, file->GetData(), file->GetSize());
}

bool FileATR::AddSeperator(void)
{
	unsigned char *direntry;
	short i;

	direntry = GetEmptyDirEntry();
	if (!direntry) return false;

	// initialize dir entry
	direntry[0] = 0x42;
	for (i=1;i< 5;i++) direntry[i] = 0;
	for (i=5;i<16;i++) direntry[i] = '-';

	return true;
}

/***************************************************************************************************/

unsigned char *FileATR::GetDirEntry(unsigned long index)
{
	unsigned short i, idx;
	unsigned char *block;
	unsigned char *entry;
	DirEntryATR *direntry;

	index--;
	entry = 0;
	direntry = new DirEntryATR;
	idx = 0;
	for (i=0;i<64;i++) {
		if ((i&7) == 0) block = GetBlock(361+(i>>3));
		direntry->Entry = block;
		if (direntry->IsNeverUsed()) break;
		if (!(direntry->GetDeleted() || direntry->GetOpened())) {
			if (idx == index) {
				entry = block;
				break;
			}
			idx++;
		}
		block += 16;
	}
	delete[] direntry;

	return (entry);
}

unsigned char *FileATR::GetHeader(void)
{
	return GetBlock(360);
}

void FileATR::ScratchFile(unsigned long index)
{
	unsigned char *block;
	unsigned short sector;

	InitMap();

	block = GetDirEntry(index);
	if (block) {
		sector  = block[4];
		sector <<= 8;
		sector |= block[3];
		block[0] |= 0x80;

		while (IsValidBlock(sector)) {
			if (AlreadyVisited(sector)) break;

			FreeBlock(sector);
			SetVisitedFlag(sector);
			sector = GetBlockLink(sector);
		}
	}
}

/***************************************************************************************************/

void FileATR::Save(const char *filename)
{
	FILE *f;
	unsigned char atrheader[16];

	if (filename == 0) return;

	f = fopen(filename, "wb");
	if (f)
	{
		CreateATRHeader(atrheader);
		fwrite((void *)atrheader, 1, 16, f);
		fwrite((void *)Image, 1, Size, f);
		fclose(f);
	}
}

void FileATR::SwapDirEntries(unsigned long index1, unsigned long index2)
{
	DirEntryATR entry1, entry2;
	DirEntryATR *temp;

	entry1.Entry = GetDirEntry(index1);
	entry2.Entry = GetDirEntry(index2);
	if ((entry1.Entry == 0) || (entry2.Entry == 0)) return;
	temp = entry1.DeepCopy();
	entry1.SetEntry(&entry2);
	entry2.SetEntry(temp);
	temp->Exit();
	delete[] temp;
}
