#include "d64.h"
#include "FileContainer.h"
#include "FileT64.h"
#include "FileLNX.h"
#include "FileP00.h"
#include "FileZipCode.h"
#include "files.h"
#include "c64file.h"
#include "petscii.h"
#include "debug.h"
#include "stringops.h"
#include "cbmlist.h"
#include "DirEntry.h"
#include "GlobalConfig.h"
#include "StringStream.h"
#include "ImageMap.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

static unsigned char rndchars[37] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

static long rnd;

static void initrnd(void)
{
	rnd = (long)time(NULL);
	if (rnd == 0) rnd = -1;
}

static unsigned long getrnd(void)
{
	if (rnd < 0) {
		rnd <<= 1;
		rnd ^= 0x04C11DB7;
	} else {
		rnd <<= 1;
	}

	return (rnd);
}

static unsigned char getrndchar(void)
{
	return (rndchars[(rand() ^ getrnd()) % 36]);
}

static unsigned char rawtypes[26] =
{
	0x04, 0x13, 0x10, 0x15, 0x12,
	0x05, 0x05, 0x12, 0x13, 0x05,
	0x0C, 0x11, 0x07, 0x12, 0x0C,

	0x88, 0x80, 0x80, 0x3F, 0x5F,
	0x3F, 0x5F, 0x91, 0x92, 0x93,
	0x95
};

static unsigned char maxsectors[40]=
{
	21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,
	19,19,19,19,19,19,19,
	18,18,18,18,18,18,
	17,17,17,17,17,
	17,17,17,17,17
};

static int trackoffsets[40];

unsigned char GetMaxSectors40(int cyl)
{
	if ((cyl <= 0) || (cyl > 40)) return (0);

	return (maxsectors[cyl-1]);
}

int GetTrackOffset40(int cyl)
{
	if ((cyl <= 0) || (cyl > 40)) return (-1);

	return (trackoffsets[cyl-1]);
}

static unsigned long getd64sectors(unsigned long tracks)
{
	if (tracks == 35) return 683;
	if (tracks == 40) return 768;
	if (tracks == 70) return 1366;
	return 0;
}

void initd64(void)
{
	int i, sum;

	sum = 0;
	for (i=0; i<40; i++) {
		trackoffsets[i] = sum;
		sum += ((int)maxsectors[i]);
	}

	initrnd();
}

/***************************************************************************************************/

FileD64::FileD64()
{
	map = 0;

#ifdef _DEBUG
	DebugPrint("FileD64(void);");
#endif
}

FileD64::FileD64(int type)
{
	Init(type);

#ifdef _DEBUG
	DebugPrint("FileD64(int type);");
#endif
}

FileD64::~FileD64()
{
#ifdef _DEBUG
	DebugPrint("~FileD64(void);");
#endif

	if (map != 0) delete map;
}

bool FileD64::HasHeader(void)
{
	return true;
}

int FileD64::GetImageSystem(void)
{
	return (IMAGESYSTEM_CBM);
}

void FileD64::InitMap(void)
{
	if (map) {
		delete map;
		map = 0;
	}

	if (Tracks > 0) map = new ImageMap(Tracks, 21);
}

void FileD64::ClearMap(void)
{
	if (map) map->Clear();
}

static int put2digits(unsigned char *buffer, int value)
{
	buffer[1] = (value % 10) + '0';
	value /= 10;
	buffer[0] = (value % 10) + '0';

	return (2);
}

static void puttime(unsigned char *buffer)
{
	time_t rtime;
	struct tm *tinfo;
	int val;

	time(&rtime);
	tinfo = localtime(&rtime);

	buffer += put2digits(buffer, (tinfo->tm_mday));
	*buffer++ = '.';
	buffer += put2digits(buffer, (tinfo->tm_mon + 1));
	*buffer++ = '.';
	val = (tinfo->tm_year + 1900);
	buffer += put2digits(buffer, val / 100);
	buffer += put2digits(buffer, val);
	*buffer++ = ' ';
	buffer += put2digits(buffer, (tinfo->tm_hour));
	*buffer++ = ':';
	buffer += put2digits(buffer, (tinfo->tm_min));
}

void FileD64::Init(int type)
{
	int i, sectors;
	unsigned char *block;

	sectors = 0;
	switch (type)
	{
	default:
	case (IMAGETYPE_D64_170K):
		type = IMAGETYPE_D64_170K;
		Tracks = 35;
		sectors = 683;
		break;
	case (IMAGETYPE_D64_192K):
		Tracks = 40;
		sectors = 768;
		break;
	case (IMAGETYPE_D64_340K):
		Tracks = 70;
		sectors = 1366;
		break;
	}
	Type = type;
	TotalSectors = sectors;

	Size = TotalSectors * 257;
	Image = new unsigned char[Size];
	for (i=0;i<Size;i++) Image[i] = 0;
	for (i=(sectors<<8);i<Size;i++) Image[i] = 1;

	Flags = IMAGEFLAG_NEW;

	map = 0;
	InitMap();

	block = GetHeaderBlock();
	block[0] = 18;
	block[1] = 1;
	block[2] = 'A';
	if (type == (IMAGETYPE_D64_340K)) block[3] = 0x80;
	for (i=0x90;i<0xAB;i++) block[i] = 0xA0;
	block[0xA2] = getrndchar();
	block[0xA3] = getrndchar();
	block[0xA5] = '2';
	block[0xA6] = 'A';
	puttime(&block[0x90]);
	block = GetBlock(block[0], block[1]);
	block[1] = 0xFF;

	Validate();
	InitMap();
}

void FileD64::Init(SP<StringStream> path, const unsigned char *data, int size)
{
	int i;

	SetPath(path);
	Flags &= (~IMAGEFLAG_NEW);

	if (size > Size) size = Size;
	for (i=0;i<size;i++) Image[i] = data[i];
	for (i=size;i<Size;i++) Image[i] = 0;

	InitMap();
	TestDirectory();
}

class FileImage *FileD64::Clone(void)
{
	FileD64 *img;

	img = new FileD64(Type);
	img->Init(Path, Image, Size);

	return (img);
}

FileD64 *CreateD64(SP<StringStream> path, int type, const unsigned char *data, int size)
{
	FileD64 *img;
	int itype;
	unsigned char *data2;
	int tracks, i;

	img = 0;
	itype = IMAGETYPE_NULL;
	data2 = 0;

	if ((type == FILETYPE_ZIP) && (data[0] == 0xFE) && (data[1] == 0x03)) {

		size = 768*256;
		data2 = new unsigned char[size];
		for (i=0;i<size;i++) data2[i] = 0;
		tracks = ParseZipCode(path->GetString(), data2);
		data = data2;

		if (tracks > 0) {
			itype = IMAGETYPE_D64_170K;
			for (i=(683*256);i<(768*256);i++) {
				if (data[i] != 0) {
					itype = IMAGETYPE_D64_192K;
					break;
				}
			}
		}

		path = StripFilename(path->GetString());
		path = path->Append(".d64");
	}

	if (type == FILETYPE_D64) {
		// 170k
		if ((size == (683*256)) || (size == (683*257))) itype = IMAGETYPE_D64_170K;

		// 192k
		if ((size == (768*256)) || (size == (768*257))) itype = IMAGETYPE_D64_192K;

		// 340k
		if ((size == (1366*256)) || (size == (1366*257))) itype = IMAGETYPE_D64_340K;
	}

	if (itype != IMAGETYPE_NULL) {
		img = new FileD64(itype);
		img->Init(path, data, size);
	}

	if (data2 != 0) delete[] data2;

	return (img);
}

static char tempbuffer[_MAX_PATH+1024];

void FileD64::SetHeader(SP<ByteBuffer> header)
{
	unsigned long i;
	unsigned char *block;

	block = GetHeaderBlock();
	for (i=0;i<16;i++) block[i+0x90] = header->GetByte(i);
}

/***************************************************************************************************/

void FileD64::SetErrorCode(short track, short sector, unsigned char error)
{
	int offs;

	offs = GetBlockOffset(track, sector);
	if (offs >= 0) {
		offs += (TotalSectors<<8);
		if (offs < Size) Image[offs] = error;
	}
}

unsigned short FileD64::WriteData(unsigned char *data, unsigned long size)
{
	short track, sector;
	unsigned long sz, i;
	unsigned short link, first;
	unsigned char *block;

	link = AllocFirstBlock();
	first = link;
	do {
		if (!link) return 0;
		track = link >> 8;
		sector = link & 0x00FF;
		block = GetBlock(track, sector);
		if (size < 254) sz = size;
		else sz = 254;
		for (i=0;i<sz;i++) block[i+2] = data[i];
		for (i=sz;i<254;i++) block[i+2] = 0;
		data += sz;
		size -= sz;

		// block errorcode: OK
		SetErrorCode(track, sector, 0x01);

		if (size > 0) {
			link = AllocNextBlock(track, sector, Config->Interleave);
			block[0] = (unsigned char)(link >> 8);
			block[1] = (unsigned char)(link & 0x00FF);
		} else {
			block[0] = 0x00;
			block[1] = (unsigned char)(sz+1);
		}
	} while (size > 0);

	return first;
}

/***************************************************************************************************/

/* erase BAM and free all blocks */

void FileD64::ClearBAM(void)
{
	unsigned char *bam;
	short i, s;

	bam = GetHeaderBlock();

	// clear BAM for tracks 1-35
	for (i=0x04;i<0x90;i++) bam[i] = 0;

	// if D71, clear BAM for tracks 36-70 too
	if (Tracks == 70) {
		for (i=0xDD;i<=0xFF;i++) bam[i] = 0;
		bam = GetBlock(53, 0);
		for (i=0;i<0x69;i++) bam[i] = 0;
	}

	// initialize BAM
	for (i=1;i<=Tracks;i++) {
		if (i != 53) {
			for (s=0;s<GetMaxSectors(i);s++) FreeBlock(i, s);
		}
	}
}

/* */

unsigned char FileD64::GetMaxSectors(short track)
{
	if (track <= 0) return 0;
	if (track > Tracks) return 0;

	if (track <= 35) return GetMaxSectors40(track);
	if ((track <= 40) && (Tracks == 40)) return GetMaxSectors40(track);
	if ((track <= 70) && (Tracks == 70)) return GetMaxSectors40(track-35);
	return 0;
}

/* TRUE if block exists or FALSE for invalid track/sector */

bool FileD64::IsValidBlock(short track, short sector)
{
	if ((track <= 0) || (track > Tracks)) return false;
	if (sector >= GetMaxSectors(track)) return false;
	return true;
}

/* test if a block is allocated. TRUE if block is free */

bool FileD64::IsFreeBlock(short track, short sector)
{
	unsigned char *bamentry;
	unsigned char *bambyte;
	unsigned char mask;

	if (!IsValidBlock(track, sector)) return false;

	mask = 1 << (sector & 7);
	if ((track > 0) && (track <= 35)) {
		bamentry = GetHeaderBlock() + (track << 2);
		bambyte = &bamentry[(sector >> 3) + 1];
		return ((bambyte[0] & mask) != 0);
	}

	if ((track > 35) && (track <= 70) && (Tracks == 70)) {
		bambyte = GetBlock(53, 0) + ((track-36)*3);
		bambyte = &bambyte[ sector >> 3 ];
		return ((bambyte[0] & mask) != 0);
	}
	return false;
}

/* allocate a block */

void FileD64::AllocBlock(short track, short sector)
{
	unsigned char *bamentry;
	unsigned char *bambyte;
	unsigned char mask;

	if (!IsValidBlock(track, sector)) return;

	mask = 1 << (sector & 7);
	if ((track > 0) && (track <= 35)) {
		bamentry = GetHeaderBlock() + (track << 2);
		bambyte = &bamentry[(sector >> 3) + 1];
		if ((bambyte[0] & mask) && (bamentry[0] > 0)) bamentry[0] -= 1;
		bambyte[0] &= ~mask;
	}

	if ((track > 35) && (track <= 70) && (Tracks == 70)) {
		bamentry = GetHeaderBlock() + (track+0xDD-36);
		bambyte = GetBlock(53, 0) + ((track-36)*3);
		bambyte = &bambyte[ sector >> 3 ];
		if ((bambyte[0] & mask) && (bamentry[0] > 0)) bamentry[0] -= 1;
		bambyte[0] &= ~mask;
	}
}

/* free a block */

void FileD64::FreeBlock(short track, short sector)
{
	unsigned char *bamentry;
	unsigned char *bambyte;
	unsigned char mask;

	if (!IsValidBlock(track, sector)) return;

	mask = 1 << (sector & 7);
	if ((track > 0) && (track <= 35)) {
		bamentry = GetHeaderBlock() + (track << 2);
		bambyte = &bamentry[(sector >> 3) + 1];
		if (!(bambyte[0] & mask)) bamentry[0] += 1;
		bambyte[0] |= mask;
	}

	if ((track > 35) && (track <= 70) && (Tracks == 70)) {
		track -= 36;
		bamentry = GetHeaderBlock() + (track+0xDD);
		bambyte = GetBlock(53, 0) + (track*3);
		bambyte = &bambyte[ sector >> 3 ];
		if (!(bambyte[0] & mask)) bamentry[0] += 1;
		bambyte[0] |= mask;
	}
}

/***************************************************************************************************/

/* TRUE if block has been visited already */

bool FileD64::AlreadyVisited(short track, short sector)
{
	if (!IsValidBlock(track, sector)) return false;
	if (!map) InitMap();
	return map->Get(track, sector);
}

/* sets the "visited"-flag for a block */

void FileD64::SetVisitedFlag(short track, short sector)
{
	if (!IsValidBlock(track, sector)) return;
	if (!map) InitMap();
	map->Set(track, sector);
}

/* returns block offset (0 to 682 on a 35 track disk) or -1 for illegal track/sector */

long FileD64::GetBlockOffset(short track, short sector)
{
	if (!IsValidBlock(track, sector)) return -1;

	if ((track > 0) && (track <= 35)) return (trackoffsets[track-1] + sector);
	if ((track > 35) && (track <= 40) && (Tracks == 40)) return (trackoffsets[track-1] + sector);
	if ((track > 35) && (track <= 70) && (Tracks == 70)) return (trackoffsets[35] + trackoffsets[track-36] + sector);
	return -1;
}

/* returns pointer to block in D64 or NULL for illegal track/sector */

unsigned char *FileD64::GetBlock(short track, short sector)
{
	long offs;

	if (!IsValidBlock(track, sector)) return 0;
	offs = GetBlockOffset(track, sector);

	return &Image[offs << 8];
}

unsigned char *FileD64::GetHeaderBlock(void)
{
	return GetBlock(18, 0);
}

/* returns next free sector on a given track according to the interleave or -1 for no free block on track */

short FileD64::AllocNextBlockOnTrack(short track, short sector, short interleave)
{
	short m, i;

	if (!IsValidBlock(track, sector)) return -1;
	m = GetMaxSectors(track);

	sector += interleave;
	for (i=0;i<m;i++) {
		while (sector < 0) sector += m;
		while (sector >= m) sector -= m;

		if (IsFreeBlock(track, sector)) {
			AllocBlock(track, sector);
			return sector;
		}
		sector ++;
	}
	return -1;
}

/* searches for the first free sector on disk which is closest to track 18 */

unsigned short FileD64::AllocFirstBlock()
{
	short i, s, t;

	switch (Config->Strategy)
	{
	default:
	case (D64STRATEGY_CLOSEST):
		for (i=1;i<18;i++) {
			t = 18-i;
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);

			t = 18+i;
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
		if (Tracks == 70) {
			for (i=1;i<18;i++) {
				t = 53-i;
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);

				t = 53+i;
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);
			}
		}
		break;

	case (D64STRATEGY_LINEAR):
		for (t=1;t<=17;t++) {
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
		for (t=19;t<=35;t++) {
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
		if (Tracks == 70) {
			for (t=36;t<=52;t++) {
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);
			}
			for (t=54;t<=70;t++) {
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);
			}
		}
		break;

	case (D64STRATEGY_SUBLINEAR):
		for (t=17;t>=1;t--) {
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
		for (t=19;t<=35;t++) {
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
		if (Tracks == 70) {
			for (t=52;t>=36;t--) {
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);
			}
			for (t=54;t<=70;t++) {
				s = AllocNextBlockOnTrack(t, 0, 0);
				if (s >= 0) return (unsigned short)((t << 8) | s);
			}
		}
		break;
	}

	if (Config->UseTrack18) {
		t = 18;
		s = AllocNextBlockOnTrack(t, 0, 0);
		if (s >= 0) return (unsigned short)((t << 8) | s);
		if (Tracks == 70) {
			t = 53;
			s = AllocNextBlockOnTrack(t, 0, 0);
			if (s >= 0) return (unsigned short)((t << 8) | s);
		}
	}

	return 0;
}

/* search for next block, if possible on same track. if not, on next track etc */

unsigned short FileD64::AllocNextBlockSide1(short track, short sector, short interleave)
{
	int i;

	sector = AllocNextBlockOnTrack(track, sector, interleave);
	if (sector >= 0) return (unsigned short)((track << 8) | sector);

	if (track < 18) {
		if (Config->Strategy == D64STRATEGY_LINEAR) {
			for (i=track+1;i<=17;i++) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		} else {
			for (i=track-1;i>=1;i--) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		}
	}

	if (track <= 35) {
		for (i=19;i<=35;i++) {
			sector = AllocNextBlockOnTrack(i, 0, 0);
			if (sector >= 0) return (unsigned short)((i << 8) | sector);
		}

		if (Config->Strategy == D64STRATEGY_LINEAR) {
			for (i=1;i<track;i++) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		} else {
			for (i=track-1;i>0;i--) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		}
	}
	return 0;
}

unsigned short FileD64::AllocNextBlockSide2(short track, short sector, short interleave)
{
	int i;

	sector = AllocNextBlockOnTrack(track, sector, interleave);
	if (sector >= 0) return (unsigned short)((track << 8) | sector);

	if (track < 53) {
		if (Config->Strategy == D64STRATEGY_LINEAR) {
			for (i=track+1;i<=52;i++) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		} else {
			for (i=track-1;i>=36;i--) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		}
	}

	if (track <= 70) {
		for (i=54;i<=70;i++) {
			sector = AllocNextBlockOnTrack(i, 0, 0);
			if (sector >= 0) return (unsigned short)((i << 8) | sector);
		}

		if (Config->Strategy == D64STRATEGY_LINEAR) {
			for (i=36;i<track;i++) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		} else {
			for (i=track-1;i>35;i--) {
				sector = AllocNextBlockOnTrack(i, 0, 0);
				if (sector >= 0) return (unsigned short)((i << 8) | sector);
			}
		}
	}
	return 0;
}

/* search for next block, if possible on same track. if not, on next track etc */

unsigned short FileD64::AllocNextBlock(short track, short sector, short interleave)
{
	unsigned short ts;

	if (track < 36) {
		ts = AllocNextBlockSide1(track, sector, interleave);
		if (ts != 0) return (ts);
		ts = AllocNextBlockSide2(52, sector, interleave);
		if (ts != 0) return (ts);
	} else {
		if (Tracks == 70) {
			ts = AllocNextBlockSide2(track, sector, interleave);
			if (ts != 0) return (ts);
			ts = AllocNextBlockSide1(17, sector, interleave);
			if (ts != 0) return (ts);
		}
	}

	if (Config->UseTrack18) {
		sector = AllocNextBlockOnTrack(18, 0, 0);
		if (sector >= 0) return (unsigned short)((18 << 8) | sector);

		if (Tracks == 70) {
			sector = AllocNextBlockOnTrack(53, 0, 0);
			if (sector >= 0) return (unsigned short)((53 << 8) | sector);
		}
	}
	return 0;
}

/***************************************************************************************************/

/* validate a file */

void FileD64::ValidateLinks(short track, short sector)
{
	unsigned char *block;

	if (!IsValidBlock(track, sector)) return;		// end validate on last block or illegal link
	if (AlreadyVisited(track, sector)) return;		// end validate on link loop
	SetVisitedFlag(track, sector);
	AllocBlock(track, sector);
	block = GetBlock(track, sector);
	ValidateLinks(block[0], block[1]);
}

/* validate all files and directory */

void FileD64::Validate()
{
	unsigned long i;
	unsigned char *direntry;

	ClearBAM();
	AllocBlock(18, 0);			// allocate BAM
	ValidateLinks(18, 1);		// allocate directory

	i = 1;
	do {
		direntry = GetDirEntry(i);
		if (direntry) ValidateLinks(direntry[3], direntry[4]);
		i++;
	} while (direntry);
}

/* deletes a file */

void FileD64::ScratchFile(unsigned long index)
{
	unsigned char *block;
	unsigned char track, sector;

	InitMap();

	block = GetDirEntry(index);
	if (block) {
		track = block[3];
		sector = block[4];
		block[2] = 0;

		while (IsValidBlock(track, sector)) {
			if (AlreadyVisited(track, sector)) break;

			FreeBlock(track, sector);
			SetVisitedFlag(track, sector);
			block = GetBlock(track, sector);
			track = block[0];
			sector = block[1];
		}
	}
}

/* returns the byte size of a file or copy file data to a buffer */

long FileD64::GetFileLinks(short track, short sector, unsigned char *buffer)
{
	unsigned char *block;
	long i, size;

	if (!IsValidBlock(track, sector)) return 0;		// end delete on last block or illegal link
	if (AlreadyVisited(track, sector)) return 0;	// end validate on link loop
	SetVisitedFlag(track, sector);
	block = GetBlock(track, sector);

	if (block[0]) {
		size = 254;
	} else {
		size = block[1];
		if (size < 2) size = 0;		// no block sizes -1 and -2 (only happens if file is badly linked to an empty block)
		else size--;
	}

	// only copy block data to a buffer if buffer point is not NULL
	if (buffer) {
		for (i=0;i<size;i++) buffer[i] = block[i+2];
		buffer += size;
	}

	return (GetFileLinks(block[0], block[1], buffer)+size);
}

/* returns the byte size of a file */

long FileD64::GetFileLength(short track, short sector)
{
	InitMap();
	return GetFileLinks(track, sector, 0);
}

/* returns the data of a file */

unsigned char *FileD64::GetFileData(short track, short sector)
{
	unsigned char *buffer;

	buffer = new unsigned char[GetFileLength(track, sector)];
	InitMap();
	GetFileLinks(track, sector, buffer);

	return buffer;
}

/***************************************************************************************************/

/* test the directory for obvious errors */

void FileD64::TestDirectory()
{
	int i, o;
	short track, sector;
	unsigned char *block;
	bool ok;

	Flags &= ~(IMAGEFLAG_CORRUPT_DIRLINK | IMAGEFLAG_LOOPED_DIR | IMAGEFLAG_INVALID_BAM);

	InitMap();

	SetVisitedFlag(18, 0);
	track = 18;
	sector = 1;
	while (true) {
		if (!IsValidBlock(track, sector)) {
			Flags |= IMAGEFLAG_CORRUPT_DIRLINK;
			break;
		}
		if (AlreadyVisited(track, sector)) {
			Flags |= IMAGEFLAG_LOOPED_DIR;
			break;
		}
		SetVisitedFlag(track, sector);
		block = GetBlock(track, sector);

		if (!block[0]) break;

		track = block[0];
		sector = block[1];
	}

	block = Image;
	Image = new unsigned char[Size];
	for (i=0;i<Size;i++) Image[i] = block[i];
	Validate();
	o = GetBlockOffset(18, 0) << 8;
	ok = true;
	for (i=(o+0x04);i<(o+0x48);i++) {
		if (block[i] != Image[i]) ok = false;
	}
	for (i=(o+0x4C);i<(o+0x90);i++) {
		if (block[i] != Image[i]) ok = false;
	}
	delete[] Image;
	Image = block;

	if (!ok) Flags |= IMAGEFLAG_INVALID_BAM;
}

/* get used dir entry by offset (starting at 1), returns NULL if failed */

unsigned char *FileD64::GetDirEntry(unsigned long index)
{
	unsigned long i, idx, count;
	short track, sector;
	unsigned char *block;

	InitMap();

	track = 18;
	sector = 1;
	SetVisitedFlag(track, sector);
	block = GetBlock(track, sector);
	i = 0;
	idx = 0;
	count = 0;
	while (block)
	{
		if (block[2+i]) idx++;
		if (idx == index) return &block[i];

		i += 32;
		if (i >= 256) {
			i = 0;
			track = block[0];
			sector = block[1];
			if (AlreadyVisited(track, sector)) return 0;
			SetVisitedFlag(track, sector);
			block = GetBlock(track, sector);
		}
	}
	return 0;
}

/* exchanges two dir entries */

void FileD64::SwapDirEntries(unsigned long index1, unsigned long index2)
{
	DirEntryD64 entry1, entry2;
	DirEntryD64 *temp;

	entry1.Entry = GetDirEntry(index1);
	entry2.Entry = GetDirEntry(index2);
	if ((entry1.Entry == 0) || (entry2.Entry == 0)) return;
	temp = entry1.DeepCopy();
	entry1.SetEntry(&entry2);
	entry2.SetEntry(temp);
	temp->Exit();
	delete[] temp;
}

/* search directory for the first free entry, returns NULL if no entry was found */

unsigned char *FileD64::GetEmptyDirEntry()
{
	short track, sector, i;
	unsigned char *block;

	ClearMap();

	track = 18;
	sector = 1;
	while (IsValidBlock(track, sector))
	{
		if (AlreadyVisited(track, sector)) break;
		SetVisitedFlag(track, sector);
		block = GetBlock(track, sector);

		for (i=0;i<256;i+=32) {
			if (!block[i+2]) return &block[i];
		}
		track = block[0];
		sector = block[1];
	}
	return 0;	/* end of dir reached or illegal link */
}

/* extend directory file by 1 block, returns NULL if failed */

unsigned char *FileD64::AddDirBlock()
{
	unsigned short link;
	short track, sector, i;
	unsigned char *block;

	ClearMap();

	track = 18;
	sector = 1;
	while (IsValidBlock(track, sector))
	{
		if (AlreadyVisited(track, sector)) return 0;
		SetVisitedFlag(track, sector);
		block = GetBlock(track, sector);

		if (!block[0]) {
			if (Config->DirTrack18Only) {
				link = AllocNextBlockOnTrack(track, sector, Config->DirInterleave);
				if (link < 0) return 0;
				sector = link;
			} else {
				link = AllocNextBlock(track, sector, Config->DirInterleave);
				if (!link) return 0;
				track = (short)(link >> 8);
				sector = (short)(link & 0x00FF);
			}
			block[0] = (unsigned char)track;
			block[1] = (unsigned char)sector;
			block = GetBlock(track, sector);
			for (i=0;i<256;i++) block[i] = 0;
			block[1] = 0xFF;

			SetErrorCode(track, sector, 0x01);
			return block;
		}
		track = block[0];
		sector = block[1];
	}
	return 0;	/* end of dir reached or illegal link */
}

/* writes a file to disk image, returns false if failed */

bool FileD64::WriteFile(SP<ByteBuffer> name, unsigned char type, unsigned char *data, unsigned long size)
{
	unsigned char *direntry;
	unsigned long blocks;
	unsigned short link;
	short i;
	SP<StringStream> sname;
	char *stuff;

	if (!data) return false;

	sname = new StringStream(name);
	stuff = sname->GetString();

	direntry = GetEmptyDirEntry();
	if (!direntry) direntry = AddDirBlock();
	if (!direntry) return false;
	link = WriteData(data, size);
	if (!link) return false;

	for (i=2;i<32;i++) direntry[i] = 0;
	for (i=0;i<16;i++) direntry[i+5] = 0xA0;
	direntry[2] = type;
	direntry[3] = (unsigned char)(link >> 8);
	direntry[4] = (unsigned char)(link & 0x00FF);
	blocks = (size+253) / 254;
	if (blocks == 0) blocks = 1;
	if (blocks > 65535) blocks = 65535;
	direntry[0x1E] = (unsigned char)(blocks & 0x00FF);
	direntry[0x1F] = (unsigned char)(blocks >> 8);

	i = 0;
	while ((stuff[i]) && (i<16)) {
		direntry[i+5] = stuff[i];
		i++;
	}
	return true;
}

bool FileD64::WriteFile(C64File *file)
{
	if (!file) return true;
	return WriteFile(file->GetName(), file->GetType(), file->GetData(), file->GetSize());
}

bool FileD64::AddSeperator(void)
{
	unsigned char *direntry;
	short i;

	direntry = GetEmptyDirEntry();
	if (!direntry) direntry = AddDirBlock();
	if (!direntry) return false;

	for (i=2;i<32;i++) direntry[i] = 0;
	for (i=0;i<16;i++) direntry[i+5] = '-';
	direntry[2] = 0x80;
	direntry[3] = 18;
	direntry[4] = 1;

	return true;
}

/***************************************************************************************************/

CbmList *FileD64::GetDirectory()
{
	unsigned long i, k, bsize, bfree;
	CbmList *nl;
	unsigned char *block;
	unsigned char *cblock;
	unsigned char line[32];
	unsigned char *cl;
	unsigned char c, last;
	short track, sector;

	ClearMap();

	block = GetHeaderBlock();
	if (!block) return 0;
	nl = new CbmList;
	nl->Init(CHARSET_COMMODORE);
	bfree = -block[0x48];
	for (i=4;i<0x90;i+=4) bfree += block[i];
	if (Tracks == 70) {
		for (i=0xDD;i<256;i++) bfree += block[i];
	}

	for (i=0;i<32;i++) line[i] = 0x20;
	cl = line;
	*cl++ = 0x30;
	*cl++ = 0x20;
	*cl++ = 0x22;
	for (i=0;i<16;i++) *cl++ = block[0x90+i];
	petscii2screen(&line[3], 0x10);
	*cl++ = 0x22;
	*cl++ = 0x20;
	for (i=0;i<4;i++) *cl++ = block[0xA2+i];

	if (block[0xA6] == 0xA0) *cl++ = 0x31;
	else *cl++ = block[0xA6];

	petscii2screen(&line[21], 5);
	for (k=0;k<24;k++) line[k+2] |= 0x80;
	nl->Add(line);

	track = 18;
	sector = 1;
	SetVisitedFlag(track, sector);
	block = GetBlock(track, sector);
	i = 0;
	while (true)
	{
		for (k=0;k<32;k++) line[k] = 0x20;
		cblock = &block[i];
		cl = line;
		if (cblock[2])
		{
			bsize   = cblock[0x1F];
			bsize <<= 8;
			bsize  |= cblock[0x1E];

			k = printint((char *)line, bsize);
			line[k] = 0x20;
			cl = &line[5];
			if (k >= 4) cl++;
			if (k > 4) cl++;
			*cl++ = 0x22;
			last = 1;
			for (k=0;k<16;k++)
			{
				c = cblock[5+k];
				if (((c == 0xA0) || (c == 0x22)) && last)
				{
					c = 0x22;
					last = 0;
				}
				if (!last) c &= 0x7F;
				*cl++ = c;
			}
			petscii2screen(&cl[-16], 16);

			if (last) c = 0x22;
			else c = 0x20;
			*cl++ = c;

			if (cblock[2] & 0x80) c = 0x20;
			else c = '*';
			*cl++ = c;

			k = cblock[2] & 0x0F;
			*cl++ = rawtypes[k];
			*cl++ = rawtypes[k+5];
			*cl++ = rawtypes[k+10];

			if (cblock[2] & 0x40) c = 0x3C;
			else c = 0x20;
			*cl++ = c;

			nl->Add(line);
		}
		i+=0x20;
		if (i >= 256)
		{
			i = 0;
			track = block[0];
			sector = block[1];
			if (track != 18) break;
			if (AlreadyVisited(track, sector)) break;
			SetVisitedFlag(track, sector);
			block = GetBlock(track, sector);
			if (!block) break;
		}
	}
	for (k=0;k<32;k++) line[k] = 0x20;
	k = printint((char *)line, bfree);
	k+= printstring((char *)&line[k], " BLOCKS FREE.");
	line[k] = 0x20;
	for (k=0;k<18;k++)
	{
		c = line[k];
		if ((c >= 'A') && (c <= 'Z')) line[k] &= 0x3F;
	}

	nl->Add(line);

	return nl;
}

/***************************************************************************************************/

C64File *FileD64::GetFile(unsigned long index)
{
	C64File *nf;
	unsigned long size;
	short track, sector;
	unsigned char *direntry;
	unsigned char *data;

	direntry = GetDirEntry(index);
	track = direntry[3];
	sector = direntry[4];
	size = GetFileLength(track, sector);
	data = GetFileData(track, sector);
	nf = 0;

	if (data)
	{
		nf = new C64File(new ByteBuffer(&direntry[5], 16), data, size, direntry[2], FILEFORMAT_CBM);
	}

	return nf;
}

unsigned char *FileD64::GetHeader(void)
{
	return GetHeaderBlock();
}

/***************************************************************************************************/

SP<StringStream> FileD64::SaveFile(SP<class StringStream> temppath, unsigned long index)
{
	C64File *cf;
	SP<StringStream> fp;
	SP<StringStream> nm;

	if (!Image) return (new StringStream((char *)0));

	fp = 0;
	if (index == 0) {
		nm = GetPath();
		if (nm == 0) nm = new StringStream("new.d64");
		fp = getfilepath(temppath, nm->GetString());
		remove(fp->GetString());

		Save(fp->GetString());
	} else {
		cf = GetFile(index);
		if (cf) {
			fp = cf->Save(temppath);
			delete cf;
		}
	}

	return fp;
}

void FileD64::Save(const char *filename)
{
	FILE *f;
	int i, sz;

	if (filename == 0) return;

	f = fopen(filename, "wb");
	if (f) {

		// test if error block is present
		sz = TotalSectors*256;
		if (Size == (sz+TotalSectors)) {

			// test if errors are present, if not: clip error block
			for (i=0;i<TotalSectors;i++) {
				if (Image[sz+i] > 0x01) {
					sz = Size;
					break;
				}
			}
		}

		fwrite((void *)Image, 1, sz, f);
		fclose(f);
	}
}
