#include "window.h"
#include "cbmlist.h"
#include "d64.h"
#include "atr.h"
#include "debug.h"
#include "dragdrop.h"
#include "files.h"
#include "stringops.h"
#include <windows.h>
#include <ddraw.h>
#include <stdio.h>
#include <shlobj.h>
#include "process.h"
#include "netarc.h"
#include "resource.h"
#include "StringStream.h"
#include "FileContainer.h"
#include "StringContainer.h"

#include "Executable.h"
#include "Win32_WindowRename.h"
#include "Win32_WindowHelp.h"
#include "Win32_WindowConfig.h"
#include "Win32_WindowNewImage.h"
#include "Win32_WindowLogDir.h"
#include "Win32_WindowCodenet.h"
#include "GlobalConfig.h"

#ifndef WM_MOUSEWHEEL
#define WM_MOUSEWHEEL 0x020A
#endif

enum
{
	CM_FILE_NEW = 1,
	CM_FILE_OPEN,
	CM_FILE_SAVE,
	CM_FILE_SVAS,
	CM_FILE_SVDIR,
	CM_FILE_LOGDIR,
	CM_FILE_EXIT,
	CM_EDIT_UNDO,
	CM_EDIT_REDO,
	CM_EDIT_RENAME,
	CM_EDIT_DELETE,
	CM_EDIT_VALIDATE,
	CM_EDIT_HEADER,
	CM_EDIT_SEPERATOR,
	CM_EDIT_SETTINGS,
	CM_HELP_HELP,
	CM_HELP_ABOUT
};

const char *Arc64_DefaultTitle = "Arc64 V2.5";

int LoadRequester(HWND handle, char *filename, int imagesys)
{
	OPENFILENAME ofn;
	BOOL ok;
	int i;

	for (i=0;i<sizeof(ofn);i++) ((unsigned char *)(&ofn))[i] = 0;

	ofn.lStructSize = sizeof(ofn);
	ofn.hwndOwner = handle;
	ofn.lpstrFile = filename;
	ofn.nMaxFile = 4096;
	ofn.nFilterIndex = 3;
	ofn.lpstrFilter = "All files\0*\0ATR Image\0*.ATR\0D64 Image\0*.D64\0\0";
	ofn.Flags = OFN_FILEMUSTEXIST;

	switch (imagesys)
	{
	default:
		ofn.nFilterIndex = 1;
		break;
	case (IMAGESYSTEM_ATR):
		ofn.nFilterIndex = 2;
		break;
	case (IMAGESYSTEM_CBM):
		ofn.nFilterIndex = 3;
		break;
	}

	ok = GetOpenFileName(&ofn);
	return ok;
}

int SaveRequester(HWND handle, char *filename, int imagesys)
{
	OPENFILENAME ofn;
	BOOL ok;
	int i;

	for (i=0;i<sizeof(ofn);i++) ((unsigned char *)(&ofn))[i] = 0;

	ofn.lStructSize = sizeof(ofn);
	ofn.hwndOwner = handle;
	ofn.lpstrFile = filename;
	ofn.nMaxFile = 4096;
	ofn.nFilterIndex = 3;
	ofn.lpstrFilter = "All files\0*\0ATR Image\0*.ATR\0D64 Image\0*.D64\0\0";

	switch (imagesys)
	{
	default:
		ofn.nFilterIndex = 1;
		break;
	case (IMAGESYSTEM_ATR):
		ofn.nFilterIndex = 2;
		break;
	case (IMAGESYSTEM_CBM):
		ofn.nFilterIndex = 3;
		break;
	}

	ok = GetSaveFileName(&ofn);

	switch (imagesys)
	{
	default:
		break;
	case (IMAGESYSTEM_ATR):
		AddExtension(filename, ".ATR");
		break;
	case (IMAGESYSTEM_CBM):
		AddExtension(filename, ".D64");
		break;
	}

	return ok;
}

/**************************************************************************************/
/*
void CreateControl(HWND parent, int label, unsigned int style, char *wclass, char *text, unsigned int flags)
{
	HWND hwnd;
	HFONT hfont;
	unsigned int pos;
	char tmp[128];

	hwnd = CreateWindowEx(style, wclass, text, flags,
		0, 0, GetSystemMetrics(SM_CYVSCROLL), 50, parent, (HMENU)label, GetModuleHandle(NULL), NULL);

	pos=printstring(tmp, "Could not create control: ");
	printstring(&tmp[pos], wclass);

	if (hwnd == NULL)
	{
		MessageBox(hwnd, tmp, "Error", MB_OK | MB_ICONERROR);
	}
	else
	{
		hfont = (HFONT)GetStockObject(DEFAULT_GUI_FONT);
		SendMessage(hwnd, WM_SETFONT, (WPARAM)hfont, MAKELPARAM(FALSE, 0));
	}
}
*/
/*
void CreateScrollBar(HWND parent, int label)
{
	CreateControl(parent, label, 0, "SCROLLBAR", 0, WS_CHILD | WS_VISIBLE | WS_GROUP | SBS_VERT | SBS_RIGHTALIGN);
}
*/
void SetItemPos(HWND hwnd, int handle, int xp, int yp, int wid, int hei)
{
	HWND sub;
	sub = GetDlgItem(hwnd, handle);
	if (sub) SetWindowPos(sub, NULL, xp, yp, wid, hei, SWP_NOZORDER);
}

void DestroyItem(HWND hwnd, int handle)
{
	HWND sub;

	sub = GetDlgItem(hwnd, handle);
	if (sub) DestroyWindow(sub);
}

/**************************************************************************************/

extern bool OpenDirectDraw();
extern void CloseDirectDraw();
extern bool OpenPrimarySurface();
extern void ClosePrimarySurface();

/**************************************************************************************/

/**/
static char path[_MAX_PATH];
CbmWindow *win;

CbmWindow::CbmWindow()
{
}

CbmWindow::~CbmWindow()
{
}

void CbmWindow::ClearSelections(void)
{
	unsigned long i;

	if (selections) {
		for (i=0;i<entries;i++) selections[i] = 0;
	}
}

void CbmWindow::RemoveSelection(unsigned long index)
{
	unsigned long i;

	if ((selections) && (entries > 0)) {
		for (i=(index+1);i<(entries-1);i++) {
			selections[i] = selections[i+1];
		}
		entries--;
		selections[entries] = 0;
	}
}

unsigned long CbmWindow::GetFirstSelection(void)
{
	unsigned long i;

	if (selections) {
		for (i=0;i<entries;i++) {
			if (selections[i]) return (i);
		}
	}
	return (0);
}

void CbmWindow::CycleImage(bool forward)
{
	class FileContainer *fc;
	FILE *f;
	unsigned char *data;
	long size;
	SP<StringStream> path;

	SP<FileImage> img;
	int type;
	bool loop;

	path = Image->GetPath();
	if (path == 0) return;

	loop = true;
	while (loop) {
		if (forward) path = nextfilename(path);
		else path = prevfilename(path);
		if (path == 0) return;

		type = getfiletype(path->GetString());
		img = Image;

		data = 0;
		size = 0;

		f = fopen(path->GetString(), "rb");
		if (f) {
			fseek(f, 0, SEEK_END);
			size = ftell(f);
			fseek(f, 0, SEEK_SET);
			if (size <= 2*1024*1024) {
				data = new unsigned char[size];
				if (size > 0) {
					size = fread((void *)data, 1, size, f);
					loop = false;
				}
			}
			fclose(f);
		} else {
			loop = false;
		}
	}

	if (data != 0) {
		img = CreateImage(path, type, data, size);

		if (img != 0) {
			Image = img;

			ClearSelections();
			RefreshImageDisplay();
			UpdateWindowHeight();
			SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
		}
	}
}


bool CbmWindow::Drop(SP<StringStream> path)
{
	class FileContainer *fc;
	FILE *f;
	unsigned char *data;
	long size;

	int type, ok;
	SP<FileImage> img;

	if (path == 0) return (false);

	type = getfiletype(path->GetString());
	ok = 0;
	img = Image;

	data = 0;
	size = 0;

	f = fopen(path->GetString(), "rb");
	if (f) {
		fseek(f, 0, SEEK_END);
		size = ftell(f);
		fseek(f, 0, SEEK_SET);
		if (size <= 2*1024*1024) {
			data = new unsigned char[size];
			if (size > 0) {
				size = fread((void *)data, 1, size, f);
			}
		}
		fclose(f);
	} else {
		MessageBox(handle, "Could not open file!", "Error!", MB_ICONEXCLAMATION | MB_OK);
	}

	if (data != 0) {
		img = CreateImage(path, type, data, size);

		if (img != 0) {
			ClearSelections();
			Image = img;
			ok = 1;
		} else {
			fc = ParseFile(path, type, data, size);
			if (fc != 0) {
				ok = fc->AddTo(Image.GetNativePointer());
				delete fc;
				if (!ok) {
					MessageBox(handle, "Disk image full!", "Error!", MB_ICONEXCLAMATION | MB_OK);
				}
			}
//			MessageBox(handle, "Not a valid disk image!", "Error!", MB_ICONEXCLAMATION | MB_OK);
		}

		delete[] data;
	}

	return (ok);
}

void CbmWindow::DropFiles(HDROP hdrop)
{
	StringContainer *sc;
	int i, num;
	bool redraw;

	sc = new StringContainer();
	num = DragQueryFile(hdrop, 0xFFFFFFFF, path, _MAX_PATH);

	redraw = false;
	for (i=0;i<num;i++) {
		DragQueryFile(hdrop, i, path, _MAX_PATH);
		sc->Add(path);
	}

	num = sc->GetCount();
	for (i=0;i<num;i++) {
		redraw |= Drop(new StringStream(sc->Get(i)));
	}
	DragFinish(hdrop);

	delete sc;

	if (redraw) {
		RefreshImageDisplay();
		UpdateWindowHeight();
	}
}

void CbmWindow::RunFile(unsigned short sel)
{
	WindowCodenet *wcn;
	SP<FileD64> d64;
	C64File *cf;

	d64 = *(SP<FileD64> *)&Image;

	sel >>= 1;
	if ((net) && (d64 != 0) && (sel)) {
		wcn = new WindowCodenet(hInstance);
		wcn->Open(handle);
		cf = d64->GetFile(sel);
		net->RunFile(handle, wcn, cf);		// TODO
	}
}

bool CbmWindow::DragOut(unsigned short sel)
{
	unsigned long i, size;
	char *all;
	char *drop;
	HGLOBAL hglobal;
	DROPFILES *dropfiles;
	SP<StringStream> tmppath;
	StringContainer *sc;

	sc = new StringContainer();

	sel /= zoom;

	GetTempPath(_MAX_PATH-1, path);
	if (Image != 0)
	{
		if (!sel) sc->Add(Image->SaveFile(new StringStream(path), 0)->GetString());
		else
		{
			for (i=1;i<entries;i++)
			{
				if (selections[i]) {
					tmppath = Image->SaveFile(new StringStream(path), i);
					if (tmppath != 0) sc->Add(tmppath->GetString());
				}
			}
		}
	}
	all = sc->GetAll();
	size = 0;
	while (all[size] || all[size+1]) size++;
	size += 2;

	hglobal = GlobalAlloc(GMEM_SHARE | GHND | GMEM_ZEROINIT, sizeof(DROPFILES)+size);
	dropfiles = (DROPFILES *) GlobalLock(hglobal);
	dropfiles->pFiles = sizeof(DROPFILES);
	dropfiles->fWide = 0;
	drop = ((char *)dropfiles) + sizeof(DROPFILES);
	for (i=0;i<size;i++) drop[i] = all[i];
	GlobalUnlock(hglobal);

	i = drag_file_out(hglobal);

	delete[] all;
	delete sc;

	return (i != 0);
}

int MouseWheelInit = 1;
int MouseWheelPos = 0;

unsigned long CbmWindow::GetPageHeight()
{
	RECT rect;

    GetClientRect(handle, &rect);

	return (rect.bottom-rect.top);
}

unsigned long CbmWindow::GetWindowWidth()
{
	return (bitmapwidth*zoom+(GetSystemMetrics(SM_CXFRAME)+GetSystemMetrics(SM_CXEDGE))*2);
}

unsigned long GetWindowBorderHeight(void)
{
	unsigned long hei;

	hei = (GetSystemMetrics(SM_CYFRAME)+GetSystemMetrics(SM_CYEDGE)) << 1;
	hei += GetSystemMetrics(SM_CYCAPTION);
	hei += GetSystemMetrics(SM_CYMENU);

	return (hei);
}

unsigned long CbmWindow::GetWindowHeight()
{
	unsigned long h;

	h = bitmapheight + 8;
	if (h < (16*8)) h = 16*8;
	h *= zoom;
	h += GetWindowBorderHeight();

	return h;
}

static LRESULT CALLBACK WinCallBack(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	int sel;

//	if ((msg != 0x84) && (msg != 0x20) && (msg != 0xA0) && (msg != 0x200))
//		sDPrintF("%08x, %08x, %08x\n", msg, wParam, lParam);

	switch (msg)
	{
	case WM_VSCROLL:
		int scrollcode;
		scrollcode = LOWORD(wParam);

		if ((scrollcode == SB_THUMBPOSITION) || (scrollcode == SB_THUMBTRACK)) {
			SetScrollPos(hwnd, SB_VERT, HIWORD(wParam), true);
			win->Redraw(false);
		}
		if (scrollcode == SB_PAGEUP) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()-win->GetPageHeight(), true);
			win->Redraw(false);
		}
		if (scrollcode == SB_PAGEDOWN) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()+win->GetPageHeight(), true);
			win->Redraw(false);
		}
		if (scrollcode == SB_LINEUP) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()-16, true);
			win->Redraw(false);
		}
		if (scrollcode == SB_LINEDOWN) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()+16, true);
			win->Redraw(false);
		}
		return 0;

	case WM_MOUSEWHEEL:
		int mwheel;
		long oldmw;
		mwheel = (short)HIWORD(wParam);
		if (MouseWheelInit) {
			MouseWheelInit = 0;
			MouseWheelPos = 0x1000000;
		}
		oldmw = MouseWheelPos;
		MouseWheelPos += mwheel;

		if ((oldmw/120) > (MouseWheelPos/120)) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()+16, true);
			win->Redraw(false);
		}
		if ((oldmw/120) < (MouseWheelPos/120)) {
			SetScrollPos(hwnd, SB_VERT, win->GetScrollPos()-16, true);
			win->Redraw(false);
		}
		return 0;

	case WM_DROPFILES:
		win->DropFiles((HDROP)wParam);
		return 0;

	case WM_LBUTTONDOWN:
		DragAcceptFiles(hwnd, false);
		sel = HIWORD(lParam) + win->GetScrollPos();
		win->Select((sel >> 3), (wParam & MK_CONTROL) != 0, (wParam & MK_SHIFT) != 0);
		win->Paint();
		win->Redraw(false);
		win->DragOut(sel >> 3);
		DragAcceptFiles(hwnd, true);
		return 0;
	case WM_RBUTTONDOWN:
		sel = HIWORD(lParam) + win->GetScrollPos();
		win->Select((sel >> 3), (wParam & MK_CONTROL) != 0, (wParam & MK_SHIFT) != 0);
		win->Paint();
		win->Redraw(false);
		return 0;
	case WM_RBUTTONUP:
		sel = HIWORD(lParam) + win->GetScrollPos();
		win->Select((sel >> 3), (wParam & MK_CONTROL) != 0, (wParam & MK_SHIFT) != 0);
		win->Paint();
		win->Redraw(false);
		win->RunFile(sel >> 3);
		return 0;
	case WM_SIZE:
		win->Size();
		win->Redraw(false);
		break;
	case WM_PAINT:
		win->Paint();
		win->Redraw(false);
		break;
	case WM_DISPLAYCHANGE:
		win->DisplayChange();
		win->Redraw(false);
		return 0;
	case WM_ERASEBKGND:
		RECT clientrect;

		GetClientRect(hwnd, &clientrect);
		ExcludeClipRect((HDC)wParam, 0, 0, clientrect.right, clientrect.bottom);
		win->Redraw(false);
		break;

	case WM_KEYDOWN:
		win->KeyDown(wParam);
		break;

	case WM_NOTIFY:
	case WM_COMMAND:
		win->Notify(wParam, lParam);
		return 0;
	case WM_CREATE:
		win->Create(hwnd);
		return 0;
	case WM_CLOSE:
		DestroyWindow(hwnd);
		break;
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	case WM_GETMINMAXINFO:
		MINMAXINFO FAR *mmi;
		mmi = (MINMAXINFO FAR *) lParam;
		mmi->ptMinTrackSize.x = win->GetWindowWidth();
		mmi->ptMaxTrackSize.x = mmi->ptMinTrackSize.x;
		mmi->ptMinTrackSize.y = GetWindowBorderHeight()+16;
		mmi->ptMaxTrackSize.y = GetSystemMetrics(SM_CYSCREEN)-32;
		break;
	default:
		return DefWindowProc(hwnd, msg, wParam, lParam);
	}
	return DefWindowProc(hwnd, msg, wParam, lParam);
//	return 0;
}

void CbmWindow::Create(HWND hwnd)
{
	HMENU hMenu;
	HMENU hSubMenu;

	hMenu = CreateMenu();

	hSubMenu = CreatePopupMenu();
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_NEW,  "&New");
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_OPEN, "&Open...");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_SAVE, "&Save");
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_SVAS, "Save &As...");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_SVDIR, "Save &Directory...");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_LOGDIR, "&Create Dirlisting...");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_FILE_EXIT, "E&xit");
	AppendMenu(hMenu, MF_STRING | MF_POPUP, (UINT)hSubMenu, "&File");

	hSubMenu = CreatePopupMenu();
//	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_UNDO, "Undo");
//	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_REDO, "Redo");
//	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_RENAME, "&Rename");
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_DELETE, "&Delete");
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_VALIDATE, "&Validate");
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_HEADER, "&Header");
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_SEPERATOR, "Seperator");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_EDIT_SETTINGS, "&Settings");
	AppendMenu(hMenu, MF_STRING | MF_POPUP, (UINT)hSubMenu, "&Edit");

	hSubMenu = CreatePopupMenu();
	AppendMenu(hSubMenu, MF_STRING, CM_HELP_HELP, "&Help");
	AppendMenu(hSubMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hSubMenu, MF_STRING, CM_HELP_ABOUT, "&About");
	AppendMenu(hMenu, MF_STRING | MF_POPUP, (UINT)hSubMenu, "&Help");

	SetMenu(hwnd, hMenu);
}

static const char *HelpText =
"R - Rename/Edit directory entry\n\
D - Delete file(s)\n\
H - Edit header\n\
V - Validate disk\n\
+/- - Next/Previous disk image\n\
LMB - Select file(s)/Drag file(s)\n\
RMB - Send file via CodeNet\n\
ESC - Exit";

static const char *AboutText = "Arc64 V2.5\n\n 2002-2013 Graham\n\n02.09.2013";

void CbmWindow::OpenImage(void)
{
	int is;
	SP<StringStream> str;

	LoadFileName[0] = 0;
	is = IMAGESYSTEM_NULL;
	if (Image != 0) {
		is = Image->GetImageSystem();

		str = Image->GetPath();
		if (str != 0) printstring(LoadFileName, Image->GetPath()->GetString());
	}

	if (LoadRequester(handle, LoadFileName, is)) {
		Drop(new StringStream(LoadFileName));
		RefreshImageDisplay();
	}
}

void CbmWindow::SaveImage(bool svas)
{
	int is, exists;
	SP<StringStream> str;

	is = IMAGESYSTEM_NULL;
	if (Image != 0) {

		LoadFileName[0] = 0;
		str = Image->GetPath();
		if (str != 0) printstring(LoadFileName, Image->GetPath()->GetString());

		if ((LoadFileName[0]) && (!svas)) {
			Image->SetPath(new StringStream(LoadFileName));
			Image->SaveFile(new StringStream(LoadFileName), 0);
			RefreshImageDisplay();
		} else {
			is = Image->GetImageSystem();
			if (SaveRequester(handle, LoadFileName, is)) {

				exists = Image->Exists();
				if (exists) {
					exists = MessageBox(handle, "File exists. Overwrite?", "Error", MB_YESNO | MB_ICONQUESTION) != IDYES;
				}

				if (!exists) {
					Image->SetPath(new StringStream(LoadFileName));
					Image->SaveFile(new StringStream(LoadFileName), 0);
					RefreshImageDisplay();
				}
			}
		}
	}
}

void CbmWindow::SaveDir(void)
{
	CbmList *list;
	char *ascii;
	FILE *f;
	int is;

	is = IMAGESYSTEM_NULL;
	if (Image != 0) {
		is = IMAGESYSTEM_NULL;
		if (SaveRequester(handle, DirFileName, is)) {
			list = Image->GetDirectory();
			if (list != 0) {
				ascii = list->GetASCII();
				f = fopen(DirFileName, "wb");
				if (f) {
					fwrite((void*)ascii, 1, stringlen(ascii), f);
					fclose(f);
				}
				delete[] ascii;
				delete[] list;
			}
		}
	}
}

void CbmWindow::Notify(WPARAM wParam, LPARAM lParam)
{
	int i;

	if (((int)wParam) == CM_FILE_NEW) {
		WindowNewImage *wn = new WindowNewImage(hInstance, this);
		wn->Open(handle);
	}

	if (((int)wParam) == CM_FILE_OPEN) OpenImage();

	if (((int)wParam) == CM_FILE_SAVE) SaveImage(false);

	if (((int)wParam) == CM_FILE_SVAS) SaveImage(true);

	if (((int)wParam) == CM_FILE_SVDIR) SaveDir();

	if (((int)wParam) == CM_FILE_LOGDIR) {
		WindowLogDir *wld = new WindowLogDir(hInstance);
		wld->Open(handle);
	}

	if (((int)wParam) == CM_FILE_EXIT) {
		DestroyWindow(handle);
		return;
	}

	if (((int)wParam) == CM_EDIT_DELETE) {
		for (i=(entries-1);i>0;i--) {
			if (selections[i]) Image->ScratchFile(i);
		}
		ClearSelections();
		RefreshImageDisplay();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
	}

	if (((int)wParam) == CM_EDIT_VALIDATE) {
		Image->Validate();
		RefreshImageDisplay();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
	}

	if (((int)wParam) == CM_EDIT_RENAME) {
		unsigned long sel;

		sel = GetFirstSelection();
		if (sel) {
			WindowRename *wr = new WindowRename(hInstance, this, sel, Image);
			wr->Open(handle);
		}
	}

	if (((int)wParam) == CM_EDIT_HEADER) {
		if (Image->HasHeader()) {
			WindowRename *wr = new WindowRename(hInstance, this, 0, Image);
			wr->Open(handle);
		}
	}

	if (((int)wParam) == CM_EDIT_SEPERATOR) {
		Image->AddSeperator();
		ClearSelections();
		RefreshImageDisplay();
//		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
	}

	if (((int)wParam) == CM_EDIT_SETTINGS) {
		WindowConfig *wc = new WindowConfig(hInstance);
		wc->Open(handle);
	}

	if (((int)wParam) == CM_HELP_HELP) {
		WindowHelp *wh = new WindowHelp(hInstance, "Help", HelpText);
		wh->Open(handle);
	}

	if (((int)wParam) == CM_HELP_ABOUT) {
		WindowHelp *wh = new WindowHelp(hInstance, "About", AboutText);
		wh->Open(handle);
	}
}

static void RunExe(HWND handle, SP<StringStream> path, const char *arg, SP<class FileImage> img)
{
	Executable *ex;
	SP<StringStream> imgpath;
	char tmppath[MAX_PATH];
	int err;


	if (path == 0) return;
	if (path->GetSize() == 0) {
		MessageBox(handle, "Could not start emulator.\n\nNo emulator path set!", "Error!", MB_ICONEXCLAMATION | MB_OK);
		return;
	}
	if (img == 0) return;
	GetTempPath(MAX_PATH, tmppath);
	imgpath = new StringStream(tmppath);
	imgpath = adddir(imgpath, "arc64_tmpdir");
	CreateDirectory(imgpath->GetString(), 0);
	imgpath = img->SaveFile(imgpath, 0);

	ex = new Executable(path->GetString());
	if (arg != 0) ex->AddArgument(arg);
	ex->AddArgument(imgpath);
	err = ex->Spawn(0);
	delete ex;

	if (err) {
		MessageBox(handle, "Could not start emulator!", "Error!", MB_ICONEXCLAMATION | MB_OK);
	}
}

void CbmWindow::KeyDown(WPARAM wParam)
{
	int i, sl;

	SP<StringStream> name;

	switch (wParam)
	{
	case (VK_RETURN):
	case (VK_NUMPAD0):
	case ('0'):
		RunExe(handle, Config->GetX64Path(), "-8", Image);
		break;

	case (VK_NUMPAD1):
	case ('1'):
		RunExe(handle, Config->GetX128Path(), "-8", Image);
		break;

	case (VK_NUMPAD2):
	case ('2'):
		RunExe(handle, Config->GetXVICPath(), "-8", Image);
		break;

	case (VK_NUMPAD3):
	case ('3'):
		RunExe(handle, Config->GetXPLUS4Path(), "-8", Image);
		break;

	case (VK_NUMPAD4):
	case ('4'):
		RunExe(handle, Config->GetYAPEPath(), 0, Image);
		break;

	case (VK_NUMPAD5):
	case ('5'):
		RunExe(handle, Config->GetA800WPath(), 0, Image);
		break;

	case (VK_F2):
	case ('R'):
		unsigned long sel;

		sel = GetFirstSelection();
		if (sel) {
			WindowRename *wr = new WindowRename(hInstance, this, sel, Image);
			wr->Open(handle);
		}
		break;

	case ('H'):
		if (Image->HasHeader()) {
			WindowRename *wr = new WindowRename(hInstance, this, 0, Image);
			wr->Open(handle);
		}
		break;

	case (VK_DELETE):
	case ('D'):
		for (i=(entries-1);i>0;i--) {
			if (selections[i]) Image->ScratchFile(i);
		}
		ClearSelections();
		RefreshImageDisplay();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
		break;

	case ('V'):
		Image->Validate();
		RefreshImageDisplay();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());
		break;

//	case (VK_OEM_PLUS):
	case (VK_ADD):
	case ('N'):
		CycleImage(true);
/*		name = nextfilename(Image->GetPath());
		if (name != 0) {
			Drop(name);
		}
		RefreshImageDisplay();
		UpdateWindowHeight();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());*/
		break;

//	case (VK_OEM_MINUS):
	case (VK_SUBTRACT):
	case ('P'):
		CycleImage(false);
/*		name = prevfilename(Image->GetPath());
		if (name != 0) {
			Drop(name);
		}
		RefreshImageDisplay();
		UpdateWindowHeight();
		SetWindowText(handle, getfilename(Image->GetInfoText())->GetString());*/
		break;

	case (VK_HOME):
		SetScrollPos(handle, SB_VERT, 0, true);
		win->Redraw(false);
		break;

	case (VK_PRIOR):
		SetScrollPos(handle, SB_VERT, GetScrollPos()-GetPageHeight(), true);
		win->Redraw(false);
		break;

	case (VK_NEXT):
		SetScrollPos(handle, SB_VERT, GetScrollPos()+GetPageHeight(), true);
		win->Redraw(false);
		break;

	case (VK_DOWN):
		for (i=(entries-2);i>1;i--) {
			if ((!selections[i]) && (selections[i-1]) && (Image != 0)) {
				Image->SwapDirEntries(i, i-1);
				sl = selections[i];
				selections[i] = selections[i-1];
				selections[i-1] = sl;
			}
		}
		SetScrollPos(handle, SB_VERT, GetScrollPos()+16, true);
		win->Redraw(true);
		break;

	case (VK_UP):
		for (i=1;i<(int)(entries-1);i++) {
			if ((!selections[i]) && (selections[i+1]) && (Image != 0)) {
				Image->SwapDirEntries(i, i+1);
				sl = selections[i];
				selections[i] = selections[i+1];
				selections[i+1] = sl;
			}
		}
		SetScrollPos(handle, SB_VERT, GetScrollPos()-16, true);
		win->Redraw(true);
		break;

	case (VK_ESCAPE):
//		DestroyWindow(handle);
		PostQuitMessage(0);
		break;
	}
}

void CbmWindow::Paint()
{
}

void CbmWindow::Size()
{
	RefreshScrollBar();
}

void CbmWindow::Init(HINSTANCE hInst, NetArc *n)
{
	unsigned long i;

	hInstance = hInst;
	handle = 0;
	visible = false;

	net = n;

	DDSurface = 0;
	DDSurfaceAllocated = false;

	entries = 0;
	lastselection = 0;

	zoom = 2;

	bitmapwidth = 0;
	bitmapheight = 0;

	selections = new unsigned char[4096];
	for (i=0;i<4096;i++) selections[i] = 0;

	scrollbar = 105;

	LoadFileName = new char[4096];
	for (i=0;i<4096;i++) LoadFileName[i] = 0;
	DirFileName = new char[4096];
	for (i=0;i<4096;i++) DirFileName[i] = 0;
}

void CbmWindow::Exit()
{
	DragAcceptFiles(handle, false);
	FreeSurface();
	ClosePrimarySurface();
	CloseDirectDraw();

	if (LoadFileName) {
		delete[] LoadFileName;
		LoadFileName = 0;
	}

	if (DirFileName) {
		delete[] DirFileName;
		DirFileName = 0;
	}

	if (selections) {
		delete[] selections;
		selections = 0;
	}

	Image = 0;
}

void CbmWindow::Show(CbmBitmap *bitmap)
{
	WNDCLASSEX wc;
	long x, y;
	char clsname[]="Arc64_MainWindow";
	LARGE_INTEGER aa;

	win = this;

	if (visible) return;
	if (!OpenDirectDraw()) return;
	if (!OpenPrimarySurface()) return;
	AllocSurface(bitmap);
	RenderSurface(bitmap);

	QueryPerformanceCounter(&aa);
	x = (long)aa.QuadPart;
	clsname[6] = (x % 10) | '0';
	x /= 10;
	clsname[7] = (x % 10) | '0';
	x /= 10;
	clsname[8] = (x % 10) | '0';
	x /= 10;
	clsname[9] = (x % 10) | '0';
	x /= 10;

//	sDPrintF("%i %i %i %i\n", GetSystemMetrics(SM_CXICON), GetSystemMetrics(SM_CYICON), GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON));

	wc.cbSize		= sizeof(WNDCLASSEX);
	wc.style		= 0;
	wc.lpfnWndProc	= WinCallBack;//GroupDemoProc;
	wc.cbClsExtra	= 0;
	wc.cbWndExtra	= 0;
	wc.hInstance	= hInstance;
	wc.hIcon		= LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON2));
	wc.hIconSm		= LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
	wc.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground= (HBRUSH)(COLOR_WINDOW);
	wc.lpszMenuName	= NULL;
	wc.lpszClassName= clsname;

	if (!RegisterClassEx(&wc))
	{
		MessageBox(NULL, "Could not register window!", "Error!",
			MB_ICONEXCLAMATION | MB_OK);
		return;
	}

	x = 256;
	y = 256;
	if (bitmap)
	{
		bitmapwidth = bitmap->GetWidth();
		bitmapheight = bitmap->GetHeight();

		x = GetWindowWidth();
		y = GetWindowHeight();
	}

	handle = CreateWindowEx(
		WS_EX_CLIENTEDGE,
		clsname,
		Arc64_DefaultTitle,
		WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT, CW_USEDEFAULT, x, y,
		NULL, NULL, hInstance, NULL);

	if (handle == NULL)
	{
		MessageBox(NULL, "Could not create window!", "Error!",
			MB_ICONEXCLAMATION | MB_OK);
		return;
	}

	ShowWindow(handle, SW_SHOW);
	ResetScrollBar();
	Size();
	RefreshImageDisplay();
	UpdateWindowHeight();
	UpdateWindow(handle);

	DragAcceptFiles(handle, true);
}

void CbmWindow::Show(SP<FileImage> fd)
{
	CbmList *list;
	CbmBitmap *bitmap;

	Image = fd;

	list = fd->GetDirectory();
	if (list != 0) {
		bitmap = list->GetBitmap();
		list->Exit();
		delete[] list;

		entries = bitmap->GetHeight() >> 3;
		Show(bitmap);
		bitmap->Exit();
		delete[] bitmap;
	}
}

bool CbmWindow::SelectAdd(unsigned short sel)
{
	if (selections[sel]) return false;

	selections[sel] = 1;
	return true;
}

bool CbmWindow::SelectExclusive(unsigned short sel)
{
	unsigned short i;
	bool change;

	change = false;
	for (i=0;i<entries;i++)
	{
		if (i == sel)
		{
			if (!selections[i])
			{
				selections[i] = 1;
				change = true;
			}
		}
		else
		{
			if (selections[i])
			{
				selections[i] = 0;
				change = true;
			}
		}
	}
	return change;
}

bool CbmWindow::SelectBunch(unsigned short sel)
{
	unsigned short i, m1, m2;
	bool change;

	m1 = sel;
	m2 = lastselection;

	if (m1 > m2)
	{
		i = m1;
		m1 = m2;
		m2 = i;
	}

	change = false;
	for (i=0;i<m1;i++)
	{
		if ((i <= 0) || (i >= (entries-1))) continue;
		if (selections[i])
		{
			selections[i] = 0;
			change = true;
		}
	}
	for (i=m1;i<=m2;i++)
	{
		if ((i <= 0) || (i >= (entries-1))) continue;
		if (!selections[i])
		{
			selections[i] = 1;
			change = true;
		}
	}
	for (i=m2+1;i<entries;i++)
	{
		if ((i <= 0) || (i >= (entries-1))) continue;
		if (selections[i])
		{
			selections[i] = 0;
			change = true;
		}
	}
	return change;
}

void CbmWindow::Select(unsigned short sel, bool control, bool shift)
{
	unsigned short i;
	bool change;

	sel /= zoom;

	change = false;
	if ((sel > 0) && (sel < (entries-1)))
	{
		if (control)
		{
			change = SelectAdd(sel);
			lastselection = sel;
		}
		else
		{
			if (shift) change = SelectBunch(sel);
			else
			{
				change = SelectExclusive(sel);
				lastselection = sel;
			}
		}
	}
	else
	{
		for (i=0;i<entries;i++)
		{
			if (selections[i])
			{
				selections[i] = 0;
				change = true;
			}
		}
		lastselection = sel;
	}
	if (change) Redraw(true);
}

unsigned long CbmWindow::GetScrollPos()
{
	SCROLLINFO si;

	memset((void *)&si, 0, sizeof(SCROLLINFO));
	si.cbSize = sizeof(SCROLLINFO);
	si.fMask = SIF_POS;
	GetScrollInfo(handle, SB_VERT, &si);
	return si.nPos;
}

void CbmWindow::RefreshScrollBar()
{
	RECT rect;
	SCROLLINFO si;

	GetClientRect(handle, &rect);
	si.cbSize = sizeof(SCROLLINFO);
	si.fMask = SIF_ALL;
	si.nMin = 0;
	si.nMax = bitmapheight*zoom;

	si.nPage = rect.bottom;
	si.nPos = GetScrollPos();
	si.nTrackPos = 0;
	SetScrollInfo(handle, SB_VERT, &si, true);
}

void CbmWindow::ResetScrollBar()
{
	RECT rect;
	SCROLLINFO si;

	GetClientRect(handle, &rect);
	si.cbSize = sizeof(SCROLLINFO);
	si.fMask = SIF_ALL;
	si.nMin = 0;
	si.nMax = bitmapheight*zoom;

	si.nPage = rect.bottom;
	si.nPos = 0;
	si.nTrackPos = 0;
	SetScrollInfo(handle, SB_VERT, &si, true);
}

void CbmWindow::RefreshImageDisplay(void)
{
	Redraw(true);
	ResetScrollBar();
	Redraw(false);
}

void CbmWindow::UpdateWindowHeight(void)
{
	int height, maxheight;
	RECT rect;

	GetWindowRect(handle, &rect);
//	sDPrintF("%i %i %i %i\n", rect.left, rect.top, rect.right, rect.bottom);
	maxheight = GetSystemMetrics(SM_CYSCREEN)-rect.top-48;
	if (maxheight < 48) maxheight = 48;
	height = GetWindowHeight();
	if (height > maxheight) height = maxheight;
	SetWindowPos(handle, 0, 0, 0, GetWindowWidth(), height, SWP_NOMOVE | SWP_NOOWNERZORDER | SWP_NOZORDER);
}

void CbmWindow::NewImage(int type)
{
	LoadFileName[0] = 0;
	if (Image != 0) {
		switch (type)
		{
		default:
		case (IMAGETYPE_D64_170K):
		case (IMAGETYPE_D64_192K):
		case (IMAGETYPE_D64_340K):
			Image = new FileD64(type);
			break;
		case (IMAGETYPE_ATR_90K):
		case (IMAGETYPE_ATR_130K):
		case (IMAGETYPE_ATR_180K):
			Image = new FileATR(type);
			break;
		}
		win->Redraw(true);
	}
	SetWindowText(handle, Arc64_DefaultTitle);
}
