#include "FileLNX.h"
#include "c64file.h"
#include "FileContainer.h"
#include "stringops.h"
#include "StringStream.h"
#include "FileStream.h"
#include "ByteBuffer.h"
#include "SmartPointer.h"
#include "debug.h"

/**************************************************************************************/

struct HeaderLNX
{
public:
	unsigned char name[17];
	unsigned short blocks, last;
	unsigned char type;
};

/**************************************************************************************/

static unsigned short getshortfromchars(const unsigned char *data)
{
	unsigned short i;

	i  = data[1];
	i <<= 8;
	i |= data[0];
	return i;
}

static int ParseLNXEntries(FileStream *fs, HeaderLNX *headers, int count)
{
	int i, k, cnt;
	unsigned char *ndata;
	HeaderLNX *hlnx;
	unsigned char c;

#ifdef _DEBUG
		sDPrintF("Parsing %i HeaderLNX:\n", count);
#endif

	cnt = 0;
	for (i=0;i<count;i++)
	{
		hlnx = &headers[i];
		ndata = fs->NextLine();
		for (k=0;k<16;k++) {
			c = ndata[k];
			if (c == 0) break;
			hlnx->name[k] = c;
		}
		for (k=k;k<16;k++) hlnx->name[k] = 0xA0;
		hlnx->name[16] = 0;
		delete[] ndata;

		// get file block size
		fs->SkipDelimiters();
		if (!fs->HasInteger()) break;
		hlnx->blocks = (unsigned short)fs->ParseInteger();
		fs->SkipDelimiters();
		c = fs->NextByte();
		if (c != 0x0D) break;

		// get filetype character (P, S, U, R...)
		fs->SkipDelimiters();
		c = fs->NextByte();
		if ((c < 'A') || ( c > 'Z')) break;
		hlnx->type = c;
		fs->SkipDelimiters();
		c = fs->NextByte();
		if (c != 0x0D) break;

		// get REL-file side sector size
		if (c == 'R') {
			fs->SkipDelimiters();
			if (!fs->HasInteger()) break;
			fs->ParseInteger();
			fs->SkipDelimiters();
			c = fs->NextByte();
			if (c != 0x0D) break;
		}

		// get file last block size
		fs->SkipDelimiters();
		if (!fs->HasInteger()) {
			// special case: some LNX files have last block size missing on the last file
			// these files will be "opened" in the directory
			if ((cnt+1) == count) {
				hlnx->last = 256;
				cnt++;
			}
			break;
		}
		hlnx->last = (unsigned short)fs->ParseInteger();
		fs->SkipDelimiters();
		c = fs->NextByte();
		if (c != 0x0D) break;

#ifdef _DEBUG
		sDPrintF("Parsed: HeaderLNX(\"%s\", %i, %i, %c)\n", hlnx->name, hlnx->blocks, hlnx->last, hlnx->type);
#endif
		cnt++;
	}

	return (cnt);
}

static void ParseLNXFiles(FileContainer *fc, FileStream *fs, int pos, HeaderLNX *headers, int count)
{
	int i, nsize, rdsize;
	unsigned char type;
	unsigned char *ndata;
	HeaderLNX *hlnx;

	for (i=0;i<count;i++)
	{
		if (fs->AtEnd()) break;

		hlnx = &headers[i];
		if (hlnx->blocks > 0) {
			nsize = ((hlnx->blocks - 1) * 254) + (hlnx->last - 1);
			ndata = new unsigned char[nsize];
			rdsize = fs->ReadData(ndata, pos, nsize);
		} else {
			nsize = 0;
			ndata = 0;
			rdsize = 0;
		}

		switch (hlnx->type)
		{
		case 'D':
			type = 0x80;
			break;
		case 'S':
			type = 0x81;
			break;
		default:
		case 'P':
			type = 0x82;
			break;
		case 'U':
			type = 0x83;
			break;
		case 'R':
			type = 0x84;
			break;
		}
		if (rdsize < nsize) type &= 0x7F;

		fc->Add(new C64File(new ByteBuffer(hlnx->name, 16), ndata, rdsize, type, FILEFORMAT_CBM));

		pos += (hlnx->blocks * 254);
	}
}

class FileContainer *ParseLNX(SP<class StringStream> path, const unsigned char *data, int size)
{
	class FileContainer *fc;
	struct FileStream fs;
	unsigned char *ndata;
	int addr, pos, count, pcount;
	bool test;
	HeaderLNX *headers;
	SP<StringStream> tmpheader;

	if ((size < 254) || (data == 0)) return (0);

	// skip basic header
	addr = getshortfromchars(data);
	pos = getshortfromchars(&data[2]) - addr + 1;
	test = true;
	test &= (addr == 0x0801);
	test &= (pos > 16);
	test &= (pos < 250);
	if (!test) return (0);

	// test basic ending bytes
	test &= (data[pos++] == 0);
	test &= (data[pos++] == 0);
	test &= (data[pos++] == 0);
	test &= (data[pos++] == 0x0D);
	if (!test) return (0);

	fs.Init(data, size);
	fs.SetPos(pos);

	// calculate file offset to data
	fs.SkipDelimiters();
	if (!fs.HasInteger()) {
		fs.Exit();
		return (0);
	}
	pos = fs.ParseInteger() * 254;
	fs.SkipDelimiters();

	// read header
	ndata = fs.NextLine();
	tmpheader = new StringStream((char *)ndata);
	delete[] ndata;

	// read file count
	fs.SkipDelimiters();
	if (!fs.HasInteger()) {
		fs.Exit();
		return (0);
	}
	count = fs.ParseInteger();
	fs.SkipDelimiters();
	fs.SkipLine();

	fc = 0;
	headers = new HeaderLNX[count];
	pcount = ParseLNXEntries(&fs, headers, count);
#ifdef _DEBUG
	if (count != pcount) {
		sDPrintF("LNX parsing error: %i from %i entries parsed!\n", pcount, count);
	}
#endif
	if (pcount > 0) {
		fc = new FileContainer(path);
		fc->SetHeader(new ByteBuffer(tmpheader));
		ParseLNXFiles(fc, &fs, pos, headers, pcount);
	}
	delete[] headers;
	fs.Exit();

	return (fc);
}
