#include "FileT64.h"
#include "c64file.h"
#include "FileContainer.h"

static unsigned char t64magic[20] = "C64 tape image file";

static unsigned char brokent64header[] =
{
/* C64S tape file<CR><LF> */
	0x43, 0x36, 0x34, 0x53, 0x20, 0x74, 0x61, 0x70,
	0x65, 0x20, 0x66, 0x69, 0x6C, 0x65, 0x0D, 0x0A,

/* Demo tap<0x1A>...... */
	0x44, 0x65, 0x6D, 0x6F, 0x20, 0x74, 0x61, 0x70,
	0x65, 0x1A, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E,

/* version $0100 / max entries: 30 / used entries: 0 */
	0x00, 0x01, 0x1E, 0x00, 0x00, 0x00, 0x00, 0x00,

/* "DEMO TAPE               " */	
	0x44, 0x45, 0x4D, 0x4F, 0x20, 0x54, 0x41, 0x50,
	0x45, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,

/* normap tape file, file type ?, $0801-$C3C6, offset $00000400 */
	0x01, 0x44, 0x01, 0x08, 0xC6, 0xC3, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

/* "FILE            " */
	0x46, 0x49, 0x4C, 0x45, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20
};

/**************************************************************************************/

unsigned long getlongfromchars(const unsigned char *data)
{
	unsigned long i;

	i  = data[3];
	i <<= 8;
	i |= data[2];
	i <<= 8;
	i |= data[1];
	i <<= 8;
	i |= data[0];
	return i;
}

static unsigned short getshortfromchars(const unsigned char *data)
{
	unsigned short i;

	i  = data[1];
	i <<= 8;
	i |= data[0];
	return i;
}

class FileContainer *ParseT64(SP<class StringStream> path, const unsigned char *data, int size)
{
	class FileContainer *fc;
	unsigned char flist[32];
	unsigned char *fname;
	unsigned char *fdata;
	unsigned char type;
	int i, cnt, offs, start, end, foffs, fsize;
	bool test, brokent64, brokenname;

	// T64 header min size: 96 bytes
	if ((size < 0x60) || (data == 0)) return (0);

	brokent64 = false;
	brokenname = false;

	// check T64 signature
	test = true;
	for (i=0;i<8;i++) test &= (data[i] == t64magic[i]);

	if (!test) {
		// check T64 signature (C64S)
		test = true;
		for (i=0;i<0x20;i++) test &= (data[i] == brokent64header[i]);

		// many T64 files contain this wrong header information
		if (test) {

			brokent64 = true;
			for (i=0x20;i<0x50;i++) brokent64 &= (data[i] == brokent64header[i]);

			if (brokent64) {
				brokenname = true;
				for (i=0x50;i<0x60;i++) brokenname &= (data[i] == brokent64header[i]);
			}
		}
	}

	// version 0x0100 or 0x0101
	test &= ((data[0x20] & 0xFE) == 0);
	test &= (data[0x21] == 1);

	cnt = getshortfromchars(&data[0x22]);

	// more than 1 file entry must exist
	test &= (cnt > 0);

	// used file entry count should not be higher than existing file entry count
	test &= (cnt >= getshortfromchars(&data[0x24]));

	if (!test) return (0);

	fc = new FileContainer(path);
	fc->SetHeader(new ByteBuffer(&data[0x28], 0x18));

	fname = &flist[16];
	offs = 0x40;
	while (cnt > 0) {

		// get entry
		for (i=0;i<0x20;i++) flist[i] = data[i+offs];
		offs += 0x20;
		cnt--;
		if (offs >= size) break;

		if (flist[0] == 1) {
			start = getshortfromchars(&flist[2]);
			end = getshortfromchars(&flist[4]);
			foffs = getlongfromchars(&flist[8]);
			if (brokent64) end = 0xFFFF;
			fsize = end-start;

			type = flist[1];
			if ((type == 0x01) || (type == 0x44)) type = 0x82;

			if (fsize > (size-foffs)) {
				fsize = (size-foffs);
				type &= 0x7F;		// file open error if file ends early
				if (brokent64) type = 0x82;
			}

			fsize += 2;
			foffs -= 2;
			fdata = new unsigned char[fsize];
			fdata[0] = (unsigned char)(start & 0x00FF);
			fdata[1] = (unsigned char)(start >> 8);
			for (i=2;i<fsize;i++) fdata[i] = data[foffs+i];

			// terminate filenames with 0xA0
			i = 15;
			while ((i>=0) && (fname[i] == 0x20)) {
				fname[i] = 0xA0;
				i--;
			}

			fc->Add(new C64File(new ByteBuffer(fname, 16), fdata, fsize, type, FILEFORMAT_CBM));

			if (brokent64) break;
		}
	}

	return (fc);
}
