; -------------------------------------------------------------------------------------------------------------- ;
; C64 Standard Zeropage / Stack / Vectors
; -------------------------------------------------------------------------------------------------------------- ;
; C64 CPU Port Data Direction Register
; ------------------------------------------------------------------------------------------------------------- ;
          ifnconst D6510    ; 
D6510     = $00             ; 6510 On-Chip I/O Data Direction
          endif             ; 
                            ;   0: Input  = Corresponding bit in R6510 can only be read
                            ;   1: Output = Corresponding bit in R6510 can also be written
                            ;
                            ; default: %00101111 ($2f) = Write Tape Motor Ctrl / Tape Data Out / CHAREN / HIRAM / LORAM
                            ; 
D6510_INI   = [$00 | Tap_MotorOff | Tap_OutInv | Mem_Ini] ; default $2f 
; ------------------------------------------------------------------------------------------------------------- ;
; C64 CPU Port Data Register
; ------------------------------------------------------------------------------------------------------------- ;                
          ifnconst R6510    ; 
R6510     = $01             ; 6510 On-Chip I/O Data
          endif             ; 
                            ; 
R6510_INI   = [$00 | Tap_Ini | Mem_Ini] ; default $37 = Tape Off / No tape data out / IO Kernal BASIC all on
                            ; -----------+-------+--------+-------. 
                            ; 76 543 210 ! $D000 ! $E000  ! $A000 ! 
                            ; nn ttt CHL ! $DFFF ! $FFFF  ! $BFFF ! 
                            ; -----------+-------+--------+-------+ 
                            ; .. ... 000 ! ram   ! ram    ! ram   ! 
                            ; .. ... 001 ! Chars ! ram    ! ram   ! 
                            ; .. ... 010 ! Chars ! Kernal ! ram   ! 
                            ; .. ... 011 ! Chars ! Kernal ! BASIC ! 
                            ; .. ... 100 ! ram   ! ram    ! ram   ! 
                            ; .. ... 101 ! I/O   ! ram    ! ram   ! 
                            ; .. ... 110 ! I/O   ! Kernal ! ram   ! 
                            ; .. ... 111 ! I/O   ! Kernal ! BASIC ! <-- default $07
; ---------------------------------------+-------+--------+-------+-------------------------------------------- ;                
; Bits 02: Select Memory Configuration
; ------------------------------------------------------------------------------------------------------------- ;                
Mem_Ctrl      = %00000111   ; memory config control
Mem_BasOn       = BASIC_ON  ; switch basic on
Mem_BasOff      = BASIC_OFF ; switch basic off
Mem_KerOn       = KERNAL_ON ; switch kernal on
Mem_KerOff      = KERNAL_OFF ; switch kernal off
Mem_IoOn        = IO_ON     ; switch IO on
Mem_IoOff       = IO_OFF    ; switch IO off

Mem_Ini         = [$00 | Mem_IoOn | Mem_KerOn | Mem_BasOn] ; 
                            ; 
LORAM       = %00000001     ; Bit 0: L
BASIC_ON      = %00000001   ;        1=BASIC
BASIC_OFF     = %11111110   ;        0=RAM
HIRAM       = %00000010     ; Bit 1: H  
KERNAL_ON     = %00000010   ;        1=Kernal
KERNAL_OFF    = %11111101   ;        0=RAM --> BASIC will be switched out too as it needs the KERNAL
CHAREN      = %00000100     ; Bit 2: C
IO_ON         = %00000100   ;        1=I/O area (VIC)
IO_OFF        = %11111011   ;        0=Char ROM instead of I/O area
CHAR_ON       = %11111011   ;        0=Char ROM instead of I/O area
; ------------------------------------------------------------------------------------------------------------- ;                
; Tape Control
; ------------------------------------------------------------------------------------------------------------- ;                
Tap_Ctrl      = %00111000   ; tape control
Tap_Ini         = [$00 & Tap_OutIni | Tap_ButtonOff | Tap_MotorOff] ; default $30
                            ; 
Tap_Out       = %00001000   ; Bit 3: Tape Data Output Pulse --> Tape Data In = Bit4 (CIAINTDATAFLG) of CIAICR
                            ;      : 
                            ;      : Digital recording: Generate a PULSE output by inverting Bit3=Tap_Out
                            ;      : 
                            ;      : .Pulse LDA R6510
                            ;      :        EOR #Tap_OutInv
                            ;      :        STA R6510
                            ;      : 
                            ;      : Mesured is the PAUSE between two PULSEs
                            ;      : 0 Bit=$60 - Value Timer B Low (216 cycles) - Value Timer B Hight always $00
                            ;      : 1 Bit=$b0 - Value Timer B Low (326 cycles) - Value Timer B Hight always $00
                            ;      : 
Tap_OutInv      = %00001000 ;      : Used to generate a PULSE
Tap_OutIni      = %11110111 ;      : 
Tap_Button    = %00010000   ; Bit 4: Tape Play Button sense
Tap_ButtonOff   = %00010000 ;      : 1=no button Pressed
Tap_ButtonOn    = %11101111 ;      : 0=one of PLAY/RECORD/FFWD/REW pressed
Tap_Motor     = %00100000   ; Bit 5: Tape Motor Control
Tap_MotorOff    = %00100000 ;      : 1=Switch tape motor off
Tap_MotorOn     = %11011111 ;      : 0=switch tape motor on
; ------------------------------------------------------------------------------------------------------------- ;                
                            ; Bit 6: Not Implemented
                            ; Bit 7: Not Implemented
; ------------------------------------------------------------------------------------------------------------- ;                
; Presets
; ------------------------------------------------------------------------------------------------------------- ;                
All_On        = [$00 | Tap_Ini | Mem_Ini]                         ; %00110111 ($37) -> IO   - Kernal - Basic  - default
All_Off       = [$00 | Tap_Ini & IO_OFF & KERNAL_OFF & BASIC_OFF] ; %00110000 ($30) -> RAM  - RAM    - RAM    - 

B_Off         = [$00 | Tap_Ini | IO_ON  | KERNAL_ON  & BASIC_OFF] ; %00110110 ($36) -> I/O  - Kernal - RAM    - 
K_Off         = [$00 | Tap_Ini | IO_ON  & KERNAL_OFF | BASIC_ON]  ; %00110101 ($35) -> I/O  - RAM    - RAM    - BASIC always off if KERNAL off
KB_Off        = [$00 | Tap_Ini | IO_ON  & KERNAL_OFF & BASIC_OFF] ; %00110100 ($34) -> RAM  - RAM    - RAM    - IO always off if KERNAL and BASIC off
IKB_Off       = [$00 | Tap_Ini & IO_OFF & KERNAL_OFF & BASIC_OFF] ; %00110000 ($30) -> RAM  - RAM    - RAM    - 

I_Off         = [$00 | Tap_Ini & IO_OFF | KERNAL_ON  | BASIC_ON]  ; %00110011 ($33) -> Char - Kernal - BASIC  - Charset ROM with KERNAL and BASIC

IKB_On        = [$00 | Tap_Ini | IO_ON  | KERNAL_ON  | BASIC_ON]  ; %00110111 ($37) -> I/O  - Kernal - BASIC  - 
CBK_On        = [$00 | Tap_Ini & IO_OFF | KERNAL_ON  | BASIC_ON]  ; %00110011 ($33) -> Char - Kernal - BASIC  - Charset ROM with KERNAL and BASIC
CK_On         = [$00 | Tap_Ini & IO_OFF | KERNAL_ON  & BASIC_OFF] ; %00110010 ($32) -> Char - Kernal - RAM    - Charset ROM with KERNAL
C_On          = [$00 | Tap_Ini & IO_OFF & KERNAL_OFF | BASIC_ON]  ; %00110001 ($31) -> Char - RAM    - RAM    - Charset ROM - BASIC always off if KERNAL off
; -------------------------------------------------------------------------------------------------------------- ;
; C64 Zero Page Equates
; -------------------------------------------------------------------------------------------------------------- ;
Unused        set $02   ;          Unused
                        ; 
ADRAY1          = $03   ; - $04    Vector: Convert FAC to Integer in (A/Y) ($B1AA)
ADRAY2          = $05   ; - $06    Vector: Convert Integer in (A/Y) to FAC ($B391)
CHARAC          = $07   ;          Search char for scanning basic text input/temp integer during INT
ENDCHR          = $08   ;          Flag: scan for Quote at end of String
INTEGR          = $07   ; - $08    Temporary integer during OR/AND
TRMPOS          = $09   ;          Cursor column pos before the last TAB or SPC
VERCK           = $0a   ;          Flag: BASIC - Type of load
VERCK_LOAD        = $00 ;                Load
VERCK_VERIFY      = $01 ;                Verify
COUNT           = $0b   ;          Index into the text input buffer/Number of array subscripts
DIMFLG          = $0c   ;          Flag: default Array dimension
VALTYP          = $0d   ;          Flag: type of data
VALTYP_NUM        = $00 ;                Numeric
VALTYP_STR        = $ff ;                String
INTFLG          = $0e   ;          Flag: type of numeric data
INTFLG_FLT        = $00 ;                Floating point
INTFLG_INT        = $80 ;                Integer
GARBFL          = $0f   ;          Flag: LIST quoted string/text line tokenization/garbage collection already tried
SUBFLG          = $10   ;          Flag: subscript reference/user function call used by PTRGET routine
INPFLG          = $11   ;          Flag: Data to get from keyword
INPFLG_INPUT      = $00 ;                INPUT
INPFLG_GET        = $40 ;                GET
INPFLG_READ       = $98 ;                READ
TANSGN          = $12   ;          Flag: TAN sign/SIN sign/comparative result
CHANNL          = $13   ;          File number of current I/O channel ($00 if default keyboard/screen)
LINNUM          = $14   ; - $15    Temp: Integer line number value for GOTO/LIST/ON/GOSUB
TEMPPT          = $16   ;          Pointer: address next available space in the temporary string stack
LASTPT          = $17   ; - $18    Pointer: address last string in the temporary string stack
TEMPST          = $19   ; - $21    descriptor stack for temporary strings
INDEX           = $22   ; - $25    Utility Pointer Area
INDEX1          = $22   ; - $23    First  Utility Pointer
INDEX2          = $24   ; - $25    Second Utility Pointer
RESHO           = $26   ; - $2a    Floating point product of Multiply and Divide
TXTTAB          = $2b   ; - $2c    Pointer: Start of BASIC Text Area ($0801)
VARTAB          = $2d   ; - $2e    Pointer: Start of BASIC Variables
ARYTAB          = $2f   ; - $30    Pointer: Start of BASIC Arrays
STREND          = $31   ; - $32    Pointer: End of BASIC Arrays + 1
FRETOP          = $33   ; - $34    Pointer: Bottom of String space
FRESPC          = $35   ; - $36    Pointer: Utility String
MEMSIZ          = $37   ; - $38    Pointer: Highest Address available to BASIC ($A000)
CURLIN          = $39   ; - $3a    Current BASIC Line number
OLDLIN          = $3b   ; - $3c    Previous BASIC Line number
OLDTXT          = $3d   ; - $3e    Pointer: BASIC Statement for CONT
DATLIN          = $3f   ; - $40    Current DATA Line number
DATPTR          = $41   ; - $42    Pointer: Used by READ - current DATA Item Address
INPPTR          = $43   ; - $44    Pointer: Temporary storage of Pointer during INPUT Routine
VARNAM          = $45   ; - $46    Name of Variable being sought in Variable Table
VARPNT          = $47   ; - $48    Pointer: to value of (VARNAM) if Integer, to descriptor if String
FORPNT          = $49   ; - $4a    Pointer: Index Variable for FOR/NEXT loop
VARTXT          = $4b   ; - $4c    Temporary storage for TXTPTR during READ, INPUT and GET
OPMASK          = $4d   ;          Mask used during FRMEVL
TEMPF3          = $4e   ; - $52    Temporary storage for FLPT value
FOUR6           = $53   ;          Length of String Variable during Garbage collection
JMPER           = $54   ; - $56    Jump Vector used in Function Evaluation - JMP + Address ($4C lo hi)
TEMPF1          = $57   ; - $5b    Temporary storage for FLPT value
TEMPF2          = $5c   ; - $60    Temporary storage for FLPT value
FAC             = $61   ; - $66    Main Floating point Accumulator
FACEXP          = $61   ;          FAC #1: Exponent
FACHO           = $62   ; - $65    FAC #1: Mantissa
FACSGN          = $66   ;          FAC #1: Sign
SGNFLG          = $67   ;          Pointer: Series Evaluation Constant
BITS            = $68   ;          FAC #1: Bit Overflow Area during normalisation Routine
AFAC            = $69   ; - $6e    Auxiliary Floating point Accumulator
ARGEXP          = $69   ;          FAC #2: Exponent
ARGHO           = $6a   ; - $6d    FAC #2: Mantissa
ARGSGN          = $6e   ;          FAC #2: Sign
ARISGN          = $6f   ;          Sign Comparison Result: FAC #1 vs FAC #2
FACOV           = $70   ;          FAC #1: Low-order rounding
FBUFPT          = $71   ; - $72    Pointer: Cassette Buffer
CHRGET          = $73   ; - $8a    Subroutine: Get next Byte of BASIC Text
CHRGOT          = $79   ;          Entry to Get same Byte again
TXTPTR          = $7a   ; - $7b    Pointer: Current Byte of BASIC Text
RNDX            = $8b   ; - $8f    Floating RND Function Seed Value
STATUS          = $90   ;          Kernal: I/O Status Word (ST)
ST_TAP_BLK_SH     = $04 ;          Block is too short (shorter than 192 bytes)
ST_TAP_BLK_LG     = $08 ;          Block is too long  (longer  than 192 bytes)
ST_TAP_ERR_READ   = $10 ;          Not all bytes read with error during pass 1 could be corrected during pass 2
ST_TAP_ERR_VER    = $10 ;          VERIFY error: The file read from the device did not match that in the memory
ST_TAP_CHKSUM     = $20 ;          Checksum error occurred
ST_TAP_EOF        = $40 ;          End of information (file) has been reached
ST_TAP_EOT        = $80 ;          End of Tape has been reached
ST_SER_TODIR      = $01 ;          Transfer direction during which the timeout occured (0=Input /1=Output)
ST_SER_TIMEOUT    = $02 ;          A Timout occurred
ST_SER_ERR_VER    = $10 ;          VERIFY error: The file read from the device did not match that in the memory
ST_SER_EOI        = $40 ;          End of information (file) has been reached
ST_SER_NO_DEV     = $80 ;          Device not present
STKEY           = $91   ;          Flag: STOP key pressed
STKEY_HIT         = $7f ;                STOP key
SVXT            = $92   ;          Timing Constant for Tape
VERCKK          = $93   ;          Flag: KERNEL - Type of load
VERCKK_LOAD       = $00 ;                Load
VERCKK_VERIFY     = $01 ;                Verify
C3PO            = $94   ;          Flag: Serial Bus - Output char buffered
BSOUR           = $95   ;          Buffered char for serial bus
SYNO            = $96   ;          Cassette sync number
TEMPX           = $97   ;          Temp storage of X Register during CHRIN
TEMPY           = $97   ;          Temp storage of Y Register during RS232 fetch
LDTND           = $98   ;          Number of open files / Index to file table
DFLTN           = $99   ;          Default Input  Device ($00=keyboard)
DFLTO           = $9a   ;          Default Output Device ($03=screen)
PRTY            = $9b   ;          Parity of byte output to tape
DPSW            = $9c   ;          Flag: Byte received from Tape
MSGFLG          = $9d   ;          Flag: Type of error messages
MSGFLG_OFF        = $00 ;                Program mode: Suppress Error Messages
MSGFLG_KERN       = $40 ;                Kernal Error Messages only
MSGFLG_FULL       = $80 ;                Direct mode: Full Error Messages
FNMIDX          = $9e   ;          Index to Cassette File name/Header ID for Tape write
PTR1            = $9e   ;          Tape error log pass 1
PTR2            = $9f   ;          Tape error log pass 2
TIME            = $a0   ; - $a2    Real-time jiffy Clock updated 1/60 sec by IRQ in UDTIMK ($F69B)
TSFCNT          = $a3   ;          Bit counter tape READ or WRITE / Flag: Serial bus EOI 
TBTCNT          = $a4   ;          Pulse counter tape READ or WRITE / Serial bus shift counter
CNTDN           = $a5   ;          Tape: Sync count down
TBUFPNT         = $a6   ;          Pointer: Tape I/O buffer
INBIT           = $a7   ;          RS232: Temp for received bit / Tape: Temp
BITC1           = $a8   ;          RS232: Input bit count       / Tape: Temp
RINONE          = $a9   ;          RS232: Flag: Start bit check / Tape: Temp
RIDATA          = $aa   ;          RS232: Input byte buffer     / Tape: Temp
RIPRTY          = $ab   ;          RS232: Input parity          / Tape: Temp
SAL             = $ac   ; - $ad    Pointer: SAVE to serial bus / Tape buffer / Screen scrolling
SAL_LO            = $ac ; 
SAL_HI            = $ad ; 
EAL             =   $ae ; - $af    Pointer: Buffer END address for WRITE/SAVE
EAL_LO            = $ae ; 
EAL_HI            = $af ; 
CMPO            = $b0   ; - $b1    Tape: Timing constants
TAPE1           = $b2   ; - $b3    Pointer: Start address tape buffer ($033C)
BITTS           = $b4   ;          RS232: Write bit count    / Tape: Read - Timing Flag
NXTBIT          = $b5   ;          RS232: Next Bit to send   / Tape: Read - End of Tape
RODATA          = $b6   ;          RS232: Output Byte Buffer / Tape: Read - Error Flag
FNLEN           = $b7   ;          Current File: Name length
LA              = $b8   ;          Current File: Logical file number
SA              = $b9   ;          Current File: Secondary address
FA              = $ba   ;          Current File: First address (Device number)- OPEN LA,FA,SA = OPEN 1,8,15,"I0":CLOSE 1
FNADR           = $bb   ; - $bc    Current File: Address of file name
ROPRTY          = $bd   ;          RS232: Output parity / Tape: Byte to be READ or WRITE
FSBLK           = $be   ;          Tape: READ or WRITE Block Count
MYCH            = $bf   ;          Serial word buffer
CAS1            = $c0   ;          Tape motor switch
STAL            = $c1   ; - $c2    Pointer: Buffer START address for SAVE / Cassette READ/WRITE
STAL_LO           = $c1 ;          
STAL_HI           = $c2 ;          
MEMUSS          = $c3   ; - $c4    Pointer: Buffer START address for LOAD/APPEND
MEMUSS_LO         = $c3 ; 
MEMUSS_HI         = $c4 ; 
LSTX            = $c5   ;          Matrix value of last key pressed (NoKey = $40)
NDX             = $c6   ;          Number of chars in keyboard buffer queue
RVS             = $c7   ;          Flag: Reverse mode
RVS_OFF           = $00 ;                Reverse Off 
RVS_ON            = $01 ;                Reverse On                 
INDX            = $c8   ;          Pointer: End of logical line for INPUT (to suppress trailing spaces)
LXSP            = $c9   ; - $ca    Cursor X/Y (Line/Column) pos at start of INPUT
SFDX            = $cb   ;          Flag: Print shifted chars
BLNSW           = $cc   ;          Flag: Cursor blink 
BLNSW_ENAB        = $00 ;                Enabled
BLNSW_DISAB       = $01 ;                Disabled
BLNCT           = $cd   ;          Timer: Count down for Cursor blink toggle
GDBLN           = $ce   ;          Char under Cursor while cursor is inverted
BLNON           = $cf   ;          Flag: Cursor status
BLNON_OFF         = $00 ;                Off
BLNON_ON          = $01 ;                On
CRSW            = $d0   ;          Flag: Input Origin
CRSW_SRCN         = $03 ;                Input from screen  
CRSW_KEYB         = $00 ;                Input from keyboard
PNT             = $d1   ; - $d2    Pointer: Current screen line address
PNTR            = $d3   ;          Cursor column on current line (including wrap-around line - if any)
QTSW            = $d4   ;          Flag: Editor mode
QTSW_OFF          = $00 ;                Editor NOT in quote mode
QTSW_ON           = $01 ;                Editor     in quote mode
LNMX            = $d5   ;          Current logical line length (39 or 79)
TBLX            = $d6   ;          Current cursor screen line number
SCHAR           = $d7   ;          Screen value of current input character / last character output
INSRT           = $d8   ;          Count: >0 = Number of outstanding insertions
LDTB1           = $d9   ; - $f2    Screen line link table / Editor temp high byte of line screen memory location
CURCOL          = $f3   ; - $f4    Pointer: Current screen colour RAM location
CURCOL_LO       = $f3   ; 
CURCOL_HI       = $f4   ; 
;USER          =   $f3   ; - $f4    Pointer: Current screen colour RAM location
KEYTAB          = $f5   ; - $f6    Vector: Current keyboard decoding table ($EB81)
RIBUF           = $f7   ; - $f8    RS232: Pointer Input  Buffer
ROBUF           = $f9   ; - $fa    RS232: Pointer Output Buffer
FREKZP          = $fb   ; - $fe    Free Zero Page space for User Programs
BASZPT          = $ff   ;          BASIC temp data area
; -------------------------------------------------------------------------------------------------------------- ;
; C64 Stack Equates
; -------------------------------------------------------------------------------------------------------------- ;
ASCWRK        = $00ff   ; - $010a  Assembly Area for Floating point to ASCII conversion
BAD           = $0100   ; - $013e  Tape Input Error log
              ifnconst STACK ; 
STACK         = $0100   ; - $01ff  6510 Hardware Stack Area
              endif     ; 
BSTACK        = $013f   ; - $01ff  BASIC Stack Area
; -------------------------------------------------------------------------------------------------------------- ;
; C64 Misc Equates
; -------------------------------------------------------------------------------------------------------------- ;
BUF           = $0200   ; - $0258  BASIC Input Buffer (Input Line from Screen)
LAT           = $0259   ; - $0262  Kernal Table: Active logical File numbers
FAT           = $0263   ; - $026C  Kernal Table: Active File First Addresses (Device numbers)
SAT           = $026d   ; - $0276  Kernal Table: Active File Secondary Addresses
KEYD          = $0277   ; - $0280  Keyboard Buffer Queue (FIFO)
MEMSTR        = $0281   ; - $0282  Pointer: Bottom of Memory for Operating System ($0800)
MEMEND        = $0283   ; - $0284  Pointer: Top of Memory for Operating System    ($A000)
TIMOUT        = $0285   ;          Serial IEEE Bus timeout defeat Flag
COLOR         = $0286   ;          Current Character Colour code
GDCOL         = $0287   ;          Background Colour under Cursor
HIBASE        = $0288   ;          High Byte of Screen Memory Address ($04)
XMAX          = $0289   ;          Maximum number of Bytes in Keyboard Buffer ($0A)
RPTFLG        = $028a   ;          Flag: Repeat keys
RPTFLG_PGM      = $00   ;                Cursors, INST/DEL & Space repeat
RPTFLG_OFF      = $40   ;                No Keys repeat
RPTFLG_ALL      = $80   ;                All Keys repeat ($00)
KOUNT         = $028b   ;          Repeat Key: Speed Counter ($04)
DELAY         = $028c   ;          Repeat Key: First repeat delay Counter ($10)
SHFLAG        = $028d   ;          Flag: Shift Keys: $00 = None
SHFLAG_SHIFT    = %00000001 ;                Bit 0 = Shift
SHFLAG_CBM      = %00000010 ;                Bit 1 = CBM
SHFLAG_CTRL     = %00000100 ;                Bit 2 = CTRL
LSTSHF        = $028e   ;          Last Shift Key used for debouncing
KEYLOG        = $028f   ; - $0290  Vector: Routine to determine Keyboard table to use based on Shift Key Pattern ($EB48)
MODE          = $0291   ;          Flag: Upper/Lower Case change: $00 = Disabled, $80 = Enabled ($00)
AUTODN        = $0292   ;          Flag: Auto scroll down: $00 = Disabled ($00)
M51CTR        = $0293   ;          RS232 Pseudo 6551 control Register Image
M51CDR        = $0294   ;          RS232 Pseudo 6551 command Register Image
M51AJB        = $0295   ; - $0296  RS232 Non-standard Bits/Second
RSSTAT        = $0297   ;          RS232 Pseudo 6551 Status Register Image
BITNUM        = $0298   ;          RS232 Number of Bits left to send
BAUDOF        = $0299   ; - $029A  RS232 Baud Rate; Full Bit time microseconds
RIDBE         = $029b   ;          RS232 Index to End of Input Buffer
RIDBS         = $029c   ;          RS232 Pointer: High Byte of Address of Input Buffer
RODBS         = $029d   ;          RS232 Pointer: High Byte of Address of Output Buffer
RODBE         = $029e   ;          RS232 Index to End of Output Buffer
IRQTMP        = $029f   ; - $02A0  Temporary store for IRQ Vector during Tape operations
ENABL         = $02a1   ;          RS232 Enables
TODSNS        = $02a2   ;          TOD sense during Tape I/O
TRDTMP        = $02a3   ;          Temporary storage during Tape READ
TD1IRQ        = $02a4   ;          Temporary D1IRQ Indicator during Tape READ
TLNIDX        = $02a5   ;          Temporary for Line Index
TVSFLG        = $02a6   ;          Flag: TV Standard:
TVSFLG_NTSC     = $00   ;                NTSC
TVSFLG_PAL      = $01   ;                PAL
                        ; 
Unused        set $02a7 ; - $02FF  Unused
                        ; 
SPR11         = $02c0   ; - $02FE  Sprite #11 Data Area. (SCREEN + $03F8 + SPR number)
                        ; 
IERROR        = $0300   ; - $0301  Vector: Indirect entry to BASIC Error Message, (X) points to Message ($E38B)
IMAIN         = $0302   ; - $0303  Vector: Indirect entry to BASIC Input Line and Decode      ($A483)
ICRNCH        = $0304   ; - $0305  Vector: Indirect entry to BASIC Tokenise Routine           ($A57C)
IQPLOP        = $0306   ; - $0307  Vector: Indirect entry to BASIC LIST Routine               ($A71A)
IGONE         = $0308   ; - $0309  Vector: Indirect entry to BASIC Character dispatch Routine ($A7E4)
IEVAL         = $030a   ; - $030B  Vector: Indirect entry to BASIC Token evaluation           ($AE86)
SAREG         = $030c   ;          Storage for 6510 Accumulator during SYS
SXREG         = $030d   ;          Storage for 6510 X-Register during SYS
SYREG         = $030e   ;          Storage for 6510 Y-Register during SYS
SPREG         = $030f   ;          Storage for 6510 Status Register during SYS
USRPOK        = $0310   ;          USR Function JMP Instruction ($4C)
USRADD        = $0311   ; - $0312  USR Address ($LB,$MB)
                                  ; 
Unused        set $0313 ;          Unused
; -------------------------------------------------------------------------------------------------------------- ;
; Kernel Indirect Vectors
; -------------------------------------------------------------------------------------------------------------- ;
CINV          = $0314   ;          Vector: Hardware IRQ Interrupt Address
CINV_LO         = $0314 ; 
CINV_HI         = $0315 ; 
CINV_Ini        = $ea31 ; 
; -------------------------------------------------------------------------------------------- ;
CNBINV        = $0316   ;          Vector: BRK Instruction Interrupt Address
CNBINV_LO       = $0316 ; 
CNBINV_HI       = $0317 ; 
CNBINV_Ini      = $fe66 ; 
; -------------------------------------------------------------------------------------------- ;
NMINV         = $0318   ;          Vector: Hardware NMI Interrupt Address
NMINV_LO        = $0318 ; 
NMINV_HI        = $0319 ; 
NMINV_Ini       = $fe47 ; 
; ---------------------------------------------------------------------------------- ;
IOPEN         = $031a   ;          Vector: Indirect entry to Kernal OPEN   Routine
IOPEN_LO        = $031a ; 
IOPEN_HI        = $031b ; 
IOPEN_Ini       = $f34a ; 
; ---------------------------------------------------------------------------------- ;
ICLOSE        = $031c   ;          Vector: Indirect entry to Kernal CLOSE  Routine
ICLOSE_LO       = $031c ; 
ICLOSE_HI       = $031d ; 
ICLOSE_Ini      = $f291 ; 
; ---------------------------------------------------------------------------------- ;
ICHKIN        = $031e   ;          Vector: Indirect entry to Kernal CHKIN  Routine
ICHKIN_LO       = $031e ; 
ICHKIN_HI       = $031f ; 
ICHKIN_Ini      = $f20e ; 
; ---------------------------------------------------------------------------------- ;
ICKOUT        = $0320   ;          Vector: Indirect entry to Kernal CHKOUT Routine
ICKOUT_LO       = $0320 ; 
ICKOUT_HI       = $0321 ; 
ICKOUT_Ini      = $f250 ; 
; ---------------------------------------------------------------------------------- ;
ICLRCH        = $0322   ;          Vector: Indirect entry to Kernal CLRCHN Routine
ICLRCH_LO       = $0322 ; 
ICLRCH_HI       = $0323 ; 
ICLRCH_Ini      = $f333 ; 
; ---------------------------------------------------------------------------------- ;
IBASIN        = $0324   ;          Vector: Indirect entry to Kernal CHRIN  Routine
IBASIN_LO       = $0324 ; 
IBASIN_HI       = $0325 ; 
IBASIN_Ini      = $f157 ; 
; ---------------------------------------------------------------------------------- ;
IBSOUT        = $0326   ;          Vector: Indirect entry to Kernal CHROUT Routine
IBSOUT_LO       = $0326 ; 
IBSOUT_HI       = $0327 ; 
IBSOUT_Ini      = $f1ca ; 
; ---------------------------------------------------------------------------------- ;
ISTOP         = $0328   ;          Vector: Indirect entry to Kernal STOP   Routine
ISTOP_LO        = $0328 ; 
ISTOP_HI        = $0329 ; 
ISTOP_Ini       = $f6ed ; 
; ---------------------------------------------------------------------------------- ;
IGETIN        = $032a   ;          Vector: Indirect entry to Kernal GETIN  Routine
IGETIN_LO       = $032a ; 
IGETIN_HI       = $032b ; 
IGETIN_Ini      = $f13e ; 
; ---------------------------------------------------------------------------------- ;
ICLALL        = $032c   ;          Vector: Indirect entry to Kernal CLALL  Routine
ICLALL_LO       = $032c ; 
ICLALL_HI       = $032d ; 
ICLALL_Ini      = $f32f ; 
; ---------------------------------------------------------------------------------- ;
USRCMD        = $032e   ;          User Defined Vector
USRCMD_LO       = $032e ; 
USRCMD_HI       = $032f ; 
USRCMD_Ini      = $fe66 ; 
; ---------------------------------------------------------------------------------- ;
ILOAD         = $0330   ;          Vector: Indirect entry to Kernal LOAD   Routine
ILOAD_LO        = $0330 ; 
ILOAD_HI        = $0331 ; 
ILOAD_Ini       = $f4a5 ; 
; ---------------------------------------------------------------------------------- ;
ISAVE         = $0332   ;          Vector: Indirect entry to Kernal SAVE   Routine
ISAVE_LO        = $0332 ; 
ISAVE_HI        = $0333 ; 
ISAVE_Ini       = $f5ed ; 
; -------------------------------------------------------------------------------------------- ;
Unused        set $0334 ; - $033b  Unused
                        ; 
TBUFFR        = $033c   ; - $03fb  Tape I/O Buffer
SPR13         = $0340   ; - $037e  Sprite #13
SPR14         = $0380   ; - $03be  Sprite #14
SPR15         = $03c0   ; - $03fe  Sprite #15
                                  
Unused        set $03fc ; - $03ff  Unused
; -------------------------------------------------------------------------------------------- ;
; C64 Standard Screen / Sprite Pointer Equates
; -------------------------------------------------------------------------------------------- ;
VICSCN        = $0400   ; - $07e7  Default: Screen Video Matrix
VICSCNX       = $07e7   ;
VICSCN_LEN      = [VICSCNX - VICSCN] ; 

Unused        set $07e8 ; Unused $0f bytes
UnusedX       set $07f7 ; 
Unused_Len    set [UnusedX - Unused] ; $0f

SPNTRS        = $07f8   ; Default: Sprite Data Pointers
SPNTRSX       = $07ff   ; 
SPNTRS_LEN      = [SPNTRSX - SPNTRS] ; 
SPNTRS_OFF      = [SPNTRS  - VICSCN] ; 
; -------------------------------------------------------------------------------------------------------------- ;
; C64 Standard Character Set Equates
; ------------------------------------------------------------------------------------------------------------- ;                
CHARGEN       = $d000   ; character generator ROM
CHAR_UP         = $d000 ; upper case
CHAR_UPR        = $d400 ; upper case / reversed
CHAR_LO         = $d800 ; lower case
CHAR_LOR        = $dc00 ; lower case / reversed
; ------------------------------------------------------------------------------------------------------------- ;                
