;============================================================
;    Various IRQ routines
;============================================================

;=================================
; Do nothing
;=================================

irq_dummy       asl $d019 ; All done
                jmp $ea81


;=================================
; Top IRQ
; Does basic stuff like play music
;=================================

irq_top         jsr playmusic ; Call music routine
                jsr handle_sounds
                lda #11 ; Restore background color
                sta 53281
                lda irq_counter ; Decrease perpetual counter
                sec
                sbc #1
                sta irq_counter
                lda irq_counter+1
                sbc #0
                sta irq_counter+1
                lda 56320
                and 56321
                ldx 203 ; Supplement joystick with keyboard 
                cpx #60
                bne irqtop1
                and #15
irqtop1         cpx #10
                bne irqtop2
                and #27
irqtop2         cpx #18
                bne irqtop3
                and #23
irqtop3         sta joystick
irq_top_jump    jmp irq_intro_countdown

;==========================
; Plain IRQ end
;==========================

irq_end         asl $d019 ; Clear interrupt flag
                jmp $ea31 ; Jump to complete IRQ standard routine


;==========================
; Continuing IRQ
;==========================

irq_prepare_drawman     lda #<drawman_top
                        ldx #>drawman_top 
                        sta $314    ; Next is top IRQ
                        stx $315
                        lda #180    ; trigger next interrupt at row 180
                        sta $d012
                        asl $d019 ; Clear interrupt flag
                        jmp $ea81 ; Jump to end of IRQ standard routine

;==========================
; Continuing IRQ
;==========================

irq_prepare_diarrhea
                lda #<diarrhea_top
                ldx #>diarrhea_top
                sta $314    ; Next is top IRQ
                stx $315
                lda #170    ; trigger next interrupt at row 170
                sta $d012
                asl $d019 ; Clear interrupt flag
                jmp $ea81 ; Jump to end of IRQ standard routine


;==========================================
; Just wait a few seconds before starting
; the music and raising the curtain
;==========================================
irq_intro_countdown     lda irq_counter+1
                        ora irq_counter
                        bne irqinco1
                        ; Done waiting. Prepare to raise curtain
                        jsr set_curtain_at_bottom
                        jsr setup_curtain_sprites
                        jsr start_intro_music ; Put this back later
                        lda #<irq_curtain_up_intro
                        sta irq_top_jump+1
                        lda #>irq_curtain_up_intro
                        sta irq_top_jump+2

irqinco1                jmp irq_end        



;=======================
; Raise curtain
;=======================

irq_curtain_up_intro
                jsr irq_raise_curtain
                lda curtain_sprite_y
                tay
                and #128
                bne icui1

                lda intro_top_count ; Game start flag already set?
                cmp #2
                bcs icui3

                tya
                and #7
                cmp #2 ; Don't allow for this scanline
                beq icui3

                lda joystick
                and #16
                bne icui3
                ; Prepare to start game
                jsr stop_intro
                lda #<irq_curtain_down_intro ; Change curtain direction to down
                sta irq_top_jump+1
                lda #>irq_curtain_down_intro
                sta irq_top_jump+2
                jmp irq_end

icui3           ; Check for cheat key
                lda 203
                cmp #33
                bne icui4
                sta infinite_lives
icui4
                cpy #38 ; Check if reached top
                bne icui1
                ; Finished raising curtain. Prepare to wait, or start game
                lda intro_top_count
                and #2
                beq icui2
                jsr start_game
                jmp irq_end

icui2           lda #<irq_intro_wait
                sta irq_top_jump+1
                lda #>irq_intro_wait
                sta irq_top_jump+2
                ; 7 seconds should be enough
                lda #50
                sta irq_counter
                lda #2
                sta irq_counter+1

icui1           jmp irq_end

;===========================
; Wait between intro screens
;============================

irq_intro_wait  lda joystick ; Check joystick button
                and #16
                bne irqinwa2
                jsr stop_intro
                jmp irqinwa3
irqinwa2        lda irq_counter+1
                ora irq_counter
                bne irqinwa1
                ; Done waiting. Prepare to lower curtain
irqinwa3        jsr set_curtain_at_top
                lda #<irq_curtain_down_intro
                sta irq_top_jump+1
                lda #>irq_curtain_down_intro
                sta irq_top_jump+2

irqinwa1        jmp irq_end

;=======================
; Lower curtain
;=======================

irq_curtain_down_intro
                lda joystick ; Check joystick
                and #16
                bne icdi3
                jsr stop_intro
icdi3           jsr irq_lower_curtain
                lda curtain_sprite_y
                cmp #144
                bne icdi1
                ; Finished lowering curtain. Raise again

                lda #<irq_curtain_up_intro
                sta irq_top_jump+1
                lda #>irq_curtain_up_intro
                sta irq_top_jump+2
                ldx intro_top_count ; Increase top half screen count number
                inx
                cpx #2
                bne icdi2
                ldx #0
icdi2           stx intro_top_count

icdi1           jmp irq_end


;=================================
; Special curtain down management 
; during Game Over
;=================================

irq_curtain_down_gameover
                jsr irq_lower_curtain
                lda curtain_sprite_y
                cmp #248
                bne icdg1
                ; Finished lowering curtain. Raise again and transition to intro
                jsr start_intro_music
                lda #<irq_curtain_up_intro
                sta irq_top_jump+1
                lda #>irq_curtain_up_intro
                sta irq_top_jump+2
                lda #0
                sta intro_top_count
                tax
                lda #$55 ; Clear fail
icdg2           sta 29979,x
                sta 29987,x
                sta 29995,x
                inx
                cpx #3
                bne icdg2
                jsr set_curtain_at_bottom

icdg1           jmp irq_end


;====================
; Stop intro
;====================

stop_intro      lda intro_top_count
                cmp #2
                beq stoint1
                lda #2 ; Set flag to start game
                sta intro_top_count
                lda #0
                sta music_on
                sta infinite_lives
                ; Randomize
                lda irq_counter
                sta random_pos
                lda #1
                jsr start_sound
stoint1         rts


;==========================
; Show level instructions
;==========================

irq_show_instructions
                lda text_source
                sta 251
                lda text_source+1
                sta 252
                ldy #0
                lda (251),y
                bne shoins1
                ; Reached end. Prepare to wait.

                lda #0
                sta irq_counter
                lda #2
                sta irq_counter+1
                lda #<irq_instructions_wait
                sta irq_top_jump+1
                lda #>irq_instructions_wait
                sta irq_top_jump+2
                jmp irq_prepare_drawman

shoins1         cmp #13 ; Check for carriage return
                bne shoins2
                lda line_dest ; Jump down one line
                clc
                adc #64
                sta line_dest
                sta text_dest
                lda line_dest+1
                adc #1
                sta line_dest+1
                sta text_dest+1

                clc
                lda 251
                adc #1
                sta 251
                lda 252
                adc #0
                sta 252

shoins2         lda text_dest
                sta 253
                lda text_dest+1
                sta 254
                lda 251
                sta text_source
                lda 252
                sta text_source+1
                lda (251),y
                tay
                lda charpos,y ; Empty char?
                beq shoins3

                clc
                adc #<fontpixels
                sta 251
                lda #>fontpixels
                adc #0
                sta 252

                ldy #0
shoins4         lda (251),y
                sta (253),y
                iny
                cpy #5
                bne shoins4

shoins3         lda 253 ; Move cursor 1 step forward
                clc
                adc #8
                sta text_dest
                lda 254
                adc #0
                sta text_dest+1
                lda text_source
                clc
                adc #1
                sta text_source
                lda text_source+1
                adc #0
                sta text_source+1

                jmp irq_prepare_drawman

;=================================
; Just wait a few seconds before
; starting the level
;=================================
irq_instructions_wait   lda irq_counter+1
                        ora irq_counter
                        beq irqinswa2
                        lda joystick
                        and #16
                        bne irqinswa1
                        lda #1
                        jsr start_sound
                        ; Done waiting. Clear screen
irqinswa2               lda #<irq_instructions_clear
                        sta irq_top_jump+1
                        lda #>irq_instructions_clear
                        sta irq_top_jump+2
                        lda #31
                        sta irq_counter

irqinswa1               jmp irq_prepare_drawman 


irq_instructions_clear
                lda irq_counter
                bne irqincl1
                ; All done. Continue to level
                jsr final_level_preparations
                jmp irq_prepare_drawman

irqincl1        and #28
                clc
                lsr
                lsr
                sta 251
                lda #96
                sta 252
                ldx #9
irqincl3        ldy #0
irqincl2        lda #255
                sta (251),y
                tya 
                clc
                adc #8
                tay
                cpy #0
                bne irqincl2
                lda 251
                clc
                adc #64
                sta 251
                lda 252
                adc #1
                sta 252
                dex 
                bne irqincl3
                jmp irq_prepare_drawman


irq_raise_curtain
                ; Move sprite curtain bottom up
                lda curtain_sprite_y
                sec
                sbc #2
                sta curtain_sprite_y
                ldy #0
raicur1         sta 53249,y
                iny
                iny
                cpy #16
                bne raicur1

                ; Check if we should draw background
                cmp #50
                bcc raicur12
                and #7
                cmp #2
                beq raicur13
raicur12        rts
raicur13
                ldx #0
raicur7         lda curtain_pix,x
                sec
                sbc curtain_delta,x
                sta curtain_pix,x
                lda curtain_pix+1,x
                sbc curtain_delta+1,x
                sta curtain_pix+1,x
                inx
                inx
                cpx #6
                bne raicur7

                lda curtain_sprite_y
                cmp #154 ; Check if lower half of screen
                bcc raicur8

                lda stage_data
                sta 253
                lda stage_data+1
                sta 254
                lda curtain_pix
                sta 251
                lda curtain_pix+1
                sta 252
                ldy #0
raicur9         lda (253),y
                sta (251),y
                iny
                bne raicur9
                inc 254
                lda curtain_bgcol
                sta 251
                lda curtain_bgcol+1
                sta 252
                ldy #0
raicur10        lda (253),y
                sta (251),y
                iny
                cpy #32
                bne raicur10
                lda 253
                clc
                adc #32
                sta 253
                lda 254
                adc #0
                sta 254
                lda curtain_fgcol
                sta 251
                lda curtain_fgcol+1
                sta 252
                ldy #0
raicur11        lda (253),y
                sta (251),y
                iny
                cpy #32
                bne raicur11
                lda 253
                clc
                adc #32
                sta stage_data
                lda 254
                adc #0
                sta stage_data+1

                rts

; Upper half of screen
raicur8         
                lda curtain_pix
                sta 251
                lda curtain_pix+1
                sta 252
                ldy #0
                lda #255
raicur4         sta (251),y
                iny
                bne raicur4
                lda curtain_bgcol
                sta 251
                lda curtain_bgcol+1
                sta 252
                ldy #0
                lda #12
raicur5         sta (251),y
                iny
                cpy #32
                bne raicur5
                lda curtain_fgcol
                sta 251
                lda curtain_fgcol+1
                sta 252
                ldy #0
                lda #15
raicur6         sta (251),y
                iny
                cpy #32
                bne raicur6
                ; Get line number
                lda curtain_sprite_y
                sec
                sbc #50
                lsr
                lsr
                lsr
                sta curtain_row
                lda intro_top_count
                bne raicur14
                jsr draw_top_logo ; Screen 0: logo
                rts
raicur14        cmp #1
                bne raicur3
                jsr draw_top_text ; Screen 1: intro text
raicur3         rts



draw_top_logo
                lda curtain_row
                cmp #0
                bne drwlog0
                lda #<logoline1
                sta line_source+1
                lda #>logoline1
                sta line_source+2
                lda #12
                sta line_length+1
                lda #80
                sta text_to+1
                lda #96
                sta text_to+2
                jsr draw_text_line
                rts
drwlog0         cmp #1
                bne drwlog1
                lda #<logoline2
                sta line_source+1
                lda #>logoline2
                sta line_source+2
                lda #19
                sta line_length+1
                lda #112
                sta text_to+1
                lda #97
                sta text_to+2
                jsr draw_text_line
                rts
drwlog1         cmp #11
                bne drwlog2
                lda #<logoline3
                sta line_source+1
                lda #>logoline3
                sta line_source+2
                lda #22
                sta line_length+1
                lda #235
                sta text_to+1
                lda #109
                sta text_to+2
                jsr draw_text_line
                rts
drwlog2         cmp #12
                bne drwlog3
                rts
drwlog3         ; Logo drawn line by line
                lda #<logo_pixels
                sta 251
                lda #>logo_pixels
                sta 252
                lda #152
                sta 253
                lda #98
                sta 254
                ldy curtain_row
drwlog5         cpy #2
                beq drwlog4
                lda 251
                clc
                adc #200
                sta 251
                lda 252
                adc #0
                sta 252
                lda 253
                clc
                adc #64
                sta 253
                lda 254
                adc #1
                sta 254
                dey
                jmp drwlog5
drwlog4         ldy #0
drwlog6         lda (251),y
                sta (253),y
                iny
                cpy #200
                bne drwlog6

                lda #<logo_bgcol
                sta 251
                lda #>logo_bgcol
                sta 252
                lda #83
                sta 253
                lda #92
                sta 254
                ldy curtain_row
drwlog8         cpy #2
                beq drwlog7
                lda 251
                clc
                adc #25
                sta 251
                lda 252
                adc #0
                sta 252
                lda 253
                clc
                adc #40
                sta 253
                lda 254
                adc #0
                sta 254
                dey
                jmp drwlog8
drwlog7         ldy #0
drwlog9         lda (251),y
                sta (253),y
                iny
                cpy #25
                bne drwlog9

                lda #<logo_fgcol
                sta 251
                lda #>logo_fgcol
                sta 252
                lda #83
                sta 253
                lda #216
                sta 254
                ldy curtain_row
drwlog11        cpy #2
                beq drwlog10
                lda 251
                clc
                adc #25
                sta 251
                lda 252
                adc #0
                sta 252
                lda 253
                clc
                adc #40
                sta 253
                lda 254
                adc #0
                sta 254
                dey
                jmp drwlog11
drwlog10        ldy #0
drwlog12        lda (251),y
                sta (253),y
                iny
                cpy #25
                bne drwlog12

                rts

draw_top_text
                ldx curtain_row
                lda introline_high,x
                bne drtote1
                rts
drtote1         sta line_source+2
                lda introline_low,x
                sta line_source+1
                lda #31
                sta line_length+1
                lda curtain_pix
                sta text_to+1
                lda curtain_pix+1
                sta text_to+2
                jsr draw_text_line
                rts

irq_lower_curtain
                ; Move sprite curtain bottom down
                lda curtain_sprite_y
                clc
                adc #2
                sta curtain_sprite_y
                ldy #0
lowcur1         sta 53249,y
                iny
                iny
                cpy #16
                bne lowcur1

                ; Check if we should draw background
                cmp #50
                bcc lowcur2
                and #7
                cmp #2
                beq lowcur3
lowcur2         rts
lowcur3
                lda curtain_pix
                sta 251
                lda curtain_pix+1
                sta 252
                ldy #0
lowcur5         lda curtain_line,y
                sta (251),y
                iny
                bne lowcur5
                lda curtain_bgcol
                sta 251
                lda curtain_bgcol+1
                sta 252
                lda curtain_fgcol
                sta 253
                lda curtain_fgcol+1
                sta 254
                ldy #0
lowcur6         lda #42
                sta (251),y
                lda #0
                sta (253),y
                iny
                cpy #32
                bne lowcur6

                ; Move pointers 1 line down
                ldx #0
lowcur4         lda curtain_pix,x
                clc
                adc curtain_delta,x
                sta curtain_pix,x
                lda curtain_pix+1,x
                adc curtain_delta+1,x
                sta curtain_pix+1,x
                inx
                inx
                cpx #6
                bne lowcur4

                rts

;===============================
; Special curtain management
; for diarrhea levels
;===============================

diarrhea_curtain_down
                jsr irq_lower_curtain
                lda curtain_sprite_y
                cmp #248
                bne diacur1
                ; Finished lowering curtain
                ; Restore control panel
                lda #<ctrl_toilet
                sta from_low+1
                lda #>ctrl_toilet
                sta from_high+1
                lda #64
                sta to_low+1
                lda #118
                sta to_high+1
                lda #8
                sta num_rows+1
                lda #64
                sta bytes_per_row+1
                jsr copy_block
                jsr set_curtain_at_bottom
                ; Check if failed or completed level
                lda game_phase
                cmp #2
                bne diacur4
                ; Prepare for next level after completion
                lda #250
                sta irq_counter
                lda #<stage_change_curtain_wait
                sta irq_top_jump+1
                lda #>stage_change_curtain_wait
                sta irq_top_jump+2
                jsr draw_board
                jmp irq_end 

diacur4         lda fail
                cmp #3
                beq diacur3
                ; Continue level after fail
                lda #<diarrhea_curtain_up
                sta irq_top_jump+1
                lda #>diarrhea_curtain_up
                sta irq_top_jump+2
                jmp irq_end 

diacur3         ; Game over, return to intro
                jsr start_intro_music
                lda #<irq_curtain_up_intro
                sta irq_top_jump+1
                lda #>irq_curtain_up_intro
                sta irq_top_jump+2
                lda #0
                sta intro_top_count
                tax
                lda #$55 ; Clear fail
diacur2         sta 29979,x
                sta 29987,x
                sta 29995,x
                inx
                cpx #3
                bne diacur2
                jmp irq_end

diacur1         lda #<diarrhea_color_change
                ldx #>diarrhea_color_change
                sta $314
                stx $315
                lda diarrhea_y
                sta $d012
                asl $d019 ; Clear interrupt flag
                jmp $ea81 ; Jump to end of IRQ standard routine

diarrhea_curtain_up
                jsr irq_raise_curtain
                lda curtain_sprite_y
                cmp #38
                bne diacurup1
                ; Continue diarrhea level
                lda #0
                sta 53277 ; Clear wide sprite bit
                lda #1
                sta 53286 ; Multicolor register 1 is white
                lda #0
                sta game_phase
                jsr reset_diarrhea_stream
                lda #<loop_diarrhea_top
                sta irq_top_jump+1
                lda #>loop_diarrhea_top
                sta irq_top_jump+2
diacurup1       jmp irq_end


diarrhea_color_change
                lda #9
                sta 53281
                lda #<irq_top
                ldx #>irq_top
                sta $314    ; Next is top IRQ
                stx $315
                lda #0
                sta $d012
                jmp irq_end               


stage_change_curtain_wait
                lda irq_counter
                bne stawai1
                lda #<stage_change_curtain_up
                sta irq_top_jump+1
                lda #>stage_change_curtain_up
                sta irq_top_jump+2
stawai1         jmp irq_end 


stage_change_curtain_up
                jsr irq_raise_curtain
                lda curtain_sprite_y
                cmp #38
                bne stacha1
                ; Continue next level
                lda #0
                sta 53277 ; Clear wide sprite bit
                lda #1
                sta 53286 ; Multicolor register 1 is white
                lda #0
                sta game_phase
                jsr prepare_level
stacha1         jmp irq_end



;========================
; Starts a sound effect
; and possibly overwrites
; the one with the lowest
; counter
;========================

start_sound     ldx #0
                ldy sound_counter
                cpy sound_counter+1
                bcc stasou1
                ldx #1
                ldy sound_counter+1
stasou1         cpy sound_counter+2
                bcc stasou2
                ldx #2
stasou2         sta sound_effect,x
                tay
                lda snd_duration,y
                sta sound_counter,x
                lda voice_offsets,x
                tax
                lda snd_wave,y
                sta 54276,x
                lda snd_ad,y
                sta 54277,x
                lda snd_sr,y
                sta 54278,x
                lda #0
                sta 54272,x ; Clear frequency
                sta 54273,x
                lda snd_wave,y
                ora #1
                sta 54276,x
                rts

;========================
; Stops the sound effect
; of the type given in
; the accumulator
;========================

stop_sound      cmp sound_effect
                bne stosou1
                lda #0
                sta sound_effect
                sta sound_counter
                sta 54276
                rts
stosou1         cmp sound_effect+1
                bne stosou2
                lda #0
                sta sound_effect+1
                sta sound_counter+1
                sta 54283
                rts
stosou2         cmp sound_effect+2
                bne stosou3
                lda #0
                sta sound_effect+2
                sta sound_counter+2
                sta 54290
stosou3         rts 

;===========================
; Called at death or wave
; completion
;===========================

stop_all_sounds
                lda #0
                sta sound_effect
                sta sound_effect+1
                sta sound_effect+2
                sta 54276
                sta 54283
                sta 54290
                rts

;===========================
; Play sounds, called
; every frame
;===========================

handle_sounds   ldx #0
hansou2         lda sound_effect,x
                beq hansou1
                tay
                lda snd_jumptable_low,y
                sta hansou3+1
                lda snd_jumptable_high,y
                sta hansou3+2
                lda voice_offsets,x
                tay
                dec sound_counter,x
                lda sound_counter,x
                bne hansou3 ; Counter reached zero?
                sta sound_effect,x
hansou3         jsr sound1

hansou1         inx
                cpx #3
                bne hansou2
                rts

sound1          asl ; Simple frequency drop
                sta 54273,y
                rts
sound2          lda #80 ; Step
                sta 54273,y
                rts
sound3          asl ; Splish
                eor #63
                sta 54273,y
                rts
sound4          and #3 ; Expunge
                asl
                asl
                asl
                adc #40
                sta 54273,y
                rts
sound5          lda #100 ; Ding
                sta 54273,y
                rts
sound6          txa ; Empty bucket, landing shit, drowning
                pha
                inc random_pos
                ldx random_pos
                lda random_table,x
                and #15
                adc #10
                sta 54273,y
                pla 
                tax
                rts
sound7          lda #10 ; Beep
                sta 54273,y
                rts
sound8          lda #30 ; Pew
                sta 54273,y
                rts
sound9          and #7 ; Bullet hit
                asl
                asl
                asl
                adc #8
                sta 54273,y
                rts
sound10         txa ; Diarrhea splash
                pha
                inc random_pos
                ldx random_pos
                lda random_table,x
                and #31
                adc #10
                sta 54273,y
                pla 
                tax
                rts
