;
; chiller 2 - the return of the jackal
;
; released by cosine systems in 2020
;
; cosine.org.uk
;
; code, graphics and music by
; andy 'aNdy' vaisey (CSDb id=6404)
;
; help and support by
; jason 'T.M.R' kelk (CSDb id=987)
;
; playtesting by
; jonathan 'moloch' mines (CSDb id=930)
; ryley 'w0rm' james (CSDb id=32190)
;
; special thanks to
; vinny mainolfi
; andrew fisher
; sailor/triad
;
; kindly sponsored by FREEZE64.com fanzine
;
; questions about this code or about how the binaries
; were created can be asked by emailing aNdy...
;
; andy@cosine.org.uk
;
; this source is formatted for the acme cross assembler
; which can be downloaded from...
;
; http://sourceforge.net/projects/acme-crossass/
;
; the final build was compressed and squished using
; exomizer 2 which can be downloaded from...
;
; http://hem.bredband.net/magli143/exo/

; the name and file format for the assembled program

		!to "chiller2.prg",cbm

; assimilate the binary data, resistance is futile...

		* = $0900
music	!binary "data\music.prg",,2

		* = $2000
		!binary "data\charset.bin"

		* = $2800
		!binary "data\sprites.bin"
		
; position of the raster interrupt(s) on-screen
; 'rp' is short for 'raster position'! duh!

raster_1_pos		= $00				; very top of screen and colour hack for ntsc
raster_2_pos		= $0f				; changes top border colour for screens
raster_3_pos		= $32				; start of main screen
raster_4_pos		= $5f				; title screen split for sprite recycling
raster_5_pos		= $f9				; bottom split to open borders

; label assignments for joystick and keyboard control

cpra				= $dc00				; cia #1, port register a
cprb				= $dc01				; cia #1, port register b

; some more label assignments (zp variables)

raster_num			= $50			; raster counter stores which split we are working in
sync				= $51			; interrupt sync variable

colourwash_count	= $53			; colour pulse counter for wash effect
colourwash_timer	= $54			; colour pulse timer for wash effect

scroll_x			= $55			; title scroller screen position
scroll_pos			= $56			; titles scroller position - $02 bytes used

ply_type			= $58			; store for player select - boy (1) or girl (0)
anim_timer			= $59			; animation timer for game sprites
title_mode	 		= $5a			; is titles mode enabled?
bounce_timer		= $5b			; bounce timer for title screen logo top sprites
anim_timer_b		= $5c			; animation timer for title screen bottom sprites
last_level			= $5d			; last level reached variable
level_num			= $5e			; level number variable

ply_move_flag		= $5f			; player move flag - player is moving or not!
ply_move_timer		= $60			; player move timer - delay x movement so visible!
ply_strength		= $61			; girl or boy strength variable
ply_jumppwr			= $62			; girl or boy jump power variable
ply_colls_flag		= $63			; flag for player death initiation! WHAAAA!
colls_timer			= $64			; timer for player -> enemy collision
ply_energy			= $65			; player energy store
coll_temp			= $66			; collision workspace - $04 bytes used
gravity_strength	= $6a			; gravity strength store
gravity_timer		= $6b			; gravity timer store
gravity_on			= $6c			; gravity on/off flag
bkgd_coll_timer		= $6d			; timer for player -> background collisions

level_comp_flag		= $6e			; flag for level complete
cross_num			= $6f			; crosses collected store

bkgd_timer			= $70			; animation timer for background chars
bkgd_lvl_tmr		= $71			; anim 'speed' for indiv level background anim 
enemy_timer			= $72			; timer for enemy movement - slow them!
raindrop			= $73			; raindrop counter
cinescrn			= $74			; cinema screen counter
snowstrip			= $75			; snow strip store

tbordblue			= $76			; flag for black or blue top border in-game
video_type			= $77			; 0=pal, 1=ntsc
music_tmr			= $78			; timer for music frame skip on ntsc
intro_mode			= $79			; additional flag for intro mode at start

; add a BASIC startline (SYS 16384) for auto-run after loading

		* = $0801
		!word code_entry-2
		!byte $00,$00,$9e
		!text "16384"
		!byte $00,$00,$00

; entry point for the code

		* = $4000					; hence sys 16384!
				
code_entry

; stop interrupts, disable the roms and set up nmi and irq interrupt pointers	

		sei							; poke the interrupts in the eye temporarily
				
		lda #<nmi					; some nmi interrupt stuff
		sta $fffa
		lda #>nmi
		sta $fffb

		lda #<int					; some irq stuff
		sta $fffe
		lda #>int
		sta $ffff
		
		ldx #$00
		lda #$00
clr_mem								; clear out memory between $0200 and $02ff	
		sta $0200,x					; not doing this makes the game bug slighty
		inx							; when using the fast loader on an action
		bne clr_mem					; replay cartridge.  dunno why it bugs....		

		lda #$35					; bin the roms and kernal, we're flying solo
		sta $01		
		
		lda #$7f					; disable timer interrupts which can be generated by the two CIA chips
		sta $dc0d					; the kernal uses such an interrupt to flash the cursor and scan
		sta $dd0d					; the keyboard, so we better stop it

		lda $dc0d					; by reading these two registers we kill any pending CIA irqs.
		lda $dd0d					; if we don't do this, a pending CIA irq might occur after we
									; finish setting up our irq and we don't want that to happen do we!?

		lda #raster_1_pos			; starting point of raster on screen ($00)
		sta $d012

		lda #$1b					; as there are more than 256 rasterlines, the topmost bit of $d011 serves as
		sta $d011					; the 9th bit for the rasterline we want our irq to be triggered.
		
		lda #$01					; acknowledge any interrupts
		sta $d019					; clear interrupt bit
		sta $d01a					; tell vic-ii to generate interrupt

		lda #$03					; select video bank
		sta $dd00		
						
		lda #$18					; use our custom charset at $2000
		sta $d018

		cli							; unleash those interrupts again

; detect what machine type is being used...

start_scan
		lda $d012
chkntsc
		cmp $d012
		beq chkntsc
		bmi start_scan
		adc #$00
		lsr
		bcs isntsc
		lsr
		bcs isoldntsc
		ldx #$00					; check drean or pal
		lda #$10
chkpal
		inx
		cmp $d012
		bne chkpal
		cpx #$70
		bcc ispal
isdrean
		lda #$01
		sta video_type				; video = ntsc
		jmp detect_dun
isntsc	
		lda #$01
		sta video_type				; video = ntsc
		jmp detect_dun
isoldntsc
		lda #$01
		sta video_type				; video = ntsc
		jmp detect_dun
ispal
		lda #$00
		sta video_type				; video = pal

detect_dun	
				
; ready the irq code for the first run
		
		lda #$01					; set irq code handler to 1
		sta raster_num				; for top of screen
		
; set up the intro screen

		lda #$01					; set to
		sta title_mode				; title screen mode!
		sta intro_mode				; hack on the intro mode!
		
        lda #$00					; load 0 for black
		sta $d021					; store to background! change colour!
		sta $d020					; store to border! change colour!		
		
		jsr clear_screen			; clear screen with subroutine
		
; prepare the intro sound, which is a blank tune!

		lda #$0a					; ready a blank tune
		jsr music+$00				; insert into music driver
		
; print intro message for either pal or ntsc and 8580 sid

		ldx #$00
in_text_loop
		lda video_type
		cmp #$00
		bne print_ntsc

		lda intro_1,x				; this is a pal machine?
		sta $05b8,x					; print the pal message!
		lda #$02
		sta $d9b8,x
		
		jmp print_8580
		
print_ntsc							; this is an ntsc machine?
		lda intro_2,x				; print the ntsc message!
		sta $05b8,x
		lda #$02
		sta $d9b8,x		

print_8580							; the music sounds best on		
		lda intro_3,x				; an 8580 sid machine
		sta $0608,x					; so tell the player!
		lda #$02
		sta $da08,x					
		inx
		cpx #$28
		bne in_text_loop

; ready the sprites for the intro 'cosine' logo	
				
		ldx #$00
i_sprite_set						; read the definitions,
		lda i_spr_x_dflt,x			; colours, positions, etc
		sta sprite_x,x				; of the cosine logo
		lda i_spr_y_dflt,x			; sprites used in the 'intro'
		sta sprite_y,x				; and pass them to the main
		lda i_spr_col_dflt,x		; sprite table so the irq's
		sta sprite_col,x			; can print them on the screen
		lda i_spr_def_dflt,x
		sta sprite_def,x
		lda i_spr_anims_dflt,x
		sta sprite_anims,x
		lda i_spr_anime_dflt,x
		sta sprite_anime,x		
		inx
		cpx #$08
		bne i_sprite_set		
		
; give player time to read the intro	
						
		ldy #$ff					; wait a bit...
		jsr sync_wait_long				
		
		lda #$00					; switch the hack for
		sta intro_mode				; intro mode off		

; set up some labels for very first run of the actual game

		sta ply_type				; make player type 0, which is the girl

		sta music_tmr				; zero the ntsc music timer
		
		lda #$09					; girls strength
		sta ply_strength			; is 9 to start
		
		lda #$fe					; the girl's jump
		sta ply_jumppwr				; distance power!
		
		lda #$01					; last level reached should
		sta last_level				; be '1' on very first run!

; now for the actual game itself...					
		
; set up and display the title screen
	
title_setup

		lda #$00					; set code to
		sta title_mode				; title screen mode and
		sta intro_mode				; intro mode off!
									; and set the flag for ntsc		
		sta tbordblue				; black top border hack!		

		jsr clear_screen			; clear screen with subroutine
		
; ready the sprites for the bouncing title logo 'chiller 2'
		
		ldx #$00
t_sprite_set1
		lda t_spr_x_dflt,x			; call up the definitions,
		sta sprite_x,x				; colours, positions, etc
		lda t_spr_y_dflt,x			; of the title screen sprites
		sta sprite_y,x				; and pass them to the main
		lda t_spr_col_dflt,x		; sprite table so the irq's
		sta sprite_col,x			; can dump them on the screen
		lda t_spr_def_dflt,x
		sta sprite_def,x
		lda t_spr_anims_dflt,x
		sta sprite_anims,x
		lda t_spr_anime_dflt,x
		sta sprite_anime,x		
		inx
		cpx #$08
		bne t_sprite_set1

; then ready the sprites for the title screen bottom sprites
; for the boy, girl and ghost...
			
		ldx #$00
t_sprite_set2
		lda t_spr_x_dflt2,x			; same as above!
		sta t_sprite_x,x
		lda t_spr_y_dflt2,x
		sta t_sprite_y,x
		lda t_spr_col_dflt2,x
		sta t_sprite_col,x
		lda t_spr_def_dflt2,x
		sta t_sprite_def,x
		lda t_spr_dflt2_anims,x
		sta t_sprite_anims,x
		lda t_spr_dflt2_anime,x
		sta t_sprite_anime,x
		inx
		cpx #$08
		bne t_sprite_set2
		
; print the title screen text and layout to screen ram		
		
		ldx #$00					; zero x register
t_scrnram_loop						; load up whatver is in the title screen
		lda t_screen_ram,x			; ram table and write to the
		sta $0400,x					; actual screen ram
		lda t_screen_ram+250,x
		sta $0400+250,x
		lda t_screen_ram+500,x
		sta $0400+500,x
		lda t_screen_ram+750,x
		sta $0400+750,x	
		inx
		cpx #$fa					; done 250 writes?
		bne t_scrnram_loop			; all written? no? loop back!

; print the compressed title screen colour data to colour ram
			
		ldx #$00					; zero x register
		ldy #$00					; zero y register
t_colram_loop
		lda t_colour_ram,x			; load up a byte
		sta $d800+1,y				; store a colour nybble to screen
		lsr							; bit shift to the right
		lsr							; 4 times so we can read
		lsr							; the other colour nybble
		lsr							; stored in the byte then
		sta $d800,y					; store the other nybble!

		lda t_colour_ram+115,x
		sta $d800+231,y
		lsr
		lsr
		lsr
		lsr
		sta $d800+230,y

		lda t_colour_ram+230,x
		sta $d800+461,y
		lsr
		lsr
		lsr
		lsr
		sta $d800+460,y

		lda t_colour_ram+345,x
		sta $d800+691,y
		lsr
		lsr
		lsr
		lsr
		sta $d800+690,y

		iny
		iny
		inx
		cpx #$73
		bne t_colram_loop								
		
; print previous score and high score onto screen

		ldx #$00
score_print_t
		lda score,x					; load up what's in the
		ora #$30					; score table and
		sta $07a8,x					; print it on screen!
		lda high_score,x			; then the same for the
		ora #$30					; high score table!
		sta $07ba,x		
		inx
		cpx #$06
		bne score_print_t
		
; reset the message scroller

		jsr scroll_reset			; reset 'scroller' subroutine
		
		lda #$50
		sta scroll_x

; prepare the title music	

		lda #$00					; ready the title music tune
		jsr music+$00				; throw it at the music driver!
		
; set flag for title screen mode to 1 (on)
		
		lda #$01
		sta title_mode
		
; which player is selected? colour word and prep dancing sprites!

		lda ply_type				; which character is selected?
		cmp #$00					; girl?
		beq prep_girl				; yes? then branch to prep_girl
									; no?
		jsr boy_select				; then prep the boy!
		
		jmp title_loop				; jump to title screen loop below!
		
prep_girl
		jsr girl_select		
		
; everything for the title screen is now on-screen
; so 'lock' the main title screen in a loop until
; the player presses a key or inputs the joystick

title_loop

		jsr sync_wait				; wait until irq finishes!

; check if 'f7' is pressed for continue from last level feature

		lda #$fe 					; load $dc00...
		sta cpra					; ...with bit row 0
		
		lda cprb					; load $dc01 to test...
		cmp #$f7					; key 'f7' pressed?
		bne ps_left					; no? check joystick...

; press 'f7' is detected
				
		jsr key_debounce			; debounce keyboard
		
		lda last_level				; load last level played...
		sta level_num				; ...into current level!

		jmp	kill_title				; then start game init!
					
; no 'f7' press detected, so check if player uses joystick...

ps_left
		lda cpra					; load 'joystick' port 2 register
		and #$04					; pull left bit activated?
		bne ps_right				; nope? test right!
									; yes?
		jsr girl_select				; select girl subroutine!
		
ps_right							; etc
		lda cpra
		and #$08
		bne ps_fire
		
		jsr boy_select
	
ps_fire	
		lda cpra
		and #$10
		bne title_loop				; nothing detected, restart title loop!
									; make sure player isn't holding
		jsr fire_debounce			; down the fire button!
		
; fire button pressed for brand new game, set the level number to '1'!
				
		lda #$01
		sta level_num

; kill title screen mode

kill_title
		lda #$00					; switch off the ghost and girl/boy
		sta title_mode				; sprites on the title screen

; reset the score to '000000'
				
		ldx #$00
		lda #$00					
score_reset				
		sta score,x					; score table set to '000000'
		inx
		cpx #$06
		bne score_reset
		
; initialise the get ready sequence	and
; set-up the next level behind the scenes
				
get_ready_setup

		lda #$00					; set the flag for
		sta tbordblue				; a black top border!
	
		jsr hide_sprites			; move the sprites off-screen!
		
		jsr clear_screen			; clear the screen

		lda #$01					; the get ready tune (1)
 		jsr music+$00				; init the music driver
 		
		lda #$21					; give player full energy
		sta ply_energy				; at the start of each level!	
		
		lda level_num				; load the level number
	
		cmp #$02					; is it level 2?
		bne *+$08					; no? check for level 3
		jsr level2_setup_subrout	; otherwise setup l2!								
		jmp get_ready_display		; jump to get ready!
		
		cmp #$03					; is it level 3?
		bne *+$08					; no? check for level 4
		jsr level3_setup_subrout	; otherwise setup l3!									
		jmp get_ready_display		; jump to get ready!
		
		cmp #$04					; level 4
		bne *+$08					; etc!
		jsr level4_setup_subrout								
		jmp get_ready_display

		cmp #$05
		bne *+$08
		jsr level5_setup_subrout								
		jmp get_ready_display
		
		cmp #$06
		bne *+$08
		jsr level6_setup_subrout								
		jmp get_ready_display
		
		cmp #$07
		bne *+$08
		jsr level7_setup_subrout								
		jmp get_ready_display
		
		cmp #$08
		bne *+$08
		jsr level8_setup_subrout								
		jmp get_ready_display
		
		cmp #$09
		bne *+$08
		jsr level9_setup_subrout								
		jmp get_ready_display
		
		cmp #$0a
		bne *+$08
		jsr level10_setup_subrout								
		jmp get_ready_display
		
		cmp #$0b
		bne *+$08
		jsr level11_setup_subrout								
		jmp get_ready_display
		
		cmp #$0c
		bne *+$08
		jsr level12_setup_subrout								
		jmp get_ready_display													

		jsr level1_setup_subrout	; none of the above? setup l1 then!!!								

; next level has been prepared, let's get ready for the game!	
		
get_ready_display	

; print level info for get ready screen

		ldx #$00
gr_text_loop
		lda get_ready_des1,x		; print level number / title
		sta $0590,x
		lda #$02					; make that text red
		sta $d990,x
		lda get_ready_des2,x		; print rhyme couplet first line
		sta $05e0,x
		lda #$06					; make that text blue
		sta $d9e0,x
		lda get_ready_des3,x		; print rhyme couplet second line
		sta $0608,x
		lda #$06					; make that text blue
		sta $da08,x					
		inx
		cpx #$28
		bne gr_text_loop

; give player time to read the level description!		
						
		ldy #$ff					; wait a bit...
		jsr sync_wait_long

; initialise the in-game music based on the level		
		
		jsr lmusic_setup
		
; set top border colour depending on level number...

		lda level_num				; which level are we on?
		
		cmp #$03					; level 3?
		bne *+$05					; no? branch to next check
		jmp set_bluebord			; yes? set blue border flag

		cmp #$07					; level 7?
		bne *+$05					; no? branch to next check
		jmp set_bluebord			; yes? set blue border flag
	
		cmp #$0c					; level 12?
		bne *+$05					; no? branch to next check
		jmp set_bluebord			; yes? set blue border flag					
		
		lda #$00					; otherwise set the flag for
		sta tbordblue				; a black top border!
		
		jmp print_gamescrn
		
set_bluebord		
		lda #$01					; set flag for a blue border
		sta tbordblue				; for levels 3 and 7!
				
; now print the level layout!

print_gamescrn						; the screen chars and colour!
		ldx #$00					; zero x register
l_scrnram_loop						; load up whatver is in the screen
		lda l_screen_ram,x			; ram table and write to the
		sta $0400,x					; screen ram
		lda l_screen_ram+230,x
		sta $0400+230,x
		lda l_screen_ram+460,x
		sta $0400+460,x
		lda l_screen_ram+690,x
		sta $0400+690,x	
		inx
		cpx #$e6						; done 250 writes?
		bne l_scrnram_loop				; all written? no? loop back!
										; yes? then...
		ldx #$00						; zero x register
l_colram_loop							; load whatever is in the colour
		lda l_colour_ram,x				; ram table and write to the
		sta $d800,x						; colour ram
		lda l_colour_ram+230,x
		sta $d800+230,x	
		lda l_colour_ram+460,x
		sta $d800+460,x	
		lda l_colour_ram+690,x
		sta $d800+690,x	
		inx
		cpx #$e6						; done 250 writes?
		bne l_colram_loop				; all written? no? loop back!

; print up the status bar

		jsr status_bar					; use subroutine!
		
; update the energy bar to correct level		
				
		jsr energy_bar					; use subroutine!
		
; get those game sprites printed on the screen!	
		
		ldx #$00
game_sprite_set
		lda sprite_x_buf,x
		sta sprite_x,x
		lda sprite_y_buf,x
		sta sprite_y,x
		lda sprite_col_buf,x
		sta sprite_col,x
		lda sprite_def_buf,x
		sta sprite_def,x
		lda sprite_anims_buf,x
		sta sprite_anims,x
		lda sprite_anime_buf,x
		sta sprite_anime,x
		lda sprite_dir_buf,x
		sta sprite_dir,x
		lda sprite_spd_buf,x
		sta sprite_spd,x
		lda sprite_min_buf,x
		sta sprite_min,x
		lda sprite_max_buf,x			
		sta sprite_max,x	
		inx
		cpx #$08
		bne game_sprite_set

; reset cross counter to 0

		ldx #$00
		lda #$00							
cross_reset								; reset the cross counter					
		sta cross_score,x				; to 00
		inx
		cpx #$02
		bne cross_reset
		
		sta cross_num					; reset cross variable to 0			
		
; print the status bar scores/counters		
		
		jsr update_status				; use subroutine!
		
; reset the in-game gravity

		lda #$00						; load accumalator with '0'
						
		sta gravity_strength			; grav strength to 0
		sta gravity_timer				; grav timer to 0
		sta gravity_on					; gravity_on is 0 (off!)
		
; reset some timers and counters

		sta colls_timer					; sprite collision timer to 0
		sta bkgd_coll_timer				; background collision timer to 0
		sta bkgd_timer					; background animation timer to 0
		sta enemy_timer					; enemy movement timer to 0
		sta raindrop					; raindrop anim counter to 0
		sta cinescrn					; cinema screen anim counter to 0
		sta snowstrip					; snow anim counter to 0
	
; reset level complete flag to 0!		
			
		sta level_comp_flag		
	
; now begin main game loop...		
		
main_game_loop

		jsr sync_wait					; wait for screen sync
		jsr player_move					; check joystick and move player
		jsr player_collision			; check for sprite collisions
		
		lda ply_colls_flag				; check if collision flag is 2
		cmp #$02						; if so, energy has run out so
		beq death_init					; begin death sequence
		
		cmp #$01						; check if collision flag is 1
		bne process_rest				; if not, branch down to 'process_rest'
										; if it is 1, carry on to code below...
		jsr score_accrue_100			; give the player 100 points!
		jsr cross_accrue				; add a cross to cross counter
		jsr update_status				; update status bar (add a cross!)
		
		lda #<sfx_collect				; start address of 'collect' sfx
        ldy #>sfx_collect				; end address of 'collect' sfx
        ldx #$0e						; play sfx on channel 3 please!
        jsr music+$06					; call sfx driver to play sound
		
		lda level_comp_flag				; load up level complete flag
		cmp #$01						; is it '1'?
		bne process_rest				; no? branch to 'process_rest'
										; yes?
		jmp level_complete				; jump to level complete sequence!
				
process_rest
		jsr process_enemies				; update enemy positions
		
		jsr process_bkgd_anims			; animate the background for current level
		
		jmp main_game_loop				; jump back to start of main game loop!

; energy is gone, start death sequence! 			
		
death_init

		lda #$00						; set the flag for
		sta tbordblue					; a black top border!

		lda #$03						; game over tune number is '3'
 		jsr music+$00					; init the music driver!
 		
		jsr hide_sprites				; hide the sprites off-screen!
		
		jsr clear_screen				; clear the screen
		
		ldx #$00
print_gover								; print the game over message
		lda game_over_txt,x
		sta $0568,x
		lda #$02
		sta $d968,x
		inx
		cpx #$f0
		bne print_gover

; load current level number into last level variable for 'continue' feature		
				
		lda level_num					; load current level number
		sta last_level					; store to last level reached variable

; give player time to read the game over message!	
						
		ldy #$ff						; wait a bit...
		jsr sync_wait_long
	
		jmp title_setup					; set up the title screen
		
; level is complete! initiate level complete sequence...

level_complete
		jsr score_accrue_1000			; give player 1000 points
		
		lda level_num					; load current level number
		inc level_num					; increase it by 1
		sta last_level					; store to last level reached variable
		
		cmp #$0c						; is the level number 12
		bne level_comp_end				; no? branch to level comp end
										; yes?
		jmp comp_init					; initiate game complete sequence!
		
level_comp_end
		jmp get_ready_setup				; set up the next level
		
; game is complete, initiate the game complete sequence

comp_init
		lda #$00						; set the flag for
		sta tbordblue					; a black top border!

		lda #$08						; some game complete music!
 		jsr music+$00					; init the music driver with tune 8
 		
 		jsr hide_sprites				; move the sprites off-screen!
		jsr clear_screen				; clear the screen

		jsr score_accrue_1000			; give player 1000 points
		jsr score_accrue_1000			; and another 1000 points				
				
		ldy #$0f						; wait a bit...
		jsr sync_wait_long		
		
		jsr status_bar					; print status bar
		
		jsr update_status				; update new score and energy		

; print game complete message!		
		
		ldx #$00
print_gcomp1
		lda game_comp_txt,x
		sta $04a0,x
		lda #$02
		sta $d8a0,x
		inx
		cpx #$a0
		bne print_gcomp1
	
; print the game complete graphic display		
			
		ldx #$00
print_gcomp2
		lda comp_screen_ram,x
		sta $05ea,x
		lda comp_screen_ram+19,x
		sta $05ea+40,x
		lda comp_screen_ram+38,x
		sta $05ea+80,x					
		lda comp_screen_ram+57,x
		sta $05ea+120,x		
		lda comp_screen_ram+76,x
		sta $05ea+160,x		
		lda comp_screen_ram+95,x
		sta $05ea+200,x		
		lda comp_screen_ram+114,x
		sta $05ea+240,x		
		lda comp_screen_ram+133,x
		sta $05ea+280,x		
		
		lda comp_colour_ram,x
		sta $d9ea,x
		lda comp_colour_ram+19,x
		sta $d9ea+40,x
		lda comp_colour_ram+38,x
		sta $d9ea+80,x					
		lda comp_colour_ram+57,x
		sta $d9ea+120,x			
		lda comp_colour_ram+76,x
		sta $d9ea+160,x	
		lda comp_colour_ram+95,x
		sta $d9ea+200,x			
		lda comp_colour_ram+114,x
		sta $d9ea+240,x			
		lda comp_colour_ram+133,x
		sta $d9ea+280,x					
		
		inx
		cpx #$13
		bne print_gcomp2
		
; throw some game complete sprites at the screen...

; ready the sprites for the title screen bottom sprites	
			
		ldx #$00
gc_sprites
		lda gc_sprite_x	,x
		sta sprite_x,x
		lda gc_sprite_y	,x
		sta sprite_y,x
		lda gc_sprite_col,x
		sta sprite_col,x
		lda gc_sprite_def,x
		sta sprite_def,x
		lda gc_sprite_anims,x
		sta sprite_anims,x
		lda gc_sprite_anime,x
		sta sprite_anime,x
		inx
		cpx #$03
		bne gc_sprites				

; now the game complete loop...		
						
comp_loop
		lda cpra						; load port 2 register
		and #$10						; wait for fire button press
		bne comp_loop
		
		jsr fire_debounce				; debounce the button
		
; swap the player characters over

		lda ply_type					; which player character at moment?
		cmp #$00						; girl?
		beq chnge2_boy					; yes? change to boy!
										; no?

		jsr girl_sprites				; insert girl sprites
		
		jmp restart_game				; restart game!
		
chnge2_boy								; change to boy!

		jsr boy_sprites					; insert boy sprites
				
restart_game		
		lda #01
		sta level_num					; reset level to 1
		sta last_level					; and last level to 1
		
		jmp get_ready_setup				; back to get ready setup!
		
																				
; subroutines called from above code =====================		


; clear screen subroutine ----------------------------------		

clear_screen
		ldx #$00						; zero x register		
clsloop
		lda #$20						; value for 'empty space'
		sta $0400,x						; clear screen ram
		sta $0500,x
		sta $0600,x
		sta $0700,x
		lda #$00						; value for black
		sta $d800,x						; clear colour ram
		sta $d900,x
		sta $da00,x
		sta $db00,x		
		inx
		bne clsloop
		
	   	rts								; return to where this was called from
	   		   		   	
; debounce subroutines for the joystick and keyboard -------------------

fire_debounce
		lda cpra						; keep looping here until the
		and #$10						; player decides to let the
		beq fire_debounce				; fire button go
		
		rts
		
key_debounce		
		lda cprb						; is there a keypress still?
		cmp #$ff						; an idiot is still pressing something???
		bne key_debounce				; lock them in a loop till they stop!!
		
		rts	
 	
; irq sync wait subroutine ---------------------------------- 
	   	 
sync_wait								; wait for sync update in irq code!
		lda #$00						; waits until the sync flag is
		sta sync						; set at the end of the interrupt
sw_loop									; code further down.
		cmp sync						; no code in the main loop above
		beq sw_loop						; is executed until the interrupt
										; has run it's course!
		rts
		
; sync wait with fire button debounce	

sync_wait_long						
		jsr sync_wait
	
		lda cpra
		and #$10
		
		bne dec_wait_tmr	

		jsr fire_debounce
		ldy #$01
		
dec_wait_tmr		
		dey
		bne sync_wait_long
	
		rts		
		
; update 'rolling' message subroutine --------------------- 
		
write_message
		ldx scroll_x
		cpx #$28
		bcs scroll_cnt

; fetch a character from the text table

scroll_read
		ldy #$00
		lda (scroll_pos),y
		bne scroll_write
		
		jsr scroll_reset
		
		jmp scroll_read

; write that character to the screen
		
scroll_write
		sta $06a8,x
		
; move ghost sprite	a little	
				
		lda t_sprite_x+$00
		clc
		adc #$04
		sta t_sprite_x+$00
		ldx scroll_x
		cpx #$27
		beq reset_ghost
		
		jmp update_txt_pos
		
reset_ghost
		lda #$00
		sta t_sprite_x+$00			

; update the text position

update_txt_pos
		inc scroll_pos+$00
		bne *+$04
		inc scroll_pos+$01

; store the current text position		
		
scroll_cnt
		inx
		stx scroll_x
		
		rts								; return to where this was called from

; reset the scroll message		
				
scroll_reset
		lda #<mess_text
		sta scroll_pos+$00
		lda #>mess_text
		sta scroll_pos+$01
		
		rts								; return to where this was called from
			
; title screen player select subroutine ------------------------
	
boy_select

; first, insert sprites to make the boy dance 
; and the girl static on the title screen

		ldx #$d8
		stx t_sprite_def+$02			; insert the values for the dancing boy
		stx t_sprite_anims+$02			; animation!
		ldx #$dc
		stx t_sprite_anime+$02
		
		ldx #$dc
		stx t_sprite_def+$01			; insert the values for still girl!
		stx t_sprite_anims+$01
		ldx #$dd
		stx t_sprite_anime+$01

; now highlight the word boy and grey the word girl!		
				
		ldx #$00						; zero x register
boy_sel_loop						
		lda #$0b						; load white
		sta $db51,x						; write 'boy' in white
		lda #$01						; load dark grey
		sta $db5d,x						; write 'girl' in dk grey
		inx
		cpx #$05						; done 5 writes?
		bne boy_sel_loop				; all written? no? loop back!
		
		lda #$00						; make player store 0
		sta ply_type					; so boy is selected

; finally, insert all the game sprites for the boy into
; the sprite store to be used by the movement routine		

boy_sprites				
		lda #$ac
		sta playwalkleft_def
		sta playwalkleft_anims
		lda #$b0
		sta playwalkleft_anime
		
		lda #$a7
		sta playwalkright_def
		sta playwalkright_anims
		lda #$ab
		sta playwalkright_anime
		
		lda #$b1
		sta playjumpleft_def
		sta playjumpleft_anims
		lda #$b2
		sta playjumpleft_anime
		
		lda #$b0
		sta playjumpright_def
		sta playjumpright_anims
		lda #$b1
		sta playjumpright_anime
		
		lda #$ab
		sta playstatic_def
		sta playstatic_anims
		lda #$ac
		sta playstatic_anime
		
		lda #$b2
		sta playfall_def
		sta playfall_anims
		lda #$b3
		sta playfall_anime		
		
		lda #$01						; make player type 1
		sta ply_type					; which is the boy
		
		lda #$0a						; boy is 'stronger'...
		sta ply_strength
		
		lda #$ff						; ...but can't jump
		sta ply_jumppwr					; quite as far!
		
		rts
		
girl_select

; first, insert sprites to make the girl dance 
; and the boy static on the title screen

		ldx #$d8
		stx t_sprite_def+$02			; insert the values for a
		stx t_sprite_anims+$02			; still boy
		ldx #$d9
		stx t_sprite_anime+$02
		
		ldx #$dc						; and values for a dancing
		stx t_sprite_def+$01			; girl animation!
		stx t_sprite_anims+$01
		ldx #$e0
		stx t_sprite_anime+$01

; now highlight the word girl and grey the word boy!		
				
		ldx #$00						; zero x register
girl_sel_loop						
		lda #$01						; load white
		sta $db51,x						; write 'boy' in white
		lda #$0b						; load dark grey
		sta $db5d,x						; write 'girl' in dk grey
		inx
		cpx #$05						; done 5 writes?
		bne girl_sel_loop				; all written? no? loop back!			
		
		lda #$01						; make player store 1
		sta ply_type					; so girl is selected
	
; finally, insert all the game sprites for the girl into
; the sprite store to be used by the movement routine

girl_sprites		
		lda #$b8
		sta playwalkleft_def
		sta playwalkleft_anims
		lda #$bc
		sta playwalkleft_anime
		
		lda #$b3
		sta playwalkright_def
		sta playwalkright_anims
		lda #$b7
		sta playwalkright_anime
		
		lda #$bd
		sta playjumpleft_def
		sta playjumpleft_anims
		lda #$be
		sta playjumpleft_anime
		
		lda #$bc
		sta playjumpright_def
		sta playjumpright_anims
		lda #$bd
		sta playjumpright_anime
		
		lda #$b7
		sta playstatic_def
		sta playstatic_anims
		lda #$b8
		sta playstatic_anime
		
		lda #$be
		sta playfall_def
		sta playfall_anims
		lda #$bf
		sta playfall_anime			
		
		lda #$00						; make player type 0
		sta ply_type					; which is the girl
		
		lda #$09						; girl is 'weaker'...
		sta ply_strength			
		
		lda #$fe						; ...but can jump
		sta ply_jumppwr					; a bit further!
					
		rts		
				
; hide sprites subroutine ---------------------------------------

hide_sprites
		ldx #$00						; load x register with 0
hide_sprites_loop						; now loop through all 8
		lda sprite_x_hide,x				; hardware sprites and write
		sta sprite_x,x					; 0's into x and y positions
		lda sprite_y_hide,x				; to hide the sprites
		sta sprite_y,x					; off-screen
		inx
		cpx #$08
		bne hide_sprites_loop
		
		rts

; player movement subroutine ---------------------------------------		
				
player_move
		lda #$00						; start with loading 0
		sta ply_move_flag				; into the move flag
		
; press fire on joystick test...

joy_fire
		lda #$7f
		sta cpra

		lda cpra						; stick button pressed?
		and #$10
		bne joy_left					; nope? go and check left!
										; yes?
		ldx gravity_on					; is gravity already on?
		cpx #$01						; yes?
		beq joy_left					; then go and check left!
		
		lda #<sfx_jump 			        ; start address of the jump sound effect
        ldy #>sfx_jump
        ldx #$0e					    ; use channel 3
        jsr music+$06					; call the sfx driver and play effect!

		ldx #$fd
		stx gravity_strength
		ldx ply_jumppwr
		stx gravity_timer
		ldx #$01
		stx gravity_on		

; pull left on joystick test...

joy_left
		lda cpra						; stick being pushed left?
		and #$04
		bne joy_right					; nope? go and check right!
										; it is being pushed left?
		ldx gravity_on					; ok, lets see if the gravity...
		cpx #$01						; ...is active first!
		beq	do_jump_left				; it's active? jump left then!
										; no gravity? ok, then...
		ldx sprite_def+$00				; load the current player sprite def
		cpx playstatic_def				; is it the static sprite? we're still?
		beq setup_move_left				; yes? better sort the left walk animation!
										; what if we're not still?
		ldx sprite_def+$00			
		cpx playjumpleft_def
		bne joy_left_move
		
setup_move_left		
		ldx playwalkleft_def
		stx sprite_def+$00
		stx sprite_anims+$00
		ldx playwalkleft_anime
		stx sprite_anime+$00
		
		jmp joy_left_move
		
do_jump_left
		ldx playjumpleft_def
		stx sprite_def+$00
		stx sprite_anims+$00
		ldx playjumpleft_anime
		stx sprite_anime+$00	
		
joy_left_move

		ldx sprite_x
		dex
		cpx #$0a
		bcc *+$05
		stx sprite_x

		inc ply_move_flag

; pull right on joystick test...

joy_right
		lda cpra
		and #$08
		bne joy_exit

		ldx gravity_on
		cpx #$01
		beq	do_jump_right

		ldx sprite_def+$00
		cpx playstatic_def
		beq setup_move_right
		
		ldx sprite_def+$00
		cpx playjumpright_def
		bne joy_right_move

setup_move_right
		ldx playwalkright_def
		stx sprite_def+$00
		stx sprite_anims+$00
		ldx playwalkright_anime
		stx sprite_anime+$00
		
		jmp joy_right_move		
		
do_jump_right
		ldx playjumpright_def
		stx sprite_def+$00
		stx sprite_anims+$00
		ldx playjumpright_anime
		stx sprite_anime+$00			

joy_right_move
		ldx sprite_x
		inx
		cpx #$a2
		bcs *+$05
		stx sprite_x

		inc ply_move_flag
		
; check if gravity is on, if so perform gravity magic!
				
joy_exit
		lda gravity_on
		cmp #$01
		bne slow_player

		lda sprite_y+$00
		clc
		adc gravity_strength
		sta sprite_y+$00

		ldx gravity_timer
		inx
		cpx #$04
		bne exit_gravity

		ldx gravity_strength
		inx
		cpx #$05
		bne *+$04
		ldx #$04
		stx gravity_strength
		
		ldx #$00
exit_gravity
		stx gravity_timer
		
; check if the player moving, if so slow 'x' movement
; otherwise display static player sprite

slow_player
		lda ply_move_flag
		cmp #$00
		beq ply_no_anim

		lda ply_move_timer
		adc #$01
		and #$07
		sta ply_move_timer
		cmp #$04
		bcc ply_move_exit
		
		jmp ply_move_exit	
			
ply_no_anim								; the player isn't moving the joystick?
		lda gravity_on					; let's check if gravity is on!
		cmp #$01			
		beq ply_freefall				; yes, gravity is active? goto to freefall!
										; gravity is not active?
		lda playstatic_def				; the static player sprite...
		sta sprite_def+$00				; ...needs to be shown then
		sta sprite_anims+$00			; because there is no input
		lda playstatic_anime			; from the player!
		sta sprite_anime+$00
		
		jmp ply_move_exit				; exit the the movement routines!

ply_freefall							; ok, we have some gravity because			
		lda playfall_def				; the player is in the air either
		sta sprite_def+$00				; falling or jumping!
		sta sprite_anims+$00			; therefore, show the fall/jump
		lda playfall_anime				; sprite frame!
		sta sprite_anime+$00		
		
ply_move_exit							; all movement checks complete

		rts								; go back to main game loop!

; player to enemy sprites subroutine ----------------------------

player_collision
		lda #$00						; set the colls flag to 0
		sta ply_colls_flag

		lda sprite_x+$00				; setup a 'bounding box'
		sec								; around the player sprite
		sbc #$04						; for software collision
		sta coll_temp+$00				; detection
		clc								; grab the some x positions
		adc #$09						; for our player sprite which	
		sta coll_temp+$01				; is sprite 0!	

		lda sprite_y+$00				; and do the same for some y
		sec								; positions of sprite 0
		sbc #$0a
		sta coll_temp+$02				; the x and y positions are
		clc								; stored in the 'coll_temp'
		adc #$16						; labels
		sta coll_temp+$03

; check for collision with the cross		

		lda sprite_x+$01				; now grab the x postion of the cross
		cmp coll_temp+$00				; from the sprite table
		bcc enemy_colls_chk				; and check against our 'bounding box'
		cmp coll_temp+$01				; coordinates to see if they	
		bcs enemy_colls_chk				; overlap...

		lda sprite_y+$01				; and do the same for the y coordinates
		cmp coll_temp+$02
		bcc enemy_colls_chk				; if no coordinates overlap, skip down
		cmp coll_temp+$03				; to 'enem_colls_skip'
		bcs enemy_colls_chk
										; if coordinates ARE overlapping		
		lda #$01						; load the player colls flag with 1
		sta ply_colls_flag				; as cross has been touched
		
; check for collision with the meanies
				
enemy_colls_chk	
		ldx #$00
enem_colls_loop
		ldy colls_timer					; first load up the collision timer
		iny								; and increase it by '1'
		cpy ply_strength				; is the timer equal to 'ply_strength' yet?
		bne coll_tmr_skip				; no? go down to coll_tmr_skip
										; and skip collision checking
								
		lda sprite_x+$02,x				; now grab the x postion of each sprite
		cmp coll_temp+$00				; from the sprite table
		bcc enem_colls_skip				; and check against our 'bounding box'
		cmp coll_temp+$01				; coordinates to see if they	
		bcs enem_colls_skip				; overlap...

		lda sprite_y+$02,x				; and do the same for the y coordinates
		cmp coll_temp+$02
		bcc enem_colls_skip				; if no coordinates overlap, skip down
		cmp coll_temp+$03				; to 'enem_colls_skip'
		bcs enem_colls_skip
										; if some coordinates overlap, collision!
		dec ply_energy					; decrease the player energy flag by 1
		
		jsr energy_bar					; redraw the energy bar
		
		lda ply_energy					; is the player out of energy?
		cmp #$00						 
		bne enem_colls_skip				; no? go down to 'enem_colls_skip'
		
		jmp energy_out					; yes? go down to 'energy_out'

enem_colls_skip
		inx
		cpx #$07
		bne enem_colls_loop	

		ldy #$00						; load x register with '0'	
coll_tmr_skip							; to reset collision timer and...				
		sty colls_timer					; store x register to colls_timer

		lda sprite_y+$00				; load up the player y position
		cmp #$01						; is it off the ground?
		bcs bkgrd_coll_start			; yes? better check background collisions

		rts								;  go back to main game loop

; check for collision with the background

		lda sprite_x+$00
		cmp #$22
		bcs bkgrd_coll_start

		rts

; sprite y coord higher than $32 so do background collision checks

bkgrd_coll_start	
		sec
		sbc #$1c
		lsr
		lsr
		lsr
		tax
		lda screen_low,x
		sta pcol_read+$01
		lda screen_high,x
		sta pcol_read+$02

		lda sprite_x+$00
		sec
		sbc #$09
		lsr
		lsr
		tax
		jsr pcol_read
		sta coll_temp+$00
		inx
		jsr pcol_read
		sta coll_temp+$01

		ldy coll_temp+$00
		lda chargrav_data,y
		cmp #$01
		beq grav_kill
		
		ldy coll_temp+$01
		lda chargrav_data,y
		cmp #$01
		beq grav_kill
						
		lda #$01
		sta gravity_on
		
		jmp grav_chk_finish

; self modifying code for player background collision

pcol_read
		lda $6464,x						; black magic and evil spirits... :D
		rts

; player sprite is "standing" on something, so kill gravity effect

grav_kill
		lda gravity_strength
		cmp #$80
		bcc *+$03
		
		rts

		lda #$00
		sta gravity_timer
		sta gravity_strength
		sta gravity_on
		
		lda sprite_y+$00
		and #$f8
		clc
		adc #$05
		sta sprite_y+$00

; gravity check is over, now check for other background collisions
				
grav_chk_finish

		ldy coll_temp+$00
		lda chardrain_data,y
		cmp #$01
		beq bkgrd_colls_loop
		
		ldy coll_temp+$01
		lda chardrain_data,y
		cmp #$01
		beq bkgrd_colls_loop
		
		jmp bkgrd_chk_end				; no collision detected? end checks!
		
; collision detected, so drain energy but with a timer to limit drain		

bkgrd_colls_loop
		ldy bkgd_coll_timer				; first load up the collision timer
		iny								; and increase it by '1'
		cpy ply_strength				; is the timer equal to player strength?
		bne bkgrd_colls_skip			; no? go down to bkgrd_colls_skip
										; and skip collision checking
		lda ply_energy
		dec ply_energy
		jsr energy_bar
		
		lda ply_energy					; is the player out of energy?
		cmp #$00
		bne reset_bacoll_tmr			; no? go down to 'reset_bacoll_tmr'	
		
		jmp energy_out					; yes? skip down to 'energy_out'

reset_bacoll_tmr						; reset the background collision timer				
		ldy #$00
bkgrd_colls_skip
		sty bkgd_coll_timer				; store y register to bkgd_coll_timer
		
; end of background checks!		
		
bkgrd_chk_end							; all background checks are complete...

		rts								; ...return to main game loop!

; energy depleted/set death flag subroutine ----------------------
					
energy_out
		lda #$02						; load the player colls flag with	
		sta ply_colls_flag				; 2 because all the energy is gone!
										; then...
		rts								; go back to main game loop where
										; player will be killed!!!! MWHAAAA!	
												
; energy bar drawing subroutine ----------------------------------

energy_bar
		ldy #$00						; update the energy bar!
		lda #$20
clr_energy_loop							; clear the row of chars that display the
		sta $07c7,y						; energy bar by printing a line of
		iny								; blank spaces!
		cpy #$21
		bne clr_energy_loop
		
		ldy #$00				
		lda #$40
drw_energy_loop							; now redraw the energy bar with char $40
		sta $07c7,y						; (the block making up the bar)
		iny								; equal to the amount of energy 
		cpy ply_energy					; remaining
		bne drw_energy_loop	
		
		rts								; go back to wherever this was called from!
		
; in-game status bar initial print routine ------------------------

status_bar
		ldx #$00
status_print
		lda status_screen_ram,x
		sta $0798,x
		lda status_colour_ram,x
		sta $db98,x		
		inx
		cpx #$50
		bne status_print
		
		rts						

; in-game status bar update subroutine ----------------------------		
		
update_status
		ldx #$00						; load x register with 0
score_print_g
		lda score,x						; read each digit of the score table
		ora #$30
		sta $079f,x						; write to the screen
		lda high_score,x				; read each digit of high socre table
		ora #$30
		sta $07ba,x						; write to screen
		inx
		cpx #$06						; do this 6 times for 6 digit score!
		bne score_print_g

		ldx #$00						; load x register with 0	
cross_print
		lda cross_score,x				; read each digit of the cross table
		ora #$30
		sta $07ae,x						; write to the screen
		inx
		cpx #$02						; repeat twice because 2 digit number!
		bne cross_print		
		
		rts								; return to main loop
		
; cross accrue subroutine ------------------------------------			
		
cross_accrue

		lda cross_num
		inc cross_num					; increase cross variable by 1

		ldx #$01						; give the player 1 cross!
ca_loop
		lda cross_score,x
		clc
		adc #$01
		cmp #$0a
		beq ca_cnt
		sta cross_score,x
		
		jmp cross_check
		
ca_cnt	
		lda #$00
		sta cross_score,x
		dex
		cpx #$ff
		bne ca_loop
		
; now check if 20 crosses have been collected!
				
cross_check						
		ldx #$00
		lda cross_score,x				; load up the tens column of the cross score
		
		cmp #$02						; is there a '2' in the tens?
		bne cross_move					; no? go down to 'cross_move'
										; yes?
		lda #$01						; set the level complete flag to 1
		sta level_comp_flag				; and go back to main game loop
										; where level will be completed!
		rts
		
; move the cross to next position on the screen
	
cross_move						
		ldx cross_num			; load the cross_num variable into x register
		lda cross_x,x			; grab x-reg value from cross x coord table
		sta sprite_x+$01		; store table value in sprite 1 x coord table
		lda cross_y,x			; grab x-reg value from cross y coord table
		sta sprite_y+$01		; store table value in sprite 1 y coord table
		
		rts						; return to main loop
		
; score accrue subroutine ------------------------------------			

score_accrue_100				; give player 100 points
		ldx #$03				; load 3 to x register
		
		jmp sa_loop				; jump down to sa_loop

score_accrue_1000				; give players 1000 points
		ldx #$02				; load 2 to x register
		
sa_loop
		lda score,x				; jump to x reg position in score table
		clc						; clear carry
		adc #$01				; add 1
		cmp #$0a				; compare to 10
		beq sa_cnt				; if 10, go down to sa_cnt to change column
		sta score,x				; else store to x reg position in score table

		jmp score_compare		; jump down to score_compare

sa_cnt	
		lda #$00
		sta score,x
		dex
		cpx #$ff
		bne sa_loop

; now compare current score to high score

score_compare
		ldx #$00
score_chk
		lda score,x
		cmp high_score,x
		beq score_chk_cnt
		bcc score_chk_end
		bcs hi_score_update
score_chk_cnt	
		inx
		cpx #$06
		bne score_chk
score_chk_end

		rts

; current score is a high score???

hi_score_update
		ldx #$00				; start at first digit
hi_up_loop
		lda score,x				; copy current digit score table to...
		sta high_score,x		; ...current digit high score table
		inx						; now next digit
		cpx #$06				; repeat 6 times because 6 digit scores
		bne hi_up_loop			; 6 done yet? no? up to hi_up_loop
								; yes?
		rts						; return to main loop
		
; update title sprites 'y' position to 'bounce' them

bounce_title

		ldx bounce_timer			; load up the bounce_timer
		inx							; increase it by '1'
		cpx #$03					; is the timer equal to '4' yet?
		bne bounce_skip				; no? don't update movement
									; yes? better do some moving then!
		ldx #$00
t_sprite_y_upd
		lda t_spr_y_dir,x
		cmp #$01
		beq t_sprite_up
		
		lda sprite_y,x
		clc
		adc #$02
		sta sprite_y,x
		cmp #$4a
		bcc t_next_sprite
		lda #$01
		sta t_spr_y_dir,x
		jmp t_next_sprite
					
t_sprite_up
		lda sprite_y,x
		sec
		sbc #$02
		sta sprite_y,x		
		cmp #$3e
		bcs t_next_sprite
		lda #$00
		sta t_spr_y_dir,x

t_next_sprite
		inx
		cpx #$08
		bne t_sprite_y_upd
		
		ldx #$00					; load x register with '0'
bounce_skip							; to reset anim_timer and...	
		stx bounce_timer			; store x register to anim_timer
		
		rts		

; enemy movement processor subroutine ------------------------		

process_enemies

		ldy enemy_timer				; load up the enemy timer
		iny							; increase it by '1'
		cpy #$02					; is the timer equal to '3' yet?
		beq move_enemies			; yes? then 'move_enemies'
									; no?
		jmp enemy_move_skip			; jump down to enemy_move_skip!
		
move_enemies
				
; deal with the vertical moving enemies first
; spider 1 and 2 (sprites 2 and 3)

		ldx #$02					; start on sprite 2
enemy_vert_check
		lda sprite_dir,x			; which direction is it moving?
		cmp #$01					; down?
		beq enemy_vert1				; yes, go down to enemy_vert1	
									; no?
		lda sprite_y,x				; load sprites y postion from table
		clc							; clear carry flag
		adc sprite_spd,x			; add sprites speed table to y position
		sta sprite_y,x				; store new y position
		cmp sprite_max,x			; load sprites max y position
		bcc next_vert_enemy			; not reached end of path? goto next_vert_enemy
		lda #$01					; reached end of path? change direction
		sta sprite_dir,x			; store direction
		lda left_enm_def,x			; load sprite defs and animation for up
		sta sprite_def,x
		sta sprite_anims,x
		lda left_enm_anime,x
		sta sprite_anime,x
		jmp next_vert_enemy			; do next sprite!

enemy_vert1
		lda sprite_y,x
		sec
		sbc sprite_spd,x
		sta sprite_y,x		
		cmp sprite_min,x
		bcs next_vert_enemy
		lda #$00
		sta sprite_dir,x
		lda rite_enm_def,x
		sta sprite_def,x
		sta sprite_anims,x
		lda rite_enm_anime,x
		sta sprite_anime,x		
		
next_vert_enemy
		inx
		cpx #$04
		bne enemy_vert_check
		
; now deal with the horizontal moving enemies
; zombie, ghost, bat, witch (sprites 4 - 7)		
				
enemy_hori_check
		lda sprite_dir,x
		cmp #$01
		beq enemy_hori1
		
		lda sprite_x,x
		clc
		adc sprite_spd,x
		sta sprite_x,x
		cmp sprite_max,x
		bcc next_hori_enemy
		lda #$01
		sta sprite_dir,x
		lda left_enm_def,x
		sta sprite_def,x
		sta sprite_anims,x
		lda left_enm_anime,x
		sta sprite_anime,x
		jmp next_hori_enemy

enemy_hori1
		lda sprite_x,x
		sec
		sbc sprite_spd,x
		sta sprite_x,x		
		cmp sprite_min,x
		bcs next_hori_enemy
		lda #$00
		sta sprite_dir,x
		lda rite_enm_def,x
		sta sprite_def,x
		sta sprite_anims,x
		lda rite_enm_anime,x
		sta sprite_anime,x		
		
next_hori_enemy
		inx
		cpx #$08
		bne enemy_hori_check

		ldy #$00					; load x register with 0
enemy_move_skip						; to reset enemy move timer
		sty enemy_timer				; and store to timer!
		
		rts							; go back to main loop!
		
; background animations subroutine ----------------------------------		

process_bkgd_anims
		
		ldy bkgd_timer				; load up the bkgrd anim timer
		iny							; increase it by '1'
		tya							; transfer to accum
		cmp bkgd_lvl_tmr			; compare to level specific anim speed
		beq bkgd_anim_chk			; it's time to animate? animate then!
			
		jmp bkgrd_anim_skip			; no? skip background anims
								
bkgd_anim_chk

		lda level_num				; which level are we on?
		
		cmp #$02					; level 2?
		bne *+$05					; no? branch to level 3 check
		jmp flame_anim				; yes? jump to flame animation!
			
		cmp #$03					; level 3?
		bne *+$05					; no? branch to next check
		jmp rain_anim				; yes? jump to rain anim!
		
		cmp #$04					; etc, etc!
		bne *+$05
		jmp flame_anim
		
		cmp #$05					; etc, etc!
		bne *+$05
		jmp slime_anim
		
		cmp #$06					; etc, etc!
		bne *+$05
		jmp cinema_anim
		
		cmp #$07					; etc, etc!
		bne *+$05
		jmp rain_anim
		
		cmp #$08					; etc, etc!
		bne *+$05
		jmp slime_anim
		
		cmp #$09					; etc, etc!
		bne *+$05
		jmp snow_anim		
		
		cmp #$0a					; etc, etc!
		bne *+$05
		jmp slime_anim
		
		cmp #$0b					; etc, etc!
		bne *+$05
		jmp flame_anim
		
		cmp #$0c					; etc, etc!
		bne *+$05
		jmp slime_anim																		
		
		jmp reset_bkgrd_anim_tmr	; no anim? reset the timer!
		
; flame anim subroutine												
									
flame_anim							; flame animation								
									
	    ldx #$00					; zero x register
flame_ror
		lda $2390,x					; load mem location of the flame char
		lsr
		lsr
		lsr
		ror $2390,x					; shift byte of the char to the right
		ror $2390,x					; shift byte of the char to the right
		ror $2390,x					; shift byte of the char to the right				
		inx
		cpx #$08					; all 8 bytes done?
		bne flame_ror				; no? do the next byte then!
		
		jmp reset_bkgrd_anim_tmr

; slime anim subroutine	
						
slime_anim							; slime animation								

; horizontal slime
									
	    ldx #$00					; zero x register
slime_rol
		lda $2738,x					; load mem location of the slime char
		asl
		rol $2738,x					; shift byte of the char to the left		
		inx
		cpx #$08					; all 8 bytes done?
		bne slime_rol				; no? do the next byte then!

; vertical slime (char location $2550 to $2557)

		lda $2557
		sta chardown_temp
		
slime_down
		lda $2550-1,x
		sta $2550,x
		dex
		bpl slime_down
		
		lda chardown_temp
		sta $2550
				
		jmp reset_bkgrd_anim_tmr
		
; snowfall anim subroutine

snow_anim
		lda snowstrip			; which snow strip should move?
		cmp #$01				; strip 2?
		beq snow_fall2			; yes? do strip 2 then!
								; no? then do strip 1!
; move snow strip 1...
; shift snow char 196 a pixel down!
; (char location $2620 to $2627)		

snow_fall1						; move strip 1			
		lda $2627				; load up the last byte of the char
		sta chardown_temp		; put it in a temporary byte store

snow_move1
		lda $2620-$01,x			; now shift each byte down
		sta $2620,x				; until 7 of the bytes are done
		dex
		bpl snow_move1
	
		lda chardown_temp		; now put the byte that was originally
		sta $2620				; last into in the first byte of the char
		
		lda #$01				; next time, do snow strip 2!
		sta snowstrip
		
		jmp reset_bkgrd_anim_tmr		

; move snow strip 2...
; shift snow char 197 a pixel down!
; (char location $2628 to $262f)	

snow_fall2						; as above, but for snow strip 2!
		lda $262f
		sta chardown_temp
		
snow_move2
		lda $2628-$01,x
		sta $2628,x
		dex
		bpl snow_move2
		
		lda chardown_temp
		sta $2628
				
		lda #$00				; next time, do snow strip 1!
		sta snowstrip

		jmp reset_bkgrd_anim_tmr			

; cinema screen anim subroutine (screen noise flicker animation)
			
cinema_anim

		lda cinescrn				; which screen to draw to produce 'flicker'?

		cmp #$01					; turn of screen 1?
		bne *+$05					; no? go down and draw screen 0!					
		jmp draw_scrn1				; yes? draw screen 1!
		
draw_scrn0							; drawn screen 0
		ldx #$00					; zero the x register
scrn0_loop
		lda scrn0_patt,x			; load a byte of screen 0 pattern
		sta $24f0,x					; shove it into the screen char
		inx							; increase x register by 1
		cpx #$08					; all 8 bytes done?
		bne scrn0_loop				; no? go back and do next byte please...
									; all 8 bytes of char are done! YAY!
		lda #$01					; next time around...
		sta cinescrn				; ... screen 1 must be drawn!

		jmp draw_scrn_dun			; screen 0 drawing all complete
		
draw_scrn1							; draw screen 1
		ldx #$00					; zero the x register
scrn1_loop
		lda scrn1_patt,x			; load a byte of screen 1 patterm	
		sta $24f0,x					; shove it into the screen char
		inx							; increase x register by 1
		cpx #$08					; all 8 bytes are read and stored?
		bne scrn1_loop				; wot? no? read another byte please!
									; all 8 bytes are done! thanks mr. 6510!
		lda #$00					; next time through...
		sta cinescrn				; ...screen 0 must be drawn!
			
draw_scrn_dun						; cinema screen drawing down
									; so...
		jmp reset_bkgrd_anim_tmr	; ...reset the background anim timer please!
													
; "i'm singing in that there rain..." anim subroutine

rain_anim							; cats and dogs animation

; clear all raindrops first

		ldx #$00
		lda #$ff					; char needs to be solid!
clear_rain		
		sta $2400,x					; raindrop 1
		sta $2408,x					; raindrop 2
		sta $2410,x					; etc
		sta $2418,x
		sta $2420,x
		inx
		cpx #$08					; all 8 bytes of each char cleared?
		bne clear_rain				; no? branch back
		
; now check which raindrop should be drawn (rain_patt)

		lda raindrop

		cmp #$01
		bne *+$05
		jmp draw_rain1
		
		cmp #$02
		bne *+$05
		jmp draw_rain2
		
		cmp #$03
		bne *+$05
		jmp draw_rain3
		
		cmp #$04
		bne *+$05
		jmp draw_rain4							
		
draw_rain0	
		ldx #$00
rain0_loop
		lda rain_patt,x		
		sta $2400,x
		inx
		cpx #$08
		bne rain0_loop
		
		jmp draw_rain_dun
		
draw_rain1	
		ldx #$00
rain1_loop
		lda rain_patt,x		
		sta $2408,x
		inx
		cpx #$08
		bne rain1_loop
		
		jmp draw_rain_dun
		
draw_rain2
		ldx #$00
rain2_loop
		lda rain_patt,x		
		sta $2410,x
		inx
		cpx #$08
		bne rain2_loop
		
		jmp draw_rain_dun
		
draw_rain3
		ldx #$00
rain3_loop
		lda rain_patt,x		
		sta $2418,x
		inx
		cpx #$08
		bne rain3_loop
		
		jmp draw_rain_dun
		
draw_rain4
		ldx #$00
rain4_loop
		lda rain_patt,x		
		sta $2420,x
		inx
		cpx #$08
		bne rain4_loop
		
draw_rain_dun
		lda raindrop
		inc raindrop
		cmp #$04
		bne reset_bkgrd_anim_tmr
		lda #$00
		sta raindrop
		
		
reset_bkgrd_anim_tmr								
		ldy #$00					; load y register with '0'
bkgrd_anim_skip						; to reset bkgrd anim timer and...	
		sty bkgd_timer				; store y register to bkgrd anim timer
	
		rts	

; in-game level music init subroutine -------------------------------		

lmusic_setup

		lda level_num				; load the level number
		
		cmp #$01					; is it 1?
		bne *+$07					; no? check for level 2
		lda #$04					; yes? init level 1 music
		jmp lmusic_init_dun			; insert into player!
		
		cmp #$02					; is it 2?
		bne *+$07					; no? check for level 3
		lda #$05					; yes? init level 2 music		
		jmp lmusic_init_dun			; insert into player!
		
		cmp #$03					; etc, etc!
		bne *+$07
		lda #$02			
		jmp lmusic_init_dun
		
		cmp #$04
		bne *+$07
		lda #$06		
		jmp lmusic_init_dun
		
		cmp #$05
		bne *+$07
		lda #$07			
		jmp lmusic_init_dun
		
		cmp #$06
		bne *+$07
		lda #$09			
		jmp lmusic_init_dun
		
		cmp #$07					; is it 7
		bne *+$07
		lda #$04					; yes? resuse level 1 music
		jmp lmusic_init_dun			; insert into player!		

		cmp #$08					; is it 8?
		bne *+$07
		lda #$05					; yes? reuse level 2 music		
		jmp lmusic_init_dun			; insert into player!		
		
		cmp #$09					; etc, etc!
		bne *+$07
		lda #$02			
		jmp lmusic_init_dun	
		
		cmp #$0a
		bne *+$07
		lda #$06		
		jmp lmusic_init_dun	

		cmp #$0a
		bne *+$07
		lda #$07		
		jmp lmusic_init_dun

		cmp #$0c
		lda #$09		
		
lmusic_init_dun
 		jsr music+$00				; shove tune number into the music player
 				
 		rts							; go back to main loop!
 				
; end subroutines =====================================		
			

; IRQ interrupt code =================================

int
		pha  
		txa
		pha
		tya
		pha
		
		lda $d019
		and #$01
		sta $d019
		bne intr_trigd
		
		jmp irq_exit				; no interrupt triggered? exit irq code!
		
; an interrupt has triggered!?!
; which split should we run? let's check...

intr_trigd

		lda raster_num				; load up the raster_num variable
		
		cmp #$02					; is it set to '2'?
		bne *+$05					; no? then jump down to check for '3'
		jmp irq_rout2				; yes? jump down to irq_rout2!

		cmp #$03					; is the raster_num set to '3'?
		bne *+$05					; no? then jump down to check for '4'
		jmp irq_rout3				; yes? jump down to irq_rout3!
		
		cmp #$04					; is the raster_num set to '4'?
		bne *+$05					; no? then jump down to check for '5'
		jmp irq_rout4				; yes? jump down to irq_rout4!
		
		cmp #$05					; is the raster_num set to '5'?
		bne *+$05					; no? jump down and run irq_rout1!
		jmp irq_rout5				; yes? jump down to irq_rout5!		
				
; raster split 1 code -------------------------------

irq_rout1

; set background to black
; n.b. this split is small and black so that in levels
; where the top 'border' is blue, the blue doesn't show
; at the bottom of the screen when running under NTSC

		lda #$00					; change colour to black background
		sta $d021					; for top part of screen
		
; set interrupt handler ready for split 2

		lda #$02					; next run through this interrupt code,
		sta raster_num				; split 2 should execute, a bit
		lda #raster_2_pos			; further down...
		sta $d012
		
		jmp irq_exit
		
; raster split 2 code -------------------------------

irq_rout2

; should top border be blue or black?

		lda tbordblue				; is the top border flag set
		cmp #$01					; to 0 or 1?
		bne blackbord				; it's not 1? set to black then!
									; it is 1? ok then...
		lda #$06					; ...change colour to blue border
		sta $d021					; for top part of screen
		
		jmp split1_cont				; continue split

blackbord		
		lda #$00					; change colour to black border
		sta $d021					; for top part of screen		
		
split1_cont		
		
; set interrupt handler ready for split 3

		lda #$03					; next run through this interrupt code,
		sta raster_num				; split 2 should execute, a bit
		lda #raster_3_pos			; further down...
		sta $d012
		
; set up and position the hardware sprites
		
		lda #$ff					; turn on..
		sta $d015					; all 8 sprites!

		lda title_mode
		cmp #$01
		bne game_sprites	
		
; title_sprites		
		
		lda #$00					; make these sprites
		sta $d01c					; hires

		lda intro_mode
		cmp #$01
		beq skip_bounce
		
		jsr bounce_title
		
skip_bounce				
		jmp sprite_plotter		

; game sprites		
				
game_sprites
		lda #$ff					; make these sprites
		sta $d01c					; multi-colour
				
		lda #$0b					; and add some colours
		sta $d025					; a nice grey and a 
		lda #$01					; splash of white
		sta $d026					; will do nicely!

; in-game sprite plotter!		
		
sprite_plotter		
		ldx #$00
		ldy #$00
sprite_plot
		lda sprite_x,x
		asl
		ror $d010
		sta $d000,y
		lda sprite_y,x
		sta $d001,y
		iny
		iny
		lda sprite_col,x
		sta $d027,x
		lda sprite_def,x
		sta $07f8,x		
		inx
		cpx #$08
		bne sprite_plot
		
; in-game sprite animation routine!	
				
		ldx anim_timer				; load up the anim_timer
		inx							; increase it by '1'
		cpx #$04					; is the timer equal to '4' yet?
		bne game_anim_skip			; no? don't update animation!
									; yes? better do some animation then!
		ldx #$00
gau_loop
		lda sprite_def,x
		clc
		adc #$01
		cmp sprite_anime,x
		bne gau_skip
		lda sprite_anims,x
gau_skip
		sta sprite_def,x
		inx
		cpx #$08
		bne gau_loop								
									; yes?
		ldx #$00					; load x register with '0'
game_anim_skip						; to reset anim_timer and...	
		stx anim_timer				; store x register to anim_timer		

		jmp irq_exit
		
	
; raster split 3 code ------------------------------------

irq_rout3

; set interrupt handler ready for split 3
	
		lda #$04
		sta raster_num
		lda #raster_4_pos
		sta $d012

		lda #$00					; black background
		sta $d021
		
; play some music!

		lda video_type				; what machine is being used
		cmp #$00					; PAL?
		beq call_music_driver		; yes? play the music!
									; no?
		ldx music_tmr				; load music timer
		inx							; increase it
		cpx #$08					; is it 8?
		bne	music_play				; no? play music!
									; yes?
		ldx #$00					; reset the timer
		stx music_tmr				; then skip playing
									; the music on this occasion
		jmp irq_exit				; to slow it for ntsc!

music_play
		stx	music_tmr
call_music_driver
		jsr music+$03				

		jmp irq_exit		

; raster split 4 code ------------------------------------

irq_rout4

; set interrupt handler ready for split 5
	
		lda #$05
		sta raster_num
		lda #raster_5_pos
		sta $d012	
				
; are we in title screen mode?

		lda title_mode
		cmp #$01
		beq intro_chk				; yes? skip down to 't_mode_do'
		
		jmp irq_exit

intro_chk		
		lda intro_mode
		cmp #$01
		bne t_mode_do
		
		jmp irq_exit
		
t_mode_do		
				
; write the title screen rolling message

		jsr write_message		
		
; set up recycled sprites for the bottom
; of the title screen

		lda #$ff					; make these sprites
		sta $d01c					; multi-colour
				
		lda #$0b
		sta $d025
		lda #$01
		sta $d026
		
		ldx #$00
		ldy #$00
sprite_plot_2
		lda t_sprite_x,x
		asl
		ror $d010
		sta $d000,y
		lda t_sprite_y,x
		sta $d001,y
		iny
		iny
		lda t_sprite_col,x
		sta $d027,x
		lda t_sprite_def,x
		sta $07f8,x		
		inx
		cpx #$08
		bne sprite_plot_2

; animate the bottom sprites on the title screen 		
				
		ldx anim_timer_b			; load up the anim_timer
		inx							; increase it by '1'
		cpx #$0c					; is the timer equal to '4' yet?
		bne bot_anim_skip			; no? don't update animation!
									; yes? better do some animation then!
		ldx #$00
au_loop
		lda t_sprite_def,x
		clc
		adc #$01
		cmp t_sprite_anime,x
		bne au_skip
		lda t_sprite_anims,x
au_skip
		sta t_sprite_def,x
		inx
		cpx #$08
		bne au_loop								
									; yes?
		ldx #$00					; load x register with '0'
bot_anim_skip						; to reset anim_timer and...	
		stx anim_timer_b			; store x register to anim_timer
		
; colour wash effect for certain title screen lines

		ldx #$00
colourwash_loop
		lda colourwash_off,x
		clc
		adc colourwash_count
		and #$0f
		tay
		
		lda colourwash_rainbow,y		
		sta $d800,x					; cosine systems
		sta $daa8,x					; rolling message

		lda colourwash_reds,y
		sta $d918,x					; sub-title		
		sta $db98,x					; last/best score
				
		lda colourwash_greys,y
		sta $d968,x					; based on chiller
		sta $da08,x					; by jason
		sta $da30,x					; by jon / ryley			

		lda colourwash_blues,y
		sta	$d9b8,x					; by andy

		inx
		cpx #$28
		bne colourwash_loop

		ldx colourwash_timer
		inx
		cpx #$03
		bcc exit_wash
		inc colourwash_count
		ldx #$00
exit_wash	
		stx colourwash_timer											
	
; split 4 is done, exit irq interrupt

		jmp irq_exit
				
; raster split 5 code ------------------------------------

irq_rout5

; set interrupt handler ready for split 1
		
		lda #$01
		sta raster_num
		lda #raster_1_pos
		sta $d012
		
; open the upper and lower borders

		lda #$14
		sta $d011

		lda #$fc
		cmp $d012
		bne *-$03
		lda #$1b
		sta $d011				
		
		lda #$00					; turn off
		sta $d015					; all 8 sprites!
				
; set sync variable for main line code synchronisation

set_sync
		lda #$01					; set the sync flag to '1'		
		sta sync					; so the main code runs!
		
; restore registers and exit irq interrupt

irq_exit
		pla
		tay
		pla
		tax
		pla
nmi		rti

; end irq interrupt code =============================


; all data tables live here! ===============================


; screen low/high byte offset tables

screen_low
		!byte $00,$28,$50,$78,$a0,$c8,$f0,$18
		!byte $40,$68,$90,$b8,$e0,$08,$30,$58
		!byte $80,$a8,$d0,$f8,$20,$48,$70,$98
		!byte $c0
		
screen_high
		!byte $04,$04,$04,$04,$04,$04,$04,$05
		!byte $05,$05,$05,$05,$05,$06,$06,$06
		!byte $06,$06,$06,$06,$07,$07,$07,$07
		!byte $07

; colour wash data tables for text colour cycling

colourwash_off
		!byte $00,$01,$02,$01,$00,$01,$02,$01
		!byte $08,$07,$06,$05,$04,$03,$02,$04
		!byte $06,$08,$0a,$0c,$0e,$00,$0f,$0e
		!byte $0d,$0c,$0b,$0a,$09,$08,$07,$06
		!byte $0c,$0c,$0b,$0a,$0b,$0c,$0c,$0e		

colourwash_blues
		!byte $06,$04,$0e,$0e,$03,$01,$0f,$03
		!byte $0f,$01,$03,$0e,$0e,$04,$06,$0f
		
colourwash_greys
		!byte $0b,$0b,$0c,$0f,$0f,$0d,$01,$0d
		!byte $0d,$01,$01,$0d,$0f,$01,$0c,$0b
		
colourwash_reds
		!byte $09,$02,$08,$08,$0a,$07,$0f,$0a
		!byte $0f,$07,$0a,$08,$08,$02,$09,$0f		
					
colourwash_rainbow
		!byte $06,$02,$04,$05,$03,$07,$01,$01
		!byte $01,$01,$01,$07,$03,$05,$04,$06

; text data for intro message		
		
intro_1		!scr "           pal  c64 detected!           "
intro_2		!scr "           ntsc c64 detected!           "
intro_3		!scr "      please use an 8580 sid chip!      "

; text data for the rolling message

mess_text
		!scr "   *  c o s i n e    s y s t e m s  *   "
		!scr "         p r e s e n t s . . .          "		
		!scr "       c  h  i  l  l  e  r      2       "
		!scr "                                        "
		!scr "    press * fire * to start the game    "
		!scr "    from the beginning of level 1...    "
		!scr "    hit * f7 * to play the game from    "
		!scr " the level reached during the last game "
		!scr "                                        "			
		!scr "  pull left on the joystick to select   "
		!scr "     and play as the girl michaela!     "
		!scr "  pull right on the joystick to select  "
		!scr "      and play as the boy michael!      "					
		!scr "                                        "	
		!scr " guide your character around each level "
		!scr "   collecting 20 magic crosses, while   "
		!scr "  avoiding the nasty meanies who drain  "
		!scr " your limited energy supply on contact! "
		!scr "                                        "
		!scr "  the boy is a little stronger and can  "
		!scr " stand more contact before energy loss. "
		!scr "  the girl is a little lighter and can  "
		!scr "   make slightly longer length jumps.   "
		!scr "                                        "
		!scr "   based on chiller from mastertronic   "
		!scr "      by richard and david darling      "		 
		!scr "                                        "
		!scr "     coding, graphics and music by      "
		!scr "      *  a n d y    v a i s e y  *      "
		!scr "                                        "
		!scr " copious amounts of help and support by "
		!scr "       *  j a s o n    k e l k  *       "
		!scr "                                        "
		!scr "     quality control and testing by     "
		!scr "    * j o n a t h a n    m i n e s *    "
		!scr "                 a n d                  "
		!scr "       * r y l e y    j a m e s *       "
		!scr "                                        "
		!scr " kindly sponsored by freeze64.com /zine "
		!scr "                                        "				
		!scr "          special thanks to...          "				
		!scr "   artstate digital    vinny mainolfi   "
		!scr "     ray lejuez     dustin chambers     "
		!scr "     andrew fisher    david darling     "
		!scr "       and of course chantelle...       "
		!scr "                                        "
		!scr "for more cosine games and demos visit..."
		!scr "           www.cosine.org.uk            "
		!scr "                                        "																					
		!byte $00		; end of text marker
					
		!scr "hi vinny! thought I might find you here!"	
						
; text data for get ready messages

gr_lvl1_des1	!scr "         level 1  -  the forest         "
gr_lvl1_des2	!scr " darkness falls across the land         "
gr_lvl1_des3	!scr "    the midnight hour is close at hand! "

gr_lvl2_des1	!scr "         level 2  -  the church         "
gr_lvl2_des2	!scr " grizzly ghouls from every tomb         "
gr_lvl2_des3	!scr "      are closing in to seal your doom! "

gr_lvl3_des1	!scr "       level 3  -  the apartments       "
gr_lvl3_des2	!scr " creatures crawl in search of blood     "
gr_lvl3_des3	!scr "     to terrorize y/all/s neighborhood! "

gr_lvl4_des1	!scr "     level 4  -  the haunted house      "
gr_lvl4_des2	!scr " demons squeal in sheer delight         "
gr_lvl4_des3	!scr "      you they spy, so plump, so right! "

gr_lvl5_des1	!scr "        level 5  -  the cemetery        "
gr_lvl5_des2	!scr " evil reaches out from the crypt        "
gr_lvl5_des3	!scr "         to crush you in it/s icy grip! "

gr_lvl6_des1	!scr "         level 6  -  the cinema         "
gr_lvl6_des2	!scr " you try to run, you try to scream      "
gr_lvl6_des3	!scr "       but no more sun you/ll ever see! "

gr_lvl7_des1	!scr "          level 7  -  the park          "
gr_lvl7_des2	!scr " ghosts and ghouls are hard to beat     "
gr_lvl7_des3	!scr "      still you stand with frozen feet! "

gr_lvl8_des1	!scr "         level 8  -  the sewers         "
gr_lvl8_des2	!scr " the foulest stench is in the air       "
gr_lvl8_des3	!scr "      the funk of forty thousand years! "

gr_lvl9_des1	!scr "       level 9  -  the log cabin        "
gr_lvl9_des2	!scr " you try to scream, but terror          "
gr_lvl9_des3	!scr "    takes the sound before you make it! "

gr_lvl10_des1	!scr "       level 10  -  the basement        "
gr_lvl10_des2	!scr " you hear the door slam but             "
gr_lvl10_des3	!scr "   realise there/s nowhere left to run! "

gr_lvl11_des1	!scr "       level 11  -  the scrapyard       "
gr_lvl11_des2	!scr " creatures are creepin/ up behind...    "
gr_lvl11_des3	!scr "                 ...you/re out of time! "  

gr_lvl12_des1	!scr "         level 12  -  the cave          "
gr_lvl12_des2	!scr " you/re fighting for your life...       "
gr_lvl12_des3	!scr "   ...inside a killer, chiller tonight! " 

; text data for game over message

game_over_txt
		!scr "  and though you fought to stay alive   "
		!scr "        you begin to shiver...          "
		!scr "   /cause no mere mortal can escape     "
		!scr "       the evil of the chiller!         "
		!scr "                                        "	
		!scr "               game over!               "
		
; text data for game complete message

game_comp_txt
		!scr "  great!  you have finished chiller 2!  "
		!scr " now play through again, with the other "
		!scr "   character!  keep your score so far   "
		!scr "   and try to achieve a higher score!   "
		
; graphics data for game complete screen

comp_screen_ram
!byte $20,$00,$00,$55,$20,$4F,$20,$20,$20,$20,$4F,$00,$20,$20,$20,$4F,$5B,$5C,$5D
!byte $20,$5B,$5C,$5D,$00,$00,$4F,$00,$00,$00,$00,$00,$4F,$20,$5B,$5C,$5C,$5D,$20
!byte $4F,$00,$00,$5B,$5C,$5D,$00,$00,$00,$00,$4F,$00,$00,$4F,$00,$5B,$5D,$20,$20
!byte $20,$4F,$00,$4F,$00,$00,$20,$4F,$20,$20,$20,$20,$4F,$20,$20,$20,$20,$20,$20
!byte $20,$20,$20,$20,$20,$41,$54,$41,$42,$41,$42,$41,$41,$42,$54,$4F,$20,$20,$20
!byte $20,$20,$20,$20,$20,$73,$F7,$F4,$F5,$F6,$F7,$F4,$F7,$F7,$73,$20,$20,$20,$20
!byte $20,$20,$20,$20,$20,$20,$20,$4B,$F3,$F6,$F4,$F7,$C6,$20,$20,$20,$20,$20,$20
!byte $20,$20,$20,$20,$20,$20,$20,$20,$20,$4C,$C6,$20,$20,$20,$20,$20,$20,$20,$20

comp_colour_ram
!byte $01,$01,$01,$01,$0B,$01,$00,$00,$00,$00,$0B,$01,$00,$00,$00,$01,$0B,$0B,$0B
!byte $01,$0B,$0B,$0B,$01,$01,$0C,$01,$01,$01,$01,$01,$01,$00,$0C,$0C,$0C,$0C,$00
!byte $01,$01,$01,$0C,$0C,$0C,$01,$01,$01,$01,$0B,$01,$01,$0C,$01,$0B,$0B,$00,$00
!byte $0C,$0C,$01,$01,$01,$01,$00,$0B,$01,$00,$00,$00,$0B,$00,$00,$00,$00,$00,$00
!byte $00,$00,$00,$00,$00,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$01,$00,$00,$00
!byte $00,$00,$00,$00,$00,$0C,$0B,$0C,$0F,$0F,$0B,$0C,$0C,$0F,$0C,$00,$00,$00,$00
!byte $00,$00,$00,$00,$00,$00,$00,$05,$0B,$0B,$0C,$0F,$05,$00,$00,$00,$00,$00,$00
!byte $00,$00,$00,$00,$00,$00,$00,$00,$00,$05,$05,$00,$00,$00,$00,$00,$00,$00,$00

; intro screen default sprite positions, colours and definitions

; cosine logo

i_spr_x_dflt		!byte $4b,$59,$65,$00,$00,$00,$00,$00
i_spr_y_dflt		!byte $6a,$6a,$6a,$00,$00,$00,$00,$00
i_spr_col_dflt		!byte $02,$02,$02,$00,$00,$00,$00,$00
i_spr_def_dflt		!byte $c0,$c1,$c2,$a3,$a3,$a4,$a5,$a6
i_spr_y_dir			!byte $00,$01,$00,$01,$00,$01,$00,$01
i_spr_anims_dflt	!byte $c0,$c1,$c2,$a3,$a3,$a4,$a5,$a6
i_spr_anime_dflt	!byte $c1,$c2,$c3,$a4,$a4,$a5,$a6,$a7		
					
; titles screen default sprite positions, colours and definitions

; title logo 'chiller 2'

t_spr_x_dflt		!byte $27,$33,$3f,$4b,$57,$63,$6f,$85
t_spr_y_dflt		!byte $3e,$4a,$3e,$4a,$3e,$4a,$3e,$4a
t_spr_col_dflt		!byte $02,$02,$02,$02,$02,$02,$02,$02
t_spr_def_dflt		!byte $a0,$a1,$a2,$a3,$a3,$a4,$a5,$a6
t_spr_y_dir			!byte $00,$01,$00,$01,$00,$01,$00,$01
t_spr_anims_dflt	!byte $a0,$a1,$a2,$a3,$a3,$a4,$a5,$a6
t_spr_anime_dflt	!byte $a1,$a2,$a3,$a4,$a4,$a5,$a6,$a7

; title screen bottom sprites (ghost, boy, girl)

t_spr_x_dflt2		!byte $00,$46,$55,$00,$00,$00,$00,$00
t_spr_y_dflt2		!byte $b2,$d3,$d3,$b2,$b2,$b2,$b2,$b2
t_spr_col_dflt2		!byte $0f,$0a,$0a,$0f,$0f,$0f,$0f,$0f
t_spr_def_dflt2		!byte $c3,$d8,$d8,$c3,$c3,$c3,$c3,$c3
t_spr_dflt2_anims	!byte $c3,$d8,$d8,$c3,$c3,$c3,$c3,$c3
t_spr_dflt2_anime	!byte $c7,$dc,$d9,$c3,$c3,$c3,$c3,$c3

; game complete sprites (hearts, boy, girl)

gc_sprite_x			!byte $56,$4f,$5e,$94,$90,$6f,$5a,$0d
gc_sprite_y			!byte $8d,$a5,$a5,$a4,$cd,$a8,$54,$85
gc_sprite_col		!byte $02,$0a,$0a,$05,$0c,$0c,$0c,$0c
gc_sprite_def		!byte $f0,$d8,$dc,$ec,$cf,$c7,$d3,$e8
gc_sprite_anims		!byte $f0,$d8,$dc,$ec,$cf,$c7,$d3,$e8
gc_sprite_anime		!byte $f5,$d9,$dd,$ee,$d3,$cb,$d7,$eb

; hide sprites off screen

sprite_x_hide		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_y_hide		!byte $00,$00,$00,$00,$00,$00,$00,$00

; level 1 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l1_sprite_x		!byte $19,$20,$4c,$94,$90,$6f,$5a,$0d
l1_sprite_y		!byte $cd,$9e,$4f,$a4,$cd,$a8,$54,$85
l1_sprite_col	!byte $0a,$07,$0c,$05,$0c,$0c,$0c,$0c
l1_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$d3,$e8
l1_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$d3,$e8
l1_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$d7,$eb
l1_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$00,$00
l1_sprite_spd	!byte $00,$00,$02,$03,$01,$02,$01,$02
l1_sprite_min	!byte $00,$00,$53,$53,$19,$5a,$5a,$0d
l1_sprite_max	!byte $00,$00,$b8,$a7,$90,$85,$85,$3a

; level 2 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l2_sprite_x		!byte $12,$80,$18,$94,$9b,$8b,$68,$0d
l2_sprite_y		!byte $cd,$59,$55,$b6,$cd,$34,$5e,$1b
l2_sprite_col	!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l2_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l2_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l2_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l2_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$00,$00
l2_sprite_spd	!byte $00,$00,$02,$01,$01,$02,$02,$03
l2_sprite_min	!byte $00,$00,$55,$82,$0d,$0d,$2a,$0d
l2_sprite_max	!byte $00,$00,$8d,$b6,$9b,$8b,$93,$9f

; level 3 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l3_sprite_x		!byte $10,$10,$58,$70,$4b,$8c,$33,$33
l3_sprite_y		!byte $cd,$7c,$5b,$af,$cd,$3b,$5c,$9c
l3_sprite_col	!byte $0a,$07,$05,$05,$0c,$0c,$0c,$0c
l3_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l3_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l3_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l3_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l3_sprite_spd	!byte $00,$00,$03,$03,$02,$03,$02,$03
l3_sprite_min	!byte $00,$00,$5b,$5b,$3b,$3c,$0f,$0f
l3_sprite_max	!byte $00,$00,$af,$af,$9f,$8c,$33,$33

; level 4 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l4_sprite_x		!byte $10,$46,$24,$30,$4b,$8c,$3d,$8f
l4_sprite_y		!byte $cd,$9d,$5b,$af,$cd,$32,$54,$9f
l4_sprite_col	!byte $0a,$07,$0c,$05,$0c,$0c,$0c,$0c
l4_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l4_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l4_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l4_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l4_sprite_spd	!byte $00,$00,$03,$03,$02,$04,$03,$04
l4_sprite_min	!byte $00,$00,$5b,$5b,$10,$10,$3d,$3f
l4_sprite_max	!byte $00,$00,$af,$af,$9f,$9f,$9f,$9f

; level 5 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l5_sprite_x		!byte $10,$9c,$20,$50,$4b,$8c,$3d,$8f
l5_sprite_y		!byte $cd,$cb,$5b,$82,$cd,$32,$5d,$a3
l5_sprite_col	!byte $0a,$07,$0c,$05,$0c,$0c,$0c,$0c
l5_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l5_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l5_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l5_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l5_sprite_spd	!byte $00,$00,$03,$02,$02,$04,$02,$02
l5_sprite_min	!byte $00,$00,$5b,$50,$10,$10,$5f,$65
l5_sprite_max	!byte $00,$00,$92,$82,$9f,$9f,$8d,$98

; level 6 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l6_sprite_x		!byte $10,$9e,$34,$78,$4b,$8c,$3d,$55
l6_sprite_y		!byte $cd,$c5,$7c,$a5,$cd,$32,$5a,$83
l6_sprite_col	!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l6_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l6_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l6_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l6_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l6_sprite_spd	!byte $00,$00,$03,$03,$02,$04,$05,$02
l6_sprite_min	!byte $00,$00,$7c,$7c,$10,$10,$10,$44
l6_sprite_max	!byte $00,$00,$b3,$a5,$9f,$9f,$9f,$65

; level 7 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l7_sprite_x		!byte $10,$87,$2d,$4d,$4b,$69,$3d,$8c
l7_sprite_y		!byte $cd,$1d,$58,$95,$cd,$95,$5d,$32
l7_sprite_col	!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l7_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l7_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l7_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l7_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l7_sprite_spd	!byte $00,$00,$03,$03,$02,$03,$03,$04
l7_sprite_min	!byte $00,$00,$58,$85,$10,$5e,$3f,$10
l7_sprite_max	!byte $00,$00,$7c,$b1,$9f,$7c,$9f,$9f

; level 8 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l8_sprite_x		!byte $10,$84,$1c,$90,$4b,$8c,$4f,$68
l8_sprite_y		!byte $c5,$bd,$5d,$82,$c5,$32,$69,$95
l8_sprite_col	!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l8_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l8_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l8_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l8_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l8_sprite_spd	!byte $00,$00,$03,$03,$01,$04,$03,$02
l8_sprite_min	!byte $00,$00,$5d,$54,$3b,$10,$28,$30
l8_sprite_max	!byte $00,$00,$a8,$b6,$59,$9f,$80,$82

; level 9 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l9_sprite_x		!byte $10,$48,$1c,$90,$4b,$69,$3d,$8c
l9_sprite_y		!byte $cd,$83,$78,$93,$4e,$cb,$83,$32
l9_sprite_col	!byte $0a,$07,$0c,$0c,$0c,$0c,$05,$0c
l9_sprite_def	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l9_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l9_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l9_sprite_dir	!byte $00,$00,$00,$01,$01,$01,$01,$00
l9_sprite_spd	!byte $00,$00,$03,$03,$02,$03,$03,$04
l9_sprite_min	!byte $00,$00,$6a,$6a,$1f,$10,$41,$10
l9_sprite_max	!byte $00,$00,$b5,$a8,$9f,$9f,$7f,$9f

; level 10 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l10_sprite_x		!byte $10,$48,$28,$80,$4b,$8c,$60,$55
l10_sprite_y		!byte $cd,$a7,$53,$73,$cd,$32,$a6,$7d
l10_sprite_col		!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l10_sprite_def		!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l10_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l10_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l10_sprite_dir		!byte $00,$00,$00,$01,$01,$01,$01,$00
l10_sprite_spd		!byte $00,$00,$03,$03,$02,$04,$03,$03
l10_sprite_min		!byte $00,$00,$53,$53,$10,$10,$44,$3c
l10_sprite_max		!byte $00,$00,$aa,$8d,$9f,$9f,$7b,$6c

; level 11 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l11_sprite_x		!byte $10,$5c,$28,$80,$4b,$8c,$60,$55
l11_sprite_y		!byte $cd,$ca,$53,$73,$cd,$32,$a1,$77
l11_sprite_col		!byte $0a,$07,$0c,$0c,$0c,$0c,$0c,$0c
l11_sprite_def		!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l11_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l11_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l11_sprite_dir		!byte $00,$00,$00,$01,$01,$01,$01,$00
l11_sprite_spd		!byte $00,$00,$04,$02,$02,$04,$03,$02
l11_sprite_min		!byte $00,$00,$56,$56,$10,$10,$46,$37
l11_sprite_max		!byte $00,$00,$a2,$80,$9f,$9f,$9f,$67

; level 12 sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

; sprites - player, cross, spider1, spider2, zombie, ghost, bat, witch

l12_sprite_x		!byte $10,$a0,$78,$94,$4b,$8c,$60,$80
l12_sprite_y		!byte $cd,$cb,$83,$64,$cd,$32,$a2,$5a
l12_sprite_col		!byte $0a,$07,$05,$05,$0c,$0c,$0c,$0c
l12_sprite_def		!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l12_sprite_anims	!byte $ab,$bf,$ee,$ec,$cf,$c7,$e0,$e8
l12_sprite_anime	!byte $ac,$c0,$f0,$ee,$d3,$cb,$e4,$ec
l12_sprite_dir		!byte $00,$00,$00,$00,$01,$01,$01,$00
l12_sprite_spd		!byte $00,$00,$02,$03,$02,$04,$04,$03
l12_sprite_min		!byte $00,$00,$83,$64,$10,$10,$10,$38
l12_sprite_max		!byte $00,$00,$c0,$a0,$9f,$9f,$6b,$85

; enemy movement animation defs

left_enm_def	!byte $00,$00,$ee,$ec,$cf,$c7,$e0,$e4
left_enm_anims	!byte $00,$00,$ee,$ec,$cf,$c7,$e0,$e4
left_enm_anime	!byte $00,$00,$f0,$ee,$d3,$cb,$e4,$e8

rite_enm_def	!byte $00,$00,$ee,$ec,$cb,$c3,$d3,$e8
rite_enm_anims	!byte $00,$00,$ee,$ec,$cb,$c3,$d3,$e8
rite_enm_anime	!byte $00,$00,$f0,$ee,$ce,$c7,$d7,$ec	

; level 1 cross positions

l1_cross_x		!byte $20,$43,$8f,$39,$6e,$9f,$11,$79,$40,$37,$a3,$11,$8f,$6d,$6e,$a1,$70,$20,$8c,$39
l1_cross_y		!byte $9e,$50,$7f,$cd,$40,$90,$b6,$86,$6a,$b6,$ba,$86,$ae,$95,$57,$65,$98,$9e,$cb,$86

; level 2 cross positions

l2_cross_x		!byte $80,$38,$0e,$38,$31,$9f,$42,$5d,$9f,$0e,$24,$5f,$50,$9f,$0b,$0b,$5d,$86,$82,$29
l2_cross_y		!byte $59,$cb,$a2,$5f,$cd,$cb,$1b,$cb,$1b,$cb,$8a,$1b,$9e,$8b,$6a,$1b,$8c,$1b,$8b,$3a

; level 3 cross positions

l3_cross_x		!byte $10,$51,$8e,$8d,$4c,$10,$3d,$58,$10,$65,$10,$64,$10,$7c,$4c,$71,$7c,$4c,$79,$4c
l3_cross_y		!byte $7c,$cd,$3b,$cd,$8b,$5c,$cd,$1c,$9c,$cd,$15,$8b,$bc,$6c,$ac,$1b,$ac,$1c,$cd,$6a

; level 4 cross positions

l4_cross_x		!byte $46,$78,$31,$9e,$38,$10,$68,$9e,$0d,$13,$7a,$29,$9c,$9c,$53,$2a,$3e,$7e,$2a,$69
l4_cross_y		!byte $9d,$1e,$cd,$76,$c4,$1e,$51,$ce,$87,$4b,$ce,$1e,$99,$1e,$5b,$93,$7c,$a1,$54,$7c

; level 5 cross positions

l5_cross_x		!byte $9c,$2c,$99,$0e,$9f,$36,$42,$67,$72,$6b,$89,$0e,$12,$86,$30,$84,$6c,$10,$3c,$60
l5_cross_y		!byte $cb,$85,$87,$63,$1b,$9b,$1b,$bc,$1b,$94,$1b,$98,$21,$51,$32,$c6,$5e,$c4,$56,$1b

; level 6 cross positions

l6_cross_x		!byte $9e,$16,$78,$80,$46,$76,$0e,$0e,$66,$9f,$0e,$9e,$2c,$46,$96,$36,$96,$4b,$0a,$56
l6_cross_y		!byte $c5,$55,$b9,$1d,$cb,$35,$a8,$1d,$6a,$1d,$7b,$7b,$1d,$a8,$55,$35,$9a,$6a,$c9,$35

; level 7 cross positions

l7_cross_x		!byte $87,$0d,$0d,$54,$56,$0d,$7a,$36,$58,$94,$97,$3f,$a2,$57,$10,$6a,$9c,$94,$26,$7b
l7_cross_y		!byte $1d,$c0,$53,$1d,$cc,$1d,$4f,$81,$51,$ca,$39,$55,$7b,$8c,$8c,$cc,$5d,$a8,$35,$81

; level 8 cross positions

l8_cross_x		!byte $84,$58,$0c,$5a,$0f,$32,$3c,$4a,$24,$9e,$a0,$a1,$10,$9e,$5e,$0a,$a1,$4e,$70,$28
l8_cross_y		!byte $bd,$94,$5c,$1c,$bd,$94,$54,$ad,$1c,$4a,$c2,$68,$1c,$1c,$5d,$93,$94,$68,$1c,$68

; level 9 cross positions

l9_cross_x		!byte $48,$2e,$0a,$7c,$60,$0e,$a2,$78,$5c,$a2,$2a,$78,$0a,$6c,$48,$9f,$30,$92,$a2,$0a
l9_cross_y		!byte $83,$32,$cb,$1e,$b8,$1e,$ad,$83,$1e,$6b,$88,$a4,$8c,$65,$a4,$32,$cb,$cb,$8c,$53

; level 10 cross positions

l10_cross_x		!byte $48,$2c,$11,$a0,$42,$9b,$12,$57,$8a,$0b,$6f,$54,$9e,$10,$31,$72,$28,$a0,$74,$76
l10_cross_y		!byte $a7,$1e,$cd,$a7,$5f,$56,$80,$1e,$cd,$98,$7e,$cd,$1e,$3f,$cd,$5f,$98,$71,$1e,$96

; level 11 cross positions

l11_cross_x		!byte $5c,$4c,$12,$53,$0e,$3e,$7a,$86,$98,$80,$a2,$0f,$4c,$98,$6e,$98,$73,$3c,$5e,$14
l11_cross_y		!byte $ca,$a1,$55,$1d,$a1,$5d,$a1,$1d,$58,$ca,$1d,$ca,$5d,$a1,$1d,$ca,$5d,$1d,$78,$1d

; level 12 cross positions

l12_cross_x		!byte $a0,$a2,$0e,$47,$0e,$37,$2c,$4e,$77,$82,$25,$0e,$78,$60,$8b,$60,$50,$2d,$0b,$9e
l12_cross_y		!byte $cb,$55,$1d,$cb,$61,$5a,$cb,$93,$32,$86,$44,$a1,$c2,$5a,$1d,$82,$32,$79,$cb,$1d

; byte stores that are updated during play

; *main* sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern

sprite_x		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_y		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_col		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_def		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_anims	!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_anime	!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_dir		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_spd		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_min		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_max		!byte $00,$00,$00,$00,$00,$00,$00,$00

; sprite store that is populated when girl or boy player
; is selected - player movement routines then pull data
; in from here!

playwalkleft_def	!byte $00
playwalkleft_anims	!byte $00
playwalkleft_anime	!byte $00

playwalkright_def	!byte $00
playwalkright_anims	!byte $00
playwalkright_anime	!byte $00

playjumpleft_def	!byte $00
playjumpleft_anims	!byte $00
playjumpleft_anime	!byte $00

playjumpright_def	!byte $00
playjumpright_anims	!byte $00
playjumpright_anime	!byte $00

playstatic_def		!byte $00
playstatic_anims	!byte $00
playstatic_anime	!byte $00

playfall_def		!byte $00
playfall_anims		!byte $00
playfall_anime		!byte $00

; *buffer* sprite default sprite positions, colours and definitions
; animations start/stop, direction, speed and max/min move pattern
; used during level setup only

sprite_x_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_y_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_col_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_def_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_anims_buf	!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_anime_buf	!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_dir_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_spd_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_min_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00
sprite_max_buf		!byte $00,$00,$00,$00,$00,$00,$00,$00

; sprite positions, colours and definitions - title screen bottom

t_sprite_x		!byte $00,$00,$00,$00,$00,$00,$00,$00
t_sprite_y		!byte $00,$00,$00,$00,$00,$00,$00,$00
t_sprite_col	!byte $00,$00,$00,$00,$00,$00,$00,$00
t_sprite_def	!byte $00,$00,$00,$00,$00,$00,$00,$00
t_sprite_anims	!byte $00,$00,$00,$00,$00,$00,$00,$00
t_sprite_anime	!byte $00,$00,$00,$00,$00,$00,$00,$00

; cross x-y positions in game copied to during level setup

cross_x		!byte $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
cross_y		!byte $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00


; byte store tables for various things in the next section
; these stores are populated during the game

; byte store for get ready message printing

get_ready_des1 !scr "                                        "	
get_ready_des2 !scr "                                        "	
get_ready_des3 !scr "                                        "

; byte store for current and high scores and cross counter

score		!byte $00,$00,$00,$00,$00,$00
high_score	!byte $00,$00,$01,$00,$00,$00
cross_score	!byte $00,$00

; status bar screen and colour data

status_screen_ram
!byte $20,$13,$03,$0F,$12,$05,$20,$20,$20,$20,$20,$20,$20,$20,$03,$12,$0F,$13,$13,$05
!byte $13,$20,$20,$20,$20,$08,$09,$2D,$13,$03,$0F,$12,$05,$20,$20,$20,$20,$20,$20,$20
!byte $05,$0E,$05,$12,$07,$19,$20,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40
!byte $40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40,$40 

status_colour_ram
!byte $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$00,$07,$07,$07,$07,$07,$07
!byte $07,$00,$07,$07,$00,$01,$01,$01,$01,$01,$01,$01,$01,$00,$01,$01,$01,$01,$01,$01
!byte $02,$02,$02,$02,$02,$02,$01,$02,$02,$02,$02,$02,$07,$07,$07,$07,$07,$07,$05,$05
!byte $05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05

; raindrop pattern data

rain_patt	!byte $ff,$ef,$ff,$ff,$ef,$ff,$ef,$ff

; temporary store for downward animating chars

chardown_temp	!byte $00

; cinema screen 'noise' anim pattern data

scrn0_patt		!byte $54,$ae,$55,$8a,$d5,$aa,$5d,$b0
scrn1_patt		!byte $a2,$55,$8e,$54,$2a,$51,$ba,$15

; sfx data (lower the sfx data, the higher play priority it has)

sfx_jump
        !byte $38,$F6,$04,$A0,$21,$A0,$A0,$A6,$A6,$A6,$20,$A9,$A9,$A9,$A2,$A2
        !byte $A2,$A6,$A6,$A6,$A9,$A9,$A9,$A2,$A2,$A2,$A6,$A6,$A6,$A9,$A9,$A9
        !byte $A2,$A2,$A2,$A6,$A6,$A6,$A9,$A9,$A9,$00
       
sfx_collect
        !byte $00,$89,$04,$A2,$41,$A2,$A2,$A6,$A6,$A6,$40,$A9,$A9,$A9,$A2,$A2
        !byte $A2,$A6,$A6,$A6,$A9,$A9,$A9,$A2,$A2,$A2,$A6,$A6,$A6,$A9,$A9,$A9
        !byte $A2,$A2,$A2,$A6,$A6,$A6,$A9,$A9,$A9,$00

; includes to import additional data

		!src "includes/screen_data.asm"			; screen / colour ram data
		!src "includes/chartype_data.asm"		; chartype for collision detection
		!src "includes/level_setup_data.asm"	; level setup subroutines data
		
; end of code!!! ========================================================================
