/******************************************************************
 * dtvmkfs - Make DTV flash filesystem                            *
 ******************************************************************
 * $Id: zipfile.c,v 1.5 2008-10-13 20:13:04 spiff Exp $
 * Access to ZIP archives                                         *
 ******************************************************************
 * Author: Mikkel Holm Olsen (dtv@symlink.dk)                     *
 ******************************************************************
 * dtvmkfs is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License    *
 * as published by the Free Software Foundation; either version 2 *
 * of the License, or (at your option) any later version.         *
 *                                                                *
 * dtvmkfs is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  *
 * GNU General Public License for more details.                   *
 *                                                                *
 * You should have received a copy of the GNU General Public      *
 * License along with dtvmkfs; if not, write to the Free Software *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,         *
 * MA  02110-1301  USA                                            *
 ******************************************************************
 * Although dtvmkfs is released under the terms of the GPL, this  *
 * part uses zlib and the unzip-library (by Gilles Vollant)       *
 * shipped with zlib. The license of both these libraries can be  *
 * found in the zlib directory, which contains an UNMODIFIED      *
 * version of the zlib source distribution.                       *
 ******************************************************************/

// Include our own header
#include "zipfile.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>

/* #ifdef unix */
# include <unistd.h>
# include <utime.h>
# include <sys/stat.h>
# include <sys/types.h>
/* #else
# include <direct.h>
# include <io.h>
#endif
*/

#include "unzip.h" // The unzip-library by Gilles Vollant (uses zlib)

#ifdef WIN32
#define USEWIN32IOAPI
#include "iowin32.h"
zlib_filefunc_def my_ffunc;
#define my_init_function() fill_win32_filefunc(&my_ffunc);
#define my_open_file(x) unzOpen2(x,&my_ffunc)
#else
#define my_init_function() /* Do nothing */
#define my_open_file(x) unzOpen(zipfilename)
#endif


#define ZIPDATABUF_SIZE 8192

char zipdatabuf[ZIPDATABUF_SIZE], *nextlineptr=NULL;
int zipdatalen = 0;
unzFile uf=NULL;

char* zip_buf_get_line(void) {
  char *ptr, *retptr;
  int nbytes;
  if (!nextlineptr) { // We must fill the buffer
    nextlineptr=zipdatabuf;
    zipdatalen=0;
  }

 ZIP_RETRY:
  // Find first newline
  for (ptr=nextlineptr;(*ptr!='\n')&&(ptr<zipdatabuf+zipdatalen);++ptr) {
    if (*ptr=='\r') *ptr=0; // Strip CR (for Windows combatability)
  }
  if (*ptr=='\n') { // Found a newline
    *ptr++=0; // Replace newline with 0-termination
    retptr=nextlineptr;
    nextlineptr=ptr;
    return retptr;
  } else { // Buffer needs to be re-filled
    if ((nextlineptr!=zipdatabuf)&&(nextlineptr<zipdatabuf+zipdatalen)) {
      zipdatalen-=(nextlineptr-zipdatabuf);
      memmove(zipdatabuf, nextlineptr, zipdatalen+1); // Move remaining data
      nextlineptr=zipdatabuf;
    } else {
      if (zipdatalen>=ZIPDATABUF_SIZE-1) {
        // Line does not fit in buffer. Return partial line.
        retptr=nextlineptr;
        nextlineptr=NULL;
        return (*retptr?retptr:NULL);
      }
    }

    nbytes=unzReadCurrentFile(uf,zipdatabuf+zipdatalen,
                              ZIPDATABUF_SIZE-zipdatalen-1);
    zipdatalen+=nbytes;
    if (nbytes>=0) zipdatabuf[zipdatalen]=0; // Ensure 0-termination
    if (nbytes<=0) { // No more data to read
      if (zipdatalen) { // Last line has no LF
        retptr=nextlineptr;
        nextlineptr=NULL;
        return retptr; 
      }
      else return NULL;
    }
    goto ZIP_RETRY;
  }
}

void zip_initialize(void) {
  my_init_function(); // For portability to Windows...
}

int zip_open_archive(const char *zipfilename) {
  if (!(uf=my_open_file(zipfilename))) {
    fprintf(stderr, "Error opening %s.\n",zipfilename);
    return -1;
  }
 
  return 0;
}

int zip_close_archive(void) {
  return unzClose(uf);
}

int zip_open_file(const char *filename, uint32_t *crc32) {
  unz_global_info gi;
  unz_file_info file_info;
  int err;
  
  if (UNZ_OK!=(err=unzGetGlobalInfo(uf,&gi))) {
    fprintf(stderr, "Error %d with zipfile in unzGetGlobalInfo.\n",err);
    return 1;
  }
  
  if (UNZ_OK!=(err=unzLocateFile(uf,filename,2))) { // 2: Not case sensitive
    fprintf(stderr,"Could not find %s in archive\n",filename);
    return 2;
  }
  
  if (UNZ_OK!=(err=unzGetCurrentFileInfo(uf,&file_info,NULL,0,NULL,0,NULL,0))) {
    fprintf(stderr,"Error %d with zipfile in unzGetCurrentFileInfo\n",err);
    return 3;
  }
  
  if (crc32)
    *crc32=file_info.crc;
  
  if (UNZ_OK!=(err=unzOpenCurrentFile(uf))) {
    fprintf(stderr,"Error %d with zipfile in unzOpenCurrentFile\n",err);
    return 4;
  }
  
  return 0;
}

int zip_get_file_size(void) {
  int err;
  unz_file_info file_info;
  if (UNZ_OK!=(err=unzGetCurrentFileInfo(uf,&file_info,NULL,0,NULL,0,NULL,0))) {
    fprintf(stderr,"Error %d with zipfile in unzGetCurrentFileInfo\n",err);
    return -1;
  }
  return file_info.uncompressed_size;
}

int zip_read_buffer(char *buf, int maxbytes) {
  return unzReadCurrentFile(uf,buf,maxbytes);
}

int zip_close_file(void) {
  nextlineptr=NULL;
  zipdatalen=0;
  return unzCloseCurrentFile(uf);
}
