/******************************************************************
 * dtvmkfs - Make DTV flash filesystem                            *
 ******************************************************************
 * $Id: dirlist.c,v 1.7 2008-12-06 17:51:19 spiff Exp $
 * Create C64 directory listing                                   *
 ******************************************************************
 * Author: Mikkel Holm Olsen (dtv@symlink.dk)                     *
 ******************************************************************
 * dtvmkfs is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License    *
 * as published by the Free Software Foundation; either version 2 *
 * of the License, or (at your option) any later version.         *
 *                                                                *
 * dtvmkfs is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  *
 * GNU General Public License for more details.                   *
 *                                                                *
 * You should have received a copy of the GNU General Public      *
 * License along with dtvmkfs; if not, write to the Free Software *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,         *
 * MA  02110-1301  USA                                            *
 ******************************************************************/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "flash.h"
#include "dirlist.h"

char default_header[] = "DTV FLASH";

struct dirlist_t {
  char *c64name;
  int size;
  struct dirlist_t *next;
} *dirlist=NULL, *lastdir=NULL;

int memaddr=0;
//int dirwidth=16;
int maxwidth=0;

/* dir_start - Start writing a C64 directory listing to mem, with
     width characters reserved for each filename. Internal memory
     starts at 0x0801. Returns address of next line. */
unsigned char *dir_start(unsigned char *mem, const char *header, int width) {
  unsigned char *startaddr=mem;
  int i;
  memaddr=0x0801;
  mem+=2; /* Skip the next-pointer for now */
  flash_set_2_bytes(mem,0); /* Line number */
  mem+=2;

  i=snprintf((char*)mem,width+3,"\x12\"%s",header);
  if (i>width+2) {
    i=width+2;
    fprintf(stderr,"Warning: header truncated.\n");
  }
  while (i<width+2) {
    mem[i++]=0x20;
  }
  mem+=i;

  memcpy(mem,"\" SP\x8B\x46",7);
  mem+=7;
  memaddr+=mem-startaddr;
  flash_set_2_bytes(startaddr,memaddr);
  
  /* Terminate the list */
  mem[0]=mem[1]=0;
  return mem;
}

/* dir_add_entry - Add an entry to the directory listing in mem.
     filesize is the BASIC line number (i.e. filesize in blocks),
     filename is the name of the file. Returns the address of next
     line. */
unsigned char *dir_add_entry(unsigned char *mem, int filesize, 
                             const char *filename, int width) {
  unsigned char *startaddr=mem;
  int i;
  if (!memaddr) return mem; /* Abort if memaddr not set */

  mem+=2; /* Skip the next-pointer for now */
  flash_set_2_bytes(mem,filesize); /* Line number */
  mem+=2;
  i=0;
  do {
    i++; /* How many digits? */
    filesize/=10;
  } while (filesize>0);
  i=(i<4?4-i:0);
  memset(mem,0x20,i); /* Space padding */
  mem+=i;

  /* Put the filename */
  i=snprintf((char*)mem,width+2,"\"%s",filename);
  if (i>width+1) {
    i=width+1;
    fprintf(stderr,"Warning: filename %s truncated.\n",filename);
  }
  mem[i++]=0x22; /* Trailing quote */
  while (i<width+2) {
    mem[i++]=0x20;
  }
  mem+=i;
  memcpy(mem,":PRG",5); /* Adds a trailing 0 */
  mem+=5;

  /* Update the next-pointer */
  memaddr+=mem-startaddr;
  flash_set_2_bytes(startaddr,memaddr);

  /* Terminate the list */
  mem[0]=mem[1]=0;
  return mem;
}


unsigned char *dir_add_blocksfree(unsigned char *mem, int freeblocks) {
  unsigned char *startaddr=mem;
  if (!memaddr) return mem; /* Abort if memaddr not set */

  mem+=2; /* Skip the next-pointer for now */
  flash_set_2_bytes(mem,freeblocks); /* Line number */
  mem+=2;
  
  memcpy(mem,"BLOCKS FREE.",13); /* Adds a trailing 0 */
  mem+=13;

  /* Update the next-pointer */
  memaddr+=mem-startaddr;
  flash_set_2_bytes(startaddr,memaddr);

  /* Terminate the list */
  mem[0]=mem[1]=0;
  return mem;
}

void dir_add_to_list(int filesize, char *filename) {
  int len;
  struct dirlist_t *newentry;
  newentry=malloc(sizeof(struct dirlist_t));
  newentry->size=filesize;
  newentry->c64name=filename;
  len=strlen(filename);
  if (len>24) {
    filename[24]=0; // Make sure filename is not too long
    len=24;
  }
  if (len>maxwidth) {
    maxwidth=len;
  }
  newentry->next=NULL;
  if (!dirlist) dirlist=newentry;
  if (lastdir) lastdir->next=newentry;
  lastdir=newentry;
}

int get_actual_width(int w, int maxw) {
  if (w==0)
    w=16; // Default width is 16 chars
  if (maxw>w)
    w=maxw;
  if (w>24)
    w=24;
  if (w<8)
    w=8;
  return w;
}

unsigned char *membuf=NULL;
int membufsize=0, dirsize=0;

int dir_build_listing(char *header, int width, int freeblocks) {
  unsigned char *oldptr, *memptr;
  int oldsize, new_width;
  struct dirlist_t *ptr=dirlist;

  new_width=get_actual_width(width, maxwidth);
  if ((width!=0)&&(new_width!=width)) {
    fprintf(stderr, "WARNING: Expanding directory width to %d.\n",new_width);
  }
  width=new_width;

  membufsize=1024;
  memptr=membuf=malloc(membufsize);
  if (!memptr) return -1;
  memptr=dir_start(memptr,header?header:default_header,width);

  while (ptr) {
    if (membufsize-100<memptr-membuf) { // Must have 100 bytes free
      oldsize=membufsize;
      oldptr=membuf;
      membufsize+=1024; // Grab another kilobyte of memory
      membuf=realloc(oldptr, membufsize);
      if (!membuf) {
        free(oldptr);
        return -1;
      }
      if (membuf!=oldptr) { // Damn... The buffer has moved
        memptr+=membuf-oldptr;
      }
      memset(membuf+oldsize,0,membufsize-oldsize); // Clear the buffer
    }
    // Now, add the directory entry
    memptr=dir_add_entry(memptr,ptr->size, ptr->c64name, width);
    ptr=ptr->next;
  }
  memptr=dir_add_blocksfree(memptr, freeblocks); 
  dirsize=memptr-membuf+2;
  return dirsize;
}

void dir_print_listing(char *header, int width, int freeblocks) {
  struct dirlist_t *ptr=dirlist;
  int i;
  char *hdr=header?header:default_header;
  width=get_actual_width(width, maxwidth);
  if (strlen(hdr)>width)
    hdr[width]=0; // Terminate header if it is too long

  printf("0 \"%s",hdr);
  for (i=strlen(hdr);i<width;i++) printf(" ");
  printf("\" SPIFF\n");
  while (ptr) {
    if (strlen(ptr->c64name)>width)
      ptr->c64name[width]=0;
    printf("%-4d \"%s\"",ptr->size,ptr->c64name);
    for (i=strlen(ptr->c64name);i<width;i++) printf(" ");
    printf(":PRG\n");
    ptr=ptr->next;
  }
  printf("%d BLOCKS FREE.\n",freeblocks);
}

unsigned char *dir_return_buffer_ptr(void) {
  return membuf;
}

int dir_save_buffer_to_prg(char *filename) {
  FILE *fp;
  if (!(fp=fopen(filename,"w"))) {
    fprintf(stderr, "Error opening %s for writing.\n",filename);
    return 1;
  }
  fwrite("\x01\x08",1,2,fp); // Prepend load address
  fwrite(membuf,1,dirsize,fp);
  fclose(fp);
  return 0;
}

void dir_free_listing_buffer(void) {
  free(membuf);
  membuf=NULL;
  membufsize=dirsize=0;
}
