/******************************************************************
 * dtvmkfs - Make DTV flash filesystem                            *
 ******************************************************************
 * $Id: dtvpack.c,v 1.3 2007-03-23 21:37:14 spiff Exp $
 * Compression and decompression of dtvpacked data                *
 ******************************************************************
 * Author: Mikkel Holm Olsen (dtv@symlink.dk)                     *
 ******************************************************************
 * dtvmkfs is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License    *
 * as published by the Free Software Foundation; either version 2 *
 * of the License, or (at your option) any later version.         *
 *                                                                *
 * dtvmkfs is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  *
 * GNU General Public License for more details.                   *
 *                                                                *
 * You should have received a copy of the GNU General Public      *
 * License along with dtvmkfs; if not, write to the Free Software *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,         *
 * MA  02110-1301  USA                                            *
 ******************************************************************
 * Portions of dtvpack.c are based on Daniel Kahlin's dtvpack:    *
 *                                                                *
 *   dtvpack - C64 DTV flash file encoder/decoder                 *
 *                                                                *
 *   Copyright (c) 2006 Daniel Kahlin <daniel@kahlin.net>         *
 *   All rights reserved.                                         *
 *                                                                *
 *   Redistribution and use in source and binary forms, with or   *
 *   without modification, are permitted provided that the        *
 *   following conditions are met:                                *
 *   1. Redistributions of source code must retain the above      *
 *      copyright notice, this list of conditions and the         *
 *      following disclaimer.                                     *
 *   2. Redistributions in binary form must reproduce the above   *
 *      copyright notice, this list of conditions and the         *
 *      following disclaimer in the documentation and/or other    *
 *      materials provided with the distribution.                 *
 *   3. Neither the names of the copyright holders nor the names  *
 *      of their contributors may be used to endorse or promote   *
 *      products derived from this software without specific      *
 *      prior written permission.                                 *
 *                                                                *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND       *
 *   CONTRIBUTORS ``AS IS´´ AND ANY EXPRESS OR IMPLIED            *
 *   WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED       *
 *   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A              *
 *   PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE    *
 *   REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,  *
 *   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES     *
 *   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 *   GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR         *
 *   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF   *
 *   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT    *
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT   *
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE          *
 *   POSSIBILITY OF SUCH DAMAGE.                                  *
 ******************************************************************/

#include <stdlib.h>
#include <stdio.h>

#define PK_WINDOW (0x100)
#define PK_RUNLEN (0x7f)
#define PK_MAXMATCH (0x7f)

int dtv_outpos=0, dtv_error=0, dtv_dstlen=0;
unsigned char *dtv_outbuf=NULL;

void dtv_output_set(unsigned char *dstbuf, int dstlen) {
  dtv_error=dtv_outpos=0;
  dtv_dstlen=dstlen;
  dtv_outbuf=dstbuf;
}

void dtv_put_byte(unsigned char data) {
  if (dtv_outpos<=dtv_dstlen) { // Check buffer size
    dtv_outbuf[dtv_outpos++]=data;
  } else {
    dtv_error=1;
  }
}


int dtv_pack(const unsigned char *srcbuf, int srclen,
             unsigned char *dstbuf, int dstlen) {
  int srcoffs;
  unsigned char buf[PK_RUNLEN];
  int len;
  int i;

  dtv_output_set(dstbuf, dstlen);
  
  srcoffs=0;
  len=0;
  while (1) {
    int st, ed;
    int match_len, match_offs;
    /* find matches */
    st=(srcoffs-PK_WINDOW > 0)?srcoffs-PK_WINDOW:0;
    ed=srcoffs;
    match_offs=0;
    match_len=0;
    for (i=st; i < ed; ++i) {
      int offs=i;
      int mlen=0;
      while (srcbuf[i+mlen] == srcbuf[srcoffs+mlen]) {
        mlen++;
        if (srcoffs+mlen == srclen)
          break; // File ends
        if (mlen == PK_MAXMATCH)
          break; // Buffer full
      }
      if (mlen > match_len) {
        match_offs=offs;
        match_len=mlen;
      }
    }
    
    /* determine if the match was good enough */
    if (match_len > 2) {
      /* flush */
      if (len>0) {
        dtv_put_byte(len);
        for (i=0; i<len; ++i)
          dtv_put_byte(buf[i]);
        len=0;
      }
      /* write a match code */
      dtv_put_byte(0x80 | match_len);
      dtv_put_byte(0x100 + (match_offs-srcoffs));    
      srcoffs+=match_len; /* one already incremented */
    } else {
      /* store a byte */
      int data;
      data=srcbuf[srcoffs];
      srcoffs++;
      
      buf[len]=data;
      len++;
      /* flush if max run length */
      if (len==PK_RUNLEN) {
        dtv_put_byte(len);
        for (i=0; i<len; ++i)
          dtv_put_byte(buf[i]);
        len=0;
      }
    }
    
    /* are we all finished? */
    if (srcoffs>=srclen) {
      break;
    }
    
  }
  /* flush */
  if (len>0) {
    dtv_put_byte(len);
    for (i=0; i<len; ++i)
      dtv_put_byte(buf[i]);
  }
  /* end marker */
  dtv_put_byte(0x00);
  
  return (dtv_error?-1:dtv_outpos);
}


#define DP_WINDOW (0x100)

unsigned char dp_buf[DP_WINDOW];
int dp_bfind;
int dp_chunkoffs;
int insize=0, outsize=0;
unsigned char *inptr=NULL;

inline unsigned char dtv_dp_get(void) {
  insize++;
  return *inptr++;
}

inline unsigned char dtv_dp_getbr(int offs) {
  unsigned char data;
  if ( (dp_bfind+offs) < 0 ) {
    fprintf(stderr,"ERROR: back-reference before start of file "
            "at offset %d\n",dp_chunkoffs);
    exit(-1);
  }
  data=dp_buf[(dp_bfind+offs) % DP_WINDOW];
  return data;
}

inline void dtv_dp_put(unsigned char data) {
    outsize++;
    dp_buf[dp_bfind % DP_WINDOW]=data;
    dp_bfind++;
}


int dtv_get_sizes(unsigned char *buf, int *compsize, int *realsize) {
  int i;
  unsigned char code,len,offs,data;
  inptr=buf;
  insize=outsize=0;
  dp_bfind=0;
  while (1) {
    dp_chunkoffs=insize;
    code=dtv_dp_get();

    /* end marker? */
    if (code == 0x00)
      break;
    
    len = code & 0x7f;
    /* sequence marker? */
    if (code & 0x80) {
      /* yes, look at back-reference */
      offs=dtv_dp_get();
      for (i=0; i < len; ++i) {
        data=dtv_dp_getbr( -0x100+offs );
        dtv_dp_put(data);
      }
    } else {
      /* no, just copy data */
      for (i=0; i < len; ++i) {
        data=dtv_dp_get();
        dtv_dp_put(data);
      }  
    }
  }
  if (compsize) *compsize=insize;
  if (realsize) *realsize=outsize;
  return 0;
}
