/******************************************************************
 * dtvmkfs - Make DTV flash filesystem                            *
 ******************************************************************
 * $Id: clopts.c,v 1.12 2008-12-06 21:44:46 spiff Exp $
 * Command line options parsing                                   *
 ******************************************************************
 * Author: Mikkel Holm Olsen (dtv@symlink.dk)                     *
 ******************************************************************
 * dtvmkfs is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License    *
 * as published by the Free Software Foundation; either version 2 *
 * of the License, or (at your option) any later version.         *
 *                                                                *
 * dtvmkfs is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  *
 * GNU General Public License for more details.                   *
 *                                                                *
 * You should have received a copy of the GNU General Public      *
 * License along with dtvmkfs; if not, write to the Free Software *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,         *
 * MA  02110-1301  USA                                            *
 ******************************************************************/

#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>

#include "config.h"

/* Our own header */
#include "clopts.h"

void clopts_clear_options(struct cmdline_opts *options) {
  memset(options,0,sizeof(struct cmdline_opts));
}

/* Parse command-line options */
int clopts_process_parameters(int argc, char **argv, struct cmdline_opts *opts) {
  int c, retflags=0;

  /* Long options for use with getopt_long */
  static struct option long_options[] = {
    {"debug",no_argument,0,'d'},
    {"help",no_argument,0,'h'},
    {"version",no_argument,0,'V'},
    {"import",required_argument,0,'i'},
    {"entries",required_argument,0,'e'},
    {"prgfile",required_argument,0,'p'},
    {"width",required_argument,0,'w'},
    {"header",required_argument,0,'H'},
    {"onefile",no_argument,0,'1'},
    {"twofiles",no_argument,0,'2'},
    {"chunked",no_argument,0,'c'},
    {"listing",no_argument,0,'l'},
    {"maindir",no_argument,0,'m'},
    {"main-dir",no_argument,0,'m'},
    {"kernel-area",no_argument,0,'k'},
    {"reserve",no_argument,0,'r'},
    {0,0,0,0}
  };
  
  while (1) {
    int option_index = 0;

    /* getopt_long does all the work */
    c = getopt_long(argc, argv, "hdVi:e:p:w:H:12clmkr",
                    long_options,&option_index);

    if (c==-1) break; /* No more options */
    switch (c) {
    case 'd':
      opts->debug++;
      break;
    case 'h': case '?': /* help */
      retflags|=RETFLAG_PROGNAME|RETFLAG_USAGE;
      break;
    case 'V': /* Version */
      retflags|=RETFLAG_PROGNAME|RETFLAG_VERSION;
      break;
    case 'i':
      if (opts->importfile) {
        retflags|=0x01;
        fprintf(stderr,"ERROR: can only import one file. Already "
                "specified %s to be imported.\n",
                opts->importfile);
      } else {
        opts->importfile=strdup(optarg);
      }
      break;
    case 'p':
      if (opts->prgfile) {
        retflags|=0x01;
        fprintf(stderr,"ERROR: can only write one .prg file. Already "
                "specified %s to be written.\n",
                opts->prgfile);
      } else {
        opts->prgfile=strdup(optarg);
      }
      break;
    case 'e':
      if (opts->numentries) {
        retflags|=0x01;
        fprintf(stderr,"ERROR: entries already specified as %d\n",
                opts->numentries);
      } else {
        char *ptr;
        int nument=strtol(optarg,&ptr,10);
        if ((0!=*ptr)||(nument>1024)||(nument<32)) {
          retflags|=0x01;
          fprintf(stderr, "ERROR: invalid value for entries arg. "
                  "(Should be 32-1024)\n");
        } else {
          opts->numentries=nument;
        }
      }
      break;
    case 'w':
      if (opts->dirwidth) {
        retflags|=0x01;
        fprintf(stderr,"ERROR: directory width already specified as %d\n",
                opts->dirwidth);
      } else {
        char *ptr;
        int dw=strtol(optarg,&ptr,10);
        if ((0!=*ptr)||(dw>24)||(dw<8)) {
          retflags|=0x01;
          fprintf(stderr, "ERROR: invalid value for width arg. "
                  "(Should be 8-24)\n");
        } else {
          opts->dirwidth=dw;
        }
      }
      break;
    case 'H':
      if (opts->dirhead) {
        retflags|=0x01;
        fprintf(stderr,"ERROR: Directory header already "
                "specified as %s.\n",
                opts->dirhead);
      } else {
        opts->dirhead=strdup(optarg);
      }
      break;
    case 'k':
      opts->kernel=1;
      break;
    case '1':
      opts->onefile=1;
      break;
    case '2':
      opts->twofiles=1;
      break;
    case 'c':
      opts->chunked=1;
      break;
    case 'l':
      opts->printlist=1;
      break;
    case 'm':
      opts->printdir=1;
      break;
    case 'r':
      opts->importraw=1;
      break;
    default:
      retflags|=RETFLAG_USAGE;
      break;
    }
  }

  /* Print program name */
  if (retflags&RETFLAG_PROGNAME) {
    fprintf(stderr,"%s - Create flash filesystem for the DTV\n",DTVMKFS_PROG);
  }

  if ((opts->importraw)&&(!(opts->importfile))) {
    fprintf(stderr, "ERROR: Option -r requires -i\n");
    retflags|=RETFLAG_USAGE;
  }

  /* Print version information */
  if (retflags&RETFLAG_VERSION) {
    fprintf(stderr,"version %s Copyright (C) 2007-2008 "
            "Mikkel Holm Olsen\n"
            "This is free software, distributed under the terms of GNU GPL.\n"
            DTVMKFS_PROG " uses zlib, Copyright (C) 1995-2005 Jean-loup "
            "Gailly and Mark Adler.\n"
            DTVMKFS_PROG " uses unzip library, Copyright (C) 1998-2005 "
            "Gilles Vollant.\n"
            DTVMKFS_PROG " uses code from dtvpack, Copyright (C) 2006 "
            "Daniel Kahlin.\n"
            "Build time: %s, %s\n",
            DTVMKFS_VERS, __DATE__, __TIME__);
  }

  /* Print usage information */
  if (retflags&RETFLAG_USAGE) {
    fprintf(stderr, "\nUSAGE:\n"
            "  %s [options] <files>\n\n",DTVMKFS_PROG);
    fprintf(stderr, "OPTIONS:\n"
            "  -h  --help           Show help (this page) and exit.\n"
            "  -V  --version        Print version information and exit.\n"
            "  -d  --debug          Increase debugging level.\n"
            "  -i  --import <file>  Initialize flash from file.\n"
            "  -r  --reserve        Find data in imported file, and reserve region.\n"
            "  -e  --entries <num>  Reserve <num> entries in the directory.\n"
            "  -m  --maindir        Print main directory to stdout.\n"
            "  -p  --prgfile <file> Dump the directory listing ($-file) to file.\n"
            "  -l  --listing        Print simulated directory listing to stdout.\n"
            "  -w  --width <n>      Specify directory width (8-24, default 16).\n"
            "  -H  --header <txt>   Set the text of the directory header to txt.\n"
            "  -k  --kernel-area    Include kernel area ($0000-$FFFF) in FS dump.\n"
            "  -1  --onefile        Write complete image to file (default).\n"
            "  -2  --twofiles       Write image to two files (for dtvtrans).\n"
            "  -c  --chunked        Write image to multiple files of 128kB.\n");
    fprintf(stderr, "\nThe files are zip-archives conforming to the required format\n"
            "  (see README for details). If the filename has a @ prepended it is\n"
            "  treated as text file listing archives to be included. If the\n"
            "  filename specified is a single -, such a list is read from\n"
            "  the standard input stream.\n");
  }

  if (opts->dirhead) { // Make sure the directory header makes sense
    char *ptr=opts->dirhead;
    int maxwidth=(opts->dirwidth?opts->dirwidth:16);
    while (*ptr) { // Convert to uppercase
      if ((*ptr>='a')&&(*ptr<='z')) *ptr-='a'-'A';
      ptr++;
    }
    if (strlen(opts->dirhead)>maxwidth) {
      if (opts->debug>=1) 
        fprintf(stderr, "WARNING: Cropping directory header to %d chars.\n",
                maxwidth);
      opts->dirhead[maxwidth]=0;
    }
  }

  // If neither --twofiles or --chunked was specified, assume --onefile
  if ((!opts->twofiles)&&(!opts->chunked))
    opts->onefile=1;

  if (retflags) return -retflags;
  return optind;
}
