/******************************************************************
 * dtvmkfs - Make DTV flash filesystem                            *
 ******************************************************************
 * $Id: main.c,v 1.18 2008-12-06 23:55:27 spiff Exp $
 * Main function                                                  *
 ******************************************************************
 * Author: Mikkel Holm Olsen (dtv@symlink.dk)                     *
 ******************************************************************
 * dtvmkfs is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License    *
 * as published by the Free Software Foundation; either version 2 *
 * of the License, or (at your option) any later version.         *
 *                                                                *
 * dtvmkfs is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  *
 * GNU General Public License for more details.                   *
 *                                                                *
 * You should have received a copy of the GNU General Public      *
 * License along with dtvmkfs; if not, write to the Free Software *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,         *
 * MA  02110-1301  USA                                            *
 ******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "config.h"
#include "clopts.h"    // Command line option parsing
#include "flash.h"     // Helper functions 
#include "hex.h"
#include "dirlist.h"   // C64 directory listing 
#include "zipfile.h"   // ZIP archives
#include "filelist.h"  // Linked list for directory management
#include "dtvpack.h"   // DTVpack compression
#include "number.h"

// The size of the flash filesystem 0x200000 = 2MB
#define FLASH_SIZE (0x200000)

struct cmdline_opts options; // Command line options

int hex_digit_value(unsigned char ch) {
  if ((ch>='0')&&(ch<='9')) {
    return ch-'0';
  } else if ((ch>='A')&&(ch<='F')) {
    return ch-'A'+10;
  } else if ((ch>='a')&&(ch<='f')) {
    return ch-'a'+10;
  } else {
    return 0;
  }
}

char *decode_c64name(char *string) {
  char *stptr=string;
  uint8_t ch, str[32];
  int index=0;
  memset(str, 0, 32);
  while ((ch=*stptr++)!=0) {
    if ((ch>=0x61)&&(ch<=0x7A)) {
      // ASCII $61..$7A are converted to PETSCII $41..$5A
      // (lowercase ASCII to uppercase unshifted PETSCII). 
      str[index++]=ch-0x20;
    } else if ((ch=='#')||(ch==',')||(ch==0x5C)) {
      // ASCII $23 (#), $2C (,), $5C give an error. 
      fprintf(stderr, "ERROR: Invalid character %c in C64name %s\n",ch, string);
      exit(1);
    } else if (ch=='%') {
      // ASCII $25 (%) not followed by a two-digit hex (upper- or lowercase) gives an error. 
      if ((!isxdigit(stptr[0]))||(!isxdigit(stptr[1]))) {
        fprintf(stderr, "ERROR: Invalid escape in C64name %s\n",string);
        exit(1);
      }
      ch=(hex_digit_value(stptr[0])<<4)|hex_digit_value(stptr[1]);
      if ((ch==0)||(ch==0xFF)) {
        // The characters %00 and %FF are invalid because they will 
        // break compatibility with the flash directory. 
        fprintf(stderr, "ERROR: Invalid char $%02X in C64name %s\n",ch,string);
        exit(1);
      }
      str[index++]=ch;
      stptr+=2;
    } else if ((ch<0x20)||(ch>0x5D)) {
      // Anything not in the ASCII $20..$5D range gives an error. 
      fprintf(stderr, "ERROR: Invalid character \"%c\" in C64name %s\n",ch,string);
      exit(1);
    } else {
      str[index++]=ch;
    }
    if (index>=25) {
      fprintf(stderr, "ERROR: C64name too long (%s)\n",string);
      exit(1);
    }
  }
  return strdup((char*)str);  
}


void process_index_line(char *line, char *arch, int allow_zip, int hidden, int nosys) {
  char *strng;
  char *file=NULL, *c64n=NULL, *st;
  char ch;
  int loadaddr=-1, sysaddr=-1;
  FileType_t file_type=FILETYPE_UNKNOWN;
  strng=line;
  
  // Strip any leading whitespace
  while ((*strng)&&(isspace(*strng))) strng++;
  st=strng;
  while (*st) { // Strip comments and NL
    if ((*st=='\n')||(*st=='\r')||(*st=='#')) { *st=0; break; }
    st++;
  }
  st--;
  while ((isspace(*st))&&(st>=strng)) { // Strip trailing spaces
    *st=0;
    st--;
  }
  if (!*strng)
    return; // We do not process empty lines (and comments)

  st=strng;
  if (*st=='"') { // Get rid of quoting
    st++;
    while ((*st)&&(*st!='"')) {st++;ch=*st;} // Find end quote
    if (ch=='"') {
      *st++=0; // Terminate filename and skip to next part
      strng++; // Skip leading quote
    } else {
      fprintf(stderr, "ERROR: Unbalanced quote in %s\n", line);
      exit(1);
    }
  }

  while ((*st)&&(*st!=',')) {st++;ch=*st;} // Find first comma
  *st++=0;
  if (strng[0]=='$') {
    strng++;
    hidden=1;
  }
  file=strdup(strng); // We have the filename
  
  strng=st;
  if (ch==',') { // Now comes C64-name
    if (*st=='"') { // Get rid of quoting
      st++;
      while ((*st)&&(*st!='"')) {st++;ch=*st;} // Find end quote
      if (ch=='"') {
        *st++=0; // Terminate filename and skip to next part
        strng++; // Skip leading quote
      } else {
        fprintf(stderr, "ERROR: Unbalanced quote in %s\n", line);
        exit(1);
      }
    }

    while ((*st)&&(*st!=',')) {st++;ch=*st;}
    *st++=0; // Terminate C64-name
    
    if (strng[0]=='$') { // Hidden file
      c64n=decode_c64name(strng+1); // Strip $
      hidden=1;
    } else {
      c64n=decode_c64name(strng);
    }
    
    if (ch==',') { // We have a load address
      strng=st;
      while ((*st)&&(*st!=',')) {st++;ch=*st;}
      *st++=0;
      if (strcasecmp(strng,"PRG")==0) {
        file_type=FILETYPE_PRG;
      } else {
        loadaddr=number_parse_from_string(strng);
        if ((loadaddr==-1)||(loadaddr>0xFFFFFF)) {
          if (strlen(strng)) {
            fprintf(stderr, "ERROR: Incorrectly formatted load address in %s %s\n",arch,file);
            exit(1);
          }
          loadaddr=-1;
        }
      }
      if (ch==',') { // We have a sys address
        strng=st;
        while ((*st)&&(*st!=',')) {st++;ch=*st;}
        *st++=0;
        if (strcasecmp(strng,"RUN")==0) {
          sysaddr=0x100;
        } else if ((strcasecmp(strng,"HIDE")==0)||(strcasecmp(strng,"HIDDEN")==0)) {
          nosys=1;
        } else {
          sysaddr=number_parse_from_string(strng);
          if ((sysaddr==-1)||(sysaddr>0xFFFF)) {
            if (strlen(strng)) {
              fprintf(stderr, "ERROR: Incorrectly formatted sys address in %s %s\n",arch,file);
              exit(1);
            }
            sysaddr=-1;
          }
          if (sysaddr==0)
            nosys=1;
        }
      }
      while (ch==',') { // Parse additional arguments
        strng=st;
        while ((*st)&&(*st!=',')) {st++;ch=*st;}
        *st++=0;
        if (strcasecmp(strng,"PRG")==0) {
          if (file_type==FILETYPE_UNKNOWN) {
            file_type=FILETYPE_PRG;
          } else {
            fprintf(stderr,"ERROR: Filetype already set %s %s\n",arch,file);
          }
        } else if (strcasecmp(strng,"BIN")==0) {
          if (file_type==FILETYPE_UNKNOWN) {
            file_type=FILETYPE_BIN;
          } else {
            fprintf(stderr,"ERROR: Filetype already set %s %s\n",arch,file);
          }
        } else if (strcasecmp(strng,"RAW")==0) {
          if (file_type==FILETYPE_UNKNOWN) {
            file_type=FILETYPE_RAW;
          } else {
            fprintf(stderr,"ERROR: Filetype already set %s %s\n",arch,file);
          }
        } else if (strcasecmp(strng,"DTV")==0) {
          if (file_type==FILETYPE_UNKNOWN) {
            file_type=FILETYPE_DTVPACKED;
          } else {
            fprintf(stderr,"ERROR: Filetype already set %s %s\n",arch,file);
          }
        } else if (strcasecmp(strng,"NOLIST")==0) {
          hidden=1;
        } else {
          fprintf(stderr, "WARNING: Unknown option %s for %s\n",strng,file);
        }
      }
    }
  }
  
  if (file_type==FILETYPE_UNKNOWN) {
    if (strcasecmp(file+strlen(file)-4,".zip")==0) {
      file_type=FILETYPE_ARCHIVE;
    } else if (strcasecmp(file+strlen(file)-4,".dtv")==0) {
      file_type=FILETYPE_DTVPACKED;
    } else if (strcasecmp(file+strlen(file)-4,".prg")==0) {
      file_type=FILETYPE_PRG;
    } else if (strcasecmp(file+strlen(file)-4,".bin")==0) {
      file_type=FILETYPE_BIN;
    } else if (strcasecmp(file+strlen(file)-4,".raw")==0) {
      file_type=FILETYPE_RAW;
    } else {
      file_type=FILETYPE_DTVPACKED;
    }      
  }  
  
  switch (file_type) {
  case FILETYPE_UNKNOWN:
    fprintf(stderr,"ERROR: Unknown file type for %s %s\n",arch,file);
    exit(1);
    break;
  case FILETYPE_ARCHIVE:
    if (!allow_zip) {
      fprintf(stderr,"ERROR: Cannot do recursive decompression (%s)\n",arch);
      exit(1);
    }
    if (c64n&&((strcmp(c64n,"0")==0)||(strcasecmp(c64n,"HIDE")==0)||(strcasecmp(c64n,"HIDDEN")==0)))
      nosys=1;
    if (zip_open_archive(file)) {
      fprintf(stderr,"ERROR: Could not open archive %s\n",file);
      exit(1);
    }
    if (zip_open_file("index.txt", NULL)) {
      fprintf(stderr,"ERROR: index.txt not found in archive %s\n",file);
      exit(1);
    }
    if (options.debug>=3) printf("Reading index.txt from archive %s\n",file);
    arch=strdup(file);
    while ((strng=zip_buf_get_line())) {
      // Process each line of index.txt recursively
      process_index_line(strng, arch, 0, hidden, nosys);
    }
    zip_close_file();
    zip_close_archive();
    break;
  case FILETYPE_PRG:
  case FILETYPE_DTVPACKED:
  case FILETYPE_RAW:
  case FILETYPE_BIN:
    if ((c64n==NULL)||(*c64n==0)) { // We really need a C64-name for the file
      char *ptr;
      char ch;
      ptr=file+strlen(file)-1;
      ch=*ptr;
      while (ptr>=file) { // Strip the path from the filename
        if ((*ptr=='/')||(*ptr=='\\'))
          break;
        ptr--;
      }
      ptr++;
      
      ptr=c64n=strdup(ptr); // Use the remaining filename
      while (*ptr) { // Convert to upper case
        if ((*ptr>='a')&&(*ptr<='z')) *ptr-='a'-'A';
        ptr++;
      }
      if ((strlen(c64n)>4)&&((0==strncmp(ptr-4,".DTV",4)) ||
                             (0==strncmp(ptr-4,".PRG",4)) ||
                             (0==strncmp(ptr-4,".RAW",4)) ||
                             (0==strncmp(ptr-4,".BIN",4)))) {
        ptr[-4]=0; // Strip extension
      }
    }
    if ((sysaddr==-1)||(nosys))
      sysaddr=0;
    if (options.debug>=5) {
      if ((arch)&&(arch[0]))
        fprintf(stderr, "zip: %s, ", arch);
      fprintf(stderr, "file: %s, c64: %s, %d, %d\n", file, c64n, loadaddr, sysaddr);
    }
    file_add_to_list(arch, file, file_type, c64n, 0, loadaddr, sysaddr, 0, !hidden);
    break;
  }
}

int abstract_read(char *buf, int size, FILE *fp) {
  if (fp) {
    return(fread(buf, 1, size, fp));
  } else {
    return(zip_read_buffer(buf,size));
  }
}

// Do actual import of the files
int import_files(unsigned char *flashfs, int size, int firstadd) {
  struct filelist_t *ptr=file_get_first_entry();
  struct filelist_t *match=NULL, *match2=NULL;
  int readbytes;
  int nbytes;
  char *tmpbuf;
  FILE *fp;
  uint32_t crc;
  while (ptr) { // Loop over all the files
    if ((ptr->archive)&&(ptr->archive[0]!=0)) {
      fp=NULL;
      if (zip_open_archive(ptr->archive)) {
        fprintf(stderr, "ERROR: Could not open archive %s.\n",
                ptr->archive);
        return 0;
      }
      if (zip_open_file(ptr->filename, &crc)) {
        fprintf(stderr, "ERROR: File %s not found in %s.\n",
                ptr->filename, ptr->archive);
        return 0;
      }
    } else { // Regular file
      if (!(fp=fopen(ptr->filename,"rb"))) {
        fprintf(stderr,"ERROR: Could not open file %s.\n",ptr->filename);
        exit(1);
      }
      crc=0;
    }
    
    if ((match=file_find_crc32(crc))!=NULL) { // A file with this CRC already exists
      if (strcmp(ptr->c64name,match->c64name)==0) { // Filenames match
        if (options.debug>=2) 
          fprintf(stderr, "NOTE: File %s was already included.\n",ptr->c64name);
        if (ptr->show_in_listing) // One entry tagged to be shown
          match->show_in_listing=1;
        file_delete_entry(ptr);
      } else { // Filenames do not match - create new directory entry
        if ((match2=file_find_name(ptr->c64name))!=NULL) { // We already have a file with this name
          fprintf(stderr, "ERROR: C64name %s used by both %s and %s(0x%08X)\n",
            ptr->c64name, match2->archive, ptr->archive, crc);
          return 0;
        }
        fprintf(stderr, "NOTE: File %s from %s matches %s from %s. Files linked in directory.\n",
          ptr->c64name,ptr->archive,match->c64name,match->archive);
        ptr->flashaddr=match->flashaddr; // Set the flash address
        ptr->unpackedsize=match->unpackedsize;
        if (ptr->show_in_listing) {
          dir_add_to_list(((ptr->unpackedsize)+253+2)/254,ptr->c64name);
        }
        ptr->packedsize=match->packedsize;            
        ptr->crc32=crc;
        if (ptr->file_type==FILETYPE_PRG) {
          if (ptr->loadaddr==-1)
            ptr->loadaddr=match->loadaddr;
          if (ptr->sysaddr==-1)
            ptr->sysaddr=match->sysaddr;
        }
        if (ptr->loadaddr==-1)
          ptr->loadaddr=0x0801;
        if (ptr->sysaddr==-1)
          ptr->sysaddr=0x0000;
      }
    } else {
      if ((match=file_find_name(ptr->c64name))!=NULL) { // We already have a file with this name
        fprintf(stderr, "ERROR: C64name %s used by both %s and %s\n",
          ptr->c64name, match->archive, ptr->archive);
        return 0;
      }
      ptr->crc32=crc;
      
      switch (ptr->file_type) {
      case FILETYPE_UNKNOWN:
      case FILETYPE_ARCHIVE:
        fprintf(stderr, "ERROR: Unhandled filetype (%d) %s %s\n",
          ptr->file_type, ptr->archive, ptr->filename);
        return 0;
      case FILETYPE_DTVPACKED:
      case FILETYPE_RAW:
        if (fp) {
          fseek(fp,0,SEEK_END);
          nbytes=ftell(fp);
          fseek(fp,0,SEEK_SET);
        } else {
          nbytes=zip_get_file_size();
        }
        if (ptr->loadaddr==-1)
          ptr->loadaddr=0x0801;
        // Include file verbatim
        if (firstadd+nbytes>size-5) {
          fprintf(stderr, "ERROR: No more room on flash.\n");
          return 0;
        }
        // Let's just read the whole file in one go.
        ptr->packedsize=abstract_read((char *)flashfs+firstadd,size-firstadd-5, fp);
        ptr->flashaddr=firstadd; // Set the flash address
        if (ptr->file_type==FILETYPE_DTVPACKED) { // Size of unpacked data
          dtv_get_sizes(flashfs+firstadd, NULL, &ptr->unpackedsize);
        } else {
          ptr->unpackedsize=ptr->packedsize;
        }
        break;
      case FILETYPE_PRG:
      case FILETYPE_BIN:
        if (fp) {
          fseek(fp,0,SEEK_END);
          nbytes=ftell(fp);
          fseek(fp,0,SEEK_SET);
        } else {
          nbytes=zip_get_file_size();
        }
        ptr->flashaddr=firstadd; // Set the flash address
        if (ptr->file_type==FILETYPE_PRG) {
          unsigned char buf[5];
          int tmpval;
          abstract_read((char*)buf, 2, fp);
          tmpval=buf[0]|((uint32_t)buf[1]<<8); // Load address set from PRG-file
          if (ptr->loadaddr==-1)
            ptr->loadaddr=tmpval;
          nbytes-=2;
        }
        if (ptr->loadaddr==-1)
          ptr->loadaddr=0x0801;
        tmpbuf=malloc(nbytes+10);
        if (!tmpbuf) {
          fprintf(stderr, "ERROR: Could not allocate %d bytes temporary buffer\n", nbytes+10);
          exit(1);
        }
        ptr->unpackedsize=abstract_read(tmpbuf, nbytes+5, fp);
        readbytes=dtv_pack((unsigned char*)tmpbuf, ptr->unpackedsize, flashfs+firstadd, size-firstadd-5);
        free(tmpbuf);
        tmpbuf=NULL;
        if (readbytes<0) {
          fprintf(stderr, "ERROR: could not pack %s (no room on fs)\n",ptr->c64name);
          return 0;
        } else {
          ptr->packedsize=readbytes;
        }
        break;
      }
  
      ptr->included=1;
      if (ptr->show_in_listing) {
        // Round up +2 to simulate load-address
        dir_add_to_list(((ptr->unpackedsize)+253+2)/254,ptr->c64name);
      }
  
      firstadd+=ptr->packedsize;
      /*
      memcpy((char *)flashfs+firstadd,"SPIFF",5);
      firstadd+=5; */
      if (firstadd>=size-1) {
        fprintf(stderr, "ERROR: No more room on flash filesystem\n");
        return 0;
      }
    }
    if (fp) {
      fclose(fp);
      fp=NULL;
    } else {
      zip_close_file();
      zip_close_archive();
    }
    ptr=ptr->next;
  }
  return firstadd;
}

int dump_file(const char *filename, unsigned char *add, int len) {
  FILE *fp;
  if (!(fp=fopen(filename,"wb"))) {
    fprintf(stderr, "Error opening %s for writing.\n", filename);
    return 1;
  }
  fwrite(add,1,len,fp);
  fclose(fp);
  return 0;
}

int main(int argc, char **argv) {
  FILE *fp;
  char *filename, linebuf[1024];
  struct filelist_t *filelistptr=NULL;
  int argctr, already_did_stdin=0, freeblocks, nbytes;
  int firstadd=0x0, dollar_add=0, dir_add=0x10000,
    dirsize=0, packedsize=0, lastaddr=FLASH_SIZE;
  char *dirheadstr=NULL;

  unsigned char *flashfs=malloc(FLASH_SIZE);
  if (!flashfs) {
  	fprintf(stderr, "ERROR: Could not allocate flash fs "
                "buffer (%d bytes).\n",FLASH_SIZE);
  	return 1;
  }
  memset((void*)flashfs,0xFF,FLASH_SIZE);
  
  clopts_clear_options(&options);
  if (0>(argctr=clopts_process_parameters(argc, argv, &options))) {
    exit(1);
  }

  if (options.importfile) { // Should we import a file?
    if (options.debug>=1) 
      fprintf(stderr, "Importing file %s.\n",options.importfile);
    if (!(fp=fopen(options.importfile,"rb"))) {
      fprintf(stderr,"ERROR: Could not open file %s.\n",options.importfile);
      exit(1);
    }
    // Determine filesize (in a portable way)
    fseek(fp,0,SEEK_END);
    nbytes=ftell(fp);
    if ((nbytes!=FLASH_SIZE)&&(nbytes!=FLASH_SIZE-0x10000)) {
      fprintf(stderr, "Imported file should be either %d or %d bytes. "
              "Not importing.\n",FLASH_SIZE,FLASH_SIZE-0x10000);
    } else {
      firstadd=(nbytes==FLASH_SIZE)?0:0x10000;
      fseek(fp,0,SEEK_SET);
      nbytes=fread(&flashfs[firstadd], 1, FLASH_SIZE-firstadd, fp);
      if (nbytes!=FLASH_SIZE-firstadd) {
        fprintf(stderr,"WARNING: Error reading %s, got %d bytes.\n",
                options.importfile,nbytes);
      }
    }
    fclose(fp);
    if (options.importraw) {
      for (lastaddr=0x10000;lastaddr<FLASH_SIZE;lastaddr++) {
        if (flashfs[lastaddr]!=0xFF) {
          fprintf(stderr,"Data detected at 0x%06X.\n",lastaddr);
          break; 
        }
      }
    } else {
      // Clear all except the kernel area
      memset((void*)&flashfs[0x10000],0xFF,FLASH_SIZE-0x10000);
      lastaddr=FLASH_SIZE;
    }
  }

  // Now is the time to get a list of archives to include
  zip_initialize();
  if (argctr<argc) { // We have command line arguments
    for (;argctr<argc;argctr++) {
      filename=argv[argctr];
      if (filename[0]=='@') { // This specifies a list-file - process it
        filename++;
        if (!(fp=fopen(filename,"r"))) {
          fprintf(stderr,"Could not open list-file %s.\n",filename);
        } else { // List-file opened
          while (fgets(linebuf,sizeof(linebuf),fp)) {
            // WARNING: If line>buffer, we get garbage here (no LF)
            process_index_line(linebuf, NULL, 1, 0, 0);
          }
          fclose(fp);
        }
      } else if (0==strcmp(filename,"-")) { // use STDIN
        if (already_did_stdin) {
          fprintf(stderr, "Already processed stdin.\n");
        } else {
          already_did_stdin=1;
          while (fgets(linebuf,sizeof(linebuf),stdin)) {
            // WARNING: If line>buffer, we get garbage here (no LF)
            process_index_line(linebuf, NULL, 1, 0, 0);
          }
        }
      } else { // Regular filename
        process_index_line(filename, NULL, 1, 0, 0);
      }
    }
  }

  // We reserve 32 bytes per directory entry
  if (options.numentries) {
    int req_entries=file_get_num_entries()+3; // +1 ent for $-file, 2 for term
    if (req_entries>options.numentries) {
      fprintf(stderr, "Directory needs %d entries, "
              "only %d requested.\n", req_entries,
              options.numentries);
      options.numentries=req_entries;
    } else if (options.debug>=2) {
      fprintf(stderr, "Requested %d directory entries, used %d\n",
              options.numentries, req_entries);

    }
  } else {
    options.numentries=file_get_num_entries()+3;
    if (options.debug>=2)
      fprintf(stderr, "Directory has %d entries\n",options.numentries);
  }

  firstadd=dir_add+32*options.numentries;
  if (firstadd>=lastaddr) {
    fprintf(stderr, "ERROR: No room for directory.\n");
    return 1;
  }

  // Now read the files (and build directory list)
  if (!(firstadd=import_files(flashfs,lastaddr,firstadd))) {
    return 1;
  }

  freeblocks=(lastaddr-firstadd)/254; // Hmm. free blocks are without the $-file
  if (options.dirhead) {
    dirheadstr=decode_c64name(options.dirhead);
  }
  dirsize=dir_build_listing(dirheadstr, options.dirwidth, freeblocks);
  if (-1==dirsize) {
    fprintf(stderr, "ERROR: Out of memory.\n");
    return 1;
  }

  dollar_add=firstadd; // The address of the $-file
  
  if (options.prgfile) {
    if (options.debug>=2) {
      fprintf(stderr, "Writing directory to %s.\n",options.prgfile);
    }
    dir_save_buffer_to_prg(options.prgfile);
  }

  if (options.printlist) {
    dir_print_listing(dirheadstr, options.dirwidth, freeblocks);
  }

  // The $-file must be DTV-packed.
  packedsize=dtv_pack(dir_return_buffer_ptr(), dirsize,
                      flashfs+firstadd, lastaddr-1-firstadd);
                      
  //  dir_free_listing_buffer();
  if (packedsize<0) {
    fprintf(stderr, "ERROR: No room for $-file\n");
    return 1;
  }
  firstadd+=packedsize;

  // Add the $-file to the list
  file_add_to_list("-", "-", FILETYPE_UNKNOWN, "$", dollar_add, 0x0801, 0, packedsize, 0);

  // Build the directory at address 0x10000
  filelistptr=file_get_first_entry();
  while (filelistptr) {
    char *c64n=filelistptr->c64name;
    strncpy((char*)&flashfs[dir_add], c64n, 24); // Pads with 0
    flash_set_3_bytes(&flashfs[dir_add+0x18],filelistptr->flashaddr);
    flash_set_3_bytes(&flashfs[dir_add+0x1b],filelistptr->loadaddr);
    flash_set_2_bytes(&flashfs[dir_add+0x1e],filelistptr->sysaddr);
    if (options.printdir) {
      printf("%-24s %06X-%06X %06X %04X\n",c64n,
             filelistptr->flashaddr,
             filelistptr->flashaddr+filelistptr->packedsize-1,
             filelistptr->loadaddr,
             filelistptr->sysaddr);
    }
    dir_add+=0x20;
    filelistptr=filelistptr->next;
  }
  if ((options.debug>=1)||(options.printdir)) {
    fprintf(stderr, "%d kB used (%.1f%%), %d kB free.\n",
            (firstadd+1023)/1024, 100.0*firstadd/lastaddr,
            (lastaddr-firstadd)/1024);
  }
  // Terminate the list
  memset(&flashfs[dir_add],0x00,0x20); // Normal end - skipped by menu
  dir_add+=0x20;

  // This should not be needed, since flash should already have
  // 0xFF where unused
  memset(&flashfs[dir_add],0xFF,0x20); // Fix for Roland's menu

  // Write the flash filesystem to disk
  if (options.onefile) {
    if (options.debug>=1) fprintf(stderr, "Writing flashfs.bin.\n");
    dump_file("flashfs.bin", &flashfs[(options.kernel?0:0x10000)],
              (options.kernel?FLASH_SIZE:FLASH_SIZE-0x10000));
  }

  if (options.twofiles) {
    if (options.debug>=1) fprintf(stderr, "Writing two files.\n");
    if (options.kernel) {
      dump_file("t000000.bin", &flashfs[0x000000], 0x100000);
    } else {
      dump_file("t010000.bin", &flashfs[0x010000], 0x0F0000);
    }
    if ((firstadd>=0x100000)||(lastaddr<0x200000)){
      dump_file("t100000.bin", &flashfs[0x100000], 0x100000);
    } else if (options.debug>=1) {
      fprintf(stderr, "Skipping t100000.bin, since it would be empty.\n");
    } 
  }

  if (options.chunked) {
    char filename[12];
    int myadd=(options.kernel?0:0x10000);
    if (options.debug>=1) fprintf(stderr, "Writing chunked (128kB) files.\n");
    while (myadd<FLASH_SIZE) {
      if ((myadd<firstadd)||((lastaddr!=0x200000)&&(myadd+0x20000>lastaddr))) {
        snprintf(filename, 12, "c%06x.bin", myadd);
        dump_file(filename, &flashfs[myadd], 
                  myadd+0x20000>FLASH_SIZE?FLASH_SIZE-myadd:0x20000);
      }
      myadd+=0x20000;
    }
  }

  return 0;
}
