; -------------------------------------------------------------------------------------------------------------- ;
; Miner 2049er - MINER.prg
; -------------------------------------------------------------------------------------------------------------- ;
; Memory Map
; -------------------------------------------------------------------------------------------------------------- ;
; $0000 - $00ff:  Zero Page Values
; $0200 - $02ff:  Work Values and C64 system
; $0300 - $03ff:  Work Values and C64 system

; $0a00 - $0ac7:  TabPtrHiresLo                     ; 
; $0ac8 - $0b8f:  TabPtrHiresHi                     ; 
; $0d00 - $0dff:  TabPtrHiresLo                     ; 
; $0e00 - $0eff:  Work Area Items                   ; 

; $4000 - $5fff:  Hires Screen                      ; 
; $6000 - $63e8:  Game  Screen                      ; 
; $6000 - $6027:  Game Status Line                  ; 
; $6008 - $63e8:  Game Level 00-09 Screen           ; 
; $60e9 - $63f8:  unused                            ; 
; $63f9 - $63ff:  Sprite Pointer                    ; 
; $6400 - $6bff:  Sprites Game Storage              ; 
; $6c00 - $6fbf:  Tiles   Game Storage              ; 
; $6fa0 - $6fdf:  Unused1 Game Storage              ; 
; $6fe0 - $6fff:  TileClr Game Storage              ; 
; $7000 - $71ff:  Character Set                     ; 
; $7200 - $73ff:  Character Set King Size           ; 

; $a080 - $a080:  Sprites Storage                   ; 
; -------------------------------------------------------------------------------------------------------------- ;
                    * equ $8000
; -------------------------------------------------------------------------------------------------------------- ;
; compiler settings
; -------------------------------------------------------------------------------------------------------------- ;
                    incdir  ..\..\inc               ; C64 System Includes

C64CIA1             include cia1.asm                ; Complex Interface Adapter (CIA) #1 Registers  $DC00-$DC0F
C64CIA2             include cia2.asm                ; Complex Interface Adapter (CIA) #2 Registers  $DD00-$DD0F
C64SID              include sid.asm                 ; Sound Interface Device (SID) Registers        $D400-$D41C
C64VicII            include vic.asm                 ; Video Interface Chip (VIC-II) Registers       $D000-$D02E
C64Kernel           include kernel.asm              ; Kernel Vectors
C64Colors           include color.asm               ; Colour RAM Address / Colours
C64Memory           include mem.asm                 ; Memory Layout

Game                include inc\M2_VarsGame.asm     ; Game Variables
WorkAreas           include inc\M2_WorkAreas.asm    ; Work Area Maps
Equates             include inc\M2_Equates.asm      ; External and Unresolved Addresses
ZeroPage            include inc\M2_Zpg.asm          ; Zero Page Addresses
; -------------------------------------------------------------------------------------------------------------- ;
ModuleStart         dc.w Start                      ; 
                    dc.w Start                      ; warm start after <restore>
; -------------------------------------------------------------------------------------------------------------- ;
                    dc.b $c3 ; C
                    dc.b $c2 ; B
                    dc.b $cd ; M
                    dc.b $38 ; 8
                    dc.b $30 ; 0
; -------------------------------------------------------------------------------------------------------------- ;
Start               sei                             ; 
                    cld                             ; 
                    ldx #$ff                        ; 
                    txs                             ; clear stack
                    
                    lda #$7f                        ; .#######
                    sta CIAICR                      ; CIA1($DC0D) Interrupt Control
                    sta CI2ICR                      ; CIA2($DD0D) Interrupt Control
                    
                    lda #$2f                        ; ..#.####
                    sta D6510                       ; D6510 - 6510 On-Chip I/O DATA Direction
                    
                    lda #$fe          ; hbu - $ff   ; ##### ##. - basic=off  i/o=on  kernal=on
                    sta R6510                       ; R6510 - 6510 On-Chip I/O Data
                    
                    lda C2DDRA                      ; CIA2($DD02) Data Dir A
                    ora #$03                        ; ......##
                    sta C2DDRA                      ; CIA2($DD02) Data Dir A
                    
                    lda CI2PRA                      ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                    and #$fc                        ; ######..
                    ora #$02                        ; ......#. - 02 = $4000-$7fff - bank 1
                    sta CI2PRA                      ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                    
                    lda #$00                        ; 
                    sta IRQMASK                     ; VIC($D01A) IRQ Mask Register
                    sta EXTCOL                      ; VIC($D020) Border Color
                    sta BGCOL0                      ; VIC($D021) Background Color 0
                    
                    sta CIDDRB                      ; CIA1($DC03) Data Direction Register B
                    
                    lda #$0b                        ; ....#.## - Bit3:1=24 rows Bit4:1=screen off bit5:0=bitmap mode off
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    
                    jsr InitTabHiResRowPtr          ; init hires row tab
                    
                    lda #<CHARGEN                   ; 
                    sta M2Z_FromAreaPtrLo           ; 
                    lda #>CHARGEN                   ; 
                    sta M2Z_FromAreaPtrHi           ; ($2b/$2c) point to $d000
                    
                    lda #<M2_CharSet                ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>M2_CharSet                ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) point to $7000
                    
                    lda #<M2_CharSetLen             ; 
                    sta M2Z_FillAreaLenBytes        ; 
                    lda #>M2_CharSetLen             ; 
                    sta M2Z_FillAreaLenPages        ; ($2d/$2e) length = $0200
                    
                    lda R6510                       ; 
                    and #$fb                        ; #####.## - char rom=on  io=off
                    sta R6510                       ; 
                    
                    jsr CopyData                    ; copy characters to $7000 - $71ff
                    
                    lda R6510                       ; 
                    ora #$04                        ; .....#.. - char rom=off  io=on
                    sta R6510                       ; 
                    
                    ldx #$07                        ; 
.CpyHeadChrData     lda TabCharLives,x              ; 
                    sta M2_CharSet,x                ; copy life sign to first position
                    dex                             ; 
                    bpl .CpyHeadChrData             ; 
                    
                    jsr DoubleCharSet               ; double chrs width
                    
                    lda #<IRQ                       ; 
                    sta C64_CINV_LO                 ; 
                    lda #>IRQ                       ; 
                    sta C64_CINV_HI                 ; Vector to IRQ Interrupt Routine
                    
                    lda #<Start                     ; 
                    sta C64_CBINV_LO                ; 
                    lda #>Start                     ; 
                    sta C64_CBINV_HI                ; Vector to BRK Instruction Interrupt Routine
                    
                    cli                             ; 
                    jmp DemoGameHandler             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitTabHiResRowPtr Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitTabHiResRowPtr  subroutine                      ; 
                    lda #$40                        ; 
                    sta M2Z_WorkLenBytes            ; 
                    lda #$01                        ; 
                    sta M2Z_WorkLenPages            ; ($1a/$1b) length = $0140 (gfx line length)
                    
                    lda #<M2_Screen_HiRes           ; 
                    sta M2Z_WorkTargetPtrLo         ; 
                    lda #>M2_Screen_HiRes           ; 
                    sta M2Z_WorkTargetPtrHi         ; ($29/$2a) point to $4000
                    
                    ldx #$00                        ; 
.CopyPointersI      ldy #$08                        ; 
.CopyPointers       lda M2Z_WorkTargetPtrHi         ; 
                    sta TabPtrHiresHi,x             ; 
                    lda M2Z_WorkTargetPtrLo         ; 
                    sta TabPtrHiresLo,x             ; 
                    inx                             ; 
                    inc M2Z_WorkTargetPtrLo         ; 
                    dey                             ; 
                    bne .CopyPointers               ; 
                    
                    and #$f0                        ; ####....
                    clc                             ; 
                    adc M2Z_WorkLenBytes            ; 
                    sta M2Z_WorkTargetPtrLo         ; 
                    
                    lda M2Z_WorkLenPages            ; 
                    adc M2Z_WorkTargetPtrHi         ; 
                    sta M2Z_WorkTargetPtrHi         ; 
                    
                    cpx #$c8                        ; 
                    bcc .CopyPointersI              ; 
                    
InitTabHiResRowPtrX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DoubleCharSet     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DoubleCharSet       subroutine                      ; 
                    lda #<M2_CharSet                ; 
                    sta M2Z_FromAreaPtrLo           ; 
                    lda #>M2_CharSet                ; 
                    sta M2Z_FromAreaPtrHi           ; ($2b/$2c) points to $7000
                    
                    lda #<M2_CharSetBig             ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>M2_CharSetBig             ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) points to $7200
                    
                    lda #$40                        ; 
                    sta M2Z_FillAreaLenBytes        ; character count
                    
.GetNextChrI        ldy #$07                        ; 
.GetNextChr         lda (M2Z_FromAreaPtr),y         ; 
                    jsr DoubleChar                  ; 
                    
                    lda M2Z_19                      ; 
                    sta (M2Z_FillAreaPtr),y         ; 
                    lda M2Z_DoubleChar2nd           ; 
                    sta M2Z_TilePart2Start,y        ; 
                    dey                             ; 
                    bpl .GetNextChr                 ; 
                    
                    ldy #$0f                        ; 
                    ldx #$07                        ; 
.Cpy2ndHalf         lda M2Z_TilePart2,x             ; 
                    sta (M2Z_FillAreaPtr),y         ; 
                    dey                             ; 
                    dex                             ; 
                    bpl .Cpy2ndHalf                 ; 
                    
                    clc                             ; 
                    lda #$08                        ; 
                    adc M2Z_FromAreaPtrLo           ; 
                    sta M2Z_FromAreaPtrLo           ; 
                    bcc .SetNextPos                 ; 
                    inc M2Z_FromAreaPtrHi           ; 
                    
.SetNextPos         clc                             ; 
                    lda #$10                        ; 
                    adc M2Z_FillAreaPtrLo           ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    bcc .DecCount                   ; 
                    inc M2Z_FillAreaPtrHi           ; 
                    
.DecCount           dec M2Z_FillAreaLenBytes        ; character count
                    bne .GetNextChrI                ; 
                    
DoubleCharSetX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DoubleChar        Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DoubleChar          subroutine                      ; 
                    ldx #$04                        ; 
.1stHalf            asl a                           ; 
                    php                             ; 
                    rol M2Z_DoubleChar1st           ; 
                    plp                             ; 
                    rol M2Z_DoubleChar1st           ; 
                    dex                             ; 
                    bne .1stHalf                    ; 
                    
                    ldx #$04                        ; 
.2ndHalf            asl a                           ; 
                    php                             ; 
                    rol M2Z_DoubleChar2nd           ; 
                    plp                             ; 
                    rol M2Z_DoubleChar2nd           ; 
                    dex                             ; 
                    bne .2ndHalf                    ; 
                    
DoubleCharX         rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
DemoGameHandler     lda #M2Z_DemoModeOn             ; 
                    sta M2Z_DemoMode                ; 
                    bne .GoDemo                     ; 
                    
.ShowInfo           jsr ShowInfoScreen              ; 
                    
                    lda M2Z_DemoMode                ; 
                    bpl .GoGame                     ; game mode
                    
.GoDemo             jsr MainLoopInit                ; 
                    
.ChkDemo            lda M2Z_DemoMode                ; 
                    bmi .ShowInfo                   ; demo mode
                    
.GoGame             lda #M2Z_DemoModeOff            ; 
                    sta M2Z_DemoMode                ; 
                    jsr MainLoopInit                ; 
                    
DemoGameHandlerX    jmp .ChkDemo                    ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MainLoopInit      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MainLoopInit        subroutine                      ; 
                    lda #$00                        ; 
                    ldx #$05                        ; 
.InitScore          sta M2Z_Score,x                 ; 
                    dex                             ; 
                    bpl .InitScore                  ; 
                    
                    lda #$03                        ; 
.InitLives          sta M2Z_NoOfLives               ; 
                    
                    lda #M2Z_LevelNoMin             ; 
.InitLevel          sta M2Z_LevelNo                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MainLoop          Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MainLoop            subroutine                      ; 
                    jsr ClearHiScreen               ; 
                    
.GetNewColor        lda M2Z_LevelColorNo            ; 
                    inc M2Z_LevelColorNo            ; set next color
                    and #$07                        ; max 7
                    asl a                           ; *2
                    tax                             ; 
                    lda TabLevelColors,x            ; 
                    pha                             ; 
                    inx                             ; 
                    lda TabLevelColors,x            ; 
                    jsr ColourScreen                ; 
                    pla                             ; 
                    jsr ColourRam                   ; 
                    
                    lda #$00                        ; BLACK
                    sta M2Z_SprSprColl04            ; 
                    sta M2Z_SprSprColl07            ; 
                    sta M2Z_WaitIRQs                ; 
                    sta M2Z_CountIRQs               ; 
                    sta EXTCOL                      ; VIC($D020) Border Color
                    sta YXPAND                      ; VIC($D017) Sprite Scale Double Width
                    sta BGCOL0                      ; VIC($D021) Background Color 0
                    sta SPBGPR                      ; VIC($D01B) Sprite to Foreground Display Priority
                    lda SPSPCL                      ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                    
                    lda #ORANGE                     ; 
                    sta SPMC0                       ; VIC($D025) Sprite Multicolor Register 0
                    lda #LT_BLUE                    ; 
                    sta SPMC1                       ; VIC($D026) Sprite Multicolor Register 1
                    
                    lda #M2Z_LevelNoDemo            ; preset demo level no
                    ldx M2Z_DemoMode                ; 
                    bmi .GetLevelHandler            ; demo mode
                    
                    lda M2Z_LevelNo                 ; get game level no
                    
.GetLevelHandler    asl a                           ; *2
                    tax                             ; 
                    lda TabLevelHandler,x           ; 
                    sta M2Z_LvlHandlerPtrLo         ; 
                    inx                             ; 
                    lda TabLevelHandler,x           ; 
                    sta M2Z_LvlHandlerPtrHi         ; ($27/$28) point to selected routine
                    
.CallLevelHandler   jsr CallLevelHandler            ; dispatch level handler routine
                    
.VolumeOff          lda #$00                        ; 
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    
.ChkDemo            lda M2Z_DemoMode                ; 
                    beq .ChkIRQs                    ; game mode
                    
                    lda #$00                        ; death
                    sta M2Z_NoOfLives               ; finish demo after one level
                    jmp .Reset                      ; 
                    
.ChkIRQs            lda M2Z_WaitIRQs                ; 
                    cmp #$41                        ; 
                    bne .ChkDemo                    ; 
                    
                    lda M2Z_NextLevel               ; 
                    bpl .DecLives                   ; something bad happened - no next level
                    
                    and #$0f                        ; ....####
                    bne .ChkLevelNoMax              ; 
                    
.SetNextLevelNo     inc M2Z_LevelNo                 ; 
                    lda M2Z_LevelNo                 ; 
                    cmp #M2Z_LevelNoMax + 1         ; 
                    bcc .Reset                      ; 
                    
                    lda #M2Z_LevelNoMin             ; 
                    sta M2Z_LevelNo                 ; 
                    jmp .Reset                      ; 
                    
.ChkLevelNoMax      cmp #M2Z_LevelNoMax + 1         ; 
                    sta M2Z_LevelNo                 ; 
                    bcc .Reset                      ; 
                    
                    lda #M2Z_LevelNoMin             ; 
                    sta M2Z_LevelNo                 ; 
                    jmp .Reset                      ; 
                    
.DecLives           dec M2Z_NoOfLives               ; 
                    bne .Reset                      ; 
                    
                    lda #$2c                        ; "game over" out pos lo
                    sta M2Z_GameOverTextPosLo       ; 
                    
                    lda #$48                        ; 
                    sta M2Z_GameOverTextCount       ; 
                    
                    ldx #$0a                        ; 
.GetVoc3Vals        lda TabSndGameOver,x            ; 
                    sta FRELO3,x                    ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .GetVoc3Vals                ; 
                    
.GetBlankPos1       lda #$20                        ; " " - blank out screen for "game over text"
                    ldx #$60                        ; "game over" out pos hi
                    jsr GraphicTextOutRev           ; 
                    
.GetBlankPos2       lda #$20                        ; " "
                    ldx #$68                        ; "game over" out pos hi
                    jsr GraphicTextOutRev           ; 
                    
                    inc M2Z_GameOverTextPosLo       ; 
                    
                    ldx #$08                        ; 
.WaitI              ldy #$00                        ; 
.Wait               dey                             ; 
                    bne .Wait                       ; 
                    dex                             ; 
                    bne .WaitI                      ; 
                    
                    lda M2Z_GameOverTextCount       ; 
                    lsr a                           ; 
                    sta FREHI3                      ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                    
                    dec M2Z_GameOverTextCount       ; 
                    bne .GetBlankPos1               ; 
                    
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    
                    lda #<TextGameOver              ; 
                    sta M2Z_LvlTextPtrLo            ; 
                    lda #>TextGameOver              ; 
                    sta M2Z_LvlTextPtrHi            ; 
                    ldx #M2Z_LevelTextReversed      ; reversed
                    ldy #$ff                        ; color pointer
                    jsr L__TextOut                  ; 
                    
                    lda #$00                        ; 
                    sta M2Z_19                      ; 
                    sta M2Z_1a                      ; 
.GetJoyStick        jsr CheckJoystick               ; 
                    
                    lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    bne .Reset                      ; 
                    
                    dec M2Z_19                      ; 
                    bne .GetJoyStick                ; 
                    
                    dec M2Z_1a                      ; 
                    bne .GetJoyStick                ; 
                    
                    lda #M2Z_DemoModeOn             ; 
                    sta M2Z_DemoMode                ; 
                    
.Reset              lda #$00                        ; 
                    tax                             ; 
.ClrSpriteWA        sta M2_WaSprite,x               ; 
                    dex                             ; 
                    bne .ClrSpriteWA                ; 
                    
                    sei                             ; 
                    lda #$00                        ; 
                    sta IRQMASK                     ; VIC($D01A) IRQ Mask Register
                    lda SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    and #$ef                        ; ###.#### - screen disable
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    cli                             ; 
                    
                    lda M2Z_NoOfLives               ; 
                    beq MainLoopX                   ; 
                    
.NextTry            jmp MainLoop                    ; 
                    
MainLoopX           rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CallLevelHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CallLevelHandler    jmp (M2Z_LvlHandlerPtr)         ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabLevelHandler     dc.w Level_00_Handler         ; level 00
                    dc.w Level_01_Handler         ; level 01
                    dc.w Level_02_Handler         ; level 02
                    dc.w Level_03_Handler         ; level 03
                    dc.w Level_04_Handler         ; level 04
                    dc.w Level_05_Handler         ; level 05
                    dc.w Level_06_Handler         ; level 06
                    dc.w Level_07_Handler         ; level 07
                    dc.w Level_08_Handler         ; level 08
                    dc.w Level_09_Handler         ; level 09
                    dc.w DemoLevelHandler         ; level 00 as demo
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTextOutRev Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTextOutRev   subroutine                      ; reverse text out
                    stx M2Z_TargetScreenPosY        ; 
                    ldx M2Z_1d                      ; 
                    stx M2Z_TargetScreenPosX        ; 
                    jsr GetBigCharPtr               ; 
                    
                    ldx #$ff                        ; 
                    ldy #$ff                        ; color pointer
                    
GraphicTextOutRevX  jmp GraphicTextOut              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ               Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ                 subroutine                      ; 
                    ldx M2Z_CountIRQs               ; 
                    bne .DecCountIRQs               ; 
                    
                    lda #$18                        ; ...##... - Bit3:1=40 cols Bit4:1=multicolor text on
                    sta SCROLX                      ; VIC($D016) VIC Control Register 2 (and Horizontal Fine Scrolling)
                    lda #$80                        ; #... ... . - 8 Bits4-7:8(screen=$2000-$23e7) Bits1-3:6(chars=$0000-$07ff) + $4000 VIC Bank 1
                    sta VMCSB                       ; VIC($D018) VIC Chip Memory Control
                    lda #$3b                        ; ..###.## - Bit3:1=25 rows Bit4:1=screen on bit5:1=bitmap mode on
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    lda M2Z_LvlRasterIrqLine        ; Line to Compare for Raster IRQ
                    sta RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    
                    inc M2Z_CountIRQs               ; 
                    jmp .ClrSpSpColl                ; 
                    
.DecCountIRQs       dex                             ; 
                    bne .ClrCountIRQs               ; 
                    
                    lda #$fa                        ; 
                    sta RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    
                    inc M2Z_CountIRQs               ; 
                    
                    lda M2Z_WaitIRQs                ; 
                    lsr a                           ; 
                    bcc .LoopSprWA04I               ; 
                    
                    lda SPSPCL                      ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                    lsr a                           ; 
                    bcc .LoopSprWA04I               ; 
                    
                    and #$0f                        ; ....####
                    beq .LoopSprWA04I               ; 
                    
                    ora M2Z_SprSprColl04            ; 
                    sta M2Z_SprSprColl04            ; 
                    
.LoopSprWA04I       ldy #$04                        ; 
.LoopSprWA04        tya                             ; 4-3-2-1
                    asl a                           ; 8-6-4-2
                    tax                             ; 8-6-4-2
                    lda M2Z_LvlSprtWaPtrPart2,x                    ; 
                    sta M2Z_SpriteWAPtrLo           ; 
                    lda M2Z_LvlSprtWaPtrPart2 + 1,x ; 
                    sta M2Z_SpriteWAPtrHi           ; 
                    tya                             ; 4-3-2-1
                    tax                             ; 4-3-2-1
                    pha                             ; 
                    jsr IRQ_SpriteHandler           ; 
                    pla                             ; 
                    tay                             ; 4-3-2-1
                    dey                             ; 
                    bne .LoopSprWA04                ; 
                    
                    jmp .ClrSpSpColl                ; 
                    
.ClrCountIRQs       ldx #$00                        ; 
                    stx M2Z_CountIRQs               ; 
                    
                    lda #$1b                        ; ...##.## - Bit3:1=24 rows Bit4:1=screen on bit5:0=bitmap mode off
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    lda #$08                        ; ....#... - Bit3:1=40 cols Bit4:0=multicolor text off
                    sta SCROLX                      ; VIC($D016) VIC Control Register 2 (and Horizontal Fine Scrolling)
                    lda #$8c                        ; #... ##. . - Bits4-7:8(screen=$2000-$23e7) Bits1-3:6(chars=$3000-$37ff) + $4000 VIC Bank 1
                    sta VMCSB                       ; VIC($D018) VIC Chip Memory Control
                    lda #$3a                        ; 
                    sta RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    
                    lda M2Z_WaitIRQs                ; 
                    lsr a                           ; 
                    bcc .LoopSprWA07I               ; 
                    
                    lda SPSPCL                      ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                    lsr a                           ; sprite $00
                    bcc .LoopSprWA07I               ; no collision
                    
                    and #$0f                        ; ....#### - enemy sprites
                    beq .LoopSprWA07I               ; no collision
                    
                    ora M2Z_SprSprColl07            ; 
                    sta M2Z_SprSprColl07            ; 
                    
.LoopSprWA07I       ldy #$07                        ; 
.LoopSprWA07        tya                             ; 7-6-5-4-3-2-1-0
                    asl a                           ; e-c-a-8-6-4-2-0
                    tax                             ; e-c-a-8-6-4-2-0
                    lda M2Z_LvlSprtWaPtrPart1,x     ; 
                    sta M2Z_SpriteWAPtrLo           ; 
                    lda M2Z_LvlSprtWaPtrPart1 + 1,x ; 
                    sta M2Z_SpriteWAPtrHi           ; 
                    tya                             ; 7-6-5-4-3-2-1-0
                    tax                             ; 7-6-5-4-3-2-1-0
                    pha                             ; 7-6-5-4-3-2-1-0
                    jsr IRQ_SpriteHandler           ; 
                    pla                             ; 7-6-5-4-3-2-1-0
                    tay                             ; 7-6-5-4-3-2-1-0
                    dey                             ; 7-6-5-4-3-2-1-0
                    bpl .LoopSprWA07                ; 
                    
                    lda M2Z_WaitIRQs                ; 
                    lsr a                           ; 
                    bcs .IncCountIRQ                ; 
                    
                    lda SPENA                       ; VIC($D015) Sprite Enable Register
                    and #$1f                        ; ...#####
                    sta SPENA                       ; VIC($D015) Sprite Enable Register
                    
.IncCountIRQ        inc M2Z_WaitIRQs                ; 

.ClrSpSpColl        lda SPSPCL                      ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                    
                    lda #$01                        ; 
                    sta VICIRQ                      ; VIC($D019) Interrupt Flags - Latched flags cleared if set to 1
                    
                    pla                             ; 
                    tay                             ; 
                    pla                             ; 
                    tax                             ; 
                    pla                             ; 
                    
IRQX                rti                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_SpriteHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_SpriteHandler   subroutine                      ; 
                    stx M2Z_SpriteNo                ; 
                    
                    txa                             ; 
                    asl a                           ; *2
                    sta M2Z_SpriteNoDoubled         ; 
                    
                    lda TabSpriteSelect,x           ; 
                    sta M2Z_SpriteSelect            ; 
                    eor #$ff                        ; 
                    sta M2Z_SpriteDeSelect          ; 
                    
                    ldy #M2Z_SpriteInfoLen - 1      ; 
.GetSpriteInfoData  lda (M2Z_SpriteWAPtr),y         ; 
                    sta M2Z_SpriteInfoStart,y       ; 
                    dey                             ; 
                    bpl .GetSpriteInfoData          ; 
                    
                    lda M2Z_SpriteEnable            ; 
                    and M2Z_SpriteSelect            ; 
                    sta M2Z_SpriteWork              ; 
                    
                    lda SPENA                       ; VIC($D015) Sprite Enable Register
                    and M2Z_SpriteDeSelect          ; 
                    ora M2Z_SpriteWork              ; 
                    sta SPENA                       ; VIC($D015) Sprite Enable Register
                    
                    ldx M2Z_SpriteNoDoubled         ; 
                    lda M2Z_SpritePosY              ; 
                    sta SP0Y,x                      ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                    
                    ldy M2Z_SpritePosX              ; 
                    lda M2Z_SpriteKingSize          ; 
                    and M2Z_SpriteSelect            ; 
                    sta M2Z_SpriteKingSize          ; 
                    beq .SetPosX                    ; 
                    
                    tya                             ; 
                    sec                             ; 
                    sbc #$04                        ; 
                    tay                             ; 
                    
.SetPosX            tya                             ; 
                    asl a                           ; *2
                    sta SP0X,x                      ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                    
                    lda #$00                        ; 
                    bcc .SprSelect                  ; 
                    lda #$ff                        ; 
.SprSelect          and M2Z_SpriteSelect            ; 
                    sta M2Z_SpriteWork              ; 
                    
                    lda MSIGX                       ; VIC($D010) MSB X-Pos Sprites 0-7
                    and M2Z_SpriteDeSelect          ; 
                    ora M2Z_SpriteWork              ; 
                    sta MSIGX                       ; VIC($D010) MSB X-Pos Sprites 0-7
                    
                    lda M2Z_SpritePointer           ; 
                    ldx M2Z_SpriteNo                ; 
                    sta M2_SpritePointers,x         ; 
                    
                    lda M2Z_SpriteColor0            ; 
                    sta SP0COL,x                    ; VIC($D027) Color Sprite 0
                    
                    lda M2Z_SpriteBackPrio          ; 
                    and M2Z_SpriteSelect            ; 
                    sta M2Z_SpriteWork              ; 
                    
                    lda SPBGPR                      ; VIC($D01B) Sprite to Foreground Display Priority
                    and M2Z_SpriteDeSelect          ; 
                    ora M2Z_SpriteWork              ; 
                    sta SPBGPR                      ; VIC($D01B) Sprite to Foreground Display Priority
                    
                    lda XXPAND                      ; VIC($D01D) Sprite Double Height
                    and M2Z_SpriteDeSelect          ; 
                    ora M2Z_SpriteKingSize          ; 
                    sta XXPAND                      ; VIC($D01D) Sprite Double Height
                    
                    lda M2Z_SpriteMultiColor        ; 
                    and M2Z_SpriteSelect            ; 
                    sta M2Z_SpriteWork              ; 
                    
                    lda SPMC                        ; VIC($D01C) Sprite Multicolor
                    and M2Z_SpriteDeSelect          ; 
                    ora M2Z_SpriteWork              ; 
                    sta SPMC                        ; VIC($D01C) Sprite Multicolor
                    
IRQ_SpriteHandlerX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Level_00_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_00_Handler    subroutine                      ; 
                    ldx #<Level_00                  ; 
                    ldy #>Level_00                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
Level00Init         jsr L__Init                     ; level init
                    jsr L__InitStatusRow            ; update status line
                    
Level00Loop         jsr L__WaitMoveSprts            ; 
                    jsr L__StdHandler               ; 
                    bcc Level00Loop                 ; level not solved - but still alive
                    
Level_00_HandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; L__WaitMoveSprts  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__WaitMoveSprts    subroutine                      ; 
                    lda M2Z_WaitIRQs                ; 
                    cmp #$02                        ; 
                    bcc L__WaitMoveSprts            ; 
                    
                    lda #$00                        ; 
                    sta M2Z_WaitIRQs                ; 
                    
                    jsr MoveEnemies                 ; 
L__WaitMoveSprtsX   jmp MovePlayer                  ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__CopyHeader     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__CopyHeader       subroutine                      ; 
                    stx M2Z_LevelHeaderPtrLo        ; 
                    sty M2Z_LevelHeaderPtrHi        ; 
                    
                    ldy #M2Z_LvlHeaderStoreLen      ; 
.Copy               lda (M2Z_LevelHeaderPtr),y      ; 
                    sta M2Z_LvlHeaderStore,y        ; 
                    dey                             ; 
                    bpl .Copy                       ; 
                    
L__CopyHeaderX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__StdHandler     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__StdHandler       subroutine                      ; 
                    jsr L__ItemChecker              ; 
                    jsr L__SprtCollHandler          ; 
                    jsr L__ShowStatusRow            ; 
                    jsr L__TimeHandler              ; 
L__StdHandlerX      jmp L__CheckSolved              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Level_01_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_01_Handler    subroutine                      ; 
                    ldx #<Level_01                  ; 
                    ldy #>Level_01                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
Level_01_HandlerX   jmp Level00Init                 ; standard init/loop
; -------------------------------------------------------------------------------------------------------------- ;
; Level_02_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_02_Handler    subroutine                      ; 
                    ldx #<Level_02                  ; 
                    ldy #>Level_02                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
                    jsr L__Init                     ; level init
                    jsr L__InitXmitDigits           ; 
                    jsr L__InitStatusRow            ; update status line
                    
Level_02_HandlerX   jmp Level00Loop                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Level_03_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_03_Handler    subroutine                      ; 
                    ldx #<Level_03                  ; 
                    ldy #>Level_03                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
Level_03_HandlerX   jmp Level00Init                 ; standard init/loop
; -------------------------------------------------------------------------------------------------------------- ;
; Level_04_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_04_Handler    subroutine                      ; 
                    ldx #<Level_04                  ; 
                    ldy #>Level_04                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    jsr L__Init                     ; level init
                    
                    ldy #$03 * M2_WaFlowLen         ; $03 moving flow tiles
.CpyFlowTileData    lda (M2Z_LvlFlowDataPtr),y      ; 
                    sta M2_WaFlow,y                 ; 
                    dey                             ; 
                    bpl .CpyFlowTileData            ; 
                    
                    jsr L__InitStatusRow            ; update status line
.Level04Loop        jsr L__WaitMoveSprts            ; 
                    jsr L__FlowTileHandler          ; 
                    
.ChkCupOfDeath      lda M2_WaSprite_PosX            ; 
                    cmp #Level_04CupTouchPosXMax    ; 
                    bcs .FlickerCupOfDeath          ; 
                    
                    lda M2_WaSprite_PosY            ; 
                    cmp #Level_04CupTouchPosYMin    ; lower
                    bcc .FlickerCupOfDeath          ; 
                    
                    cmp #Level_04CupTouchPosYMax    ; 
                    bcs .FlickerCupOfDeath          ; greater/equal
                    
.TouchedCupOfDeath  lda #M2Z_PlayerDeadWobbleYes    ; 
                    sta M2Z_PlayerDeadWobble        ; 
                    
.FlickerCupOfDeath  lda RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    and #$07                        ; .....### - reduce to colors $00-$07
                    sta M2Z_19                      ; 
                    lda Level_04CupColorPos         ; 
                    and #$f0                        ; ####.... - clear old color
                    ora M2Z_19                      ; 
                    sta Level_04CupColorPos10       ; 
                    sta Level_04CupColorPos11       ; 
                    sta Level_04CupColorPos20       ; 
                    sta Level_04CupColorPos21       ; 
                    jsr L__StdHandler               ; 
                    bcc .Level04Loop                ; level not solved - but still alive
                    
Level_04_HandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; Level_05_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_05_Handler    subroutine                      ; 
                    ldx #<Level_05                  ; 
                    ldy #>Level_05                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
                    lda #$aa                        ; #.#.#.#. - color pointer
                    ldx #$f0                        ; 
.SetWasteColorType  sta Level_05WasteRow1,x         ; hiscreen row 01
                    sta Level_05WasteRow2,x         ; hiscreen row 02
                    sta Level_05WasteRow3,x         ; hiscreen row 03
                    sta Level_05WasteRow4,x         ; hiscreen row 04
                    dex                             ; 
                    bne .SetWasteColorType          ; 
                    
                    jsr L__Init                     ; level init
                    
                    ldy #$01 * M2_WaFlowLen         ; $01 moving flow tile
.CpyFlowTileData    lda (M2Z_LvlFlowDataPtr),y      ; 
                    sta M2_WaFlow,y                 ; 
                    dey                             ; 
                    bpl .CpyFlowTileData            ; 
                    
                    ldx #M2Z_LevelTextReversed      ; reversed
                    ldy #$aa                        ; color pointer
                    jsr L__TextOut                  ; 
                    jsr L__InitStatusRow            ; update status line
                    
.Level05Loop        jsr L__WaitMoveSprts            ; 
                    jsr L__FlowTileHandler          ; 
                    
.ChkPlayerInWaste   lda M2_WaSprite_PosY            ; 
                    cmp #M2_WaSprite_PosYMax        ; 
                    bcc .GetWasteColor              ; lower - player not on waste surface line
                    
.ChkLeftPole        lda M2_WaSprite_PosX            ; 
                    cmp #Level_05WastePoleLeMin     ; outer side
                    beq .StepBack                   ; 
                    cmp #Level_05WastePoleLeMax     ; inner side
                    beq .StepFore                   ; 
                    
.ChkRightPole       cmp #Level_05WastePoleRiMin     ; inner side
                    beq .StepBack                   ; 
                    cmp #Level_05WastePoleRiMax     ; outer side
                    bne .ChkPoles                   ; 
                    
.StepFore           inc M2_WaSprite_PosX            ; 
                    bne .ChkPoles                   ; 
                    
.StepBack           dec M2_WaSprite_PosX            ; 
                    
.ChkPoles           lda M2_WaSprite_PosX            ; 
                    cmp #Level_05WastePoleLeMax     ; left inner side
                    bcc .GetWasteColor              ; lower
                    
                    cmp #Level_05WastePoleRiMax     ; right outer side
                    bcs .GetWasteColor              ; higher/equal
                    
.SetDeath           lda #M2Z_PlayerDeadWobbleYes    ; player in waste range
                    sta M2Z_PlayerDeadWobble        ; 
                    
.GetWasteColor      lda Level_05WasteColor + $01    ; flicker waste color
                    and #$0f                        ; ....#### - isolate
                    cmp #GREEN                      ; 
                    bcc .IncWasteColor              ; lower
                    
.IniWasteColor      lda #HR_BlackBlack              ; 
                    
.IncWasteColor      adc #HR_GreyWhite               ; 
                    ldx #$a0                        ; 
.SetWasteColor      sta Level_05WasteColor,x        ; 
                    dex                             ; 
                    bne .SetWasteColor              ; 
                    
                    jsr L__StdHandler               ; 
                    bcc .Level05Loop                ; level not solved - but still alive
                    
Level_05_HandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; Level_06_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_06_Handler    subroutine                      ; 
                    ldy #>Level_06                  ; 
                    ldx #<Level_06                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    jsr L__Init                     ; level init
                    jsr L__InitXmitDigits           ; 
                    
                    ldy #$04 * M2_WaFlowLen         ; $04 moving flow tiles
.CpyFlowTileData    lda (M2Z_LvlFlowDataPtr),y      ; 
                    sta M2_WaFlow,y                 ; 
                    dey                             ; 
                    bpl .CpyFlowTileData            ; 
                    
                    jsr L__InitStatusRow            ; update status line
                    
.Level06Loop        jsr L__WaitMoveSprts            ; 
                    jsr L__FlowTileHandler          ; 
                    jsr L__StdHandler               ; 
                    bcc .Level06Loop                ; level not solved - but still alive
                    
Level_06_HandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; Level_07_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_07_Handler    subroutine                      ; 
                    ldx #<Level_07                  ; 
                    ldy #>Level_07                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    jsr L__Init                     ; level init
                    
                    lda #$64                        ; 
                    sta M2Z_LifterPlatformPosX      ; 
                    lda #$a9                        ; 
                    sta M2Z_LifterPlatformPosY      ; 
                    
                    ldy #M2_WaSpriteLen * $02 + $01 ; 2 synchronized flow tiles
.CpyKillFlowData    lda (M2Z_LvlXtraDataPtr),y      ; kill flow tile data
                    sta M2_WaSprite_0a,y            ; 
                    dey                             ; 
                    bpl .CpyKillFlowData            ; 
                    
                    lda #<Level_07LifterScreen      ; 
                    sta M2Z_LiftColorRamPtrLo       ; 
                    lda #>Level_07LifterScreen      ; 
                    sta M2Z_LiftColorRamPtrHi       ; 
                    
                    lda #Level_07LifterHeight       ; height
                    sta M2Z_LiftHeight              ; 
                    
                    lda #<Level_07LifterColor       ; 
                    sta M2Z_LiftColorScrPtrLo       ; 
                    lda #>Level_07LifterColor       ; 
                    sta M2Z_LiftColorScrPtrHi       ; 
                    
.SetLifterWidth     ldy #$0f                        ; 
.ColorLifter        lda #HR_WhiteWhite              ; 
                    sta (M2Z_LiftColorRamPtr),y     ; color ram
                    sta (M2Z_LiftColorScrPtr),y     ; color screen
                    dey                             ; 
                    bpl .ColorLifter                ; 
                    
.SetNextScreenRow   clc                             ; 
                    lda #M2_ScreenGameRowLen        ; next color ram line
                    adc M2Z_LiftColorRamPtrLo       ; 
                    sta M2Z_LiftColorRamPtrLo       ; 
                    bcc .SetNextColorRow            ; 
                    inc M2Z_LiftColorRamPtrHi       ; 
                    
.SetNextColorRow    clc                             ; 
                    lda #M2_ScreenGameRowLen        ; next color screen row
                    adc M2Z_LiftColorScrPtrLo       ; 
                    sta M2Z_LiftColorScrPtrLo       ; 
                    bcc .DecLifterHeight            ; 
                    inc M2Z_LiftColorScrPtrHi       ; 
                    
.DecLifterHeight    dec M2Z_LiftHeight              ; 
                    bne .SetLifterWidth             ; 
                    
                    jsr L__InitStatusRow            ; update status line
                    
.Level07Loop        lda M2Z_WaitIRQs                ; 
                    cmp #$02                        ; 
                    bcc .Level07Loop                ; 
                    
                    lda #$00                        ; 
                    sta M2Z_WaitIRQs                ; 
                    
                    jsr MoveEnemies                 ; 
                    jsr CheckLifterStart            ; 
                    
                    lda M2Z_LifterStatus            ; 
                    bpl .HandlePlayer               ; inactive
                    
                    jsr L07LifterHandler            ; 
                    jmp .HandleLevel                ; 
                    
.HandlePlayer       jsr MovePlayer                  ; 
                    jsr L__SprtCollHandler          ; 
                    
.HandleLevel        jsr L__ItemChecker              ; 
                    jsr L07BadFlowHandler           ; 
                    jsr L__TimeHandler              ; 
                    jsr L__ShowStatusRow            ; 
                    jsr L__CheckSolved              ; 
                    bcc .Level07Loop                ; no - but still alive
                    
Level_07_HandlerX   rts                             ; solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; Level_08_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_08_Handler    subroutine                      ; 
                    ldx #<Level_08                  ; 
                    ldy #>Level_08                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
                    ldy #M2_WaMisc_StomperWaLen - 1 ; 
.SaveStomperData    lda (M2Z_LvlStompDataPtr),y     ; 
                    sta M2_WaMisc_Stomper,y         ; 
                    dey                             ; 
                    bpl .SaveStomperData            ; 
                    
                    ldy #$06 * M2_WaFlowLen         ; $06 moving flow tiles
.CpyFlowTileData    lda (M2Z_LvlFlowDataPtr),y      ; 
                    sta M2_WaFlow,y                 ; 
                    dey                             ; 
                    bpl .CpyFlowTileData            ; 
                    
                    jsr L__Init                     ; level init
                    jsr L__InitStatusRow            ; update status line
                    
.Level08Loop        jsr L__WaitMoveSprts            ; 
                    jsr L08FlowHandler              ; 
                    jsr L08StompHandler             ; 
                    jsr L__StdHandler               ; 
                    bcc .Level08Loop                ; level not solved - but still alive
                    
Level_08_HandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; Level_09_Handler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Level_09_Handler    subroutine                      ; 
                    ldx #<Level_09                  ; 
                    ldy #>Level_09                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    
                    ldx #M2Z_LevelTextNormal        ; non reversed
                    ldy #$aa                        ; color pointer
                    jsr L__TextOut                  ; color pointer
                    
                    lda #$3d                        ; "="
                    ldx #$52                        ; PosX
                    ldy #$10                        ; PosY
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda #$30                        ; "0"
                    ldx #$6a                        ; PosX
                    ldy #$10                        ; PosY
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda #HR_BlackBlack              ; 
                    ldx #$0d                        ; 
.HideWarning        sta Level_09TextWarnColor,x     ; make "DANGER" invisible
                    dex                             ; 
                    bpl .HideWarning                ; 
                    
                    lda Level_09Cannon              ; get actual color
                    and #$0f                        ; clear color 1
                    ora #HR_DkGreyBlack             ; set color 1
                    tay                             ; save cannon base color
                    ldx #Level_09CannonHeight       ; amount
                    tya                             ; 
.ColourCannon       lda #HR_DkGreyDkGrey            ; 
                    sta Level_09CannonRow1,x        ; set cannon color
                    sta Level_09CannonRow3,x        ; 
                    sta Level_09CannonRow4,x        ; 
                    sta Level_09CannonRow5,x        ; set cannon base color
                    lda #HR_GreyGrey                ; .hbu001.
                    sta Level_09CannonRow2,x        ; .hbu001.
                    dex                             ; 
                    bpl .ColourCannon               ; 
                    
                    lda #M2Z_CannonAtPosXMax        ; 
                    sta M2Z_CannonAtPosX            ; 
                    
                    lda #$00                        ; 
                    sta M2Z_CannonStatus            ; 
                    sta M2Z_AmountHex               ; 
                    sta M2Z_AmountTNT10             ; 
                    sta M2Z_AmountTNT100            ; 
                    jsr L__Init                     ; level init
                    jsr L__InitStatusRow            ; update status line
                    
.Level09Loop        jsr L__WaitMoveSprts            ; 
                    jsr L09CannonHandler            ; 
                    jsr L__ItemChecker              ; 
                    jsr L09TNTHandler               ; 
                    jsr L09TNTBoxWallChk            ; 
                    jsr L__SprtCollHandler          ; 
                    jsr L__ShowStatusRow            ; 
                    jsr L__TimeHandler              ; 
                    jsr L__CheckSolved              ; 
                    bcc .Level09Loop                ; no - but still alive
                    
Level_09_HandlerX   rts                             ; solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; DemoLevelHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DemoLevelHandler    subroutine                      ; 
                    ldx #<Level_00                  ; 
                    ldy #>Level_00                  ; 
                    jsr L__CopyHeader               ; copy $40 bytes to $5c-$9b
                    jsr L__Init                     ; level init
                    jsr L__InitStatusRow            ; update status line
                    
                    lda #$00                        ; 
                    sta M2Z_DemoPtrMoveTab          ; 
                    lda M2_WaSprite_PosX            ; 
                    sta M2Z_DemoSpritePosX          ; 
                    
.LevelDemoLoop      lda M2Z_WaitIRQs                ; 
                    eor #$02                        ; 
                    bne .LevelDemoLoop              ; 
                    
                    sta M2Z_WaitIRQs                ; $00
                    
                    jsr MoveEnemies                 ; 
                    jsr CheckJoystick               ; 
                    lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    beq .DemoNextMove               ; continue
                    
                    lda #$01                        ; 
                    sta M2Z_DemoMode                ; 
                    
.ExitDemo           rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
.DemoNextMove       lda M2Z_DemoSpritePosX          ; 
                    beq .ChkArivalPosY              ; 
                    
.ChkArivalPosX      cmp M2_WaSprite_PosX            ; 
                    bne .CopyDemoMovesI             ; 
                    beq .SetNextArivalPos           ; 
                    
.ChkArivalPosY      ldx M2Z_DemoSpritePosY          ; 
                    cpx M2_WaSprite_PosY            ; 
                    beq .SetNextArivalPos           ; 
                    
                    dex                             ; 
                    cpx M2_WaSprite_PosY            ; 
                    bne .CopyDemoMovesI             ; 
                    
.SetNextArivalPos   ldx M2Z_DemoPtrMoveTab          ; 
                    lda TabDemoSpritePosX,x         ; 
                    sta M2Z_DemoSpritePosX          ; 
                    
                    lda TabDemoSpritePosY,x         ; 
                    sta M2Z_DemoSpritePosY          ; 
                    
                    ldy #$00                        ; preset no fire
                    lda TabDemoMoves,x              ; 
                    bpl .SetDemoFire                ; 
                    
                    dey                             ; $ff - fire
                    
.SetDemoFire        sty M2Z_DemoJoyFire             ; 
                    
.GetDemoMoveLiRe    and #$03                        ; ......##
                    tay                             ; 
                    lda TabDemoMoveDir,y            ; 
                    sta M2Z_DemoJoyLeRi             ; 
                    
.GetDemoMoveUpDo    lda TabDemoMoves,x              ; ....##..
                    lsr a                           ; .....##.
                    lsr a                           ; ......##
                    and #$03                        ; 
                    tay                             ; 
                    lda TabDemoMoveDir,y            ; 
                    sta M2Z_DemoJoyUpDo             ; 
                    
                    inc M2Z_DemoPtrMoveTab          ; 
                    
.CopyDemoMovesI     ldx #$02                        ; 
.CopyDemoMoves      lda M2Z_DemoJoyMove,x           ; 
                    sta M2Z_JoystickMove,x          ; le=$ff ri=$01 / up=$ff do=$01 / fi=ff
                    dex                             ; 
                    bpl .CopyDemoMoves              ; 
                    
                    jsr PlayerMoveTypes             ; 
                    jsr L__StdHandler               ; 
                    bcc .LevelDemoLoop              ; level not solved - but still alive
                    
DemoLevelHandlerX   rts                             ; level solved or dead
; -------------------------------------------------------------------------------------------------------------- ;
; ItemGraphicOut    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ItemGraphicOut      subroutine                      ; 
.SavePtr            lda M2Z_TargetScreenPosX        ; 
                    sta M2Z_21                      ; 
                    lda M2Z_TargetScreenPosY        ; 
                    sta M2Z_22                      ; 
                    
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTileOut              ; 
                    
                    lda M2Z_22                      ; 
                    clc                             ; 
                    adc #$08                        ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    jsr IncItemCtrlPtr              ; 
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTileOut              ; 
                    
                    lda M2Z_22                      ; 
                    sta M2Z_TargetScreenPosY        ; 
                    lda M2Z_21                      ; 
                    clc                             ; 
                    adc #$04                        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    
                    jsr IncItemCtrlPtr              ; 
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTileOut              ; 
                    
                    lda M2Z_22                      ; 
                    clc                             ; 
                    adc #$08                        ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    jsr IncItemCtrlPtr              ; 
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTileOut              ; 
                    
.RestorePtr         lda M2Z_21                      ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda M2Z_22                      ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
ItemGraphicOutX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IncItemCtrlPtr    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IncItemCtrlPtr      subroutine                      ; 
                    clc                             ; 
                    lda #$08                        ; 
                    adc M2Z_TileGfxPtrLo            ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    bcc IncItemCtrlPtrX             ; 
                    inc M2Z_TileGfxPtrHi            ; 
                    
IncItemCtrlPtrX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearScoreToAdd   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearScoreToAdd     subroutine                      ; 
                    lda #$00                        ; 
                    ldx #$05                        ; 
.Clear              sta M2Z_ScoreToAdd,x            ; 
                    dex                             ; 
                    bpl .Clear                      ; 
                    
ClearScoreToAddX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IncScore          Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IncScore            subroutine                      ; 
                    ldx #$05                        ; 
                    clc                             ; 
.GetScore           lda M2Z_Score,x                 ; 
                    adc M2Z_ScoreToAdd,x            ; 
                    cmp #$0a                        ; 
                    bcc .SetScore                   ; 
                    
                    adc #$05                        ; 
                    and #$0f                        ; 
                    sec                             ; 
.SetScore           sta M2Z_Score,x                 ; 
                    dex                             ; 
                    bpl .GetScore                   ; 
                    
IncScoreX           rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShowStatusScore   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShowStatusScore     subroutine                      ; 
                    ldx #$00                        ; 
                    stx M2Z_StatusScoreDecPt        ; 
                    
.FirstThree         lda M2Z_Score,x                 ; 
                    jsr DigitOrBlank                ; 
                    
                    sta M2_ScreenStatusLine,x       ; 
                    inx                             ; 
                    cpx #$03                        ; 
                    bcc .FirstThree                 ; lower
                    
                    lda #$20                        ; " "
                    ldy M2Z_StatusScoreDecPt        ; 
                    beq .SetDecPoint                ; 
                    
                    lda #$2c                        ; ","
                    
.SetDecPoint        sta M2_ScreenStatusLine,x       ; 
                    
.LastThree          lda M2Z_Score,x                 ; 
                    jsr DigitOrBlank                ; 
                    
                    sta M2_ScreenStatusLine + 1,x   ; 
                    inx                             ; 
                    cpx #$06                        ; 
                    bcc .LastThree                  ; lower
                    
ShowStatusScoreX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DigitOrBlank      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DigitOrBlank        subroutine                      ; 
                    beq .ChkFirst                   ; 
                    
.SetDecimalPoint    inc M2Z_StatusScoreDecPt        ; 
                    
.ChkFirst           cpx #$05                        ; always print first digit
                    bcs .ReturnDigit                ; greater/equal
                    
                    ldy M2Z_StatusScoreDecPt        ; 
.ChkDecimalPoint    bne .ReturnDigit                ; 
                    
.ReturnBlank        lda #$20                        ; " " - return blank if zero
                    rts                             ; 
                    
.ReturnDigit        ora #$30                        ; ..##.... - make digit
DigitOrBlankX       rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShowStatusLives   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShowStatusLives     subroutine                      ; 
                    lda #$20                        ; " "
                    sta M2Z_StatusLives01           ; 
                    sta M2Z_StatusLives02           ; 
                    sta M2Z_StatusLives03           ; 
                    
                    ldy M2Z_NoOfLives               ; 
                    beq .SetStausLiveHeads          ; $00
                    
                    dey                             ; 
                    beq .SetStausLiveHeads          ; $01
                    
                    lda #$00                        ; char head
                    sta M2Z_StatusLives01           ; 
                    
                    dey                             ; 
                    beq .SetStausLiveHeads          ; $02
                    
                    sta M2Z_StatusLives02           ; 
                    
                    dey                             ; 
                    beq .SetStausLiveHeads          ; $03
                    
                    sta M2Z_StatusLives03           ; 
                    
.SetStausLiveHeads  lda M2Z_StatusLives01           ; 
                    sta M2_ScreenStatusLives3       ; 
                    lda M2Z_StatusLives02           ; 
                    sta M2_ScreenStatusLives2       ; 
                    lda M2Z_StatusLives03           ; 
                    sta M2_ScreenStatusLives1       ; 
                    
ShowStatusLivesX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShowStatusTime    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShowStatusTime      subroutine                      ; 
                    lda M2Z_TimeTen                 ; 
                    ora #$30                        ; ..##.... - make digit
                    sta M2_ScreenStatusTime10       ; 
                    
                    lda M2Z_TimeOne                 ; 
                    ora #$30                        ; ..##.... - make digit
                    sta M2_ScreenStatusTime1        ; 
                    
ShowStatusTimeX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__InitStatusRow  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__InitStatusRow    subroutine                      ; 
                    lda #$02                        ; 
                    sta M2Z_CountIRQs               ; 
                    
.WaitRaster         lda RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    cmp #$fa                        ; 
                    bne .WaitRaster                 ; 
                    
                    lda #$fe                        ; 
                    sta RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    
                    lda #$01                        ; Bit 0: Flag: Raster Compare IRQ caused an IRQ? (1=yes)
                    sta IRQMASK                     ; VIC($D01A) IRQ Mask Register
                    sta VICIRQ                      ; VIC($D019) Interrupt Flags - Latched flags cleared if set to 1
                    
                    lda #$00                        ; 
L__InitStatusRowX   sta M2Z_WaitIRQs                ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__ShowStatusRow  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__ShowStatusRow    subroutine                      ; 
                    jsr ShowStatusScore             ; display score in status line
                    jsr ShowStatusLives             ; display no of lives as heads in status line
L__ShowStatusRowX   jmp ShowStatusTime              ; display time in status line
; -------------------------------------------------------------------------------------------------------------- ;
; IncItemTabPtr     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IncItemTabPtr       subroutine                      ; 
                    clc                             ; 
                    adc M2Z_ItemTabPtrLo            ; 
                    sta M2Z_ItemTabPtrLo            ; 
                    bcc IncItemTabPtrX              ; 
                    inc M2Z_ItemTabPtrHi            ; 
                    
IncItemTabPtrX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CopyData          Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CopyData            subroutine                      ; 
                    ldx #$00                        ; 
                    ldy #$00                        ; 
.GetNext            lda (M2Z_FromAreaPtr),y         ; 
                    sta (M2Z_FillAreaPtr),y         ; 
                    iny                             ; 
                    bne .ChkMax                     ; 
                    
                    inx                             ; 
                    inc M2Z_FromAreaPtrHi           ; 
                    inc M2Z_FillAreaPtrHi           ; 
.ChkMax             cpy M2Z_FillAreaLenBytes        ; 
                    bne .GetNext                    ; 
                    
                    cpx M2Z_FillAreaLenPages        ; 
                    bne .GetNext                    ; 
                    
CopyDataX           rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; FillArea          Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
FillArea            subroutine                      ; 
                    ldx #$00                        ; no of pages
                    ldy #$00                        ; no of bytes
.FillWithAc         sta (M2Z_FillAreaPtr),y         ; 
                    iny                             ; 
                    bne .ChkBytes                   ; 
                    
                    inx                             ; 
                    inc M2Z_FillAreaPtrHi           ; 
                    
.ChkBytes           cpy M2Z_FillAreaLenBytes        ; max bytes
                    bne .FillWithAc                 ; 
                    
.ChkPages           cpx M2Z_FillAreaLenPages        ; max pages
                    bne .FillWithAc                 ; 
                    
FillAreaX           rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTileCopy   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTileCopy     subroutine                      ; 
                    lda #$00                        ; 
                    ldx #$08                        ; 
.ClrTile1Store      sta M2Z_TilePart1Mask,x         ; mask1 and part1 to $00
                    dex                             ; 
                    bpl .ClrTile1Store              ; 
                    
                    stx M2Z_TilePart2Mask           ; mask2 to $ff
                    
                    ldy #$07                        ; 
.GetSimpleTileParts lda (M2Z_TileGfxPtr),y          ; part2 to tile values
                    sta M2Z_TilePart1Start,y        ; 
                    dey                             ; 
                    bpl .GetSimpleTileParts         ; 
                    
                    lda M2Z_TargetScreenPosX        ; 
                    and #$03                        ; 4 different screen positions
                    beq GraphicTileCopyX            ; no coorection for position $00
                    
                    asl a                           ; *2 - correction pos counter
                    tay                             ; 
.GetTilePartMax     ldx #$07                        ; shift correction for position $01-$03
                    lsr M2Z_TilePart2Mask           ; ######## ........  ->  .####### #.......
                    ror M2Z_TilePart1Mask           ; 
.ShiftTileParts     lsr M2Z_TilePart1,x             ; 
                    ror M2Z_TilePart2,x             ; 
                    dex                             ; 
                    bpl .ShiftTileParts             ; 
                    
                    dey                             ; 
                    bne .GetTilePartMax             ; 
                    
GraphicTileCopyX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetHiresPtr       Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetHiresPtr         subroutine                      ; 
                    ldx M2Z_TargetScreenPosY        ; 
                    lda TabPtrHiresLo,x             ; 
                    sta M2Z_HiResScreenPtrLo        ; 
                    lda TabPtrHiresHi,x             ; 
                    sta M2Z_HiResScreenPtrHi        ; 
                    
                    lda M2Z_TargetScreenPosX        ; 
                    and #$fc                        ; ######..
                    asl a                           ; 
                    bcc B_899c                      ; 
                    
                    inc M2Z_HiResScreenPtrHi        ; 
                    
B_899c              clc                             ; 
                    adc M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_HiResScreenPtrLo        ; 
                    bcc GetHiresPtrX                ; 
                    inc M2Z_HiResScreenPtrHi        ; 
                    
GetHiresPtrX        rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTileOut    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTileOut      subroutine                      ; 
                    ldx #$00                        ; part count
.NextTilePart       txa                             ; 
                    pha                             ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    pla                             ; 
                    tax                             ; 
                    
                    ldy #$00                        ; 
                    lda (M2Z_HiResScreenPtr),y      ; 
                    and M2Z_TilePart1Mask           ; 
                    ora M2Z_TilePart1,x             ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    ldy #$08                        ; 
                    lda (M2Z_HiResScreenPtr),y      ; 
                    and M2Z_TilePart2Mask           ; 
                    ora M2Z_TilePart2,x             ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    inc M2Z_TargetScreenPosY        ; 
                    inx                             ; 
                    cpx #$08                        ; $08 parts
                    bne .NextTilePart               ; 
                    
GraphicTileOutX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DisplayLevel      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DisplayLevel        subroutine                      ; 
                    ldy #M2Z_LvlTileWorkLen - 1     ; get level tile data block
.CpyItemData        lda (M2Z_ItemTabPtr),y          ; 
                    sta M2Z_LvlTileWork,y           ; 
                    dey                             ; 
                    bpl .CpyItemData                ; 
                    
                    tay                             ; 
                    bne .DisplayLevelTiles          ; 
                    
.ExitEnOfTileData   rts                             ; 
                    
.DisplayLevelTiles  lda #M2Z_LvlTileWorkLen         ; 
                    jsr IncItemTabPtr               ; point to next data block
                    
                    lda M2Z_LvlTile                 ; 
                    and #LevelTileTypeMaskClr       ; 
                    sta M2Z_LvlTileClearCount       ; 
                    
                    lda M2Z_LvlTile                 ; 
                    and #LevelTileTypeMaskGrnd + LevelTileTypeMaskNo ; check flag / isolate level tile no
                    bpl .SetTileNo                  ; 
                    
                    and #LevelTileTypeMaskNo        ; isolate level tile no
                    clc                             ; 
                    adc #$08                        ; point to next tile - ground
                    
.SetTileNo          sta M2Z_LvlTileDataPtrEven      ; 
                    
                    lda M2Z_LvlTile                 ; 
                    and #LevelTileTypeMaskNo        ; isolate level tile no
                    bne .SetGroundTileType          ; 
                   
                    lda #NoLvlTileGroundNew2        ; NoLvlTileGroundNew2 - 2nd form
                   
.SetGroundTileType  sta M2Z_LvlTileDataPtrOdd       ; 
                    
                    lda M2Z_LvlTileDir              ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; isolate left half
                    cmp #$0f                        ; 
                    bne .SetNextPosY                ; 
                    
                    ora #$f0                        ; x-1 y-1
                    
.SetNextPosY        sta M2Z_LvlTileNextPosY         ; 
                    
                    lda M2Z_LvlTileDir              ; 
                    and #$0f                        ; 
                    cmp #$0f                        ; 
                    bne .SetNextPosX                ; 
                    
                    ora #$f0                        ; x-1 y-1
                    
.SetNextPosX        sta M2Z_LvlTileNextPosX         ; 
                    
                    lda #$00                        ; 
                    sta M2Z_LvlOddEvenFlag          ; tile part1/part2 selector
                    
.SetNextTilePart    inc M2Z_LvlOddEvenFlag          ; 
                    lda M2Z_LvlOddEvenFlag          ; 
                    lsr a                           ; odd/even flip-flop
                    
                    ldx M2Z_LvlTileDataPtrOdd       ; 
                    bcs .SetLvlTilePtr              ; any odd  value
                    ldx M2Z_LvlTileDataPtrEven      ; any even value 
.SetLvlTilePtr      clc                             ; .hbu001.
                    txa                             ; .hbu001.
                    adc #<LevelTileStore            ; .hbu001.
                    sta M2Z_TileGfxPtrLo            ; .hbu001.
                    lda #$00                        ; .hbu001.
                    adc #>LevelTileStore            ; .hbu001.
                    sta M2Z_TileGfxPtrHi            ; 
                    
                    lda M2Z_LvlTilePosX             ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda M2Z_LvlTilePosY             ; 
                    sta M2Z_TargetScreenPosY        ; 
                    jsr GraphicTileCopy             ; 
                    
                    lda #$ff                        ; reset masks
                    sta M2Z_TilePart1Mask           ; 
                    sta M2Z_TilePart2Mask           ; 
                    
                    lda #$00                        ; 
                    ldx M2Z_LvlTileClearCount       ; bits .....### of M2Z_LvlTile
                    beq .GfxTileOut                 ; 
                    
.ClrPartTileData    sta M2Z_TilePart1 - 1,x         ; cut off info from ladders top to fit the different grounds
                    sta M2Z_TilePart2 - 1,x         ; 
                    dex                             ; 
                    bne .ClrPartTileData            ; 
                    
                    stx M2Z_LvlTileClearCount       ; $00
                    
.GfxTileOut         jsr GraphicTileOut              ; 
                    
                    dec M2Z_LvlTileCount            ; 
                    bne .SetNextTilePtr             ; 
                    
.GetNextDataBlock   jmp DisplayLevel                ; 
                    
.SetNextTilePtr     clc                             ; 
                    lda M2Z_LvlTilePosX             ; 
                    adc M2Z_LvlTileNextPosX         ; 
                    sta M2Z_LvlTilePosX             ; 
                    
                    clc                             ; 
                    lda M2Z_LvlTilePosY             ; 
                    adc M2Z_LvlTileNextPosY         ; 
                    sta M2Z_LvlTilePosY             ; 
                    
DisplayLevelTilesX  jmp .SetNextTilePart            ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearHiScreen     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearHiScreen       subroutine                      ; 
                    lda #<M2_Screen_HiRes           ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>M2_Screen_HiRes           ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) point to hires screen
                    
                    lda #<M2_Screen_HiResLen        ; 
                    sta M2Z_FillAreaLenBytes        ; 
                    lda #>M2_Screen_HiResLen        ; 
                    sta M2Z_FillAreaLenPages        ; ($2d/$2e) length = $2000
                    
                    lda #$00                        ; fill byte
ClearHiScreenX      jmp FillArea                    ; fill area
; -------------------------------------------------------------------------------------------------------------- ;
; ColourScreen      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ColourScreen        subroutine                      ; 
                    pha                             ; save colour
                    
                    lda #<M2_ScreenGameLen          ; 
                    sta M2Z_FillAreaLenBytes        ; 
                    lda #>M2_ScreenGameLen          ; 
                    sta M2Z_FillAreaLenPages        ; ($2d/$2e) length = $03e8
                    lda #<M2_ScreenGame             ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>M2_ScreenGame             ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) point to $6000
                    
                    pla                             ; restore colour - fill byte
ColourScreenX       jmp FillArea                    ; fill area
; -------------------------------------------------------------------------------------------------------------- ;
; ColourRam         Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ColourRam           subroutine                      ; 
                    pha                             ; save colour
                    
                    lda #<M2_ScreenGameLen          ; 
                    sta M2Z_FillAreaLenBytes        ; 
                    lda #>M2_ScreenGameLen          ; 
                    sta M2Z_FillAreaLenPages        ; ($2d/$2e) length = $03e8
                    
                    lda #<COLORAM                   ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>COLORAM                   ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) point to $d800
                    
                    pla                             ; restore colour - fill byte
ColourRamX          jmp FillArea                    ; fill area
; -------------------------------------------------------------------------------------------------------------- ;
; EnemyMoveOne      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
EnemyMoveOne        subroutine                      ; 
                    ldy #$08                        ; 
                    lda (M2Z_SpriteWaPtr),y         ; M2_WaSprite_StepWait
                    tax                             ; 
                    dex                             ; count down
                    txa                             ; 
                    sta (M2Z_SpriteWaPtr),y         ; 
                    bne EnemyMoveOneX               ; 
                    
                    iny                             ; 
                    lda (M2Z_SpriteWaPtr),y         ; M2_WaSprite_Speed
                    dey                             ; 
                    sta (M2Z_SpriteWaPtr),y         ; M2_WaSprite_StepWait
                    
                    ldy #$0d                        ; 
                    lda (M2Z_SpriteWaPtr),y         ; 
                    sec                             ; 
                    adc #$00                        ; 
                    and #$01                        ; 
                    sta (M2Z_SpriteWaPtr),y         ; 
                    
                    ldy #$0a                        ; 
                    lda (M2Z_SpriteWaPtr),y         ; M2_WaSprite_MoveDir - $00 / $01(+1) $ff(-1)
                    tax                             ; 
                    
                    clc                             ; 
                    ldy #$01                        ; 
                    adc (M2Z_SpriteWaPtr),y         ; M2_WaSprite_PosX
                    sta (M2Z_SpriteWaPtr),y         ; M2_WaSprite_PosX
                    sta M2Z_SpriteWAPosX            ; 
                    
                    txa                             ; M2_WaSprite_MoveDir
                    bmi .ChkMaxLeft                 ; 
                    
.ChkMaxRight        ldy #$0c                        ; 
                    lda M2Z_SpriteWAPosX            ; 
                    cmp (M2Z_SpriteWaPtr),y         ; M2_WaSprite_MoveMaxX
                    bcs .ReverseMoveDir             ; greater /equal
                    bcc .GetSpritePtr               ; lower
                    
.ChkMaxLeft         ldy #$0b                        ; 
                    lda M2Z_SpriteWAPosX            ; 
                    cmp (M2Z_SpriteWaPtr),y         ; M2_WaSprite_MoveMinX
                    bcs .GetSpritePtr               ; greater /equal
                    
.ReverseMoveDir     txa                             ; M2_WaSprite_MoveDir
                    eor #$ff                        ; reverse
                    sec                             ; 
                    adc #$00                        ; +1
                    ldy #$0a                        ; 
                    sta (M2Z_SpriteWaPtr),y         ; M2_WaSprite_MoveDir - $00 / $01(+1) $ff(-1)
                    
.GetSpritePtr       ldx #PtrSprEneMovSm01           ; preset
                    
                    lda M2Z_EnemySmileTime          ; 
                    bne .SetSpritePtr               ; 
                    
                    ldx #PtrSprEneMovRi01           ; 
                    ldy #$0a                        ; 
                    lda (M2Z_SpriteWaPtr),y         ; M2_WaSprite_MoveDir - $00 / $01(+1) $ff(-1)
                    bpl .SetSpritePtr               ; 
                    
                    ldx #PtrSprEneMovLe01           ; spritepointer $02
                    
.SetSpritePtr       txa                             ; 
                    
                    ldy #$0d                        ; 
                    clc                             ; 
                    adc (M2Z_SpriteWaPtr),y         ; M2_WaSprite_No
                    
                    ldy #$03                        ; 
                    sta (M2Z_SpriteWaPtr),y         ; M2_WaSprite_Pointer
                    
EnemyMoveOneX       rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerMoveLeRi    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerMoveLeRi      subroutine                      ; 
                    txa                             ; move_left=$ff  move_right=$01
                    clc                             ; 
                    adc M2_WaSprite_PosX            ; 
                    cmp #M2_WaSprite_PosXMax        ; 
                    bcs .SetLastDir                 ; greater/equal
                    
                    cmp #M2_WaSprite_PosXMin        ; 
                    bcc .SetLastDir                 ; lower
                    
.SetNewPosX         sta M2_WaSprite_PosX            ; 
                    
.SetLastDir         txa                             ; move_left=$ff  move_right=$01
                    eor M2_WaSprite_LastDir         ; 
                    stx M2_WaSprite_LastDir         ; 
                    bpl .SetNextSprtLeRi            ; 
                    
                    lda #$02                        ; 
                    sta M2_WaSprite_NoPlayer        ; 
                    
                    lda #$01                        ; 
                    sta M2_WaSprite_NoCounter       ; 
                    bpl .SetSprtPtrLeRi             ; always
                    
.SetNextSprtLeRi    lda M2_WaSprite_NoPlayer        ; 
                    clc                             ; 
                    adc M2_WaSprite_NoCounter       ; 
                    sta M2_WaSprite_NoPlayer        ; 
                    beq .SetSprtNoCounter           ; lower bound
                    
                    cmp #$04                        ; upper bound - 5 different sprite le/ri images
                    bne .SetSprtPtrLeRi             ; 
                    
.SetSprtNoCounter   lda M2_WaSprite_NoCounter       ; flip between $01 and $ff (-1)
                    eor #$ff                        ; 
                    sec                             ; 
                    adc #$00                        ; 
                    sta M2_WaSprite_NoCounter       ; 
                    
                    lda #M2Z_SoundPlayerStepOn      ; 
                    sta M2Z_SoundPlayerStep         ; 
                    
.SetSprtPtrLeRi     ldy #PtrSprPlrMovLe01           ; preset
                    
                    txa                             ; move_left=$ff  move_right=$01
                    bmi .GetSprtPtrLeRi             ; 
                    
                    ldy #PtrSprPlrMovRi01           ; 
                    
.GetSprtPtrLeRi     tya                             ; 
                    clc                             ; 
                    adc M2_WaSprite_NoPlayer        ; 
                    sta M2_WaSprite_Pointer         ; 
                    
PlayerMoveLeRiX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerJumpUp      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerJumpUp        subroutine                      ; 
                    dec M2_WaSprite_PosY            ; move up a bit
                    
                    lda M2_WaSprite_PosX            ; 
                    clc                             ; 
                    adc M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    cmp #M2_WaSprite_PosXMax        ; max PosX on screen
                    bcs .DecJumpHeight              ; greater/equal
                    
                    cmp #M2_WaSprite_PosXMin        ; min PosX on screen
                    bcc .DecJumpHeight              ; lower
                    
                    sta M2_WaSprite_PosX            ; 
                    
.DecJumpHeight      dec M2_WaSprite_MoveTime        ; jump heigth
                    lda M2_WaSprite_MoveTime        ; 
                    bne .ChkMaxUp                   ; 
                    
                    lda #M2_WaSprite_MoveFallDo     ; fall down again
                    sta M2_WaSprite_MoveType        ; 
                    
                    lda M2_WaSprite_PosY            ; 
                    cmp #M2_WaSprite_PosYMin        ; max PosY on screen
                    bcs .Exit                       ; greater/equal
                    
                    lda #PtrSprPlrFallDown          ; player shot up over top of screen
                    sta M2_WaSprite_Pointer         ; 
                    
.Exit               rts                             ; 

.ChkMaxUp           cmp #M2_WaSprite_MoveTimeJuUp   ; moved up max
                    bcc .ChkJumpDir                 ; lower=not at max up
                    
                    dec M2_WaSprite_MoveTime        ; 
                    dec M2_WaSprite_PosY            ; move up a bit
                    
.ChkJumpDir         ldx #PtrSprPlrJumpLe            ; preset
                    
                    lda M2_WaSprite_LastDir         ; move_no=$00  move_left=$ff  move_right=$01
                    bmi .SetSpritePtr               ; 
                    
                    ldx #PtrSprPlrJumpRi            ; 
                    
.SetSpritePtr       stx M2_WaSprite_Pointer         ; 
                    
PlayerJumpUpX       rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerFallDown    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerFallDown      subroutine                      ; 
                    inc M2_WaSprite_MoveTime        ; 
                    inc M2_WaSprite_PosY            ; move down a bit
                    
                    lda M2_WaSprite_PosX            ; 
                    clc                             ; 
                    adc M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    cmp #M2_WaSprite_PosXMax        ; 
                    bcs .SetNextPosX                ; 
                    
                    cmp #M2_WaSprite_PosXMin        ; 
                    bcc .SetNextPosX                ; 
                    
                    sta M2_WaSprite_PosX            ; 
                    
.SetNextPosX        lda M2_WaSprite_PosX            ; 
                    sec                             ; 
                    sbc #$09                        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    
                    lda M2_WaSprite_PosY            ; 
                    sbc #$1e                        ; 
                    bcc .Exit1                      ; 
                    
                    sta M2Z_TargetScreenPosY        ; 
                    
                    jsr PlayerCheckGround           ; 
                    
                    php                             ; 
                    inc M2Z_TargetScreenPosY        ; 
                    plp                             ; 
                    beq .NextTargetPosX             ; 
                    
                    jsr PlayerCheckGround           ; 
                    beq .ChkNextPosYUp              ; 
                    
.NextTargetPosX     inc M2Z_TargetScreenPosX        ; 
                    
                    jsr PlayerCheckGround           ; 
                    beq .ChkNextPosYUp              ; 
                    
                    lda M2_WaSprite_MoveTime        ; 
                    cmp #$05                        ; 
                    bcc .Exit1                      ; 
                    
                    inc M2_WaSprite_PosY            ; move down a bit
                    inc M2_WaSprite_MoveTime        ; 
                    inc M2Z_TargetScreenPosY        ; 
                    
                    jsr PlayerCheckGround           ; 
                    beq .ChkNextPosYUp              ; 
                    
                    dec M2Z_TargetScreenPosX        ; 
                    
                    jsr PlayerCheckGround           ; 
                    beq .ChkNextPosYUp              ; 
                    
.Exit1              rts                             ; 

.ChkNextPosYUp      dec M2Z_TargetScreenPosY        ; move up a bit
                    
                    jsr PlayerCheckGround           ; 
                    beq .Exit1                      ; 
                    
                    dec M2Z_TargetScreenPosY        ; 
                    jsr PlayerCheckGround           ; 
                    beq .Exit1                      ; 
                    
                    lda M2_WaSprite_MoveTime        ; 
                    cmp #M2_WaSprite_MoveTimeJuDo   ; max jump down
                    bcc .ChkDeathFlag               ; lower
                    
                    ldx #M2Z_PlayerDeadSinkYes      ; 
                    stx M2Z_PlayerDeadSink          ; 
                    
                    ldx #M2Z_SoundKillOn            ; 
                    stx M2Z_SoundKill               ; 
                    
.SetNewMove         ldx #M2_WaSprite_MoveClear      ; 
                    stx M2_WaSprite_MoveType        ; 
                    
.ExitNewMove        rts                             ; 

.ChkDeathFlag       lda M2Z_PlayerDeadSink          ; 
                    bne .SetNewMove                 ; 
                    
                    jsr PlayerSprInitLeRi           ; 
PlayerFallDownX     jmp PlayerTryFlipGrnd           ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerSprInitLeRi Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerSprInitLeRi   subroutine                      ; 
                    lda #$02                        ; 
                    sta M2_WaSprite_StepWait        ; 
                    
                    lda #M2_WaSprite_MoveClear      ; 
                    sta M2_WaSprite_MoveType        ; 
                    
                    ldx #PtrSprPlrMovLe01           ; preset
                    
.GetLastMoveDir     lda M2_WaSprite_LastDir         ; 
                    bmi .SetStartSprite             ; 
                    
                    ldx #PtrSprPlrMovRi01           ; 
                    
.SetStartSprite     txa                             ; 
                    clc                             ; 
                    adc #$02                        ; 
                    sta M2_WaSprite_Pointer         ; 
                    
PlayerSprInitLeRiX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerCheckGround Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerCheckGround   subroutine                      ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_TargetScreenPosX        ; 
                    and #$03                        ; ......## - 4 different screen positions
                    tax                             ; 
                    
                    ldy #$00                        ; 
                    lda (M2Z_HiResScreenPtr),y      ; 
                    and TabGroundMasks,x            ; 
                    cmp TabGroundMasks,x            ; 
                    
PlayerCheckGroundX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerSlideHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerSlideHandler  subroutine                      ; 
                    inc M2_WaSprite_PosY            ; 
                    
                    lda M2_WaSprite_PosX            ; 
                    clc                             ; 
                    adc M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    sta M2_WaSprite_PosX            ; 
                    
                    ldx M2_WaSprite_MoveTime        ; 
                    dex                             ; 
                    beq .ChkLeRi                    ; 
                    
                    dec M2_WaSprite_MoveTime        ; 
                    
                    inc M2_WaSprite_PosY            ; 
                    
                    lda M2_WaSprite_PosX            ; 
                    clc                             ; 
                    adc M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    sta M2_WaSprite_PosX            ; 
                    
.ChkLeRi            ldx #PtrSprPlrJumpLe            ; 
                    lda M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    bmi .SetSpritePtr               ; 
                    
                    inx                             ; PtrSprPlrJumpRi
                    lda M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    bne .SetSpritePtr               ; 
                    
                    lda M2_WaSprite_LastDir         ; 
                    bpl .SetSpritePtr               ; 
                    
                    dex                             ; 
                    
.SetSpritePtr       stx M2_WaSprite_Pointer         ; 

                    dec M2_WaSprite_MoveTime        ; 
                    bne PlayerSlideHandlerX         ; 
                    
                    jsr PlayerSprInitLeRi           ; 
                    
                    lda M2Z_PlayerDeadSink          ; 
                    bne PlayerSlideHandlerX         ; 
                    
                    jsr PlayerTryFlipGrnd           ; 
                    
PlayerSlideHandlerX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerOnSlide     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerOnSlide       subroutine                      ; 
                    lda M2Z_LvlSlideDataPtrLo       ; 
                    sta M2Z_SlideDataPtrLo          ; 
                    lda M2Z_LvlSlideDataPtrHi       ; 
                    sta M2Z_SlideDataPtrHi          ; 
                    
.GetNextSlide       ldy #$00                        ; 
                    lda (M2Z_SlideDataPtr),y        ; LevelSlideData
                    bne .ChkOnSlideX                ; 
                    
.SetNotOnSlide      clc                             ; EndOfLevelData
.Exit               rts                             ; 

.ChkOnSlideX        iny                             ; 
                    lda (M2Z_SlideDataPtr),y        ; LevelSlidePosX
                    tax                             ; 
                    dex                             ; 
                    cpx M2_WaSprite_PosX            ; 
                    beq .ChkOnSlideY                ; 
                    
                    inx                             ; 
                    cpx M2_WaSprite_PosX            ; 
                    beq .ChkOnSlideY                ; 
                    
                    inx                             ; 
                    cpx M2_WaSprite_PosX            ; 
                    beq .ChkOnSlideY                ; 
                    
.SetNextSlideBlock  clc                             ; 
                    lda #LevelSlideDataLen          ; 
                    adc M2Z_SlideDataPtrLo          ; 
                    sta M2Z_SlideDataPtrLo          ; 
                    bcc .GetNextSlide               ; 
                    inc M2Z_SlideDataPtrHi          ; 
                    jmp .GetNextSlide               ; 
                    
.ChkOnSlideY        tax                             ; 
                    iny                             ; 
                    lda (M2Z_SlideDataPtr),y        ; LevelSlidePosY
                    cmp M2_WaSprite_PosY            ; 
                    bne .SetNextSlideBlock          ; 
                    
                    stx M2_WaSprite_PosX            ; 
                    
                    iny                             ; 
                    lda (M2Z_SlideDataPtr),y        ; LevelSlideDir
                    sta M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    ldy #$00                        ; 
                    lda (M2Z_SlideDataPtr),y        ; LevelSlideLength
                    sta M2_WaSprite_MoveTime        ; 
                    lda #M2_WaSprite_MoveSlide      ; 
                    sta M2_WaSprite_MoveType        ; 
                    
.SetOnSlide         sec                             ; 
PlayerOnSlideX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckJoystick     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckJoystick       subroutine                      ; 
                    lda CIAPRB                      ; CIA1($DC01) Data Port Register B - Joystick 1
                    jsr TestJoystickBits            ; 
                    stx M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    
                    jsr TestJoystickBits            ; 
                    stx M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    
                    jsr TestJoystickBits            ; fire=$ff
                    
                    txa                             ; 
                    and M2Z_JoystickFireRev         ; reversed
                    sta M2Z_JoystickFire            ; 
                    
                    txa                             ; 
                    eor #$ff                        ; 
                    sta M2Z_JoystickFireRev         ; reversed
                    
CheckJoystickX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TestJoystickBits  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TestJoystickBits    subroutine                      ; 
.NoMove             ldx #$00                        ; $00 - none
                    
                    lsr a                           ; Bit4:Fire Bit3:Right Bit2:Left Bit1:Down Bit0:Up  0=pressed
                    bcs .Test2nd                    ; 
                    
.1stMove            dex                             ; $ff - move_up  or  move_left  or  fire
                    
.Test2nd            lsr a                           ; 
                    bcs TestJoystickBitsX           ; 
                    
.2ndMove            inx                             ; $01 - move_down  or  move_right
                    
TestJoystickBitsX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerMoveTypes   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerMoveTypes     subroutine                      ; 
                    ldx M2_WaSprite_MoveType        ; 
                    bne .ChkJumpUp                  ; 
                    
                    jmp PlayerMoveHandler           ; $00 - get next move
                    
.ChkJumpUp          dex                             ; 
                    bne .ChkJumpDown                ; 
                    
                    jmp PlayerJumpUp                ; $01
                    
.ChkJumpDown        dex                             ; 
                    bne .ChkJumpClimb               ; 
                    
                    jmp PlayerFallDown              ; $02
                    
.ChkJumpClimb       dex                             ; 
                    bne .ChkJumpSlide               ; 
                    
                    jmp PlayerClimbHandler          ; $03
                    
.ChkJumpSlide       dex                             ; 
                    bne PlayerMoveTypesX            ; 
                    
                    jmp PlayerSlideHandler          ; $04

PlayerMoveTypesX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerMoveHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerMoveHandler   subroutine                      ; 
                    lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    beq .ChkUpDown                  ; 
                    
                    jmp PlayerJumpInit              ; 
                    
.ChkUpDown          lda M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    beq .ChkSlide                   ; 
                    bpl .ChkLadderDown              ; 
                    
                    jsr PlayerOnLadderBot           ; 
                    beq .ChkSlide                   ; no ladder
                    bne .SetMoveLadder              ; 
                    
.ChkLadderDown      jsr PlayerOnLadderTop           ; 
                    beq .ChkSlide                   ; no ladder
                    
.SetMoveLadder      lda #M2_WaSprite_MoveClimb      ; 
                    sta M2_WaSprite_MoveType        ; 
                    jmp PlayerClimbHandler          ; 
                    
.ChkSlide           jsr PlayerOnSlide               ; 
                    bcc .ChkLeftRight               ; not on slide
                    
                    jmp PlayerSlideHandler          ; 
                    
.ChkLeftRight       ldx M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    bne .GoPlayerMoveLeRi           ; 
                    
                    rts                             ; 

.GoPlayerMoveLeRi   jsr PlayerMoveLeRi              ; 
                    jsr PlayerChkGroundX            ; 
                    
                    lda M2_WaSprite_MoveType        ; 
                    cmp #M2_WaSprite_MoveFallDo     ; 
                    beq .SetStartSptPtr             ; 
                    
                    jmp PlayerTryFlipGrnd           ; and return

.SetStartSptPtr     jsr PlayerSprInitLeRi           ; 
                    
                    lda #M2_WaSprite_MoveFallDo     ; 
                    sta M2_WaSprite_MoveType        ; 
                    
                    lda #$00                        ; 
                    sta M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    sta M2_WaSprite_MoveTime        ; 
                    
PlayerMoveHandlerX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerClimbHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerClimbHandler  subroutine                      ; 
                    lda M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
.ChkClimbDir        bmi .ClimbUp                    ; 
                    bne .ClimbDown                  ; 
                    
.ExitChkClimbDir    rts                             ; 

.ClimbUp            dec M2_WaSprite_PosY            ; move up a bit
                    
                    lda M2Z_LadderPosYTop           ; 
                    cmp M2_WaSprite_PosY            ; 
                    bcc .ChkLadderTop               ; lower
                    
.SetNewPosY         sta M2_WaSprite_PosY            ; 
                    
                    jmp PlayerSprInitLeRi           ; and return

.ClimbDown          inc M2_WaSprite_PosY            ; move down a bit
                    
                    lda M2Z_LadderPosYBot           ; 
                    cmp M2_WaSprite_PosY            ; 
                    bcc .SetNewPosY                 ; lower
                    
.ChkLadderTop       lda M2_WaSprite_PosY            ; 
                    sec                             ; 
                    sbc M2Z_LadderPosYTop           ; 
                    cmp #$06                        ; 
                    bcs .SetSpriteLadder            ; greater/equal
                    
.SetSpriteLadderTop lda M2_WaSprite_PosY            ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    ror a                           ; 
                    lda #$00                        ; 
                    adc #PtrSprPlrMovLa03           ; top of ladder 01
                    sta M2_WaSprite_Pointer         ; 
                    
.ExitClimbDown      rts                             ; 

.SetSpriteLadder    lda M2_WaSprite_PosY            ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lda #$00                        ; 
                    adc #PtrSprPlrMovLa01           ; 
                    sta M2_WaSprite_Pointer         ; 
                    
PlayerClimbHandlerX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerJumpInit    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerJumpInit      lda #M2_WaSprite_MoveJumpUp     ; 
                    sta M2_WaSprite_MoveType        ; 
                    
                    lda #M2_WaSprite_MoveTimeJump   ; jump duration
                    sta M2_WaSprite_MoveTime        ; 
                    
                    lda M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    sta M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    
PlayerJumpInitX     jmp PlayerJumpUp                ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerChkGroundX  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerChkGroundX    subroutine                      ; 
                    jsr PlayerGetGroundPos          ; 
                    
                    lda M2Z_PlayerTilePosY          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    dec M2Z_TargetScreenPosY        ; check up a bit
                    
.ChkMaxXforY1       lda M2Z_PlayerTilePosXMax       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    bne .ChkMinXforY1               ; 
                    
                    dec M2_WaSprite_PosY            ; step up a bit
                    jmp PlayerChkGroundXX           ; 
                    
.ChkMinXforY1       lda M2Z_PlayerTilePosXMin       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    bne .StepDown1st                ; 
                    
                    dec M2_WaSprite_PosY            ; step up a bit
                    jmp PlayerChkGroundXX           ; 
                    
.StepDown1st        inc M2Z_TargetScreenPosY        ; step down a bit
                    
.ChkMaxXforYdown1   lda M2Z_PlayerTilePosXMax       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    beq PlayerChkGroundXX           ; 
                    
.ChkMinXforYdown1   lda M2Z_PlayerTilePosXMin       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    beq PlayerChkGroundXX           ; 
                    
.StepDown2nd        inc M2_WaSprite_PosY            ; step down a bit again
                    inc M2Z_TargetScreenPosY        ; 
                    
                    jsr PlayerCheckGround           ; 
                    beq PlayerChkGroundXX           ; 
                    
                    lda M2Z_PlayerTilePosXMax       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    beq PlayerChkGroundXX           ; 
                    
.LostSolidGround    lda #M2_WaSprite_MoveFallDo     ; 
                    sta M2_WaSprite_MoveType        ; 
                    
PlayerChkGroundXX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerTryFlipGrnd Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerTryFlipGrnd   subroutine                      ; 
                    jsr PlayerGetGroundPos          ; 
                    
                    lda M2Z_PlayerTilePosY          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    cmp #M2_WaSprite_PosYMax        ; 
                    bcc .ChkMaxPosX                 ; lower
                    
.ExitMaxPosY        rts                             ; 

.ChkMaxPosX         lda M2Z_PlayerTilePosXMax       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    bne .ChkMinPosX                 ; 
                    
                    inc M2Z_TargetScreenPosY        ; 
                    jsr PlayerCheckGround           ; 
                    beq .NextPosY                   ; 
                    
                    jsr PlayerFlipGround            ; 
                    jmp .ChkMinPosX                 ; 
                    
.NextPosY           inc M2Z_TargetScreenPosY        ; step down a bit
                    
                    jsr PlayerCheckGround           ; 
                    beq .ChkMinPosX                 ; 
                    
                    jsr PlayerFlipGround            ; 
                    
.ChkMinPosX         lda M2Z_PlayerTilePosXMin       ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda M2Z_PlayerTilePosY          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    jsr PlayerCheckGround           ; 
                    beq .SetNextPosY1               ; 
                    
                    rts                             ; 

.SetNextPosY1       inc M2Z_TargetScreenPosY        ; step down a bit
                    
                    jsr PlayerCheckGround           ; 
                    beq .SetNextPosY2               ; 
                    
                    jmp PlayerFlipGround            ; and return

.SetNextPosY2       inc M2Z_TargetScreenPosY        ; step down a bit
                    
                    jsr PlayerCheckGround           ; 
                    bne PlayerTryFlipGrndX          ; go and flip groud
                    
.ExitNoFlip         rts                             ; 

PlayerTryFlipGrndX  jmp PlayerFlipGround            ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerFlipGround  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerFlipGround    subroutine                      ; 
                    dec M2Z_TargetScreenPosX        ; 
                    jsr PlayerCheckGround           ; 
                    beq .SetFlipPos                 ; 
                    
                    
                    inc M2Z_TargetScreenPosY        ; 
                    jsr PlayerCheckGround           ; 
                    bne .SetFlipPos                 ; 
                    
                    dec M2Z_TargetScreenPosY        ; 
                    jmp PlayerFlipGround            ; 
                    
.SetFlipPos         inc M2Z_TargetScreenPosX        ; 
                    
                    lda M2Z_PlayerTilePosY          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
.SetFixGround       lda #<LevelTileGroundFix        ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>LevelTileGroundFix        ; 
                    sta M2Z_TileGfxPtrHi            ; 
                    jsr GraphicTileCopy             ; 
                    
                    lda #$ff                        ; 
                    sta M2Z_TilePart1Mask           ; 
                    sta M2Z_TilePart2Mask           ; 
                    jsr GraphicTileOut              ; 
                    
.DecCounter         dec M2Z_LvlFieldsToStep         ; 
                    jsr ClearScoreToAdd             ; 
                    
.AddScore           lda #$05                        ; 05
                    sta M2Z_ScoreToAdd1             ; 
                    
PlayerFlipGroundX   jmp IncScore                    ; inc score
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerOnLadderTop Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerOnLadderTop   subroutine                      ; 
                    ldy #M2Z_OnLadderPosTop         ; 
                    sty M2Z_OnLadderPos             ; 
                    
                    ldy #$00                        ; 
                    jmp GetNextLadder               ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerOnLadderBot Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerOnLadderBot   subroutine                      ; 
                    ldy #M2Z_OnLadderPosBot         ; 
                    sty M2Z_OnLadderPos             ; 
                    
GetNextLadder       lda (M2Z_LvlLadderDataPtr),y    ; LevelLadderPosX
                    bne .SavLadderPosX              ; 
                    
.Exit               rts                             ; 

.SavLadderPosX      sta M2Z_LadderPosX              ; LevelLadderPosX
                    
.ChkOnLadderX       iny                             ; 
                    cmp M2_WaSprite_PosX            ; 
                    beq .GetBothPosY                ; 
                    
                    sec                             ; 
                    adc #$00                        ; +1
                    cmp M2_WaSprite_PosX            ; 
                    beq .GetBothPosY                ; 
                    
                    clc                             ; 
                    adc #$fe                        ; -2
                    cmp M2_WaSprite_PosX            ; 
                    beq .GetBothPosY                ; 
                    
.SetNextLadderBlock iny                             ; 
                    iny                             ; 
                    jmp GetNextLadder               ; 
                    
.GetBothPosY        lda (M2Z_LvlLadderDataPtr),y    ; LevelLadderPosY Top
                    sta M2Z_LadderPosYTop           ; 
                    iny                             ; 
                    lda (M2Z_LvlLadderDataPtr),y    ; LevelLadderPosY Bottom
                    sta M2Z_LadderPosYBot           ; 
                    
                    iny                             ; 
                    ldx M2Z_OnLadderPos             ; 
                    beq .ChkPlayerPosY              ; 
                    
                    lda M2Z_LadderPosYTop           ; 
                    
.ChkPlayerPosY      sec                             ; 
                    sbc M2_WaSprite_PosY            ; 
                    bpl .ChkEnd                     ; 
                    
                    eor #$ff                        ; 
                    
.ChkEnd             and #$fc                        ; ######..
                    bne GetNextLadder               ; 
                    
                    lda M2Z_LadderPosX              ; LevelLadderPosX
                    sta M2_WaSprite_PosX            ; 
                    
PlayerOnLadderX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerGetGroundPos Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerGetGroundPos  subroutine                      ; 
                    ldx #$0a                        ; max posX
                    ldy #$07                        ; min posX
                    
                    lda M2_WaSprite_NoPlayer        ; 
                    cmp #NoSprPlrMovLe03            ; stand still le/ri
                    bne .SetPosition                ; 
                    
                    dex                             ; adjust
                    iny                             ; 
                    
.SetPosition        stx M2Z_PlayerTilePosXMax       ; 
                    sty M2Z_PlayerTilePosXMin       ; 
                    
                    sec                             ; 
                    lda M2_WaSprite_PosX            ; 
                    sbc M2Z_PlayerTilePosXMax       ; 
                    sta M2Z_PlayerTilePosXMax       ; 
                    
                    lda M2_WaSprite_PosX            ; 
                    sbc M2Z_PlayerTilePosXMin       ; 
                    sta M2Z_PlayerTilePosXMin       ; 
                    
                    lda M2_WaSprite_PosY            ; 
                    sbc #$1d                        ; 
                    sta M2Z_PlayerTilePosY          ; 
                    
PlayerGetGroundPosX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__Init           Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__Init             subroutine                      ; 
                    lda #$0f                        ; ....####
                    sta M2_Wa_SprColl04             ; 
                    sta M2_Wa_SprColl07             ; 
                    
                    lda #$00                        ; 
                    sta M2Z_SinkSprtDatLen          ; 
                    sta M2Z_PlayerDeadSink          ; 
                    sta M2_Wa_SpriteSinkPosY        ; 
                    sta M2Z_PlayerDeadWobble        ; 
                    sta M2Z_EnemySmileTime          ; 
                    sta M2Z_XmitState               ; 
                    
                    ldx #$1b                        ; 
.ClrSID             sta FRELO1,x                    ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
.ClrSoundControls   sta M2Z_SoundCtrlArea,x         ; 
                    dex                             ; 
                    bpl .ClrSID                     ; 
                    
                    lda #$0f                        ; 
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    lda #$0a                        ; 
                    sta ATDCY3                      ; SID($D413) Oscillator 3 Attack/Decay
                    lda #$a0                        ; 
                    sta SUREL3                      ; SID($D414) Oscillator 3 Sustain/Release
                    
                    lda #M2Z_NextLevelCheatTime     ; 
                    sta M2Z_NextLevel               ; 
                    
.LevelStructure     lda M2Z_LvlStructurePtrLo       ; 
                    sta M2Z_ItemTabPtrLo            ; 
                    lda M2Z_LvlStructurePtrHi       ; 
                    sta M2Z_ItemTabPtrHi            ; 
                    
                    jsr DisplayLevel                ; 
                    
                    ldy #$7f                        ; length
.CpyWAItem          lda (M2Z_LvlItemDataPtr),y      ; filled from level header
                    sta M2_WaItem,y                 ; store actual work area data
                    dey                             ; 
                    bpl .CpyWAItem                  ; 
                    
                    ldy #$20                        ; 
.GetWAItem          lda M2_WaItem,y                 ; 
                    beq .SetWAItem                  ; end of data found
                    
                    tya                             ; 
                    clc                             ; 
                    adc #M2_WaItemLen               ; length of eacj item work area block
                    tay                             ; 
                    jmp .GetWAItem                  ; 
                    
.SetWAItem          sta M2_WaItem,y                 ; clear work area after end of actual data
                    iny                             ; 
                    bpl .SetWAItem                  ; 
                    
                    lda #<M2_WaItem                 ; 
                    sta M2Z_WaItemPtrLo             ; 
                    lda #>M2_WaItem                 ; 
                    sta M2Z_WaItemPtrHi             ; 
                    
.DisplayLevelItems  ldy #M2_WaItem_PaintPosX        ; 
                    lda (M2Z_WaItemPtr),y           ; 
                    beq .ClrWaSpriteI               ; end of data
                    
                    sta M2Z_TargetScreenPosX        ; 
                    
                    iny                             ; M2_WaItem_PosY
                    lda (M2Z_WaItemPtr),y           ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    iny                             ; M2_WaItem_No
                    lda (M2Z_WaItemPtr),y           ; 
                    and #$f8                        ; #####...
                    
                    ldy #$00                        ; 
                    sty M2Z_TileGfxPtrHi            ; 
                    asl a                           ; 
                    rol M2Z_TileGfxPtrHi            ; ptr *2
                    asl a                           ; 
                    rol M2Z_TileGfxPtrHi            ; ptr *4 - length of each item gfx
                    adc #<ItemStore                 ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda M2Z_TileGfxPtrHi            ; 
                    adc #>ItemStore                 ; 
                    sta M2Z_TileGfxPtrHi            ; 
                    
                    jsr ItemGraphicOut              ; 
                    
                    clc                             ; point to nex item WA block
                    lda M2Z_WaItemPtrLo             ; 
                    adc #M2_WaItemLen               ; 
                    sta M2Z_WaItemPtrLo             ; 
                    bcc .DisplayLevelItems          ; 
                    inc M2Z_WaItemPtrHi             ; 
                    jmp .DisplayLevelItems          ; 
                    
.ClrWaSpriteI       lda #$00                        ; 
                    tax                             ; 
.ClrWaSprite        sta M2_WaSprite,x               ; 
                    dex                             ; 
                    bne .ClrWaSprite                ; 
                    
                    tay                             ; 
.GetWaSpriteLen     ldx #M2_WaSpriteLen + 1         ; 
                    iny                             ; 
                    lda (M2Z_LvlSpriteDataPtr),y    ; filled from level header
                    beq .CpySpriteData              ; 
                    
                    dey                             ; 
.GetWaSpriteData    lda (M2Z_LvlSpriteDataPtr),y    ; filled from level header
                    sta M2_WaSprite,y               ; 
                    iny                             ; 
                    dex                             ; 
                    bne .GetWaSpriteData            ; 
                    beq .GetWaSpriteLen             ; 
                    
.CpySpriteData      lda #<MinerGfxStore             ; 
                    sta M2Z_FromAreaPtrLo           ; 
                    lda #>MinerGfxStore             ; 
                    sta M2Z_FromAreaPtrHi           ; ($2b/$2c) points to $a080
                    
                    lda #<M2_GameGfxStore           ; 
                    sta M2Z_FillAreaPtrLo           ; 
                    lda #>M2_GameGfxStore           ; 
                    sta M2Z_FillAreaPtrHi           ; ($29/$2a) points to $6400
                    
                    lda #<M2_GameGfxStoreLen        ; 
                    sta M2Z_FillAreaLenBytes        ; 
                    lda #>M2_GameGfxStoreLen        ; 
                    sta M2Z_FillAreaLenPages        ; ($2d/$2e) length = $0c00
                    jsr CopyData                    ; copy sprite data from $a080 to $6400-$6fff
                    
                    lda #$20                        ; blank
                    ldx #$27                        ; length - 40 - 1
.ClrStatusLine      sta M2_ScreenStatusLine,x       ; 
                    dex                             ; 
                    bpl .ClrStatusLine              ; 
                    
                    lda #LT_BLUE                    ; 
                    ldx #$27                        ; length - 40 - 1
.ColorStatusLine    sta COLORAM,x                   ; color status line
                    dex                             ; 
                    bpl .ColorStatusLine            ; 
                    
                    lda M2Z_LvlTimeDiscount         ; 
                    and #$0f                        ; ....#### - isolate ones
                    sta M2Z_TimeOne                 ; 
                    
                    lda M2Z_LvlTimeDiscount         ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; shift tenths to right
                    and #$0f                        ; ....#### - isolate tenths
                    sta M2Z_TimeTen                 ; 
                    
                    lda #M2Z_TimeTicsInit           ; 
                    sta M2Z_TimeTics                ; 
                    
L__InitX            rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveEnemies       Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveEnemies         subroutine                      ; 
                    lda #<M2_WaSprite_01            ; 
                    sta M2Z_SpriteWaPtrLo           ; 
                    lda #>M2_WaSprite_01            ; 
                    sta M2Z_SpriteWaPtrHi           ; 
                    
                    lda #$08                        ; max no of enemies
.SetnextEnemyNo     pha                             ; 
                    
                    ldy #$00                        ; 
                    lda (M2Z_ItemTabPtr),y          ; 
                    lsr a                           ; 
                    bcc .SetNextEnemyWA             ; was $00 - dead pr inactive
                    
                    jsr EnemyMoveOne                ; 
                    
.SetNextEnemyWA     lda #M2_WaSpriteLen + 1         ; 
                    jsr IncItemTabPtr               ; 
                    
                    pla                             ; 
                    sec                             ; 
                    sbc #$01                        ; 
                    bne .SetnextEnemyNo             ; 
                    
MoveEnemiesX        rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__TimeHandler    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__TimeHandler      subroutine                      ; 
                    jsr PlaySoundHandler            ; 
                    jsr ScoreTimeHandler            ; 
                    
                    dec M2Z_TimeTics                ; 
                    bne .ChkEnemySmileTime          ; 
                    
                    lda #M2Z_TimeTicsInit           ; 
                    sta M2Z_TimeTics                ; 
                    
                    lda M2Z_TimeOne                 ; 
                    ora M2Z_TimeTen                 ; 
                    beq .ChkEnemySmileTime          ; 
                    
                    dec M2Z_TimeOne                 ; 
                    bpl .ChkEnemySmileTime          ; 
                    
                    lda #$09                        ; 
                    sta M2Z_TimeOne                 ; 
                    
                    dec M2Z_TimeTen                 ; 
                    bpl .ChkEnemySmileTime          ; 
                    
                    lda #$09                        ; 
                    sta M2Z_TimeTen                 ; 
                    
.ChkEnemySmileTime  lda M2Z_EnemySmileTime          ; 
                    beq .ChkTime10th                ; 
                    
                    dec M2Z_EnemySmileTime          ; 
                    
.ChkTime10th        lda M2Z_TimeTen                 ; 
                    bne L__TimeHandlerX             ; 
                    
                    lda M2Z_TimeOne                 ; 
                    cmp #$07                        ; 
                    bcs L__TimeHandlerX             ; greater/equal
                    
                    lda M2Z_TimeTics                ; 
                    and #$0f                        ; ....####
                    bne L__TimeHandlerX             ; 
                    
                    lda #M2Z_SoundTimeLowOn         ; 
                    sta M2Z_SoundTimeLow            ; 
                    
L__TimeHandlerX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__CheckSolved    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__CheckSolved      subroutine                      ; 
                    lda M2Z_LvlFieldsToStep         ; 
                    beq AddTimeToScore              ; all fields handled
                    
.ChkLevelTime       bit M2Z_TimeTics                ; 
                    bvs .ChkPlayerSunken            ; bit6=1
                    
                    lda M2Z_TimeOne                 ; 
                    ora M2Z_TimeTen                 ; 
                    bne .ChkPlayerSunken            ; 
                    
.SetDeathTimeUp     lda #M2Z_PlayerDeadWobbleYes    ; 
                    sta M2Z_PlayerDeadWobble        ; 
                    
.ChkPlayerSunken    lda M2Z_SinkSprtDatLen          ; 
                    cmp #M2Z_SinkSprtDatLenHat      ; 
                    beq .SetExitWithDeath           ; leave hat visible
                    
                    ldx #M2Z_NextLevelCheatTime     ; preset
                    
                    lda M2Z_JoystickFireRev         ; reversed
                    bne .SetExitContinue            ; 
                    
.NextLevelCheat     dec M2Z_NextLevel               ; keep fire pressed for timer value
                    beq SetExitSolved               ; 
                    
                    ldx M2Z_NextLevel               ; 
                    
.SetExitContinue    stx M2Z_NextLevel               ; 
                    clc                             ; level unsolved and alive
                    rts                             ; 
                    
SetExitSolved       lda #M2Z_NextLevelYes           ; alive
                    sta M2Z_NextLevel               ; 
                    sec                             ; level solved
                    rts                             ; 
                    
.SetExitWithDeath   lda #M2Z_NextLevelNo            ; dead
                    sta M2Z_NextLevel               ; 
                    sec                             ; level unsolved but dead
L__CheckSolvedX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; AddTimeToScore    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
AddTimeToScore      subroutine
                    lda M2Z_TimeOne                 ; 
                    ora M2Z_TimeTen                 ; 
.Exit               beq SetExitSolved               ; whole time handled
                    
.DecRemainingTime   dec M2Z_TimeOne                 ; 
                    bpl .ClearScoreToAdd            ; 
                    lda #$09                        ; 
                    sta M2Z_TimeOne                 ; 
                    dec M2Z_TimeTen                 ; 
                    
.ClearScoreToAdd    jsr ClearScoreToAdd             ; 
                    
                    lda #$01                        ; 100
                    sta M2Z_ScoreToAdd100           ; 
                    jsr IncScore                    ; inc score
                    jsr L__ShowStatusRow            ; 
                    jsr InitSid                     ; 
                    
                    ldx #$0a                        ; 
.GetScoreBeep       lda TabSndScoreBeep,x           ; 
                    sta FRELO3,x                    ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .GetScoreBeep               ; 
                    
.WaitBeep           lda M2Z_WaitIRQs                ; 
                    eor #$05                        ; 
                    bne .WaitBeep                   ; 
                    
                    sta M2Z_WaitIRQs                ; $00
.BeepOff            sta VCREG3                      ; SID($D412) Oscillator 3 Control
                    
.WaitBeepOff        lda M2Z_WaitIRQs                ; 
                    eor #$04                        ; 
                    bne .WaitBeepOff                ; 
                    
                    sta M2Z_WaitIRQs                ; $00
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    
AddTimeToScoreX     jmp AddTimeToScore              ; next remaining time
; -------------------------------------------------------------------------------------------------------------- ;
; InitSid           Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitSid             subroutine                      ; 
                    ldx #$17                        ; 
                    lda #$00                        ; 
.Reset              sta FRELO1,x                    ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .Reset                      ; 
                    
InitSidX            rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlaySoundHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlaySoundHandler    subroutine                      ; 
                    ldx #$00                        ; $00
.SoundPlayerStep    jsr PlaySoundSelect             ; M2Z_SoundPlayerStep
                    inx                             ; $01
.SoundTimeLow       jsr PlaySoundSelect             ; M2Z_SoundTimeLow
                    beq .PlayerFlickerOff           ; 
                    
.PlayerFlickerOn    lda #RED                        ; flicker player sprite during beeps
                    bne .SetFlickeColor             ; always
.PlayerFlickerOff   lda #ORANGE                     ; flicker player sprite during beeps
.SetFlickeColor     sta SPMC0                       ; VIC($D025) Sprite Multicolor Register 0
                    
                    inx                             ; $02
.SoundPickItem      jsr PlaySoundSelect             ; M2Z_SoundPickItem
                    
.SoundJump          inx                             ; $03
                    lda M2_WaSprite_MoveTime        ; get sound start value
                    ldy M2_WaSprite_MoveType        ; 
                    dey                             ; 
                    beq .SetJumpUp                  ; M2_WaSprite_MoveJumpUp
                    
                    dey                             ; 
                    beq .SetFallDownHeight          ; M2_WaSprite_MoveFallDo
                    
                    dey                             ; 
                    dey                             ; 
                    beq .SetSlideDown               ; M2_WaSprite_MoveSlide
                    
.ResetAction        lda #$00                        ; 
                    sta M2Z_SoundPlayerAction,x     ; 
                    sta M2Z_SndActionCtrl,x         ; 
                    beq .SoundXmit                  ; always
                    
.SetJumpUp          eor #$ff                        ; M2_WaSprite_MoveTime
                    adc #$32                        ; 
                    bne .SavActionFreq              ; 
                    
.SetFallDownHeight  ldy #$32                        ; and handle the very deep fall tunes
                    
                    cmp #$06                        ; 
                    bcc .SetFallDown                ; lower ac
                    sbc #$06                        ; 
                    
                    lsr a                           ; 
                    ldy #$2c                        ; action fequnency
                    
                    cmp #$08                        ; 
                    bcc .SetFallDown                ; lower ac
                    sbc #$08                        ; 
                    
                    lsr a                           ; 
                    ldy #$24                        ; action fequnency
                    
                    cmp #$10                        ; 
                    bcc .SetFallDown                ; lower ac
                    sbc #$10                        ; 
                    
                    lsr a                           ; 
                    ldy #$14                        ; action fequnency
                    
.SetFallDown        eor #$ff                        ; 
                    sta M2Z_19                      ; 
                    tya                             ; 
                    clc                             ; 
                    adc M2Z_19                      ; 
                    jmp .SavActionFreq              ; 
                    
.SetSlideDown       lsr a                           ; 
                    adc #$14                        ; 
                    cmp #$46                        ; 
                    bcc .SavActionFreq              ; lower
                    
                    lda #$46                        ; action fequnency
                    
.SavActionFreq      tay                             ;  
                    jsr GetSoundActionTab           ; 
                    
.GetActionFreq      tya                             ; 
                    sta M2Z_SndActionFreqs,x        ; 
                    
.SoundXmit          inx                             ; $04
                    jsr PlaySoundSelect             ; M2Z_SoundXmit
                    beq .GoSelectActSound           ; 
                    
.RandomizeXmit      tya                             ; 
                    lsr a                           ; 
                    and #$07                        ; 
                    tay                             ; 
                    lda TabSndXmitRnd,y             ; 
                    sta M2Z_SndActionFreqs,x        ; 
                    
.GoSelectActSound   jsr SelectActionSound           ; 
                    
                    lda M2Z_PlayerDeadWobble        ; 
                    bne .ChkKillSoundWobble         ; M2Z_PlayerDeadWobbleYes
                    
                    sta M2Z_SoundKillWobble         ; M2Z_PlayerDeadWobbleNo
                    beq .ChkSoundKillTime           ; always
                    
.ChkKillSoundWobble lda M2Z_SoundKillWobble         ; 
                    bne .NextWobbleSound            ; 
                    
                    dec M2Z_SoundKillWobble         ; $ff
                    
                    ldx #$0d                        ; 
.CpyKilledSoundWob  lda TabSndKilledWobble,x        ; 
                    sta FRELO1,x                    ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .CpyKilledSoundWob          ; 
                    
                    lda #$7d                        ; 
                    sta CUTHI                       ; SID($D416) Filter Cutoff Frequency (high byte)
                    lda #$f3                        ; 
                    sta RESON                       ; SID($D417) Filter Resonance Ctrl
                    lda #$3f                        ; 
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    
.ExitKill           rts                             ; 
                    
.NextWobbleSound    dec M2Z_SoundKillWobble         ; 
                    
                    ldx #$81                        ; bit7=1 random noise   bit0=1 Start attack/decay/sustain
                    stx VCREG1                      ; SID($D404) Oscillator 1 Control
                    
                    lsr a                           ; 
                    lsr a                           ; 
                    bcc .SetVoc2Ctrl                ; 
                    
                    ldx #$00                        ; 
                    
.SetVoc2Ctrl        stx VCREG2                      ; SID($D40B) Oscillator 2 Control
                    
                    rts                             ; 
                    
.ChkSoundKillTime   lda M2Z_SoundKill               ; 
                    beq .ChkLifterCannon            ; 
                    bpl .DecSoundKillTime           ; 
                    
                    lda #M2Z_SoundKillTimeInit      ; 
                    sta M2Z_SoundKill               ; 
                    
                    ldx #$0d                        ; 
.CpyKilledSoundFal  lda TabSndKilledFall,x          ; 
                    sta FRELO1,x                    ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .CpyKilledSoundFal          ; 
                    
                    lda #$7d                        ; 
                    sta CUTHI                       ; SID($D416) Filter Cutoff Frequency (high byte)
                    lda #$f3                        ; 
                    sta RESON                       ; SID($D417) Filter Resonance Ctrl
                    lda #$3f                        ; 
                    sta SIGVOL                      ; SID($D418) Volume/Filter Select
                    
                    rts                             ; 
                    
.DecSoundKillTime   dec M2Z_SoundKill               ; time
                    
                    lda M2Z_SoundKill               ; preset
                    beq .SetVoc1Voc2Ctrl            ; $00
                    
                    lda #$81                        ; bit7=1 random noise   bit0=1 Start attack/decay/sustain
                    
.SetVoc1Voc2Ctrl    sta VCREG1                      ; SID($D404) Oscillator 1 Control
                    sta VCREG2                      ; SID($D40B) Oscillator 2 Control
                    
                    rts                             ; 
                    
.ChkLifterCannon    lda M2Z_SoundLifterCannon       ; 
                    beq .ClrVoc1Voc2                      ; 
                    
                    ldx #$0d                        ; 
.CpyLiftCannSound   lda TabSndLifterCannon,x        ; 
                    sta FRELO1,x                    ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                    dex                             ; 
                    bpl .CpyLiftCannSound           ; 
                    
                    inx                             ; $00
                    
                    stx RESON                       ; SID($D417) Filter Resonance Ctrl
                    lda M2Z_TimeTics                ; 
                    ldx #$02                        ; 
                    and #$04                        ; 
                    beq .SetLiftCannFreq            ; 
                    
                    ldx #$05                        ; 
                    
.SetLiftCannFreq    stx FREHI1                      ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                    
                    rts                             ; 
                    
.ClrVoc1Voc2        sta VCREG1                      ; SID($D404) Oscillator 1 Control
                    sta VCREG2                      ; SID($D40B) Oscillator 2 Control
                    
PlaySoundHandlerX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SelectActionSound Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SelectActionSound   subroutine                      ; 
                    ldx #$01                        ; 
                    lda M2Z_SoundPlayerAction,x     ; M2Z_SoundTimeLow
                    bne .PutActSndToSid             ; 
                    
                    inx                             ; $02
                    lda M2Z_SoundPlayerAction,x     ; M2Z_SoundPickItem
                    bne .PutActSndToSid             ; 
                    
                    ldx #$00                        ; 
                    lda M2Z_SoundPlayerAction,x     ; M2Z_SoundPlayerStep
                    bne .PutActSndToSid             ; 
                    
                    ldx #$03                        ; 
                    lda M2Z_SoundPlayerAction,x     ; M2Z_SoundXmit
                    bne .PutActSndToSid             ; 
                    
                    inx                             ; $04
                    lda M2Z_SoundPlayerAction,x     ; M2Z_SndActionTimes
                    bne .PutActSndToSid             ; 
                    
                    sta VCREG3                      ; SID($D412) Oscillator 3 Control
                    
                    rts                             ; 
                    
.PutActSndToSid     lda M2Z_SndActionFreqs,x        ; 
                    sta FREHI3                      ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                    lda M2Z_SndActionWaveWi,x       ; 
                    sta PWHI3                       ; SID($D411) Oscillator 3 Pulse Waveform Width (high nybble)
                    lda M2Z_SndActionCtrl,x         ; 
                    sta VCREG3                      ; SID($D412) Oscillator 3 Control
                    
SelectActionSoundX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlaySoundSelect   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlaySoundSelect     subroutine                      ; 
                    lda M2Z_SoundPlayerAction,x     ; 
                    beq PlaySoundSelectX            ; 
                    bpl .PlayActionSound            ; M2Z_SoundPlayerPlays
                    
                    jmp GetSoundActionTab           ; and return
                    
.PlayActionSound    jsr DecSoundActionTime          ; 
                    
PlaySoundSelectX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetSoundActionTab Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetSoundActionTab   subroutine                      ; 
                    lda #M2Z_SoundPlayerPlays       ; 
                    sta M2Z_SoundPlayerAction,x     ; 
                    
                    lda TabSndActionTime,x          ; 
                    sta M2Z_SndActionTimes,x        ; duration time for action sounds
                    lda TabSndActionFreqHi,x        ; 
                    sta M2Z_SndActionFreqs,x        ; frequency
                    lda TabSndActionWaveWi,x        ; 
                    sta M2Z_SndActionWaveWi,x       ; wave worm width
                    lda TabSndActionCtrl,x          ; 
                    sta M2Z_SndActionCtrl,x         ; wave form / attack-release
                    
GetSoundActionTabX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DecSoundActionTime Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DecSoundActionTime  subroutine                      ; 
                    lda M2Z_SndActionTimes,x        ; 
                    tay                             ; 
                    dey                             ; 
                    tya                             ; 
                    sta M2Z_SndActionTimes,x        ; 
                    bne DecSoundActionTimeX         ; 
                    
.SwitchOff          sta M2Z_SoundPlayerAction,x     ; $00
                    sta M2Z_SndActionCtrl,x         ; 
                    
DecSoundActionTimeX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__ItemChecker    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__ItemChecker      subroutine                      ; 
                    lda #<M2_WaItem                 ; 
                    sta M2Z_ItemTabPtrLo            ; 
                    lda #>M2_WaItem                 ; 
                    sta M2Z_ItemTabPtrHi            ; 
                    
.GetNextItem        ldy #$00                        ; 
                    lda (M2Z_ItemTabPtr),y          ; 
                    bne .WasItem                    ; 
                    
.Exit               rts                             ; all items handled

.WasItem            ldy #$03                        ; 
                    lda (M2Z_ItemTabPtr),y          ; 
                    beq .SetNextItemTabPos          ; 
                    
                    iny                             ; 
                    lda M2_WaSprite_PosX            ; 
                    cmp (M2Z_ItemTabPtr),y          ; 
                    bcc .SetNextItemTabPos          ; 
                    
                    iny                             ; 
                    cmp (M2Z_ItemTabPtr),y          ; 
                    bcs .SetNextItemTabPos          ; 
                    
                    iny                             ; 
                    lda M2_WaSprite_PosY            ; 
                    cmp (M2Z_ItemTabPtr),y          ; 
                    bcc .SetNextItemTabPos          ; 
                    
                    iny                             ; 
                    cmp (M2Z_ItemTabPtr),y          ; 
                    bcs .SetNextItemTabPos          ; 
                    
                    jsr ItemHandler                 ; 
                    
.SetNextItemTabPos  lda #$08                        ; 
                    jsr IncItemTabPtr               ; 
                    
L__ItemCheckerX     jmp .GetNextItem                ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ItemHandler       Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ItemHandler         subroutine                      ; 
                    ldy #M2_WaItem_PaintPosX        ; 
                    lda (M2Z_ItemTabPtr),y          ; 
                    sta M2Z_TargetScreenPosX        ; 
                    iny                             ; M2_WaItem_PosY
                    lda (M2Z_ItemTabPtr),y          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    lda #<TileClearStore            ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>TileClearStore            ; 
                    sta M2Z_TileGfxPtrHi            ; 
                    
                    jsr ItemGraphicOut              ; 
                    jsr ClearScoreToAdd             ; 
                    
                    ldy #M2_WaItem_Score            ; 
                    lda (M2Z_ItemTabPtr),y          ; 
                    tax                             ; save score
                    and #$0f                        ; ....#### - isolate right halfbyte
                    sta M2Z_ScoreToAdd100           ; 
                    
                    txa                             ; restore score
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; 
                    lsr a                           ; isolate left halfbyte
                    and #$01                        ; .......# - max 1000
                    sta M2Z_ScoreToAdd1000          ; 
                    jsr IncScore                    ; inc score
                    
                    ldx #M2_WaSpriteLen             ; 
.CpyTabVals         lda TabWaSpriteScoreIni,x       ; 
                    sta M2_WaSprite_Score,x         ; 
                    dex                             ; 
                    bpl .CpyTabVals                 ; 
                    
                    ldy #M2_WaItem_PaintPosX        ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaItem_PaintPosX
                    adc #$0c                        ; 
                    sta M2_WaSprite_ScorePosX       ; 
                    
                    iny                             ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaItem_PosY
                    adc #$32                        ; 
                    sta M2_WaSprite_ScorePosY       ; 
                    
                    iny                             ; 
                    iny                             ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaItem_Score
                    lsr a                           ; .#######
                    lsr a                           ; ..######
                    lsr a                           ; ...#####
                    lsr a                           ; ....####
                    lsr a                           ; .....###
                    clc                             ; 
                    adc #$aa                        ; sprite pointer scores base
                    sta M2_WaSprite_ScoreSpPtr      ; 
                    
                    lda #$00                        ; 
                    sta (M2Z_ItemTabPtr),y          ; clear M2_WaItem_Score
                    
                    dey                             ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaItem_No
                    and #$03                        ; ......##
                    bne .ChkItem_01                 ; 
                    
                    ldx #M2_WaSpriteLen             ; 
.CpyItem_00         lda M2_WaSprite_Score,x         ; 
                    sta M2_WaSprite_09,x            ; 
                    dex                             ; 
                    bpl .CpyItem_00                 ; 
                    
                    jmp .SetExitVals                ; 
                    
.ChkItem_01         cmp #$01                        ; 
                    bne .CpyItem_02I                ; 
                    
                    ldx #M2_WaSpriteLen             ; 
.CpyItem_01         lda M2_WaSprite_Score,x         ; 
                    sta M2_WaSprite_0a,x            ; 
                    dex                             ; 
                    bpl .CpyItem_01                 ; 
                    
                    jmp .SetExitVals                ; 
                    
.CpyItem_02I        ldx #M2_WaSpriteLen             ; 
.CpyItem_02         lda M2_WaSprite_Score,x         ; 
                    sta M2_WaSprite_0b,x            ; 
                    dex                             ; 
                    bpl .CpyItem_02                 ; 
                    
.SetExitVals        lda #M2Z_EnemySmileTimeInit     ; 
                    sta M2Z_EnemySmileTime          ; 
                    
                    lda #M2Z_SoundPickItemOn        ; 
                    sta M2Z_SoundPickItem           ; 
                    
ItemHandlerX        rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__SprtCollHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__SprtCollHandler  subroutine                      ; 
                    lda M2Z_PlayerDeadWobble        ;  
                    bne L__SprtCollHandlerX         ; already dead
                    
                    ldx M2Z_SprSprColl04            ; 
                    sta M2Z_SprSprColl04            ; $00
                    txa                             ; 
                    and M2_Wa_SprColl04             ; 
                    beq .ChkColl07                  ; 
                    
                    tax                             ; 
                    eor #$0f                        ; 
                    and M2_Wa_SprColl04             ; 
                    sta M2_Wa_SprColl04             ; 
                    
                    lda M2Z_EnemySmileTime          ; 
                    beq .ExitDeath                  ; 
                    
                    txa                             ; 
                    ldx #$5e                        ; pointer enemy wa ponter area 07
                    jsr KillEnemyHandler            ; 
                    
.ChkColl07          lda M2Z_SprSprColl07            ; 
                    ldx #$00                        ; 
                    stx M2Z_SprSprColl07            ; 
                    and M2_Wa_SprColl07             ; 
                    beq L__SprtCollHandlerX         ; 
                    
                    tax                             ; save collision flags
                    eor #$0f                        ; 
                    and M2_Wa_SprColl07             ; 
                    sta M2_Wa_SprColl07             ; 
                    
                    lda M2Z_EnemySmileTime          ; 
                    beq .ExitDeath                  ; 
                    
                    txa                             ; restore collision flags
                    ldx #$6e                        ; pointer enemy wa ponter area 04
                    jmp KillEnemyHandler            ; and return
                    
.ExitDeath          lda #M2Z_PlayerDeadWobbleYes    ; 
                    sta M2Z_PlayerDeadWobble        ; 
                    
L__SprtCollHandlerX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; KillEnemyHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
KillEnemyHandler    subroutine                      ; 
                    sta M2Z_SprSprColFlags          ; 
                    stx M2Z_PtrEnemyWaPtr           ; 
                    
                    lda #$04                        ; 
                    sta M2Z_EnemiesToHandle         ; 
                    
.ChkNext            lsr M2Z_SprSprColFlags          ; 
                    bcc .SetNextEnemyWaPtr          ; 
                    
                    ldx M2Z_PtrEnemyWaPtr           ; 
                    lda M2Z_D6510,x                 ; $5e/$6e - enemy wa ptr lo
                    sta M2Z_WaS_EnemyPtrLo          ; 
                    
                    lda M2Z_R6510,x                 ; $5f/$6f - enemy wa ptr hi
                    sta M2Z_WaS_EnemyPtrHi          ; 
                    
                    ldy #$00                        ; 
                    lda (M2Z_WaS_EnemyPtr),y        ; M2_WaSprite_State
                    lsr a                           ; 
                    bcc .SetNextEnemyWaPtr          ; dead already
                    
                    asl a                           ; 
                    sta (M2Z_WaS_EnemyPtr),y        ; M2_WaSprite_State
                    
                    ldy #$03                        ; 
                    lda #PtrSprScore0090            ; 
                    sta (M2Z_WaS_EnemyPtr),y        ; M2_WaSprite_Pointer
                    
                    iny                             ; 
                    lda #LT_BLUE                    ; 
                    sta (M2Z_WaS_EnemyPtr),y        ; M2_WaSprite_Color0
                    
                    ldy #$08                        ; 
                    lda #M2_WaSprite_StepWaitScore  ; score show time
                    sta (M2Z_WaS_EnemyPtr),y        ; M2_WaSprite_StepWait
                    
                    jsr ClearScoreToAdd             ; 
                    
                    lda #$09                        ; 90
                    sta M2Z_ScoreToAdd10            ; 
                    jsr IncScore                    ; inc score
                    
                    lda #M2Z_SoundKillOn            ; 
                    sta M2Z_SoundKill               ; 
                    
.SetNextEnemyWaPtr  inc M2Z_PtrEnemyWaPtr           ; 
                    inc M2Z_PtrEnemyWaPtr           ; 
                    
                    dec M2Z_EnemiesToHandle         ; 
                    bne .ChkNext                    ; 
                    
                    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScoreTimeHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScoreTimeHandler    subroutine                      ; 
                    lda #<(M2_WaSprite + M2_WaSpriteLen + 1) ; $0e
                    sta M2Z_ItemTabPtrLo            ; 
                    lda #>(M2_WaSprite + M2_WaSpriteLen + 1) ; $0d
                    sta M2Z_ItemTabPtrHi            ; 
                    
                    ldx #M2_WaSprite_BlocksMax - 1  ; discount player WA = 1st
                    
.GetNextState       ldy #$00                        ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaSprite_State
                    beq .SetNextEneWaPtr            ; M2_WaSprite_StateDead
                    
                    lsr a                           ; 
                    bcs .SetNextEneWaPtr            ; no score sprite
                    
.GetScoreShowTime   ldy #$08                        ; 
                    lda (M2Z_ItemTabPtr),y          ; M2_WaSprite_StepWait
                    adc #$ff                        ; -1
                    sta (M2Z_ItemTabPtr),y          ; 
                    bne .SetNextEneWaPtr            ; 
                    
.SwitchOffScore     ldy #$00                        ; 
                    tya                             ; M2_WaSprite_StateDead
                    sta (M2Z_ItemTabPtr),y          ; M2_WaSprite_State
                    
.SetNextEneWaPtr    lda #M2_WaSpriteLen + 1         ; 
                    jsr IncItemTabPtr               ; 
                    
                    dex                             ; 
                    bne .GetNextState               ; 
                    
ScoreTimeHandlerX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MovePlayer        Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MovePlayer          subroutine                      ; 
                    jsr CheckJoystick               ; 
                    
                    lda M2Z_PlayerDeadWobble        ; 
                    beq .ChkDeathFlag               ; 
                    
                    sta M2Z_PlayerDeadSink          ; 
                    
.SetDeathColors     lda #PURPLE                     ; 
                    sta M2_WaSprite_Color0          ; 
                    
                    lda #RED                        ; 
                    sta SPMC0                       ; VIC($D025) Sprite Multicolor Register 0
                    
.DeathWobbler       lda M2_WaSprite_KingSize        ; 
                    eor #$ff                        ; 
                    sta M2_WaSprite_KingSize        ; 
                    
                    lda M2_WaSprite_MoveType        ; 
                    cmp #M2_WaSprite_MoveClimb      ; 
                    bne .ChkDeathFlag               ; 
                    
                    lda #M2_WaSprite_MoveFallDo     ; 
                    sta M2_WaSprite_MoveType        ; 
                    sta M2_WaSprite_StepWait        ; 
                    
                    lda #$00                        ; 
                    sta M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    
                    lda #PtrSprPlrJumpLe            ; 
                    sta M2_WaSprite_Pointer         ; 
                    
.ChkDeathFlag       lda M2Z_PlayerDeadSink          ; 
                    beq .ChkXmitBoothX              ; 
                    
.DoNotMove          lda #M2Z_JoystickReset          ; stop moving whilest sinking
                    sta M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    sta M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    sta M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    
.ChkXmitBoothX      lda M2Z_LvlXmitBoothXpos        ; 
                    beq .ChkSink                    ; 
                    
                    jsr DecXmitNonact               ; 
                    
.ChkSink            lda M2Z_SinkSprtDatLen          ; 
                    bne PlayerDeathSinkNext         ; 
                    
                    lda M2Z_PlayerDeadSink          ; 
                    beq .ChkXmitState               ; 
                    
                    lda M2_WaSprite_MoveType        ; 
                    beq PlayerDeathSink             ; 
                    
                    cmp #M2_WaSprite_MoveFly        ; 
                    bne .ChkXmitState               ; 
                    
                    lda M2_WaSprite_PosY            ; 
                    cmp #M2_WaSprite_PosYMaxSink    ; 
                    beq PlayerDeathSink             ; 
                    
.ChkXmitState       lda M2Z_XmitState               ; 
                    bne MovePlayerX                 ; on
                    
                    jsr PlayerMoveTypes             ; 
                    
MovePlayerX         rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerDeathSink   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerDeathSink     lda M2_WaSprite_Pointer         ; 
                    
                    ldx #$00                        ; 
                    stx M2Z_SinkSprtDatPtrLo        ; 
                    
.GetSprtDataPtr     lsr a                           ; 
                    ror M2Z_SinkSprtDatPtrLo        ; 
                    lsr a                           ; 
                    ror M2Z_SinkSprtDatPtrLo        ; 
                    ora #>M2_Screen_HiRes           ; 
                    sta M2Z_SinkSprtDatPtrHi        ; point to actual sprite data
                    
                    lda #$3e                        ; do not sink the hat
                    sta M2Z_SinkSprtDatLen          ; 
                    
PlayerDeathSinkNext inc M2_WaSprite_PosY            ; 
                    inc M2_Wa_SpriteSinkPosY        ; 
                    
                    ldx #$00                        ; 
                    ldy M2Z_SinkSprtDatLen          ; 
                    
                    dey                             ; 
                    lda (M2Z_SinkSprtDatPtr),y      ; 
                    sta M2Z_19                      ; 
                    txa                             ; $00
                    sta (M2Z_SinkSprtDatPtr),y      ; 
                    sta M2_SavePlayerSprite,y       ; clear 1st sprite row byte from bottom up
                    
                    dey                             ; 
                    lda (M2Z_SinkSprtDatPtr),y      ; 
                    sta M2Z_1a                      ; 
                    txa                             ; $00
                    sta (M2Z_SinkSprtDatPtr),y      ; 
                    sta M2_SavePlayerSprite,y       ; clear 2nd sprite row byte from bottom up
                    
                    dey                             ; 3rd sprite row byte always $00
                    sty M2Z_SinkSprtDatLen          ; 
                    
                    dey                             ; 
                    lda (M2Z_SinkSprtDatPtr),y      ; 
                    ora M2Z_19                      ; 
                    sta (M2Z_SinkSprtDatPtr),y      ; 
                    
                    dey                             ; 
                    lda (M2Z_SinkSprtDatPtr),y      ; 
                    ora M2Z_1a                      ; 
                    sta (M2Z_SinkSprtDatPtr),y      ; 
                    
PlayerDeathSinkX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
;A_954e              dc.b $ff ; honz
;                    dc.b $ff ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__InitXmitDigits Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__InitXmitDigits   subroutine                      ; if used always 4 stacked one directly over the other
                    lda M2Z_LvlXmitDigitXpos        ; xpos
                    sta M2Z_TargetScreenPosX        ; 
                    lda #$1b                        ; ypos
                    sta M2Z_TargetScreenPosY        ; 
                    lda #<(M2_CharSet + $0540)      ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>(M2_CharSet + $0540)      ; big digit 4
                    sta M2Z_TileGfxPtrHi            ; 
                    jsr GraphicTextLtRed            ; 
                    
                    lda M2Z_LvlXmitDigitXpos        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda #$49                        ; ypos
                    sta M2Z_TargetScreenPosY        ; 
                    lda #<(M2_CharSet + $0530)      ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>(M2_CharSet + $0530)      ; big digit 3
                    sta M2Z_TileGfxPtrHi            ; 
                    jsr GraphicTextLtRed            ; 
                    
                    lda M2Z_LvlXmitDigitXpos        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda #$77                        ; ypos
                    sta M2Z_TargetScreenPosY        ; 
                    lda #<(M2_CharSet + $0520)      ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>(M2_CharSet + $0520)      ; big digit 2
                    sta M2Z_TileGfxPtrHi            ; 
                    jsr GraphicTextLtRed            ; 
                    
                    lda M2Z_LvlXmitDigitXpos        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    lda #$a5                        ; ypos
                    sta M2Z_TargetScreenPosY        ; 
                    lda #<(M2_CharSet + $0510)      ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda #>(M2_CharSet + $0510)      ; big digit 1
                    sta M2Z_TileGfxPtrHi            ; 
                    jsr GraphicTextLtRed            ; 
                    
                    lda #$00                        ; 
                    sta M2Z_XmitNonAct              ; 
                    sta M2Z_XmitLevel               ; 
                    lda #HR_GreenLtRed              ; 
                    
L__InitXmitDigitsX  jmp ColorXmitBooth              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTextLtRed  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTextLtRed    subroutine                      ; 
                    ldx #$00                        ; 
                    ldy #HR_LtRedLtRed              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTextOut    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTextOut      subroutine                      ; 
                    stx M2Z_19                      ; $00 or $ff
                    sty M2Z_1a                      ; color pointer
                    
                    lda M2Z_TargetScreenPosY        ; 
                    pha                             ; 
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTextAdapt            ; 
                    jsr GraphicTileOut              ; 
                    
                    pla                             ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    lda M2Z_TargetScreenPosX        ; 
                    clc                             ; 
                    adc #$04                        ; 
                    sta M2Z_TargetScreenPosX        ; 
                    
                    lda M2Z_TileGfxPtrLo            ; 
                    adc #$08                        ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    jsr GraphicTileCopy             ; 
                    jsr GraphicTextAdapt            ; 
                    
GraphicTextOutX     jmp GraphicTileOut              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTextAdapt  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTextAdapt    subroutine                      ; 
                    ldx #$07                        ; 
.NextTextTilePart   lda M2Z_TilePart1,x             ; 
                    eor M2Z_19                      ; $00 or $ff
                    and M2Z_1a                      ; color pointer
                    sta M2Z_TilePart1,x             ; 
                    
                    lda M2Z_TilePart2,x             ; 
                    eor M2Z_19                      ; $00 or $ff
                    and M2Z_1a                      ; color pointer
                    sta M2Z_TilePart2,x             ; 
                    dex                             ; 
                    bpl .NextTextTilePart           ; 
                    
GraphicTextAdaptX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ColorXmitLevelNo  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ColorXmitLevelNo    subroutine                      ; 
                    and #$0f                        ; isolate color
                    sta M2Z_19                      ; 
                    
.Level4             dex                             ; level number
                    bne .Level3                     ; 
                    
                    ldx #$20                        ; 
                    ldy #$63                        ; 
                    jmp .SetDigitLine01             ; 
                    
.Level3             dex                             ; level number
                    bne .Level2                     ; 
                    
                    ldx #$30                        ; 
                    ldy #$62                        ; 
                    jmp .SetDigitLine01             ; 
                    
.Level2             dex                             ; level number
                    bne .Level1                     ; 
                    
                    ldx #$68                        ; 
                    ldy #$61                        ; 
                    jmp .SetDigitLine01             ; 
                    
.Level1             ldx #$78                        ; 
                    ldy #$60                        ; 
                    
.SetDigitLine01     sty M2Z_1b                      ; 
                    clc                             ; 
                    txa                             ; 
                    adc M2Z_LvlXmitBoothXpos        ; 
                    sta M2Z_1a                      ; 
                    bcc .SetDigitLine02             ; 
                    inc M2Z_1b                      ; 
                    
.SetDigitLine02     lda M2Z_1a                      ; 
                    clc                             ; 
                    adc #$28                        ; 
                    sta M2Z_1c                      ; 
                    lda M2Z_1b                      ; 
                    adc #$00                        ; 
                    sta M2Z_1d                      ; 
                    
                    ldy #$02                        ; 
.SetDigitColor      lda (M2Z_1a),y                  ; get old xmitter level color
                    and #$f0                        ; isolate color
                    ora M2Z_19                      ; set new color
                    sta (M2Z_1a),y                  ; store at digit line 1
                    sta (M2Z_1c),y                  ; store at digit line 2
                    dey                             ; 
                    bpl .SetDigitColor              ; 
                    
ColorXmitLevelNoX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DecXmitNonact     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DecXmitNonact       subroutine                      ; 
                    lda M2Z_XmitNonAct              ; 
                    beq CheckPlayerPos              ; xmit booth already active
                    
                    dec M2Z_XmitNonAct              ; 
                    beq ResetXmitNonact             ; 
                    
DecXmitNonactX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
ResetXmitNonact     lda #HR_GreenLtRed              ; 
                    jsr ColorXmitBooth              ; 
                    
CheckPlayerPos      lda M2_WaSprite_MoveType        ; 
                    beq CheckPlayerInXmit           ; 
                    
ResetXmitNonactX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
CheckPlayerInXmit   lda M2_WaSprite_PosX            ; 
.ChkInsideLeft      cmp M2Z_LvlInXmitMinXpos        ; 
                    bcc .NotFullyInBooth            ; lower
                    
.ChkInsideRight     cmp M2Z_LvlInXmitMaxXpos        ; lower
                    bcc PlayerInXmitBooth           ; 
                    
.NotFullyInBooth    lda #M2Z_XmitStateOff           ; 
                    sta M2Z_XmitState               ; 
                    sta M2Z_XmitLevel               ; 
                    
CheckPlayerInXmitX  jmp MarkXmitLevelNo             ; 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerInXmitBooth   lda M2Z_XmitState               ; 
                    bne SwitchPlayerPos             ; on
                    
                    lda M2Z_XmitLevel               ; 
                    bne .ChkJoyFire                 ; 
                    
                    jsr IncMarkXmitLevelNo          ; 
                    
.ChkJoyFire         lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    beq DecXmitLevelTime            ; 
                    
                    lda M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    bne DecXmitLevelTime            ; still moving
                    
PlayerInXmitBoothX  jmp PrepareXmit                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
DecXmitLevelTime    dec M2Z_XmitLevelTime           ; 
                    bne DecXmitLevelTimeX           ; 
                    
                    jsr IncMarkXmitLevelNo          ; 
                    
DecXmitLevelTimeX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
SwitchPlayerPos     lda #M2Z_JoystickReset          ; 
                    sta M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    sta M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    sta M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    
                    ldy #M2Z_SaveSpriteDataLen - 1  ; 
.RestoreSpriteData  lda M2_SavePlayerSprite,y       ; 
                    sta (M2Z_SaveSpritePtr),y       ; 
                    dey                             ; 
                    bpl .RestoreSpriteData          ; 
                    
                    dec M2Z_XmitLevelTime           ; 
                    bne FlickerPlayer               ; 
                    
                    lda #M2Z_XmitStateOff           ; 
                    sta M2_WaSprite_09              ; 
                    sta M2Z_XmitState               ; 
                    sta M2Z_XmitLevel               ; 
                    
                    lda #M2Z_XmitNonActTime         ; 
                    sta M2Z_XmitNonAct              ; 
                    lda #HR_LtRedLtRed              ; 
                    
SwitchPlayerPosX    jmp ColorXmitBooth              ; 
; -------------------------------------------------------------------------------------------------------------- ;
FlickerPlayer       lda #>M2_ProgramStart           ; 
                    sta M2Z_RndPatternPtrHi         ; 
                    inc M2Z_RndPatternPtrLo         ; 
                    
                    ldy #M2Z_SaveSpriteDataLen - 1  ; 
.GetPlayerData      lda (M2Z_SaveSpritePtr),y       ; 
                    and (M2Z_RndPatternPtr),y       ; 
                    sta (M2Z_SaveSpritePtr),y       ; 
                    dey                             ; 
                    bpl .GetPlayerData              ; 
                    
FlickerPlayerX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MarkXmitLevelNo   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MarkXmitLevelNo     subroutine                      ; 
                    ldx #$01                        ; 
                    lda #LT_RED                     ; 
                    cpx M2Z_XmitLevel               ; 
                    bne .Level1                     ; 
                    
                    lda #WHITE                      ; 
.Level1             jsr ColorXmitLevelNo            ; 
                    
                    ldx #$02                        ; 
                    lda #LT_RED                     ; 
                    cpx M2Z_XmitLevel               ; 
                    bne .Level2                     ; 
                    
                    lda #WHITE                      ; 
.Level2             jsr ColorXmitLevelNo            ; 
                    
                    ldx #$03                        ; 
                    lda #LT_RED                     ; 
                    cpx M2Z_XmitLevel               ; 
                    bne .Level3                     ; 
                    
                    lda #WHITE                      ; 
.Level3             jsr ColorXmitLevelNo            ; 
                    
                    ldx #$04                        ; 
                    lda #LT_RED                     ; 
                    cpx M2Z_XmitLevel               ; 
                    bne MarkXmitLevelNoX            ; 
                    
                    lda #WHITE                      ; 
MarkXmitLevelNoX    jmp ColorXmitLevelNo            ; level4
; -------------------------------------------------------------------------------------------------------------- ;
; IncMarkXmitLevelNo Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IncMarkXmitLevelNo  subroutine                      ; 
                    ldx #$01                        ; 
                    lda M2_WaSprite_PosY            ; 
                    cmp #M2_WaSprite_XmitPosY1      ; 
                    beq .IncXmitLevel               ; 
                    
                    inx                             ; $02
                    cmp #M2_WaSprite_XmitPosY2      ; 
                    beq .IncXmitLevel               ; 
                    
                    inx                             ; $03
                    cmp #M2_WaSprite_XmitPosY3      ; 
                    beq .IncXmitLevel               ; 
                    
                    inx                             ; $04
.IncXmitLevel       inc M2Z_XmitLevel               ; 
                    
                    lda M2Z_XmitLevel               ; 
                    cmp #M2Z_XmitLevelMax + 1       ; 
                    bcc .ChkXmitLevel               ; lower
                    
                    lda #M2Z_XmitLevelMin           ; restart
                    sta M2Z_XmitLevel               ; 
                    
.ChkXmitLevel       cpx M2Z_XmitLevel               ; 
                    beq .IncXmitLevel               ; do not set the actual xmit booth for xmits
                    
                    jsr MarkXmitLevelNo             ; 
                    
                    lda #M2Z_XmitLevelTimeWait      ; 
                    sta M2Z_XmitLevelTime           ; 
                    
IncMarkXmitLevelNoX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PrepareXmit       Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PrepareXmit         subroutine                      ; 
                    lda #M2Z_SoundXmitOn            ; 
                    sta M2Z_SoundXmit               ; 
                    
                    lda #M2Z_XmitLevelTimeXmit      ; 
                    sta M2Z_XmitLevelTime           ; 
                    lda #M2Z_XmitStateOn            ; 
                    sta M2Z_XmitState               ; 
                    
                    ldx #M2_WaSpriteLen             ; 
.CpyPlayerWA        lda M2_WaSprite,x               ; 
                    sta M2_WaSprite_09,x            ; 
                    dex                             ; 
                    bpl .CpyPlayerWA                ; 
                    
                    ldx M2Z_XmitLevel               ; 
                    lda #M2_WaSprite_XmitPosY1      ; 
                    dex                             ; 
                    beq .SetPlayerSpritePos         ; 
                    
                    lda #M2_WaSprite_XmitPosY2      ; 
                    dex                             ; 
                    beq .SetPlayerSpritePos         ; 
                    
                    lda #M2_WaSprite_XmitPosY3      ; 
                    dex                             ; 
                    beq .SetPlayerSpritePos         ; 
                    
                    lda #M2_WaSprite_XmitPosY4      ; 
                    
.SetPlayerSpritePos sta M2_WaSprite_PosY            ; 

                    lda M2_WaSprite_Pointer         ; 
                    sta M2Z_SaveSpritePtrHi         ; 
                    
                    lda #M2Z_JoystickReset          ; 
                    sta M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    sta M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    sta M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    
                    lsr M2Z_SaveSpritePtrHi         ; 
                    ror a                           ; *2
                    lsr M2Z_SaveSpritePtrHi         ; 
                    ror a                           ; *4
                    adc #<M2_Screen_HiRes           ; 
                    sta M2Z_SaveSpritePtrLo         ; 
                    
                    lda M2Z_SaveSpritePtrHi         ; 
                    adc #>M2_Screen_HiRes           ; 
                    sta M2Z_SaveSpritePtrHi         ; 
                    
                    ldy #M2Z_SaveSpriteDataLen - 1  ; 
.SaveSpriteData     lda (M2Z_SaveSpritePtr),y       ; 
                    sta M2_SavePlayerSprite,y       ; 
                    dey                             ; 
                    bpl .SaveSpriteData             ; 
                    
PrepareXmitX        rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ColorXmitBooth    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ColorXmitBooth      subroutine                      ; 
                    and #$f0                        ; isolate new color
                    sta M2Z_1a                      ; 
                    
                    ldx #$18                        ; 
                    stx M2Z_19                      ; 
                    
                    lda #<M2_ScreenLevel            ; 
                    sta M2Z_27                      ; 
                    lda #>M2_ScreenLevel            ; 
                    sta M2Z_28                      ; 
                    
.SetBothCols        ldx #$05                        ; 
                    ldy M2Z_LvlXmitBoothXpos        ; 
                    
                    dey                             ; 
                    
.ColorBooth         lda (M2Z_27),y                  ; get booth screen color
                    and #$0f                        ; clear new color
                    ora M2Z_1a                      ; set new color
                    sta (M2Z_27),y                  ; 
                    
                    iny                             ; next color pos
                    dex                             ; 
                    bne .ColorBooth                 ; 
                    
                    dec M2Z_19                      ; 
                    beq ColorXmitBoothX             ; 
                    
                    lda M2Z_27                      ; 
                    clc                             ; 
                    adc #$28                        ; set next line
                    sta M2Z_27                      ; 
                    bcc .SetBothCols                ; 
                    inc M2Z_28                      ; 
                    jmp .SetBothCols                ; 
                    
ColorXmitBoothX     jmp MarkXmitLevelNo             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__FlowTileHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__FlowTileHandler  subroutine                      ; 
                    lda #<M2_WaFlow                 ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    lda #>M2_WaFlow                 ; 
                    sta M2Z_FlowTileWaPtrHi         ; 
                    
.GetNextFlowYpos    ldy #M2_WaFlow_Ypos             ; 
                    lda (M2Z_FlowTileWaPtr),y       ; 
                    bne FlowTiles                   ; 
                    
.Exit               rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
FlowTiles           iny                             ; 
                    lda (M2Z_FlowTileWaPtr),y       ; M2_WaFlow_Delay
                    beq .SaveFlowWAI                ; 
                    
.DecStartDelayTime  sec                             ; 
                    sbc #$01                        ; 
                    sta (M2Z_FlowTileWaPtr),y       ; 
                    jmp .SetNextFlowWA              ; 
                    
.SaveFlowWAI        ldy #M2_WaFlowLen - 1           ; 
.SaveFlowWA         lda (M2Z_FlowTileWaPtr),y       ; 
                    sta M2Z_FlowWorkStart,y         ; 
                    dey                             ; 
                    bpl .SaveFlowWA                 ; 
                    
                    lda M2Z_FlowSpeed               ; 
                    sta M2Z_FlowDelay               ; 
                    
                    jsr MoveFlowTile                ; 
                    jsr MoveFlowTileSprite          ; 
                    jsr CheckFlowDirMax             ; 
                    bcc .RestoreFlowWAI             ; max not reached
                    
                    lda M2Z_FlowDir                 ; 
.ReverseFlowDir     eor #$ff                        ; 
                    sec                             ; 
                    adc #$00                        ; 
                    sta M2Z_FlowDir                 ; 
                    
.RestoreFlowWAI     ldy #M2_WaFlowLen - 1           ; 
.RestoreFlowWA      lda M2Z_FlowWorkStart,y         ; 
                    sta (M2Z_FlowTileWaPtr),y       ; 
                    dey                             ; 
                    bpl .RestoreFlowWA              ; 
                    
.SetNextFlowWA      clc                             ; 
                    lda #M2_WaFlowLen               ; 
                    adc M2Z_FlowTileWaPtrLo         ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    bcc .GetNextFlowYpos            ; 
                    inc M2Z_FlowTileWaPtrHi         ; 
                    
L__FlowTileHandlerX jmp .GetNextFlowYpos            ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveFlowTile      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveFlowTile        subroutine                      ; 
                    lda #$03                        ; 
                    sta M2Z_TileThickness           ; 
                    lda M2Z_FlowXpos                ; 
                    clc                             ; 
                    adc M2Z_FlowDir                 ; 
                    sta M2Z_FlowXpos                ; 
                    
                    lda M2Z_FlowPosY                ; 
                    sta M2Z_TargetScreenPosY        ; 
                    lda M2Z_FlowDir                 ; 
                    bne MoveFlowTileLiRe            ; not = M2_WaFlow_DirStop
                    
MoveFlowTileX       rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
MoveFlowTileLiRe    bmi MoveFlowTileLeft            ; = M2_WaFlow_DirLeft
; -------------------------------------------------------------------------------------------------------------- ;
MoveFlowTileRight   subroutine                      ; 
                    ldx M2Z_FlowXpos                ; 
                    dex                             ; 
                    stx M2Z_TargetScreenPosX        ; 
.SetTileHiResPtr    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldx M2Z_FlowLength              ; 
                    inx                             ; point to first flow tile pos
                    
                    ldy #$00                        ; 
.IniFillBits        sty M2Z_TileFillUpBits          ; ........        ......##        ......##
.ShiftTileDataRight lda (M2Z_HiResScreenPtr),y      ; ########        ########        ........
                    ror M2Z_TileFillUpBits          ; #.......        .......# -> c   .......# -> c
                    ror a                           ; .####### -> c   ######## -> c   #.......
                    ror M2Z_TileFillUpBits          ; ##......        #....... -> c   ........ -> c
                    ror a                           ; ..###### -> c   ######## -> c   ##......
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    rol M2Z_TileFillUpBits          ; #......# -> c   .......# -> c   ........
                    
.Fill00Bits1st      lda #$00                        ;                                 ........
                    bcc .StoreFiller                ; 
                    
.Fill02BitsNext     lda #$03                        ; ......##        ......##
                    
.StoreFiller        sta M2Z_TileFillUpBits          ; ......##        ......##        ........
                    
.SetNextTilePartPos clc                             ; 
                    tya                             ; 
                    adc #$08                        ; 
                    tay                             ; 
                    
                    dex                             ; M2Z_FlowLength
                    bne .ShiftTileDataRight         ; 
                    
.SetNextTileLayer   inc M2Z_TargetScreenPosY        ; 
                    dec M2Z_TileThickness           ; 
                    bne .SetTileHiResPtr            ; 
                    
MoveFlowTileRightX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
MoveFlowTileLeft    subroutine                      ; 
                    lda M2Z_FlowXpos                ; 
                    sta M2Z_TargetScreenPosX        ; 
.SetTileHiResPtr    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_FlowLength              ; ......#.
                    asl a                           ; .....#..
                    asl a                           ; ....#...
                    asl a                           ; ...#....
                    tay                             ; point to last flow tile pos
                    
                    lda #$00                        ; 
.IniFillBits        sta M2Z_TileFillUpBits          ; ........        ##......        ##......
.ShiftTileDataLeft  lda (M2Z_HiResScreenPtr),y      ; ########        ########        ........
                    rol M2Z_TileFillUpBits          ; ........        #......# -> c   #......# -> c
                    rol a                           ; #######. -> c   ######## -> c   .......#
                    rol M2Z_TileFillUpBits          ; .......#        ......## -> c   ......#. -> c
                    rol a                           ; ######.. -> c   ######## -> c   ......##
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    lda #$00                        ;                                 ........
                    bcc .StoreFiller                ; 
                    
                    lda #$c0                        ; ##......        ##......
                    
.StoreFiller        sta M2Z_TileFillUpBits          ; ##......        ##......        ........
                    
.SetNextTilePartPos tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .ShiftTileDataLeft          ; 
                    
.SetNextTileLayer   inc M2Z_TargetScreenPosY        ; 
                    dec M2Z_TileThickness           ; 
                    bne .SetTileHiResPtr            ; 
                    
MoveFlowTileLeftX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckSpriteFlows  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckSpriteFlows    subroutine                      ; 
                    clc                             ; NO:preset
                    lda M2_WaSprite_MoveType        ; 
                    bne CheckSpriteFlowsX           ; some kind of automove active - not movable
                    
                    lda M2_WaSprite_PosY            ; 
                    sec                             ; 
                    sbc M2_Wa_SpriteSinkPosY        ; 
                    sec                             ; 
                    sbc #$1d                        ; 
                    cmp M2Z_FlowPosY                ; 
                    clc                             ; NO:sprite not on flow tile
                    bne CheckSpriteFlowsX           ; 
                    
                    sec                             ; YES:sprite on flow tile and movable
                    
CheckSpriteFlowsX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveFlowTileSprite Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveFlowTileSprite  subroutine                      ; 
                    jsr CheckSpriteFlows            ; 
                    bcc MoveFlowTileSpriteX         ; NO
                    
                    lda M2_WaSprite_PosX            ; 
                    clc                             ; 
                    adc M2Z_FlowDir                 ; 
                    sta M2_WaSprite_PosX            ; 
                    
MoveFlowTileSpriteX rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckFlowDirMax   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckFlowDirMax     subroutine                      ; 
                    lda M2Z_FlowXpos                ; 
                    ldx M2Z_FlowDir                 ; 
                    bmi .ChkFlowDirMaxLe            ; M2_WaFlow_DirLeft
                    
.ChkFlowDirMaxRi    cmp M2Z_FlowMaxRight            ; 
                    
.exit               rts                             ; 

.ChkFlowDirMaxLe    lda M2Z_FlowMaxLeft             ; 
                    cmp M2Z_FlowXpos                ; 
                    
CheckFlowDirMaxX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L__TextOut        Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L__TextOut          subroutine                      ; 
                    stx M2Z_LevelTextType           ; $ff or $00
                    sty M2Z_LevelTextColorPtr       ; color pointer
                    
.GetNextOutPtr      ldy #$00                        ; 
                    lda (M2Z_LvlTextPtr),y          ; 
                    bne .SetTextOutPtr              ; 
                    
.Exit               rts                             ; 

.SetTextOutPtr      sta M2Z_TargetScreenPosX        ; 
                    iny                             ; 
                    lda (M2Z_LvlTextPtr),y          ; 
                    sta M2Z_TargetScreenPosY        ; 
                    iny                             ; 
                    lda (M2Z_LvlTextPtr),y          ; 
                    asl a                           ; *2
                    sta M2Z_TileGfxPtrLo            ; 
                    
                    lda #$00                        ; 
                    adc #>M2_CharSetBig             ; 
                    sta M2Z_TileGfxPtrHi            ; 
                    
                    ldx M2Z_LevelTextType           ; $ff or $00
                    ldy M2Z_LevelTextColorPtr       ; color pointer
                    jsr GraphicTextOut              ; 
                    
                    clc                             ; set next text pos
                    lda M2Z_LvlTextPtrLo            ; 
                    adc #$03                        ; text char tab entry length
                    sta M2Z_LvlTextPtrLo            ; 
                    bcc .GetNextOutPtr              ; 
                    inc M2Z_LvlTextPtrHi            ; 
                    
L__TextOutX         jmp .GetNextOutPtr              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L08FlowHandler    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L08FlowHandler      subroutine                      ; 
                    lda #<M2_WaFlow                 ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    lda #>M2_WaFlow                 ; 
                    sta M2Z_FlowTileWaPtrHi         ; 
                    
.GetNextFlowTile    ldy #$00                        ; 
                    lda (M2Z_FlowTileWaPtr),y       ; M2_WaFlow_Ypos
                    beq .SetHiResSlideDown          ; no slide left
                    
                    sta M2Z_FlowPosY                ; 
                    
                    ldy #$04                        ; 
                    lda (M2Z_FlowTileWaPtr),y       ; 
                    bne .CpyFlowWaToZpgI            ; 
                    
                    jsr CheckSpriteFlows            ; 
                    bcc .SetFlowTileWaPtr           ; NO
                    
                    ldx M2_WaSprite_PosX            ; 
                    lda #M2_WaFlow_DirLeft          ; 
                    cpx #$64                        ; max pos left
                    bcs .SetFlowMoveDir             ; greater/equal
                    
                    lda #M2_WaFlow_DirRight         ; 
                    
.SetFlowMoveDir     sta (M2Z_FlowTileWaPtr),y       ; 
                    jmp .SetFlowTileWaPtr           ; 
                    
.CpyFlowWaToZpgI    ldy #M2_WaFlowLen - 1           ; 
.CpyFlowWaToZpg     lda (M2Z_FlowTileWaPtr),y       ; 
                    sta M2Z_FlowWorkStart,y         ; 
                    dey                             ; 
                    bne .CpyFlowWaToZpg             ; 
                    
                    jsr MoveFlowTile                ; 
                    jsr MoveFlowTileSprite          ; 
                    jsr CheckFlowDirMax             ; 
                    bcc .CpyFlowZpgToWaI            ; 
                    
                    lda #M2_WaFlow_DirStop          ; 
                    sta M2Z_FlowDir                 ; 
                    
.CpyFlowZpgToWaI    ldy #M2_WaFlowLen - 1           ; 
.CpyFlowZpgToWa     lda M2Z_FlowWorkStart,y         ; 
                    sta (M2Z_FlowTileWaPtr),y       ; 
                    dey                             ; 
                    bpl .CpyFlowZpgToWa             ; 
                    
.SetFlowTileWaPtr   clc                             ; 
                    lda M2Z_FlowTileWaPtrLo         ; 
                    adc #M2_WaFlowLen               ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    bcc .GetNextFlowTile            ; 
                    inc M2Z_FlowTileWaPtrHi         ; 
                    jmp .GetNextFlowTile            ; 
                    
.SetHiResSlideDown  lda #<Level_08FlowTile01        ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    lda #>Level_08FlowTile01        ; 
                    sta M2Z_FlowTileWaPtrHi         ; 
                    
                    lda #$10                        ; 
                    sta M2Z_SlideDownHeight         ; 
                    
.ColorSlideDownBarI ldy #$0f                        ; M2Z_SlideDownWidth
.ColorSlideDownBar  lda (M2Z_FlowTileWaPtr),y       ; 
                    ora #$aa                        ; color pointer
                    sta (M2Z_FlowTileWaPtr),y       ; 
                    dey                             ; 
                    bpl .ColorSlideDownBar          ; 
                    
                    dec M2Z_SlideDownHeight         ; 
                    bne .SetNextHiResLine           ; 
                    
.Exit               rts                             ; 

.SetNextHiResLine   clc                             ; 
                    lda M2Z_FlowTileWaPtrLo         ; 
                    adc #<M2_Screen_HiResRowLen     ; 
                    sta M2Z_FlowTileWaPtrLo         ; 
                    lda M2Z_FlowTileWaPtrHi         ; 
                    adc #>M2_Screen_HiResRowLen     ; 
                    sta M2Z_FlowTileWaPtrHi         ; 
                    
L08FlowHandlerX     jmp .ColorSlideDownBarI         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckLifterStart  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckLifterStart    subroutine                      ; 
                    lda M2_WaSprite_MoveType        ; 
                    bne .SetLifterInactive          ; some kind of auto move ongoing
                    
                    lda M2Z_LifterPlatformPosY      ; 
                    clc                             ; 
                    adc #$1d                        ; 
                    cmp M2_WaSprite_PosY            ; 
                    bne .SetLifterInactive          ; player not on platform
                    
                    lda M2_WaSprite_PosX            ; 
                    cmp #$44                        ; 
                    bcc .SetLifterInactive          ; lower
                    
                    cmp #$80                        ; 
                    bcs .SetLifterInactive          ; higher/equal
                    
                    lda M2Z_PlayerDeadWobble        ; 
                    ora M2Z_PlayerDeadSink          ; 
                    bne .SetLifterInactive          ; 
                    
                    lda CIAPRB                      ; CIA1($DC01) Data Port Register B
                    eor #$ff                        ; 
                    and #$10                        ; fire
                    and M2Z_JoystickFireRev         ; reversed
                    bne .ChkLifterStatus            ; 
                    
.ExitLifterOut      rts                             ; 
                    
.ChkLifterStatus    inc M2Z_LifterStatus            ; 
                    lda M2Z_LifterStatus            ; 
                    bne .ChkActivation              ; 
                    
                    sta M2Z_JoystickFireRev         ; $00 - reversed
                    sta M2Z_SoundLifterCannon       ; 
                    
                    lda #WHITE                      ; 
                    jmp .ColorLifterWagonI          ; 
                    
.ChkActivation      cmp #$02                        ; 
                    beq .SetLifterActive            ; 
                    
.ExitLifterInact    rts                             ; 
                    
.SetLifterActive    lda #M2Z_SoundLifterCannonOn    ; 
                    sta M2Z_SoundLifterCannon       ; lifter/cannon
                    
                    lda #M2Z_LifterActive           ; 
                    sta M2Z_LifterStatus            ; 
                    
                    lda #LT_RED                     ; 
.ColorLifterWagonI  ldx #$10                        ; 
.ColorLifterWagon   sta COLORAM + $03a6,x           ; 
                    dex                             ; 
                    bpl .ColorLifterWagon           ; 
                    
.ExitLifterIsAct    rts                             ; 
                    
.SetLifterInactive  lda #M2Z_LifterInActive         ; 
                    sta M2Z_LifterStatus            ; 
                    
CheckLifterStartX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L07LifterHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L07LifterHandler    subroutine                      ; 
                    jsr CheckJoystick               ; 
                    
.ChkLifterLeRi      lda M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    beq .GetJoyUpDo                 ; 
                    bmi .ChkLifterMaxLeft           ; 
                    
.ChkLifterMaxRight  lda M2Z_LifterPlatformPosX      ; 
                    cmp #$68                        ; right limit
                    bcs .GetJoyUpDo                 ; greater/equal
                    
.MovLifterRight     sta M2Z_TargetScreenPosX        ; 
                    lda M2Z_LifterPlatformPosY      ; 
                    sta M2Z_TargetScreenPosY        ; 
                    inc M2Z_LifterPlatformPosX      ; 
                    inc M2_WaSprite_PosX            ; 
                    jsr MoveLifterRight             ; 
                    jmp .GetJoyUpDo                 ; 
                    
.ChkLifterMaxLeft   ldx M2Z_LifterPlatformPosX      ; 
                    cpx #$3f                        ; left limit
                    bcc .GetJoyUpDo                 ; lower
                    
.MovLifterLeft      dex                             ; 
                    stx M2Z_LifterPlatformPosX      ; 
                    stx M2Z_TargetScreenPosX        ; 
                    lda M2Z_LifterPlatformPosY      ; 
                    sta M2Z_TargetScreenPosY        ; 
                    dec M2_WaSprite_PosX            ; 
                    jsr MoveLifterLeft              ; 
                    
.GetJoyUpDo         lda M2Z_JoystickUpDo            ; move_no=$00  move_up=$ff  move_down=$01
                    bne .ChkLifterUpDo              ; 
                    
.ExitLifterNoLeRi   rts                             ; 
                    
.ChkLifterUpDo      bpl .ChkLifterMaxDown           ; 
                    
.ChkLifterMaxUp     ldx M2Z_LifterPlatformPosY      ; 
                    cpx #$69                        ; 
                    bcs .MoveLifterUp               ; greater/equal
                    
                    rts                             ; 
                    
.MoveLifterUp       dec M2_WaSprite_PosY            ; 
                    dex                             ; 
                    stx M2Z_LifterPlatformPosY      ; 
                    stx M2Z_TargetScreenPosY        ; 
                    lda M2Z_LifterPlatformPosX      ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_27                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_28                      ; 
                    inc M2Z_TargetScreenPosY        ; 
                    inc M2Z_TargetScreenPosY        ; 
                    inc M2Z_TargetScreenPosY        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_25                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_26                      ; 
                    lda M2Z_TargetScreenPosY        ; 
                    clc                             ; 
                    adc #$07                        ; 
                    sta M2Z_TargetScreenPosY        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$20                        ; 
.LifterUp           lda (M2Z_25),y                  ; 
                    sta (M2Z_27),y                  ; 
                    lda (M2Z_HiResScreenPtr),y      ; 
                    sta (M2Z_25),y                  ; 
                    
                    tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .LifterUp                   ; 
                    
.ExitLifterUp       rts                             ; 
                    
.ChkLifterMaxDown   ldx M2Z_LifterPlatformPosY      ; 
                    cpx #$ad                        ; 
                    bcc .MoveLifterDown             ; 
                    
.ExitLifterMaxDown  rts                             ; 
                    
.MoveLifterDown     inc M2_WaSprite_PosY            ; 
                    inc M2Z_LifterPlatformPosY      ; 
                    stx M2Z_TargetScreenPosY        ; 
                    lda M2Z_LifterPlatformPosX      ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_27                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_28                      ; 
                    inc M2Z_TargetScreenPosY        ; 
                    inc M2Z_TargetScreenPosY        ; 
                    inc M2Z_TargetScreenPosY        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$20                        ; 
.LifterDown         lda (M2Z_27),y                  ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    lda #$00                        ; 
                    sta (M2Z_27),y                  ; 
                    
                    tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .LifterDown                 ; 
                    
L07LifterHandlerX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveLifterRight   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveLifterRight     subroutine                      ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$00                        ; 
                    sty M2Z_19                      ; 
.Move               lda (M2Z_HiResScreenPtr),y      ; 
                    ror M2Z_19                      ; 
                    ror a                           ; /2
                    ror M2Z_19                      ; 
                    ror a                           ; /4
                    rol M2Z_19                      ; 
                    rol M2Z_19                      ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    tya                             ; 
                    clc                             ; 
                    adc #$08                        ; 
                    tay                             ; 
                    cpy #$21                        ; 
                    bcc .Move                       ; 
                    
.SetNextRight       ldx M2Z_TargetScreenPosY        ; 
                    inx                             ; 
                    stx M2Z_TargetScreenPosY        ; 
                    
.ChkMaxRight        cpx #$c3                        ; 
                    bcc MoveLifterRight             ; lower
                    
MoveLifterRightX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveLifterLeft    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveLifterLeft      subroutine                      ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda #$00                        ; 
                    sta M2Z_19                      ; 
                    ldy #$20                        ; 
.Move               lda (M2Z_HiResScreenPtr),y      ; 
                    rol M2Z_19                      ; 
                    rol a                           ; 
                    rol M2Z_19                      ; 
                    rol a                           ; 
                    ror M2Z_19                      ; 
                    ror M2Z_19                      ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .Move                       ; 
                    
.SetNextLeft        ldx M2Z_TargetScreenPosY        ; 
                    inx                             ; 
                    stx M2Z_TargetScreenPosY        ; 
.ChkMaxLeft         cpx #$c3                        ; 
                    bcc MoveLifterLeft              ; lower
                    
MoveLifterLeftX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L07BadFlowHandler Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L07BadFlowHandler   subroutine                      ; 
.FlickerColor       lda M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_Color0 ; 
                    and #$07                        ; .....### - WHITE to ORANGE
                    tax                             ; 
                    inx                             ; 
                    stx M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_Color0 ; 
                    stx M2_WaSprite_0b - M2_WaSprite + M2_WaSprite_Color0 ; 
                    
                    lda M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_StepWait ; 
                    bmi .MoveLeft                   ; 
                    
.MoveRight          inc M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_PosX ; 
                    inc M2_WaSprite_0b - M2_WaSprite + M2_WaSprite_PosX ; 
                    
                    lda M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_PosX ; 
.ChkMaxRight        cmp #$6c                        ; 
                    bcc .ChkAtPlayerPosY            ; lower
                    bcs .FlipMoveDir                ; greater/equal
                    
.MoveLeft           dec M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_PosX ; 
                    dec M2_WaSprite_0b - M2_WaSprite + M2_WaSprite_PosX ; 
                    
                    lda M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_PosX ; 
.ChkMaxLeft         cmp #$2c                        ; 
                    bcs .ChkAtPlayerPosY            ; reater/equal
                    
.FlipMoveDir        lda M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_StepWait ; 
                    eor #$ff                        ; 
                    sta M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_StepWait ; 
                    
.ChkAtPlayerPosY    lda M2_WaSprite_PosY            ; 
                    cmp #$d7                        ; M2_WaSprite_PosY - no of sprite rows $15
                    bcs .ChkAtPlayerPosX            ; greater/equal
                    
.ExitPlayerNoPosY   rts                             ; 
                    
.ChkAtPlayerPosX    lda M2_WaSprite_PosX            ; 
                    sec                             ; 
                    sbc M2_WaSprite_0a - M2_WaSprite + M2_WaSprite_PosX ; 
                    bmi .AddIRQs                    ; 
                    
                    cmp #$1e                        ; 
                    bcc .SetDeath                   ; lower
                    
.ExitPlayerNoPosX   rts                             ; 
                    
.AddIRQs            adc M2Z_WaitIRQs                ; 
                    bpl .SetDeath                   ; 
                    
.ExitAlive          rts                             ; 
                    
.SetDeath           lda #M2Z_PlayerDeadWobbleYes    ; 
                    sta M2Z_PlayerDeadWobble        ; 
                    
L07BadFlowHandlerX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L08StompHandler   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L08StompHandler     subroutine                      ; 
                    lda #<M2_WaMisc_Stomper         ; 
                    sta M2Z_StomperWaPtrLo          ; 
                    lda #>M2_WaMisc_Stomper         ; 
                    sta M2Z_StomperWaPtrHi          ; 
                    
.CpyStompWa2WorkI   ldy #M2_WaMisc_StomperLen - 1   ; 
.CpyStompWa2Work    lda (M2Z_StomperWaPtr),y        ; 
                    sta M2Z_StompWorkStart,y        ; 
                    dey                             ; 
                    bpl .CpyStompWa2Work            ; 
                    
                    tax                             ; 
                    beq .GetStompOldColor           ; end of stomper data
                    
                    lda M2Z_StompTime               ; 
                    bmi .ChkStomperDir              ; 
                    
                    dec M2Z_StompTime               ; 
                    bne .ChkTouchPlayer             ; 
                    
                    lda #$c0                        ; 
                    sta M2Z_StompTime               ; 
                    bne .ChkTouchPlayer             ; always
                    
.ChkStomperDir      asl a                           ; 
                    bpl .GoMoveStomperDown          ; 
                    
.GoMoveStomperUp    jsr MoveStompUp                 ; 
                    
                    lda M2Z_StompPosY               ; 
                    cmp #$c0                        ; 
                    bne .ChkTouchPlayer             ; 
                    
                    lda #$80                        ; 
                    sta M2Z_StompTime               ; 
                    bne .ChkTouchPlayer             ; 
                    
.GoMoveStomperDown  jsr MoveStompDown               ; 
                    
                    lda M2Z_StompPosY               ; 
                    cmp #$ac                        ; 
                    bne .ChkTouchPlayer             ; 
                    
                    lda #$55                        ; 
                    sta M2Z_StompTime               ; 
                    
.ChkTouchPlayer     lda M2Z_StompPosX               ; 
                    clc                             ; 
                    adc #$05                        ; 
                    cmp M2_WaSprite_PosX            ; 
                    bcs .CpyStompWork2WaI           ; 
                    
                    adc #$13                        ; 
                    cmp M2_WaSprite_PosX            ; 
                    bcc .CpyStompWork2WaI           ; 
                    
                    lda M2Z_StompPosY               ; 
                    adc #$2c                        ; 
                    cmp M2_WaSprite_PosY            ; 
                    bcs .CpyStompWork2WaI           ; 
                    
                    adc #$04                        ; 
                    cmp M2_WaSprite_PosY            ; 
                    bcc .CpyStompWork2WaI           ; 
                    
.Touched            lda #M2Z_PlayerDeadWobbleYes    ; kill player
                    sta M2Z_PlayerDeadWobble        ; 
                    
.CpyStompWork2WaI   ldy #$02                        ; 
.CpyStompWork2Wa    lda M2Z_StompWorkStart,y        ; 
                    sta (M2Z_StomperWaPtr),y        ; 
                    dey                             ; 
                    bpl .CpyStompWork2Wa            ; 
                    
.SetNextStompBlock  clc                             ; 
                    lda M2Z_StomperWaPtrLo          ; 
                    adc #M2_WaMisc_StomperLen       ; 
                    sta M2Z_StomperWaPtrLo          ; 
                    bcc .CpyStompWa2WorkI           ; 
                    inc M2Z_StomperWaPtrHi          ; 
                    jmp .CpyStompWa2WorkI           ; 
                    
.GetStompOldColor   lda Level_08Stomper             ; get actual color
                    and #$f0                        ; ####....
                    cmp #HR_BlueBlack               ; 
                    bcc .AddStompNewColor           ; lower
                    
                    lda #HR_WhiteBlack              ; 
                    
.AddStompNewColor   clc                             ; 
                    adc #HR_WhiteBlack              ; 
                    ora #HR_BlackRed                ; 
                    
                    ldx #Level_08StomperHeight      ; 
.SetStompNewColor   sta Level_08StomperRow1,x       ; color the whole area
                    sta Level_08StomperRow2,x       ; 
                    sta Level_08StomperRow3,x       ; 
                    sta Level_08StomperRow4,x       ; 
                    dex                             ; 
                    bpl .SetStompNewColor           ; 
                    
L08StompHandlerX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveStompUp       Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveStompUp         subroutine                      ; 
                    inc M2Z_StompPosY               ; 
                    
                    lda M2Z_StompPosY               ; 
                    jsr GetStompPtrs                ; 
                    
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_21                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_22                      ; 
                    dec M2Z_TargetScreenPosY        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$10                        ; 
.MoveUp             lda (M2Z_23),y                  ; 
                    sta (M2Z_25),y                  ; 
                    
                    lda (M2Z_21),y                  ; 
                    sta (M2Z_23),y                  ; 
                    
                    lda (M2Z_HiResScreenPtr),y      ; 
                    sta (M2Z_21),y                  ; 
                    
                    tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .MoveUp                     ; 
                    
MoveStompUpX        rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MoveStompDown     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MoveStompDown       subroutine                      ; 
                    lda M2Z_StompPosY               ; 
                    dec M2Z_StompPosY               ; 
                    jsr GetStompPtrs                ; 
                    
                    ldy #$10                        ; 
.MoveDown           lda (M2Z_23),y                  ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    lda (M2Z_25),y                  ; 
                    sta (M2Z_23),y                  ; 
                    
                    lda #$00                        ; 
                    sta (M2Z_25),y                  ; 
                    
                    tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .MoveDown                   ; 
                    
MoveStompDownX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetStompPtrs      Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetStompPtrs        subroutine                      ; 
                    sta M2Z_TargetScreenPosY        ; M2Z_StompPosY
                    lda M2Z_StompPosX               ; 
                    sta M2Z_TargetScreenPosX        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_25                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_26                      ; 
                    
                    dec M2Z_TargetScreenPosY        ; 
                    dec M2Z_TargetScreenPosY        ; 
                    jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    lda M2Z_HiResScreenPtrLo        ; 
                    sta M2Z_23                      ; 
                    lda M2Z_HiResScreenPtrHi        ; 
                    sta M2Z_24                      ; 
                    
                    dec M2Z_TargetScreenPosY        ; 
                    dec M2Z_TargetScreenPosY        ; 
GetStompPtrsX       jmp GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
; -------------------------------------------------------------------------------------------------------------- ;
; L09TNTWallCheck   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L09TNTBoxWallChk    subroutine                      ; 
                    ldx M2_WaSprite_PosX            ; 
                    ldy M2_WaSprite_PosY            ; 
                    
.ChkWallCeiling1st  cpy #$87                        ; M2_WaSprite_PosY
                    bcc .ExitNoWallOut              ; lower - was ceiling
                    
.ChkWallMaxRightOut cpx #$41                        ; M2_WaSprite_PosX
                    bne .ChkWallPosX                ; 
                    
.ChkWallDoorOut     cpy #$e4                        ; M2_WaSprite_PosY
                    beq .ExitNoWallOut              ; 
                    
.ExitHitWallOut     inx                             ; 
                    stx M2_WaSprite_PosX            ; 
                    
.ExitNoWallOut      rts                             ; 
                    
.ChkWallPosX        bcs .ExitNoWallOut              ; greater/equal
                    
.ChkWallMaxRightIn  cpx #$39                        ; M2_WaSprite_PosX
                    bcc .ChkWallMaxLeftIn           ; lower
                    bne .ChkWallDoorIn              ; 
                    
                    cpy #$e1                        ; M2_WaSprite_PosY
                    bcs .ChkWallDoorIn              ; greater/equal
                    
                    dex                             ; 
                    bne .ExitHitWallIn              ; always
                    
.ChkWallMaxLeftIn   cpx #$0e                        ; M2_WaSprite_PosX
                    bcs .ChkWallCeiling2nd          ; greater/equal
                    
                    inx                             ; 
.ExitHitWallIn      stx M2_WaSprite_PosX            ; 
                    
.ChkWallCeiling2nd  cpy #$9d                        ; M2_WaSprite_PosY
                    bcs .ExitNoWallIn               ; greater/equal
                    
.SetMoveTime        ldy #$01                        ; 
                    sty M2_WaSprite_MoveTime        ; 
                    
.ExitNoWallIn       rts                             ; 
                    
.ChkWallDoorIn      cpy #$e4                        ; M2_WaSprite_PosY
                    bcc .SetMoveTime                ; lower
                    
L09TNTBoxWallChkX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L09TNTHandler     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L09TNTHandler       subroutine                      ; 
                    ldx M2_WaSprite_09 + M2_WaSprite_State    - M2_WaSprite ;
                    ldy M2_WaSprite_09 + M2_WaSprite_StepWait - M2_WaSprite ; 
                    lda M2_WaSprite_09 + M2_WaSprite_Pointer  - M2_WaSprite ; 
                    jsr GetTNTHandler               ; 
                    
                    ldx M2_WaSprite_0a + M2_WaSprite_State    - M2_WaSprite ;
                    ldy M2_WaSprite_0a + M2_WaSprite_StepWait - M2_WaSprite ; 
                    lda M2_WaSprite_0a + M2_WaSprite_Pointer  - M2_WaSprite ; 
                    jsr GetTNTHandler               ; 
                    
                    ldx M2_WaSprite_0b + M2_WaSprite_State    - M2_WaSprite ;
                    ldy M2_WaSprite_0b + M2_WaSprite_StepWait - M2_WaSprite ; 
                    lda M2_WaSprite_0b + M2_WaSprite_Pointer  - M2_WaSprite ; 
                    
L09TNTHandlerX      jmp GetTNTHandler               ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetTNTHandler     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetTNTHandler       subroutine                      ; 
.GetOnlyOnce        cpy #M2_WaSprite_StepWaitScore  ; item score time init
                    bne .ExitNoTNT                  ; 
                    
                    cpx #M2_WaSprite_StateScore     ; TNT item taken
                    beq .GetTNT                     ; 
                    
.ExitNoTNT          rts                             ; 
                    
.GetTNT             sec                             ; 
                    sbc #PtrSprScoreBase            ; convert M2_WaSprite_Pointer to digit
                    tax                             ; 
                    clc                             ; 
                    adc M2Z_AmountHex               ; 
                    sta M2Z_AmountHex               ; 
                    
                    txa                             ; 
                    adc M2Z_AmountTNT10             ; 
                    sta M2Z_AmountTNT10             ; 
                    
                    cmp #$0a                        ; 
                    bcc .GetTNT10                   ; lower
                    
                    adc #$05                        ; 
                    and #$0f                        ; 
                    sta M2Z_AmountTNT10             ; 
                    
                    inc M2Z_AmountTNT100            ; 
                    
.GetTNT10           lda M2Z_AmountTNT100            ; 
                    beq .GetTNT1                    ; 
                    
                    ora #$30                        ; make digit
                    ldx #Level_09TNTPosX100         ; 
                    ldy #Level_09TNTPosY            ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
.GetTNT1            lda M2Z_AmountTNT10             ; 
                    ora #$30                        ; make digit
                    ldx #Level_09TNTPosX100         ; 
                    ldy #Level_09TNTPosY            ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda M2Z_AmountHex               ; 
                    cmp #M2Z_AmountHexAlert         ; 
                    bcc GetTNTHandlerX              ; lower
                    
                    ldx #$0e                        ; 
                    lda #HR_LtRedLtRed              ; 
.ShowWarning        sta Level_09TextWarnColor,x     ; make "DANGER" visible
                    dex                             ; 
                    bpl .ShowWarning                ; 
                    
GetTNTHandlerX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GraphicTextOutNoR Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicTextOutNoR   subroutine                      ; non reversed text out
                    stx M2Z_TargetScreenPosX        ; 
                    sty M2Z_TargetScreenPosY        ; 
                    jsr GetBigCharPtr               ; 
                    
                    ldx #M2Z_LevelTextNormal        ; 
                    ldy #$aa                        ; color pointer
                    
GraphicTextOutNoRX  jmp GraphicTextOut              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetBigCharPtr     Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetBigCharPtr       subroutine                      ; 
                    and #$3f                        ; ..######
                    ldx #$00                        ; 
                    stx M2Z_TileGfxPtrHi            ; 
                    asl a                           ; 
                    asl a                           ; 
                    asl a                           ; 
                    rol M2Z_TileGfxPtrHi            ; 
                    asl a                           ; 
                    rol M2Z_TileGfxPtrHi            ; 
                    
                    adc #<M2_CharSetBig             ; 
                    sta M2Z_TileGfxPtrLo            ; 
                    lda M2Z_TileGfxPtrHi            ; 
                    adc #>M2_CharSetBig             ; 
                    sta M2Z_TileGfxPtrHi            ; 
                    
GetBigCharPtrX      rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; L09CannonHandler  Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
L09CannonHandler    subroutine                      ; 
                    ldx M2Z_CannonStatus            ; 
                    bne .ChkCannonStatus            ; 
                    
.AutoMoveBack       jsr CannonMoveRight             ; M2Z_CannonStatusEmpty - move back to start pos
                    
                    lda M2_WaSprite_MoveType        ; 
                    cmp #M2_WaSprite_MoveFallDo     ; 
                    bne .ExitNotInCannon            ; 
                    
.ChkPosXMin         lda M2_WaSprite_PosX            ; 
                    cmp #$7f                        ; 
                    bcc .ExitNotInCannon            ; 
                    
.ChkPosXMax         cmp #$82                        ; 
                    bcs .ExitNotInCannon            ; 
                    
                    lda M2_WaSprite_PosY            ; 
.ChkPosYMin         cmp #$b5                        ; 
                    bcc .ExitNotInCannon            ; 
                    
.ChkPosYMax         cmp #$bd                        ; 
                    bcs .ExitNotInCannon            ; 
                    
.ChkTNT             lda M2Z_AmountHex               ; 
                    beq .ExitNotInCannon            ; 
                    
.SetHasTNT          inc M2Z_CannonStatus            ; M2Z_CannonStatusHasTNT
                    
                    lda #$ff                        ; 
                    sta M2_WaSprite_BackPrio        ; hide in cannon
                    
                    lda #$80                        ; 
                    sta M2_WaSprite_PosX            ; 
                    
.ExitNotInCannon    rts                             ; 
                    
.ChkCannonStatus    dex                             ; 
                    bne .ChkCannonLoaded            ; 
                    
.WasCannonHasTNT    lda M2_WaSprite_PosY            ; 
                    cmp #$ca                        ; 
                    bcc .ExitNotInCannon            ; lower
                    
                    lda #M2_WaSprite_MoveFly        ; 
                    sta M2_WaSprite_MoveType        ; 
                    
.SetLoaded          inc M2Z_CannonStatus            ; M2Z_CannonStatusLoaded
                    
.ExitLoaded         rts                             ; 
                    
.ChkCannonLoaded    dex                             ; 
                    bne .WasCannonFired             ; 
                    
.WasCannonLoaded    lda #M2Z_SoundLifterCannonOn    ; 
                    sta M2Z_SoundLifterCannon       ; 
                    
                    lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    beq .ChkCannonMove              ; 
                    
                    lda #M2Z_SoundLifterCannonOff   ; 
                    sta M2Z_SoundLifterCannon       ; 
                    
                    lda #M2Z_SoundKillOn            ; 
                    sta M2Z_SoundKill               ; 
                    
.ClrTNTDisplay      lda #$20                        ; " "
                    ldx #Level_09TNTPosX100         ; 
                    ldy #Level_09TNTPosY            ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda #$20                        ; " "
                    ldx #Level_09TNTPosX10          ; 
                    ldy #Level_09TNTPosY            ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda #$98                        ; target distance $01
                    
                    ldx M2Z_AmountHex               ; 
                    dex                             ; $01
                    beq .SetShotDistance            ; 
                    
                    lda #$6d                        ; target distance $02
                    dex                             ; $02
                    beq .SetShotDistance            ; 
                    
                    lda #$45                        ; target distance $03
                    dex                             ; 
                    beq .SetShotDistance            ; $04
                    
                    lda #$1b                        ; target distance $04 - all other values
                    
.SetShotDistance    sta M2Z_CannonShotDist          ; 
                    
.SetFired           inc M2Z_CannonStatus            ; M2Z_CannonStatusFired
                    
.ExitFired          rts                             ; 
                    
.ChkCannonMove      lda M2Z_JoystickLeRi            ; move_no=$00  move_left=$ff  move_right=$01
                    beq L09CannonHandlerX           ; 
                    bmi .GoCannonMoveLeft           ; 
                    
.GoCannonMoveRight  jmp CannonMoveRight             ; and return
.GoCannonMoveLeft   jmp CannonMoveLeft              ; and return
                    
.WasCannonFired     lda M2_WaSprite_PosY            ; 
                    sec                             ; 
                    sbc M2Z_CannonShotDist          ; 
                    bcc .CannonShotAtTarget         ; 
                    
.FlightControl      ldx #$fd                        ; height add init
                    cmp #$0a                        ; 
                    bcc .SetFightHeight             ; lower
                    
                    dex                             ; height add
                    cmp #$1e                        ; lower
                    bcc .SetFightHeight             ; 
                    
                    dex                             ; height add
                    cmp #$50                        ; 
                    bcc .SetFightHeight             ; lower
                    
                    dex                             ; height add
                    dex                             ; 
                    
.SetFightHeight     txa                             ; 
                    clc                             ; 
                    adc M2_WaSprite_PosY            ; 
                    sta M2_WaSprite_PosY            ; 
                    
                    cmp #$b9                        ; 
                    bcs .ExitCannonShotFly          ; greater/equal
                    
                    lda #$00                        ; 
                    sta M2_WaSprite_BackPrio        ; 
                    
.ExitCannonShotFly  rts                             ; 
                    
.CannonShotAtTarget lda #M2_WaSprite_MoveJumpUp     ; 
                    sta M2_WaSprite_MoveType        ; 
                    
                    lda #$10                        ; 
                    sta M2_WaSprite_MoveTime        ; 
                    
                    lda #$00                        ; 
                    sta M2_WaSprite_MoveDirX        ; $00=no $01=right $ff=left
                    sta M2Z_CannonStatus            ; 
                    sta M2Z_AmountHex               ; 
                    sta M2Z_AmountTNT10             ; 
                    sta M2Z_AmountTNT100            ; 
                    
L09CannonHandlerX   rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonMoveLeft    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonMoveLeft      subroutine                      ; 
                    ldx M2Z_CannonAtPosX            ; 
                    cpx #M2Z_CannonAtPosXMin        ; 
                    bcc CannonMoveLeftX             ; 
                    
                    dec M2_WaSprite_PosX            ; 
                    dex                             ; 
                    stx M2Z_CannonAtPosX            ; 
                    stx M2Z_TargetScreenPosX        ; 
                    
                    lda #$a0                        ; 
                    sta M2Z_TargetScreenPosY        ; 
                    
                    jsr WaitBeamOffScreen           ; 
.GetNextMovePtr     jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$00                        ; 
                    sty M2Z_19                      ; 
                    ldy #$28                        ; 
.Move               lda (M2Z_HiResScreenPtr),y      ; 
                    rol M2Z_19                      ; 
                    rol a                           ; 
                    rol M2Z_19                      ; 
                    rol a                           ; 
                    ror M2Z_19                      ; 
                    ror M2Z_19                      ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
.SetNextLeft        tya                             ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    tay                             ; 
                    bpl .Move                       ; 
                    
.ChkMaxLeft         ldx M2Z_TargetScreenPosY        ; 
                    inx                             ; 
                    stx M2Z_TargetScreenPosY        ; 
                    cpx #$c3                        ; 
                    bcc .GetNextMovePtr             ; 
                    
CannonMoveLeftX     rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonMoveRight   Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonMoveRight     subroutine                      ; 
                    ldx M2Z_CannonAtPosX            ; 
                    cpx #M2Z_CannonAtPosXMax        ; 
                    bcs CannonMoveRightX            ; 
                    
                    stx M2Z_TargetScreenPosX        ; 
                    inx                             ; 
                    stx M2Z_CannonAtPosX            ; 
                    
                    ldx #$a0                        ; 
                    stx M2Z_TargetScreenPosY        ; 
                    
                    lda M2Z_CannonStatus            ; 
                    cmp #$02                        ; 
                    bne .WaitBeamOff                ; 
                    
                    inc M2_WaSprite_PosX            ; 
                    
.WaitBeamOff        jsr WaitBeamOffScreen           ; 
.GetNextMovePtr     jsr GetHiresPtr                 ; in:M2Z_TargetScreenPosX/Y  out:M2Z_HiResScreenPtr
                    
                    ldy #$00                        ; 
                    sty M2Z_19                      ; 
.Move               lda (M2Z_HiResScreenPtr),y      ; 
                    ror M2Z_19                      ; 
                    ror a                           ; 
                    ror M2Z_19                      ; 
                    ror a                           ; 
                    rol M2Z_19                      ; 
                    rol M2Z_19                      ; 
                    sta (M2Z_HiResScreenPtr),y      ; 
                    
                    tya                             ; 
                    clc                             ; 
                    adc #$08                        ; 
                    tay                             ; 
                    cpy #$29                        ; 
                    bcc .Move                       ; 
                    
.SetNextRight       ldx M2Z_TargetScreenPosY        ; 
                    inx                             ; 
                    stx M2Z_TargetScreenPosY        ; 
                    
.ChkMaxRight        cpx #$c3                        ; 
                    bcc .GetNextMovePtr             ; lower
                    
CannonMoveRightX    rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; WaitBeamOffScreen Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
WaitBeamOffScreen   subroutine                      ; 
                    lda RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    cmp #$f5                        ; 
                    bcc WaitBeamOffScreen           ; lower
                    
WaitBeamOffScreenX  rts                             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShowInfoScreen    Does    : 
;                   Expects : 
;                   Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShowInfoScreen      subroutine                      ; 
                    sei                             ; 

                    jsr ClearHiScreen               ; 
                    lda #HR_YellowYellow            ; 
                    jsr ColourScreen                ; 
                    lda #HR_YellowYellow            ; 
                    jsr ColourRam                   ; 
                    
                    ldx #$00                        ; 
.BlankOutLoop       lda #$20                        ; blank
                    sta M2_ScreenGame + $0208,x     ; 
                    sta M2_ScreenGame + $02f8,x     ; 
                    
                    lda #ORANGE                     ; 
                    sta COLORAM + $0208,x           ; 
                    sta COLORAM + $02f8,x           ; 
                    
                    dex                             ; 
                    bne .BlankOutLoop               ; 
                    
                    lda #$6e                        ; hbu - $8a
                    sta M2Z_InfoTextPosX            ; 
                    lda #$0b                        ; hbu - $11
                    sta M2Z_InfoTextLen             ; 
                    
.OutTextLine01      ldx M2Z_InfoTextPosX            ; PosX
                    ldy M2Z_InfoTextLen             ; 
                    lda InfoTextLine01,y            ; 
                    ldy #$28                        ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda M2Z_InfoTextPosX            ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    sta M2Z_InfoTextPosX            ; 
                    
                    dec M2Z_InfoTextLen             ; 
                    bpl .OutTextLine01              ; 
                    
                    lda #$72                        ; hbu - $76
                    sta M2Z_InfoTextPosX            ; 
                    lda #$0c                        ; 
                    sta M2Z_InfoTextLen             ; 
                    
.OutTextLine02      ldx M2Z_InfoTextPosX            ; PosX
                    ldy M2Z_InfoTextLen             ; 
                    lda InfoTextLine02,y            ; 
                    ldy #$38                        ; 
                    jsr GraphicTextOutNoR           ; non reversed text out
                    
                    lda M2Z_InfoTextPosX            ; 
                    sec                             ; 
                    sbc #$08                        ; 
                    sta M2Z_InfoTextPosX            ; 
                    
                    dec M2Z_InfoTextLen             ; 
                    bpl .OutTextLine02              ; 
                    
                    ldx #$0f                        ; 
.OutTextLines       lda InfoTextLine03,x            ; 
                    sta M2_ScreenGame + $020c,x     ; 
                    lda InfoTextLine04,x            ; 
                    sta M2_ScreenGame + $02dd,x     ; 
                    lda InfoTextLine05,x            ; 
                    sta M2_ScreenGame + $0305,x     ; 
                    lda InfoTextLine06,x            ; 
                    sta M2_ScreenGame + $0386,x     ; 
                    dex                             ; 
                    bpl .OutTextLines               ; 
                    
                    lda #$38                        ; 
                    sta SP0X                        ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                    lda #$aa                        ; 
                    sta SP0Y                        ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                    lda #$00                        ; 
                    
                    sta MSIGX                       ; VIC($D010) MSB X-Pos Sprites 0-7
                    sta SPMC                        ; VIC($D01C) Sprite Multicolor
                    sta EXTCOL                      ; VIC($D020) Border Color
                    sta BGCOL0                      ; VIC($D021) Background Color 0
                    
                    lda #$01                        ; 
                    sta SPENA                       ; VIC($D015) Sprite Enable Register
                    sta SP0COL                      ; VIC($D027) Color Sprite 0
                    sta XXPAND                      ; VIC($D01D) Sprite Double Height
                    sta YXPAND                      ; VIC($D017) Sprite Scale Double Width
                    
                    lda #PtrSprLogo                 ; logo sprite
                    sta M2_SpritePointer00          ; 
                    
.SetTimer           lda #$04                        ; 
                    sta M2Z_23                      ; 
                    sta M2Z_24                      ; 
                    
.ChkRasterBot       lda RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    cmp #$fa                        ; 
                    bne .ChkRasterBot               ; 
                    
                    jsr CheckJoystick               ; get joystick
                    
                    lda M2Z_JoystickFire            ; pressed_no=$00  pressed=$ff
                    beq .DecTimer                   ; 
                    
                    lda #M2Z_DemoModeOff            ; 
                    sta M2Z_DemoMode                ; 
                    
.SetScreenOffAndX   lda #$0b                        ; ....#.## - Bit3:1=24 rows Bit4:0=screen off bit5:0=bitmap mode off
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    
                    cli                             ; 
.ExitInfo           rts                             ; 
                    
.DecTimer           dec M2Z_23                      ; 
                    bne .SetGfxScreenOn             ; 
                    dec M2Z_24                      ; 
                    beq .SetScreenOffAndX           ; 
                    
.SetGfxScreenOn     lda #$18                        ; ...##... - Bit3:1=40 cols Bit4:1=multicolor text on
                    sta SCROLX                      ; VIC($D016) VIC Control Register 2 (and Horizontal Fine Scrolling)
                    lda #$80                        ; #....... - Bits4-7:8(screen=$2000-$23e7) + $4000 VIC Bank 1
                    sta VMCSB                       ; VIC($D018) VIC Chip Memory Control
                    lda #$3b                        ; ..###.## - Bit3:1=24 rows Bit4:1=screen on bit5:0=bitmap mode on
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    
.WaitRasterTop      lda RASTER                      ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                    cmp #$9a                        ; 
                    bne .WaitRasterTop              ; 
                    
                    lda #$1b                        ; ...##.## - Bit3:1=24 rows Bit4:1=screen on bit5:0=bitmap mode off
                    sta SCROLY                      ; VIC($D011) VIC Control Register 1 (and Vertical Fine Scrolling)
                    lda #$08                        ; ....#... - Bit3:1=40 cols Bit4:1=multicolor text off
                    sta SCROLX                      ; VIC($D016) VIC Control Register 2 (and Horizontal Fine Scrolling)
                    lda #$8c                        ; #... ##. . - Bits4-7:8(screen=$2000-$23e7) Bits1-3:6(chars=$3000-$37ff) + $4000 VIC Bank 1
                    sta VMCSB                       ; VIC($D018) VIC Chip Memory Control
                    
ShowInfoScreenX     jmp .ChkRasterBot               ; 
; -------------------------------------------------------------------------------------------------------------- ;
MinerInfoScreen     include inc\M2_DataInfoScreen.asm
MinerTileLevels     include inc\M2_DataTileLevels.asm
MinerGfxStore       equ *
MinerSprites        include inc\M2_DataSprites.asm
MinerTileItems      include inc\M2_DataTileItems.asm
MinerUnused1        include inc\M2_DataUnused1.asm
MinerTileClear      include inc\M2_DataTileClear.asm
MinerGfxStoreX      equ *
MinerLevel00        include inc\M2_DataLevel00.asm
MinerLevel01        include inc\M2_DataLevel01.asm
MinerLevel02        include inc\M2_DataLevel02.asm
MinerLevel03        include inc\M2_DataLevel03.asm
MinerLevel04        include inc\M2_DataLevel04.asm
MinerLevel05        include inc\M2_DataLevel05.asm
MinerLevel06        include inc\M2_DataLevel06.asm
MinerLevel07        include inc\M2_DataLevel07.asm
MinerLevel08        include inc\M2_DataLevel08.asm
MinerLevel09        include inc\M2_DataLevel09.asm
; -------------------------------------------------------------------------------------------------------------- ;
;MinerUnused2        include inc\M2_DataUnused2.asm
; -------------------------------------------------------------------------------------------------------------- ;
MinerNoises         include inc\M2_Sound.asm
MinerTables         include inc\M2_Tables.asm
; -------------------------------------------------------------------------------------------------------------- ;
