/* Helper functions for using WAV files. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "UberCassette.h"
#include "WAV.h"

struct WAV_ChunkPrototype;


struct WAV_ChunkPrototype
{
	char ID[ 4 ];
	unsigned long dataSize;
};

struct WAV_Header
{
	char ID[ 4 ];
	unsigned long dataSize;
	char RIFFtype[ 4 ];
	struct WAV_ChunkPrototype firstChunk;
};

struct WAV_FormatChunk
{
	char ID[ 4 ];
	unsigned long dataSize;
	unsigned short compressionCode;
	unsigned short numberOfChannels;
	unsigned long sampleRate;
	unsigned long bytesPerSecond;
	unsigned short blockAlign;
	unsigned short significantBitsPerSample;
	unsigned short extraFormatBytes;
};

struct WAV_DataChunk
{
	char ID[ 4 ];
	unsigned long dataSize;
	signed char data[0];
};

struct WAV_Format gWAVFormat;
struct RAW_Format gRAWFormat;

void WAV_ParseBlock_FMT( struct WAV_FormatChunk *chunk )
{
	gWAVFormat.numberOfChannels = CorrectEndianShort( chunk->numberOfChannels );
	gWAVFormat.sampleRate = CorrectEndianLong( chunk->sampleRate );
	gWAVFormat.significantBitsPerSample = CorrectEndianShort( chunk->significantBitsPerSample );
	return;
}

void WAV_ParseBlock_DATA( struct WAV_DataChunk *chunk, bool invert )
{
	int tRawSize, tChannel = 0;
	unsigned long tInputByte;
	signed char *tOutputPtr;
    unsigned long tDataSize;

	// Convert the data to 8-bit signed, 44KHz.
	tRawSize = CorrectEndianLong( chunk->dataSize );
	if ( gWAVFormat.numberOfChannels == 2 )
		tRawSize /= 2;
	if ( gWAVFormat.significantBitsPerSample != 8 )
		tRawSize /= (gWAVFormat.significantBitsPerSample / 8);

	gRAWFormat.size = tRawSize;
	tOutputPtr = gRAWFormat.data = (signed char *)malloc( tRawSize );
    if  (tOutputPtr == NULL )
    {
        printf( "Out of memory allocating WAV data.\n" );
        return;
    }

    tDataSize = CorrectEndianLong( chunk->dataSize );

	for ( tInputByte = 0 ; tInputByte < tDataSize ; tOutputPtr++ )
	{
		if ( tChannel == 1 )
		{
			// Skip the right channel.
			tInputByte += gWAVFormat.significantBitsPerSample / 8;
			continue;
		}

		switch (gWAVFormat.significantBitsPerSample )
		{
		case 8:
			*tOutputPtr = chunk->data[ tInputByte++ ] * (invert ? -1 : 1 );
			break;
		case 16:
			tInputByte++; // Skip the LSB.
			*tOutputPtr = chunk->data[ tInputByte++ ] * (invert ? -1 : 1 );
			break;
		default:
			break;
		}

		tChannel++;
		if ( tChannel > (gWAVFormat.numberOfChannels-1))
			tChannel = 0;
	}

	return;
}

unsigned long WAV_ParseBlock( struct WAV_ChunkPrototype *chunk, bool invert )
{
	if (!strnicmp( ((struct WAV_ChunkPrototype *)chunk)->ID, "fmt ", 4))
	{
		WAV_ParseBlock_FMT( (struct WAV_FormatChunk *)chunk );
	}
	if ( !strnicmp( ((struct WAV_ChunkPrototype *)chunk)->ID, "data", 4 ))
	{
		WAV_ParseBlock_DATA( (struct WAV_DataChunk *)chunk, invert );
	}

	return CorrectEndianLong( chunk->dataSize ) + 8;
}

struct RAW_Format WAV_GetRAW( unsigned char *wavData, bool invert )
{
	struct WAV_Header *tHeader = (struct WAV_Header *)wavData;
	int tFileSize = 0;
	struct WAV_ChunkPrototype *tChunk = NULL;
	unsigned char *tAddress;
	unsigned long tChunkLength;

    printf( "WAV_GetRaw.\n" );

	gRAWFormat.data = NULL;
	gRAWFormat.size = 0;

	if ( strnicmp( (char *)tHeader->ID, "RIFF", 4 ) )
	{
		// Not a valid RIFF file.
		return gRAWFormat;
	}

	if ( strnicmp( (char *)tHeader->RIFFtype, "WAVE", 4 ) )
	{
		// Not a valid WAVE file.
		return gRAWFormat;
	}

	// We have a valid file!
	tFileSize = CorrectEndianLong( tHeader->dataSize ) - 8;

	// Loop through the chunks till we get useful data.
	tChunk = &tHeader->firstChunk;
	tAddress = (unsigned char *)tChunk;
	printf( "tFileSize is %d.\n", tFileSize );

	while ( ((unsigned char *)tChunk - wavData) < tFileSize )
	{
		// Add on the length of the chunk.
		tChunkLength = WAV_ParseBlock( tChunk, invert );
		tAddress += tChunkLength;
		tChunk = (struct WAV_ChunkPrototype *)tAddress;
	}

    if ( gRAWFormat.data == NULL )
        return gRAWFormat;

#if defined(DUMP_RAW)
	{
		FILE *tFile;
        printf( "Dumping RAW file of size %d bytes.\n", gRAWFormat.size );
		tFile = fopen( "temp.raw", "wb" );
		if ( tFile )
		{
			fwrite( gRAWFormat.data, gRAWFormat.size, 1, tFile );
			fclose( tFile );
		}
        printf( "Done.\n" );
	}
#endif

	return gRAWFormat;
}
