; Alien Flash minitools
; ---------------------
;
; (c) 2011 Hannu Nuotio <nojoopa@users.sf.net>
; (based heavily on EasyFlash programs, see README_fw)
;
; This software is provided 'as-is', without any express or implied
; warranty.  In no event will the authors be held liable for any damages
; arising from the use of this software.
;
; Permission is granted to anyone to use this software for any purpose,
; including commercial applications, and to alter it and redistribute it
; freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must not
;    claim that you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation would be
;    appreciated but is not required.
; 2. Altered source versions must be plainly marked as such, and must not be
;    misrepresented as being the original software.
; 3. This notice may not be removed or altered from any source distribution.

; Alternative firmware for the Alien Flash cart.


!initmem $ff
!ct scr ; Screencode

; --- Constants

ROM_start = $8000
EASYFS_start = $b000
EAPI_start = $b800
BOOT_start = $be00


Startup_dest = $0100
keyhandler_dest = $0400
irq_dest = $02a7


; - Alien Flash registers ($de00)

af_reg_bank = $de00
af_reg_bankh = $de01

af_reg_control = $de02
MODE_off = %00000100
MODE_16k = %00000111
MODE_usb = %01000000
MODE_led = %10000000

af_reg_control2 = $de03
MODE_romx = %00000010
MODE_ar_m = %00000100
; protections off
MODE_poff = %01000000


; - Alien Flash RAM ($df00)

Af_addr_stash = $df2b
af_addr_stash_load = Af_addr_stash + 0
af_addr_stash_loadh = Af_addr_stash + 1
af_addr_stash_end = Af_addr_stash + 2
af_addr_stash_endh = Af_addr_stash + 3

; table of format flags: is drive N formatting?
af_formatting = $df7f - 11

; EAPI
; jump table
af_easyapi = $df80
af_eapi_erase = af_easyapi + 3
af_eapi_setbank = af_easyapi + 6
af_eapi_getbank = af_easyapi + 9
af_eapi_setfptr = af_easyapi + 12
af_eapi_setlen = af_easyapi + 15
af_eapi_readf = af_easyapi + 18
af_eapi_writef = af_easyapi + 21

; state variables
af_eapi_bankmode = $dfe0
af_eapi_fptr = $dfd7
af_eapi_fptrh = $dfd8





; banks

; PRG in $010000 ->
PRG_bank = (($10000 / $2000) + Base_bank)

; D64 in $040000 ->
D64_bank = (($40000 / $2000) + Base_bank)

; AR in $870000 - $877fff
; ... ROMH $070000, ROMX on.
AR_bank = (($70000 / $2000) + Base_bank)


; magic values in Alien Flash file header
AFFILE_d64 = $7f
AFFILE_prg = $0e


; - keys

KEY_x = $58


; - kernal functions

k_CLRCHN = $ffcc
k_CHRIN = $ffcf
k_CHKIN = $ffc6
k_CHROUT = $ffd2
k_CHKOUT = $ffc9
k_SETLFS = $ffba        ; a = filenr, x = unitnr, y = sec.addr/command (255|1)
k_SETNAM = $ffbd        ; a = namelen, x:y->name
k_OPEN = $ffc0
k_CLOSE = $ffc3         ; x = filenr
k_SCNKEY = $ff9f
k_GETIN = $ffe4         ; ret: a = 0: no keys pressed, otherwise a = ASCII code


; - hw addresses

Screen = $f800          ; Screen address
color = $d800           ; color ram address
vicii_bank = Screen & $c000
ownfont = $e000


; - colors

Color_back = 0
Color_border = 0
Color_fore = $e

Color_usb = 6
Color_format = 4
Color_d64_from = 9
Color_d64_to = 10


; --- Zero page variables

af_found_romslot = $23
af_found_romsloth = $24

tmp = $25

af_found_file = $26
MASK_found_ar  = %00000001
MASK_found_prg = %00000010
MASK_found_d64 = %00000100

zp_src = $b7
zp_srch = $b8
zp_dst = $ae
zp_dsth = $af

tmpptr = $fd
tmpptrh = $fe


; --- Main

; start of cart ROM
* = ROM_start

; -- screen help text

Cvbar = '5' + 40
Chbar = 'A' - 1
Cubar = 'I' + 40
Cdbar = 'J' + 40

;    |0123456789012345678901234567890123456789|
prgtxt:
!by Cdbar
!fi 28, Chbar
!by Cdbar
!fi 10, Chbar    ; line 0

; line 0
!scr Cvbar, " Alien Flash minitools v"
!src "version.inc"
!scr " ", Cvbar, " q : quit " ; line 1

!by Cubar
!fi 18, Chbar
!by Cdbar
!fi 9, Chbar
!by Cubar
!fi 10, Chbar    ; line 2

;    |0123456789012345678
;     901234567890123456789|
; line 3
!scr "Boot               ", Cvbar
!scr "Transfer            "

; line 4
!scr "0-f: ROM SLOT 0-15 ", Cvbar
!scr " u : USB mode       "

; line 5
!scr " z : Action Replay ", Cvbar
!scr " w : drive _ D64    "

; line 6
!scr " p : PRG in Flash  ", Cvbar
!scr " r : D64 _ drive    "

; line 7
!scr "                   ", Cvbar
!scr " j : format drive   "

; line 8
!scr "                   ", Cvbar
textloc_drivenum = * - prgtxt + Screen + 12
!scr "n,m: drive #xx      "

; line 9
!scr "                   ", Cvbar
!scr " l : PRG _ RAM      "

; line 10
!scr "                   ", Cvbar
!scr "                    "

; line 11
!scr "                   ", Cvbar
!scr "                    "

; line 12
!scr "                   ", Cvbar
!scr "                    "

!fi 19, Chbar
!by Cubar
!fi 20, Chbar   ; line 13

textloc_romslotinfo = * - prgtxt + Screen + 5
textloc_have_ar = * - prgtxt + Screen + 39
!scr "ROM: xxxxxxxxxxxxxxxx              AR: x" ; 14
textloc_fileinfo = * - prgtxt + Screen + 5
!scr "D64: 0123456789abcdef,xx                " ; 15
!scr "PRG: 0123456789abcdef load $xxxx - $xxxx" ; 16
!fi 40, Chbar   ; 17
textloc_statusinfo = * - prgtxt + Screen
!scr "Welcome.                "
textloc_usbdump = * - prgtxt + Screen
!scr                         "                " ; 18
!scr "                                        " ; 19
!scr "Remember that only the key x works while" ; 20
!scr "USB mode is on. Don't exit while in the " ; 21
!scr "middle of a transfer.  Good luck.       " ; 22
!scr "                                        " ; 23
!scr "                                        " ; 24

Prgtxt_no_base:
prgtxt_no_prg_offset = * - Prgtxt_no_base
!scr "No PRG in Flash!", 0

prgtxt_no_d64_offset = * - Prgtxt_no_base
!scr "No D64 in Flash!", 0

prgtxt_flash_error_offset = * - Prgtxt_no_base
!scr "Flash error!", 0

prgtxt_drive_error_offset = * - Prgtxt_no_base
!scr "Drive error $xx!", 0

prgtxt_no_ar_offset = * - Prgtxt_no_base
!scr "Action Replay not found!",0

prgtxt_usb_on_offset = * - Prgtxt_no_base
!scr "USB mode (x to exit) BD:",0

prgtxt_ram_prg_offset = * - Prgtxt_no_base
!scr "No PRG in RAM!", 0

prgtxt_wait_offset = * - Prgtxt_no_base
!scr "Waiting...", 0


; -- IRQ handler
;
main_irq:
!pseudopc irq_dest {
irq_normal:
    asl $d019
    jmp $ea31     ; return to the auxiliary raster interrupt

irq_killed:
-   inc $d020
    inc Screen+40
    jmp -
    asl $d019
    bit $dc00
    rti
} ; !pseudopc irq_dest
main_irq_end:
LEN_main_irq = main_irq_end - main_irq


; -- main entrypoint

main_entrypoint:
; - adapted from spritemcbase.asm
    jsr $fda3

; clone_fd50
    lda #0
    tay
-   sta $0002,y
    sta $0200,y
    sta $0300,y
    iny
    bne -

    ldx #$03
    lda #$3c
    sta $b2
    stx $b3
    jsr $fd8c   ;Set MemBounds

; clone_fd15
    ldy #$1f
-   lda $fd30,y
    sta $0314,y
    dey
    bpl -

; clone_ff5b
    lda #3
    sta $9a
    lda #0
    sta $99
    jsr $e51b
; - end spritemcbase.asm stealing

    lda #8
    sta $ba
    lda #64
    sta $c5
    lda #0
    sta $c6

    ; copy screen to destination
    ldy #0
-   lda prgtxt,y
    sta Screen,y
    lda prgtxt + 0x100,y
    sta Screen + 0x100,y
    lda prgtxt + 0x200,y
    sta Screen + 0x200,y
    lda prgtxt + 0x300,y
    sta Screen + 0x300,y
    iny
    bne -

    ; setup VICII
    ; set up bank & $d018
    lda $dd00
    and #$fc
    ora #((vicii_bank >> 14) XOR 3)
    sta $dd00
    lda #((>(Screen & $3fff)) << 2) | ((>(ownfont & $3800)) >> 2)
    sta $d018
    lda #$c8
    sta $d016
    lda #Color_back
    sta $d021
    lda #Color_fore
    sta 646

    ; we're not formatting on any drive (hopefully)
    lda #0
    ldy #3
-   sta af_formatting + 8,y
    dey
    bpl -

initints:
    sei

    ; setup IRQ line and screen
    lda #$1b
    sta $d011     ; set the raster interrupt location
    lda #$12
    sta $d012

    ; copy our IRQ handlers to RAM
    ldy #(LEN_main_irq - 1)
-   lda main_irq,y
    sta irq_dest,y
    dey
    bpl -

    ; point to our normal IRQ handler
    lda #<irq_normal
    sta $0314
    lda #>irq_normal
    sta $0315

    lda #$7f
    sta $dc0d   ; disable timer interrupts
    sta $dd0d
    lda $dc0d
    lda $dc0d   ; ack timer interrupts
    ldx #1
    stx $d01a   ; enable raster interrupt
    lsr $d019   ; ack video interrupts
    cli

initmain:
    ; (re)set border color
    lda #Color_border
    sta $d020

    ; (re)set background color
    lda #Color_back
    sta $d021

    ; detect rom slots, d64, prg, ar
    ; assume nothing will be found
    lda #0
    sta af_found_file
    sta af_found_romslot
    sta af_found_romsloth

    ; copy the detecter to RAM
    ldy #key_index_detect_offset
    jsr copy_handler_to_ram

detect_d64_file:
    ; a = bank number
    lda #D64_bank
    ; type
    ldy #AFFILE_d64
    ; offset to print location
    ldx #0
    jsr keyhandler_dest
    ; store "found" bit C
    rol af_found_file

detect_prg_file:
    ; a = bank number
    lda #PRG_bank
    ; type
    ldy #AFFILE_prg
    ; offset to print location
    ldx #40
    jsr keyhandler_dest
    ; store "found" bit C
    rol af_found_file

detect_ar_file:
    ; AR bank, check $a000
    lda #AR_bank
    ldy #0
    ldx #$a0
    jsr detect_ar_file_handler
    ; store "found" bit C
    rol af_found_file
    ; show info
    ldx #'n'
    lda #(MASK_found_ar)
    and af_found_file
    beq +
    ldx #'y'
+
    stx textloc_have_ar

detect_rom_slot:
    ldx #0
detect_rom_slot_loop:
    lda rom_slot_bankh,x
    tay
    lda rom_slot_bank,x
    jsr detect_rom_vec_handler
    ror af_found_romsloth
    ror af_found_romslot
    lda #'.'
    bit af_found_romsloth
    bmi +
    lda hex_lut,x
+   sta textloc_romslotinfo,x
    inx
    cpx #$10
    bne detect_rom_slot_loop
    jmp show_drivenum


show_drivenum:
    ; print drive number
    lda #<textloc_drivenum
    sta tmpptr
    lda #>textloc_drivenum
    sta tmpptrh
    lda $ba
    ldy #0
    jsr printdecimal


menuwait:
    ; wait for key
    jsr waitkey

    ; erase statustext
    jsr erasestatus

DEBUG_SHOWKEY = 0
!if DEBUG_SHOWKEY = 1 {
    ; display keycode
    tax
    ldy #<Screen
    sty tmpptr
    ldy #>Screen
    sty tmpptrh
    jsr printhex
    txa
}

menu_find_key:
    ; find index to match key
    ldy #(-1)
-   iny
    cmp key_index_table,y
    bcc menuwait    ; table value was smaller, abort
    bne -

    ; found index (in y)
menu_found_key:
    ; copy the handler to RAM
    jsr copy_handler_to_ram

    ; check if we need to kill interrupts
    lda key_handler_table_kill_ints,y
    beq +

    ; kill interrupts
    sei

    ; point to nop IRQ handler
    lda #<irq_killed ;$fd30
    sta $0314
    lda #>irq_killed ;$fd31
    sta $0315

    ; disable raster interrupt
    lda #0
    sta $d01a
    lsr $d019   ; ack video interrupts
+

    ; jump to handler
    jmp keyhandler_dest


; - possible key values
; key is searched for and the index (if found) is
; used to index the key handler table.
; must be in increasing order! (except the end marker)
; commands 0-f must be placed in order!
;
key_index_table:
;   run/stop
!by $03
key_index_romslot_offset = * - key_index_table
;    0    1    2    3    4    5    6    7    8    9
!by $30, $31, $32, $33, $34, $35, $36, $37, $38, $39
;    a    b    c    d    e    f
!by $41, $42, $43, $44, $45, $46
;    j    l    m    n
!by $4a, $4c, $4d, $4e
;    p    q    r    u    w    z
!by $50, $51, $52, $55, $57, $5a
;  (end of list)
key_index_detect_offset = * - key_index_table
!by $00 ; used as fake index for detect_af_file)


; - keypress handler addresses
; same order as above table
; handlers get index in y
;
key_handler_table_start:
!wo quit_to_basic       ; run/stop
!wo boot_rom_slot       ; 0
!wo boot_rom_slot       ; 1
!wo boot_rom_slot       ; 2
!wo boot_rom_slot       ; 3
!wo boot_rom_slot       ; 4
!wo boot_rom_slot       ; 5
!wo boot_rom_slot       ; 6
!wo boot_rom_slot       ; 7
!wo boot_rom_slot       ; 8
!wo boot_rom_slot       ; 9
!wo boot_rom_slot       ; a
!wo boot_rom_slot       ; b
!wo boot_rom_slot       ; c
!wo boot_rom_slot       ; d
!wo boot_rom_slot       ; e
!wo boot_rom_slot       ; f
!wo drive_format        ; j
!wo prg_from_ram        ; l
!wo drivenum_inc        ; m
!wo drivenum_dec        ; n
!wo boot_prg            ; p
!wo quit_to_basic       ; q
!wo d64_from_drive      ; r
!wo usb_mode            ; u
!wo d64_to_drive        ; w
!wo boot_ar             ; z
!wo detect_af_file      ; (fake entry)

key_handler_table_end:
!wo quit_to_basic_end   ; run/stop
!wo boot_rom_slot_end   ; 0
!wo boot_rom_slot_end   ; 1
!wo boot_rom_slot_end   ; 2
!wo boot_rom_slot_end   ; 3
!wo boot_rom_slot_end   ; 4
!wo boot_rom_slot_end   ; 5
!wo boot_rom_slot_end   ; 6
!wo boot_rom_slot_end   ; 7
!wo boot_rom_slot_end   ; 8
!wo boot_rom_slot_end   ; 9
!wo boot_rom_slot_end   ; a
!wo boot_rom_slot_end   ; b
!wo boot_rom_slot_end   ; c
!wo boot_rom_slot_end   ; d
!wo boot_rom_slot_end   ; e
!wo boot_rom_slot_end   ; f
!wo drive_format_end    ; j
!wo prg_from_ram_end    ; l
!wo drivenum_inc_end    ; m
!wo drivenum_dec_end    ; n
!wo boot_prg_end        ; p
!wo quit_to_basic_end   ; q
!wo d64_from_drive_end  ; r
!wo usb_mode_end        ; u
!wo d64_to_drive_end    ; w
!wo boot_ar_end         ; z
!wo detect_af_file_end  ; (fake entry)

key_handler_table_kill_ints:
!by $ff ; quit_to_basic       ; run/stop
!by $ff ; boot_rom_slot       ; 0
!by $ff ; boot_rom_slot       ; 1
!by $ff ; boot_rom_slot       ; 2
!by $ff ; boot_rom_slot       ; 3
!by $ff ; boot_rom_slot       ; 4
!by $ff ; boot_rom_slot       ; 5
!by $ff ; boot_rom_slot       ; 6
!by $ff ; boot_rom_slot       ; 7
!by $ff ; boot_rom_slot       ; 8
!by $ff ; boot_rom_slot       ; 9
!by $ff ; boot_rom_slot       ; a
!by $ff ; boot_rom_slot       ; b
!by $ff ; boot_rom_slot       ; c
!by $ff ; boot_rom_slot       ; d
!by $ff ; boot_rom_slot       ; e
!by $ff ; boot_rom_slot       ; f
!by $ff ; drive_format        ; j
!by $ff ; prg_from_ram        ; l
!by $00 ; drivenum_inc        ; m
!by $00 ; drivenum_dec        ; n
!by $ff ; boot_prg            ; p
!by $ff ; quit_to_basic       ; q
!by $ff ; d64_from_drive      ; r
!by $00 ; usb_mode            ; u
!by $ff ; d64_to_drive        ; w
!by $ff ; boot_ar             ; z
!by $ff ; detect_af_file      ; (fake entry)

; -- Key handlers
; copied to keyhandler_dest (in RAM)

; - quit_to_basic
;
quit_to_basic:
!pseudopc keyhandler_dest {
    lda #MODE_off
    sta af_reg_control
    jmp ($fffc) ; reset
} ; pseudopc keyhandler_dest
quit_to_basic_end:


; - boot_rom_slot
; y = ROM SLOT number
;
boot_rom_slot:
!pseudopc keyhandler_dest {
    ; add a small delay
    jsr smalldelay

    ; turn off Axx protection
    lda #(MODE_poff)
    sta af_reg_control2

    ; y = ROM SLOT + key_index_romslot_offset
    tya
    sec
    sbc #key_index_romslot_offset
    tax
    lda rom_slot_bankh,x
    tay
    lda rom_slot_bank,x
    sta af_reg_bank
    sty af_reg_bankh

    ; set default mode
    lda #0
    sta af_reg_control
    ; turn on A22-A19 protection
    sta af_reg_control2
    ; reset to default VICII bank
    lda #$3f
    sta $dd00
    ; jump to reset
    jmp ($fffc)
} ; pseudopc keyhandler_dest
boot_rom_slot_end:


; - ar_mode_start
;
boot_ar:
!pseudopc keyhandler_dest {
    ; check if there's an AR cart to start
    lda af_found_file
    and #(MASK_found_ar)
    bne +
    ; offset for error message
    ldy #prgtxt_no_ar_offset
    jmp show_found_error
+
    ; got an AR, now boot it.

    ; ...but insert some delay as AR likes
    ; to boot to freeze mode if a key is held
    ; during the reset sequence
    jsr smalldelay

    ; set correct bank
    lda #AR_bank
    sta af_reg_bank
    ; set cart off (ultimax in AR mode)
    lda #(MODE_off)
    sta af_reg_control
    ; set AR mode
    lda #(MODE_ar_m | MODE_romx)
    sta af_reg_control2
    ; jump to reset
    jmp ($fffc)
} ; pseudopc keyhandler_dest
boot_ar_end:


; - usb_mode
;
usb_mode:
!pseudopc keyhandler_dest {
    ; print informative text
    ldy #prgtxt_usb_on_offset
    jsr printstatus

    ; set the background color
    lda #Color_usb
    sta $d021

    ; activate USB mode with (disabled) ROM on $8000->$bfff
    lda #(MODE_16k | MODE_usb | MODE_led)
    sta af_reg_control

usb_mode_loop:
usb_mode_y_smc = * + 1
    ldy #0
    ; load value from disabled ROM -> bus value
    lda $8000
    ; store it in plain sight
    sta textloc_usbdump,y
    ; invert it so inactive is black
    eor #$ff
    ; ... and set the border color
    sta $d020
    ; next print location
    inc usb_mode_y_smc
    ; check for key 'x'
    jsr k_GETIN     ; read key (if any)
    cmp #KEY_x
    bne usb_mode_loop

    ; TODO check if _usbdump is all $ff?

    ; disable USB mode
    lda #MODE_16k
    sta af_reg_control

    ; reset bank after USB has messed with it
    lda #Base_bank
    sta af_reg_bank
    lda #0
    sta af_reg_bankh

    ; erase statustext
    jsr erasestatus

    jmp initints
} ; pseudopc keyhandler_dest
usb_mode_end:


; - prg_from_ram
;
prg_from_ram:
!pseudopc keyhandler_dest {
eapi_prg_from_ram = * + $100
    jmp prg_from_ram_code

; - Alien Flash File signature
; we build the PRG entry here...
af_prg_header: ; "RRBY64AF"
!by $52, $52, $42, $59, $36, $34, $41, $46
; af_file_sig_type:
!by AFFILE_prg
!fi 15, $ff
; file size
af_prg_header_size:
!by $00
af_prg_header_sizeh:
!by $00
!by $00, $00
!fi 4, $ff
; name
;     0123456789abcdef
!scr "PRG from RAM    "
LEN_af_prg_header = * - af_prg_header

prg_from_ram_flash_start:
    ; copy EAPI
    ldy #0
-   lda EAPI_start,y
    sta eapi_prg_from_ram,y
    lda EAPI_start + $100,y
    sta eapi_prg_from_ram + $100,y
    lda EAPI_start + $200,y
    sta eapi_prg_from_ram + $200,y
    iny
    bne -

    ; flash init
    jsr eapi_prg_from_ram + 20  ; EAPIInit
    ; really should check C here...

    ; erase sector $010000 - $020000
    lda #PRG_bank
    ldy #$80    ; ROML
    jsr af_eapi_erase   ; EAPIEraseSector
    bcc +
    jmp prg_from_ram_flash_error
+

    ; set bank (needed?)
    lda #PRG_bank
    jsr af_eapi_setbank ; EAPISetBank

    ; set pointer
    lda #$b0            ; bank change mode: (ROM)L only
    ldx #<($8000)
    ldy #>($8000)       ; point to on-flash header
    jsr af_eapi_setfptr ; EAPISetPtr

    ; flash header
    ldy #0
-   lda af_prg_header,y
    jsr prg_from_ram_flash_byte
    iny
    cpy #LEN_af_prg_header
    bne -

    ; point to start of prg data
    lda #$b0            ; bank change mode: (ROM)L only
    ldx #<($9100)
    ldy #>($8100)       ; point to on-flash header
    jsr af_eapi_setfptr ; EAPISetPtr

    ; flash load addr
    lda zp_src
    jsr prg_from_ram_flash_byte
    lda zp_srch
    jsr prg_from_ram_flash_byte

    ; flash data
    ldy #0
-   lda (zp_src),y
    sta $d020
    jsr prg_from_ram_flash_byte
    inc zp_src
    bne +
    inc zp_srch
+
prg_from_ram_endah = * + 1
    lda #$12
    cmp zp_srch
    bne -
prg_from_ram_enda = * + 1
    lda #$12
    cmp zp_src
    bne -

    ; right bank
    lda #Base_bank
    sta af_reg_bank

    ; set cart to 16k
    lda #MODE_16k
    sta af_reg_control

    ; get back to menu
    jmp initints

prg_from_ram_flash_byte:
    sta $d020
    jsr af_eapi_writef  ; EAPIWriteFlashInc
    bcc +
    ; error occurred
    ; unwind stack
    pla
    pla
prg_from_ram_flash_error:
    ; offset for error message
    ldy #prgtxt_flash_error_offset    ; assumed to be != 0
    bne prg_from_ram_error_jmp
+
    rts

prg_from_ram_error:
    ; offset for error message
    ldy #prgtxt_ram_prg_offset
prg_from_ram_error_jmp:
    jmp show_found_error

!if * > eapi_prg_from_ram {
!error "prg_from_ram code over EAPI!"
}


prg_from_ram_code:
    ; detect PRG end
    lda af_addr_stash_endh
    cmp #$d0
    sta prg_from_ram_endah
    bcs prg_from_ram_error ; addr >= $d0xx not supported
    lda af_addr_stash_end
    sta prg_from_ram_enda

    ; detect PRG start
    lda af_addr_stash_loadh
    cmp #$08
    bcc prg_from_ram_error ; addr < $08xx not supported
    sta zp_srch
    cmp #$d0
    bcs prg_from_ram_error ; addr >= $d0xx not supported
    lda af_addr_stash_load
    sta zp_src

    ; calculate size for the PRG header
    ldy af_addr_stash_endh
    lda af_addr_stash_end
    ; add the size of the load addr
    clc
    adc #2
    bcc +
    iny
+   sec
    sbc af_addr_stash_load
    sta af_prg_header_size
    tya
    sbc af_addr_stash_loadh
    sta af_prg_header_sizeh

    ; check for too large/small program
    cmp #(>($d000 - $0800) + 1)
    bcs prg_from_ram_error ; end < start or otherwise too long
    cmp #0
    bne +
    lda af_prg_header_size
    cmp #24
    bcc prg_from_ram_error ; len < 24 bytes, probably no program
+   jmp prg_from_ram_flash_start

} ; pseudopc keyhandler_dest
prg_from_ram_end:


; - d64_to_drive
;
d64_to_drive:
!pseudopc keyhandler_dest {
    jmp d64_to_drive_start

; time:  0:40:00 .. 0:49:22 ->  9:22

; drive commands
drive_cmd_ux:
!pet "u2 2 0 x x", $0d, 0, 0, 0, 0 ,0

drive_cmd_ch:
!pet "#"

; variables
drive_track:
!by 1
drive_sector:
!by 0
drive_sector_max:
!by 0
d64_bank:
!by D64_bank

drive_cmd_bp:
!pet "b-p 2 0"
Drive_cmd_bp_len = * - drive_cmd_bp

d64_to_drive_buf = $5000

d64_to_drive_start:
    ; check and wait if we're formatting
    jsr drive_format_wait
    ; check if there's any D64 to copy
    lda af_found_file
    and #(MASK_found_d64)
    bne +
    ; offset for error message
    ldy #prgtxt_no_d64_offset
    jmp show_found_error

    ; got a d64, now write it.

    ; set the background color
    lda #Color_d64_from
    sta $d021

    ; drive init
    jsr d64_to_drive_kernal_init
    bcc +
    ; error
    jsr printdriveerror
    jmp initints
+

d64_to_drive_loop:
    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

    ; set max sector
    ldy drive_track
    lda disk_track_to_max_sector,y
    sta drive_sector_max
    tax

    ; fill buffer with track data
    lda #>d64_to_drive_buf
    sta d64_to_drive_data_desth
--  ldy #0
    lda d64_bank
    sta af_reg_bank
d64_to_drive_data_srch = * + 2
-   lda $8100,y
d64_to_drive_data_desth = * + 2
    sta d64_to_drive_buf,y
    iny
    bne -
    inc d64_to_drive_data_desth
    ldy d64_to_drive_data_srch
    iny
    cpy #$a0
    bne +
    inc d64_bank
    ldy #$80
+   sty d64_to_drive_data_srch
    dex
    bne --

    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

    ; write track to drive
    jsr d64_to_drive_kernal_write_track
    bcc +
    ; error
    jsr printdriveerror
    ; shutdown drive
    jsr d64_to_drive_kernal_shutdown
    jmp initints
+
    ; next track
    lda #0
    sta drive_sector
    iny
    sty drive_track
    cpy #36
    beq d64_to_drive_done
    jmp d64_to_drive_loop

d64_to_drive_done:
    ; shutdown drive
    jsr d64_to_drive_kernal_shutdown
    bcc +
    ; error
    jsr printdriveerror
    ; shutdown drive
    jsr d64_to_drive_kernal_shutdown
    jmp initints
+
    ; erase statustext
    jsr erasestatus
    jmp initints




!align $f, 0

; - d64_to_drive_kernal jump table
;
d64_to_drive_kernal_init:
    jmp d64_to_drive_kernal_init_start
d64_to_drive_kernal_shutdown:
    jmp d64_to_drive_kernal_shutdown_start


; - d64_to_drive_kernal_write_track
; parameters:
;  drive_track = track to write
;  drive_sector_max = number of sectors on track
;  d64_to_drive_buf = data buffer to read from
; returns:
;  C = 1 -> error, A = error code, drive_sector = sector where error happened
;  C = 0 -> OK
;
d64_to_drive_kernal_write_track:
    ; reset buffer pointer
    lda #>d64_to_drive_buf
    sta d64_to_drive_src_smc

d64_to_drive_kernal_write_track_loop:
    ; build command string info
    ldx drive_track
    lda drive_sector
    ldy #7
    jsr printtracksector

    ; show it
    jsr printhandlerstatus

    ; stash the string length
    sty d64_to_drive_cmdlen_smc

    ; open channels

    ; open 2,device,2,"#"
    ldx #<drive_cmd_ch
    ldy #>drive_cmd_ch
    lda #1              ; len("#")
    jsr k_SETNAM
    lda #2              ; filenr
    ldx $ba             ; unitnr
    ldy #2              ; sec.address (,2)
    jsr k_SETLFS
    jsr k_OPEN          ; OPEN 2,device,2,"#"
    bcc +
    ; TODO read error channel
    rts
+

    ; open 15,device,15,"b-p 2 0"
    ldx #<drive_cmd_bp
    ldy #>drive_cmd_bp
    lda #Drive_cmd_bp_len
    jsr k_SETNAM
    lda #15             ; filenr
    ldx $ba             ; unitnr
    ldy #15             ; sec.address (,15)
    jsr k_SETLFS
    jsr k_OPEN          ; OPEN 15,device,15,"b-p 2 0"
    bcc +
    ; TODO read error channel
    rts
+

    ; set output device to 2
    ldx #2
    jsr k_CHKOUT

    ; get and send bytes
    ldy #0
d64_to_drive_src_smc = * + 2
-   lda d64_to_drive_buf,y
    sta $d020
    jsr k_CHROUT
    iny
    bne -

    ; set output device to 15
    ldx #15
    jsr k_CHKOUT

    ; send "u2 ..."
    ldy #0
-   lda drive_cmd_ux,y
    jsr k_CHROUT
    iny
d64_to_drive_cmdlen_smc = * + 1
    cpy #$12
    bne -

    ; close
    jsr k_CLRCHN
    lda #15
    jsr k_CLOSE
    lda #2
    jsr k_CLOSE

    ; next sector
    inc d64_to_drive_src_smc
    inc drive_sector
    ldy drive_track
    lda drive_sector_max
    cmp drive_sector
    beq +
    jmp d64_to_drive_kernal_write_track_loop
+   clc
    rts


; - d64_to_drive_kernal_init
; returns:
;  C = 1 -> error, A = error code
;  C = 0 -> OK
;
d64_to_drive_kernal_init_start:
    ; TODO
    rts

; - d64_to_drive_kernal_shutdown
; returns:
;  C = 1 -> error, A = error code
;  C = 0 -> OK
;
d64_to_drive_kernal_shutdown_start:
    ; TODO
    ; set output device to 0
    ldx #0
    jsr k_CHKOUT
    clc
    rts

} ; pseudopc keyhandler_dest
d64_to_drive_end:



; - d64_from_drive
;
d64_from_drive:
!pseudopc keyhandler_dest {
    jmp d64_from_drive_start

; time:  0:12:00 .. 0:21:22 ->  9:22
;        0:28:20 .. 0:38:48 -> 10:28

; drive commands
drive_cmd_ux:
!pet "u1 2 0 x x", $0d, 0, 0, 0, 0 ,0

drive_cmd_ch:
!pet "#"

; variables
drive_track:
!by 1
drive_sector:
!by 0
drive_sector_max:
!by 0
d64_bank:
!by D64_bank

eapi_d64_from_drive = $7d00
d64_from_drive_buf = $5000

d64_from_drive_start:
    ; check and wait if we're formatting
    jsr drive_format_wait
    ; no real sense to continue if we were, but let's do it anyway

    ; set the background color
    lda #Color_d64_from
    sta $d021

    ; drive init
    jsr d64_from_drive_kernal_init
    bcc d64_from_drive_init_ok
    ; error
    jsr printdriveerror
    jmp initints

d64_from_drive_init_ok:
    ; copy EAPI
    ldy #0
-   lda EAPI_start,y
    sta eapi_d64_from_drive,y
    lda EAPI_start + $100,y
    sta eapi_d64_from_drive + $100,y
    lda EAPI_start + $200,y
    sta eapi_d64_from_drive + $200,y
    iny
    bne -

    ; flash init
    jsr eapi_d64_from_drive + 20  ; EAPIInit
    ; really should check C here...

    ; erase $040000 - $06ffff
    lda #D64_bank
    ldy #$80    ; ROML
-   jsr af_eapi_erase   ; EAPIEraseSector
    bcc +
    jmp d64_from_drive_flash_error
+   adc #1
    cmp #(D64_bank + ($30000 / $2000))
    bne -

    ; set bank (needed?)
    lda #D64_bank
    sta d64_bank
    jsr af_eapi_setbank ; EAPISetBank

    ; set pointer
    lda #$b0            ; bank change mode: (ROM)L only
    ldx #<($8100)
    ldy #>($8100)       ; point to on-flash data
    jsr af_eapi_setfptr ; EAPISetPtr

d64_from_drive_loop:
    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

    ; set max sector
    ldy drive_track
    lda disk_track_to_max_sector,y
    sta drive_sector_max

    ; read track to RAM
    jsr d64_from_drive_kernal_read_track
    bcc d64_from_drive_flash_bytes
    ; error
    jsr printdriveerror
    ; shutdown drive
    jsr d64_from_drive_kernal_shutdown
    jmp initints

d64_from_drive_flash_bytes:
    sei

    ; flash bytes
    lda #>d64_from_drive_buf
    sta d64_from_drive_flash_bytes_srch
    ldx drive_sector_max
--  ldy #0
-
d64_from_drive_flash_bytes_srch = * + 2
    lda d64_from_drive_buf,y
    jsr d64_from_drive_flash_byte
    iny
    bne -
    inc d64_from_drive_flash_bytes_srch
    dex
    bne --

    ; reset the border color
    lda #Color_border
    sta $d020

    ; next track
    ldy drive_track
    iny
    sty drive_track
    cpy #36 ; Drive_track_num
    beq +
    jmp d64_from_drive_loop
+

d64_from_drive_got_all_sectors:
    ; all sectors read

    ; get diskname
    ; set bank
    lda #(D64_bank + ($16000 / $2000))
    sta af_reg_bank

    ldy #15
    ; diskname offset 0x16590 ( + header)
-   lda $8690,y
    sta af_d64_header + $20,y
    dey
    bpl -

    ; pad name with ' ' instead of $a0
    ldy #15
    ; diskname offset 0x16590 ( + header)
-   lda af_d64_header + $20,y
    cmp #$a0
    bne +
    lda #' '
    sta af_d64_header + $20,y
    dey
    bpl -
+

    ; set bank
    lda #D64_bank
    jsr af_easyapi + 6  ; EAPISetBank

    ; program header
    ; set pointer
    lda #$b0            ; bank change mode: (ROM)L only
    ldx #<($8000)
    ldy #>($8000)       ; point to on-flash header
    jsr af_eapi_setfptr ; EAPISetPtr

    ; flash header
    ldy #0
-   lda af_d64_header,y
    jsr d64_from_drive_flash_byte
    iny
    cpy #LEN_af_d64_header
    bne -

    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

    ; drive shutdown
    jsr d64_from_drive_kernal_shutdown
    bcc d64_from_drive_done
    ; error
    jsr printdriveerror
    jmp initints

d64_from_drive_done:
    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

    ; erase statustext
    jsr erasestatus
    jmp initints

d64_from_drive_flash_byte:
    sta $d020
    jsr af_eapi_writef  ; EAPIWriteFlashInc
    bcc +
    ; error occurred
    ; unwind stack
    pla
    pla
d64_from_drive_flash_error:
    ; shutdown drive
    jsr d64_from_drive_kernal_shutdown
    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank
    ; offset for error message
    ldy #prgtxt_flash_error_offset  ; assumed to be != 0
    bne d64_from_drive_error_jmp
+
    rts

d64_from_drive_no_d64_error:
    ; offset for error message
    ldy #prgtxt_no_d64_offset
d64_from_drive_error_jmp:
    jmp show_found_error

; - Alien Flash File signature
; we build the D64 entry here...
af_d64_header: ; "RRBY64AF"
!by $52, $52, $42, $59, $36, $34, $41, $46
; af_file_sig_type:
!by AFFILE_d64
!fi 15, $ff
; file size
!by $00, $ab, $02   ; "normal" d64 size, 174848 B
!by $00
!fi 4, $ff
; name
;     0123456789abcdef
!fi 16, $ff
LEN_af_d64_header = * - af_d64_header


!align $f, 0

; - d64_from_drive_kernal jump table
;
d64_from_drive_kernal_init:
    jmp d64_from_drive_kernal_init_start
d64_from_drive_kernal_shutdown:
    jmp d64_from_drive_kernal_shutdown_start

; - d64_from_drive_kernal_read_track
; parameters:
;  drive_track = track to read
;  drive_sector_max = number of sectors on track
;  (d64_from_drive_buf = data buffer to read to)
; returns:
;  data_buf = track data
;  C = 1 -> error, A = error code, drive_sector = sector where error happened
;  C = 0 -> OK
;
d64_from_drive_kernal_read_track:
    ; set destination addr MSB
    lda #>d64_from_drive_buf
    sta d64_from_drive_kernal_read_track_dest

    ; reset sector number
    lda #0
    sta drive_sector

    ; set bank to normal
    lda #Base_bank
    sta af_reg_bank

d64_from_drive_kernal_read_track_loop:
    ; build command string info
    ldx drive_track
    lda drive_sector
    ldy #7
    jsr printtracksector

    ; stash the string length
    sty d64_from_drive_kernal_cmdlen_smc

    ; show it
    jsr printhandlerstatus

    ; open channels
    ; open 2,device,2,"#"
    ldx #<drive_cmd_ch
    ldy #>drive_cmd_ch
    lda #1              ; len("#")
    jsr k_SETNAM
    lda #2              ; filenr
    ldx $ba             ; unitnr
    ldy #2              ; sec.address (,2)
    jsr k_SETLFS
    jsr k_OPEN          ; OPEN 2,device,2,"#"
    bcc +
    ; TODO read error channel
    sei
-   jmp -
    rts
+
    ; open 15,device,15,"u1 2 0 ..."
    ldx #<drive_cmd_ux
    ldy #>drive_cmd_ux
d64_from_drive_kernal_cmdlen_smc = * + 1
    lda #$12
    jsr k_SETNAM
    lda #15             ; filenr
    ldx $ba             ; unitnr
    ldy #15             ; sec.address (,15)
    jsr k_SETLFS
    jsr k_OPEN          ; OPEN 15,device,15,"u1 2 0 ..."
    bcc +
    ; TODO read error channel
    sei
-   jmp -
    rts
+

    ; set input device to 2
    ldx #2
    jsr k_CHKIN

    ; read sector
    ldy #0
-   jsr k_CHRIN
d64_from_drive_kernal_read_track_dest = * + 2
    sta d64_from_drive_buf,y
    sta textloc_usbdump,y
    iny
    bne -
    inc d64_from_drive_kernal_read_track_dest

d64_from_drive_kernal_got_sector:

    ; close files
    lda #15
    jsr k_CLOSE
    lda #2
    jsr k_CLOSE

    ; next sector
    ldy drive_sector
    iny
    cpy drive_sector_max
    beq +
    sty drive_sector
    jmp d64_from_drive_kernal_read_track_loop
+
    clc
    rts


; - d64_from_drive_kernal_init
; returns:
;  C = 1 -> error, A = error code
;  C = 0 -> OK
;
d64_from_drive_kernal_init_start:
    ; TODO
    clc
    rts

; - d64_from_drive_kernal_shutdown
; returns:
;  C = 1 -> error, A = error code
;  C = 0 -> OK
;
d64_from_drive_kernal_shutdown_start:
    ; TODO
    ; set input device to 0
    ldx #0
    jsr k_CHKIN
    clc
    rts


!if * > d64_from_drive_buf {
!error "from_drive goes over buffer!"
}

} ; pseudopc keyhandler_dest
d64_from_drive_end:


; - drivenum_dec
;
drivenum_dec:
!pseudopc keyhandler_dest {
    ldy $ba
    dey
    cpy #8
    bcc +
    sty $ba
+   jmp show_drivenum
} ; pseudopc keyhandler_dest
drivenum_dec_end:


; - drivenum_inc
;
drivenum_inc:
!pseudopc keyhandler_dest {
    ldy $ba
    iny
    cpy #12
    bcs +
    sty $ba
+   jmp show_drivenum
} ; pseudopc keyhandler_dest
drivenum_inc_end:


; - drive_format
;
drive_format:
!pseudopc keyhandler_dest {
    jmp drive_format_begin

; drive format string
drive_format_string:
!pet "n:"
drive_format_string_name:
!pet "afminitools,12"
drive_format_default_name_len = * - drive_format_string_name
; enough padding for full name + ID
!fi 10, 0

drive_format_begin:
    lda #Color_format
    sta $d020

    ; check and wait if we're already formatting
    jsr drive_format_wait
    bcc +
    ; was already formatting, no need to again
    jmp initints
+
    ldy #drive_format_default_name_len
    ; check if there's any D64 for the diskname & ID
    lda af_found_file
    and #(MASK_found_d64)
    beq drive_format_got_name
    ; got a d64, now use it's name / ID
    ; copy name from AF header
    lda #D64_bank
    sta af_reg_bank
    ldy #15
-   lda $8000 + $20,y
    sta drive_format_string_name,y
    dey
    bpl -

    ; find name length
    ldy #15
-   lda drive_format_string_name,y
    cmp #' '
    bne +
    dey
    bne -
+
    iny

    ; store the ','
!ct pet ; ...in petscii
    lda #','
    sta drive_format_string_name,y
!ct scr
    iny

    ; copy ID from D64 data
    ; id offset 0x165a2 ( + header)
    lda #(D64_bank + ($16000 / $2000))
    sta af_reg_bank

    ldx #0
-   lda $86a2,x
    sta drive_format_string_name,y
    iny
    inx
    cpx #2
    bne -

    ; restore right bank
    lda #Base_bank
    sta af_reg_bank

drive_format_got_name:
    ; drive format string ready, now use it
    ; y = len(diskname,id)

    ; print the string
    jsr printhandlerstatus

    ; open device,device,15,"n:disk,id"
    tya
    clc
    adc #2              ; add "n:"
    ldx #<drive_format_string
    ldy #>drive_format_string
    jsr k_SETNAM
    ldx $ba             ; unitnr
    txa                 ; = filenr
    sta af_formatting,x ; set flag "formatting unit X"
    ldy #15             ; sec.address (,15)
    jsr k_SETLFS
    jsr k_OPEN          ; OPEN device,device,15,"n:..."
    bcc drive_format_done

drive_format_got_error:
    jsr printdriveerror
    jsr drive_format_close

drive_format_done:
    ; get back to menu
    jmp initints

} ; pseudopc keyhandler_dest
drive_format_end:


; - detect_af_file
; parameters:
;  a = bank
;  y = type
;  x = offset to print location
; returns:
;  zp_src -> real (found) or fake (didn't find) file
;  C = 1 if af file found
;
detect_af_file:
!pseudopc keyhandler_dest {
    ; store type to end of reference signature
    sty af_file_sig_type
    ; assume we don't find a file
    ldy #<af_file_sig
    sty zp_src
    ldy #>af_file_sig
    sty zp_srch
    ; set bank
    sta $de00
    ; header (if any) is now at $8000
    ; now to compare...
    ldy #(7+1)
-   lda $8000,y
    sta af_file_was_type
    cmp af_file_sig,y
    bne detect_af_file_nope
    dey
    bpl -
    ; signature matches
    ; file found, read info from $8000
    lda #0
    sta zp_src
    lda #$80
    sta zp_srch
detect_af_file_nope:
    ; doesn't match, no file

detect_af_print:
    ldy #$20
    ; print name
-   lda (zp_src),y
    sta textloc_fileinfo,x
    inx
    iny
    cpy #$30
    bne -

    ; if d64, get id
    lda af_file_sig_type
    cmp #AFFILE_d64
    bne detect_af_got_prg

    ; copy ID from D64 data
    ; id offset 0x165a2 ( + header)
    lda #(D64_bank + ($16000 / $2000))
    sta af_reg_bank
    ldy #0
-   inx
    lda $86a2,y
    sta textloc_fileinfo,x
    iny
    cpy #2
    bne -

    ; all info from ROM taken
    ; restore normal bank
    lda #Base_bank
    sta $de00
    ; we're done for d64
    jmp detect_af_finished

detect_af_got_prg:
    ; increase offset to point to addr and
    ; store current place to tmpptr for printhex
    txa
    clc
    adc #<(textloc_fileinfo + 7)
    sta tmpptr
    lda #>(textloc_fileinfo + 7)
    adc #0
    sta tmpptrh

    ; check if we really found a prg
    bit zp_srch
    bmi +

    ; set fake file size and len
    lda #0
    pha
    pha
    pha
    pha
    sta detect_af_prg_load
    sta detect_af_prg_loadh
    beq ++

+
    ; get file size (2 bytes, MSB first)
    ldx #2
    ldy #$19
-   lda (zp_src),y
    pha
    dey
    dex
    bne -

    ; get load addr (2 bytes, LSB first)
    inc zp_srch
    ldy #0
    lda (zp_src),y
    pha
    sta detect_af_prg_load
    iny
    lda (zp_src),y
    pha

    ; substract the load address (and 1) from size for display
    tax
    lda detect_af_prg_load
    sec
    sbc #(2 + 1)
    sta detect_af_prg_load
    bcs +
    dex
+
    stx detect_af_prg_loadh

++
    ; all info from ROM taken
    ; restore normal bank
    lda #Base_bank
    sta $de00

    ; print load addr
    ldx #2
-   pla
    jsr printhex
    dex
    bne -

    ; point to " - $" location on screen
    lda tmpptr
    clc
    adc #4
    sta tmpptr
    bcc +
    inc tmpptrh
+
    ; calc and print end addr
    pla     ; size LSB
    clc
detect_af_prg_load = * + 1
    adc #$12
    tay
    pla     ; size MSB
detect_af_prg_loadh = * + 1
    adc #$12
    jsr printhex
    tya
    jsr printhex

detect_af_finished:
    ; return C = 1 if found
    lda zp_srch ; $80 if found, < $80 if not
    asl
    rts


; - Alien Flash File signature
; we also have a fake entry here...
af_file_sig: ; "RRBY64AF"
!by $52, $52, $42, $59, $36, $34, $41, $46
af_file_sig_type:
!by $00
!fi 15, $ff
; file size
!by $00, $00, $00, $00
!fi 4, $ff
; name
;     0123456789abcdef
!scr "(no file exists)"

af_file_was_type: !by 0


; - detect_rom_vec_handler
; parameters:
;  a = bank
;  y = bankh
;  (x = number)
; returns:
;  C = 0 if bank has something in it
;
detect_rom_vec_handler:
    sta af_reg_bank
    sty af_reg_bankh

    ; check ROM vectors
    ldy #$5
-   lda $bfff - 5,y
    cmp #$ff
    bne +
    dey
    bpl -
+
    ; restore bank
    lda #Base_bank
    sta af_reg_bank
    lda #0
    sta af_reg_bankh
    rts


; - detect_ar_file_handler
; parameters:
;  a = bank
;  y = bankh
;  x = addr MSB ($80 or $a0)
; returns:
;  C = 1 if bank has something in it
;
detect_ar_file_handler:
    sta af_reg_bank
    sty af_reg_bankh
    stx detect_ar_file_handler_bank

    ; check for "CBM80"
    ldx #4
    ldy #8
detect_ar_file_handler_bank = * + 2
-   lda $8000,y
    cmp cbm80_signature,x
    beq +
    dey
    dex
    bne -
    clc
+
    ; restore bank
    lda #Base_bank
    sta af_reg_bank
    lda #0
    sta af_reg_bankh
    rts

cbm80_signature:
!by $c3, $c2, $cd, $38, $30

;>C:8000  09 80 0c 80  c3 c2 cd 38  30
} ; pseudopc keyhandler_dest
detect_af_file_end:

; - boot_prg
;
boot_prg:
!pseudopc keyhandler_dest {
    ; offset for error message
    ldy #prgtxt_no_prg_offset
    ; check if there's any PRG to start
    lda af_found_file
    and #(MASK_found_prg)
    bne +
    jmp show_found_error
    ; got a prg, now boot it.
+

    ; - the following is adapted from EasyLoader

load_prg:
    ; copy load code to cart RAM ($df00)
    ldx #0
-   lda af_prgload_scr,x
    sta $df00,x
    inx
    cpx #LEN_af_prgload_scr
    bne -

    ; setup parameters
    ; set bank where the prg is
    lda #PRG_bank
    sta af_reg_bank

    ; get prg load addr
    lda $8000+$100
    sta af_prgload_dst
    lda $8000+$101
    sta af_prgload_dst+1

    ; add to length, store end addr
    lda af_prgload_dst
    clc
    adc $8000+$18
    sta af_loading_enda
    lda af_prgload_dst+1
    adc $8000+$19
    sta af_loading_endah

    ; do a partial reset via code in I/O
    jmp do_partial_reset

    ; continues in back_from_partial_reset, which must reside in ROM
} ; pseudopc keyhandler_dest
boot_prg_end:


    ; (this stays in ROM)
back_from_partial_reset:
    ; change CHRIN vector
    lda $324
    sta smc_RESTORE_LOWER+1
    lda $325
    sta smc_RESTORE_UPPER+1

    ; add our vector
    lda #<reset_trap
    sta $324
    lda #>reset_trap
    sta $325

    ; continue reset-routine
    jmp go_reset

file_copier:
    ; setup parameters
    lda af_prgload_dst
    sta zp_dst

    lda af_prgload_dst+1
    sta zp_dst+1

    lda #<($8102)
    sta zp_src

    lda #>($8102)
    sta zp_src+1

    lda af_loading_enda
    sta $2d
    sta $2f
    sta $31

    lda af_loading_endah
    sta $2e
    sta $30
    sta $32

    lda #<($0801)
    sta $2b
    lda #>($0801)
    sta $2c

    lda #8
    sta $ba

    ; inject "run\n"
    lda #$52
    sta $277
    lda #$55
    sta $278
    lda #$4e
    sta $279
    lda #$0d
    sta $27a
    lda #$04
    sta $c6

    ; copy file in I/O
    jmp copy_file


; executed at $df00
af_prgload_scr:    ; start of code to copy
!pseudopc $df00 {

do_partial_reset:
    ; disable rom
    lda #MODE_off
    sta af_reg_control

    ; do a partial reset
    sei
    ldx #$ff
    txs
    ldx #$05
    stx $d016
    jsr $fda3
    jsr $fd50
    jsr $fd15
    jsr $ff5b

    ; enable rom
    lda #MODE_16k
    sta af_reg_control

    ; select correct bank
    lda #Base_bank
    sta af_reg_bank

    jmp back_from_partial_reset

go_reset:
    ; disable rom
    lda #MODE_off
    sta af_reg_control
    jmp $fcfe

; RESET IS DONE
; RESTORE VECTOR
; JUMP BACK IN CARTRIDGE

reset_trap:
    ; restore A,X,Y
    sei
    pha
    txa
    pha
    tya
    pha

    ; restore_vector (by self-modifying-code)
smc_RESTORE_LOWER:
    lda #$00
    sta $324
smc_RESTORE_UPPER:
    lda #$00
    sta $325

    ; activate cart
    lda #MODE_16k
    sta af_reg_control

    ; jump back to program
    jmp file_copier

    ; copy the file
copy_file:
    ldy #0
--
af_loading_bank = * + 1
    lda #PRG_bank
    sta af_reg_bank

    ; copy a byte
-
    lda (zp_src),y
    sta (zp_dst),y

    ; increase dest, check for end addr
    inc zp_dst
    bne +
    inc zp_dsth
af_loading_endah = * + 1
+   lda #$12
    cmp zp_dsth
    bne +
af_loading_enda = * + 1
    lda #$12
    cmp zp_dst
    beq af_loading_done

    ; increase source, check for bank changes
+
    inc zp_src
    bne -
    inc zp_srch
    lda #$a0
    cmp zp_srch
    bne -
    ; next bank, reset address
    inc af_loading_bank
    lda #$80
    sta zp_srch
    bne --

af_loading_done:
    ; disable cart
    lda #MODE_off
    sta af_reg_control

    ; restore A,X,Y
    pla
    tay
    pla
    tax
    pla

    cli
    jmp ($324)

; stashed data
af_prgload_dst:
!by 0, 0

} ; pseudopc $df00
LEN_af_prgload_scr = * - af_prgload_scr

    ; - the EasyLoader stealing is over.


; -- Misc subroutines

; - copy_handler_to_ram
; parameters:
;  y = handler index
;
copy_handler_to_ram:
    tya
    tax     ; stash y
    ; get handler address range
    asl
    tay     ; y = index * sizeof(word)
    lda key_handler_table_start,y
    sta zp_src
    lda key_handler_table_end,y
    sta tmpptr
    iny
    lda key_handler_table_start,y
    sta zp_srch
    lda key_handler_table_end,y
    sta tmpptrh

    ; copy to destination (overcopies last bank)
    lda #<keyhandler_dest
    sta zp_dst
    lda #>keyhandler_dest
    sta zp_dsth

    ldy #0
-   lda (zp_src),y
    sta (zp_dst),y
    iny
    bne -
    inc zp_dsth
    inc zp_srch
    lda tmpptrh
    cmp zp_srch
    bcs -

    ; restore index
    txa
    tay
    rts


; - drive_format_close
;
drive_format_close:
    ; close device
    ldx $ba
    lda #0
    sta af_formatting,x
    txa
    jsr k_CLOSE     ; CLOSE device
    rts

; - drive_format_wait
; returns:
;  C = 1 -> was formatting
;
drive_format_wait:
    clc ; assume no need to wait
    ; get flag "formatting unit X"
    ldx $ba
    lda af_formatting,x
    beq +
    ; already formatting, wait for it to finish and leave
    ldy #prgtxt_wait_offset
    jsr printstatus
    jsr drive_format_close
    jsr erasestatus
    sec ; signal waiting
+   rts


; - printstatus
; parameters:
;  y = offset to text
; changes:
;  a, y
;
printstatus:
    txa
    pha
    ldx #0
-   lda Prgtxt_no_base,y
    beq +
    sta textloc_statusinfo,x
    iny
    inx
    bne -
+   pla
    tax
    rts

; - printhandlerstatus
; changes:
;  a
;
printhandlerstatus:
    txa
    pha
    ldx #0
-   lda keyhandler_dest + 3,x
    beq +
    sta textloc_statusinfo,x
    inx
    bne -
+   pla
    tax
    rts


; - erasestatus
; changes:
;  x
;
erasestatus:
    pha
    ldx #0
    lda #' '
-   sta textloc_statusinfo,x
    sta textloc_usbdump,x
    inx
    bne -
    pla
    rts


; - error_feedback
;
error_feedback:
    ; set the background color
    lda #2
    sta $d021

    ; TODO play sound?

    ; wait some frames
    ldy #40
-   jsr waitframe
    dey
    bne -

    ; reset the background color
    lda #0
    sta $d021
    rts


; - show_found_error
; parameters
;  y = offset to error text
;
show_found_error:
    ; print error text
    jsr printstatus
    jsr error_feedback
    jmp initints


; printdriveerror
; parameters:
;  a = error code
;
printdriveerror:
    pha
    ldy #prgtxt_drive_error_offset
    jsr printstatus
    pla
    ldy #<(textloc_statusinfo + 13)
    sty tmpptr
    ldy #>(textloc_statusinfo + 13)
    sty tmpptrh
    jsr printhex
    jsr error_feedback
    rts


; - printhex
; parameters:
;  tmpptr -> screen location to print to
;  a = value to print
; returns:
;  tmpptr += 2
; changes:
;  tmp
;
printhex:
    ; store input value
    sta tmp
    ; store y & x
    txa
    pha
    tya
    pha
    ; get input value
    lda tmp
    ; mask lower
    and #$0f
    ; lookup
    tax
    lda hex_lut,x
    ; print
    ldy #1
    sta (tmpptr),y
    ; lsr x4
    lda tmp
    lsr
    lsr
    lsr
    lsr
    ; lookup
    tax
    lda hex_lut,x
    ; print
    dey
    sta (tmpptr),y
    ; tmpptr += 2
    lda tmpptr
    clc
    adc #$02
    sta tmpptr
    bcc +
    inc tmpptr+1
+   ; restore registers
    pla
    tay
    pla
    tax
    lda tmp
    rts

; - hex lookup table
hex_lut:
!scr "0123456789abcdef"


; - printdecimal_div
; parameters:
;  tmpptr,y -> location to print to
;  a = value
;  x = divisor
; returns:
;  x = result
;  a = remainder
;  y = (dec(a/y) < 1) ? 0 : 1
; changes:
;  tmp
;
printdecimal_div:
    stx tmp
    ldx #0
-   cmp tmp
    bcc +
    inx
    sbc tmp
    bne -
+
    cpx #0
    beq +
printdecimal_print:
    pha
    txa
    clc
    adc #'0'
    sta (tmpptr),y
    iny
    pla
+
    rts


; - printdecimal
; parameters:
;  tmpptr -> location to print to
;  y = offset to location
;  a = value to print
; returns:
;  a = x = y = offset + length of number
; changes:
;  tmp
;
printdecimal:
    ldx #100
    jsr printdecimal_div
    ; x = hundreds, a = tens & ones
    ldx #10
    jsr printdecimal_div
    ; x = tens, a = ones
    tax
    jsr printdecimal_print

    ; print 1 space
    tya
    lda #' '
    sta (tmpptr),y
    iny

    ; return new offset (pointing to space)
    rts


; - printtracksector
; parameters:
;   a = sector
;   x = track
;   y = offset to string
; changes:
;   tmpptr
;
printtracksector:
    pha     ; stash sector
    lda #<(keyhandler_dest + 3)
    sta tmpptr
    lda #>(keyhandler_dest + 3)
    sta tmpptrh
    ; print track
    txa
    jsr printdecimal
    ; print sector
    pla
    jsr printdecimal
    ; terminate string
    lda #$0d
    sta (tmpptr),y
    iny
    lda #0
    sta (tmpptr),y
    rts


; - waitframe
;
waitframe:
-   lda $d011
    bpl -
-   lda $d011
    bmi -
    rts

; - waitkey
; returns:
;  a = x = pressed key
;
waitkey:
-   jsr waitframe
    jsr k_GETIN     ; read key (if any)
    tax             ; x = pressed key (or 0)
    beq -           ; if no keys pressed, no input -> wait
    rts

; - smalldelay
;
smalldelay:
    pha
    txa
    pha
    ldx #20
-   lda delaycolor,x
    sta $d020
    jsr waitframe
    dex
    bpl -
    pla
    tax
    pla
    rts


; -- Data

; border colors for smalldelay (reverse order of showing)
delaycolor:
!by 0
!fi 2, 11
!fi 2, 12
!fi 2, 15
!fi 6, 1
!fi 2, 15
!fi 2, 12
!fi 2, 11
!by 0

; table of sector amounts on disk tracks
disk_track_to_max_sector:
!by 0
!by 21, 21, 21, 21, 21, 21, 21, 21, 21, 21  ; 1 - 10
!by 21, 21, 21, 21, 21, 21, 21, 19, 19, 19  ; 11 - 20
!by 19, 19, 19, 19, 18, 18, 18, 18, 18, 18  ; 21 - 30
!by 17, 17, 17, 17, 17                      ; 31 - 35
!by 0

; tables for ROM SLOT number -> bank regs translation

rom_slot_bank:
!for .i, 16 {
!by <((.i - 1) * ($80000 / $2000))
}

rom_slot_bankh:
!for .i, 16 {
!by >((.i - 1) * ($80000 / $2000))
}

menu_end = *


ROM_bank0_end:

!if ROM_bank0_end > EAPI_start {
!error "menu code too big"
}



; --- EAPI
; (skip .prg loading addr)
;
* = EAPI_start
!bin "eapi-am29f040-04",,2
EAPI_end = *


; --- Startup code

; heavily based on banking-test.s by Thomas 'skoe' Giesel

* = $bf00
!pseudopc $ff00 {
; minimal interrupt handlers, just in case
startup_retn:
;    bit $dd0c
    rti

startup_reti:
;    lsr $d019
;    bit $dc0c
    rti

cold_start:
    ; === the reset vector points here ===
    sei
    ldx #$ff
    txs
    cld

    ; enable VIC (e.g. RAM refresh)
    lda #8
    sta $d016

    ; write to RAM to make sure it starts up correctly (=> RAM datasheets)
-   sta Startup_dest,x
    dex
    bne -

    ; sprites off
    lda #0
    sta $d015

    ; preserve $2b-$2e in $dfxx ram
    ldy #3
-   lda $2b,y
    sta Af_addr_stash,y
    dey
    bpl -

    ; copy the final start-up code to RAM (bottom of CPU stack)
    ldx #0
-   lda startup_start,x
    sta Startup_dest,x
    inx
    cpx #Startup_size
    bne -
    jmp Startup_dest

startup_start:
!pseudopc Startup_dest {
    ; === this code is copied to the stack area, does some inits ===
    ; === scans the keyboard and kills the cartridge or          ===
    ; === starts the main application                            ===

    ; set cart off
    lda #MODE_off
    sta af_reg_control

    ; Check if one of the magic kill keys is pressed
    ; This should be done in the same way on any EasyFlash cartridge!

    ; Prepare the CIA to scan the keyboard
    lda #$7f
    sta $dc00   ; pull down row 7 (DPA)

    ldx #$ff
    stx $dc02   ; DDRA $ff = output (X is still $ff from copy loop)
    inx
    stx $dc03   ; DDRB $00 = input

    ; Read the keys pressed on this row
    lda $dc01   ; read coloumns (DPB)

    ; Restore CIA registers to the state after (hard) reset
    stx $dc02   ; DDRA input again
    stx $dc00   ; Now row pulled down

    ; Check if one of the magic kill keys was pressed
    and #$e0    ; only leave "Run/Stop", "Q" and "C="
    cmp #$e0
    bne startup_kill  ; branch if one of these keys is pressed

fontsetup:
    ; character rom on
    lda #$2f
    sta $00
    lda #$31
    sta $01

    ; copy character rom to RAM
    ldx #8
fontsetup_next:
    ldy #0
fontptrh = * + 2
-   lda $d800,y     ; load from char-rom
ownfontptrh = * + 2
    sta ownfont,y   ; store to ram
    iny
    bne -

    ; update pointers
    inc fontptrh
    inc ownfontptrh

    dex
    bne fontsetup_next

    ; character rom off, I/O & ROM on
    lda #$37
    sta $01

    ; set cart to 16k
    lda #MODE_16k
    sta af_reg_control

    ; protections off
    lda #MODE_poff
    sta af_reg_control2

    ; start the application code
    jmp main_entrypoint

startup_kill:
    jmp ($fffc) ; reset
}   ; pseudopc Startup_dest
startup_end:
Startup_size = startup_end - startup_start

!if startup_end > $fffa {
!error "startup code too big"
}
} ; pseudopc $fb00

; -- vectors
!align $3fff, $3ffa, $ff

!pseudopc $fffa {
!word startup_retn  ; NMI
!word cold_start    ; RESET
!word startup_reti  ; IRQ
} ; pseudopc $fffa
