#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "aftt.h"
#include "afusb.h"
#include "config.h"
#include "flashcore.h"
#include "globals.h"
#include "util.h"

/* -------------------------------------------------------------------------- */
/* settings */

static unsigned int aftt_erase_reads = 0x4000;
static unsigned int aftt_prog_reads = 0x1;
static unsigned int aftt_usb_read_blocksize = 0x40;
static unsigned int aftt_usb_write_blocksize = 0x800 * 5;
static int aftt_do_erases = 0;

int set_usb_read_blocksize(int argc, char **argv)
{
    int v = 0;

    if (parse_hex(argv[0], &v, 6, NULL) < 0) {
        util_error("invalid hex string '%s'", argv[0]);
        return -1;
    }

    if (v < 1) {
        util_error("block size '%s' too small", argv[0]);
        return -1;
    }

    aftt_usb_read_blocksize = (unsigned int)v;

    return 0;
}

int set_usb_write_blocksize(int argc, char **argv)
{
    int v = 0;

    if (parse_hex(argv[0], &v, 6, NULL) < 0) {
        util_error("invalid hex string '%s'", argv[0]);
        return -1;
    }

    if (v < 1) {
        util_error("block size '%s' too small", argv[0]);
        return -1;
    }

    if (v > AFTT_USB_WRITE_BLOCKSIZE_MAX) {
        util_error("block size '%s' too large", argv[0]);
        return -1;
    }

    aftt_usb_write_blocksize = (unsigned int)(v * 5);

    return 0;
}

int set_dummy_erase_amount(int argc, char **argv)
{
    int v = 0;

    if (parse_hex(argv[0], &v, 6, NULL) < 0) {
        util_error("invalid hex string '%s'", argv[0]);
        return -1;
    }

    aftt_erase_reads = (unsigned int)v;

    return 0;
}

int set_dummy_prog_amount(int argc, char **argv)
{
    int v = 0;

    if (parse_hex(argv[0], &v, 6, NULL) < 0) {
        util_error("invalid hex string '%s'", argv[0]);
        return -1;
    }

    aftt_prog_reads = (unsigned int)v;

    return 0;
}


int set_do_erases(int argc, char **argv)
{
    aftt_do_erases = 1;
    util_message("Enabled erases on created/sent AFTT.");
    return 0;
}

/* -------------------------------------------------------------------------- */
/* "emulation" of the Alien Flash USB.

   If fd_usb != -1, send the commands and read the results via USB.
   Otherwise, perform the writes/reads on the emulated Flash chips.
*/

static unsigned int aftt_parse_param(unsigned char command[5])
{
    return command[1]
          | (command[2] << 8)
          | (command[3] << 16);
}

int aftt_run(FILE *fd_i, FILE *fd_o, int vlevel, int fd_usb)
{
    int rc = 1;
    int tristate = 0;
    unsigned int addr;
    int command_count = 0;
    int same_command_count = 1;
    unsigned char command[5];
    unsigned char command_last = 0xff;
    unsigned char b;
    int command_len;

    while (rc > 0) {
        command_len = 0;

        if (fread(&command, 1, 1, fd_i) < 1) {
            if (feof(fd_i)) {
                rc = 0;
                break;
            }

            util_error("problems reading input file.");
            break;
        }

        if (command_last == *command) {
            same_command_count++;
        } else {
            if ((vlevel > 0) && (same_command_count > 2)) {
                util_message("AFTT: got CMD $%02x %i times in row", command_last, same_command_count);
            }
            command_last = *command;
            same_command_count = 1;
        }

        switch (*command) {
            /* invalid? */
            case 0x00:
                if (vlevel > 0) {
                    util_message("AFTT: CMD $00, tristate %i -> 0", tristate);
                }
                tristate = 0;
                command_len = 1;
                break;

            /* write */
            case 0x55:
                if (fread(&command[1], 4, 1, fd_i) < 1) {
                    util_error("problems reading input file.");
                    rc = -1;
                    break;
                }

                /* special tristate command */
                if (!tristate &&
                    (command[1] == 0x55) &&
                    (command[2] == 0x55) &&
                    (command[3] == 0x55) &&
                    (command[4] == 0x55)) {
                    if (vlevel > 0) {
                        util_message("AFTT: CMD $55, tristate %i -> 1", tristate);
                    }
                    tristate = 1;
                } else if (!tristate) {
                    util_error("write command while not in tristate, aborting.");
                    rc = -1;
                    break;
                } else {
                    addr = aftt_parse_param(command);

                    if (vlevel > 2) {
                        util_message("AFTT: store $%06x <- $%02x", addr, command[4]);
                    }

                    if (fd_usb == -1) {
                        flashcore_store(addr, command[4]);
                    }
                }

                command_len = 5;
                break;

            /* read */
            case 0xd5:
                if (fread(&command[1], 4, 1, fd_i) < 1) {
                    util_error("problems reading input file.");
                    rc = -1;
                    break;
                }

                if (!tristate) {
                    util_error("read command while not in tristate, aborting.");
                    rc = -1;
                    break;
                } else if (command[4] != 0) {
                    util_error("read command last param $%02x != $00, aborting.", command[4]);
                    rc = -1;
                    break;
                }

                /* perform the read after (possibly) sending the command via USB */

                command_len = 5;
                break;

            /* invalid */
            default:
                util_error("got invalid command $%02x, aborting.", *command);
                tristate = 0;
                rc = -1;
                break;
        }

        if ((rc != -1) && (fd_usb != -1) && (command_len > 0)) {
            /* got valid command, send it */
            if (afusb_write(fd_usb, (const unsigned char *)command, command_len) < command_len) {
                rc = -1;
                break;
            }
        }

        if (*command == 0xd5) {
            /* handle reading at this point */
            addr = aftt_parse_param(command);

            if (fd_usb == -1) {
                b = flashcore_read(addr);
            } else if (afusb_read(fd_usb, &b, 1) != 1) {
                rc = -1;
                break;
            }

            if (vlevel > 2) {
                util_message("AFTT:  read $%06x -> $%02x", addr, b);
            }

            /* write out the read byte if an output file was given */
            if ((fd_o != NULL) && (fwrite(&b, 1, 1, fd_o) < 1)) {
                util_error("problems writing output file.");
                rc = -1;
                break;
            }
        }

        command_count++;
    }

    if (vlevel > 0) {
        util_message("AFTT: %i commands, last was $%02x %i times", command_count, *command, same_command_count);
        util_message("AFTT: $%06x-$%06x written", flash_prog_low, flash_prog_end - 1);
    }

    if (tristate) {
        util_warning("still in tristate.");
        /* try to set Alien Flash to a sane state */
        if ((fd_usb != -1) && (afusb_write(fd_usb, (const unsigned char *)"\x00", 1) < 1)) {
            return -1;
        }
    }

    return 0;
}


/* -------------------------------------------------------------------------- */
/* write buffering */

#define AFTT_FL_IS_USB    (1 << 0)
#define AFTT_FL_DO_ERASES (1 << 1)

static unsigned char aftt_write_buffer[(AFTT_USB_WRITE_BLOCKSIZE_MAX + 1) * 5];
static int aftt_write_buffer_pos = 0;

inline static void aftt_write_init(void)
{
    aftt_write_buffer_pos = 0;
}

inline static int aftt_write_flush(int fd, int flags)
{
    int rc = 0;
    int num = aftt_write_buffer_pos;

    if (num == 0) {
        /* nothing to flush */
        return 0;
    }

    aftt_write_buffer_pos = 0;

    if (flags & AFTT_FL_IS_USB) {
        rc = afusb_write(fd, aftt_write_buffer, num);
    } else {
        rc = write(fd, aftt_write_buffer, num);
    }

    if (rc != num) {
        return -1;
    }

    return 0;
}

inline static int aftt_write(int fd, const unsigned char* data, int num, int flags)
{
    int len;
    int do_flush;

    while (num > 0) {
        len = num;

        if ((aftt_write_buffer_pos + len) >= aftt_usb_write_blocksize) {
            len = aftt_usb_write_blocksize - aftt_write_buffer_pos;
            do_flush = 1;
        } else {
            do_flush = 0;
        }

        memcpy(&aftt_write_buffer[aftt_write_buffer_pos], data, len);

        aftt_write_buffer_pos += len;
        num -= len;
        data += len;

        if (do_flush && (aftt_write_flush(fd, flags) < 0)) {
            return -1;
        }
    }

    return 0;
}


/* -------------------------------------------------------------------------- */
/* AFTT command generation */

inline static int aftt_add_reads(int fd, unsigned int addr, int num_reads, int flags)
{
    unsigned char rcmdseq[] =
        { 0xd5, 0x00, 0x00, 0x00, 0x00 };

    /* set the address */
    rcmdseq[1] = addr & 0xff;
    rcmdseq[2] = (addr >> 8) & 0xff;
    rcmdseq[3] = (addr >> 16) & 0xff;

    /* read the addr for a (max) number of times */
    while (num_reads--) {
        if (aftt_write(fd, rcmdseq, 5, flags) < 0) {
            return -1;
        }
    }

    return 0;
}

static int aftt_add_erase(int fd, unsigned int addr, int flags)
{
    unsigned char cmdseq[] =
        { 0x55, 0x55, 0x05, 0x00, 0xaa,
          0x55, 0xaa, 0x02, 0x00, 0x55,
          0x55, 0x55, 0x05, 0x00, 0x80,
          0x55, 0x55, 0x05, 0x00, 0xaa,
          0x55, 0xaa, 0x02, 0x00, 0x55,
          0x55, 0x00, 0x00, 0x00, 0x30 };

    /* set address to the command sequence */
    cmdseq[3 + (0 * 5)] = (addr >> 16) & 0xff;
    cmdseq[3 + (1 * 5)] = (addr >> 16) & 0xff;
    cmdseq[3 + (2 * 5)] = (addr >> 16) & 0xff;
    cmdseq[3 + (3 * 5)] = (addr >> 16) & 0xff;
    cmdseq[3 + (4 * 5)] = (addr >> 16) & 0xff;
    cmdseq[3 + (5 * 5)] = (addr >> 16) & 0xff;

    if (aftt_write(fd, cmdseq, 6 * 5, flags) < 0) {
        return -1;
    }

    if (flags & AFTT_FL_IS_USB) {
        /* don't do extra reads with real HW, instead:
           - flush the write buffer to get the erase command transmitter
           - sleep for a while so the erase can complete */
        if (aftt_write_flush(fd, flags) < 0) {
            return -1;
        }

        util_msleep(1000);
        return 0;
    }

    /* read the last addr for a number of times */
    if (aftt_add_reads(fd, addr | 0xffff, aftt_erase_reads, flags) < 0) {
        return -1;
    }

    return 0;
}

static int aftt_add_prog(int fd, unsigned int addr, unsigned char data, int flags)
{
    unsigned char cmdseq[] =
        { 0x55, 0x55, 0x05, 0x00, 0xaa,
          0x55, 0xaa, 0x02, 0x00, 0x55,
          0x55, 0x55, 0x05, 0x00, 0xa0,
          0x55, 0x00, 0x00, 0x00, 0x00 };

    /* let's optimize by not writing the 0xff bytes */
    if (data == 0xff) {
        return 0;
    }

    /* set address and data to the command sequence */
    cmdseq[3 + (0 * 5)] = ((addr & 0xff0000) >> 16) & 0xff;
    cmdseq[3 + (1 * 5)] = ((addr & 0xff0000) >> 16) & 0xff;
    cmdseq[3 + (2 * 5)] = ((addr & 0xff0000) >> 16) & 0xff;
    cmdseq[3 + (3 * 5)] = ((addr & 0xff0000) >> 16) & 0xff;
    cmdseq[2 + (3 * 5)] = (addr >> 8) & 0xff;
    cmdseq[1 + (3 * 5)] = addr & 0xff;
    cmdseq[4 + (3 * 5)] = data;

    if (aftt_write(fd, cmdseq, 4 * 5, flags) < 0) {
        return -1;
    }

    if (flags & AFTT_FL_IS_USB) {
        /* don't do extra reads with real HW */
        return 0;
    }

    /* read the last addr for a number of times */
    if (aftt_add_reads(fd, addr, aftt_prog_reads, flags) < 0) {
        return -1;
    }

    return 0;
}

static int aftt_add_start(int fd, int flags)
{
    unsigned char cmdseq[] =
        { 0x00,
          0x55, 0x55, 0x55, 0x55, 0x55 };

    aftt_write_init();

    if (aftt_write(fd, cmdseq, 6, flags) < 0) {
        return -1;
    }

    /* flush the command out now */
    if (aftt_write_flush(fd, flags) < 0) {
        return -1;
    }

    return 0;
}

static int aftt_add_end(int fd, int flags)
{
    unsigned char cmdseq[] = { 0x00 };

    if (aftt_write(fd, cmdseq, 1, flags) < 0) {
        return -1;
    }

    if (aftt_write_flush(fd, flags) < 0) {
        return -1;
    }

    return 0;
}


/* -------------------------------------------------------------------------- */

static int aftt_save_internal(int fd, unsigned char *data, size_t datasize, unsigned int addr, int flags)
{
    int rc = 0;

    /* start with the tristate sequence */
    if (aftt_add_start(fd, flags) < 0) {
        return -1;
    }

    while (datasize--) {
        /* erase sector before it's written to (if enabled) */
        if ((flags & AFTT_FL_DO_ERASES) && ((addr & 0xffff) == 0)) {
            if (aftt_add_erase(fd, addr, flags) < 0) {
                /* signal an error and try to send the 00 at the end */
                rc = -1;
                break;
            }
        }

        /* program the byte */
        if (aftt_add_prog(fd, addr, *data, flags) < 0) {
            /* signal an error and try to send the 00 at the end */
            rc = -1;
            break;
        }

        /* next addr and data */
        addr++;
        data++;
    }

    /* end with the 00 command */
    if (aftt_add_end(fd, flags) < 0) {
        return -1;
    }

    return rc;
}

int aftt_save(const char *filename, unsigned char *data, size_t datasize, unsigned int addr)
{
    int fd;
    int rc;
    int flags;

    if (data == NULL) {
        return -1;
    }

    util_message("Saving AFTT file '%s' (at $%06x - $%06x)...", filename, addr, (unsigned int)(addr + datasize - 1));

    fd = open(filename,
    /* FIXME this could be handled better... */
#ifdef IS_WINDOWS
                O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, S_IRUSR | S_IWUSR
#else
                O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH
#endif
             );

    if (fd < 0) {
        util_error("couldn't open file '%s' for writing!", filename);
        return -1;
    }

    flags = aftt_do_erases ? AFTT_FL_DO_ERASES : 0;
    rc = aftt_save_internal(fd, data, datasize, addr, flags);
    close(fd);

    if (rc < 0) {
        util_error("problems saving file '%s'!", filename);
    }

    return rc;
}

/* -------------------------------------------------------------------------- */
/* send/receive to/from a real Alien Flash device */

static int aftt_load_internal(int fd, unsigned char *data, int offset, int offset_end)
{
    unsigned int i, j;
    int rc = 0;
    unsigned char *p = &data[offset];


    /* start with the tristate sequence */
    if (aftt_add_start(fd, AFTT_FL_IS_USB) < 0) {
        return -2;
    }

    for (i = (unsigned int)offset, j = 0; i <= (unsigned int)offset_end; ++i, ++j) {
        if (aftt_add_reads(fd, i, 1, AFTT_FL_IS_USB) < 0) {
            rc = -1;
            break;
        }

        if (j == aftt_usb_read_blocksize) {
            if (aftt_write_flush(fd, AFTT_FL_IS_USB) < 0) {
                rc = -6;
                break;
            }

            if (afusb_read(fd, p, j) != j) {
                rc = -4;
                break;
            }

            p += j;
            j = 0;
        }
    }

    /* read leftover bytes */
    if ((rc == 0) && (j > 0)) {
        if (aftt_write_flush(fd, AFTT_FL_IS_USB) < 0) {
            rc = -7;
        }

        if ((rc == 0) && (afusb_read(fd, p, j) != j)) {
            rc = -5;
        }

        p += j;
        j = 0;
    }

    /* end with the 00 command */
    if (aftt_add_end(fd, AFTT_FL_IS_USB) < 0) {
        return -3;
    }

    return rc;
}

static int aftt_keep_old_internal(unsigned char *data, unsigned int *addr_out, int *addr_end_out)
{
    int addr = (int)(*addr_out);
    int addr_end = *addr_end_out;

    /* due to the Flash sector structure, we need to read the old data within
       the "gaps" in the start and end sectors of the range, also extending
       the sent range to contain that old data */
    if ((addr & 0x00ffff) != 0) {
        if (aftt_usb_load(addr & 0xff0000, addr - 1, 0) < 0) {
            return -1;
        }
        addr &= 0xff0000;
    }
    if ((addr_end & 0x00ffff) != 0x00ffff) {
        if (aftt_usb_load(addr_end + 1, addr_end | 0x00ffff, 0) < 0) {
            return -1;
        }
        addr_end |= 0x00ffff;
    }

    *addr_out = (unsigned int)addr;
    *addr_end_out = addr_end;

    return 0;
}

int aftt_usb_send(unsigned char *data, size_t datasize, unsigned int addr, int romh_too, int keep_old, int do_erases)
{
    int fd;
    int rc = -1;
    int i;
    int flags = AFTT_FL_IS_USB
              | (do_erases ? AFTT_FL_DO_ERASES : 0);

    if (data == NULL) {
        return -1;
    }

    fd = afusb_get_fd();

    if (fd < 0) {
        util_error("Alien Flash not connected!");
        return -1;
    }

    for (i = 0; i <= romh_too; ++i) {
        unsigned int addr_l = addr;
        size_t datasize_l = datasize;
        int addr_end = (unsigned int)(addr_l + datasize_l - 1);

        if (keep_old) {
            rc = aftt_keep_old_internal(data, &addr_l, &addr_end);
            if (rc < 0) {
                util_error("problems reading old data!");
                break;
            }
            datasize_l = (size_t)(addr_end - addr_l + 1);
        }

        util_message("Sending $%06x - $%06x to Alien Flash...", addr_l, addr_end);

        rc = aftt_save_internal(fd, &data[addr_l], datasize_l, addr_l, flags);

        if (rc < 0) {
            util_error("problems sending!");
            break;
        }
        addr += 0x800000;
    }

    util_message("Sending to Alien Flash done, status: %i.", rc);

    return rc;
}

int aftt_usb_load(int offset, int offset_end, int romh_too)
{
    int fd;
    int rc = -1;
    int i;

    fd = afusb_get_fd();

    if (fd < 0) {
        util_error("Alien Flash not connected!");
        return -1;
    }

    for (i = 0; i <= romh_too; ++i) {
        util_message("Receiving $%06x - $%06x from Alien Flash...", offset, offset_end);

        rc = aftt_load_internal(fd, flash_data, offset, offset_end);

        if (rc < 0) {
            util_error("problems receiving! %i", rc);
            break;
        }

        offset += 0x800000;
        offset_end += 0x800000;
    }

    util_message("Receiving from Alien Flash done, status: %i.", rc);

    return rc;
}
