/*
 * afusb.c - Alien Flash USB access wrapper
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>

#define AFUSB_INTERNAL_API
#include "afusb.h"
#undef AFUSB_INTERNAL_API
#include "config.h"
#include "util.h"

/* -------------------------------------------------------------- */

/* The drivers, default is first */
typedef struct afusb_driver_s {
    const char *name;
    int (*init)(void);
    int (*get_fd)(void);
    int (*shutdown)(void);
    int (*write)(int fd, const unsigned char *data, int num);
    int (*read)(int fd, unsigned char *data, int num);
    int (*set_param)(char *param);
    int (*get_param)(const char **param);
    int (*set_eepromdump)(int argc, char **argv);
} afusb_driver_t;

static afusb_driver_t afusb_driver[] = {
#ifdef HAVE_AFUSB_LINUX
    {
        "linux",
        afusb_linux_init,
        afusb_linux_get_fd,
        afusb_linux_shutdown,
        afusb_linux_write,
        afusb_linux_read,
        afusb_linux_set_device_param,
        afusb_linux_get_device_param,
        NULL    /* afusb_linux_set_eeprom_dump_name */
    },
#endif /* HAVE_AFUSB_LINUX */

#ifdef HAVE_AFUSB_WIN32
    {
        "win32",
        afusb_win32_init,
        afusb_win32_get_fd,
        afusb_win32_shutdown,
        afusb_win32_write,
        afusb_win32_read,
        afusb_win32_set_device_param,
        afusb_win32_get_device_param,
        NULL    /* afusb_linux_set_eeprom_dump_name */
    },
#endif /* HAVE_AFUSB_WIN32 */

#ifdef HAVE_AFUSB_FTD2XX
    {
        "ftd2xx",
        afusb_ftd2xx_init,
        afusb_ftd2xx_get_fd,
        afusb_ftd2xx_shutdown,
        afusb_ftd2xx_write,
        afusb_ftd2xx_read,
        NULL,   /* afusb_ftd2xx_set_device_param, */
        NULL,   /* afusb_ftd2xx_get_device_param, */
        NULL    /* afusb_ftd2xx_set_eeprom_dump_name */
    },
#endif /* HAVE_AFUSB_FT2XX */

#ifdef HAVE_AFUSB_FTDI
    {
        "ftdi",
        afusb_ftdi_init,
        afusb_ftdi_get_fd,
        afusb_ftdi_shutdown,
        afusb_ftdi_write,
        afusb_ftdi_read,
        afusb_ftdi_set_device_param,
        afusb_ftdi_get_device_param,
        afusb_ftdi_set_eeprom_dump_name
    },
#endif /* HAVE_AFUSB_FTDI */


    {
        "dummy",
        afusb_dummy_init,
        afusb_dummy_get_fd,
        afusb_dummy_shutdown,
        afusb_dummy_write,
        afusb_dummy_read,
        afusb_dummy_set_device_param,
        afusb_dummy_get_device_param,
        NULL    /* afusb_linux_set_eeprom_dump_name */
    },

    {
        NULL,
        NULL,   /* afusb_init, */
        NULL,   /* afusb_get_fd, */
        NULL,   /* afusb_shutdown, */
        NULL,   /* afusb_write, */
        NULL,   /* afusb_read, */
        NULL,   /* afusb_set_device_param, */
        NULL,   /* afusb_get_device_param, */
        NULL    /* afusb_set_eeprom_dump_name */
    }
};

static int driver_i = 0;
static int driver_active = -1;

/* buffer for parameters */
static char parambuf[1024] = "";

/* -------------------------------------------------------------- */

int afusb_select_driver(const char *driver_name)
{
    int i = 0, found = 0;

    if (driver_name == NULL) {
        return -1;
    }

    while (afusb_driver[i].name != NULL) {
        if (strcmp(afusb_driver[i].name, driver_name) == 0) {
            found = 1;
            break;
        }
        ++i;
    }

    if (found) {
        if ((driver_active >= 0) && (driver_active != i)) {
            util_error("can't change driver to '%s' while '%s' is active!", driver_name, afusb_driver[i].name);
            return -1;
        }

        if (driver_i == i) {
            /* already selected */
            return 0;
        }
        util_message("Selected USB driver '%s'.", driver_name);
        driver_i = i;
        return 0;
    } else {
        util_error("unknown driver '%s'!", driver_name);
        return -1;
    }
}

const char *afusb_get_driver_name(void)
{
    return afusb_driver[driver_i].name;
}

int afusb_driver_has_params(void)
{
    return (afusb_driver[driver_i].set_param != NULL);
}

int afusb_driver_has_eepromdump(void)
{
    return (afusb_driver[driver_i].set_eepromdump != NULL);
}

int afusb_is_connected(void)
{
    return (driver_active == driver_i);
}

/* -------------------------------------------------------------- */

int afusb_set_eeprom_dump_name(int argc, char **argv)
{
    if (afusb_driver[driver_i].set_eepromdump != NULL) {
        return (afusb_driver[driver_i].set_eepromdump)(argc, argv);
    }

    return -1;
}

int afusb_set_device_param(int argc, char **argv)
{
    if (afusb_driver[driver_i].set_param != NULL) {
        if (driver_active == driver_i) {
            util_error("can't change parameters while driver is active!");
            return -1;
        }

        /* copy parameter to buffer */
        strcpy(parambuf, argv[0]);

        return (afusb_driver[driver_i].set_param)(parambuf);
    }

    util_error("driver doesn't have parameters!");
    return -1;
}

int afusb_get_device_param(const char **param)
{
    if (afusb_driver[driver_i].get_param != NULL) {
        return (afusb_driver[driver_i].get_param)(param);
    }

    util_error("driver doesn't have parameters!");
    return -1;
}

int afusb_init(void)
{
    int rc;

    if (driver_active == driver_i) {
        util_warning("driver already active!");
        return 0;
    }

    rc = (afusb_driver[driver_i].init)();

    if (rc >= 0) {
        driver_active = driver_i;
    }

    return rc;
}

int afusb_get_fd(void)
{
    return (afusb_driver[driver_i].get_fd)();
}

int afusb_shutdown(void)
{
    int rc;

    if (driver_active < 0) {
        util_warning("driver shutdown while not active.");
        return 0;
    }

    rc = (afusb_driver[driver_i].shutdown)();
    driver_active = -1;

    return rc;
}

int afusb_write(int fd, const unsigned char *data, int num)
{
    return (afusb_driver[driver_i].write)(fd, data, num);
}

int afusb_read(int fd, unsigned char *data, int num)
{
    return (afusb_driver[driver_i].read)(fd, data, num);
}
