/*
 * afusb_ftd2xx.c - Alien Flash USB access with ftd2xx.dll
 *
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>

#include "config.h"

#ifdef HAVE_WTYPES_H
#include <wtypes.h>
#endif
#ifdef HAVE_WINNT_H
#include <winnt.h>
#endif

#include <ftd2xx.h>

#include "afusb.h"
#include "util.h"

/* -------------------------------------------------------------- */

/* handle to the device */
static FT_HANDLE afusb_handle;

/* just a flag */
static int afusb_fd = -1;

/* -------------------------------------------------------------- */

int afusb_ftd2xx_init(void)
{
    FT_STATUS frc;

    ui_message("Opening Alien Flash USB device...");

    /* open device */
    frc = FT_OpenEx(AFUSB_NAME, FT_OPEN_BY_DESCRIPTION, &afusb_handle);
    if (frc != FT_OK) {
        ui_error("Unable to open ftd2xx device!");
        return -1;
    }

    afusb_fd = 1;

    /* set baudrate */
    frc = FT_SetBaudRate(afusb_handle, AFUSB_BAUDRATE);
    if (frc != FT_OK) {
        ui_error("Unable to set baudrate!");
        goto fail;
    }

    /* set line property */
    frc = FT_SetDataCharacteristics(afusb_handle, FT_BITS_8, FT_STOP_BITS_1, FT_PARITY_NONE);
    if (frc != FT_OK) {
        ui_error("Unable to set line property!");
        goto fail;
    }

    ui_message("USB device opened successfully.");

    return 0;

fail:
    FT_Close(afusb_handle);
    afusb_fd = -1;
    return -1;
}

int afusb_ftd2xx_get_fd(void)
{
    return afusb_fd;
}

int afusb_ftd2xx_shutdown(void)
{
    if (afusb_fd < 0) {
        return 0;
    }

    ui_message("Closing Alien Flash USB device...");
    FT_Close(afusb_handle);
    afusb_fd = -1;
    return 0;
}

int afusb_ftd2xx_write(int fd, const unsigned char *data, int num)
{
    FT_STATUS frc;
    DWORD got_bytes = 0;
    DWORD nb = (DWORD)num;

    frc = FT_Write(afusb_handle, (LPVOID)data, nb, &got_bytes);

    if ((frc != FT_OK) || (got_bytes < nb)) {
        ui_error("Problems sending: %lu of %lu", got_bytes, nb);
        return -1;
    }

    return num;
}

int afusb_ftd2xx_read(int fd, unsigned char *data, int num)
{
    FT_STATUS frc;
    DWORD got_bytes = 0;

    frc = FT_Read(afusb_handle, data, num, &got_bytes);

    if (frc != FT_OK) {
        ui_error("Problems reading!");
        return -1;
    } else if (got_bytes != num) {
        ui_error("Problems reading, got no data!");
        return -1;
    }

    return num;
}
