/*
 * afusb_ftdi.c - Alien Flash USB access with libftdi
 *
 * Based on serial_read.c in libftdi-0.18 examples.
 * "This program is distributed under the GPL, version 2"
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>

#include <ftdi.h>

#include "afusb.h"
#include "config.h"
#include "util.h"

/* -------------------------------------------------------------- */

static struct ftdi_context ftdic;

/* buffer for raw EEPROM contents */
#define AFUSB_EEPROM_SIZE 128
static unsigned char afusb_eeprom_raw[AFUSB_EEPROM_SIZE];

/*
   Address/mask (in EEPROM) for the D2XX/VCP mode bit.
   This crucial information was digged from:
   http://rtr.ca/ft232r/
*/
#define AFUSB_MODEBIT_ADDR 0x00
#define AFUSB_MODEBIT_MASK 0x08

/* just a flag */
static int afusb_fd = -1;

/* filename to dump EEPROM to */
static const char *afusb_dump_eeprom_name = NULL;

int afusb_ftdi_set_eeprom_dump_name(int argc, char **argv)
{
    afusb_dump_eeprom_name = argv[0];
    util_message("Got request to dump EEPROM to file '%s'.", afusb_dump_eeprom_name);
    return 0;
}

/* for get_param */
static const char *afusb_param = NULL;

/* latency timer, 0 -> don't change */
static int afusb_latency_timer = 0;

/* D2XX/VCP mode change request */
static enum {
    MODE_CHANGE_NONE,
    MODE_CHANGE_VCP,
    MODE_CHANGE_D2XX
} change_mode_request = MODE_CHANGE_NONE;

int afusb_ftdi_set_device_param(char *param)
{
    int got_mode_change = 0;
    int rc;
    const char *e = NULL;

    /* store param for get_param */
    afusb_param = param;

    /* check the parameter for mode change requests */
    if (strcmp(param, AFUSB_MAGIC_D2XX_OFF_STRING) == 0) {
        change_mode_request = MODE_CHANGE_VCP;
        got_mode_change = 1;
    } else if (strcmp(param, AFUSB_MAGIC_D2XX_ON_STRING) == 0) {
        change_mode_request = MODE_CHANGE_D2XX;
        got_mode_change = 1;
    }

    if (got_mode_change) {
        util_message("Got request to change mode to %s.", (change_mode_request == MODE_CHANGE_D2XX) ? "D2XX" : "VCP");
        return 0;
    }

    /* not a mode change, check for a hex value to be used as latency timer */
    rc = parse_hex(param, &afusb_latency_timer, 2, &e);

    if (rc || (*e != '\0') || (afusb_latency_timer < 1)) {
        util_error("invalid USB parameter '%s'!", param);
        return -1;
    }

    util_message("Got request to set latency to $%02x.", afusb_latency_timer);

    return 0;
}

int afusb_ftdi_get_device_param(const char **param)
{
    *param = afusb_param;
    return 0;
}

static int afusb_ftdi_write_eeprom(void)
{
    /*
        Unfortunately the ftdi_eeprom_build/decode functions in
        libftdi-0.19 don't know about this D2XX/VCP bit, so we
        have to calculate the checksum ourselves.

        This code is taken from ftdi_eeprom_build (libftdi-0.19).
    */
    unsigned char i;
    unsigned short checksum, value;

    checksum = 0xAAAA;

    for (i = 0; i < ((AFUSB_EEPROM_SIZE / 2) - 1); i++) {
        value = afusb_eeprom_raw[i * 2];
        value += afusb_eeprom_raw[(i * 2) + 1] << 8;

        checksum = value ^ checksum;
        checksum = (checksum << 1) | (checksum >> 15);
    }

    afusb_eeprom_raw[AFUSB_EEPROM_SIZE - 2] = checksum;
    afusb_eeprom_raw[AFUSB_EEPROM_SIZE - 1] = checksum >> 8;

    /* write back the EEPROM */
    return ftdi_write_eeprom(&ftdic, afusb_eeprom_raw);
}

static int afusb_ftdi_dump_eeprom(void)
{
    FILE *fd;

    if (afusb_dump_eeprom_name == NULL) {
        return 0;
    }

    util_message("Dumping EEPROM to file '%s'...", afusb_dump_eeprom_name);

    fd = fopen(afusb_dump_eeprom_name, "wb");

    if (fd == NULL) {
        util_error("problems creating file '%s'!", afusb_dump_eeprom_name);
        return -1;
    }

    if (fwrite(afusb_eeprom_raw, 1, AFUSB_EEPROM_SIZE, fd) < 1) {
        util_error("problems writing file '%s'!", afusb_dump_eeprom_name);
        fclose(fd);
        return -1;
    }

    fclose(fd);
    return 0;
}

/* -------------------------------------------------------------- */

int afusb_ftdi_init(void)
{
    int rc = 0;
    int interface = INTERFACE_ANY;
    int do_change_mode = 0;

    util_message("Opening Alien Flash USB device...");

    if (ftdi_init(&ftdic) < 0) {
        util_error("Failed to init, aborting.");
        return -1;
    }

    /* select first interface */
    ftdi_set_interface(&ftdic, interface);

    /* open device */
    rc = ftdi_usb_open_desc(&ftdic, AFUSB_VID, AFUSB_PID, AFUSB_NAME, NULL);
    if (rc < 0) {
        util_error("Unable to open ftdi device: %d (%s)", rc, ftdi_get_error_string(&ftdic));
        ftdi_deinit(&ftdic);
        return -1;
    }

    /* read EEPROM */
    rc = ftdi_read_eeprom(&ftdic, afusb_eeprom_raw);
    if (rc < 0) {
        util_error("Unable to read EEPROM: %d (%s)", rc, ftdi_get_error_string(&ftdic));
        goto fail;
    }

    /* dump EEPROM to a file (if requested) */
    if (afusb_ftdi_dump_eeprom() < 0) {
        goto fail;
    }

    /* check if the D2XX mode bit is on */
    if (afusb_eeprom_raw[AFUSB_MODEBIT_ADDR] & AFUSB_MODEBIT_MASK) {
        /* D2XX mode: by default, complain and offer help */
        if (change_mode_request == MODE_CHANGE_NONE) {
            util_error("Alien Flash was found, but it is set to D2XX mode which is not supported.\n"
                       "        Try again with \"-u " AFUSB_MAGIC_D2XX_OFF_STRING "\" to set Alien Flash to VCP mode.\n"
                       "        Starting with \"-u " AFUSB_MAGIC_D2XX_ON_STRING "\" sets Alien Flash to D2XX mode.\n"
                       "        (WARNING: changing the mode is an UNTESTED, DANGEROUS, YOU KEEP BOTH PARTS thing)"
                    );
            goto fail;
        } else if (change_mode_request == MODE_CHANGE_VCP) {
            /* reset mode bit */
            afusb_eeprom_raw[AFUSB_MODEBIT_ADDR] &= ~AFUSB_MODEBIT_MASK;
            /* signal change */
            do_change_mode = 1;
        }
    } else {
        /* VCP mode */
        if (change_mode_request == MODE_CHANGE_D2XX) {
            /* set mode bit */
            afusb_eeprom_raw[AFUSB_MODEBIT_ADDR] |= AFUSB_MODEBIT_MASK;
            /* signal change */
            do_change_mode = 1;
        }
    }

    /* asked to change mode and need to change mode */
    if (do_change_mode) {
        util_message("Changing mode to %s...", (change_mode_request == MODE_CHANGE_D2XX) ? "D2XX" : "VCP" );
        util_warning("you may need to dis/reconnect the USB cable for the change to take effect.");
        rc = afusb_ftdi_write_eeprom();
        if (rc < 0) {
            fprintf(stderr, "Unable to write EEPROM: %d (%s)\n", rc, ftdi_get_error_string(&ftdic));
            goto fail;
        }
        util_message("Mode changed to %s.", (change_mode_request == MODE_CHANGE_D2XX) ? "D2XX" : "VCP" );
        util_error("Not checking mode again, see above warning. Aborting.");
        goto fail;
    } else if (change_mode_request != MODE_CHANGE_NONE) {
        util_warning("D2XX/VCP mode change requested when already in the mode, ignoring.");
    }

    /* set baudrate */
    rc = ftdi_set_baudrate(&ftdic, AFUSB_BAUDRATE);
    if (rc < 0) {
        util_error("Unable to set baudrate: %d (%s)", rc, ftdi_get_error_string(&ftdic));
        goto fail;
    }

    /* set line property */
    rc = ftdi_set_line_property(&ftdic, 8, 1, NONE);
    if (rc < 0) {
        util_error("Unable to set line property: %d (%s)", rc, ftdi_get_error_string(&ftdic));
        goto fail;
    }

    /* set latency if asked to */
    if (afusb_latency_timer > 0) {
        rc = ftdi_set_latency_timer(&ftdic, (unsigned char)afusb_latency_timer);
        if (rc < 0) {
            util_error("Unable to set latency: %d (%s)", rc, ftdi_get_error_string(&ftdic));
            goto fail;
        }
    }

    afusb_fd = 1;

    util_message("USB device opened successfully.");

    return 0;

fail:
    ftdi_usb_close(&ftdic);
    ftdi_deinit(&ftdic);
    return -1;
}

int afusb_ftdi_get_fd(void)
{
    return afusb_fd;
}

int afusb_ftdi_shutdown(void)
{
    if (afusb_fd < 0) {
        return 0;
    }

    util_message("Closing Alien Flash USB device...");
    ftdi_usb_close(&ftdic);
    ftdi_deinit(&ftdic);

    afusb_fd = -1;
    return 0;
}

int afusb_ftdi_write(int fd, const unsigned char *data, int num)
{
    int rc = -1;

    rc = ftdi_write_data(&ftdic, (unsigned char *)data, num);
    if (rc < num) {
        util_error("Problems sending: %s", ftdi_get_error_string(&ftdic));
        return -1;
    }

    return num;
}

int afusb_ftdi_read(int fd, unsigned char *data, int num)
{
    int rc;

    rc = ftdi_read_data(&ftdic, data, num);

    if (rc < 0) {
        util_error("Problems reading: %d (%s)", rc, ftdi_get_error_string(&ftdic));
        return -1;
    } else if (rc != num) {
        util_error("Problems reading, got %i / %i!", rc, num);
        return -1;
    }

    return num;
}
