/*
 * afusb_linux.c - Linux specific Alien Flash USB access
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <stdint.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <linux/serial.h>

#include "afusb.h"
#include "config.h"
#include "util.h"

/* -------------------------------------------------------------- */

/* timeout for serial read (deciseconds) */
#define RX_TIMEOUT 10

static int afusb_fd = -1;

static const char *afusb_device_name = AFUSB_LINUX_DEFAULT_VALUE;

/* -------------------------------------------------------------- */

int afusb_linux_set_device_param(char *param)
{
    afusb_device_name = param;
    util_message("Set Alien Flash USB device name '%s'.", afusb_device_name);
    return 0;
}

int afusb_linux_get_device_param(const char **param)
{
    *param = afusb_device_name;
    return 0;
}

/* -------------------------------------------------------------- */

static int afusb_linux_close(void)
{
    if (afusb_fd < 0) {
        return 0;
    }

    util_message("Closing Alien Flash USB device '%s'...", afusb_device_name);

    if (close(afusb_fd)) {
        perror("afusb_close");
        afusb_fd = -1;
        return -1;
    }

    afusb_fd = -1;
    return 0;
}

static int afusb_setup(void)
{
    struct termios tio;
    struct serial_struct serinfo;

    memset(&tio, 0, sizeof(tio));
    /* B38400 is used to select the custom baud rate */
    tio.c_cflag = B38400 | CS8 | CLOCAL | CREAD;
    tio.c_iflag = IGNPAR;
    tio.c_oflag = 0;
    tio.c_lflag = 0;
    tio.c_cc[VTIME] = RX_TIMEOUT;
    tio.c_cc[VMIN]  = 0;

    util_message("Flushing USB device...");

    if (tcflush(afusb_fd, TCIFLUSH)) {
        perror("afusb_setup flush");
        goto fail;
    }

    util_message("Setting attributes...");

    if (tcsetattr(afusb_fd, TCSANOW, &tio)) {
        perror("afusb_setup tcsetattr");
        goto fail;
    }

    util_message("Setting custom speed...");

    if (ioctl(afusb_fd, TIOCGSERIAL, &serinfo) < 0) {
        perror("afusb_setup ioctl get");
        goto fail;
    }

    serinfo.custom_divisor = 16; /* 48000000 / 3000000 */
    serinfo.flags &= ~ASYNC_SPD_MASK;
    serinfo.flags |= ASYNC_SPD_CUST;

    if (ioctl(afusb_fd, TIOCSSERIAL, &serinfo) < 0) {
        perror("afusb_setup ioctl set");
        goto fail;
    }

    return 0;

fail:
    afusb_linux_close();
    return -1;
}

/* -------------------------------------------------------------- */

int afusb_linux_init(void)
{
    int fd;

    util_message("Opening Alien Flash USB device '%s'...", afusb_device_name);

    fd = open(afusb_device_name, O_RDWR | O_NOCTTY);

    if (fd < 0) {
        perror(afusb_device_name);
        util_error("Failed to open '%s', aborting.", afusb_device_name);
        return -1;
    }

    afusb_fd = fd;

    if (afusb_setup()) {
        util_error("Failed to set up '%s', aborting.", afusb_device_name);
        return -1;
    }

    util_message("USB device opened successfully.");

    return 0;
}

int afusb_linux_get_fd(void)
{
    return afusb_fd;
}

int afusb_linux_shutdown(void)
{
    return afusb_linux_close();
}

int afusb_linux_write(int fd, const unsigned char *data, int num)
{
    ssize_t rc = -1;
    size_t sent_bytes;
    size_t sent_bytes_total = 0;
    size_t bytes_to_go = (size_t)num;

    do {
        rc = write(fd, data, bytes_to_go);

        if (rc < 0) {
            perror("afusb_write");
            return -1;
        }

        sent_bytes = (size_t)rc;
        sent_bytes_total += sent_bytes;
        data += sent_bytes;
        bytes_to_go -= sent_bytes;
    } while (bytes_to_go > 0);

    if (sent_bytes_total != (size_t)num) {
        util_error("sent only %i / %i!", sent_bytes_total, num);
        return -1;
    }

    return (int)sent_bytes_total;
}

int afusb_linux_read(int fd, unsigned char *data, int num)
{
    ssize_t rc;
    size_t got_bytes;
    size_t got_bytes_total = 0;
    size_t bytes_to_go = (size_t)num;

    do {
        rc = read(fd, data, bytes_to_go);

        if (rc < 0) {
            perror("afusb_read");
            return -1;
        }

        got_bytes = (size_t)rc;
        got_bytes_total += got_bytes;
        data += got_bytes;
        bytes_to_go -= got_bytes;
    } while (bytes_to_go > 0);

    if (got_bytes_total != (size_t)num) {
        util_error("got only %i / %i!", got_bytes_total, num);
        return -1;
    }

    return (int)got_bytes_total;
}
