/*
 * afusb_win32.c - win32 specific Alien Flash USB access
 *
 * Based on rs232dev.c from VICE, written by
 *  Spiro Trikaliotis <spiro.trikaliotis@gmx.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */


#include <errno.h>
#include <stdio.h>
#include <string.h>

#include <winsock.h>
#include <io.h>

#include "afusb.h"
#include "config.h"
#include "util.h"

/* ------------------------------------------------------------------------- */

static HANDLE afusb_handle;
static DCB restore_dcb;

/* just a flag */
static int afusb_fd = -1;

static const char *afusb_device_name = AFUSB_WIN32_DEFAULT_VALUE;

/* -------------------------------------------------------------- */

int afusb_win32_set_device_param(char *param)
{
    if (strlen(param) > 8) {
        util_error("invalid parameter '%s'!", param);
        return -1;
    }

    afusb_device_name = param;
    util_message("Set Alien Flash USB device number '%s'.", afusb_device_name);
    return 0;
}

int afusb_win32_get_device_param(const char **param)
{
    *param = afusb_device_name;
    return 0;
}

/* -------------------------------------------------------------- */

int afusb_win32_get_fd(void)
{
    return afusb_fd;
}


int afusb_win32_init(void)
{
    HANDLE serial_port;
    int ret = -1;
                          /*0 1 23 456789abcdef */
    char open_string[16] = "\\\\.\\COMx        ";

    int i;

    for (i = 0; i < 3; ++i) {
        open_string[i + 7] = afusb_device_name[i];
        if (afusb_device_name[i] == '\0') {
            break;
        }
    }

    util_message("Opening Alien Flash USB device '%s'...", open_string);

    do {
        DCB dcb;
        COMMTIMEOUTS comm_timeouts;

        serial_port = CreateFile(open_string, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);

        if (serial_port == INVALID_HANDLE_VALUE) {
            util_error("CreateFile '%s' failed: %lu!", open_string, GetLastError());
            break;
        }

        memset(&dcb, 0, sizeof(dcb));
        dcb.DCBlength = sizeof(dcb);

        if (!GetCommState(serial_port, &dcb)) {
            util_error("GetCommState '%s' failed: %lu!", open_string, GetLastError());
            break;
        }

        /* FIXME? */
        restore_dcb = dcb;

        dcb.BaudRate = AFUSB_BAUDRATE;
        dcb.fBinary = TRUE;
        dcb.ByteSize = 8;
        dcb.Parity = NOPARITY;
        dcb.StopBits = ONESTOPBIT;
        dcb.fOutxCtsFlow = FALSE;
        dcb.fOutxDsrFlow = FALSE;
        dcb.fDtrControl = DTR_CONTROL_DISABLE;
        dcb.fDsrSensitivity = FALSE;
        dcb.fOutX = FALSE;
        dcb.fInX = FALSE;
        dcb.fNull = FALSE;
        dcb.fRtsControl = RTS_CONTROL_DISABLE;

        if (!SetCommState(serial_port, &dcb)) {
            util_error("SetCommState '%s' failed: %lu!", open_string, GetLastError());
            break;
        }

        memset(&comm_timeouts, 0, sizeof(comm_timeouts));

        /* wait max. 1000 ms for bytes */
        comm_timeouts.ReadIntervalTimeout = MAXDWORD;
        comm_timeouts.ReadTotalTimeoutMultiplier = MAXDWORD;
        comm_timeouts.ReadTotalTimeoutConstant = 1000;

        comm_timeouts.WriteTotalTimeoutConstant = 10;
        comm_timeouts.WriteTotalTimeoutMultiplier = 10;

        if (!SetCommTimeouts(serial_port, &comm_timeouts)) {
            util_error("SetCommTimeouts '%s' failed: %lu!", open_string, GetLastError());
            break;
        }

        afusb_fd = 1;
        afusb_handle = serial_port;
        ret = 0;
        serial_port = INVALID_HANDLE_VALUE;
    } while (0);

    if (serial_port != INVALID_HANDLE_VALUE) {
        CloseHandle(serial_port);
    }

    return ret;
}

int afusb_win32_shutdown(void)
{
    if (afusb_fd < 0) {
        return 0;
    }

    util_message("Closing Alien Flash USB device...");

    if (!SetCommState(afusb_handle, &restore_dcb)) {
        util_error("SetCommState failed: %lu!", GetLastError());
        return -1;
    }

    CloseHandle(afusb_handle);
    afusb_fd = -1;
    return 0;
}

int afusb_win32_write(int fd, const unsigned char *data, int num)
{
    DWORD number_of_bytes = 1;

    if (WriteFile(afusb_handle, data, num, &number_of_bytes, NULL) == 0) {
        util_error("Problems sending: %lu!", GetLastError());
        return -1;
    }

    if (number_of_bytes != (DWORD)num) {
        util_error("Problems sending: sent only %lu / %i!", number_of_bytes, num);
        return -1;
    }

    return (int)number_of_bytes;
}

int afusb_win32_read(int fd, unsigned char *data, int num)
{
    DWORD number_of_bytes = 0;
    DWORD dnum = (DWORD)num;
    DWORD got_bytes = 0;

    do {
        if (ReadFile(afusb_handle, data, dnum, &number_of_bytes, NULL) == 0) {
            util_error("Problems reading: %lu!", GetLastError());
            return -1;
        }
        got_bytes += number_of_bytes;
        data += number_of_bytes;
        dnum -= number_of_bytes;
    } while (dnum > 0);

    if (got_bytes != (DWORD)num) {
        util_error("Problems reading, got %lu / %i!", got_bytes, num);
    }

    return (int)got_bytes;
}
