/*
 * cart.c - crt load/save.
 *
 * Based on easyflash.c and crt.c from VICE, written by
 *  Andreas Boose <viceteam@t-online.de>
 *  ALeX Kazik <alx@kazik.de>
 *  Marco van den Heuvel <blackystardust68@yahoo.com>
 *
 */

#include <stdio.h>
#include <string.h>

#include "cart.h"
#include "config.h"
#include "flashcore.h"
#include "globals.h"
#include "util.h"

/* ------------------------------------------------------------------------- */
/* globals */
                              /* 0123456789abcdef */
char text_description[16 + 1] = "";
char text_cartname[32 + 1] = "afapack v." PACKAGE_VERSION;

/* -------------------------------------------------------------------------- */

static const char CRT_HEADER[] = "C64 CARTRIDGE   ";
static const char CHIP_HEADER[] = "CHIP";

#define ALIENFLASH_BIN_SIZE (16 * 1024 * 1024)
#define ALIENFLASH_SLOT_SIZE (ALIENFLASH_BIN_SIZE / 16)
#define ALIENFLASH_CHIPS    4
#define ALIENFLASH_CHIP_SIZE (ALIENFLASH_BIN_SIZE / ALIENFLASH_CHIPS)
#define ALIENFLASH_CHIP_BANKS (ALIENFLASH_CHIP_SIZE / 0x2000)
#define ALIENFLASH_ROMLH_BANKS (ALIENFLASH_CHIP_BANKS * 2)
#define EASYFLASH_CHIP_BANKS ((1024 * 1024) / 2 / 0x2000)

/* -------------------------------------------------------------------------- */

int af_check_empty(const unsigned char *data)
{
    int i;

    for (i = 0; i < 0x2000; i++) {
        if (data[i] != 0xff) {
            return 0;
        }
    }
    return 1;
}

static int write_chip(FILE* fd, const unsigned char* chipheader, const unsigned char* data)
{
    if (fwrite(chipheader, 1, 0x10, fd) != 0x10) {
        return -1;
    }

    if (fwrite(data, 1, 0x2000, fd) != 0x2000) {
        return -1;
    }

    return 0;
}

static int write_chip_if_not_empty(FILE* fd, const unsigned char* chipheader, const unsigned char* data)
{
    if (af_check_empty(data) == 0) {
        return write_chip(fd, chipheader, data);
    }
    return 0;
}

static int aefcart_load_crt(FILE *fd, unsigned int rom_offset, unsigned int *dest_out, int *romh_too_out)
{
    unsigned char chipheader[0x10];
    unsigned int bank, offset, length, dest = 0, dest_max = 0;
    int romh_too = 0;

    while (1) {
        if (fread(chipheader, 0x10, 1, fd) < 1) {
            break;
        }

        bank = (chipheader[0xa] << 8) | chipheader[0xb];
        offset = (chipheader[0xc] << 8) | chipheader[0xd];
        length = (chipheader[0xe] << 8) | chipheader[0xf];

        if (length != 0x2000) {
            return -1;
        }

        if (bank >= ALIENFLASH_ROMLH_BANKS || !(offset == 0x8000 || offset == 0xa000 || offset == 0xe000)) {
            return -1;
        }

        if (offset != 0x8000) {
            bank += ALIENFLASH_ROMLH_BANKS;
            romh_too = 1;
        }

        dest = bank * 0x2000 + rom_offset;

        if ((dest + 0x2000) > ALIENFLASH_BIN_SIZE) {
            util_error("destination $%07x > $0ffffff, aborting. (bank = $%03x, ROM offset = $%06x)", dest, bank, rom_offset);
            return -1;
        }

        if (fread(&flash_data[dest], 0x2000, 1, fd) < 1) {
            return -1;
        }

        if ((dest & 0x7fffff) > (dest_max & 0x7fffff)) {
            dest_max = dest;
        }
    }

    if (romh_too_out != NULL) {
        *romh_too_out = romh_too;

        if (rom_offset < 0x800000) {
            dest_max &= 0x7fffff;
        }
   }

    if (dest_out != NULL) {
        *dest_out = dest_max + 0x2000;
    }

    return 0;
}

static FILE *crt_load(const char* filename, int *crt_id, int *want_crt_id)
{
    unsigned char header[0x40];
    FILE *fd = NULL;
    int new_crt_id;

    fd = fopen(filename, "rb");

    if (fd == NULL) {
        util_error("problems opening file '%s'!", filename);
        return NULL;
    }

    if (fread(header, 0x40, 1, fd) < 1) {
        util_error("problems reading file '%s'!", filename);
        fclose(fd);
        return NULL;
    }

    if (strncmp((char*)header, CRT_HEADER, 16)) {
        util_error("file '%s' doesn't have CRT header!", filename);
        fclose(fd);
        return NULL;
    }

    new_crt_id = (header[0x17] + (header[0x16] * 256));

    *crt_id = new_crt_id;

    if ((want_crt_id != NULL) && (new_crt_id != *want_crt_id)) {
        util_error("file '%s' has CRT ID %i instead of %i!", filename, new_crt_id, *want_crt_id);
        fclose(fd);
        return NULL;
    }

    return fd;
}

/* -------------------------------------------------------------------------- */

static int align_and_check_offsets(int *offset_out, int *offset_end_out, int max_offset, int max_diff)
{
    int old;
    int offset = *offset_out;
    int offset_end = *offset_end_out;

    /* mask offsets to bank limits */
    if (offset & 0x001fff) {
        old = offset;
        offset &= 0xffe000;
        util_warning("alinging offset $%06x to bank limit $%06x", old, offset);
    }

    if ((offset_end & 0x001fff) != 0x001fff) {
        old = offset_end;
        offset_end |= 0x001fff;
        util_warning("alinging end offset $%06x to bank limit $%06x", old, offset_end);
    }

    /* check ranges */
    if (offset > max_offset) {
        util_error("offset $%06x too large!", offset);
        return -1;
    }

    if ((offset_end - offset) > max_diff) {
        old = offset_end;
        offset_end = offset + max_diff;
        util_warning("range $%06x-$%06x too large, shrinking to $%06x-$%06x", offset, old, offset, offset_end);
    }

    if (offset_end > max_offset) {
        util_error("end offset $%06x too large!", offset_end);
        return -1;
    }

    *offset_out = offset;
    *offset_end_out = offset_end;

    return 0;
}

int afcart_save(const char* filename, int offset, int offset_end)
{
    FILE *fd;
    unsigned char header[0x40], chipheader[0x10];
    unsigned char *data;
    unsigned int bank, max_bank;
    unsigned int romlh;

    if (filename == NULL) {
        return -1;
    }

    if (offset < 0) {
        offset = 0;
    }

    if (offset_end < 0) {
        offset_end = 0xffffff;
    }

    if (align_and_check_offsets(&offset, &offset_end, 0xffffff, 0xffffff) < 0) {
        return -1;
    }

    max_bank = (offset_end + 1 - offset) / 0x2000 / 2;

    util_message("Saving Alien Flash cart '%s' from $%06x-$%06x (banks $000-$%03x)...", filename, offset, offset_end, max_bank - 1);

    fd = fopen(filename, "wb");

    if (fd == NULL) {
        return -1;
    }

    memset(header, 0x0, 0x40);
    memset(chipheader, 0x0, 0x10);

    strcpy((char *)header, CRT_HEADER);

    header[0x13] = 0x40;
    header[0x14] = 0x01;
    header[0x17] = CARTRIDGE_ALIEN_FLASH;
    header[0x18] = 0x01;
    strncpy((char *)&header[0x20], text_cartname, 32);
    if (fwrite(header, 1, 0x40, fd) != 0x40) {
        fclose(fd);
        return -1;
    }

    strcpy((char *)chipheader, CHIP_HEADER);
    chipheader[0x06] = 0x20;
    chipheader[0x07] = 0x10;
    chipheader[0x09] = 0x02;
    chipheader[0x0e] = 0x20;

    for (romlh = 0; romlh < 2; ++romlh) {
        chipheader[0x0c] = (romlh == 0) ? 0x80 : 0xa0;
        for (bank = 0; bank < max_bank; bank++) {
            chipheader[0x0a] = ((bank >> 8) & 0xff);
            chipheader[0x0b] = (bank & 0xff);

            data = flash_data + bank * 0x2000 + offset + romlh * (ALIENFLASH_BIN_SIZE / 2);

            if (write_chip_if_not_empty(fd, chipheader, data) != 0) {
                fclose(fd);
                return -1;
            }
        }
    }

    fclose(fd);
    return 0;
}

int afcart_load(const char* filename)
{
    FILE *fd = NULL;
    int crt_id = 0;
    int want_crt_id = CARTRIDGE_ALIEN_FLASH;

    util_message("Loading Alien Flash cart '%s'...", filename);

    fd = crt_load(filename, &crt_id, &want_crt_id);

    if (fd == NULL) {
        return -1;
    }

    if (aefcart_load_crt(fd, 0, NULL, NULL) < 0) {
        util_error("problems reading file '%s'!", filename);
        fclose(fd);
        return -1;
    }

    fclose(fd);
    return 0;
}

int efcart_save(const char* filename, int offset, int offset_end)
{
    FILE *fd;
    unsigned char header[0x40], chipheader[0x10];
    unsigned int bank, dest, max_bank;

    if (filename == NULL) {
        return -1;
    }

    if (align_and_check_offsets(&offset, &offset_end, 0x7fffff, 0x07ffff) < 0) {
        return -1;
    }

    max_bank = (offset_end + 1 - offset) / 0x2000;

    util_message("Saving EasyFlash cart '%s' from $%06x-$%06x (banks $00-$%02x)...", filename, offset, offset_end, max_bank - 1);

    fd = fopen(filename, "wb");

    if (fd == NULL) {
        util_error("problems creating file '%s'!", filename);
        return -1;
    }

    memset(header, 0x0, 0x40);
    memset(chipheader, 0x0, 0x10);

    strcpy((char *)header, CRT_HEADER);

    header[0x13] = 0x40;
    header[0x14] = 0x01;
    header[0x17] = CARTRIDGE_EASYFLASH;
    header[0x18] = 0x01;
    strncpy((char *)&header[0x20], text_cartname, 32);
    if (fwrite(header, 1, 0x40, fd) != 0x40) {
        fclose(fd);
        return -1;
    }

    strcpy((char *)chipheader, CHIP_HEADER);
    chipheader[0x06] = 0x20;
    chipheader[0x07] = 0x10;
    chipheader[0x09] = 0x02;
    chipheader[0x0e] = 0x20;

    for (bank = 0; bank < max_bank; bank++) {
        chipheader[0x0a] = ((bank >> 8) & 0xff);
        chipheader[0x0b] = (bank & 0xff);

        chipheader[0x0c] = 0x80;
        dest = bank * 0x2000 + offset;

        if (write_chip_if_not_empty(fd, chipheader, &flash_data[dest]) != 0) {
            fclose(fd);
            return -1;
        }

        chipheader[0x0c] = 0xa0;

        if (write_chip_if_not_empty(fd, chipheader, &flash_data[dest + (ALIENFLASH_BIN_SIZE / 2)]) != 0) {
            fclose(fd);
            return -1;
        }
    }

    fclose(fd);
    return 0;
}

static int get_default_cart_params(int crt_id, int *default_offset, const char **cart_name)
{
    switch (crt_id) {
        case CARTRIDGE_ACTION_REPLAY:
            *default_offset = 0x870000;
            *cart_name = "Action Replay";
            break;
        case CARTRIDGE_EASYFLASH:
            *default_offset = 0x080000;
            *cart_name = "EasyFlash";
            break;
        case CARTRIDGE_ALIEN_FLASH:
            *default_offset = 0;
            *cart_name = "Alien Flash";
            break;
        default:
            util_error("Unsupported CRT ID %i.", crt_id);
            return -2;
    }

    return 0;
}

int detect_cart_type(const char* filename, int *default_offset_out, int *crt_id_out, const char **cart_name)
{
    FILE *fd = NULL;

    fd = crt_load(filename, crt_id_out, NULL);

    if (fd == NULL) {
        util_error("problems loading file '%s'!", filename);
        return -1;
    }

    fclose(fd);

    return get_default_cart_params(*crt_id_out, default_offset_out, cart_name);
}

int anycart_load(const char* filename, int *offset_out, unsigned int *last_addr, int *romh_too)
{
    FILE *fd = NULL;
    int crt_id = 0;
    const char *cart_name = "???";
    int default_offset = 0;
    int offset = *offset_out;

    util_message("Loading unknown cart '%s'...", filename);

    fd = crt_load(filename, &crt_id, NULL);

    if (fd == NULL) {
        util_error("problems loading file '%s'!", filename);
        return -1;
    }

    if (get_default_cart_params(crt_id, &default_offset, &cart_name) < 0) {
        fclose(fd);
        return -1;
    }

    if (offset < 0) {
        offset = default_offset;
        *offset_out = offset;
    }

    util_message("Loading %s cart '%s' to offset $%06x...", cart_name, filename, offset);

    if (aefcart_load_crt(fd, (unsigned int)(offset), last_addr, romh_too) < 0) {
        util_error("problems reading file '%s'!", filename);
        fclose(fd);
        return -1;
    }

    fclose(fd);
    return 0;
}

int arcart_save(const char* filename, int rom_offset, int banks, int crt_id)
{
    FILE *fd;
    unsigned char header[0x40], chipheader[0x10];
    unsigned int bank, dest;

    if (filename == NULL) {
        return -1;
    }

    util_message("Saving cart CRT ID %i '%s' with %i banks from offset $%06x...", crt_id, filename, banks, rom_offset);

    fd = fopen(filename, "wb");

    if (fd == NULL) {
        util_error("problems creating file '%s'!", filename);
        return -1;
    }

    memset(header, 0x0, 0x40);
    memset(chipheader, 0x0, 0x10);

    strcpy((char *)header, CRT_HEADER);

    header[0x13] = 0x40;
    header[0x14] = 0x01;
    header[0x17] = crt_id;
    header[0x18] = 0x01;
    strncpy((char *)&header[0x20], text_cartname, 32);
    if (fwrite(header, 1, 0x40, fd) != 0x40) {
        fclose(fd);
        return -1;
    }

    strcpy((char *)chipheader, CHIP_HEADER);
    chipheader[0x06] = 0x20;
    chipheader[0x07] = 0x10;
    chipheader[0x09] = 0;
    chipheader[0x0e] = 0x20;


    for (bank = 0; bank < (unsigned int)banks; bank++) {
        chipheader[0x0a] = ((bank >> 8) & 0xff);
        chipheader[0x0b] = (bank & 0xff);

        chipheader[0x0c] = 0x80;
        dest = bank * 0x2000 + rom_offset;

        if (write_chip(fd, chipheader, &flash_data[dest]) != 0) {
            fclose(fd);
            return -1;
        }
    }

    fclose(fd);
    return 0;
}

int bin_save(const char* filename, int offset, int offset_end)
{
    FILE *fd;

    if (filename == NULL) {
        return -1;
    }

    util_message("Saving binary '%s' from $%06x-$%06x...", filename, offset, offset_end);

    fd = fopen(filename, "wb");

    if (fd == NULL) {
        util_error("problems creating file '%s'!", filename);
        return -1;
    }

    if (fwrite(&flash_data[offset], 1, offset_end + 1 - offset, fd) < 1) {
        util_error("problems writing file '%s'!", filename);
        fclose(fd);
        return -1;
    }

    fclose(fd);
    return 0;
}

int detect_aftt_header(int base, int type, int *len_out, char *buf, int *load_out)
{
    if (memcmp(&flash_data[base], aftt_header_magic, 8) == 0) {
        memcpy(buf, &flash_data[base + 0x20], 16);
        buf[16] = '\0';

        *load_out = flash_data[base + 0x100]
                  + (flash_data[base + 0x101] << 8);

        *len_out = flash_data[base + 0x18]
                 + (flash_data[base + 0x19] << 8)
                 + (flash_data[base + 0x1a] << 16)
                 + (flash_data[base + 0x1b] << 24);

        return type == flash_data[base + 0x08];
    }

    return 0;
}

int has_aftt_file(int type, char *buf)
{
    int x, y;
    int base = 0;

    switch (type) {
        case AFTT_TYPE_PRG:
            base = 0x10000;
            break;

        case AFTT_TYPE_D64:
            base = 0x40000;
            break;

        default:
            return -1;
    }

    return detect_aftt_header(base, type, &x, buf, &y);
}
